; C Library
;
; Copyright (C) 2022 Kestrel Institute (http://www.kestrel.edu)
; Copyright (C) 2022 Kestrel Technology LLC (http://kestreltechnology.com)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "C")

(include-book "../language/abstract-syntax")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ atc-abstract-syntax
  :parents (atc-implementation)
  :short "Abstract syntax of C for ATC."
  :long
  (xdoc::topstring
   (xdoc::p
    "ATC generates C code by
     generating abstract syntax trees
     and pretty-printing them.
     For now we use the "
    (xdoc::seetopic "abstract-syntax" "abstract syntax")
    " from the C language formalization.
     In fact, that abstract syntax has been initially motivated by ATC.")
   (xdoc::p
    "However, in the future we may have a separate syntax for ATC,
     for the following reasons:")
   (xdoc::ol
    (xdoc::li
     "The abstract syntax from the C language formalization
      captures syntax after preprocessing,
      but at some point we may want ATC to generate code
      with at least some of the preprocessing constructs,
      such as @('#include')s, and possibly also some (simple) macros.
      This means that the ATC abstract syntax will have to mix
      preprocessing constructs with the preprocessed constructs:
      this is something that may not be part, in this mixed form,
      of the language formalization,
      which should differentiate between
      preprocessing translation units and
      (preprocessed) translation units.")
    (xdoc::li
     "We might want ATC to generate certain comments in the code,
      which would require the ATC abstract syntax to incorporate
      some information about comments.
      However, in the language formalization,
      comments, and their removal during one of C's translation phases,
      would be captured differently,
      not as part of the abstract syntax
      over which the language semantics is defined.")
    (xdoc::li
     "While the abstract syntax from the language formalization
      may be generalized to cover much more of the C language,
      the abstract syntax for ATC can incorporate restrictions
      that make it simpler and that fit the C code generated by ATC.
      In particular,
      the C syntax for declarations and related entities is quite complex,
      with significant mutual recursion,
      and with many constraints not directly captured by the C grammar.
      For instance, @('typedef') is classified as a storage class specifier,
      for syntactic convenience apparently [C:6.7.1/5],
      even though its role is very different from the others.
      Thus, by differentiating more explicitly, in our ATC abstract syntax,
      among different kinds of declarations and related entities,
      we make things more clear overall, as far as ATC is concerned."))
   (xdoc::p
    "Notwithstanding the above three reasons,
     in the short term, for expediency, we might actually
     incorporate preprocessing directives and comments,
     and impose restrictions,
     on the abstract syntax from the language formalization,
     rather than creating a separate abstract syntax for ATC.
     So long as the language formalization appropriately covers a subset of C,
     there is nothing inherently wrong with that approach.
     However, longer-term, as the language formalization is made more general,
     in particular covering the translation phases [C:5.1.12] explicitly,
     we will likely need to differentiate the abstract syntax for ATC
     from the one(s) from the language formalization.
     For proof generation,
     we would provide a mapping from the former to the latter,
     or in fact we may have the proof apply to the actual concrete syntax,
     if the language formalization includes parsing.")
   (xdoc::p
    "Some observations about some parts of the abstract syntax
     in relation to ATC:")
   (xdoc::ul
    (xdoc::li
     "The fixtype @(tsee ident) allows any ACL2 string,
      which may thus not be valid C identifiers.
      However, ATC always generates valid C identifiers.")
    (xdoc::li
     "The fixtype @(tsee ident) models only ASCII identifiers.
      For ATC, ASCII identifiers may be all that we need
      in the foreseeable future:
      we may not need to generate C programs with
      identifiers that include non-ASCII characters,
      some aspects of which may be
      implementation-dependent or locale-dependent.
      Since ASCII identifiers are portable,
      we plan for ATC to generate only ASCII identifiers,
      unless there will be reasons to support a broader range.")
    (xdoc::li
     "The fixtype @(tsee ident) allows identifiers of any length.
      In the interest of portability, it is our intention to have ATC
      generate identifiers of 31 characters or less
      (which is the minimum of significant characters in (external) identifiers
      [C:6.4.2.1/5] [C:6.4.2.1/6] [C:5.2.4.1]
      which may not be a significant limitation.")
    (xdoc::li
     "The fixtype @(tsee iconst) does not capture leading 0s
      in octal and hexadecimal constants.
      This is not a severe limitation,
      but at some point we may want ATC to generate
      something like @('0x0000ffff').")
    (xdoc::li
     "The fixtype @(tsee const) includes enuemration constants,
      which, as discussed there,
      cannot be differentiated from identifier expressions
      during parsing, but only after static semantic checking.
      This is not an issue for ATC, which generates, does not parse, C code:
      ATC can generate one or the other case in the code.")
    (xdoc::li
     "The fixtypes for declarations do not support function pointers.
      Most likely, this support will not be neded for ATC,
      given that ACL2 is first-order,
      and thus cannot readily represent C function pointers.
      (However, perhaps there is a way
      to represent function pointers with @(tsee apply$).)")
    (xdoc::li
     "The fixtype @(tsee block-item) only supports object declarations.
      This suffices for ATC currently.")
    (xdoc::li
     "The fixtype @(tsee fundef) could be alteratively defined as
      consisting of a function declaration and a body.
      However, even though this would work in abstract syntax,
      in concrete syntax a function declaration has to end with a semicolon
      (and that is why the grammar rule in [C:6.9.1/1]
      does not use a declaration, but rather its components):
      thus, for the ATC pretty-printer,
      we want to differentiate between
      the type specifier sequences and declarators
      that form a full function declaration,
      and the type specifier sequences and declarators
      that are part of a function definition.")))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-ident ()
  :returns (id identp)
  :short "An irrelevant identifier, usable as a dummy return value."
  (with-guard-checking :none (ec-call (ident-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-ident))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-iconst-length ()
  :returns (tysuff iconst-lengthp)
  :short "An irrelevant length suffix, usable as a dummy return value."
  (with-guard-checking :none (ec-call (iconst-length-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-iconst-length))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-iconst ()
  :returns (iconst iconstp)
  :short "An irrelevant integer constant, usable as a dummy return value."
  (with-guard-checking :none (ec-call (iconst-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-iconst))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-tyspecseq ()
  :returns (ty tyspecseqp)
  :short "An irrelevant type specifier sequence,
          usable as a dummy return value."
  (with-guard-checking :none (ec-call (tyspecseq-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-tyspecseq))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-tyname ()
  :returns (tyname tynamep)
  :short "An irrelevant type name, usable as a dummy value."
  (with-guard-checking :none (ec-call (tyname-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-tyname))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-unop ()
  :returns (op unopp)
  :short "An irrelevant unary operator, usable as a dummy return value."
  (with-guard-checking :none (ec-call (unop-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-unop))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-binop ()
  :returns (op binopp)
  :short "An irrelevant binary operator, usable as a dummy return value."
  (with-guard-checking :none (ec-call (binop-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-binop))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-expr ()
  :returns (expr exprp)
  :short "An irrelevant expression, usable as a dummy return value."
  (with-guard-checking :none (ec-call (expr-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-expr))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-tag-declon ()
  :returns (declon tag-declonp)
  :short "An irrelevant structure/union/enumeration declaration,
          usable as a dummy return value."
  (with-guard-checking :none (ec-call (tag-declon-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-tag-declon))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-param-declon ()
  :returns (param param-declonp)
  :short "An irrelevant parameter declaration, usable as a dummy return value."
  (with-guard-checking :none (ec-call (param-declon-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-param-declon))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-stmt ()
  :returns (stmt stmtp)
  :short "An irrelevant statement, usable as a dummy return value."
  (with-guard-checking :none (ec-call (stmt-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-stmt))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-ext-declon ()
  :returns (ext ext-declonp)
  :short "An irrelevant external declaration, usable as a dummy return value."
  (with-guard-checking :none (ec-call (ext-declon-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-ext-declon))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define irr-transunit ()
  :returns (tunit transunitp)
  :short "An irrelevant translation unit, usable as a dummy return value."
  (with-guard-checking :none (ec-call (transunit-fix :irrelevant)))
  ///
  (in-theory (disable (:e irr-transunit))))
