------------------------------------------------------------------------------
--                                                                          --
--                           GNATELIM COMPONENTS                            --
--                                                                          --
--                     G N A T E L I M . O P T I O N S                      --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 2013-2014, AdaCore                     --
--                                                                          --
-- GNATELIM  is  free software;  you can  redistribute it and/or  modify it --
-- under the terms of the  GNU  General Public License  as published by the --
-- Free Software Foundation; either version 3 or (at your option) any later --
-- version. GNATELIM is distributed in the hope that it will be useful, but --
-- WITHOUT ANY WARRANTY; without even the implied warranty of  MERCHANTABI- --
-- LITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public Li- --
-- cense for  more details.  You should  have  received  a copy of the  GNU --
-- General Public License distributed with GNAT; see file COPYING3. If not, --
-- go to http://www.gnu.org/licenses for a complete copy of the license.    --
--                                                                          --
-- The original version  of  Gnatelim  was developed by  Alain  Le  Guennec --
-- It is now maintained by AdaCore (http://www.adacore.com)                 --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains definitions of gnatelim-specific options

with GNAT.OS_Lib;              use GNAT.OS_Lib;

with ASIS_UL.Common;           use ASIS_UL.Common;
with ASIS_UL.Compiler_Options; use ASIS_UL.Compiler_Options;
with ASIS_UL.Debug;            use ASIS_UL.Debug;
with ASIS_UL.Environment;      use ASIS_UL.Environment;
with ASIS_UL.Options;          use ASIS_UL.Options;
with ASIS_UL.Output;           use ASIS_UL.Output;
with ASIS_UL.Projects;
with ASIS_UL.Source_Table;     use ASIS_UL.Source_Table;

package body Gnatelim.Options is

   --------------------
   -- Scan_Arguments --
   --------------------

   procedure Scan_Arguments (Parser : Opt_Parser := Command_Line_Parser) is
      In_Project_File : constant Boolean := Parser /= Command_Line_Parser;
   begin
      loop
         case GNAT.Command_Line.Getopt
                (
                 "P: U X! "            &   --  project-specific options
                 "t v q  "             &
                 "l? "                 & --  Specifying the log file
                 "wq ws wn wf "        & --  Warning message control
                 "o= "                 & --  Specify the output file
                 "files= "             & --  Specifies a set of files
                 --  to process
                 "main= "              & --  specifies the main subprogram
                 "-version -help "     & --  print version and usage
                 "-no-elim-dispatch "  & --  Turns OFF elimination of
                 --  dispatching operations,

                 --  Non-documented options:
                 "d? "                 & --  debug mode/options
                 "a "                  & --  Process RTL components, no effect
                 --  at the moment
                 "-ignore= "           & --  specifies a set of units not to
                 --  generate pragmas for
                 "-elim-dispatching "  & --  Turns ON elimination of
                 --  dispatching operations, does nothing for now, elimination
                 --  of dispatching operations is ON by default

                 "-old-pragma-format " & --  Generate UNIT_NAME pragma
                 --  Parameter (Parser => Parser)

                 --  obsolete features, for backward compatibility
                 "b: I: C: -GCC=: -GNATMAKE= m"
                 )
         is

            when ASCII.NUL =>
               exit when not More_Arguments;
            when 'a' =>
               Process_RTL_Units := True;
            when 'b' =>
               null;  --  obsolete features, for backward compatibility

            when 'C' =>
               --  In old gnatelim this switch has been used to specify a
               --  configuration file, so:
               Store_GNAT_Option_With_Path
                 ("gnatec", Parameter (Parser => Parser));
            when 'd' =>

               if Full_Switch (Parser => Parser) = "d" then
                  Set_Debug_Options (Parameter (Parser => Parser));
               end if;

            when 'f' =>

               if Full_Switch (Parser => Parser) = "files" then
                  Read_Args_From_File
                    (Parameter (Parser => Parser),
                     Arg_Project         => Gnatelim.Options.Gnatelim_Prj,
                     Store_With_No_Check => True);
               end if;

            when 'I' =>
               Store_I_Option (Parameter (Parser => Parser));

            when 'l' =>

               if Full_Switch (Parser => Parser) = "l" then
                  Log_Mode := True;
                  Set_Log_File_Name (Parameter (Parser => Parser));
                  Set_Log_File;
               end if;

            when 'm' =>

               if Full_Switch (Parser => Parser) = "main" then
                  ASIS_UL.Options.Main_Subprogram_Name :=
                    new String'(Parameter (Parser => Parser));
               --  elsif Full_Switch (Parser => Parser) = "m" then
               --     null; --  Obsolete switch, for backwards compatibility
               end if;

            when 'o' =>

               if Full_Switch (Parser => Parser) = "o" then
                  Set_Pipe_Mode (False);
                  Set_Report_File_Name (Parameter (Parser => Parser));
               end if;

            when 'P' =>
               Gnatelim.Projects.Store_Project_Source
                 (Gnatelim_Prj,
                  Parameter (Parser => Parser));

            when 'q' =>
               Quiet_Mode := True;

            when 't' =>
               if Full_Switch (Parser => Parser) = "t" then
                  Compute_Timing := True;
               end if;

            when 'U' =>
               if ASIS_UL.Projects.U_Option_Set then
                  Error ("-U can be specified only once");
                  raise Parameter_Error;
               end if;

               ASIS_UL.Projects.U_Option_Set := True;

            when 'v' =>
               Verbose_Mode := True;

            when 'w' =>

               if Full_Switch (Parser => Parser) = "wq" then
                  Warning_Mode := Quiet;
               elsif Full_Switch (Parser => Parser) = "ws" then
                  Warning_Mode := Short;
               elsif Full_Switch (Parser => Parser) = "wn" then
                  Warning_Mode := Normal;
               elsif Full_Switch (Parser => Parser) = "wf" then
                  Warning_Mode := Full;
               end if;

            when 'X' =>
               ASIS_UL.Projects.Store_External_Variable
                 (Var                  => Parameter  (Parser => Parser),
                  Is_From_Command_Line => not In_Project_File);

            when '-' =>

               if Full_Switch (Parser => Parser) = "-GCC" then
                  null;  --  obsolete features, for backward compatibility
               elsif Full_Switch (Parser => Parser) = "-GNATMAKE" then
                  null;  --  obsolete features, for backward compatibility
               elsif Full_Switch (Parser => Parser) = "-elim-dispatching" then
                  Gnatelim.Options.Eliminate_Dispatching_Operations := True;
               elsif Full_Switch (Parser => Parser) = "-no-elim-dispatch" then
                  Gnatelim.Options.Eliminate_Dispatching_Operations := False;
               elsif Full_Switch (Parser => Parser) = "-help" then
                  Print_Usage := True;
                  return;
               elsif Full_Switch (Parser => Parser) = "-ignore" then
                  if Is_Regular_File (Parameter (Parser => Parser)) then
                     Gnatelim.Options.Exempted_Units :=
                       new String'(Normalize_Pathname
                                     (Parameter (Parser => Parser)));
                  else
                     Error (Parameter (Parser => Parser) & " not found");
                     raise Parameter_Error;
                  end if;
               elsif Full_Switch (Parser => Parser) = "-old-pragma-format" then
                  --  Junk for now, should be changed to opposite when
                  --  K108-003 is complete
                  Gnatelim.Options.Long_Pragma_Format := False;
               elsif Full_Switch (Parser => Parser) = "-version" then
                  Print_Version := True;
                  return;
               end if;

            when others =>
               raise Parameter_Error;
         end case;
      end loop;

      Process_cargs_Section;

   exception
      when GNAT.Command_Line.Invalid_Switch =>
         Error ("invalid switch : " & Full_Switch (Parser => Parser));
         raise Parameter_Error;

      when GNAT.Command_Line.Invalid_Parameter =>
         Error ("missing Parameter (Parser => Parser) for: " &
                Full_Switch (Parser => Parser));
         raise Parameter_Error;

   end Scan_Arguments;

end Gnatelim.Options;
