#!/bin/bash
#*******************************************************************************
#
#   This file is part of the ESO Pipeline Tool
#+  Copyright (C) 2004 European Southern Observatory
#
#   This program is free software; you can redistribute it and/or modify
#+  it under the terms of the GNU General Public License as published by
#+  the Free Software Foundation; either version 2 of the License, or
#+  (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#+  but WITHOUT ANY WARRANTY; without even the implied warranty of
#+  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#+  GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#+  along with this program; if not, write to the Free Software
#+  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#  Author: Nick Kornweibel for ESO
#
#  This is an example script that may be used
#+ with the CPL command line utility's "--output-script" option.
#
#  When specified, the --output-script option will run the
#+ given program once for each product file produced by a
#+ recipe. The program is launched with 3 command line parameters,
#+ which are required by this example script:
#
#  $1: the name of the recipe
#  $2: the full path of the product file
#  $3: the PRO_CATG keyword of the product file
#
#  This example script renames the product file with an
#+ appropriate 4-digit number suffix based on the 
#+ recipe and tag information.
#
#  The script returns 0 on success, else non-zero. If there is
#+ no specified case for the recipe/product category combination,
#+ then 0 is returned (i.e. success).
#
#  Other possible uses for the --output-script option could be:
#  - compressing or archiving product files
#  - sending products to a visualisation tool
#  - triggering the next step of data reduction
#
#  This script may be extended as required to include
#+ suffixes for additional recipes and product tags.
#
#  For more information on CPL please see:
#  http://www.eso.org/observing/cpl
#
#*******************************************************************************

if [ $# -ne "3" ]
then
  echo
  echo "`basename $0`: expected 3 command-line arguments"
  echo "  1: the name of the recipe"
  echo "  2: the full path of the product file"
  echo "  3: the PRO_CATG keyword of the product file"
  exit 1
fi  

recipeName=$1
productFile=$2
productCat=$3

suffix=""

#
#  the return code, set to non-zero to indicate an error
#
returnCode=0

#
#  rename the productFile by adding $suffix
#+ to the end of the file (before the extension, if found), as follows:
#+ (Assuming suffix is "_0000")
#+
#+ For files with no extension, append the suffix
#+ e.g. fred would become fred_0000
#+
#+ For files with an extension, append the suffix before the extension
#+ e.g. fred.fits would become fred_0000.fits
#+
#+ For files with a 4-digit number before an extension, 
#+ append the suffix before the extension, replacing the 4-digit number.
#+ e.g. fred_0001.fits would become fred_0000.fits
#
#  on error the returnCode is set
#
#  note that this function assumes that the productFile
#+ contains only a single "." character.
#
renameProduct() {
    dirName=`dirname $productFile`
    fileName=`basename $productFile`
    newFileName=""

    # if the file has 4-digit name part before an extension
    echo "$fileName" | egrep -e "_[0-9][0-9][0-9][0-9]\." > /dev/null
    if [ $? -eq 0 ] 
    then
        # product filename contains a "XXXX." (X is [0-9])
        newFileName=`echo $fileName | \
            sed -e "s/_[0-9][0-9][0-9][0-9]\./$suffix\./g"`
    else 
        # if the file has an extension
        echo "$fileName" | egrep -e "\." > /dev/null
        if [ $? -eq 0 ] 
        then
            # product filename contains a "."
            newFileName=`echo $fileName | sed -e "s/\./$suffix\./g"`
        else
            # product filename contains no "."
            newFileName=$fileName$suffix
        fi
    fi
    newProductName=$dirName/$newFileName

    # echo "Rename $productFile to $newProductName"

    # rename only if required
    if [ "$fileName" != "$newFileName" ]
    then
        mv -f $productFile $newProductName
    fi
    returnCode=$?
}

#
#  Nested Case statements for determining the
#+ correct suffix based on the recipe and the product category
#
#  Edit these case statements as required. An example recipe
#+ code "block" is given at the end of the case statement which
#+ can be copied and modified.
#
case "$recipeName" in

  "vmbias")

	case "$productCat" in
            "MASTER_BIAS")                suffix="0000";;
        esac
  ;;

  "vmspflat")
	case "$productCat" in
            "MOS_MASTER_SCREEN_FLAT")     suffix="0000";;
            "MOS_COMBINED_SCREEN_FLAT")   suffix="0001";;
            "PAF")                        suffix="0002";;
        esac
  ;;

  "vmspcaldisp")
	case "$productCat" in
            "EXTRACT_TABLE")              suffix="0000";;
            "MOS_ARC_SPECTRUM_EXTRACTED") suffix="0001";;
            "PAF")                        suffix="0002";;
        esac
  ;;

  "vmmosobsstare")
	case "$productCat" in
            "MOS_SCIENCE_REDUCED")        suffix="0000";;
            "MOS_SCIENCE_EXTRACTED")      suffix="0001";;
            "MOS_SCIENCE_SKY")            suffix="0002";;
            "OBJECT_TABLE")               suffix="0003";;
            "WINDOW_TABLE")               suffix="0004";;
        esac
  ;;


#
#  copy and paste the following block to add a new recipe
# =============================================================================
  "new_recipe_here")
        case "$productCat" in
            "NEW_PROCATG_1_HERE")         suffix="0000";;
            "NEW_PROCATG_2_HERE")         suffix="0001";;
            # etc...
        esac
  ;;
# =============================================================================
#

esac

#
#  if the suffix is set we can rename the product
#
if [ -n "$suffix" ]
then
    renameProduct
fi

exit $returnCode

#
#  end of script
#
