/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "jpeginput.h"
#include "exroutput.h"

static double voltage_to_intensity_srgb( double val )
{
    /* Handle invalid values before doing a gamma transform. */
    if( val < 0.0 ) return 0.0;
    if( val > 1.0 ) return 1.0;

    if( val <= 0.04045 ) {
        return val / 12.92;
    }
    return pow( ( val + 0.055 ) / 1.055, 2.4 );
}

int main( int argc, char **argv )
{
    jpeginput_t *jpegin;
    exroutput_t *exrout;
    int width, height;
    double *scanline;
    int y;

    if( argc < 3 ) {
        fprintf( stderr, "Usage: jpegtoexr input.jpg output.exr\n" );
        return 1;
    }

    jpegin = jpeginput_new( argv[ 1 ], 1 );
    if( !jpegin ) {
        fprintf( stderr, "jpegtoexr: Cannot open %s\n", argv[ 1 ] );
        return 1;
    }
    width = jpeginput_get_width( jpegin );
    height = jpeginput_get_height( jpegin );
    fprintf( stdout, "jpegtoexr: %s [%dx%d]\n", argv[ 1 ], width, height );

    scanline = malloc( width * 4 * sizeof( double ) );
    if( !scanline ) {
        fprintf( stderr, "jpegtoexr: Cannot allocate memory.\n" );
        jpeginput_delete( jpegin );
        return 1;
    }

    exrout = exroutput_new( argv[ 2 ], width, height );
    if( !exrout ) {
        fprintf( stderr, "jpegtoexr: Cannot open %s\n", argv[ 2 ] );
        jpeginput_delete( jpegin );
        free( scanline );
        return 1;
    }

    for( y = 0; y < height; y++ ) {
        uint8_t *input = jpeginput_get_scanline( jpegin, y );
        int x;

        fprintf( stderr, "jpegtoexr: Output scanline %d\r", y );
        for( x = 0; x < width; x++ ) {
            /* Yes, JFIF says 256 not 255. */
            double yp = ((double) input[ (x * 3) + 0 ]) / 256.0;
            double u = (((double) input[ (x * 3) + 1 ]) - 128.0) / 256.0;
            double v = (((double) input[ (x * 3) + 2 ]) - 128.0) / 256.0;
            double rp, gp, bp;
            double r, g, b;

            /* Derived using matlab, maybe I should find a better reference. */
            rp = yp - (0.0009 * u) + (1.4017 * v);
            gp = yp - (0.3437 * u) - (0.7142 * v);
            bp = yp + (1.7722 * u) + (0.0010 * v);

            r = voltage_to_intensity_srgb( rp );
            g = voltage_to_intensity_srgb( gp );
            b = voltage_to_intensity_srgb( bp );

            scanline[ (x * 4) + 0 ] = r;
            scanline[ (x * 4) + 1 ] = g;
            scanline[ (x * 4) + 2 ] = b;
            scanline[ (x * 4) + 3 ] = 1.0;
        }
        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\njpegtoexr: Done.\n" );
    exroutput_delete( exrout );
    jpeginput_delete( jpegin );
    free( scanline );
    return 0;
}

