------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2006 Dmitriy Anisimkov.                                   --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-containers.ads,v 1.15 2008/03/13 10:25:27 vagul Exp $

with Ada.Calendar;
with Ada.Containers.Vectors;
with Ada.Strings.Unbounded;
with OCI.Thick.Lobs;

package OCI.Thick.Containers is

   type Data_Holder is private;

   Null_Data : constant Data_Holder;

   type Container_Type is tagged private;

   subtype Data_Container is Container_Type;
   --  ??? Old name for some time backward compartibility.

   procedure Clear (Container : in out Container_Type);
   pragma Inline (Clear);
   --  Clear container.

   function Length (Container : in Container_Type) return Natural;
   pragma Inline (Length);
   --  Number of field sets in container.

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Integer;
      Position  : in     Positive;
      Iteration : in     Positive := 1);
   pragma Inline (Set);

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Long_Float;
      Position  : in     Positive;
      Iteration : in     Positive := 1);
   pragma Inline (Set);

   procedure Set
     (Container : in out Container_Type;
      Item      : in     String;
      Position  : in     Positive;
      Iteration : in     Positive := 1);
   pragma Inline (Set);

   procedure Set
     (Container : in out Container_Type;
      Item      : in     OCINumber;
      Position  : in     Positive;
      Iteration : in     Positive := 1);
   pragma Inline (Set);

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Ada.Calendar.Time;
      Position  : in     Positive;
      Iteration : in     Positive := 1);
   pragma Inline (Set);

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Lobs.Locator;
      Position  : in     Positive;
      Iteration : in     Positive := 1);
   pragma Inline (Set);

   procedure Set
     (Container : in out Container_Type;
      Item      : in     Data_Holder;
      Position  : in     Positive;
      Iteration : in     Positive := 1);

   procedure Clear
     (Container : in out Container_Type;
      Position  : in     Positive;
      Iteration : in     Positive := 1);
   pragma Inline (Clear);

   function Is_Null
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Boolean;

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Integer;
   pragma Inline (Get);

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Long_Float;
   pragma Inline (Get);

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return String;
   pragma Inline (Get);

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return OCINumber;
   pragma Inline (Get);

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Ada.Calendar.Time;
   pragma Inline (Get);

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Lobs.Locator;
   pragma Inline (Get);

   function Get
     (Container : in Container_Type;
      Position  : in Positive;
      Iteration : in Positive := 1) return Data_Holder;
   pragma Inline (Get);

   procedure Get
     (Container : in     Container_Type;
      Item      :    out Data_Holder;
      Position  : in     Positive;
      Iteration : in     Positive := 1);

   procedure Append
     (Container : in out Container_Type;
      Item      : in     String;
      Position  : in     Positive;
      Iteration : in     Positive := 1);

   procedure Append (Holder : in out Data_Holder; Item : in String);

   procedure C_Slice
     (Holder : in     Data_Holder;
      Low    : in     Positive;
      Item   :    out C.char_array;
      Last   :    out C.size_t;
      Done   :    out Boolean);

   function Is_Null (Item : in Data_Holder) return Boolean;

   function Value (Item : in Data_Holder) return Integer;
   pragma Inline (Value);

   function Value (Item : in Data_Holder) return String;
   pragma Inline (Value);

   function Value (Item : in Data_Holder) return Long_Float;
   pragma Inline (Value);

   function Value (Item : in Data_Holder) return OCINumber;
   pragma Inline (Value);

   function Value (Item : in Data_Holder) return Ada.Calendar.Time;
   pragma Inline (Value);

   function Value (Item : in Data_Holder) return Lobs.Locator;
   pragma Inline (Value);

   function Value (Item : in Data_Holder; Default : in Integer) return Integer;
   pragma Inline (Value);

   function Value (Item : in Data_Holder; Default : in String) return String;
   pragma Inline (Value);

   function Value
     (Item : in Data_Holder; Default : in Long_Float) return Long_Float;
   pragma Inline (Value);

   function Value
     (Item : in Data_Holder; Default : in OCINumber) return OCINumber;
   pragma Inline (Value);

   function Value
     (Item : in Data_Holder; Default : in Ada.Calendar.Time)
      return Ada.Calendar.Time;
   pragma Inline (Value);

   function To_Data (Item : Integer) return Data_Holder;
   pragma Inline (To_Data);

   function To_Data (Item : String) return Data_Holder;
   pragma Inline (To_Data);

   function To_Data (Item : C.char_array) return Data_Holder;
   pragma Inline (To_Data);

   function To_Data (Item : OCINumber) return Data_Holder;
   pragma Inline (To_Data);

   function To_Data (Item : Long_Float) return Data_Holder;
   pragma Inline (To_Data);

   function To_Data (Item : Ada.Calendar.Time) return Data_Holder;
   pragma Inline (To_Data);

   function To_Data (Item : Lobs.Locator) return Data_Holder;
   pragma Inline (To_Data);

   function Image
     (Item : in Data_Holder; Null_Image : in String := "") return String;

private

   type Data_Type is
     (Type_Null,
      Type_Integer,
      Type_Long_Float,
      Type_String,
      Type_Number,
      Type_Date,
      Type_Lob);

   type Data_Holder (Kind : Data_Type := Type_Null) is record
      case Kind is
      when Type_Number     => Numb : OCINumber;
      when Type_Integer    => Int  : Integer;
      when Type_Long_Float => Flt  : Long_Float;
      when Type_Date       => Dat  : Ada.Calendar.Time;
      when Type_String     => Str  : Ada.Strings.Unbounded.Unbounded_String;
      when Type_Lob        => Loc  : Lobs.Locator;
      when Type_Null       => null;
      end case;
   end record;

   Null_Data : constant Data_Holder := (Kind => Type_Null);

   package Positions is new Ada.Containers.Vectors
     (Index_Type => Positive, Element_Type => Data_Holder);

   package Data_Vectors is new Ada.Containers.Vectors
     (Index_Type   => Positive,
      Element_Type => Positions.Vector,
      "="          => Positions."=");

   type Container_Type is tagged record
      Data : Data_Vectors.Vector;
   end record;

end OCI.Thick.Containers;
