/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ishell.h"


#include "icontrolmodule.h"
#include "icontrolscript.h"
#include "idirectory.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iimagefactory.h"
#include "ioutputchannel.h"
#include "iparallelmanager.h"
#include "ishellfactory.h"
#include "iversion.h"
#include "iviewmodule.h"
#include "ivolumeviewsubject.h"

#include <vtkRenderWindow.h>
#include <vtkTimerLog.h>

#ifdef I_OFFSCREEN
#include <vtkGraphicsFactory.h>
#include <vtkImagingFactory.h>
#endif

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


const iString iShell::mOptionPrefix = "-";


namespace iShell_Private
{
	void CheckSystem();
	void TestPerformance();

	class BatchModeObserver : public iScriptObserver
	{

	public:

		BatchModeObserver(iScript *s) : iScriptObserver(s)
		{
		}

	protected:

		virtual void OnScriptStartBody(){}
		virtual void OnScriptStopBody(const iString &error){}

		virtual void OnScriptBeginLineBody(int line, const iString &text)
		{
			iConsole::Display(iConsole::_Info,"Executing line #"+iString::FromNumber(line)+": "+text);
		}

		virtual void OnScriptEndLineBody(int , const iString &){}
		virtual bool OnScriptCheckAbortBody(int cur, int num, int level)
		{
			if(cur>-1 && num>0)
			{
				iConsole::Display(iConsole::_Info,"Loop: level #"+iString::FromNumber(level)+", iteration "+iString::FromNumber(cur)+" out of "+iString::FromNumber(num));
			}
			return false;
		}
	};
};


using namespace iShell_Private;


void iShell::RunApplication(const iString &t, int argc, char **argv)
{
#ifdef I_DEBUG
//	iHelpFactory::CreateUserGuide(iHelpFactory::_Publish); return;
#endif
	iShell *tmp = iShell::New(t,argc,argv);
	if(tmp != 0)
	{
		tmp->Run();
		tmp->Delete();
	}
}


iShell::iShell(const iString &type, int, char **) : mType(type)
{
	int i;

	mRunning = false;
	mControlModule = 0; // control module must be created after the constructor is finished
	mNumProcs = 1;

	mCurInitStep = 0;
	for(i=0; i<4; i++)
	{
		mInitSteps[i].Current = 0;
	}
	mInitSteps[0].Total = 10;		// needs to be set manually during development. How to do it automatically?
	mInitSteps[1].Total = 161;		// needs to be set manually during development. How to do it automatically?
	mInitSteps[2].Total = 1382;		// needs to be set manually during development. How to do it automatically?
	mInitSteps[3].Total = 13;		// needs to be set manually during development. How to do it automatically?

	//
	//  Define some basic command-line options
	//
	this->AddCommandLineOption("h",false,"show this help");
	this->AddCommandLineOption("help",false,"show this help");
	this->AddCommandLineOption("-help",false,"show this help");
	this->AddCommandLineOption("i",true,"load the state from a file with name <arg>");
	this->AddCommandLineOption("b",true,"execute a control script from a file <arg> in the screenless mode");
	this->AddCommandLineOption("np",true,"set the number of processors to use to <arg>");
	this->AddCommandLineOption("test",false,"test performance and exit (must be the first and only option)");
	this->AddCommandLineOption("GPU",false,"try to use GPU even if VTK does not find it");
	this->AddCommandLineOption("no-GPU",false,"do not use GPU even if it is present on the system (useful for running IFrIT remotely)");

	//
	//  Set environment variables
	//  Is the base-dir environment variable set?
	//
	char *e = getenv("IFRIT_DIR");
	if(e != 0)
	{
		mEnvBaseDir = iString(e);
		if(mEnvBaseDir.IsEmpty()) mEnvBaseDir = iDirectory::Current();
		if(!mEnvBaseDir.EndsWith(iDirectory::Separator())) mEnvBaseDir += iDirectory::Separator();
	}
	else
	{
		//
		//  Get the home directory, if it exists
		//
		e = getenv("APPDATA"); // windows?
		if(e != 0)
		{
			mEnvBaseDir = iString(e) + iDirectory::Separator() + "IfrIT" + iDirectory::Separator();
		}
		else // unix?
		{
			e = getenv("HOME");
			mEnvBaseDir = iString(e) + iDirectory::Separator() + ".ifrit" + iDirectory::Separator();
		}
		//
		//  Is it readable?
		//
		iDirectory dir;
		if(!dir.Open(mEnvBaseDir))
		{
			//
			//  try to create
			//
			if(!dir.Make(mEnvBaseDir))
			{
				mEnvBaseDir = iDirectory::Current();
			}
		}
	}

	//    
	//  Read other environment here    
	//    
	e = getenv("IFRIT_DATA_DIR");               if(e != 0) mEnvDataDir = iString(e) + iDirectory::Separator(); else mEnvDataDir = iDirectory::Current();
	e = getenv("IFRIT_IMAGE_DIR");              if(e != 0) mEnvImageDir = iString(e) + iDirectory::Separator(); else mEnvImageDir = iDirectory::Current();
	e = getenv("IFRIT_SCRIPT_DIR");             if(e != 0) mEnvScriptDir = iString(e) + iDirectory::Separator(); else mEnvScriptDir = mEnvBaseDir;
	e = getenv("IFRIT_PALETTE_DIR");            if(e != 0) mEnvPaletteDir = iString(e) + iDirectory::Separator(); else mEnvPaletteDir = mEnvBaseDir;

	iDirectory::ExpandFileName(mEnvBaseDir);
	iDirectory::ExpandFileName(mEnvDataDir);
	iDirectory::ExpandFileName(mEnvImageDir);
	iDirectory::ExpandFileName(mEnvScriptDir);
	iDirectory::ExpandFileName(mEnvPaletteDir);

	mInitTimer = vtkTimerLog::New(); IERROR_ASSERT(mInitTimer);
	mInitTimer->StartTimer();

#ifdef I_OFFSCREEN
	//
	// Configure Graphics Factory
	//
	vtkGraphicsFactory *graphics_factory = vtkGraphicsFactory::New(); IERROR_ASSERT(graphics_factory);
	graphics_factory->SetOffScreenOnlyMode(1);
	graphics_factory->SetUseMesaClasses(1);
	//
	// Configure Imaging Factory
	vtkImagingFactory *imaging_factory = vtkImagingFactory::New(); IERROR_ASSERT(imaging_factory);
	imaging_factory->SetUseMesaClasses(1);
#endif
}


iShell::~iShell()
{
	//
	//  This ensures that control module is deleted after the shell is deleted.
	//
	//	mControlModule->Delete();
	mInitTimer->Delete();
}


iShell* iShell::New(const iString &t, int argc, char **argv)
{
	//
	//  Check that sizes of basic types are correct
	//
	CheckSystem();
	//
	//  Check if a test call
	//
	if(argc==2 && iString(argv[1])==(mOptionPrefix+"test"))
	{
		TestPerformance();
		return 0;
	}
	//
	//  Also use options to specify shell - they overwrite the default choice
	//
	iString st = t;
	int firstOption = 1;
	if(argc>1 && iString(argv[1]).Part(0,mOptionPrefix.Length())==mOptionPrefix)
	{
		iString list, help, s = iString(argv[1]).Part(mOptionPrefix.Length());
		iShellFactory::GetSupportedShells(list,help);
		if(list.Contains(s+"/") == 1)
		{
			st = s;
			firstOption = 2;
		}
	}
	//
	//  If there is no default and no option, ask ShellFactory to query the user
	//
	if(st.IsEmpty())
	{
		iConsole::Display(iConsole::_FatalError,"No shell has been specified. Use a \"-<shell>\" option to specify a two-letter shell abbreviation <shell>.");
		return 0;
	}
	//
	//  Create shell
	//
	iShell *tmp = iShellFactory::CreateShell(st,argc,argv);
	if(tmp == 0) 
	{
		iConsole::Display(iConsole::_FatalError,"Shell '"+st+"' is not included in this installation.");
		return 0;
	}
	tmp->mFirstOption = firstOption;
	//
	// Make sure Image factory is up to date
	//
	iImageFactory::FindIcon("ok.png");
	//
	//  Parse command line options
	//
	tmp->DefineSpecificCommandLineOptions();
	tmp->ParseCommandLineOptions(argc,argv);

#ifdef I_DEBUG
	//tmp->mNumProcs = 2;
#endif

	if(tmp->mScriptFileName.IsEmpty())
	{
		tmp->PrepareForConstruction();
	}

#ifdef I_DEBUG
	tmp->InitHelper();
#endif
	//
	//  This ensures that "tmp->this" pointer is correct before a control module is created.
	//
	tmp->mCurInitStep = 1;
	tmp->mControlModule = iControlModule::New(tmp); IERROR_ASSERT(tmp->mControlModule);
	if(tmp->mNumProcs >= 0) tmp->mControlModule->GetParallelManager()->SetNumberOfProcessors(tmp->mNumProcs);
	//
	//  Set the state file name
	//
	if(tmp->mStateFileName.IsEmpty())
	{
		iString fname = tmp->GetEnvironment(Environment::Base) + "ifrit.ini";
		if(iFile::IsReadable(fname)) tmp->mStateFileName = fname;
	}

#ifdef I_DEBUG
	tmp->InitHelper();
#endif
	//
	//  Really construct the shell: control module needs to be fully initialized before a shell 
	//  is really constructed.
	//
	tmp->mCurInitStep = 2;
	if(tmp->mScriptFileName.IsEmpty())
	{
		tmp->ConstructorBody(); 
	}
	//
	//   Load widget info into ObjectKeyHelp objects.
	//
#ifdef I_DEBUG
	tmp->InitHelper();
#endif
	//
	//  Load the state file if needed
	//
	tmp->mCurInitStep = 3;
	if(!tmp->mStateFileName.IsEmpty())
	{
		if(tmp->mScriptFileName.IsEmpty())
		{
			tmp->PrepareToLoadStateFile(); 
		}
		if(!tmp->GetControlModule()->LoadStateFromFile(tmp->mStateFileName))
		{
			iConsole::Display(iConsole::_HighError,"Unable to load the state file.\n Default values of parameters will be used.");
		}
	}
#ifdef I_DEBUG
	tmp->InitHelper();
#endif

	if(tmp->mScriptFileName.IsEmpty())
	{
#ifdef I_OFFSCREEN
		//
		//  In the off-screen mode we need the script
		//
		iConsole::Display(iConsole::_FatalError,"A batch script filename is required in the off-screen-only mode.");
		return 0;
#else
		//
		//  Start the shell
		//
		tmp->Start();
		return tmp;
#endif
	}
	else
	{
		//
		//  Execute script and exit
		//
		int i, n = tmp->mControlModule->GetNumberOfViewModules();
		for(i=0; i<n; i++)
		{
			tmp->mControlModule->GetViewModule(i)->GetRenderWindow()->SetOffScreenRendering(1);
		}

		if(tmp->mControlModule->GetViewModule(0)->GetRenderWindow()->GetOffScreenRendering() == 0)
		{
			iConsole::Display(iConsole::_FatalError,"This platform+shell combination does not support off-screen rendering.\n Off-screen rendering is needed for executing a script in a batch mode.");
			return 0;
		}
		//
		//  Execute the batch script
		//
		iFile F(tmp->mScriptFileName);

		if(!F.Open(iFile::_Read,iFile::_Text)) 
		{
			iConsole::Display(iConsole::_FatalError,iString("File ")+tmp->mScriptFileName+" cannot be open for reading.");
			return 0;
		}

		iConsole::Display(iConsole::_Info,"Reading the file...");
		iString text, line;
		while(F.ReadLine(line)) text += "\n" + line;
		text += "\n";

		iControlScript *cs = tmp->mControlModule->GetControlScript();
		iConsole::Display(iConsole::_Info,"Compiling the script...");

		cs->SetText(text);
		if(!cs->Compile())
		{
			iConsole::Display(iConsole::_FatalError,iString("Syntax error in script, line ")+iString::FromNumber(cs->GetThisLine())+" : "+cs->GetErrorStatus()->Message());
			return 0;
		}

		cs->SetAutoRender(false);

		iConsole::Display(iConsole::_Info,"Running the script...");
		BatchModeObserver *obs = new BatchModeObserver(cs);
		cs->Execute();
		if(obs != 0) delete obs;
		if(cs->GetErrorStatus()->IsError())
		{
			iConsole::Display(iConsole::_HighError,iString("Runtime error in script, line ")+iString::FromNumber(cs->GetThisLine())+" : "+cs->GetErrorStatus()->Message());
		}
		else iConsole::Display(iConsole::_Info,"Done.");

		tmp->Delete();
		return 0;
	}
}


void iShell::Delete()
{
	//
	//  We are just about to start deleting objects. First thing we do is to block all
	//  observers since we do not know at each moment in the deleting process which objects
	//  are still present and which have been already deleted.
	//
	iEventObserver::BlockAllEventObservers(true);

	//
	//  Check whether the error log was created by iConsole:
	//
	iOutputChannel::GetInstance()->NotifyIfLogCreated();

	//
	//  Go on rampage!
	//
	if(mScriptFileName.IsEmpty())
	{
		this->DestructorBody();
	}

	//
	//  This ensures that control module is deleted before the shell is deleted.
	//
	mControlModule->Delete();

	this->vtkObjectBase::Delete();
}


void iShell::Run()
{
	mRunning = true;
	this->RunBody();
	mRunning = false;
}


void iShell::IgnoreCommandLineArgument(const iString &arg)
{
	mIgnoredCommandLineArguments.Add(arg);
}


void iShell::AddCommandLineOption(const iString &root, bool hasValue, const iString &help)
{
	Option tmp;
	tmp.Name = root;
	tmp.NeedsValue = hasValue;
	tmp.Help = help;
	mOptions.Add(tmp);
}


void iShell::ParseCommandLineOptions(int argc, char **argv)
{
	int i, j;
	Option o;
	bool done;

	i = mFirstOption;
	while(i < argc)
	{
		//
		//  Check if this word needs to be ignored
		//
		if(mIgnoredCommandLineArguments.Find(argv[i]) > -1) break;

		done = false;
		if(iString(argv[i]).Part(0,mOptionPrefix.Length()) == mOptionPrefix)
		{
			o.Name = argv[i] + mOptionPrefix.Length();
			for(j=0; !done && j<mOptions.Size(); j++) if(o.Name == mOptions[j].Name)
			{
				if(mOptions[j].NeedsValue)
				{
					i++;
					if(i < argc) o.Value = argv[i]; else
					{
						iConsole::Display(iConsole::_HighError,"Option "+o.Name+" requires an argument.\nIFrIT will now exit.");
						exit(1);
					}
				}
				if(!AnalyseOneCommandLineOption(o))
				{
					iConsole::Display(iConsole::_HighError,"Error has occured in analysing option "+o.Name+"\nIFrIT will now exit.");
					exit(1);
				}
				else done = true;
			}
			if(done) i++;
		}

		//
		//  Not an option
		//
		if(!done && i==argc-1)
		{
			//
			//  This must be the name of the default data directory
			//
			iDirectory d;
			if(d.Open(argv[i]))
			{
				d.Close();
				mEnvDataDir = argv[i];
				i++;
				done = true;
			}
		}

		//
		//  Unrecognizable argument
		//
		if(!done)
		{
			iConsole::Display(iConsole::_HighError,"Command-line argument "+iString(argv[i])+" is not supported.\nIFrIT will now exit.");
			exit(1);
		}
	}
}


bool iShell::AnalyseOneCommandLineOption(const struct Option &o)
{
	if(o.Name=="h" || o.Name=="help" || o.Name=="-help")
	{
		iString sl, sh, s = "Format: ifrit [shell] [option] ... [data_dir_name]\n";
		s += "Shell: -<id>,  where a two-letter shell id is one of the following:\n";
		iShellFactory::GetSupportedShells(sl,sh);
		int i, c = sl.Contains('/');
		for(i=0; i<c; i++)
		{
			s += "\t" + sl.Section("/",i,i) + "\t" + sh.Section("/",i,i) + "\n";
		}
		s += "Options:\n";
		for(i=0; i<mOptions.Size(); i++)
		{
			s += "\t-" + mOptions[i].Name + " ";
			if(mOptions[i].NeedsValue) s += "<arg> "; else s += "      ";
			s += "\t" + mOptions[i].Help + "\n";
		}
		iConsole::Display(iConsole::_Info,s);
		exit(0);
	}

	if(o.Name == "np")
	{
		bool ok;
		int n = o.Value.ToInt(ok);
		if(!ok || n<0 || n>16384) return false;
		mNumProcs = n;
		return true;
	}

	if(o.Name == "i")
	{
		iString fn(o.Value);
		iDirectory::ExpandFileName(fn,this->GetEnvironment(Environment::Base));
		if(iFile::IsReadable(fn))
		{
			mStateFileName = fn;
			return true;
		}
		else
		{
			iConsole::Display(iConsole::_HighError,"File "+fn+" is not accessible.\nIFrIT will now exit.");
			exit(1);
		}
	}

	if(o.Name == "b")
	{
		iString fn (o.Value);
		iDirectory::ExpandFileName(fn,this->GetEnvironment(Environment::Base));
		if(iFile::IsReadable(fn))
		{
			mScriptFileName = fn;
			return true;
		}
		else
		{
			iConsole::Display(iConsole::_HighError,"File "+fn+" is not accessible.\nIFrIT will now exit.");
			exit(1);
		}
	}

	if(o.Name == "GPU")
	{
		iVolumeViewSubject::SetGPUMode(1);
	}

	if(o.Name == "no-GPU")
	{
		iVolumeViewSubject::SetGPUMode(-1);
	}

	return this->AnalyseOneExtendedCommandLineOption(o);
}


//
//  Environment querying
//
const iString& iShell::GetEnvironment(int key) const
{
	static const iString none;

	switch(key)
	{
	case Environment::Base:
		{
			return mEnvBaseDir;
		}
	case Environment::Data:
		{
			return mEnvDataDir;
		}
	case Environment::Image:
		{
			return mEnvImageDir;
		}
	case Environment::Script:
		{
			return mEnvScriptDir;
		}
	case Environment::Palette:
		{
			return mEnvPaletteDir;
		}
	default:
		{
			return none;
		}
	}
}


bool iShell::HasStateFile() const
{
	return !mStateFileName.IsEmpty();
}


bool iShell::LoadShellStateFromFile(iFile &F)
{
	if(mScriptFileName.IsEmpty())
	{
		return this->LoadShellStateFromFileBody(F);
	}
	else return true;
}


bool iShell::SaveShellStateToFile(iFile &F) const
{
	if(mScriptFileName.IsEmpty())
	{
		return this->SaveShellStateToFileBody(F);
	}
	else return true;
}


void iShell::OnInitAtom()
{
	mInitSteps[mCurInitStep].Current++;
	mInitTimer->StopTimer();
	if(mInitTimer->GetElapsedTime() > 0.1)
	{
		mInitTimer->StartTimer();
		this->OnInitAtomBody(true);
	}
	else
	{
		this->OnInitAtomBody(false);
	}
}


void iShell::OnInitAtomBody(bool)
{
}

	
void iShell::OnLoadStateFileAtom(int prog)
{
	this->OnLoadStateFileAtomBody(prog);
}


void iShell::OnLoadStateFileAtomBody(int)
{
}


#ifdef I_DEBUG

void iShell::InitHelper()
{
	if(mInitSteps[mCurInitStep].Current != mInitSteps[mCurInitStep].Total)
	{
		int step = mCurInitStep;
		int total = mInitSteps[mCurInitStep].Current;
		int bp = 0;
	}
}

#endif


#include <vtkActor.h>
#include <vtkCamera.h>
#include <vtkCubeSource.h>
#include <vtkGlyph3D.h>
#include <vtkImageData.h>
#include <vtkLookupTable.h>
#include <vtkPointData.h>
#include <vtkPolyDataMapper.h>
#include <vtkProperty.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>

namespace iShell_Private
{
	void CheckSystem()
	{
		//
		//  Check that sizes of basic types are correct
		//
		if(sizeof(int)!=4 || sizeof(float)!=4 || sizeof(double)!=8)
		{
			int sint = sizeof(int);
			int sflt = sizeof(float);
			int sdbl = sizeof(double);
			int sptr = sizeof(void*);
			int slng = sizeof(long);
			int svtk = sizeof(vtkIdType);
			int ssiz = sizeof(size_t);
			iConsole::Display(iConsole::_FatalError,"IFrIT has not been ported to this machine.");
			exit(0);
		}
		//
		//  Check that vtkIdType can address a full pointer
		//
		if(sizeof(vtkIdType) < sizeof(void*))
		{
			iConsole::Display(iConsole::_Notification,"This machine is "+iString::FromNumber(8*int(sizeof(void*)))+"-bit, but VTK has been compiled with "+iString::FromNumber(8*int(sizeof(vtkIdType)))+"-bit ids.\n"
				"IFrIT will not be able to use more than 2GB of memory per single array.\n"
				"Just letting you know.\n");
		}
		if(sizeof(vtkIdType) > sizeof(void*))
		{
			iConsole::Display(iConsole::_FatalError,"This machine is "+iString::FromNumber(8*int(sizeof(void*)))+"-bit, but VTK has been compiled with "+iString::FromNumber(8*int(sizeof(vtkIdType)))+"-bit ids.\n"
				"This configuration is inconsistent and is prone to crashes.\n"
				"Please recompile VTK with the advanced option VTK_USE_64BIT_IDS set to OFF.\n");
			exit(0);
		}
	}

	float TestRenderingPerformance(vtkRenderer *ren)
	{
		//
		// Set up the timer
		//
		ren->GetActiveCamera()->SetPosition(0.0,0.0,1.0);
		ren->ResetCamera();

		ren->GetRenderWindow()->Render();

		vtkTimerLog *timer = vtkTimerLog::New();
		timer->StartTimer();

		int i;
		float r = 0;
		do
		{
			for(i=0; i<35; i++)
			{
				ren->GetActiveCamera()->Azimuth(9);
				ren->GetRenderWindow()->Render();
			}
			for(i=0; i<35; i++)
			{
				ren->GetActiveCamera()->Elevation(9);
				ren->GetActiveCamera()->OrthogonalizeViewUp();
				ren->GetRenderWindow()->Render();
			}
			for(i=0; i<40; i++)
			{
				ren->GetActiveCamera()->Roll(9);
				ren->GetRenderWindow()->Render();
			}
			for(i=0; i<5; i++)
			{
				ren->GetActiveCamera()->Elevation(9);
				ren->GetActiveCamera()->OrthogonalizeViewUp();
				ren->GetRenderWindow()->Render();
			}
			for(i=0; i<5; i++)
			{
				ren->GetActiveCamera()->Azimuth(9);
				ren->GetRenderWindow()->Render();
			}
			timer->StopTimer();
			r += 1;
		}
		while(timer->GetElapsedTime() < 3.0);

		r /= timer->GetElapsedTime();

		timer->Delete();

		return r;
	}


	void TestPerformance()
	{
		int i;
		const float renRateScale[2] = { 7.5f, 7.5f };

		vtkRenderer *ren = vtkRenderer::New();
		vtkRenderWindow *win = vtkRenderWindow::New();
		win->AddRenderer(ren);
		ren->Delete();

		vtkImageData *data = vtkImageData::New();
		data->SetDimensions(17,17,17);
		data->SetSpacing(0.125,0.125,0.125);
		data->SetOrigin(-1.0,-1.0,-1.0);
		data->SetScalarTypeToUnsignedChar();
		data->SetNumberOfScalarComponents(1);
		data->AllocateScalars();

		unsigned char *ptr = (unsigned char *)data->GetScalarPointer();
		for(i=0; i<data->GetNumberOfPoints(); i++)
		{
			ptr[i] = i % 256;
		}

		vtkCubeSource *cube = vtkCubeSource::New();
		cube->Update();

		vtkGlyph3D *glyph = vtkGlyph3D::New();
		glyph->SetScaleModeToDataScalingOff();
		glyph->SetColorModeToColorByScalar();
		glyph->SetScaleFactor(0.08);
		glyph->SetSource(cube->GetOutput());
		glyph->SetInput(data);
		cube->Delete();
		glyph->Update();

		vtkPolyDataMapper *mapper = vtkPolyDataMapper::New();
		vtkActor *actor = vtkActor::New();

		mapper->ScalarVisibilityOn();
		mapper->SetColorModeToMapScalars();

		vtkLookupTable *lut = vtkLookupTable::New();
		lut->SetRange(0.0,255.0);
		//lut->SetHueRange(0.0,255.0);
		mapper->SetLookupTable(lut);
		mapper->UseLookupTableScalarRangeOn();
		lut->Delete();

		actor->GetProperty()->SetColor(0.0,0.0,0.0);
		actor->SetMapper(mapper);
		mapper->Delete();

		mapper->SetInput(glyph->GetOutput());
		glyph->Delete();

		ren->AddActor(actor);
		actor->Delete();

		//
		// set the scene
		//
		win->SetSize(512,512);
		ren->SetBackground(1.0,1.0,1.0);

		//
		// draw the resulting scene
		//
		win->SetWindowName("IFrIT - Performance Test");
		win->Render();
		win->SetWindowName("IFrIT - Performance Test");

		iString ws = "Rendering performance:\n\n";
		float rr;

		actor->GetProperty()->SetOpacity(1.0);
		actor->GetProperty()->SetColor(0.0,0.0,1.0);
		rr = TestRenderingPerformance(ren)*renRateScale[0];
		ws += "Opaque:\t\t " + iString::FromNumber(rr,"%.1f") + "\n";

		actor->GetProperty()->SetOpacity(0.03);
		actor->GetProperty()->SetColor(1.0,0.0,0.0);
		rr = TestRenderingPerformance(ren)*renRateScale[1];
		ws += "Transparent:\t " + iString::FromNumber(rr,"%.1f") + "\n";

		ws += "\n";
		ws += "Representative rangers for values:\n";
		ws += "1-2\tLaptop with a chipset\n";
		ws += "5-20\tDesktop with a low-end video card\n";
		ws += "50-200\tWorkstation with a high-end video card\n\n";

		ws += iString("VTK Version: ") + VTK_VERSION + "\n";
		ws += "IFrIT Version: " + iVersion::GetVersion() + "\n";

#ifdef _WIN32
		iConsole::Display(iConsole::_Notification,ws);
#else
		iConsole::Display(iConsole::_Info,ws);
#endif

		data->Delete();
		win->RemoveRenderer(ren);
		win->Delete();
	}
};
