/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.io;

import static org.testng.Assert.assertFalse;
import static org.testng.Assert.assertNull;
import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertNotNull;
import static org.testng.AssertJUnit.assertTrue;

import java.awt.Color;
import java.io.File;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import jalview.datamodel.AlignmentAnnotation;
import jalview.datamodel.AlignmentI;
import jalview.datamodel.HiddenColumns;
import jalview.datamodel.SequenceGroup;
import jalview.gui.AlignFrame;
import jalview.gui.JvOptionPane;
import jalview.io.AnnotationFile.ViewDef;

public class AnnotationFileIOTest
{

  @BeforeClass(alwaysRun = true)
  public void setUpJvOptionPane()
  {
    JvOptionPane.setInteractiveMode(false);
    JvOptionPane.setMockResponse(JvOptionPane.CANCEL_OPTION);
  }

  static String TestFiles[][] = { { "Test example annotation import/export",
      "examples/uniref50.fa", "examples/testdata/example_annot_file.jva" },
      { "Test multiple combine annotation statements import/export",
          "examples/uniref50.fa",
          "examples/testdata/test_combine_annot.jva" },
      { "Test multiple combine annotation statements with sequence_ref import/export",
          "examples/uniref50.fa", "examples/testdata/uniref50_iupred.jva" },
      { "Test group only annotation file parsing results in parser indicating annotation was parsed",
          "examples/uniref50.fa", "examples/testdata/test_grpannot.jva" },
      { "Test hiding/showing of insertions on sequence_ref",
          "examples/uniref50.fa",
          "examples/testdata/uniref50_seqref.jva" },
      { "Test example annotation with calcid import/export",
            "examples/uniref50.fa", "examples/testdata/example_annot_file_calcid.jva" },
      { "Test example annotation with annotation property import/export",
              "examples/uniref50.fa", "examples/testdata/example_annot_file_property.jva" }
  };

  @Test(groups = { "Functional" })
  public void exampleAnnotationFileIO() throws Exception
  {
    for (String[] testPair : TestFiles)
    {
      testAnnotationFileIO(testPair[0], new File(testPair[1]),
              new File(testPair[2]));
    }
  }

  protected AlignmentI readAlignmentFile(File f)
  {
    System.out.println("Reading file: " + f);
    String ff = f.getPath();
    try
    {
      FormatAdapter rf = new FormatAdapter();

      AlignmentI al = rf.readFile(ff, DataSourceType.FILE,
              new IdentifyFile().identify(ff, DataSourceType.FILE));

      // make sure dataset is initialised ? not sure about this
      for (int i = 0; i < al.getSequencesArray().length; ++i)
      {
        al.getSequenceAt(i).createDatasetSequence();
      }
      assertNotNull("Couldn't read supplied alignment data.", al);
      return al;
    } catch (Exception e)
    {
      e.printStackTrace();
    }
    Assert.fail(
            "Couln't read the alignment in file '" + f.toString() + "'");
    return null;
  }

  /**
   * test alignment data in given file can be imported, exported and reimported
   * with no dataloss
   * 
   * @param f
   *          - source datafile (IdentifyFile.identify() should work with it)
   * @param ioformat
   *          - label for IO class used to write and read back in the data from
   *          f
   */
  void testAnnotationFileIO(String testname, File f, File annotFile)
  {
    System.out.println("Test: " + testname + "\nReading annotation file '"
            + annotFile + "' onto : " + f);
    String af = annotFile.getPath();
    try
    {
      AlignmentI al = readAlignmentFile(f);
      HiddenColumns cs = new HiddenColumns();
      assertTrue("Test " + testname
              + "\nAlignment was not annotated - annotation file not imported.",
              new AnnotationFile().readAnnotationFile(al, cs, af,
                      DataSourceType.FILE));

      AnnotationFile aff = new AnnotationFile();
      // ViewDef is not used by Jalview
      ViewDef v = aff.new ViewDef(null, al.getHiddenSequences(), cs,
              new Hashtable());
      String anfileout = new AnnotationFile().printAnnotations(
              al.getAlignmentAnnotation(), al.getGroups(),
              al.getProperties(), null, al, v);
      assertTrue("Test " + testname
              + "\nAlignment annotation file was not regenerated. Null string",
              anfileout != null);
      assertTrue("Test " + testname
              + "\nAlignment annotation file was not regenerated. Empty string",
              anfileout.length() > "JALVIEW_ANNOTATION".length());

      System.out.println(
              "Output annotation file:\n" + anfileout + "\n<<EOF\n");

      AlignmentI al_new = readAlignmentFile(f);
      assertTrue("Test " + testname
              + "\nregenerated annotation file did not annotate alignment.",
              new AnnotationFile().readAnnotationFile(al_new, anfileout,
                      DataSourceType.PASTE));

      // test for consistency in io
      StockholmFileTest.testAlignmentEquivalence(al, al_new, false, false,
              false);
      return;
    } catch (Exception e)
    {
      e.printStackTrace();
    }
    Assert.fail("Test " + testname
            + "\nCouldn't complete Annotation file roundtrip input/output/input test for '"
            + annotFile + "'.");
  }

  @Test(groups = "Functional")
  public void testAnnotateAlignmentView()
  {
    long t1 = System.currentTimeMillis();
    /*
     * JAL-3779 test multiple groups of the same name get annotated
     */
    AlignFrame af = new FileLoader().LoadFileWaitTillLoaded(
            ">Seq1\nQRSIL\n>Seq2\nFTHND\n>Seq3\nRPVSL\n",
            DataSourceType.PASTE);
    long t2 = System.currentTimeMillis();
    System.err.println("t0: " + (t2 - t1));
    // seq1 and seq3 are in distinct groups both named Group1
    String annotationFile = "JALVIEW_ANNOTATION\nSEQUENCE_GROUP\tGroup1\t*\t*\t1\n"
            + "SEQUENCE_GROUP\tGroup2\t*\t*\t2\n"
            + "SEQUENCE_GROUP\tGroup1\t*\t*\t3\n"
            + "PROPERTIES\tGroup1\toutlineColour=blue\tidColour=red\n";
    new AnnotationFile().annotateAlignmentView(af.getViewport(),
            annotationFile, DataSourceType.PASTE);

    AlignmentI al = af.getViewport().getAlignment();
    List<SequenceGroup> groups = al.getGroups();
    assertEquals(3, groups.size());
    SequenceGroup sg = groups.get(0);
    assertEquals("Group1", sg.getName());
    assertTrue(sg.contains(al.getSequenceAt(0)));
    assertEquals(Color.BLUE, sg.getOutlineColour());
    assertEquals(Color.RED, sg.getIdColour());
    sg = groups.get(1);
    assertEquals("Group2", sg.getName());
    assertTrue(sg.contains(al.getSequenceAt(1)));

    /*
     * the bug fix: a second group of the same name is also given properties
     */
    sg = groups.get(2);
    assertEquals("Group1", sg.getName());
    assertTrue(sg.contains(al.getSequenceAt(2)));
    assertEquals(Color.BLUE, sg.getOutlineColour());
    assertEquals(Color.RED, sg.getIdColour());
  }

  @Test(groups = "Functional")
  public void testCalcIdRestore()
  {
    long t1 = System.currentTimeMillis();
    /*
     * JAL-3779 test multiple groups of the same name get annotated
     */
    AlignFrame af = new FileLoader().LoadFileWaitTillLoaded(
            ">Seq1\nQRSIL\n>Seq2\nFTHND\n>Seq3\nRPVSL\n",
            DataSourceType.PASTE);
    long t2 = System.currentTimeMillis();
    System.err.println("t0: " + (t2 - t1));
    // seq1 and seq3 are in distinct groups both named Group1
    String annotationFile = "JALVIEW_ANNOTATION\nCALCID\tmyCalcId\tNO_GRAPH\tA Label\tfoolabel\tt|t|t|t\n"
            + "CALCID\tmyCalcId\tROWPROPERTIES\tA Label\tspecial=property\n"
            + "NO_GRAPH\tWithout CALCID 1\tfoolabel\tt|t|t|t\n"
            + "ROWPROPERTIES\tWithout CALCID 1\tspecial=property\n"
            + "CALCID\tNO_GRAPH\tWithout CALCID 2\tfoolabel\tt|t|t|t\n"
            + "CALCID\tROWPROPERTIES\tWithout CALCID 2\tspecial=property\n"
            + "CALCID\tmyCalcId2\tNO_GRAPH\tA Label\tfoolabel\tt|t|t|t\n"
            + "CALCID\tmyCalcId2\tROWPROPERTIES\tA Label\tspecial=property\n"
            + "SEQUENCE_REF\tSeq3\n"
            + "CALCID\tmyCalcId\tNO_GRAPH\tA Label\tfoolabel\tt|t|t|t\n"
            + "CALCID\tmyCalcId\tROWPROPERTIES\tA Label\tspecial=seqproperty\n"
            + "SEQUENCE_REF\t\n"
            + "BAR_GRAPH\tSeqTrack\tseqs 4\tt|t|t|t\tNO_OF_SEQUENCES=4\tNO_OF_TRACKS=1\n"
            ;
    
    
    assertTrue(new AnnotationFile().annotateAlignmentView(af.getViewport(),
            annotationFile, DataSourceType.PASTE));
    AlignmentI al = af.getViewport().getAlignment();
    int length = al.getAlignmentAnnotation().length;
    AlignmentAnnotation[] alignAnnot = al.getAlignmentAnnotation();
    Iterable<AlignmentAnnotation> myCalcIdRows = al
            .findAnnotation("myCalcId");
    assertNotNull(myCalcIdRows);
    // Expect 2 - one for the alignment and one for the sequence
    Iterator<AlignmentAnnotation> rows = myCalcIdRows.iterator();
    assertTrue(rows.hasNext());
    AlignmentAnnotation nextan = rows.next();
    assertEquals(nextan.label, "A Label");
    assertEquals("property", nextan.getProperty("special"));
    assertTrue(nextan.sequenceRef == null);
    

    // second expected to be sequence associated
    assertTrue(rows.hasNext());
    nextan = rows.next();
    assertEquals("seqproperty", nextan.getProperty("special"));
    assertTrue(nextan.sequenceRef != null);
    // Now check that the track is transferred to the dataset
    AlignmentAnnotation dsseqan = nextan.sequenceRef.getDatasetSequence()
            .getAlignmentAnnotations(nextan.getCalcId(), nextan.label)
            .get(0);
    assertEquals("seqproperty", dsseqan.getProperty("special"));
    assertFalse(rows.hasNext());
    
    Iterable<AlignmentAnnotation> myCalcId2Rows = al
            .findAnnotation("myCalcId2");
    assertNotNull(myCalcId2Rows);
    rows = myCalcId2Rows.iterator();
    assertTrue(rows.hasNext());
    nextan = rows.next();
    assertEquals(nextan.label, "A Label");
    assertEquals("property", nextan.getProperty("special"));
    assertTrue(nextan.sequenceRef == null);
    
    Iterable<AlignmentAnnotation> withoutCalcIdRows = al
            .findAnnotation("");
    assertNotNull(withoutCalcIdRows);
    rows = withoutCalcIdRows.iterator();
    assertTrue(rows.hasNext());
    nextan = rows.next();
    assertEquals(nextan.label, "Without CALCID 1");
    assertTrue(rows.hasNext());
    nextan = rows.next();
    assertEquals(nextan.label, "Without CALCID 2");
    assertEquals(-1,nextan.getNoOfSequencesIncluded());
    assertEquals(-1,nextan.getNoOfTracksIncluded());
    
    // Check sequence and annotation track counts are imported
    nextan = rows.next();
    assertEquals("SeqTrack",nextan.label);
    assertEquals(4,nextan.getNoOfSequencesIncluded());
    assertEquals(1,nextan.getNoOfTracksIncluded());
    assertFalse(nextan.getProperties().contains(jalview.io.AnnotationFile.NO_OF_SEQUENCES));
    assertFalse(nextan.getProperties().contains(jalview.io.AnnotationFile.NO_OF_TRACKS));
    AlignmentAnnotation consAnn=null;
    long tries=100;
    while (tries-->0 && (consAnn = af.getViewport().getAlignmentConsensusAnnotation())==null 
            || consAnn.getNoOfSequencesIncluded()==-1) {
      try {
        Thread.sleep(30);
      } catch (Exception x) { Assert.fail("Interrupted wating for calculation...");};
    }
    if (tries==0)
    {
      Assert.fail("Gave up waiting for consensus calculation.");
    }
    // Now check we have expected metadata - ideally we'd do a round trip here..
    String annFile = new jalview.io.AnnotationFile().printAnnotationsForView(af.getViewport());
    assertTrue(annFile.contains("NO_OF_SEQUENCES=4"));
    assertTrue(annFile.contains("NO_OF_TRACKS=1"));
    // expect at least one NO_OF_SEQUENCES=3 that occurs after 'Consensus'
    assertTrue(annFile.contains("Consensus"));
    int consensusLineEnds = annFile.indexOf("\n",annFile.indexOf("Consensus"));
    int no_of_seq_pos=annFile.indexOf("NO_OF_SEQUENCES=3",annFile.indexOf("Consensus"));
    assertTrue("Couldn't find NO_OF_SEQUENCES after start of Consensus annotation row",no_of_seq_pos>-1);
    assertTrue("Couldn't find NO_OF_SEQUENCES before end of Consensus annotation row",no_of_seq_pos<consensusLineEnds);
    
  }
  
  
  @Test(groups = "Functional")
  public void testProviderRestore()
  {
    long t1 = System.currentTimeMillis();
    /*
     * JAL-3779 test multiple groups of the same name get annotated
     */
    AlignFrame af = new FileLoader().LoadFileWaitTillLoaded(
            ">Seq1\nQRSIL\n>Seq2\nFTHND\n>Seq3\nRPVSL\n",
            DataSourceType.PASTE);
    long t2 = System.currentTimeMillis();
    System.err.println("t0: " + (t2 - t1));
    // seq1 and seq3 are in distinct groups both named Group1
    String annotationFile = "JALVIEW_ANNOTATION\n"
            + "CALCID\tmyCalcId\tNO_GRAPH\tA Label\tfoolabel\tt|t|t|t\tSS_PROVIDER=PDB\tproperty2=abc\t4property=xyz\n"
            + "CALCID\tmyCalcId\tROWPROPERTIES\tA Label\tspecial=property\n"
            + "CALCID\tmyCalcId\tNO_GRAPH\tA Label\tfoolabel\tt|t|t|t\t\n"
            + "CALCID\tmyCalcId\tROWPROPERTIES\tA Label\tspecial=property\n"
            + "SEQUENCE_REF\tSeq3\n"
            + "\nCALCID\tmyCalcId\tNO_GRAPH\tA Label\tfoolabel\tt|t|t|t\t1.0\t2.0\t3.0\tSS_PROVIDER=JPred\tproperty3=value1\n"
            + "\nCALCID\tmyCalcId\tROWPROPERTIES\tA Label\tspecial=seqproperty\n";
    new AnnotationFile().annotateAlignmentView(af.getViewport(),
            annotationFile, DataSourceType.PASTE);
    AlignmentI al = af.getViewport().getAlignment();
    Iterable<AlignmentAnnotation> myCalcIdRows = al
            .findAnnotation("myCalcId");
    assertNotNull(myCalcIdRows);
    // Expect 3 - 2 for the alignment and one for the sequence
    Iterator<AlignmentAnnotation> rows = myCalcIdRows.iterator();
    assertTrue(rows.hasNext());
    AlignmentAnnotation nextan = rows.next();
    assertEquals(nextan.label, "A Label");
    assertEquals("property", nextan.getProperty("special"));
    assertEquals("PDB", nextan.getProperty("SS_PROVIDER"));
    assertEquals("abc", nextan.getProperty("property2"));
    assertEquals(null, nextan.getProperty("4property"));
    assertTrue(nextan.sequenceRef == null);
    // second row without data properties
    assertTrue(rows.hasNext());
    nextan = rows.next();
    assertEquals(nextan.label, "A Label");
    // third expected to be sequence associated
    assertTrue(rows.hasNext());
    nextan = rows.next();
    assertEquals("seqproperty", nextan.getProperty("special"));
    assertEquals("JPred", nextan.getProperty("SS_PROVIDER"));
    assertEquals("value1", nextan.getProperty("property3"));
    assertEquals(3.0, nextan.score);
    assertTrue(nextan.sequenceRef != null);
    // Now check that the track is transferred to the dataset
    AlignmentAnnotation dsseqan = nextan.sequenceRef.getDatasetSequence()
            .getAlignmentAnnotations(nextan.getCalcId(), nextan.label)
            .get(0);
    assertEquals("seqproperty", dsseqan.getProperty("special"));
    assertEquals("JPred", dsseqan.getProperty("SS_PROVIDER")); 
  }
  
  @Test
  public void testIsPropertyValueToken() {
    AnnotationFile annotationFile = new AnnotationFile();

    // Valid cases
    assertTrue(annotationFile.isPropertyValueToken("property=value"));
    assertTrue(annotationFile.isPropertyValueToken("_myProperty123=myValue123"));
    assertTrue(annotationFile.isPropertyValueToken("_myProperty123=123"));

    // Invalid cases
    assertFalse(annotationFile.isPropertyValueToken("=value"));           // no key
    assertFalse(annotationFile.isPropertyValueToken("key="));             // no value
    assertFalse(annotationFile.isPropertyValueToken("123key=value"));     // invalid key
    assertFalse(annotationFile.isPropertyValueToken("@key=value"));     // invalid key
    assertFalse(annotationFile.isPropertyValueToken("property"));         // no =
    assertFalse(annotationFile.isPropertyValueToken(""));                 // empty string
  }
  
  @Test
  public void testValidKeywords() {
    AnnotationFile annotationFile = new AnnotationFile();
      assertTrue(annotationFile.isAnnotationKeyword("BAR_GRAPH"));
      assertTrue(annotationFile.isAnnotationKeyword("bar_graph"));
      assertTrue(annotationFile.isAnnotationKeyword("BarGraph"));
      assertTrue(annotationFile.isAnnotationKeyword("rowproperties"));
      assertTrue(annotationFile.isAnnotationKeyword("VIEW_HIDECOLS"));        
      assertFalse(annotationFile.isAnnotationKeyword("INVALID"));
      assertFalse(annotationFile.isAnnotationKeyword("ANNOTATION_ROW"));
      assertFalse(annotationFile.isAnnotationKeyword(null));
  }

}
