/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_test.c
    \brief  Program to test the LCMAPS and its plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This program has elements of the edg-gatekeeper to be able to test the
    LCMAPS and its plugins without having the edg-gatekeeper installed.
    To run it : just run ./lcmaps-test
    It is not possible (yet) to feed a user credential (proxy) to the program.
*/

#include "lcmaps_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <gssapi.h>
#include "globus_gss_assist.h"

/* Programming interface to dynamic linking loader */
#if HAVE_DLFCN_H
#include <dlfcn.h>
#endif

#if LINKED_LCMAPS
#include "lcmaps.h"                                                                             
#endif

static void failure(short failure_type, char *s);
static void notice(int, char *s);

#define FAILED_AUTHORIZATION        1
#define FAILED_SERVICELOOKUP        2
#define FAILED_SERVER               3
#define FAILED_NOLOGIN              4
#define FAILED_AUTHENTICATION       5
#define FAILED_PING                 6

static char     tmpbuf[1024];
#define notice2(i,a,b) {sprintf(tmpbuf, a,b); notice(i,tmpbuf);}
#define notice3(i,a,b,c) {sprintf(tmpbuf, a,b,c); notice(i,tmpbuf);}
#define notice4(i,a,b,c,d) {sprintf(tmpbuf, a,b,c,d); notice(i,tmpbuf);}
#define failure2(t,a,b) {sprintf(tmpbuf, a,b); failure(t,tmpbuf);}
#define failure3(t,a,b,c) {sprintf(tmpbuf, a,b,c); failure(t,tmpbuf);}
#define failure4(t,a,b,c,d) {sprintf(tmpbuf, a,b,c,d); failure(t,tmpbuf);}

static char *   lcmapsmod_name= NULL;
static gss_cred_id_t delegated_cred_handle = GSS_C_NO_CREDENTIAL;
static FILE *   usrlog_fp=NULL;

int main()
{
    char * lcmaps_request="Test lcmaps_request";
    char * client_name="/O=dutchgrid/O=users/O=nikhef/CN=Martijn Steenbakkers";
    int    retval=0;

    usrlog_fp=stderr;

    /* Load proxy */
    {
        OM_uint32 major_status;
        OM_uint32 minor_status;
        char * proxyname = "/home/gridtest/cvs/fabric_mgt/gridification/lcmaps/modules/voms/x509up_u500";
    
        setenv("X509_USER_PROXY",proxyname,1);
        major_status = globus_gss_assist_acquire_cred(&minor_status,
                                                      GSS_C_INITIATE, /* or GSS_C_ACCEPT */
                                                      &delegated_cred_handle);

        if (major_status != GSS_S_COMPLETE)
        {
            globus_gss_assist_display_status(stderr,
                                             "Some failure message here",
                                             major_status,
                                             minor_status,
                                             0);
            return 1;
        }
    }

#if LINKED_LCMAPS
    {
        int retval;

        notice(0,"Using linked in version of LCMAPS");
        /* Initialize, send authorization request to and terminate the LCMAPS */
        retval=lcmaps_init(usrlog_fp);
        if (retval)
        {
            failure(FAILED_SERVER, "LCMAPS initialization failure.");
        }
#if ALLOW_EMPTY_CREDENTIALS
        retval=lcmaps_run(client_name,delegated_cred_handle, lcmaps_request);
#else
        retval=lcmaps_run(delegated_cred_handle, lcmaps_request);
#endif /* ALLOW_EMPTY_CREDENTIALS */
        if (retval)
        {
            failure(FAILED_AUTHORIZATION, "LCMAPS failed authorization.");
        }
        retval=lcmaps_term();
        if (retval)
        {
            failure(FAILED_SERVER, "LCMAPS termination failure.");
        }
    }
#else /* LINKED_LCMAPS */
    lcmapsmod_name="/usr/lib/lcmaps/lcmaps.mod";
    {
        void *handle;
        char *error;
        int retval;
        int (*LcmapsInit)(FILE *);
        int (*LcmapsTerm)();
#if ALLOW_EMPTY_CREDENTIALS
        int (*LcmapsRun)(char*, gss_cred_id_t, char*);
        notice(0,"temporarily ALLOW empty credentials");
#else
        int (*LcmapsRun)(gss_cred_id_t, char*);
#endif

        notice2(0,"lcmapsmod_name = %s",lcmapsmod_name);
        handle = dlopen(lcmapsmod_name,RTLD_LAZY|RTLD_GLOBAL);
        if (!handle)
        {
            notice2(0,"dlopen error: %s",dlerror());
            failure2(FAILED_SERVER,"Cannot open LCMAPS module of %s",lcmapsmod_name);
        }
        else
        {
            /* Check the symbols */
            LcmapsInit=dlsym(handle,"lcmaps_init");
            if ((error = dlerror()) != NULL)
            {
                notice2(0,"dlsym error: %s",error);
                dlclose(handle);
                failure(FAILED_SERVER,"LCMAPS module not compliant.");
            }
            LcmapsRun=dlsym(handle,"lcmaps_run");
            if ((error = dlerror()) != NULL)
            {
                notice2(0,"dlsym error: %s",error);
                dlclose(handle);
                failure(FAILED_SERVER,"LCMAPS module not compliant.");
            }
            LcmapsTerm=dlsym(handle,"lcmaps_term");
            if ((error = dlerror()) != NULL)
            {
                notice2(0,"dlsym error: %s",error);
                dlclose(handle);
                failure(FAILED_SERVER,"LCMAPS module not compliant.");
            }

            /* Initialize, send authorization request to and terminate the LCMAPS */
            retval=(*LcmapsInit)(usrlog_fp);
            if (retval)
            {
                dlclose(handle);
                failure(FAILED_SERVER, "LCMAPS initialization failure.");
            }
#if ALLOW_EMPTY_CREDENTIALS
            retval=(*LcmapsRun)(client_name, delegated_cred_handle, lcmaps_request);
#else
            retval=(*LcmapsRun)(delegated_cred_handle, lcmaps_request);
#endif
            if (retval)
            {
                if ((*LcmapsTerm)())
                {
                    dlclose(handle);
                    failure(FAILED_SERVER, "LCMAPS termination failure.");
                }
                dlclose(handle);
                failure(FAILED_NOLOGIN, "LCMAPS failed user mapping.");
            }
            retval=(*LcmapsTerm)();
            if (retval)
            {
                dlclose(handle);
                failure(FAILED_SERVER, "LCMAPS termination failure.");
            }
            dlclose(handle);
        }
    }
#endif /* LINKED_LCMAPS */
    fprintf(usrlog_fp,"return value= %d\n",retval);
    return 0;
}
/******************************************************************************
Function:       notice()
Description:    
Parameters: prty is the syslog priority, but if = 0, then dont syslog. 
Returns:
******************************************************************************/
static void 
notice(int prty, char * s)
{
    {
        fprintf(usrlog_fp, "Notice: %d: %s\n", prty, s);
    }
} /* notice() */

/******************************************************************************
Function:       failure()
Description:    
Parameters:
Returns:
******************************************************************************/
static void 
failure(short failure_type, char * s)
{
    fprintf(stderr,"Failure: %s\n", s);
    {
        fprintf(usrlog_fp, "Failure: %s\n", s);
    }
    exit(1);
} /* failure() */
/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/lcmaps_test.c,v $
    $Date: 2011-04-13 15:58:19 +0200 (Wed, 13 Apr 2011) $
    $Revision: 15235 $
    $Author: okoeroo $
******************************************************************************/
