package tests::functional::LrMailTest;

use strict;

use base qw/Lire::Test::FunctionalTestCase/;

use Lire::Config::Value;
use Lire::Config::Build qw/ac_path/;
use Lire::Utils qw/create_file tempdir/;

# For some reason related to the use of MIME::Parser
# we need to use this module if we want to be able to run it
# directly from the TestRunner

{
    use utf8;
    use MIME::Parser;
}

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->{'lr_mail_path'} = ac_path( 'libexecdir', 'PACKAGE' ) . "/lr_mail";
    my $tmpdir = tempdir( "MIME::Parser_XXXXXX", 'CLEANUP' => 1 );
    $self->{'parser'} = new MIME::Parser();
    $self->{'parser'}->output_under( $tmpdir );

    return $self;

}
sub test_lr_mail {
    my $self = $_[0];

    my $content = <<EOF;
This is a plain text file.

With some French characters: .
EOF

    my $file = $self->rundir() . "/test.txt";
    create_file( $file, $content );
    my $email = 'flacoste@logreport.org';
    my $result = $self->lire_run( "$self->{'lr_mail_path'} -f $email -s 'Subject with spaces' text/plain $file $email" );
    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
                                  "There were warnings or error messages." );
    $self->assert( ! $result->stdout(), "STDOUT should be empty" );

    my $msg = $result->sent_mail();
    $self->assert_equals( 1, scalar @$msg );
    $self->assert_deep_equals( [ $email ], $msg->[0]{'recipients'} );
    my $entity = $self->{'parser'}->parse_data( $msg->[0]{'message'} );

    my $head = $entity->head();
    $self->assert_equals( "Subject with spaces\n", $head->get( 'Subject' ) );
    $self->assert_equals( "$email\n", $head->get( 'From' ) );
    $self->assert_equals( 'text/plain', $head->mime_type() );

    my $body = $entity->bodyhandle();
    $self->assert_equals( $content, $body->as_string() );
}

sub test_lr_mail_from_config {
    my $self = $_[0];

    my $file = $self->rundir() . "/test.txt";
    create_file( $file, "Test\n" );
    my $email = 'flacoste@logreport.org';

    my $cfg = $self->create_test_cfg_file( 'lr_mail_from_config' );
    $cfg->get( "lr_mail_from" )->set( $email );
    $cfg->get( "lr_mail_reply_to" )->set( $email );
    $cfg->save();

    my $result = $self->lire_run( "$self->{'lr_mail_path'} text/plain $file $email" );
    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
                                  "There were warnings or error messages." );
    $self->assert( ! $result->stdout(), "STDOUT should be empty" );

    my $msg = $result->sent_mail();
    $self->assert_equals( 1, scalar @$msg );
    $self->assert_deep_equals( [ $email ], $msg->[0]{'recipients'} );
    my $entity = $self->{'parser'}->parse_data( $msg->[0]{'message'} );

    my $head = $entity->head();
    $self->assert_equals( "$email\n", $head->get( 'From' ) );
    $self->assert_equals( "$email\n", $head->get( 'Reply-To' ) );
}

1;
