/**
 * \file pappsomspp/processing/cbor/psm/psmfilescanprocessandcopy.cpp
 * \date 15/07/2025
 * \author Olivier Langella
 * \brief PSM file reader designed to parallelize scan process and then copy the results in PSM cbor
 * output stream
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "psmfilescanprocessandcopy.h"
#include "pappsomspp/core/pappsoexception.h"
#include "pappsomspp/core/utils.h"


namespace pappso
{
namespace cbor
{
namespace psm
{

PsmFileScanProcessAndCopy::PsmFileScanProcessAndCopy(std::size_t buffer_scan_size,
                                                     CborStreamWriter *cbor_output_p,
                                                     const QString &operation)
  : PsmFileScanProcess(buffer_scan_size)
{
  mp_cborOutput = cbor_output_p;
  m_operation   = operation;


  mp_cborOutput->startMap();
  mp_copyCborOutput = nullptr;
}


PsmFileScanProcessAndCopy::~PsmFileScanProcessAndCopy()
{
  if(mpa_bufferAfterProteinMap != nullptr)
    {
      delete mpa_bufferAfterProteinMap;
    }
  if(mpa_bufferWriterAfterProteinMap != nullptr)
    {
      delete mpa_bufferWriterAfterProteinMap;
    }
}


void
PsmFileScanProcessAndCopy::delayProteinMapInMemory()
{
  mp_copyCborOutput = mp_cborOutput;

  mpa_bufferAfterProteinMap = new QBuffer();
  mpa_bufferAfterProteinMap->open(QIODevice::ReadWrite);
  mpa_bufferWriterAfterProteinMap = new pappso::cbor::CborStreamWriter(mpa_bufferAfterProteinMap);
}

void
PsmFileScanProcessAndCopy::close()
{
  if(mp_copyCborOutput == nullptr)
    {
      mp_cborOutput->endMap();
    }
  else
    {

      // mpa_bufferWriterAfterProteinMap->endArray(); // end scan_list
      // mpa_bufferWriterAfterProteinMap->endMap();   // end sample
      // mpa_bufferWriterAfterProteinMap->endArray(); // end sample_list

      mp_copyCborOutput->append("protein_map");
      m_psmNewProteinMap.writeMap(*mp_copyCborOutput);

      // copy buffer

      mp_copyCborOutput->append("sample_list");

      // mpa_writer->startArray(); // sample_list
      // mpa_writer->endArray();   // end sample_list


      mpa_bufferAfterProteinMap->seek(0);
      // qWarning() << "size=" << mpa_buffer->data().size();

      mp_copyCborOutput->device()->write(mpa_bufferAfterProteinMap->data());

      mp_copyCborOutput->endMap();
    }
}

void
PsmFileScanProcessAndCopy::informationsReady(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  mp_cborOutput->writeInformations("pappsomspp_copy", Utils::getVersion(), "psm", m_operation);
}


void
PsmFileScanProcessAndCopy::logReady(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  m_cborLog.append(m_cborInformations);
  mp_cborOutput->append("log");
  mp_cborOutput->writeCborArray(m_cborLog);
}

void
PsmFileScanProcessAndCopy::parameterMapReady(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  mp_cborOutput->append("parameter_map");
  mp_cborOutput->writeCborMap(m_cborParameterMap);
}

void
PsmFileScanProcessAndCopy::fastaFilesReady(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  // "target_fasta_files": ["zea_mays.fasta", "contaminant.fasta"],
  if(!m_targetFastaFiles.isEmpty())
    {
      mp_cborOutput->append("target_fasta_files");
      mp_cborOutput->writeArray(m_targetFastaFiles);
    }
  //"decoy_fasta_files" : ["rev_zea_mays.fasta", "rev_contaminant.fasta"],

  if(!m_decoyFastaFiles.isEmpty())
    {
      mp_cborOutput->append("decoy_fasta_files");
      mp_cborOutput->writeArray(m_decoyFastaFiles);
    }
}

void
PsmFileScanProcessAndCopy::proteinMapReady(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  if(mp_copyCborOutput == nullptr)
    {
      mp_cborOutput->append("protein_map");
      m_proteinMap.writeMap(*mp_cborOutput);
    }
  else
    {
      // protein map writing is delayed : swap to temporary buffer writer
      mp_cborOutput = mpa_bufferWriterAfterProteinMap;
    }
}


void
PsmFileScanProcessAndCopy::processBufferScanDone(pappso::UiMonitorInterface &monitor
                                                 [[maybe_unused]])
{

  // monitor.setStatus(QObject::tr("PsmFileScanProcessAndCopy::processBufferScanDone"));
  for(CborScanMapBase *cbor_scan_p : m_cborScanList)
    {
      // qDebug() << cbor_scan_p->keys();
      cbor_scan_p->filterAndSortPsmList();

      if(mp_copyCborOutput != nullptr)
        {
          // populate new protein map
          cbor_scan_p->populateProteinMapUsingOldProteinMap(m_proteinMap, m_psmNewProteinMap);
        }


      // qWarning() << "PsmFileScanProcessAndCopy::processBufferScanDone "
      //           << cbor_scan_p->value("psm_list").toArray().size();

      // cbor_scan_p->write(*mp_cborOutput);
      mp_cborOutput->writeCborMap(*cbor_scan_p);

      if(!cbor_scan_p->keys().contains("id"))
        {
          throw pappso::PappsoException(
            QObject::tr("missing scan id %1").arg(cbor_scan_p->keys().size()));
        }
    }
}

void
PsmFileScanProcessAndCopy::sampleStarted(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  mp_cborOutput->startMap();
  mp_cborOutput->append("name");
  mp_cborOutput->append(m_currentSampleName);
  if(m_currentIdentificationFileList.size() > 0)
    {
      mp_cborOutput->append("identification_file_list");
      writePsmFileList(*mp_cborOutput, m_currentIdentificationFileList);
    }
  mp_cborOutput->append("peaklist_file");
  writePsmFile(*mp_cborOutput, m_currentPeaklistFile);

  mp_cborOutput->append("scan_list");
  mp_cborOutput->startArray();
}

void
PsmFileScanProcessAndCopy::sampleFinished(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{

  processBufferScan(monitor);
  clearScanBuffer();

  mp_cborOutput->endArray();
  mp_cborOutput->endMap();
}

void
PsmFileScanProcessAndCopy::sampleListStarted(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  if(mp_copyCborOutput == nullptr)
    {
      mp_cborOutput->append("sample_list");
    }
  mp_cborOutput->startArray();
}

void
PsmFileScanProcessAndCopy::sampleListFinished(pappso::UiMonitorInterface &monitor [[maybe_unused]])
{
  mp_cborOutput->endArray();
}


} // namespace psm
} // namespace cbor
} // namespace pappso
