#include "manetGraph.h"
#include <protoDebug.h>

NetGraph::Cost::Cost()
{
}

NetGraph::Cost::~Cost()
{
}

NetGraph::Link::Link()
{
}

NetGraph::Link::~Link()
{
}

void NetGraph::Link::SetCost(const Cost& cost)
{
    AccessCost() = cost;
}  // end NetGraph::Link::SetCost()

NetGraph::Interface::Interface(Node& theNode, const ProtoAddress& addr)
 : node(theNode), address(addr)
{
}

NetGraph::Interface::~Interface()
{
}

NetGraph::Interface::AdjacencyIterator::AdjacencyIterator(Interface& theInterface)
 : ProtoGraph::Vertice::AdjacencyIterator(theInterface)
{
}

NetGraph::Interface::AdjacencyIterator::~AdjacencyIterator()
{
}

NetGraph::Interface::SimpleList::SimpleList(ItemPool* itemPool)
 : Vertice::SimpleList(itemPool)
{
}

NetGraph::Interface::SimpleList::~SimpleList()
{
}

NetGraph::Interface::SimpleList::Iterator::Iterator(const SimpleList& theList)
 : ProtoGraph::Vertice::SimpleList::Iterator(theList)
{
}

NetGraph::Interface::SimpleList::Iterator::~Iterator()
{
}

NetGraph::Interface::SortedList::SortedList(ItemPool* itemPool)
 : Vertice::SortedList(itemPool)
{
}

NetGraph::Interface::SortedList::~SortedList()
{
}

NetGraph::Interface::SortedList::Iterator::Iterator(const SortedList& theList)
 : ProtoGraph::Vertice::SortedList::Iterator(theList)
{
}

NetGraph::Interface::SortedList::Iterator::~Iterator()
{
}

NetGraph::Interface::PriorityQueue::Item::Item()
 : prev_hop(NULL), next_hop_link(NULL), level(0)
{
}

NetGraph::Interface::PriorityQueue::Item::~Item()
{
}

NetGraph::Interface::PriorityQueue::ItemFactory::ItemFactory()
{
}

NetGraph::Interface::PriorityQueue::ItemFactory::~ItemFactory()
{
    Destroy();
}

NetGraph::Interface::PriorityQueue::Item* NetGraph::Interface::PriorityQueue::ItemFactory::GetItem()
{
    Item* item = static_cast<PriorityQueue::Item*>(item_pool.QueueStatePool::Get());
    if (NULL == item) item = CreateItem();
    if (NULL == item)
        PLOG(PL_ERROR, "NetGraph::Interface::PriorityQueue::ItemFactory::GetItem() CreateItem() error: %s\n",
                       GetErrorString());
    return item;
}  // end NetGraph::Interface::PriorityQueue::ItemFactory::GetItem()

NetGraph::Interface::PriorityQueue::PriorityQueue(PriorityQueue::ItemFactory& itemFactory)
 : item_factory(itemFactory)
{
}

NetGraph::Interface::PriorityQueue::~PriorityQueue()
{
    Empty();
    item_factory.Destroy();
}

bool NetGraph::Interface::PriorityQueue::Insert(Interface& iface, const Cost& cost)
{
    Item* item = item_factory.GetItem();
    if (NULL == item)
    {
        PLOG(PL_ERROR, "NetGraph::Interface::PriorityQueue::Insert() error: couldn't allocate item\n");
        return false;
    }
    item->SetCost(cost);
    Associate(iface, *item);
    InsertItem(*item);
    return true;
}  // end NetGraph::Interface::PriorityQueue::Insert()

void NetGraph::Interface::PriorityQueue::Adjust(Interface& iface, const Cost& newCost)
{
    Item* item = static_cast<Item*>(GetQueueState(iface));
    ASSERT(item != NULL);
    RemoveItem(*item);
    item->SetCost(newCost);
    InsertItem(*item);
}  // end ProtoGraph::Vertice::PriorityQueue::Adjust()

bool NetGraph::Interface::PriorityQueue::AdjustDownward(Interface& iface, const Cost& newCost)
{
    Item* item = static_cast<Item*>(GetQueueState(iface));
    ASSERT(item != NULL);
    if (newCost < item->GetCost())
    {
        RemoveItem(*item);
        item->SetCost(newCost);
        InsertItem(*item);
        return true;
    }
    else
    {
        return false;
    }
}  // end ProtoGraph::Vertice::PriorityQueue::AdjustDownward()

bool NetGraph::Interface::PriorityQueue::Append(Interface& iface)
{
    Item* item = item_factory.GetItem();
    if (NULL == item)
    {
        PLOG(PL_ERROR, "NetGraph::Interface::PriorityQueue::Insert() error: couldn't allocate item\n");
        return false;
    }
    Associate(iface, *item);
    SortedList::AppendItem(*item);
    return true;
}  // end NetGraph::Interface::PriorityQueue::Append()

NetGraph::Interface::PriorityQueue::Iterator::Iterator(PriorityQueue& theQueue)
 : SortedList::Iterator(theQueue)
{
}

NetGraph::Interface::PriorityQueue::Iterator::~Iterator()
{
}


NetGraph::Node::Node()
 : iface_list(true)
{
}

NetGraph::Node::~Node()
{
    // Delete interfaces
    Interface* iface;
    while (NULL != (iface = static_cast<Interface*>(iface_list.RemoveHead())))
        delete iface;
}       

NetGraph::Node::InterfaceIterator::InterfaceIterator(Node& node)
 : ProtoSortedTree::Iterator(node.iface_list)
{
}

NetGraph::Node::InterfaceIterator::~InterfaceIterator()
{
}

NetGraph::NetGraph()
{
}

NetGraph::~NetGraph()
{
}

bool NetGraph::InsertNode(Node& node, Interface* iface)
{
    // need to init iface_tree on first insert !!!
    if (NULL == iface) 
        iface = node.GetDefaultInterface();
    if (NULL != iface)
    {
        return ProtoGraph::InsertVertice(*iface);
    }
    else
    {
        PLOG(PL_ERROR, "NetGraph::InsertNode() error: node has no interfaces?!\n");
        return false;
    }
}  // end NetGraph::InsertNode()

void NetGraph::RemoveNode(Node& node, Interface* iface)
{
    if (NULL == iface)
    {
        // Remove all interfaces associated with this node
        Node::InterfaceIterator it(node);
        Interface* iface;
        while (NULL != (iface = it.GetNextInterface()))
            RemoveVertice(*iface);
    }
    else
    {   
        // Remove only the interface specified
        RemoveVertice(*iface);
    }                                  
}  // end NetGraph::RemoveNode()

bool NetGraph::Connect(Interface& srcIface, Interface& dstIface, const Cost& cost, bool duplex)
{
    // TBD - allow multiple links srcIface <-> dstIface ???
    
    // Is there already a link, "srcIface" to "dstIface"?
    if (srcIface.HasEdgeTo(dstIface))
    {
        PLOG(PL_ERROR, "NetGraph::Connect() error: srcIface -> dstIface already connected\n");
        return false;
    }
    // Get a new link from our edge factory (from pool or created as needed)
    Link * link = static_cast<Link*>(GetEdge());
    if (NULL == link)
    {
        PLOG(PL_ERROR, "NetGraph::Connect() error: unable to allocate link\n");
        return false;
    }
    if (duplex)
    {
        // Is there already a reverse link, "dstIface" to "srcIface"?
        if (dstIface.HasEdgeTo(srcIface))
        {
            PLOG(PL_ERROR, "NetGraph::Connect() error: dstIface -> srcIface already connected\n");
            PutEdge(*link);
            return false;
        }
        // Get another "Link" from our "edge_pool" (or create one)
        Link* reverseLink  = static_cast<Link*>(GetEdge());
        if (NULL == reverseLink)
        {
            PLOG(PL_ERROR, "NetGraph::Connect() error: unable to allocate reverse link\n");
            PutEdge(*link);
            return false;
        }
        reverseLink->SetCost(cost);
        dstIface.Connect(srcIface, *reverseLink);
    }
    link->SetCost(cost);
    srcIface.Connect(dstIface, *link);
    return true;
}  // end NetGraph::Connect()

NetGraph::InterfaceIterator::InterfaceIterator(NetGraph& theGraph)
  : ProtoGraph::VerticeIterator(static_cast<ProtoGraph&>(theGraph))
{
}

NetGraph::InterfaceIterator::~InterfaceIterator()
{
}

NetGraph::DijkstraTraversal::DijkstraTraversal(NetGraph&    theGraph,   
                                               Node&        startNode,  
                                               Interface*   startIface)
 : manet_graph(theGraph),
   start_iface((NULL != startIface) ? startIface : startNode.GetDefaultInterface()),
   queue_pending(static_cast<ItemFactory&>(*this)), 
   queue_visited(static_cast<ItemFactory&>(*this)),
   trans_iface(NULL), current_level(0), dijkstra_completed(false), in_update(false)
{
    ASSERT(&start_iface->GetNode() == &startNode);
}

NetGraph::DijkstraTraversal::~DijkstraTraversal()
{
    queue_pending.Empty();
    queue_visited.Empty();
}

bool NetGraph::DijkstraTraversal::Reset()
{
    // (TBD) use a dual queue approach so that we can avoid 
    //      visiting every iface twice everytime we run the Dijkstra.
    //      I.e., we could maintain "visited" & "unvisited" queue, removing ifaces
    //      from that queue as they are visited, and then at the end of
    //      the Dijkstra, update the cost of any remaining unvisited ifaces
    //      to COST_MAX and then swap visited and unvisited queues for next
    //      Dijkstra run ... Thus each iface would be visited only once _or_ as
    //      needed for Dijkstra instead of once _plus_ as needed for Dijkstra
    queue_visited.Empty();
    queue_pending.Empty();
    Cost& startCost = AccessCostTemp();
    startCost.Minimize();
    if (NULL != start_iface)
    {
        if (!queue_pending.Insert(*start_iface, startCost))
        {
            PLOG(PL_ERROR, "NetGraph::DijkstraTraversal::Reset() error: couldn't enqueue start_iface\n");
            return false;
        }
        dijkstra_completed = false;
    }
    else
    {
        dijkstra_completed = true;
    }
    return true;
}  // end NetGraph::DijkstraTraversal::Reset()


// Dijkstra traversal step
NetGraph::Interface* NetGraph::DijkstraTraversal::GetNextInterface()
{
    // (TBD) We could be a little more efficient if we worked with PriorityQueue::Items directly
    Interface* currentIface = queue_pending.GetHead();
    if (NULL != currentIface)
    {
        queue_pending.TransferInterface(*currentIface, queue_visited);
        const Cost* currentCost = queue_visited.GetCost(*currentIface);
        ASSERT(NULL != currentCost);
        Interface::AdjacencyIterator linkIterator(*currentIface);
        Link* nextLink;
        while ((nextLink = linkIterator.GetNextLink()))
        {
            Interface* nextDst = nextLink->GetDst();
            ASSERT(NULL != nextDst);
            
            if (!AllowLink(*currentIface, *nextLink)) continue;
            
            const Cost& linkCost = nextLink->GetCost();
            Cost& nextCost = AccessCostTemp();
            nextCost = linkCost;
            nextCost += *currentCost;
            
            if (nextDst->IsInQueue(queue_pending))
            {
                // We have found another path to this pending iface.
                // If it is a shorter path update the cost to lower value
                if (!queue_pending.AdjustDownward(*nextDst, nextCost))
                        continue;
            }
            else if (!nextDst->IsInQueue(queue_visited))
            {
                // This is the first path found to this iface,
                // so enqueue it in our "queue_pending"
                if (!queue_pending.Insert(*nextDst, nextCost))
                {
                    PLOG(PL_ERROR, "NetGraph::DijkstraTraversal::GetNextInterface() error: couldn't enqueue iface\n");
                    return NULL;
                }
            }
            else if (in_update)
            {
                // I _think_ this is currently broken?  It seems we would need to shuffle nodes 
                // from "queue_visited" to "queue_pending" here to work properly? ... Note that
                // it depends heavily on "Update()" being called with a proper "startIface"
                // Check for shorter path to a "visited" iface
                // (must be in "Update()" mode!)
                if (queue_visited.AdjustDownward(*nextDst, nextCost))
                {
                    // Update the routing tree state 
                    if (currentIface == start_iface)
                        queue_visited.SetRouteInfo(*nextDst, nextLink, currentIface);
                    else
                        queue_visited.SetRouteInfo(*nextDst, queue_visited.GetNextHopLink(*currentIface), currentIface);
                }
                continue;
            }
            else
            {
                continue;
            }
            
            // Save our routing tree state as we go
            if (currentIface == start_iface)
                queue_pending.SetRouteInfo(*nextDst, nextLink, currentIface);
            else
                queue_pending.SetRouteInfo(*nextDst, queue_visited.GetNextHopLink(*currentIface), currentIface);
        }  // end while ((nextLink = linkIterator.GetNextLink()))
    }
    else
    {
        dijkstra_completed = true;
    }   
    return currentIface;
}  // end NetGraph::DijkstraTraversal::GetNextInterface()
       
void NetGraph::DijkstraTraversal::Update(Interface& startIface)
{
    in_update = true;
    // (TBD) fix this for new Dijkstra approach
    if (!dijkstra_completed)
    {
        // Complete dijkstra traversal
        while (NULL != GetNextInterface());
    }
    
    ASSERT(queue_pending.IsEmpty());
    
    if (startIface.IsInQueue(queue_visited))
    {
        queue_visited.TransferInterface(startIface, queue_pending);
    }
    else
    {
        ASSERT(0);
    }
    
    while (NULL != GetNextInterface());
    in_update = false;
}  // end NetGraph::DijkstraTraversal::Update()

// Call this to setup re-traverse of tree computed via Dijkstra
bool NetGraph::DijkstraTraversal::TreeWalkReset()
{
    // If Dijkstra was not completed, run full Dijkstra
    if (!dijkstra_completed)
    {
        Reset();
        while (NULL != GetNextInterface());
    }
    ASSERT(queue_pending.IsEmpty());
    if (NULL != start_iface)
    {
        if (!queue_pending.Append(*start_iface))
        {
            PLOG(PL_ERROR, "NetGraph::DijkstraTraversal::TreeWalkReset() error: couldn't append start_iface\n");
            return false;
        }
    }
    trans_iface = NULL;
    current_level = 0;
    return true;
}  // end NetGraph::DijkstraTraversal::TreeWalkReset()

NetGraph::Interface* NetGraph::DijkstraTraversal::TreeWalkNext(unsigned int* level)
{
    Interface* currentIface =  queue_pending.RemoveHead();
    if (NULL != currentIface)
    {
        // Find selected links
        Interface::AdjacencyIterator linkIterator(*currentIface);
        Link* nextLink;
        Link* firstLink = NULL;
        while ((nextLink = linkIterator.GetNextLink()))
        {
            Interface* nextDst = nextLink->GetDst();
            ASSERT(NULL != nextDst);
            if (currentIface == GetPrevHop(*nextDst))
            {
                if (NULL == firstLink) firstLink = nextLink;
                queue_pending.Append(*nextDst);
            }
        }
        // Track depth as walk progresses ...
        if (NULL == trans_iface)
        {
            trans_iface = firstLink ? firstLink->GetDst() : NULL;
        }
        else if (trans_iface == currentIface)
        {
            trans_iface = firstLink ? firstLink->GetDst() : NULL; 
            current_level++;  
        }
        if (NULL != level) *level = current_level;
    }
    return currentIface;
}  // end NetGraph::DijkstraTraversal::TreeWalkNext()
