/*
 * Copyright (C) 2011 Colin Walters <walters@verbum.org>
 *
 * SPDX-License-Identifier: LGPL-2.0+
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see <https://www.gnu.org/licenses/>.
 *
 * Author: Colin Walters <walters@verbum.org>
 */

#include "config.h"

#include <gio/gio.h>

#include <locale.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mount.h>
#include <sys/statvfs.h>

#include "ostree.h"
#include "ot-admin-functions.h"
#include "otutil.h"

static char *opt_repo;
static char *opt_sysroot;
static gboolean opt_verbose;
static gboolean opt_version;
static gboolean opt_print_current_dir;

// TODO: make this public?  But no one sane wants to use our C headers
// to find where to put files.  Maybe we can make it printed by the CLI?
#define _OSTREE_EXT_DIR PKGLIBEXECDIR "/ext"

static GOptionEntry global_entries[]
    = { { "verbose", 'v', 0, G_OPTION_ARG_NONE, &opt_verbose,
          "Print debug information during command processing", NULL },
        { "version", 0, 0, G_OPTION_ARG_NONE, &opt_version, "Print version information and exit",
          NULL },
        { NULL } };

static GOptionEntry repo_entry[]
    = { { "repo", 0, 0, G_OPTION_ARG_FILENAME, &opt_repo,
          "Path to OSTree repository (defaults to current directory or /sysroot/ostree/repo)",
          "PATH" },
        { NULL } };

static GOptionEntry global_admin_entries[] = {
  /* No description since it's hidden from --help output. */
  { "print-current-dir", 0, G_OPTION_FLAG_HIDDEN, G_OPTION_ARG_NONE, &opt_print_current_dir, NULL,
    NULL },
  { "sysroot", 0, 0, G_OPTION_ARG_FILENAME, &opt_sysroot, "Create a new OSTree sysroot at PATH",
    "PATH" },
  { NULL }
};

static GOptionContext *
ostree_option_context_new_with_commands (OstreeCommand *commands)
{
  GOptionContext *context = g_option_context_new ("COMMAND");

  g_autoptr (GString) summary = g_string_new ("Builtin Commands:");

  while (commands->name != NULL)
    {
      if ((commands->flags & OSTREE_BUILTIN_FLAG_HIDDEN) == 0)
        {
          g_string_append_printf (summary, "\n  %-18s", commands->name);

          if (commands->description != NULL)
            g_string_append_printf (summary, "%s", commands->description);
        }

      commands++;
    }

  g_option_context_set_summary (context, summary->str);

  return context;
}

int
ostree_usage (OstreeCommand *commands, gboolean is_error)
{
  g_autoptr (GOptionContext) context = ostree_option_context_new_with_commands (commands);
  g_option_context_add_main_entries (context, global_entries, NULL);

  g_autofree char *help = g_option_context_get_help (context, FALSE, NULL);
  if (is_error)
    g_printerr ("%s", help);
  else
    g_print ("%s", help);

  return (is_error ? 1 : 0);
}

/* If we're running as root, booted into an OSTree system and have a read-only
 * /sysroot, then assume we may need write access.  Create a new mount namespace
 * if so, and return *out_ns = TRUE.  Otherwise, *out_ns = FALSE.
 */
static gboolean
maybe_setup_mount_namespace (gboolean *out_ns, GError **error)
{
  *out_ns = FALSE;

  /* If we're not root, then we almost certainly can't be remounting anything */
  if (getuid () != 0)
    return TRUE;

  /* If the system isn't booted via libostree, also nothing to do */
  if (!glnx_fstatat_allow_noent (AT_FDCWD, OSTREE_PATH_BOOTED, NULL, 0, error))
    return FALSE;
  if (errno == ENOENT)
    return TRUE;

  if (unshare (CLONE_NEWNS) < 0)
    return glnx_throw_errno_prefix (error, "setting up mount namespace: unshare(CLONE_NEWNS)");

  *out_ns = TRUE;
  return TRUE;
}

static void
message_handler (const gchar *log_domain, GLogLevelFlags log_level, const gchar *message,
                 gpointer user_data)
{
  /* Make this look like normal console output */
  if (log_level & G_LOG_LEVEL_DEBUG)
    g_printerr ("OT: %s\n", message);
  else
    g_printerr ("%s: %s\n", g_get_prgname (), message);
}

int
ostree_main (int argc, char **argv, OstreeCommand *commands)
{
  g_autoptr (GError) error = NULL;

  setlocale (LC_ALL, "");

  g_set_prgname (argv[0]);

  int ret = ostree_run (argc, argv, commands, &error);

  if (error != NULL)
    {
      g_printerr ("%s%serror:%s%s %s\n", ot_get_red_start (), ot_get_bold_start (),
                  ot_get_bold_end (), ot_get_red_end (), error->message);
    }

  return ret;
}

/**
 * ostree_command_lookup_external:
 * @argc: number of entries in @argv
 * @argv: array of command-line arguments
 * @commands: array of hardcoded internal commands
 *
 * Search for a relevant ostree extension binary in $PATH. Given a verb
 * from argv, if it is not an internal command, it tries to locate a
 * corresponding 'ostree-$verb' executable on the system.
 *
 * Returns: (transfer full) (nullable): callname (i.e. argv[0]) for the
 * external command if found, or %NULL otherwise.
 */
gchar *
ostree_command_lookup_external (int argc, char **argv, OstreeCommand *commands)
{
  g_assert (commands != NULL);

  // Find the first verb (ignoring all earlier flags), then
  // check if it is a known native command. Otherwise, try to look it
  // up in /usr/lib/ostree/ostree-$cmd or $PATH.
  // We ignore argv[0] here, the ostree binary itself is not multicall.
  for (guint arg_index = 1; arg_index < argc; arg_index++)
    {
      char *current_arg = argv[arg_index];
      if (current_arg == NULL || g_str_has_prefix (current_arg, "-")
          || g_strcmp0 (current_arg, "") == 0)
        continue;

      for (guint cmd_index = 0; commands[cmd_index].name != NULL; cmd_index++)
        {
          if (g_strcmp0 (current_arg, (commands[cmd_index]).name) == 0)
            return NULL;
        }

      g_autofree gchar *ext_command = g_strdup_printf ("ostree-%s", current_arg);

      /* First, search in our libdir /usr/lib/ostree/ostree-$cmd */
      g_autofree char *ext_lib = g_strconcat (_OSTREE_EXT_DIR, "/", ext_command, NULL);
      struct stat stbuf;
      if (stat (ext_lib, &stbuf) == 0)
        return g_steal_pointer (&ext_lib);

      /* Otherwise, look in $PATH */
      if (g_find_program_in_path (ext_command) == NULL)
        return NULL;
      return g_steal_pointer (&ext_command);
    }

  return NULL;
}

/**
 * ostree_command_exec_external:
 * @argv: array of command-line arguments
 *
 * Execute an ostree extension binary.
 *
 * Returns: diverge on proper execution, otherwise return 1.
 */
int
ostree_command_exec_external (char **argv)
{
  int r = execvp (argv[0], argv);
  g_assert (r == -1);

  setlocale (LC_ALL, "");
  g_printerr ("%s%serror:%s%s: Executing %s: %s\n", ot_get_red_start (), ot_get_bold_start (),
              ot_get_bold_end (), ot_get_red_end (), argv[0], g_strerror (errno));
  return 1;
}

int
ostree_run (int argc, char **argv, OstreeCommand *commands, GError **res_error)
{
  GError *error = NULL;
  GCancellable *cancellable = NULL;
#ifndef BUILDOPT_TSAN
  g_autofree char *prgname = NULL;
#endif
  const char *command_name = NULL;
  gboolean success = FALSE;
  int in, out;

  /* avoid gvfs (http://bugzilla.gnome.org/show_bug.cgi?id=526454) */
  if (!g_setenv ("GIO_USE_VFS", "local", TRUE))
    {
      (void)glnx_throw (res_error, "Failed to set environment variable GIO_USE_FVS");
      return 1;
    }

  g_log_set_handler (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE, message_handler, NULL);

  /*
   * Parse the global options. We rearrange the options as
   * necessary, in order to pass relevant options through
   * to the commands, but also have them take effect globally.
   */

  for (in = 1, out = 1; in < argc; in++, out++)
    {
      /* The non-option is the command, take it out of the arguments */
      if (argv[in][0] != '-')
        {
          if (command_name == NULL)
            {
              command_name = argv[in];
              out--;
              continue;
            }
        }

      argv[out] = argv[in];
    }

  argc = out;

  OstreeCommand *command = commands;
  while (command->name)
    {
      if (g_strcmp0 (command_name, command->name) == 0)
        break;
      command++;
    }

  if (!command->fn)
    {
      g_autoptr (GOptionContext) context = ostree_option_context_new_with_commands (commands);

      /* This will not return for some options (e.g. --version). */
      if (ostree_option_context_parse (context, NULL, &argc, &argv, NULL, NULL, cancellable,
                                       &error))
        {
          if (command_name == NULL)
            {
              g_set_error_literal (&error, G_IO_ERROR, G_IO_ERROR_FAILED, "No command specified");
            }
          else
            {
              g_set_error (&error, G_IO_ERROR, G_IO_ERROR_FAILED, "Unknown command '%s'",
                           command_name);
            }
        }

      ostree_usage (commands, TRUE);
      goto out;
    }

#ifndef BUILDOPT_TSAN
  prgname = g_strdup_printf ("%s %s", g_get_prgname (), command_name);
  g_set_prgname (prgname);
#endif
  OstreeCommandInvocation invocation = { .command = command };
  if (!command->fn (argc, argv, &invocation, cancellable, &error))
    goto out;

  success = TRUE;
out:
  g_assert (success || error);

  if (error)
    {
      g_propagate_error (res_error, error);
      return 1;
    }
  return 0;
}

/* Process a --repo arg. */
static OstreeRepo *
parse_repo_option (GOptionContext *context, const char *repo_path, gboolean skip_repo_open,
                   GCancellable *cancellable, GError **error)
{
  g_autoptr (OstreeRepo) repo = NULL;

  if (repo_path == NULL)
    {
      g_autoptr (GError) local_error = NULL;

      repo = ostree_repo_new_default ();
      if (!ostree_repo_open (repo, cancellable, &local_error))
        {
          if (g_error_matches (local_error, G_IO_ERROR, G_IO_ERROR_NOT_FOUND))
            {
              g_autofree char *help = NULL;

              g_set_error_literal (error, G_IO_ERROR, G_IO_ERROR_FAILED,
                                   "Command requires a --repo argument");

              help = g_option_context_get_help (context, FALSE, NULL);
              g_printerr ("%s", help);
            }
          else
            {
              g_propagate_error (error, g_steal_pointer (&local_error));
            }
          return NULL;
        }
    }
  else
    {
      g_autoptr (GFile) repo_file = g_file_new_for_path (repo_path);

      repo = ostree_repo_new (repo_file);
      if (!skip_repo_open)
        {
          if (!ostree_repo_open (repo, cancellable, error))
            return NULL;
        }
    }

  return g_steal_pointer (&repo);
}

/* Process a --repo arg, determining if we should remount /sysroot; used below, and for the remote
 * builtins */
static OstreeRepo *
parse_repo_option_and_maybe_remount (GOptionContext *context, const char *repo_path,
                                     gboolean skip_repo_open, GCancellable *cancellable,
                                     GError **error)
{
  g_autoptr (OstreeRepo) repo
      = parse_repo_option (context, repo_path, skip_repo_open, cancellable, error);
  if (!repo)
    return NULL;

  /* This is a bit of a brutal hack; we set up a mount
   * namespace if it appears that we may need it.  It'd
   * be better to do this more precisely in the future.
   */
  if (ostree_repo_is_system (repo) && !ostree_repo_is_writable (repo, NULL))
    {
      gboolean setup_ns = FALSE;
      if (!maybe_setup_mount_namespace (&setup_ns, error))
        return FALSE;
      if (setup_ns)
        {
          if (mount ("/sysroot", "/sysroot", NULL, MS_REMOUNT | MS_SILENT, NULL) < 0)
            return glnx_null_throw_errno_prefix (error, "Remounting /sysroot read-write");

          /* Reload the repo so it's actually writable. */
          g_clear_pointer (&repo, g_object_unref);
          repo = parse_repo_option (context, repo_path, skip_repo_open, cancellable, error);
          if (!repo)
            return NULL;
        }
    }

  return g_steal_pointer (&repo);
}

/* Used by the remote builtins which are special in taking --sysroot or --repo */
gboolean
ostree_parse_sysroot_or_repo_option (GOptionContext *context, const char *sysroot_path,
                                     const char *repo_path, OstreeSysroot **out_sysroot,
                                     OstreeRepo **out_repo, GCancellable *cancellable,
                                     GError **error)
{
  g_autoptr (OstreeSysroot) sysroot = NULL;
  g_autoptr (OstreeRepo) repo = NULL;
  if (sysroot_path)
    {
      g_autoptr (GFile) sysroot_file = g_file_new_for_path (sysroot_path);
      sysroot = ostree_sysroot_new (sysroot_file);
      if (!ostree_sysroot_load (sysroot, cancellable, error))
        return FALSE;
      if (!ostree_sysroot_get_repo (sysroot, &repo, cancellable, error))
        return FALSE;
    }
  else
    {
      repo = parse_repo_option_and_maybe_remount (context, repo_path, FALSE, cancellable, error);
      if (!repo)
        return FALSE;
    }

  ot_transfer_out_value (out_sysroot, &sysroot);
  ot_transfer_out_value (out_repo, &repo);
  return TRUE;
}

gboolean
ostree_option_context_parse (GOptionContext *context, const GOptionEntry *main_entries, int *argc,
                             char ***argv, OstreeCommandInvocation *invocation,
                             OstreeRepo **out_repo, GCancellable *cancellable, GError **error)
{
  g_autoptr (OstreeRepo) repo = NULL;
  /* When invocation is NULL,  do not fetch repo */
  const OstreeBuiltinFlags flags
      = invocation ? invocation->command->flags : OSTREE_BUILTIN_FLAG_NO_REPO;

  if (invocation && invocation->command->description != NULL)
    {
      const char *context_summary = g_option_context_get_summary (context);

      /* If the summary is originally empty, we set the description, but
       * for root commands(command with subcommands), we want to prepend
       * the description to the existing summary string
       */
      if (context_summary == NULL)
        g_option_context_set_summary (context, invocation->command->description);
      else
        {
          /* TODO: remove this part once we deduplicate the ostree_option_context_new_with_commands
           * function from other root commands( command with subcommands). Because
           * we can directly add the summary inside the ostree_option_context_new_with_commands
           * function.
           */
          g_autoptr (GString) new_summary_string = g_string_new (context_summary);

          g_string_prepend (new_summary_string, "\n\n");
          g_string_prepend (new_summary_string, invocation->command->description);

          g_option_context_set_summary (context, new_summary_string->str);
        }
    }
  /* Entries are listed in --help output in the order added.  We add the
   * main entries ourselves so that we can add the --repo entry first. */

  if (!(flags & OSTREE_BUILTIN_FLAG_NO_REPO))
    g_option_context_add_main_entries (context, repo_entry, NULL);

  if (main_entries != NULL)
    g_option_context_add_main_entries (context, main_entries, NULL);

  g_option_context_add_main_entries (context, global_entries, NULL);

  if (!g_option_context_parse (context, argc, argv, error))
    return FALSE;

  /* Filter out the first -- we see; g_option_context_parse() leaves it in */
  int in, out;
  gboolean removed_double_dashes = FALSE;
  for (in = 1, out = 1; in < *argc; in++, out++)
    {
      if (g_str_equal ((*argv)[in], "--") && !removed_double_dashes)
        {
          removed_double_dashes = TRUE;
          out--;
          continue;
        }

      (*argv)[out] = (*argv)[in];
    }
  *argc = out;

  if (opt_version)
    {
      /* Ignore SIGPIPE so that piping the output to grep or similar
       * doesn't cause the process to fail. */
      if (signal (SIGPIPE, SIG_IGN) == SIG_ERR)
        return glnx_throw_errno_prefix (error, "Ignoring SIGPIPE");

      /* This should now be YAML, like `docker version`, so it's both nice to read
       * possible to parse */
      g_auto (GStrv) features = g_strsplit (OSTREE_FEATURES, " ", -1);
      g_print ("%s:\n", PACKAGE_NAME);
      g_print (" Version: '%s'\n", PACKAGE_VERSION);
      if (strlen (OSTREE_GITREV) > 0)
        g_print (" Git: %s\n", OSTREE_GITREV);
#ifdef BUILDOPT_IS_DEVEL_BUILD
      g_print (" DevelBuild: yes\n");
#endif
      g_print (" Features:\n");
      for (char **iter = features; iter && *iter; iter++)
        g_print ("  - %s\n", *iter);
      exit (EXIT_SUCCESS);
    }

  if (opt_verbose)
    g_log_set_handler (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, message_handler, NULL);

  if (!(flags & OSTREE_BUILTIN_FLAG_NO_REPO))
    {
      repo = parse_repo_option_and_maybe_remount (
          context, opt_repo, (flags & OSTREE_BUILTIN_FLAG_NO_CHECK) > 0, cancellable, error);
      if (!repo)
        return FALSE;
    }

  if (out_repo)
    *out_repo = g_steal_pointer (&repo);

  return TRUE;
}

static void
on_sysroot_journal_msg (OstreeSysroot *sysroot, const char *msg, void *dummy)
{
  g_print ("%s\n", msg);
}

gboolean
ostree_admin_sysroot_load (OstreeSysroot *sysroot, OstreeAdminBuiltinFlags flags,
                           GCancellable *cancellable, GError **error)
{
  if ((flags & OSTREE_ADMIN_BUILTIN_FLAG_UNLOCKED) == 0)
    {
      /* Set up the mount namespace, if applicable */
      if (!ostree_sysroot_initialize_with_mount_namespace (sysroot, cancellable, error))
        return FALSE;

      /* Released when sysroot is finalized, or on process exit */
      if (!ot_admin_sysroot_lock (sysroot, error))
        return FALSE;
    }

  if (!ostree_sysroot_load (sysroot, cancellable, error))
    return FALSE;

  if (flags & OSTREE_ADMIN_BUILTIN_FLAG_SUPERUSER)
    {
      OstreeDeployment *booted = ostree_sysroot_get_booted_deployment (sysroot);

      /* Only require root if we're manipulating a booted sysroot. (Mostly
       * useful for the test suite)
       */
      if (booted && getuid () != 0)
        {
          g_set_error (error, G_IO_ERROR, G_IO_ERROR_PERMISSION_DENIED,
                       "You must be root to perform this command");
          return FALSE;
        }
    }

  return TRUE;
}

gboolean
ostree_admin_option_context_parse (GOptionContext *context, const GOptionEntry *main_entries,
                                   int *argc, char ***argv, OstreeAdminBuiltinFlags flags,
                                   OstreeCommandInvocation *invocation, OstreeSysroot **out_sysroot,
                                   GCancellable *cancellable, GError **error)
{
  /* Entries are listed in --help output in the order added.  We add the
   * main entries ourselves so that we can add the --sysroot entry first. */

  g_option_context_add_main_entries (context, global_admin_entries, NULL);

  if (!ostree_option_context_parse (context, main_entries, argc, argv, invocation, NULL,
                                    cancellable, error))
    return FALSE;

  if (!opt_print_current_dir && (flags & OSTREE_ADMIN_BUILTIN_FLAG_NO_SYSROOT))
    {
      g_assert_null (out_sysroot);
      /* Early return if no sysroot is requested */
      return TRUE;
    }

  g_autoptr (GFile) sysroot_path = NULL;
  if (opt_sysroot != NULL)
    sysroot_path = g_file_new_for_path (opt_sysroot);

  g_autoptr (OstreeSysroot) sysroot = ostree_sysroot_new (sysroot_path);
  if (!ostree_sysroot_initialize (sysroot, error))
    return FALSE;
  g_signal_connect (sysroot, "journal-msg", G_CALLBACK (on_sysroot_journal_msg), NULL);

  if (opt_print_current_dir || (flags & OSTREE_ADMIN_BUILTIN_FLAG_NO_LOAD) == 0)
    {
      if (!ostree_admin_sysroot_load (sysroot, flags, cancellable, error))
        return FALSE;
    }

  if (opt_print_current_dir)
    {
      g_autoptr (GPtrArray) deployments = NULL;
      OstreeDeployment *first_deployment;
      g_autoptr (GFile) deployment_file = NULL;
      g_autofree char *deployment_path = NULL;

      deployments = ostree_sysroot_get_deployments (sysroot);
      if (deployments->len == 0)
        return glnx_throw (error, "Unable to find a deployment in sysroot");
      first_deployment = deployments->pdata[0];
      deployment_file = ostree_sysroot_get_deployment_directory (sysroot, first_deployment);
      deployment_path = g_file_get_path (deployment_file);

      g_print ("%s\n", deployment_path);

      /* The g_autoptr, g_autofree etc. don't happen when we explicitly
       * exit, making valgrind complain about leaks */
      g_clear_object (&sysroot);
      g_clear_object (&sysroot_path);
      g_clear_object (&deployment_file);
      g_clear_pointer (&deployments, g_ptr_array_unref);
      g_clear_pointer (&deployment_path, g_free);
      exit (EXIT_SUCCESS);
    }

  if (out_sysroot)
    *out_sysroot = g_steal_pointer (&sysroot);

  return TRUE;
}

gboolean
ostree_ensure_repo_writable (OstreeRepo *repo, GError **error)
{
  if (!ostree_repo_is_writable (repo, error))
    return glnx_prefix_error (error, "Cannot write to repository");
  return TRUE;
}

#ifndef OSTREE_DISABLE_GPGME
void
ostree_print_gpg_verify_result (OstreeGpgVerifyResult *result)
{
  guint n_sigs = ostree_gpg_verify_result_count_all (result);

  /* XXX If we ever add internationalization, use ngettext() here. */
  g_print ("GPG: Verification enabled, found %u signature%s:\n", n_sigs, n_sigs == 1 ? "" : "s");

  g_autoptr (GString) buffer = g_string_sized_new (256);

  for (guint ii = 0; ii < n_sigs; ii++)
    {
      g_string_append_c (buffer, '\n');
      ostree_gpg_verify_result_describe (result, ii, buffer, "  ",
                                         OSTREE_GPG_SIGNATURE_FORMAT_DEFAULT);
    }

  g_print ("%s", buffer->str);
}
#endif /* OSTREE_DISABLE_GPGME */

gboolean
ot_enable_tombstone_commits (OstreeRepo *repo, GError **error)
{
  gboolean tombstone_commits = FALSE;
  GKeyFile *config = ostree_repo_get_config (repo);

  tombstone_commits = g_key_file_get_boolean (config, "core", "tombstone-commits", NULL);
  /* tombstone_commits is FALSE either if it is not found or it is really set to FALSE in the
   * config file.  */
  if (!tombstone_commits)
    {
      g_key_file_set_boolean (config, "core", "tombstone-commits", TRUE);
      if (!ostree_repo_write_config (repo, config, error))
        return FALSE;
    }

  return TRUE;
}
