///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/ViewportConfiguration.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/ViewportPanel.h>

#include <core/gui/ApplicationManager.h>
#include <core/gui/mainwnd/MainFrame.h>

#include <core/data/DataSetManager.h>

#include <core/scene/objects/AbstractCameraObject.h>
#include <core/scene/ObjectNode.h>

namespace Core {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ViewportRecord, RefTarget)
DEFINE_FLAGS_REFERENCE_FIELD(ViewportRecord, ObjectNode, "ViewNode", PROPERTY_FIELD_NO_UNDO|PROPERTY_FIELD_NEVER_CLONE_TARGET, _viewNode)

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(ViewportConfiguration, RefTarget)
DEFINE_FLAGS_VECTOR_REFERENCE_FIELD(ViewportConfiguration, ViewportRecord, "Records", PROPERTY_FIELD_NO_UNDO|PROPERTY_FIELD_ALWAYS_CLONE, _viewRecords)

/******************************************************************************
* Default constructor.
******************************************************************************/
ViewportRecord::ViewportRecord(bool isLoading) : RefTarget(isLoading),
	_viewType(Viewport::VIEW_NONE), _shadingMode(Viewport::SHADING_WIREFRAME), _showGrid(false),
	_fieldOfView(100), _viewMatrix(IDENTITY), _showRenderFrame(false),
	_viewport(NULL), _orbitCenter(ORIGIN), _useOrbitCenter(false)
{
	INIT_PROPERTY_FIELD(ViewportRecord, _viewNode);
}

/******************************************************************************
* Initialization constructor.
******************************************************************************/
ViewportRecord::ViewportRecord(Viewport::ViewportType viewType, Viewport::ViewportShadingMode shadingMode, bool showGrid,
	FloatType fieldOfView, const AffineTransformation& viewMatrix, bool showRenderFrame) : RefTarget(false),
	_viewType(viewType), _shadingMode(shadingMode), _showGrid(showGrid),
	_fieldOfView(fieldOfView), _viewMatrix(viewMatrix), _showRenderFrame(showRenderFrame),
	_viewport(NULL), _orbitCenter(ORIGIN), _useOrbitCenter(false)
{
	INIT_PROPERTY_FIELD(ViewportRecord, _viewNode);
}

/******************************************************************************
* Returns a description the viewport's view at the given animation time.
******************************************************************************/
void ViewportRecord::setRenderFrameShown(bool show)
{
	if(_showRenderFrame != show) {
		_showRenderFrame = show;
		if(_viewport) {
			if(APPLICATION_MANAGER.guiMode())
				MAIN_FRAME->viewportPanel()->layoutViewports();
			_viewport->updateViewport(true);
		}
	}
}

/******************************************************************************
* Turns the grid on or off.
******************************************************************************/
void ViewportRecord::setGridShown(bool visible)
{
	if(visible == _showGrid) return;
	_showGrid = visible;
	if(_viewport) _viewport->updateViewport(true);
}

/******************************************************************************
* Sets the view type.
******************************************************************************/
void ViewportRecord::setViewType(Viewport::ViewportType type)
{
	if(type == viewType()) return;

	// Reset camera node.
	setViewNode(NULL);

	// Setup default view matrix.
	switch(type) {
		case Viewport::VIEW_TOP:
			setViewMatrix(IDENTITY);
			setGridMatrix(IDENTITY);
			break;
		case Viewport::VIEW_BOTTOM:
			setViewMatrix(AffineTransformation(-1,0,0,0,  0,1,0,0,  0,0,-1,0));
			setGridMatrix(inverseViewMatrix());
			break;
		case Viewport::VIEW_LEFT:
			setViewMatrix(AffineTransformation(0,-1,0,0,  0,0,1,0,  -1,0,0,0));
			setGridMatrix(inverseViewMatrix());
			break;
		case Viewport::VIEW_RIGHT:
			setViewMatrix(AffineTransformation(0,1,0,0,  0,0,1,0,  1,0,0,0));
			setGridMatrix(inverseViewMatrix());
			break;
		case Viewport::VIEW_FRONT:
			setViewMatrix(AffineTransformation(1,0,0,0,  0,0,1,0,  0,-1,0,0));
			setGridMatrix(inverseViewMatrix());
			break;
		case Viewport::VIEW_BACK:
			setViewMatrix(AffineTransformation(-1,0,0,0,  0,0,1,0,  0,1,0,0));
			setGridMatrix(inverseViewMatrix());
			break;
		case Viewport::VIEW_ORTHO:
			if(viewType() == Viewport::VIEW_NONE)
				setViewMatrix(IDENTITY);
			setGridMatrix(IDENTITY);
			break;
		case Viewport::VIEW_PERSPECTIVE:
			if(viewType() >= Viewport::VIEW_TOP && viewType() <= Viewport::VIEW_ORTHO) {
				setViewMatrix(AffineTransformation::translation(Vector3(0,0,-fieldOfView())) * viewMatrix());
			}
			else if(viewType() != Viewport::VIEW_PERSPECTIVE)
				setViewMatrix(AffineTransformation::translation(Vector3(0,0,-50)));
			setGridMatrix(IDENTITY);
			break;
		case Viewport::VIEW_SCENENODE:
			setGridMatrix(IDENTITY);
			break;
	}

	// Setup default zoom.
	if(type == Viewport::VIEW_PERSPECTIVE) {
		if(viewType() != Viewport::VIEW_PERSPECTIVE)
			setFieldOfView(DEFAULT_PERSPECTIVE_FIELD_OF_VIEW);
	}
	else {
		if(viewType() == Viewport::VIEW_PERSPECTIVE || viewType() == Viewport::VIEW_NONE)
			setFieldOfView(DEFAULT_ORTHOGONAL_FIELD_OF_VIEW);
	}

	_viewType = type;

	if(_viewport) {

		_viewport->_cameraView.validityInterval = TimeForever;
		_viewport->_cameraView.isPerspective = (type == Viewport::VIEW_PERSPECTIVE);

		// Load the caption string for the new view type.
		_viewport->updateViewportTitle();
		// Render the scene again.
		_viewport->updateViewport(true);
	}
}

/******************************************************************************
* Sets the orientation of the grid plane.
******************************************************************************/
void ViewportRecord::setGridMatrix(const AffineTransformation& gridmat)
{
	if(_viewport)
		_viewport->grid().setGridMatrix(gridmat);
}

/******************************************************************************
* Sets the shading mode to use for scene rendering.
******************************************************************************/
void ViewportRecord::setShadingMode(Viewport::ViewportShadingMode mode)
{
	if(shadingMode() == mode) return;
    _shadingMode = mode;

    // Do a complete scene update.
	if(_viewport) _viewport->updateViewport(true);
}

/******************************************************************************
* Returns the world to camera (view) transformation without projection.
******************************************************************************/
const AffineTransformation& ViewportRecord::viewMatrix() const
{
	if(_viewport) return _viewport->viewMatrix();
	return _viewMatrix;
}

/******************************************************************************
* Returns the camera (view) to world transformation without the projection part.
******************************************************************************/
AffineTransformation ViewportRecord::inverseViewMatrix() const
{
	if(_viewport) return _viewport->inverseViewMatrix();
	return _viewMatrix.inverse();
}

/******************************************************************************
* Sets the view matrix of the viewport.
******************************************************************************/
void ViewportRecord::setViewMatrix(const AffineTransformation& tm)
{
	_viewMatrix = tm;
	if(_viewport) _viewport->setViewMatrix(tm);
}

/******************************************************************************
* Returns the zoom or field of view.
******************************************************************************/
FloatType ViewportRecord::fieldOfView() const
{
	if(_viewport) return _viewport->fieldOfView();
	return _fieldOfView;
}

/******************************************************************************
* Sets the zoom of the viewport.
******************************************************************************/
void ViewportRecord::setFieldOfView(FloatType fov)
{
	_fieldOfView = fov;
	if(_viewport) _viewport->setFieldOfView(fov);
}

/******************************************************************************
* Returns a description the viewport's view at the given animation time.
******************************************************************************/
CameraViewDescription ViewportRecord::getViewDescription(TimeTicks time, FloatType aspectRatio, const Box3& bb)
{
	Box3 sceneBoundingBox = bb;
	if(sceneBoundingBox.isEmpty())
		sceneBoundingBox = DATASET_MANAGER.currentSet()->sceneRoot()->worldBoundingBox(time);

	CameraViewDescription d;
	d.validityInterval = TimeForever;
	d.aspectRatio = aspectRatio;

	// Get transformation from view scene node.
	if(viewType() == Viewport::VIEW_SCENENODE && viewNode()) {
		PipelineFlowState state = viewNode()->evalPipeline(time);
		AbstractCameraObject* camera = dynamic_object_cast<AbstractCameraObject>(state.result());
		if(camera) {
			// Get camera transformation.
			d.inverseViewMatrix = viewNode()->getWorldTransform(time, d.validityInterval);
			d.viewMatrix = d.inverseViewMatrix.inverse();
			// Calculate znear and zfar clipping plane distances.
			Box3 bb = sceneBoundingBox.transformed(d.viewMatrix);
			d.zfar = -bb.minc.Z;
			d.znear = max(-bb.maxc.Z, -bb.minc.Z * (FloatType)1e-5);
			camera->getCameraDescription(time, d);
		}
	}
	else if(viewType() == Viewport::VIEW_PERSPECTIVE) {
        // Get camera position.
		//Point3 camera = ORIGIN + inverseViewMatrix().getTranslation();
		// Enlarge bounding box to include camera.
		//Box3 bb = sceneBoundingBox;
		//bb.addPoint(camera);

		//d.zfar = Length(bb.size());
		//d.zfar = max(d.zfar, (FloatType)1e-5);
		//d.znear = d.zfar * 1e-5;

		Box3 bb = sceneBoundingBox.transformed(viewMatrix());
		if(bb.minc.Z < -1e-5) {
			d.zfar = -bb.minc.Z;
			d.znear = max(-bb.maxc.Z, -bb.minc.Z * (FloatType)1e-5);
		}
		else {
			d.zfar = 1000.0f;
			d.znear = 0.1f;
		}

		d.fieldOfView = fieldOfView();
		d.viewMatrix = viewMatrix();
		d.inverseViewMatrix = inverseViewMatrix();
		d.isPerspective = true;
		d.projectionMatrix = Matrix4::perspective(d.fieldOfView, 1.0/d.aspectRatio, d.znear, d.zfar);
		d.inverseProjectionMatrix = d.projectionMatrix.inverse();
	}
	else {
		// Transform scene to camera space.
		Box3 bb = sceneBoundingBox.transformed(viewMatrix());
		if(!bb.isEmpty()) {
			d.znear = -bb.maxc.Z;
			d.zfar  = max(-bb.minc.Z, d.znear + 1.0f);
		}
		else {
			d.znear = 1;
			d.zfar = 100;
		}
		d.viewMatrix = viewMatrix();
		d.inverseViewMatrix = inverseViewMatrix();
		d.isPerspective = false;
		d.fieldOfView = fieldOfView();
		d.projectionMatrix = Matrix4::ortho(-d.fieldOfView, d.fieldOfView,
							-d.fieldOfView*d.aspectRatio, d.fieldOfView*d.aspectRatio,
							d.znear, d.zfar);
		d.inverseProjectionMatrix = d.projectionMatrix.inverse();
	}
	return d;
}


/******************************************************************************
* This method is called when a reference target has sent a message.
******************************************************************************/
bool ViewportRecord::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == _viewNode && msg->type() == REFTARGET_CHANGED) {
		// Update viewport when camera has moved
		if(_viewport)
			_viewport->updateViewport(true);
	}
	else if(source == _viewNode && msg->type() == SCHEMATIC_TITLE_CHANGED) {
		// Update viewport title when camera has been renamed.
		if(_viewport) {
			_viewport->updateViewportTitle();
			_viewport->updateViewport();
		}
	}
	return RefMaker::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Is called when the value of a reference field of this RefMaker changes.
******************************************************************************/
void ViewportRecord::onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget)
{
	// Switch to perspective mode when camera has been deleted.
	if(newTarget == NULL && viewType() == Viewport::VIEW_SCENENODE) {
		setViewType(Viewport::VIEW_PERSPECTIVE);
	}
	else {
		// Update viewport when the camera has been replaced by another scene node.
		if(_viewport) {
			_viewport->updateViewportTitle();
			_viewport->updateViewport(true);
		}
	}

	RefMaker::onRefTargetReplaced(field, oldTarget, newTarget);
}

#define VIEWPORT_RECORD_FILE_FORMAT_VERSION		2

/******************************************************************************
* Saves the viewport record to the given stream.
******************************************************************************/
void ViewportRecord::saveToStream(ObjectSaveStream& stream)
{
	RefTarget::saveToStream(stream);
	stream.beginChunk(0x00ACACAC + VIEWPORT_RECORD_FILE_FORMAT_VERSION);
	stream.writeEnum(viewType());
	stream.writeEnum(shadingMode());
	stream << isGridShown();
	stream << fieldOfView();
	stream << viewMatrix();
	stream << renderFrameShown();
	stream << useOrbitCenter();
	stream << orbitCenter();
	stream.endChunk();
}

/******************************************************************************
* Loads the viewport record from the given stream.
******************************************************************************/
void ViewportRecord::loadFromStream(ObjectLoadStream& stream)
{
	RefTarget::loadFromStream(stream);
	int fileVersion = stream.expectChunkRange(0x00ACACAC, VIEWPORT_RECORD_FILE_FORMAT_VERSION);
	stream.readEnum(_viewType);
	stream.readEnum(_shadingMode);
	stream >> _showGrid;
	stream >> _fieldOfView;
	stream >> _viewMatrix;
	stream >> _showRenderFrame;
	if(fileVersion >= 2) {	// Added in version 0.9.2
		stream >> _useOrbitCenter;
		stream >> _orbitCenter;
	}
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr ViewportRecord::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	ViewportRecord::SmartPtr clone = static_object_cast<ViewportRecord>(RefTarget::clone(deepCopy, cloneHelper));

	clone->_viewType = this->viewType();
	clone->_shadingMode = this->shadingMode();
	clone->_showGrid = this->isGridShown();
	clone->_fieldOfView = this->fieldOfView();
	clone->_viewMatrix = this->viewMatrix();
	clone->_showRenderFrame = this->renderFrameShown();
	clone->_useOrbitCenter = this->useOrbitCenter();
	clone->_orbitCenter = this->orbitCenter();

	return clone;
}

/******************************************************************************
* The takes the current configuration of the viewports and
* saves it into this object.
******************************************************************************/
void ViewportConfiguration::saveConfiguration()
{
	if(APPLICATION_MANAGER.consoleMode())
		return;

	// Delete all records.
	_viewRecords.clear();
	_activeViewport = -1;
	_maximizedViewport = -1;

	CloneHelper cloneHelper;

	ViewportPanel* vpPanel = MAIN_FRAME->viewportPanel();
	int index = 0;
	Q_FOREACH(Viewport* vp, vpPanel->viewports()) {

		addViewport(cloneHelper.cloneObject(vp->settings(), false));
		if(vp == VIEWPORT_MANAGER.activeViewport())
			_activeViewport = index;
		if(vp == VIEWPORT_MANAGER.maximizedViewport())
			_maximizedViewport = index;

		index++;
	}
}

/******************************************************************************
* This applies the saved configuration to the viewports in the viewport panel.
******************************************************************************/
void ViewportConfiguration::restoreConfiguration()
{
	if(APPLICATION_MANAGER.consoleMode())
		return;

	ViewportPanel* vpPanel = MAIN_FRAME->viewportPanel();

	// Enlarge viewports array if neccessary.
	while(vpPanel->viewports().size() < records().size())
		vpPanel->addViewport();
	// Dispose any viewports that are too much.
	while(vpPanel->viewports().size() > records().size())
		vpPanel->removeViewport(vpPanel->viewports().back());

	VIEWPORT_MANAGER.setMaximizedViewport(NULL);
	VIEWPORT_MANAGER.setActiveViewport(NULL);

	for(int i=0; i<records().size(); i++) {
		Viewport* vp = vpPanel->viewports()[i];
		ViewportRecord* record = _viewRecords[i];
		vp->settings()->setViewType(record->viewType());
		vp->settings()->setShadingMode(record->shadingMode());
		vp->settings()->setGridShown(record->isGridShown());
		vp->setFieldOfView(record->fieldOfView());
		vp->setViewMatrix(record->viewMatrix());
		vp->settings()->setRenderFrameShown(record->renderFrameShown());
		vp->settings()->setViewNode(record->viewNode());
		vp->settings()->setUseOrbitCenter(record->useOrbitCenter());
		vp->settings()->setOrbitCenter(record->orbitCenter());
		if(i == _activeViewport)
			VIEWPORT_MANAGER.setActiveViewport(vp);
		if(i == _maximizedViewport)
			VIEWPORT_MANAGER.setMaximizedViewport(vp);
	}
}

/******************************************************************************
* Saves the viewport configuration to the given stream.
******************************************************************************/
void ViewportConfiguration::saveToStream(ObjectSaveStream& stream)
{
	RefTarget::saveToStream(stream);
	stream.beginChunk(0x00ABABAB);
	stream << _activeViewport;
	stream << _maximizedViewport;
	stream.endChunk();
}

/******************************************************************************
* Loads the viewport configuration from the given stream.
******************************************************************************/
void ViewportConfiguration::loadFromStream(ObjectLoadStream& stream)
{
	RefTarget::loadFromStream(stream);
	stream.expectChunk(0x00ABABAB);
	stream >> _activeViewport;
	stream >> _maximizedViewport;
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr ViewportConfiguration::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	ViewportConfiguration::SmartPtr clone = static_object_cast<ViewportConfiguration>(RefTarget::clone(deepCopy, cloneHelper));

	clone->_activeViewport = this->_activeViewport;
	clone->_maximizedViewport = this->_maximizedViewport;

	return clone;
}

};
