/*	Stage_Manager

PIRL CVS ID: Stage_Manager.java,v 1.55 2012/04/16 06:04:12 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Messenger.*;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Configuration.*;
import	PIRL.Utilities.UNIX_Process;

import	java.io.FileNotFoundException;
import	java.io.IOException;
import	java.io.BufferedReader;
import	java.io.InputStreamReader;
import	java.util.concurrent.BlockingQueue;
import	java.util.concurrent.LinkedBlockingQueue;
import	java.util.Iterator;
import	java.util.StringTokenizer;


/**	A <i>Stage_Manager</i> is a Messenger manager for communication
	between Conductor processes and client processes.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.55
	@see		Theater
	@see		Messenger
*/
public class Stage_Manager
	extends Dispatcher
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Stage_Manager (1.55 2012/04/16 06:04:12)";


/**	The name used by the base Dispatcher class for the Stage_Manager service.
<p>
	@see	Dispatcher#Dispatcher_Name(String)
*/
public static final String
	STAGE_MANAGER_NAME				= "Stage_Manager";


//	Configuration parameters:

/**	The name of the Configuration parameter that specifies the
	Conductor command name.
<p>
	If this parameter is not present or its value is empty or "disabled"
	remote Conductor command execution is disabled.
*/
public static final String
	CONDUCTOR_COMMAND_PARAMETER_NAME
		= "Conductor_Command_Name";

/**	The name of the Configuration parameter that specifies the maximum
	number of remote Conductor commands that may be executed by a single
	request.
<p>
	If the Count of remote commands exceeds this value no commands will
	be executed and an exception response will be sent in reply.
<p>
	@see	#CONDUCTOR_COMMAND_PARAMETER_NAME
*/
public static final String
	MAX_START_CONDUCTORS_COUNT_PARAMETER_NAME
		= "Max_Start_Conductors_Count";

/**	The default maximum number of remote Conductor commands that may be
	executed by a single request.
<p>
	@see	#MAX_START_CONDUCTORS_COUNT_PARAMETER_NAME
*/
public static final int
	DEFAULT_MAX_START_CONDUCTORS_COUNT	= 10;

private int
	Max_Start_Conductors_Count			= DEFAULT_MAX_START_CONDUCTORS_COUNT;


//	Message Actions and their parameters:
public static final String
	START_CONDUCTORS_ACTION			= "Start_Conductors",
		COUNT_PARAMETER_NAME			= "Count",
		EXPLANATION_PARAMETER_NAME		= Messenger.EXPLANATION_PARAMETER_NAME,
		COMMAND_PARAMETER_NAME			= "Command",
		REPORT_PARAMETER_NAME			= "Report",
		EXCEPTION_PARAMETER_NAME		= Messenger.EXCEPTION_PARAMETER_NAME;

private String
	Conductor_Command_Name			= null;

public static final String
	ASSISTANT_NAME					= "Assistant";

private Assistant
	Assistant						= null;
private Start_Conductors_Assistant
	Start_Conductors_Assistant		= null;

private final BlockingQueue<Message_Delivered_Event>
	Tasks
		= new LinkedBlockingQueue<Message_Delivered_Event> (),
	Start_Conductors_Tasks
		= new LinkedBlockingQueue<Message_Delivered_Event> ();


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 1,
	DEBUG_CONFIG				= 1 << 2,
	DEBUG_MESSAGES				= 1 << 3,
	DEBUG_ASSISTANT				= 1 << 4,
	DEBUG_START_CONDUCTORS		= 1 << 5,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Stage_Manager using a Configuration and port number.
<p>
	The base {@link Dispatcher} class is constructed using the
	{@link #STAGE_MANAGER_NAME} along with the specified Configuration
	and port number. The {@link Dispatcher#Dispatcher_ID} is set to the
	{@link #ID} of this class.
<p>
	@param	configuration	A Configuration. If null an attempt will
		be made to use a default configuration source.
	@param	port	The communications port number. If less than or equal
	to zero a {@link Dispatcher#DEFAULT_PORT} number will be used.
*/
public Stage_Manager
	(
	Configuration	configuration,
	int				port
	)
{
super (STAGE_MANAGER_NAME, configuration, port);
Dispatcher_ID = ID;
}

/**	Construct a Stage_Manager using a Configuration.
<p>
	The base Dispatcher class will provide a default communications port
	number.
<p>
	@param	configuration	A Configuration. If null an attempt will
		be made to use a default configuration source.
	@see	#Stage_Manager(Configuration, int)
*/
public Stage_Manager
	(
	Configuration	configuration
	)
{this (configuration, 0);}

/**	Construct a Stage_Manager.
<p>
	Defaults will be provided for the Configuration source and
	communications port number.
<p>
	@see	#Stage_Manager(Configuration, int)
*/
public Stage_Manager ()
{this (null, 0);}

/*==============================================================================
	Configuration
*/
/**	Configures the Stage_Manager.
<p>
	The Configuration class is set to use this class when using a
	class-relative search for a configuration source when a Configuration
	is not provided.
<p>
	The base Dispatcher class is {@link Dispatcher#Configure(Configuration)
	configured} and then Stage_Manager specific parameters are obtained.
<dl>
<dt>{@link #CONDUCTOR_COMMAND_PARAMETER_NAME}
	<dd>The name of the Conductor command used for remote execution
		requests. For best security this should be an absolute pathname
		to the executable file used to run a Conductor. If not present,
		empty or set to "DISABLED" remote commend execution is disabled.
<dt>{@link #MAX_START_CONDUCTORS_COUNT_PARAMETER_NAME}
	<dd>The maximum number of command executions allowed for each
		request. This is used to prevent accidental hordes of Conductors
		from being run. The default value is {@link
		#DEFAULT_MAX_START_CONDUCTORS_COUNT}; the minimum value is 1.
</dl><p>
	<b>N.B.</b>: All parameters are sought in the {@link
	#Config_Pathname(String) in the Configuration Group} with the {@link
	Dispatcher#Dispatcher_Name() Stage_Manager name}.
<p>
	@param	configuration	The Configuration to use. If null and
		defatult configuration source will sought.
	@throws	Configuration_Exception	If there was a problem parsing the
		configuration source, accessing its contents, or establishing
		a log file.
	@throws	SecurityException	If the connection authentication keys
		could not be generated, no {@link #PASSWORD_PARAMETER_NAME} was
		found but {@link Dispatcher#UNAUTHENTICATED_CONNECTIONS_ALLOWED}
		is false, or a log file could not be opened due to a security
		violation.
*/
protected void Configure
	(
	Configuration	configuration
	)
	throws	Configuration_Exception, SecurityException
{
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		(">>> Stage_Manager.Configure");
Configuration.Relative_to_Class (Stage_Manager.class);
super.Configure (configuration);

Conductor_Command_Name = The_Configuration.Get_One
	(Config_Pathname (CONDUCTOR_COMMAND_PARAMETER_NAME));
if (Conductor_Command_Name != null &&
	(Conductor_Command_Name.length () == 0 ||
	Conductor_Command_Name.equalsIgnoreCase ("DISABLED")))
	Conductor_Command_Name = null;

Max_Start_Conductors_Count = (int)The_Configuration.Get_Number
	(Config_Pathname (MAX_START_CONDUCTORS_COUNT_PARAMETER_NAME),
		DEFAULT_MAX_START_CONDUCTORS_COUNT);
if (Max_Start_Conductors_Count <= 0)
	Max_Start_Conductors_Count = 1;

if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println
		("<<< Stage_Manager.Configure");
}

/*==============================================================================
	Messages
*/
//	Message_Delivered_Listener
public synchronized void Message_Delivered
	(
	Message_Delivered_Event	event
	)
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Stage_Manager.Message_Delivered: " + event.Message.Action () + NL
		+"    From -" + NL
		+ (Messenger)event.getSource () + NL
		+ ((Messenger)event.getSource ()).Identity () + NL
		+"    Message -" + NL
		+ event.Message.Routing () + NL
		+ event.Message);
try {Tasks.put (event);}
catch (InterruptedException exception)
	{NACK ((Messenger)event.getSource (), event.Message);}
if (Assistant == null)
	{
	Assistant = new Assistant ();
	Assistant.start ();
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Stage_Manager.Message_Delivered");
}

/*==============================================================================
	Assistants
*/
private class Assistant
	extends Thread
{
public Assistant ()
{super (STAGE_MANAGER_NAME + ' ' + ASSISTANT_NAME);}


public void run ()
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("==> Stage_Manager.Assistant");
Message_Delivered_Event
	event;
Messenger
	messenger;
String
	action;
while (true)
	{
	try {event = Tasks.take ();}
	catch (InterruptedException exception) {break;}
	messenger = (Messenger)event.getSource ();
	action = event.Message.Action ();
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			(NL
			+"--> Stage_Manager.Assistant: " + action + NL
			+"    From -" + NL
			+ messenger + NL
			+ messenger.Identity () + NL
			+"    Message -" + NL
			+ event.Message.Routing () + NL
			+ event.Message);
	if (START_CONDUCTORS_ACTION.equals (action))
		{
		try {Start_Conductors_Tasks.put (event);}
		catch (InterruptedException exception)
			{NACK ((Messenger)event.getSource (), event.Message);}
		if (Start_Conductors_Assistant == null)
			{
			Start_Conductors_Assistant = new Start_Conductors_Assistant ();
			Start_Conductors_Assistant.start ();
			}
		}
	else
		Dispatch (event);
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			(NL
			+"<-- Stage_Manager.Assistant");
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<== Stage_Manager.Assistant quit");
Log (NL + "The Stage_Manager Assistant quit." + NL);
Assistant = null;
}
}	//	Assistant class.


private void Dispatch
	(
	Message_Delivered_Event	event
	)
{super.Message_Delivered (event);}


private class Start_Conductors_Assistant
	extends Thread
{
public Start_Conductors_Assistant ()
{super (STAGE_MANAGER_NAME + " Start_Conductors_" + ASSISTANT_NAME);}


public void run ()
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("==> Stage_Manager.Assistant");
Message_Delivered_Event
	event = null;
while (true)
	{
	try {event = Start_Conductors_Tasks.take ();}
	catch (InterruptedException exception) {break;}
	Start_Conductors ((Messenger)event.getSource (), event.Message);
	}
Log (NL + "The Stage_Manager Start_Conductors_Assistant quit." + NL);
Start_Conductors_Assistant = null;
}
}	//	Start_Conductors_Assistant class.

/*==============================================================================
	Message handlers
*/
private void Start_Conductors
	(
	Messenger	messenger,
	Message		message
	)
{
if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
	System.out.println
		(">>> Stage_Manager.Start_Conductors:" + NL
		+ messenger + NL
		+ message.Routing () + NL
		+ message);	
Message
	reply = Message.Action (START_CONDUCTORS_ACTION)
			.Reply_To (message);
if (Conductor_Command_Name == null)
	{
	//	Remote command execution disabled.
	Send (messenger,
		reply
		.Set (EXPLANATION_PARAMETER_NAME,
			"Remote command execution is not enabled."));
	if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
		System.out.println
			("    Remote command execution is not enabled." + NL
			+"<<< Stage_Manager.Start_Conductors");
	return;
	}

//	Confirm a valid command name.
String
	command_line = message.Get (COMMAND_PARAMETER_NAME);
if (command_line == null ||
	command_line.length () == 0)
	command_line = Conductor_Command_Name;
else
if (! command_line.equals (Conductor_Command_Name))
	{
	//	Invalid command name.
	Send (messenger,
		reply
		.Set (EXPLANATION_PARAMETER_NAME,
			"Invalid Conductor command name: " + command_line));
	if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
		System.out.println
			("    Invalid command name: " + command_line + NL
			+"<<< Stage_Manager.Start_Conductors");
	return;
	}

//	Assemble the command line from the message parameters.
int
	count = 1;
String
	argument;
boolean
	wait_to_start = false,
	manager = false;
Parameter
	parameter;
Iterator
	parameters = message.iterator ();
while (parameters.hasNext ())
	{
	//	Option name.
	parameter = (Parameter)parameters.next ();
	argument = parameter.Name ();
	if (argument.length () == 0 ||
		argument.equalsIgnoreCase (ACTION_PARAMETER_NAME) ||
		argument.equals (COMMAND_PARAMETER_NAME))
		continue;

	if (argument.equalsIgnoreCase (COUNT_PARAMETER_NAME))
		{
		try {count = (int)parameter.Value ().long_Data ();}
		catch (PVL_Exception exception)
			{/* Ignore this parameter */}
		continue;
		}

	if (argument.charAt (0) != '-')
		//	Add the option name marker.
		argument = "-" + argument;

	//	Check for the -Wait_to_start option.
	if (argument.toUpperCase ().startsWith ("-W"))
		wait_to_start = true;

	command_line += " " + argument;

	if (parameter.Has_Value ())
		{
		//	Option value.
		try {argument = parameter.Value ().String_Data ();}
		catch (PVL_Exception exception) {/* Already checked */}
		if (argument.length () != 0)
			{
			//	Check for a hidden -Manager option.
			StringTokenizer
				tokens = new StringTokenizer (argument);
			while (! manager &&
					tokens.hasMoreTokens ())
				if (tokens.nextToken ().toUpperCase ().startsWith ("-M"))
					manager = true;

			command_line += " " + argument;
			}
		}
	}

String
	report;
if (count <= 0)
	count = 1;
else
if (count > Max_Start_Conductors_Count)
	{
	String
		explanation =
		"Invalid Conductor command request.";
	report =
		"A Conductor command count of " + count
			+ " exceeds the maximum of " + Max_Start_Conductors_Count + '.';
	Log_Time
		(explanation + NL
		+ report);
	Send (messenger,
		reply
		.Set (EXPLANATION_PARAMETER_NAME, explanation)
		.Set (EXCEPTION_PARAMETER_NAME, report));
	if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
		System.out.println
			("    " + explanation + NL
			+"    " + report + NL
			+"<<< Stage_Manager.Start_Conductors");
	return;
	}

if (! wait_to_start &&
	manager)
	//	Add the wait-to-start option when a manager is also used.
	command_line += " -Wait_to_start";

reply.Set (COMMAND_PARAMETER_NAME, command_line);

//	Execute the command line.
Log_Time
	("Starting " + count + " Conductor"
		+ ((count == 1) ? "" : "s") + " -" + NL
	+ command_line);

//	Hold connection notifications in abeyance.
boolean
	report_messenger_connections = Report_Messenger_Connections ();
Report_Messenger_Connections (false);

report = "";
int
	total = count;
count = 0;
while (count++ < total)
	{
	if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
		System.out.println
			("    Starting Conductor " + count + " of " + total);
	String
		exception_identification = null;
	UNIX_Process
		procedure = null;
	try {procedure = new UNIX_Process
		(Runtime.getRuntime ().exec (command_line));}
	catch (IOException exception)
		{
		exception_identification = exception.toString ();
		report += NL
			+ "Unable to execute Conductor command -" + NL
			+ command_line + NL
			+ exception_identification;
		}
	catch (NoSuchFieldException exception)
		{
		exception_identification = exception.toString ();
		report += NL
			+ "The Process does not appear to be a UNIX Process." + NL
			+ exception_identification;
		}
	catch (SecurityException exception)
		{
		exception_identification = exception.toString ();
		report += NL
			+ "Unable to access the UNIX Process information." + NL
			+ exception_identification;
		}
	if (exception_identification != null)
		{
		Send (messenger,
			reply
			.Set (EXPLANATION_PARAMETER_NAME, report)
			.Set (EXCEPTION_PARAMETER_NAME, exception_identification));
		Log
			("Started " + --count + " of " + total
				+ " Conductor" + ((total == 1) ? "" : "s") + ':'
			+ report + NL);
		report = null;		//	Abort Conductor startups.
		break;
		}

	//	Check startup messages.
	String
		line = null,
		listing = "";
	BufferedReader
		procedure_stdout = null;
	try
		{
		procedure_stdout =
			new BufferedReader (
				new InputStreamReader (
					procedure.getInputStream ()));
		while ((line = procedure_stdout.readLine ()) != null)
			{
			if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
				System.out.println (line);
			if (line.equals (">>> READY <<<"))
				//	Startup was successful.
				break;
			listing += line + NL;
			}
		}
	catch (IOException except) {}
	try
		{
		if (procedure_stdout != null)
			procedure_stdout.close ();
		}
	catch (IOException except) {}

	if (line == null)
		{
		//	EOF from Conductor stdout.
		report += NL
			+ "Conductor start in the " + Location ()
				+ " theater failed with command -" + NL
			+ command_line;
		try
			{
			int
				status = procedure.waitFor ();
			report += NL
				+ "Conductor " + procedure.ID ()
					+ " exited with status " + status + ".";
			}
		catch (InterruptedException exception) {}

		reply
			.Set (EXPLANATION_PARAMETER_NAME, report)
			.Set (EXCEPTION_PARAMETER_NAME, listing);
		Log
			("Started " + --count + " of " + total
				+ " Conductor" + ((total == 1) ? "" : "s") + ':'
			+ report + NL
			+ listing);
		if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
			System.out.println
				("    " + report + NL
				+"    Conductor listing -" + NL
				+ listing);
		report = null;		//	Abort Conductor startups.
		break;
		}
	else
		report += NL
			+ "Started Conductor " + procedure.ID () + '.';
	}

if (report != null)
	{
	//	All Conductors started.
	Log_Time
		("Started " + total + " Conductor" + ((total == 1) ? "" : "s") + ':'
		+ report);
	reply.Set (REPORT_PARAMETER_NAME, report);
	}
Send (messenger, reply);

if (report_messenger_connections)
	{
	Report_Messenger_Connections (report_messenger_connections);
	Report_Messengers ();
	}

if ((DEBUG & DEBUG_START_CONDUCTORS) != 0)
	System.out.println
		("<<< Stage_Manager.Start_Conductors");
}

/*==============================================================================
	Application main
*/
public static void main
	(
	String[]	args
	)
{
System.out.println (ID + NL);

Default_Dispatcher_Name (STAGE_MANAGER_NAME);

String
	configuration_filename = null,
	log_filename = null,
	name = "";
int
	port = 0;

for (int count = 0;
	 count < args.length;
	 count++)
	{
	if (args[count].length () > 1 &&
		args[count].charAt (0) == '-')
		{
		switch (args[count].charAt (1))
			{
			case 'C':	//	Configuration
			case 'c':
				if (++count == args.length ||
					args[count].length () == 0 ||
					args[count].charAt (0) == '-')
					{
					System.out.println ("Missing configuration filename");
					Usage ();
					}
				if (configuration_filename != null)
					{
					System.out.println ("Multiple configuration files -" + NL
						+ configuration_filename + NL
						+ "and" + NL
						+ args[count]);
					Usage ();
					}
				configuration_filename = args[count];
				break;

			case 'L':	//	Logging
			case 'l':
				if (++count == args.length ||
					(args[count].length () > 1 &&
					 args[count].charAt (0) == '-'))
					{
					name = Default_Log_Filename ();
					--count;
					}
				else
					name = args[count];

				if (log_filename != null &&
					! log_filename.equals (name))
					{
					System.out.println ("Multiple log files -" + NL
						+ "    " + log_filename + NL
						+ "  and" + NL
						+ "    " + name);
					Usage ();
					}
				if (name.length () == 0)
					name = STDOUT;
				log_filename = name;
				break;

			case 'P':	//	Port
			case 'p':
				if (++count == args.length ||
					args[count].length () == 0 ||
					args[count].charAt (0) == '-')
					Usage ();
				try {port = Integer.parseInt (args[count]);}
				catch (NumberFormatException exception)
					{
					System.out.println ("Invalid port number: " + args[count]);
					Usage ();
					}
				break;

			default:
				System.out.println
					("Unknown argument: " + args[count]);

			case 'H':	//	Help
			case 'h':
				Usage ();
			}
		}
	else
		{
		System.out.println
			("Unknown argument: " + args[count]);
		Usage ();
		}
	}

Configuration
	configuration = null;
if (configuration_filename != null)
	{
	try {configuration = new Configuration (configuration_filename);}
	catch (IllegalArgumentException exception)
		{
		System.out.println ("Unable to find the configuration file - "
			+ configuration_filename);
		System.exit (EXIT_CONFIG_FILE_PROBLEM);
		}
	catch (Configuration_Exception exception)
		{
		System.out.println ("Could not load the configuration file - "
			+ configuration_filename + NL
			+ exception.getMessage ());
		System.exit (EXIT_CONFIG_FILE_PROBLEM);
		}
	}

Stage_Manager
	stage_manager = new Stage_Manager (configuration, port);

if (DEBUG != DEBUG_OFF)
	{
	if (log_filename == null)
		log_filename = STDOUT;
	}
if (log_filename != null)
	{
	try {System.out.println (stage_manager.Log_Pathname (log_filename));}
	catch (FileNotFoundException exception)
		{
		System.out.println (exception.getMessage ());
		System.exit (EXIT_NO_LOG_FILE);
		}
	stage_manager.Logging (true);
	}

try {stage_manager.Start ();}
catch (Configuration_Exception exception)
	{
	stage_manager.Log_Time ("Could not load the configuration." + NL
		+ exception.getMessage () + NL);
	System.exit (EXIT_CONFIG_FILE_PROBLEM);
	}

catch (IOException exception)
	{
	stage_manager.Log_Time ("Server I/O failure:" + NL
		+ exception.getMessage () + NL);
	System.exit (EXIT_IO_ERROR);
	}
catch (SecurityException exception)
	{
	stage_manager.Log_Time ("Server security violation:" + NL
		+ exception.getMessage () + NL);
	System.exit (EXIT_SECURITY_VIOLATION);
	}
catch (Exception exception)
	{
	stage_manager.Log_Time ("Server failure:");
	if (stage_manager.Logging ())
		exception.printStackTrace (stage_manager.Log_Stream ());
	stage_manager.Log ();
	System.exit (EXIT_UNKNOWN_EXCEPTION);
	}
}


public static void Usage ()
{
System.out.println 
	("Usage: " + Default_Dispatcher_Name () + " <Options>" + NL
	+"  Options -" + NL
	+"    -Configuration <source>"
		+ " (default: " + Default_Configuration_Source () + ')' + NL
	+"    -Logging [<pathname>]"
		+ " (default: " + Default_Log_Filename () + ')' + NL
	+"    -Port <number>"
		+ " (default: " + DEFAULT_PORT + ')' + NL
	+"    -Help" + NL
	);
System.exit (EXIT_COMMAND_LINE_SYNTAX);
}


}
