/*	Theater_List_Model

PIRL CVS ID: Theater_List_Model.java,v 1.29 2012/04/16 06:04:12 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Messenger.Messenger;
import	PIRL.Utilities.Host;

import	javax.swing.AbstractListModel;
import	javax.swing.MutableComboBoxModel;
import	java.util.List;
import	java.util.Vector;


/**	A <i>Theater_List_Model</i> extends the AbstractListModel to provide
	management of a list of Theater objects.
<p>
@author		Bradford Castalia - UA/PIRL
@version	1.29
*/
public class Theater_List_Model
	extends AbstractListModel
	implements MutableComboBoxModel
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Theater_List_Model (1.29 2012/04/16 06:04:12)";


/*	List of Theaters and their locations.

	All Locations should be non-null, unique Strings. Each location
	should be a fully qualified Theater.Location(). The Names - the
	Theater.Location(String) portion from the location - are provided
	for display. If the name is the same as some other name, all the
	identical names are reset to their fully qualified location
	versions.

	The Theaters are Theater objects that correspond to the
	Locations. A Location may have a null theater. Every non-null
	theater should have a Theater.Location () identical to its
	corresponding Locations entry.
*/
private Vector<String>
	Names					= new Vector<String> (),
	Locations				= new Vector<String> ();
private Vector<Theater>
	Theaters				= new Vector<Theater> ();

//	Supports ComboBoxModel interface {set,get}SelectedItem methods.
private String
	Selected_Location		= null,
	Previous_Location		= null;


private static final String
	NL						= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTOR		= 1 << 1,
	DEBUG_ADD				= 1 << 2,
	DEBUG_LOCATION			= 1 << 3,
	DEBUG_SELECTION			= 1 << 4,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	MutableComboBoxModel
*/
public void addElement
	(
	Object	element
	)
{
if (element == null)
	return;
if (element instanceof Theater)
	Add ((Theater)element);
else
if (element instanceof String)
	{
	String
		location = (String)element;
	if (Index (location) < 0)
		{
		//	New location.
		int
			index = getSize ();
		if (Location (null, location, index))
			fireContentsChanged (this, -1, -1);
		else
			fireIntervalAdded (this, index, index);
		}
	}
}


public void insertElementAt
	(
	Object	element,
	int		index
	)
{
if (element == null ||
	index < 0 ||
	index > getSize ())
	return;

Theater
	theater = null;
String
	location = null;
if (element instanceof Theater)
	{
	theater = (Theater)element;
	if (Index (theater) >= 0)
		//	The theater is already present.
		return;

	location = theater.Location ();
	}
else
if (element instanceof String)
	{
	location = (String)element;
	if (Index (location) >= 0)
		//	The location is already present.
		return;
	}

if (location != null)
	{
	Names.insertElementAt (null, index);
	Locations.insertElementAt (null, index);
	Theaters.insertElementAt (null, index);
	if (Location (theater, location, index))
		fireContentsChanged (this, -1, -1);
	else
		fireIntervalAdded (this, index, index);
	}
}


public void removeElement
	(
	Object	element
	)
{
if (element == null)
	return;

if (element instanceof Theater)
	Remove ((Theater)element);
else
if (element instanceof String)
	Remove (Index ((String)element));
}


public void removeElementAt
	(
	int		index
	)
{Remove (index);}

/*------------------------------------------------------------------------------
	ComboBoxModel
*/
/**	Set the currently selected item.
<p>
	If the item is null nothing is done.
<p>
	If the item is a Theater its {@link Theater#Location() location} is used.
	Otherwise the item must be a location String. The location is converted
	to the {@link Theater#Location(String) abbreviated location} form.
<p>
	If the item location is equal to the currently {@link #getSelectedItem()
	selected item location} nothing is done.
<p>
	The {@link #Previous_Location() previously selected item location} is
	set to the currently selected item location and the currently selected
	item location is then set to the new item location. All listeners are
	notified that the selected item has changed.
<p>
	@param	item	A Theater or location String.
*/
public void setSelectedItem
	(
	Object	item
	)
{
if (item == null)
	return;

String
	location = null;
if (item instanceof Theater)
	location = ((Theater)item).Location ();
else
if (item instanceof String)
	location = (String)item;
location = Theater.Location (location);
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		(">-< Theater_List_Model.setSelectedItem: " + location);

if (location != null &&
	! location.equals (Selected_Location))
	{
	Previous_Location = Selected_Location;
	Selected_Location = location;
	fireContentsChanged (this, -1, -1);
	}
}

/**	Get the currently selected item location.
<p>
	<b>N.B.</b>: The abbreviated form of the Theater location is returned.
<p>
	@return	The currently selected item location String. This will be null
		if, and only if, no item has yet been selected.
	@see	Theater#Location(String)
	@see	Theater#Full_Location(String)
*/
public Object getSelectedItem ()
{
if ((DEBUG & DEBUG_SELECTION) != 0)
	System.out.println
		(">-< Theater_List_Model.getSelectedItem: " + Selected_Location);
return Selected_Location;
}

/**	Get the previously selected item location.
<p>
	Each time the currently selected item is {@link #setSelectedItem(Object)
	updated} with a new value the previously selected item is set to the
	currently selected item before it is changed.
<p>
	<b>N.B.</b>: The abbreviated form of the Theater location is returned.
<p>
	@return	The previously selected item location String. This will always
		be different than the {@link #getSelectedItem() currently selected
		item location} unless both are null. This will be null until the
		second change of the currently selected item.
*/
public String Previous_Location ()
{return Previous_Location;}

/*------------------------------------------------------------------------------
	ListModel
*/
/**	Get the Theater name at the data model index.
<p>
	@param	index	A data model index.
	@return	The Theater name for the data model idex, If the index is not
		valid null is returned.
*/
public Object getElementAt
	(
	int		index
	)
{
if (index >= 0 &&
	index < Names.size ())
	return Names.get (index);
return null;
}

/**	Get the number of entries in the data model.
<p>
	@return	The total number of entries in the data model.
	@see	#Total_Opened()
*/
public int getSize ()
{return Theaters.size ();}

/*==============================================================================
	Accessors
*/
/**	Add a Theater to the data model.
<p>
	If a Theater at the {@link Theater#Location() location} of the specified
	Theater is already in the data model that entry is replaced. Otherwise
	a new entry for the specified Theater is added to the data model.
<p>
	If addition of the specified Theater resulted in a {@link
	#Location(Theater, String, int) location name change}, notification is
	sent to all listeners that the entire list may have changed;
	otherwise notification of a change to only the affected data model
	index is sent.
<p>
	@param	theater	A Theater to be added to the data model. If null nothing
		is done.
	@return	This Theater_List_Model.
*/
public Theater_List_Model Add
	(
	Theater	theater
	)
{
if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		(">>> Theater_List_Model.Add:" + NL
		+ theater);
if (theater == null)
	{
	if ((DEBUG & DEBUG_ADD) != 0)
		System.out.println
			("<<< Theater_List_Model.Add");
	return this;
	}

String
	location = theater.Location ();
int
	size = getSize (),
	index = Index_of (location);
if (index < 0)
	{
	//	New theater location.
	index = size;
	if ((DEBUG & DEBUG_ADD) != 0)
		System.out.println
			("    New Theater");
	}
if (Location (theater, location, index))
	fireContentsChanged (this, -1, -1);
else
if (index == size)
	fireIntervalAdded (this, index, index);
else
	fireContentsChanged (this, index, index);

if ((DEBUG & DEBUG_ADD) != 0)
	System.out.println
		("<<< Theater_List_Model.Add");
return this;
}

/**	Set a location at a data model index.
<p>
	If the theater is non-null the {@link Theater#Location() Theater
	location} overrides the specified location. Otherwise, if the
	specified location is non-null it is set to the {@link
	Theater#Full_Location(String) standard location format}. The {@link
	Theater#Location(String) abbreviated theater name} is obtained from
	the location.
<p>
	If the name is different from the location the names list is
	searched for names not at the specified index that match the new
	name. If a match is found a name changed flag is set and all
	matching names are reset to the location (the name is changed to the
	location). This ensures that names will always be unique.
<p>
	If the specified index equals the list size - the location is a
	new addition to the list - the name and location are added to their
	lists and a null place-holder entry is added to the Theaters list.
	Otherwise the location replaces an existing location and the name
	replaces the existing name.
<p>
	The name changed flag is returned as a signal that a notification
	should be sent that the entire list may have changed due to the name
	change; the change is not limited to the specified index.
<p>
	@param	theater		A Theater object. If null and the location is
		null nothing is done and false is returned.
	@param	location	A Theater location String. If null or empty the
		{@link Theater#Location() Theater location} is used.
	@param	index	A data model index where the location is to be set.
		If less than zero or greater than the list size nothing is done
		and false is returned.
	@return	true if the list of Theater names was changed - the change
		is not limited to the specified index; false otherwise.
*/
protected boolean Location
	(
	Theater	theater,
	String	location,
	int		index
	)
{
if ((DEBUG & DEBUG_LOCATION) != 0)
	System.out.println
		(">>> Theater_List_Model.Location:" + NL
		+"    theater -" + NL
		+ theater + NL
		+"    location - " + location + NL
		+"    index - " + index);
if (index < 0 ||
	index > getSize ())
	{
	if ((DEBUG & DEBUG_LOCATION) != 0)
	System.out.println
		("<<< Theater_List_Model.Location: false");
	return false;
	}

String
	name = Theater.Location (location);

//	Ensure that the standard location format is used.
if (theater != null)
	location = theater.Location ();
else
if ((location = Theater.Full_Location (location)) == null)
	{
	if ((DEBUG & DEBUG_LOCATION) != 0)
	System.out.println
		("<<< Theater_List_Model.Location: false");
	return false;
	}

if ((DEBUG & DEBUG_LOCATION) != 0)
	System.out.println
		("        Theater name: " + name + NL
		+"    Theater location: " + location);

//	Check for duplicate names.
boolean
	name_change = false;
if (! name.equals (location))
	{
	for (int
			name_index = 0;
			(name_index = Names.indexOf (name, name_index)) >= 0;
			name_index++)
		{
		if (name_index != index)
			{
			//	Duplicate name. Use the location.
			Names.set (name_index, location);
			name_change = true;
			}
		}
	if (name_change)
		{
		name = location;
		if ((DEBUG & DEBUG_LOCATION) != 0)
			System.out.println
				("    Duplicate name; using location");
		}
	}

if (index == getSize ())
	{
	if ((DEBUG & DEBUG_LOCATION) != 0)
		System.out.println
			("    Add new entries");
	Names.add (name);
	Locations.add (location);
	Theaters.add (theater);
	if (index == 0 &&
		Selected_Location == null)
		Selected_Location = name;
	}
else
	{
	if ((DEBUG & DEBUG_LOCATION) != 0)
		System.out.println
			("    Set existing entries");
	Names.set (index, name);
	Locations.set (index, location);
	Theaters.set (index, theater);
	}
if ((DEBUG & DEBUG_LOCATION) != 0)
	System.out.println
		("<<< Theater_List_Model.Location: " + name_change);
return name_change;
}
	

public Theater_List_Model Replace
	(
	List<Theater>	theaters
	)
{
Theater
	theater;
boolean
	changed = false;
int
	index,
	row = getSize ();
while (--row >= 0)
	{
	//	Find the current theater in the new theaters list.
	if ((theater = Theaters.get (row)) == null)
		//	No theater.
		continue;

	index = Matching_Index (theater, theaters);
	if (index < 0)
		{
		//	Obsolete: Current theater is not in the new list.
		Remove_at (row);
		changed = true;
		}
	else
		//	Redundant: Current theater is in the new list.
		theaters.remove (index);
	}

if (theaters.size () > 0)
	{
	//	What's left are new theaters.
	index = -1;
	row = theaters.size ();
	while (++index < row)
		Location (theaters.get (index), null, getSize ());
	changed = true;
	}	

if (changed)
	fireContentsChanged (this, -1, -1);
return this;
}


public Theater Remove
	(
	int		index
	)
{
Theater
	theater = null;
if (index >= 0 &&
	index < getSize ())
	{
	theater = Theaters.get (index);
	Remove_at (index);
	fireIntervalRemoved (this, index, index);
	}
return theater;
}


private void Remove_at
	(
	int		index
	)
{
if (Names.get (index).equals (Selected_Location))
	{
	if (index == 0)
		Selected_Location =
			(Locations.size () == 1) ? null : Names.get (1);
	else
		Selected_Location = Names.get (index - 1);
	}
Theaters.remove (index);
Locations.remove (index);
Names.remove (index);
}


public Theater Remove
	(
	Theater	theater
	)
{return Remove (Index (theater));}

/**	Clear the data model of all entries.
<p>
	<b>N.B.</b>: The Theaters are not {@link #Close(Theater) closed}.
<p>
	@see	#Close_All()
*/
public void Clear ()
{
Selected_Location = null;
if (getSize () > 0 )
	{
	int
		size = getSize () - 1;
	Theaters.clear ();
	Locations.clear ();
	Names.clear ();
	fireIntervalRemoved (this, 0, size);
	}
}

//------------------------------------------------------------------------------

public int Index
	(
	Theater	theater
	)
{
if (theater != null)
	{
	int
		index = getSize ();
	while (--index >= 0)
		if (theater == Theaters.get (index))
			return index;
	}
return -1;
}


public int Index
	(
	String	location
	)
{
if (location != null &&
	location.length () != 0)
	//	Ensure a standard location format.
	return Index_of (Theater.Full_Location (location));
return -1;
}


private int Index_of
	(
	//	Standard location format.
	String	location
	)
{
int
	index = Locations.size ();
while (--index >= 0)
	if (location.equals (Locations.get (index)))
		break;
return index;
}

/**	Get the index of a Theater in a list of Theaters that matches.
<p>
	Theaters match when their {@link Theater#Location() locations} are equal.
<p>
	@param	theater	A Theater to be matched.
	@param	theaters	A List of Theater objects.
	@return	The index of the first matching Theater in the list.
*/
protected static int Matching_Index
	(
	Theater			theater,
	List<Theater>	theaters
	)
{
if (theater == null ||
	theaters == null)
	return -1;

String
	location = theater.Location ();
int
	index = -1,
	size = theaters.size ();
while (++index < size)
	if ((theater = theaters.get (index)) != null &&
		theater.Location ().equals (location))
		return index;
return -1;
}


public Theater Theater
	(
	int		index
	)
{
if (index >= 0 &&
	index < getSize ())
	return Theaters.get (index);
return null;
}


public Theater Theater
	(
	String	location
	)
{return Theater (Index (location));}


public Theater Theater
	(
	Messenger	messenger
	)
{
if (messenger == null)
	return null;

Theater
	theater;
int
	index = -1,
	size = getSize ();
while (++index < size)
	if ((theater = Theaters.get (index)) != null &&
		theater.Messenger () == messenger)
		return theater;
return null;
}

/**	Get the Theater for a Messenger address.
<p>
	@param	messenger_address	A Messenger {@link Messenger#Address()
		address}.
	@return	The Theater for the Messenger address. This will be null the
		Messenger address is null or no Theater with a Messenger having
		the specified address can be found.
*/
public Theater Messenger_Theater
	(
	String	messenger_address
	)
{
if (messenger_address == null)
	return null;

Theater
	theater;
int
	index = -1,
	size = getSize ();
while (++index < size)
	if ((theater = Theaters.get (index)) != null &&
		theater.Messenger ().Address ().equals (messenger_address))
		return theater;
return null;
}

/**	Get the Theater location for a data model index.
<p>
	@param	index	A data model index.
	@return	The Theater location String for the data model index. This
		will be null if the index is invalid. <b>N.B.</b>: This will be
		the same as the {@link Theater#Location() Theater location} for the
		Theater at the same index if the index has a Theater; but since
		there may be no Theater for an index the locations list is used
		instead.
*/
public String Location
	(
	int		index
	)
{
if (index >= 0 &&
	index < getSize ())
	return Locations.get (index);
return null;
}

public String Location
	(
	Messenger	messenger
	)
{return Location (Theater (messenger));}


public String Location
	(
	String	messenger_address
	)
{return Location (Messenger_Theater (messenger_address));}


/**	Get the location for a Theater in the data model.
<p>
	@param	theater	A Theater.
	@return	The location from the data model list associated with the
		Theater. This will be null if the theater is not in the data model.
*/
public String Location
	(
	Theater	theater
	)
{
String
	location = null;
int
	index = Index (theater);
if (index >= 0)
	location = Locations.get (index);
return location;
}

/**	Set the maximum amount of time, in seconds, that will be used when
	waiting for a Theater protocol Message to be received.
<p>
	@param	timeout	The {@link Theater#Receive_Timeout(int) timeout
		value} to be applied to all Theaters in the data model.
*/
public void Receive_Timeout
	(
	int		timeout
	)
{
Theater
	theater;
for (int
		index = 0;
		index < getSize ();
		index++)
	{
	theater = Theaters.get (index);
	if (theater != null)
		theater.Receive_Timeout (timeout);
	}
}


public boolean Opened
	(
	int		index
	)
{
if (index >= 0 &&
	index < getSize ())
	{
	Theater
		theater = Theaters.get (index);
	return
		theater != null &&
		theater.Opened ();
	}
return false;
}


public boolean Opened
	(
	String	location
	)
{return Opened (Index (location));}


public int Total_Opened ()
{
Theater
	theater;
int
	total = 0,
	index = getSize ();
while (--index >= 0)
	if ((theater = Theaters.get (index)) != null &&
		 theater.Opened ())
		++total;
return index;
}


public void Update
	(
	int		index
	)
{fireContentsChanged (this, index, index);}


public void Update ()
{fireContentsChanged (this, 0, getSize () - 1);}

/**	Close a Theater at an index in the data model.
<p>
	If the Theater is closed (the return value is true) a {@link
	#fireContentsChanged(Object, int, int) change notification} is sent
	to all {@link #addListDataListener(ListDataListener) listeners}.
<p>
	@param	index	A data model index. If the index is invalid or no
		Theater is present at the index nothing is done.
	@return	true if an open theater was present at the index and it was
		closed; false otherwise.
*/
public boolean Close
	(
	int		index
	)
{
boolean
	closed = Close_Theater (Theater (index));
if (closed)
	fireContentsChanged (this, index, index);
return closed;
}

/**	Close a Theater registered in the data model.
<p>
	If the Theater is closed (the return value is true) a {@link
	#fireContentsChanged(Object, int, int) change notification} is sent
	to all {@link #addListDataListener(ListDataListener) listeners}.
<p>
	@param	theater	A Theater to be closed. <b>N.B.</b>: If the theater
		is not registered in this data model nothing is done.
	@return	true if the theater is in this data model and it was closed,
		having been found to be open; false otherwise.
*/
public boolean Close
	(
	Theater	theater
	)
{
boolean
	closed = false;
int
	index = Index (theater);
if (index >= 0)
	closed = Close_Theater (theater);
if (closed)
	fireContentsChanged (this, index, index);
return closed;
}

/**	Close all Theaters registered in the data model.
<p>
	If any Theater is closed (the return value is true) a {@link
	#fireContentsChanged(Object, int, int) change notification} is sent
	to all {@link #addListDataListener(ListDataListener) listeners}.
<p>
	@return	true if any Theater in the data model was found to be open and
		was closed; false if there was no Theater to be closed.
*/
public boolean Close_All ()
{
boolean
	closed = false;
int
	index = Theaters.size ();
while (--index >= 0)
	if (Close_Theater (Theaters.get (index)))
		closed = true;
if (closed)
	fireContentsChanged (this, -1, -1);
return closed;
}


private static boolean Close_Theater
	(
	Theater	theater
	)
{
if (theater != null)
	return theater.Close ();
return false;
}


}
