#!/usr/bin/python

"""Create power domains for IBM System X machines with PowerExecutive."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import popen2
import time
import discovery
import pwrkap_data
import util
import dircache

SYSFS_NODE_DIR = "/sys/devices/system/node/"

def cmp_meter(x, y):
	"""Compare two meters by sysfs name."""
	xi = x.inventory()
	yi = y.inventory()

	(xt, xd) = xi
	(yt, yd) = yi

	assert xt == "sysfsmeter" and yt == "sysfsmeter"

	xn = xd["name"][22:]
	yn = yd["name"][22:]

	xs = xn.find("/")
	ys = yn.find("/")
	assert xs >= 0 and ys >= 0

	return util.cmp_string_as_number(xn[:xs], yn[:ys])

def find_device_in_list(list, devname):
	for listdev in list:
		x = listdev.inventory()
		if devname == x[0]:
			return listdev
	return None

def find_domain_with_dev(domains, dev):
	for dom in domains:
		if dev in dom.devices:
			return dom
	return None

def ibm_aem_discover():
	"""Configure power domains on AEM systems."""
	global SYSFS_NODE_DIR

	def find_aem_energy_meters():
		"""Find AEM energy meters."""
		meters = []

		# energy2 meter is usually more accurate, but if it's
		# zero then it's not connected
		for meter in discovery.PWRKAP_ENERGY_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("sysfsmeter"):
				continue
			if not data["chip"] == "aem2":
				continue
			if data["name"].endswith("energy2_input"):
				if meter.read() != 0:
					meters.append(meter)

		if len(meters) > 0:
			return meters

		# Try for energy1 if there aren't any energy2s.
		for meter in discovery.PWRKAP_ENERGY_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("sysfsmeter"):
				continue
			if not data["chip"] == "aem2":
				continue
			if data["name"].endswith("energy1_input"):
				meters.append(meter)

		return meters

	def find_aem_power_meters():
		"""Find AEM power meters."""
		meters = []

		# power2 meter is usually more accurate, but if it's
		# zero then it's not connected
		for meter in discovery.PWRKAP_POWER_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("sysfsmeter"):
				continue
			if not data["chip"] == "aem2":
				continue
			if data["name"].endswith("power2_average"):
				if meter.read() != 0:
					meters.append(meter)

		if len(meters) > 0:
			return meters

		# Try for power1 if there aren't any power1s.
		for meter in discovery.PWRKAP_POWER_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("sysfsmeter"):
				continue
			if not data["chip"] == "aem2":
				continue
			if data["name"].endswith("power1_average"):
				meters.append(meter)

		return meters

	# Go look for meters
	emeters = find_aem_energy_meters()
	pmeters = find_aem_power_meters()
	if len(emeters) == 0 or len(pmeters) == 0:
		return

	emeters.sort(cmp = cmp_meter)
	pmeters.sort(cmp = cmp_meter)

	# Some AEM systems are actually NUMA systems.  For this to work,
	# we must grab node info from /sys/devices/system/node/nodeX/.
	# Each node tells us which CPUs are in that node; to find the
	# power/energy meter we'll simply sort them by /sys/class/hwmon/hwmonX
	# order and assign them to the nodes.
	# (This system should work for the non-NUMA systems too.)

	i = 0
	for node in dircache.listdir(SYSFS_NODE_DIR):
		if not node.startswith("node"):
			continue
		cpus = []
		emeter = emeters[i]
		pmeter = pmeters[i]
		dev_domains = []
		devs = []

		for cpu in dircache.listdir(SYSFS_NODE_DIR + node):
			if not cpu.startswith("cpu"):
				continue
			cpu_device = find_device_in_list(discovery.PWRKAP_DEVICES, cpu)
			if cpu_device == None:
				continue
			cpu_domain = find_domain_with_dev(discovery.PWRKAP_DEVICE_DOMAINS, cpu_device)
			assert cpu_domain != None
			for dev in cpu_domain.devices:
				discovery.PWRKAP_DEVICES.remove(dev)
				devs.append(dev)
			discovery.PWRKAP_DEVICE_DOMAINS.remove(cpu_domain)
			dev_domains.append(cpu_domain)

		discovery.PWRKAP_POWER_METERS.remove(pmeter)
		discovery.PWRKAP_ENERGY_METERS.remove(emeter)
		idomains = pwrkap_data.detect_idomains_for_devices(devs)
		pd = pwrkap_data.power_domain(dev_domains, idomains, pmeter, emeter, 1000)
		discovery.PWRKAP_POWER_DOMAINS.append(pd)
			
		i = i + 1

def ibm_simple_pex_discover():
	"""Configure power domain on PEx systems."""

	def find_ibmaem_energy_sensor():
		"""Return an appropriate ibmaem energy meter."""
		p2 = p1 = None

		for meter in discovery.PWRKAP_ENERGY_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("sysfsmeter"):
				continue
			if not data["chip"] == "aem2":
				continue
			if data["name"].endswith("energy2_input"):
				p2 = meter
			elif data["name"].endswith("energy1_input"):
				p1 = meter
		if p2 != None:
			return p2

		return p1

	def find_ibmaem_power_sensor():
		"""Return an appropriate ibmaem power meter."""
		p2 = p1 = None

		for meter in discovery.PWRKAP_POWER_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("sysfsmeter"):
				continue
			if not data["chip"] == "aem2":
				continue
			if data["name"].endswith("power2_average"):
				p2 = meter
			elif data["name"].endswith("power1_average"):
				p1 = meter
		if p2 != None:
			return p2

		return p1

	def find_ibmpex_sensor():
		"""Return an appropriate ibmpex power meter."""
		for meter in discovery.PWRKAP_POWER_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("sysfsmeter"):
				continue
			if not data["chip"] == "ibmpex":
				continue
			if not data["name"].endswith("power11_average"):
				continue
			return meter
		return None

	def find_ipmi_sensor():
		"""Return an appropriate ipmi power meter."""
		for meter in discovery.PWRKAP_POWER_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("ipmimeter"):
				continue
			if data["name"] == "AVG Power":
				return meter
			elif data["name"] == "AvgPwrIns1":
				return meter
		return None

	# First see if we can find an ibmaem meter
	pmeter = find_ibmaem_power_sensor()
	emeter = find_ibmaem_energy_sensor()

	# Then try to find an ibmpex power meter
	if pmeter == None:
		pmeter = find_ibmpex_sensor()

	# If none found, go for IPMI sensor
	if pmeter == None:
		pmeter = find_ipmi_sensor()

	# No meter?
	if pmeter == None:
		return

	form_domain_from_all_devices(pmeter, emeter)

def ibm_x3_discover():
	"""Configure power domain on simple X3 systems."""
	def find_ipmi_sensor():
		"""Return an appropriate ipmi power meter."""
		for meter in discovery.PWRKAP_POWER_METERS:
			(name, data) = meter.inventory()
			if not name.startswith("ipmimeter"):
				continue
			if not data["name"] == "RFG 1PS 220":
				continue
			return meter
		return None

	# If none found, go for IPMI sensor
	meter = find_ipmi_sensor()

	# No meter?
	if meter == None:
		return

	form_domain_from_all_devices(meter, None)

def form_domain_from_all_devices(pmeter, emeter):
	"""Assume system has one meter for all devices and create domain."""
	# No devices?
	if len(discovery.PWRKAP_DEVICE_DOMAINS) == 0:
		return

	# Assume all CPUs have identical power curves
	idomains = pwrkap_data.detect_idomains_for_devices(discovery.PWRKAP_DEVICES)

	# Take all device domains for this power domain
	domains = discovery.PWRKAP_DEVICE_DOMAINS

	# Remove all devices, domains, and meters that we intend to use.
	discovery.PWRKAP_DEVICES = []
	discovery.PWRKAP_DEVICE_DOMAINS = []
	discovery.PWRKAP_POWER_METERS.remove(pmeter)
	if emeter != None:
		discovery.PWRKAP_ENERGY_METERS.remove(emeter)

	# Create power domain
	pd = pwrkap_data.power_domain(domains, idomains, pmeter, emeter, 1000)
	discovery.PWRKAP_POWER_DOMAINS.append(pd)

known_systems = [
	(["IBM System x3650",
	"IBM System x3655",
	"IBM System x3755",
	"IBM System x3350",
	"BladeCenter LS21",
	"IBM eServer BladeCenter HS21",
	"IBM System x3550"], ibm_simple_pex_discover),
	(["eserver xSeries 366",
	"IBM x3850",
	"IBM x3800"], ibm_x3_discover),
	(["IBM 3850 M2 / x3950 M2"], ibm_aem_discover),
]

def ibm_system_discover():
	"""Configure power domains on IBM systems."""
	def get_dmi_system_name():
		"""Retrieve system name from DMI."""
		proc = popen2.Popen4("dmidecode -s system-product-name")
		input = proc.fromchild
		while proc.poll() == -1:
			time.sleep(0.1)
		return input.readline().strip()

	global known_systems
	# First determine if this is one of the known systems.
	sys_name = get_dmi_system_name()
	if sys_name == None:
		return

	sys_func = None
	for (sys_list, func) in known_systems:
		for sys in sys_list:
			if sys_name.startswith(sys):
				sys_func = func
				break
		if sys_func != None:
			break;
	if sys_func == None:
		return
	return func()

def ibm_init():
	"""Set up IBM system discovery functions."""
	discovery.PWRKAP_POWER_DOMAIN_DISCOVERY.append(ibm_system_discover)

ibm_init()
