
#include "Document.h"
#include "Utility.h"

#include "ThumbnailGenerator.h"


Glib::RefPtr<Gdk::Pixbuf> ThumbnailGenerator::defaultthumb_;
Glib::RefPtr<Gdk::Pixbuf> ThumbnailGenerator::thumbframe_;

ThumbnailGenerator &ThumbnailGenerator::instance ()
{
	static ThumbnailGenerator inst;

	return inst;
}

ThumbnailGenerator::ThumbnailGenerator ()
{
	Glib::signal_idle().connect(
		sigc::bind_return (sigc::mem_fun (this, &ThumbnailGenerator::run), false));
}


void ThumbnailGenerator::run ()
{
	Glib::Thread::create(
		sigc::mem_fun (*this, &ThumbnailGenerator::mainLoop), false);
}

void ThumbnailGenerator::mainLoop ()
{
	for (;;) {
		Glib::ustring file;

		bool gotJob = false;
		taskLock_.lock ();
		if (taskList_.size () > 0) {
			std::multimap<Glib::ustring, Document *>::iterator it = taskList_.begin ();
			std::pair<Glib::ustring, Document *> task = *it;

			file = task.first;
			gotJob = true;
		}

		taskLock_.unlock ();

		if (!gotJob) {
			sleep (1);
			continue;
		}
		
		Glib::RefPtr<Gdk::Pixbuf> result;
		gdk_threads_enter ();
		result = lookupThumb (file);
		gdk_threads_leave ();
		
		if (result) {
			gdk_threads_enter ();
			taskLock_.lock ();

			typedef std::multimap<Glib::ustring, Document*>::iterator Iterator;
			const std::pair<Iterator,Iterator> docs = taskList_.equal_range(file);
			for (Iterator i = docs.first; i!= docs.second; ++i) {
				Document *doc = i->second;
				doc->setThumbnail (result);
			}

			taskList_.erase (file);

			taskLock_.unlock ();
			gdk_threads_leave ();
			sleep (0.1);
		}
	}
}

Glib::RefPtr<Gdk::Pixbuf> ThumbnailGenerator::getThumbnailSynchronous (Glib::ustring const &file)
{
	return lookupThumb (file);
}

Glib::RefPtr<Gdk::Pixbuf> ThumbnailGenerator::lookupThumb (Glib::ustring const &file)
{
	Glib::RefPtr<Gdk::Pixbuf> thumbnail;
	GdkPixbuf* thumbnail_c;

	Glib::RefPtr<Gio::File> uri = Gio::File::create_for_uri (file);


	if (!file.empty () && Utility::uriIsFast (uri) && uri->query_exists()) {
		Glib::RefPtr<Gio::FileInfo> fileinfo = uri->query_info ();

		Glib::ustring thumbnail_path = fileinfo->get_attribute_as_string("thumbnail::path");
		//DEBUG("gio thumbnail path: '%1'", thumbnail_path);
		if (thumbnail_path != "") {
			thumbnail = Gdk::Pixbuf::create_from_file(thumbnail_path);
		}
	}

	// Disallow crazy sized thumbs
	int const minimumDimension = 8;

	if (!thumbnail || thumbnail->get_width() < minimumDimension || thumbnail->get_height() < minimumDimension) {
		if (defaultthumb_) {
			thumbnail = defaultthumb_;
		} else {
			thumbnail = Gdk::Pixbuf::create_from_file
				(Utility::findDataFile ("unknown-document.png"));
			float const desiredwidth = 64.0 + 9;
			int oldwidth = thumbnail->get_width ();
			int oldheight = thumbnail->get_height ();
			int newwidth = (int)desiredwidth;
			int newheight = (int)((float)oldheight * (desiredwidth / (float)oldwidth));
			thumbnail = thumbnail->scale_simple (
				newwidth, newheight, Gdk::INTERP_BILINEAR);
			defaultthumb_ = thumbnail;
		}
	} else {
		float const desiredwidth = 64.0;
		int oldwidth = thumbnail->get_width ();
		int oldheight = thumbnail->get_height ();
		int newwidth = (int)desiredwidth;
		int newheight = (int)((float)oldheight * (desiredwidth / (float)oldwidth));
		thumbnail = thumbnail->scale_simple (
			newwidth, newheight, Gdk::INTERP_BILINEAR);

		if (!thumbframe_) {
			thumbframe_ = Gdk::Pixbuf::create_from_file (
				Utility::findDataFile ("thumbnail_frame.png"));
		}

		int const left_offset = 3;
		int const top_offset = 3;
		int const right_offset = 6;
		int const bottom_offset = 6;
		thumbnail = Utility::eelEmbedImageInFrame (
			thumbnail, thumbframe_,
			left_offset, top_offset, right_offset, bottom_offset);
	}

	return thumbnail;
}

/*
 * Always called holding gdk_threads lock
 */
void ThumbnailGenerator::registerRequest (Glib::ustring const &file, Document *doc)
{
	taskLock_.lock ();
	taskList_.insert (std::pair<Glib::ustring, Document*>(file,doc));
	taskLock_.unlock ();
}

/*
 * Always called holding gdk_threads lock
 */
void ThumbnailGenerator::deregisterRequest (Document *doc)
{
	taskLock_.lock ();

	/* Erase exactly one entry which has its second field equal to doc */
	typedef std::multimap<Glib::ustring, Document*>::iterator Iterator;
	for (Iterator i = taskList_.begin(); i!= taskList_.end(); ++i) {
		if (i->second == doc) {
			taskList_.erase (i);
			taskLock_.unlock ();
			return;
		}
	}

	taskLock_.unlock ();
}


