///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

typedef Float (*function_type)(const point&);

struct list_type {
        string        name;
        function_type function;
};
Float monom_1  (const point& x) { return 1; }
Float monom_x  (const point& x) { return x[0]; }
Float monom_y  (const point& x) { return x[1]; }
Float monom_x2 (const point& x) { return x[0]*x[0]; }
Float monom_xy (const point& x) { return x[0]*x[1]; }
Float monom_y2 (const point& x) { return x[1]*x[1]; }

list_type xy_monom_list [] = {
        //   monom
        // id   fct
        {"1",   monom_1},
        {"x",   monom_x},
        {"y",   monom_y},
        {"x2",  monom_x2},
        {"xy",  monom_xy},
        {"y2",  monom_y2}
};
const unsigned int xy_monom_max = sizeof(xy_monom_list)/sizeof(list_type);

point e0 (const point& x) { return point(0,1); }
point e1 (const point& x) { return point(1,0); }

void usage()
{
      cerr << "form_on_locked_space: usage: form_div_div_tst"
	   << " {-Igeodir}*"
	   << " -|mesh[.geo]"
	   << " [-lock string]"
	   << " [-lock-u-component int]"
	   << " [-lock-v-component int]"
	   << " [-form string]" 
	   << endl << "\t"
	   << " [-approx-u string]"
	   << " [-approx-v string]"
	   << " [-u-component int]"
	   << " [-v-component int]"
	   << " [-u-monom string]"
	   << " [-v-monom string]"
	   << " [-coord string]"
	   << endl;
      cerr << "example:\n";
      cerr << "  form_on_locked_space square -lock top -lock-u-component 1 -lock-v-component -1 -form mass\n";
      cerr << "      -u-monom xy -v-monom x -u-component 0 -v-component 0 -coord rz\n";
      cerr << "lock_?_component with a negative value does not lock the component.";
      exit (1);
}
int main(int argc, char**argv)
{
    //
    // load geometry and options
    //
    geo omega;  
    string coord_sys = "cartesian";
    string lock_domain = "top";
    string approx_u = "P2";
    string approx_v = "P2";
    string form_name = "mass";
    bool mesh_done = false;
    int lock_u_component = 0;
    int lock_v_component = 0;
    size_t i_comp_u = 0;
    size_t i_comp_v = 0;
    string        u_monom_id = "";
    function_type u_monom_function = 0;
    unsigned int  u_monom_idx = 0;
    string        v_monom_id = "";
    function_type v_monom_function = 0;
    unsigned int  v_monom_idx = 0;

    if (argc <= 1) usage() ;

    for (int i = 1; i < argc; i++ ) {

      if (argv [i][0] == '-' && argv [i][1] == 'I')
	  append_dir_to_rheo_path (argv[i]+2) ;
      else if (strcmp(argv[i], "-coord") == 0)
	  coord_sys = argv[++i];
      else if (strcmp(argv[i], "-lock") == 0)
	  lock_domain = argv[++i];
      else if (strcmp(argv[i], "-form") == 0)
	  form_name = argv[++i];
      else if (strcmp(argv[i], "-approx-u") == 0)
	  approx_u = argv[++i];
      else if (strcmp(argv[i], "-approx-v") == 0)
	  approx_v = argv[++i];
      else if (strcmp(argv[i], "-lock-u-component") == 0)
	  lock_u_component = atoi(argv[++i]);
      else if (strcmp(argv[i], "-lock-v-component") == 0)
	  lock_v_component = atoi(argv[++i]);
      else if (strcmp(argv[i], "-u-component") == 0)
	  i_comp_u = atoi(argv[++i]);
      else if (strcmp(argv[i], "-v-component") == 0)
	  i_comp_v = atoi(argv[++i]);
      else if (strcmp(argv[i], "-u-monom") == 0) {
	  u_monom_id = argv[++i];
          for (unsigned int i = 0; i < xy_monom_max; i++) {
            if (u_monom_id == xy_monom_list [i].name) {
	      u_monom_function = xy_monom_list [i].function;
	      u_monom_idx = i;
            }
          }
      } else if (strcmp(argv[i], "-v-monom") == 0) {
	  v_monom_id = argv[++i];
          for (unsigned int i = 0; i < xy_monom_max; i++) {
            if (v_monom_id == xy_monom_list [i].name) {
	      v_monom_function = xy_monom_list [i].function;
	      v_monom_idx = i;
            }
          }
      } else if (strcmp(argv[i], "-") == 0) {
	  // input geo on standard input
	  if (mesh_done) usage() ;
	  cerr << " ! load geo on stdin" << endl ;
	  cin >> omega ;
	  mesh_done = true ;
      } else {
	  // input geo on file
	  if (mesh_done) usage() ;
	  cerr << " ! load " << argv[i] << endl ;
	  omega = geo(argv[i]);
	  mesh_done = true ;
      }
    }
    if (!mesh_done) {
	cerr << "form_div_div_tst: no mesh specified" << endl;
	usage() ;
    }
    omega.set_coordinate_system (coord_sys);
    if (!u_monom_function) {
	error_macro("invalid u-monom identifier: " << u_monom_id);
    }
    if (!v_monom_function) {
	error_macro("invalid v-monom identifier: " << v_monom_id);
    }
    cerr << "syscoord = " << omega.coordinate_system() << endl;
    cerr << "u_monom = " << u_monom_id << endl;
    cerr << "v_monom = " << v_monom_id << endl;
    space Uh(omega, approx_u,"vector");
    //Uh.block("left");
    space U0h = (space) Uh[0];
    space Vh(omega, approx_u,"vector");
    space V0h = (space) Vh[0];
    space ref_U0h(omega, approx_u);
    space ref_Uh = ref_U0h*ref_U0h;
    space ref_V0h(omega, approx_u);
    space ref_Vh = ref_V0h*ref_V0h;
    if (lock_u_component==0) Uh.lock_components(lock_domain,e0);
    else if (lock_u_component==1) Uh.lock_components(lock_domain,e1);
    if (lock_v_component==0) Vh.lock_components(lock_domain,e0);
    else if (lock_v_component==1) Vh.lock_components(lock_domain,e1);
    field ref_u_monom (ref_Uh, Float(0));
    ref_u_monom[i_comp_u] = interpolate (ref_U0h, u_monom_function);
    field ref_v_monom (ref_Vh, Float(0));
    ref_v_monom[i_comp_v] = interpolate (ref_V0h, v_monom_function);
    field u_monom (Uh, Float(0));
    u_monom[i_comp_u] = interpolate (U0h, u_monom_function);
    field v_monom (Vh, Float(0));
    v_monom[i_comp_v] = interpolate (V0h, v_monom_function);
    form  a (Uh,Vh,form_name,true);
    form  ref_a (ref_Uh,ref_Vh,form_name);
    u_monom.set_locked_boundaries();
    v_monom.set_locked_boundaries();
    Float result = a(u_monom, v_monom);
    Float expect = ref_a(ref_u_monom, ref_v_monom);
    string u_vec_monom_id;
    string v_vec_monom_id;
    if (i_comp_u == 0) u_vec_monom_id = u_monom_id + ",0";
    else               u_vec_monom_id = "0," + u_monom_id;
    if (i_comp_v == 0) v_vec_monom_id = v_monom_id + ",0";
    else               v_vec_monom_id = "0," + v_monom_id;
    cerr << setprecision(numeric_limits<Float>::digits10)
         << "a(omega," << approx_u << "[" << u_vec_monom_id << "]," << approx_v << "[" << v_vec_monom_id << "]) = " << double(result) << endl;
    Float tol = 1e-10;

    if (fabs(result - expect) <= tol) {
        cerr << "ok" << endl;
        return 0;
    } else {
        cerr << "but it was expected " << expect << endl;
	cerr << "and error = " << fabs(result - expect) << endl;
	cerr << "and tol = " << tol << endl;
        return 1;
    }
}
