///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/hazel_1d.h"
#include "rheolef/georep.h"
using namespace rheolef;
using namespace std;
void
hazel_1d::insert (pair<size_type,size_type>& v2e_ix, size_type K_idx)
{
  static const size_type none = numeric_limits<size_type>::max();
  if (v2e_ix.first  == none) { v2e_ix.first  = K_idx; return; }
  if (v2e_ix.second == none) { v2e_ix.second = K_idx; return; }
  error_macro ("unexpected 1d mesh connectivity arround element " << K_idx);
}
hazel_1d::hazel_1d (const georep& Ih)
 : vector<vector<size_type> >(),
   _interval()
{
  initialize(Ih);
}
void
hazel_1d::initialize (const georep& Ih)
{
  check_macro (Ih.map_dimension() == 1, "only 1D mesh here");
  // ------------------------------------
  // 1) initialize vertex2edge table
  // ------------------------------------
  size_type dimension = Ih.dimension(); // number of physical coordinates
  static const size_type none = numeric_limits<size_type>::max();
  pair<size_type,size_type> none_pair = make_pair(none,none);
  vector<pair<size_type,size_type> > vertex2edge (Ih.n_vertex(), none_pair);
  georep::const_iterator K = Ih.begin();
  georep::const_iterator_vertex x = Ih.begin_vertex();
  for (size_type k = 0; k < Ih.size(); k++) {
    insert (vertex2edge[K[k][0]], k);
    insert (vertex2edge[K[k][1]], k);
  }
  // sanity check
  for (size_type i = 0; i < vertex2edge.size(); i++) {
    check_macro (vertex2edge[i].first != none, "vertex " << i
	<< " has no elements");
  }
  // -------------------------------------
  // 2) build list of connected components
  // -------------------------------------
  list<list<size_type> > path_list;
  vector<bool> mark (Ih.n_vertex(), false);
  list<pair<Float,Float> > interval_list;
  for (size_type i = 0; i < Ih.n_vertex(); i++) {
    // search for an extremity of a connected component
    // i.e a boundary none that is included in only one element
    if (mark[i]) continue;
    if (dimension == 1 && vertex2edge[i].second != none) continue;
    path_list.push_back(list<size_type>());
    list<size_type>& path = *(path_list.rbegin());
    mark[i] = true;
    size_type k = vertex2edge[i].first;
    size_type j = i;
    do {
      size_type j_prec = j;
      j = (K[k][0] != j_prec) ? K[k][0] : K[k][1];
      path.push_back(k);
      if (mark[j]) {
	// closed surface
        break;
      }
      mark[j] = true;
      size_type k_prev = k;
      k = (vertex2edge[j].first != k_prev) ? vertex2edge[j].first : vertex2edge[j].second;
    } while (k != none);
    // path from i to j : ordered by increasing coordinate ?
    if (dimension == 1) {
      pair<Float,Float> boundary;
      if (x[i][0] < x[j][0]) {
        boundary = make_pair(x[i][0], x[j][0]);
      } else {
        boundary = make_pair(x[j][0], x[i][0]);
        path.reverse();
      }
      interval_list.push_back(boundary);
    }
  }
  // copy list<list> into hard-sized vector<vector> : save memory
  size_type n_component = path_list.size();
  vector<vector<size_type> >::resize(n_component);
  size_type i_component = 0;
  for (list<list<size_type> >::const_iterator p = path_list.begin(); p != path_list.end(); p++, i_component++) {
    const list<size_type>& path = *p;
    vector<size_type>& new_path = vector<vector<size_type> >::operator[](i_component);
    new_path.resize(path.size());
    copy (path.begin(), path.end(), new_path.begin());
  }
  // copy also interval list into hard-sized version (when dimension = 1)
  if (dimension == 1) {
    _interval.resize(n_component);
    copy (interval_list.begin(), interval_list.end(), _interval.begin());
  }
}
hazel_1d::size_type
hazel_1d::localize (const georep& Ih, const Float& x0) const
{
  check_macro (_interval.size() != 0, "cannot localize in a " << Ih.dimension() 
	<< "D line path");
  // step 1 : find connex component containing x0
  static const size_type none = numeric_limits<size_type>::max();
  size_type i0 = none;
  for (size_type i = 0; i < _interval.size(); i++) {
    if (_interval[i].first <= x0 && _interval[i].second >= x0) {
      i0 = i;
      break;
    }
  }
  if (i0 == none) return none;
  const vector<size_type>& component = operator[](i0);
  size_type n = component.size();
  // step 2 : find element containing x0 in component
  // by using dichotomy on elements
  georep::const_iterator K = Ih.begin();
  georep::const_iterator_vertex x = Ih.begin_vertex();
  size_type kmin = 0;
  size_type kmax = n;
  size_type k = 0;
  while (kmin+1 < kmax) {
    k = size_type((kmin + kmax)/2.0);
    Float xk = min(x[K[k][0]][0],x[K[k][1]][0]);
    if (x0 >= xk) kmin = k;
    else          kmax = k;
  }
  k = kmin;
  // K[k] contains x
#undef PARANO
#ifdef PARANO
  Float x1 = min(x[K[k][0]][0],x[K[k][1]][0]);
  Float x2 = max(x[K[k][0]][0],x[K[k][1]][0]);
  check_macro (x1 <= x0 && x0 <= x2, "locate error");
#endif // PARANO
  return k;
}
bool
hazel_1d::localize_nearest (const georep& Ih, const Float& x0,
	Float& y0, size_type& k) const
{
  static const size_type none = numeric_limits<size_type>::max();
  k = localize (Ih, x0);
  if (k != none) { y0 = x0; return true; }
  // loop on intervals and search closest boundary
  size_type i0 = none;
  Float dist_min = numeric_limits<Float>::max();
  size_t extremity = 0;
  for (size_type i = 0; i < _interval.size(); i++) {
    Float d1 = fabs(x0 - _interval[i].first);
    if (d1 < dist_min) {
      y0 = _interval[i].first;
      i0 = i;
      extremity = 0;
      dist_min = d1;
    }
    Float d2 = fabs(x0 - _interval[i].second);
    if (d2 < dist_min) {
      y0 = _interval[i].second;
      i0 = i;
      extremity = 1;
      dist_min = d2;
    }
  }
  const vector<size_type>& component = operator[](i0);
  if (extremity == 0) {
    k = component[0];
  } else {
    k = component[component.size()-1];
  }
  return false;
}
// trace x0+v :
// return true if x0+v is inside the mesh
// otherwise return nearest element containing x,
// also t such that x=x0+t*v (for the method of characteristic)
bool
hazel_1d::trace (const georep& Ih, const Float& x0, const Float& v, Float& x, Float& t, size_type& k) const 
{
  static const size_type none = numeric_limits<size_type>::max();
  bool is_inside = localize_nearest (Ih, x0+v, x, k);
  if (is_inside) {
    t = 1;
  } else {
    t = (1+v != 1) ? (x-x0)/v : 0;
  }
  return is_inside;
}
