#!/usr/bin/env python

__authors__ = "Martin Sandve Alnes"
__date__ = "2008-09-09 2008-09-10"

import unittest
import os, sys, glob, shutil, commands
import swiginac
import ufl

from ufl import FiniteElement
from ufl import VectorElement
from ufl import TensorElement
from ufl import MixedElement

from ufl import Argument
from ufl import TestFunction
from ufl import TrialFunction

from ufl import Coefficient
from ufl import Constant

from ufl import dx, ds

import SyFi
import sfc as sfc
from sfc.representation import ElementRepresentation
from sfc.quadrature import find_quadrature_rule

import instant
instant.set_logging_level("warning")



class ElementRepresentationTest(unittest.TestCase):
    def __init__(self, *args, **kwargs):
        unittest.TestCase.__init__(self, *args, **kwargs)
    
    def setUp(self):
        SyFi.initSyFi(2)
        polygon = "triangle"
        
        self.P1 = FiniteElement("CG", polygon, 1)
        self.P2 = FiniteElement("CG", polygon, 2)
        self.Q  = FiniteElement("Q",  polygon, 1)
        self.VQ = VectorElement("Q",  polygon, 1)
        self.V1 = VectorElement("CG", polygon, 1)
        self.V2 = VectorElement("CG", polygon, 2)
        self.T1 = TensorElement("CG", polygon, 1, symmetry=True)
        self.T2 = TensorElement("CG", polygon, 2, symmetry=True)
        self.TH = self.V2 * self.P1
        self.M  = MixedElement(self.T1, self.V1, self.P1)
        
        self.P1rep = ElementRepresentation(self.P1)
        self.P2rep = ElementRepresentation(self.P2)
        self.Qrep  = ElementRepresentation(self.Q, quad_rule = find_quadrature_rule(polygon, 2))
        self.VQrep = ElementRepresentation(self.VQ, quad_rule = find_quadrature_rule(polygon, 2))
        self.V1rep = ElementRepresentation(self.V1)
        self.V2rep = ElementRepresentation(self.V2)
        self.T1rep = ElementRepresentation(self.T1)
        self.T2rep = ElementRepresentation(self.T2)
        self.THrep = ElementRepresentation(self.TH)
        self.Mrep  = ElementRepresentation(self.M)
        
        self.reps = [getattr(self, d) for d in dir(self) if isinstance(getattr(self, d), ElementRepresentation)]
    
    def tearDown(self):
        pass
    
    def testSetup(self):
        pass

    def testValueShapes(self):
        self.assertTrue( self.P1rep.value_shape == () )
        self.assertTrue( self.P2rep.value_shape == () )
        self.assertTrue( self.Qrep.value_shape  == () )
        self.assertTrue( self.VQrep.value_shape == (2,) )
        self.assertTrue( self.V1rep.value_shape == (2,) )
        self.assertTrue( self.V2rep.value_shape == (2,) )
        self.assertTrue( self.T1rep.value_shape == (2,2) )
        self.assertTrue( self.T2rep.value_shape == (2,2) )
        self.assertTrue( self.THrep.value_shape == (2+1,) )
        self.assertTrue( self.Mrep.value_shape  == (4+2+1,) )

    def testGeometryDimensions(self):
        self.assertTrue( self.P1rep.geometric_dimension == 2 )
        self.assertTrue( self.P2rep.geometric_dimension == 2 )
        self.assertTrue( self.Qrep.geometric_dimension  == 2 )
        self.assertTrue( self.VQrep.geometric_dimension == 2 )
        self.assertTrue( self.V1rep.geometric_dimension == 2 )
        self.assertTrue( self.V2rep.geometric_dimension == 2 )
        self.assertTrue( self.T1rep.geometric_dimension == 2 )
        self.assertTrue( self.T2rep.geometric_dimension == 2 )
        self.assertTrue( self.THrep.geometric_dimension == 2 )
        self.assertTrue( self.Mrep.geometric_dimension  == 2 )
    
    def testTopologyDimensions(self):
        self.assertTrue( self.P1rep.topological_dimension == 2 )
        self.assertTrue( self.P2rep.topological_dimension == 2 )
        self.assertTrue( self.Qrep.topological_dimension  == 2 )
        self.assertTrue( self.VQrep.topological_dimension == 2 )
        self.assertTrue( self.V1rep.topological_dimension == 2 )
        self.assertTrue( self.V2rep.topological_dimension == 2 )
        self.assertTrue( self.T1rep.topological_dimension == 2 )
        self.assertTrue( self.T2rep.topological_dimension == 2 )
        self.assertTrue( self.THrep.topological_dimension == 2 )
        self.assertTrue( self.Mrep.topological_dimension  == 2 )
    
    def testArguments(self):
        # Tests that for each basisfunction there is at least one nonzero value component
        #for rep in (self.P1rep, self.V1rep, self.T1rep):
        #for rep in (self.V1rep, ):
        for rep in self.reps:
            if rep.ufl_element.family() == "Quadrature"\
               or rep.ufl_element.family() == "Boundary Quadrature":
                continue
            #print 
            #print rep
            for i in range(rep.local_dimension):
                zeros = 0
                for c in ufl.permutation.compute_indices(rep.value_shape):
                    #print "Calling basis_function(", i, c, ") for element ", repr(rep.ufl_element)
                    N = rep.basis_function(i, c)
                    #print i, c, N
                    if N == 0.0:
                        zeros += 1
                self.assertTrue(zeros < rep.value_size)
                if rep.local_dimension > 1 and rep.value_size > 1:
                    self.assertTrue(zeros > 0) # This should hold for compositions of scalar elements
        # FIXME: Test something more
    
    def testDofTopology(self):
        def triangle_entities():
            for d in range(2):
                for i in range((3,3,1)[d]):
                    yield (d,i)
        for rep in self.reps:
            for (d,i) in triangle_entities():
                dofs = rep.entity_dofs[d][i]
                self.assertTrue(len(dofs) == rep.num_entity_dofs[d])
        # FIXME: Test something more
    
    def testPointEvaluation(self):
        # Tests that the property { N_k(\xi_j) == \delta_{kj} } holds for scalar elements.
        for rep in (self.P1rep, self.P2rep):
            for i in range(rep.local_dimension):
                x  = rep.dof_x[i]
                xi = rep.dof_xi[i]
                
                repmap = swiginac.exmap()
                for j in range(2):
                    repmap[rep.p[j]] = xi[j]
                
                for k in range(rep.local_dimension):
                    c = () # Assuming scalar element here:
                    N = rep.basis_function(k, c)
                    Nxi = N.subs(repmap)
                    if i == k:
                        self.assertTrue(Nxi == 1.0)
                    else:
                        self.assertTrue(Nxi == 0.0)
    
    def testCoordinates(self):
        for rep in self.reps:
            #print
            #print rep.ufl_element
            for i in range(rep.local_dimension):
                x  = rep.dof_x[i]
                xi = rep.dof_xi[i]
                #print i, xi
                #x2 = rep.xi_to_x(xi)
                #xi2 = rep.x_to_xi(x)
                #self.assertTrue(x2 == x)
                #self.assertTrue(xi2 == xi)
        # FIXME: Test something more
    
    def testSubHierarchy(self):
        self.assertTrue(len(self.P1rep.sub_elements) == 0 )
        self.assertTrue(len(self.P2rep.sub_elements) == 0 )
        self.assertTrue(len(self.Qrep.sub_elements)  == 0 )
        self.assertTrue(len(self.VQrep.sub_elements) == 2 )
        self.assertTrue(len(self.V1rep.sub_elements) == 2 )
        self.assertTrue(len(self.V2rep.sub_elements) == 2 )
        self.assertTrue(len(self.T1rep.sub_elements) == 3 )
        self.assertTrue(len(self.T2rep.sub_elements) == 3 )
        self.assertTrue(len(self.THrep.sub_elements) == 2 )
        self.assertTrue(len(self.Mrep.sub_elements)  == 3 )


tests = [ElementRepresentationTest]

if __name__ == "__main__":
    unittest.main()

