/*******************************************************************************
 * Copyright (C) 2004-2008 Intel Corp. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       LocalAgent.cpp
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <cstdio>
#include <cstdlib>
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>

#include "ZTCLocalAgent.h"
#include "PTHIDisplay.h"
#include "HECILinux.h"
#include "resource.h"
#include "version.h"
#include "Utils.h"
#include "Types.h"
#include "HECI_if.h"
#include "StringTableUtils.h"
#include "FWULCommand.h"
#ifdef PACKAGE_DATA_DIR
#define STATUS_STRINGS_FILE PACKAGE_DATA_DIR "/StatusStrings.dat"
#define ZTC_LOCALAGENT_STRINGS_FILE PACKAGE_DATA_DIR "/ZTCLocalAgent.dat"
#else
#define STATUS_STRINGS_FILE "StatusStrings.dat"
#define ZTC_LOCALAGENT_STRINGS_FILE "ZTCLocalAgent.dat"
#endif

static bool _verbose = false;

static int ztc_lock_file_fd = -1;
#define ZTC_LOCK_FILE "/tmp/ztcla.lock"


void ztc_exitcleanup()
{
	if (-1 != ztc_lock_file_fd) {
		close(ztc_lock_file_fd);
		ztc_lock_file_fd = -1;
		unlink(ZTC_LOCK_FILE);
	}
}

int ztc_lock_file(const char *lockfile)
{
	int lfp = -1;
	int error = 0;

	/* Create the lock file as the current user */
	if (lockfile && lockfile[0]) {
		lfp = open(lockfile, O_RDWR | O_CREAT, 0644);
		if (lfp < 0) {
			return -1;
		}
		if (-1 == flock(lfp, LOCK_EX | LOCK_NB)) {
			error = errno;
			if (EWOULDBLOCK == errno) {
				close(lfp);
			} else {
				close(lfp);
				unlink(lockfile);
				return -2;
			}
			return -1;
		}
		ztc_lock_file_fd = lfp;
	}
	return 0;
}


#define _CFCON_AMT_DISABLED 0
#define _CFCON_AMT_ENABLED  1
#define _CFCON_AMT_UNKNOWN  2
#define _CFCON_AMT_AT3      3
int isAMTEnabled(PTHICommand &command, bool verbose, bool useOpenPTHI = false)
{
	int ret = _CFCON_AMT_UNKNOWN;

	if (useOpenPTHI) {
		CODE_VERSIONS ver;

		AMT_STATUS ast = command.GetCodeVersions(ver);
		if (PTHI_STATUS_EMPTY_RESPONSE == ast) {
			return _CFCON_AMT_DISABLED;
		}
		if (AMT_STATUS_SUCCESS == ast) {
			return _CFCON_AMT_ENABLED;
		}
	} else {
		FWU_GET_VERSION_MSG_REPLY verMsg;
		FWU_GET_INFO_MSG_REPLY infoMsg;
		HECI_STATUS heciRet;
		FWULCommand _fwulCommand(verbose);

		if (_fwulCommand.FWULClient.Init()) {
			heciRet = _fwulCommand.GetFWUVersionAndInfo(verMsg, infoMsg);
			_fwulCommand.FWULClient.Deinit();
			if (HECI_STATUS_OK == heciRet) {
				if (infoMsg.MessageType == FWU_GET_INFO_REPLY) {
					return ((MEFWCAPS_MANAGEABILITY_SUPP_AMT
					     == infoMsg.ManageabilityMode)
					    ? _CFCON_AMT_ENABLED
					    : _CFCON_AMT_DISABLED);
				}
				ret = _CFCON_AMT_AT3;
			}
		}
	}

	return ret;
}


/*
* This is a main entry point
*/
int main(UINT32 argc, CHAR *argv[])
{
	AMT_STATUS 	status;
	LOCAL_AGENT_PARAMS param;
	ZtcLocalState   lState;

	StringTableUtils::StringTableInit(STATUS_STRINGS_FILE, TRUE);
	StringTableUtils::StringTableInit(ZTC_LOCALAGENT_STRINGS_FILE, FALSE);

	status = Parse(argc, argv, &param);
	if (AMT_STATUS_SUCCESS != status) {
		PrintHelp();
		return status;
	}
	_verbose = param.Verbose;

	// Print Local Agent version
	DisplayLocalAgentVersion();

	if (true == param.Activate) {
		int lockresult = ztc_lock_file(ZTC_LOCK_FILE);
		if (-2 == lockresult) {
			lockresult = ztc_lock_file(ZTC_LOCK_FILE);
		}
		if (0 != lockresult) {
			DisplayErrorMessage(ZTC_ALREADY_STARTED);
			return status;
		}
		atexit(ztc_exitcleanup);
	}

	PTHICommand command(_verbose, 3000);
	status = DiscoveryTest(command, param, lState);
	if ((AMT_STATUS_SUCCESS == status) && (true == param.Activate)) {
		status = Activate(command, param, lState);
		PTHIDisplay::DisplayConfigActivate(status);
	}
	command.PTHIClient.Deinit();

	if (AMT_STATUS_SUCCESS != status) {
		DisplayStatusString(status);
		return status;
	}
	return status;
}

/*
* Change SKU to AMT
* ReInit PTHI connection
* and report errors
*/
static UINT32 TryChangeToAMT(PTHICommand &command)
{
	command.PTHIClient.Deinit();
	if (true != ChangeToAMT()) {
		DisplayErrorMessage(CHANGE_TO_AMT_FAILURE);
		return AMT_STATUS_INTERNAL_ERROR;
	}
	DisplayMessage(WORD_CHANGE_TO_AMT);
	DisplayMessage(WORD_SUCCESS);
	if (true != ReInitPTHI(command)) {
		DisplayErrorMessage(HECI_CONNECT_TO_PTHI_CLIENT_FAILURE);
		return AMT_STATUS_INTERNAL_ERROR;
	}
	return AMT_STATUS_SUCCESS;
}

/*
* Calls to DiscoveryTest for check AMT configuration.
* Arguments:
*	isActivate - if this test running for start amt configuration
* Return values:
*	AMT_STATUS_SUCCESS - on success for get amt configuration data
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
UINT32 DiscoveryTest(PTHICommand &command, LOCAL_AGENT_PARAMS param,
		     ZtcLocalState &lState)
{
	AMT_STATUS status = AMT_STATUS_SUCCESS;
	CODE_VERSIONS CodeVersions;
	AMT_HASH_HANDLES HashHandles;
	AMT_PROVISIONING_STATE provstate;
	CFG_PROVISIONING_MODE mode;
	AMT_BOOLEAN legacy;
	HECI_VERSION hver;
	bool isReinit = false;
	bool isActivate = param.Activate;
	bool amtEnabled = false;

	lState.newCode = false;
	lState.mayBeStarted = false;

	if (true != command.PTHIClient.Init()) {
		if (true != command.PTHIClient.GetHeciVersion(hver)) {
			DisplayErrorMessage(OPEN_HECI_FAILURE);
			status = AMT_STATUS_INTERNAL_ERROR;
			return status;
		}
	} else {
		amtEnabled = (isAMTEnabled(command, param.Verbose, true) == _CFCON_AMT_ENABLED);
	}

	if (!amtEnabled) {
		DisplayErrorMessage(HECI_CONNECT_TO_PTHI_CLIENT_FAILURE);
		if (true != isActivate) {
			status = AMT_STATUS_INVALID_AMT_MODE;
			return status;
		}
		status = TryChangeToAMT(command);
		if (AMT_STATUS_SUCCESS != status) {
			return status;
		}
		lState.mayBeStarted = true;
	}

	status = command.GetCodeVersions(CodeVersions);
	if (AMT_STATUS_SUCCESS != status) {
		return status;
	}
	PTHIDisplay::DisplayCodeVersions(CodeVersions);

	// provisioning state
	status = command.GetProvisioningState(provstate);
	if (AMT_STATUS_SUCCESS != status) {
		return status;
	}
	PTHIDisplay::DisplayProvisioningState(provstate);

	lState.newCode = IsNewCode(CodeVersions);

	if (((true != lState.newCode) || (true != lState.mayBeStarted))
	    && (true == isActivate)
	    && (PROVISIONING_STATE_PRE != provstate)) {
		status = AMT_STATUS_INVALID_AMT_MODE;
		return status;
	}

	// provisioning  mode
	status = command.GetProvisioningMode(mode, legacy);
	if (AMT_STATUS_SUCCESS != status) {
		return status;
	}

	if ((PROVISIONING_STATE_PRE == provstate) &&
	    (CFG_PROVISIONING_MODE_SMALL_BUSINESS == mode)) {
		if (true != isActivate) {
			status = AMT_STATUS_INVALID_AMT_MODE;
			return status;
		}
		status = TryChangeToAMT(command);
		if (AMT_STATUS_SUCCESS != status) {
			return status;
		}
		lState.mayBeStarted = true;
	}

	if (true != lState.newCode) {
		status = command.GetProvisioningMode(mode, legacy);
		if (AMT_STATUS_SUCCESS != status) {
			return status;
		}
		PTHIDisplay::DisplayAMTMode(legacy);
		if ((legacy) && (0 != param.OneTimePassword.Length)) {
			status = AMT_STATUS_INVALID_AMT_MODE;
			return status;
		}
	}

	//enumerate hash handles
	if (CFG_PROVISIONING_MODE_ENTERPRISE == mode) {
		status = command.EnumerateHashHandles(HashHandles);
		if (AMT_STATUS_SUCCESS != status) {
			return status;
		}
	} else {
		HashHandles.Length = 0;
	}

	// print all handles in here if any
	if (HashHandles.Length > 0) {
		DisplayString(FOUND);
		fprintf(stdout, " %d ", HashHandles.Length);
		DisplayMessage(CERT_HASHES_IN_FW);
		for (UINT32 i = 0; i < HashHandles.Length; i++) {
			fprintf(stdout, "%d,", HashHandles.Handles[i]);
		}
		fprintf(stdout, "\n");

		//print certificate hashes
		for (UINT32 i = 0; i < HashHandles.Length; i++) {
			CERTHASH_ENTRY HashEntry;
			status = command.GetCertificateHashEntry(HashHandles.Handles[i], HashEntry);
			if (AMT_STATUS_SUCCESS != status) {
				DisplayStatusString(status);
			} else {
				PTHIDisplay::DisplayHashEntry(HashEntry);
				if (NULL != HashEntry.Name.Buffer) {
					free(HashEntry.Name.Buffer);
				}
			}
		}
	} else {
		DisplayMessage(NO_HANDLES_FOUND);
	}

	return status;
}

/*
* Calls to Activate AMT configuration.
* Arguments:
*	param - Local agent parameters structure
* Return values:
*	PT_STATUS_SUCCESS - on success for get amt configuration data
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
UINT32 Activate(PTHICommand &command, LOCAL_AGENT_PARAMS param,
		ZtcLocalState &lState)
{
	AMT_STATUS status;
	AMT_BOOLEAN ztcEnabled;
	AMT_PROVISIONING_TLS_MODE provisioningTlsMode;
	AMT_RNG_STATUS rngStatus;
	AMT_ANSI_STRING dnsSuffix;
	AMT_ANSI_STRING tempDnsSuffix;
	AMT_ANSI_STRING otp;
	AMT_PROVISIONING_STATE provstate;

	// ZTC enabled
	status = command.GetZeroTouchEnabled(ztcEnabled);
	if (AMT_STATUS_SUCCESS != status) {
		return status;
	}
	PTHIDisplay::DisplayZTCEnabled(ztcEnabled);

	if ((!ztcEnabled) && (0 != param.OneTimePassword.Length)) {
		status = AMT_STATUS_INVALID_AMT_MODE;
		return status;
	}

	// Get provisioning TLS mode
	status = command.GetProvisioningTlsMode(provisioningTlsMode);
	if (AMT_STATUS_SUCCESS != status) {
		return status;
	}
	PTHIDisplay::DisplayProvisioningTlsMode(provisioningTlsMode);

	if ((PSK == provisioningTlsMode) && (0 != param.OneTimePassword.Length)) {
		status = PTSDK_STATUS_INVALID_PARAM;
		return status;
	}

	if ((PSK != provisioningTlsMode) && (ztcEnabled)) {
		//Generate Rng Key
		status = command.GenerateRngKey();
		if ((AMT_STATUS_SUCCESS != status) &&
		    (AMT_STATUS_RNG_GENERATION_IN_PROGRESS != status)) {
			return status;
		} else if (AMT_STATUS_RNG_GENERATION_IN_PROGRESS == status) {
			for (int i = 0; i < 12; i++) {
				sleep(5);
				if (true != command.PTHIClient.Init()) {
					continue;
				}
				// Calling GetRngSeedStatus() too soon might fail.
				// We need to give the firmware time to finish its reset
				// and to generate the RNG seed.
				sleep(5);//seconds
				status = command.GetRngSeedStatus(rngStatus);
				if (status == AMT_STATUS_SUCCESS) {
					break;
				}
			}
		} else {
			status = command.GetRngSeedStatus(rngStatus);
		}

		if (AMT_STATUS_SUCCESS != status) {
			return status;
		}
		PTHIDisplay::DisplayRngSeedStatus(rngStatus);

		if (RNG_STATUS_EXIST != rngStatus) {
			status = AMT_STATUS_PKI_MISSING_KEYS;
			return status;
		}

		if (0 != param.DnsSuffix.Length) {
			dnsSuffix.Length = param.DnsSuffix.Length;
			dnsSuffix.Buffer = (CHAR *)malloc(dnsSuffix.Length);
			if (NULL == dnsSuffix.Buffer) {
				status = PTSDK_STATUS_INTERNAL_ERROR;
				DisplayErrorMessage(ALLOCATE_MEMORY_ERROR);
				return status;
			}
			memcpy(dnsSuffix.Buffer, param.DnsSuffix.String, dnsSuffix.Length);
			status = command.SetDnsSuffix(dnsSuffix);
			if (AMT_STATUS_SUCCESS != status) {
				if (NULL != dnsSuffix.Buffer) {
					free(dnsSuffix.Buffer);
				}
				return status;
			}
			status = command.GetDnsSuffix(tempDnsSuffix);
			if (AMT_STATUS_SUCCESS != status) {
				if (NULL != dnsSuffix.Buffer) {
					free(dnsSuffix.Buffer);
				}
				return status;
			}
			if ((tempDnsSuffix.Length != dnsSuffix.Length) ||
			    (0 != memcmp(dnsSuffix.Buffer, tempDnsSuffix.Buffer, tempDnsSuffix.Length))) {
				status = AMT_STATUS_INTERNAL_ERROR;
				if (NULL != dnsSuffix.Buffer) {
					free(dnsSuffix.Buffer);
				}
				if (NULL != tempDnsSuffix.Buffer) {
					free(tempDnsSuffix.Buffer);
				}
				return status;
			}

			if (NULL != dnsSuffix.Buffer) {
				free(dnsSuffix.Buffer);
			}
			if (NULL != tempDnsSuffix.Buffer) {
				free(tempDnsSuffix.Buffer);
			}
		}

		if (0 != param.OneTimePassword.Length) {
			otp.Length = param.OneTimePassword.Length;
			otp.Buffer = (CHAR *)malloc(otp.Length);
			if (NULL == otp.Buffer) {
				status = PTSDK_STATUS_INTERNAL_ERROR;
				DisplayErrorMessage(ALLOCATE_MEMORY_ERROR);
				return status;
			}
			memcpy(otp.Buffer, param.OneTimePassword.String, otp.Length);
			status = command.SetProvisioningServerOTP(otp);
			if (NULL != otp.Buffer) {
				free(otp.Buffer);
			}
			if (AMT_STATUS_SUCCESS != status) {
				return status;
			}
		}
	}

	if (true == lState.mayBeStarted) {
		if (AMT_STATUS_SUCCESS == command.GetProvisioningState(provstate)) {
			if (PROVISIONING_STATE_IN == provstate) {
				PTHIDisplay::DisplayProvisioningState(provstate);
				return status;
			}
		}
		DisplayMessage(ZTC_NEED_REBOOT);
		return status;
	}

	//Start configuration
	status = command.StartConfiguration();
	if (AMT_STATUS_CERTIFICATE_NOT_READY == status) {
		if (PSK == provisioningTlsMode) {
			return status;
		}
		for (int i = 0; i < 20; i++) {
			sleep(30);//30 seconds
			// provisioning state
			status = command.GetProvisioningState(provstate);
			if (AMT_STATUS_SUCCESS != status) {
				return status;
			}
			if (PROVISIONING_STATE_IN == provstate) {
				PTHIDisplay::DisplayProvisioningState(provstate);
				return status;
			}
		}

		PTHIDisplay::DisplayProvisioningState(provstate);
		status = AMT_STATUS_INVALID_PROVISIONING_STATE;
		return status;
	}

	if (AMT_STATUS_SUCCESS == command.GetProvisioningState(provstate)) {
		PTHIDisplay::DisplayProvisioningState(provstate);

		if ((AMT_STATUS_SUCCESS == status) &&
		    (PROVISIONING_STATE_IN != provstate)) {
			status = AMT_STATUS_INVALID_PROVISIONING_STATE;
		}
	}

	return status;
}

// Define GUID used to connect to the Watchdog client (via the HECI device)
// {05B79A6F-4628-4D7F-899D-A91514CB32AB}
const GUID HECI_WATCHDOG_GUID = {0x05B79A6F, 0x4628, 0x4D7F, {0x89, 0x9D, 0xA9, 0x15, 0x14, 0xCB, 0x32, 0xAB}};

/*
* Change SKU to AMT
*/
bool ChangeToAMT()
{
	UCHAR *rxBuff = (UCHAR *)NULL;
	DWORD bytesRead, bytesWritten;
	unsigned long bufSize;
	HECILinux WDClient(HECI_WATCHDOG_GUID, _verbose);

	if (true != WDClient.Init()) {
		if (_verbose) {
			DisplayErrorMessage(HECI_CONNECT_TO_WD_CLIENT_FAILURE);
		}
		return false;
	}
	bufSize = WDClient.GetBufferSize();

	if (bufSize < sizeof(STATE_INDEPNDENCE_COMMAND)) {
		if (_verbose) {
			DisplayErrorMessage(FW_BUFFER_IS_TO_SMALL);
		}
		goto err;
	}

	rxBuff = (UCHAR *)malloc(bufSize);
	if (NULL == rxBuff) {
		if (_verbose) {
			DisplayErrorMessage(ALLOCATE_MEMORY_ERROR);
		}
		goto err;
	}
	STATE_INDEPNDENCE_COMMAND msg;
	msg.Cmd = STATE_INDEPNDENCE_CMD;
	msg.ByteCount = STATE_INDEPNDENCE_BYTE_COUNT;
	msg.SubCmd = STATE_INDEPNDENCE_ENABLED;
	msg.Version = STATE_INDEPNDENCE_VERSION;

	bytesWritten = WDClient.SendMessage((UCHAR *)&msg, sizeof(msg));
	if (bytesWritten != sizeof(msg)) {
		if (_verbose) {
			DisplayErrorMessage(SEND_DATA_TO_FW_FAILURE);
		}
		goto err;
	}
	STATE_INDEPENDECE_IS_CHANGE_ENABLED_REPLY repEnabledMsg;
	bytesRead = WDClient.ReceiveMessage(rxBuff, bufSize);
	if (bytesRead != sizeof(STATE_INDEPENDECE_IS_CHANGE_ENABLED_REPLY)) {
		if (_verbose) {
			DisplayErrorMessage(RECEIVE_DATA_FROM_FW_FAILURE);
		}
		goto err;
	}
	memcpy(&repEnabledMsg, rxBuff, sizeof(STATE_INDEPENDECE_IS_CHANGE_ENABLED_REPLY));
	if (!repEnabledMsg.Enabled) {
		goto err;
	}
	//Send change to AMT
	msg.SubCmd = STATE_INDEPNDENCE_CHANGE_TO_AMT;
	bytesWritten = WDClient.SendMessage((UCHAR *)&msg, sizeof(msg));
	if (bytesWritten != sizeof(msg)) {
		if (_verbose) {
			DisplayErrorMessage(SEND_DATA_TO_FW_FAILURE);
		}
		goto err;
	}
	STATE_INDEPENDECE_CHANGE_TO_AMT_REPLY repChangeToAMTMsg;
	bytesRead = WDClient.ReceiveMessage(rxBuff, bufSize);
	if (bytesRead != sizeof(STATE_INDEPENDECE_CHANGE_TO_AMT_REPLY)) {
		if (_verbose) {
			DisplayErrorMessage(RECEIVE_DATA_FROM_FW_FAILURE);
		}
		goto err;
	}
	memcpy(&repChangeToAMTMsg, rxBuff, sizeof(STATE_INDEPENDECE_CHANGE_TO_AMT_REPLY));
	if (HECI_STATUS_OK != repChangeToAMTMsg.Status) {
		goto err;
	}
	free(rxBuff);
	WDClient.Deinit();
	return true;

err:
	if (rxBuff) {
		free(rxBuff);
	}
	WDClient.Deinit();

	return false;
}

void DisplayLocalAgentVersion()
{
	fprintf(stdout, "\n");
	DisplayString(VERSION_MESSAGE);
	fprintf(stdout, "%d.%d.%d.%d\n\n",
		MAJOR_VERSION, MINOR_VERSION,
		QUICK_FIX_NUMBER, VER_BUILD);
}

bool ReInitPTHI(PTHICommand &command)
{
	for (int i = 0; i < 12; i++) {
		sleep(5);

		if (true != command.PTHIClient.Init()) {
			continue;
		} else {
			return true;
		}
	}
	return false;
}

bool IsNewCode(CODE_VERSIONS &versions)
{
	UINT32 i;

	for (i = 0; i < versions.VersionsCount; i++) {
		if ((3 == versions.Versions[i].Description.Length)
		    && ('A' == versions.Versions[i].Description.String[0])
		    && ('M' == versions.Versions[i].Description.String[1])
		    && ('T' == versions.Versions[i].Description.String[2])
		    ) {
			if ('5' == versions.Versions[i].Version.String[0]) {
				return true;
			}
			break;
		}
	}
	return false;
}

