; ACL2 Version 6.3 -- A Computational Logic for Applicative Common Lisp
; Copyright (C) 2013, Regents of the University of Texas

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the LICENSE file distributed with ACL2.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; LICENSE for more details.

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Science
; University of Texas at Austin
; Austin, TX 78701 U.S.A.

(in-package "ACL2")

; This file, ld.lisp, provides the definition of the ACL2 macro ld,
; which implements both the ACL2 read-eval-print loop and the ACL2
; file loader.

(defun default-print-prompt (channel state)

; This is the default function for printing the ACL2 ld loop prompt.  A typical
; prompt looks like: ACL2 !>, where the number of >'s indicates the ld-level.
; The prompt is printed by (fmt "~@0~sr ~@1~*2" a channel state nil), where a
; is an alist computed from current-package, ld-level, default-defun-mode,
; guard-checking-on, and ld-skip-proofsp, and #\r is bound to "" except for the
; #+:non-standard-analysis version, where it is bound to "(r)".  To keep from
; consing up this alist every time, we memoize it, storing in 'prompt-memo the
; tuple (pkg level skipp defun-mode+ gc-on a), where defun-mode+ is the
; default-defun-mode except in raw-mode, where defun-mode+ is nil.  Thus, if
; the current settings are as in the memo, we use the a in the memo.
; Otherwise, we compute and store a new memo.

; Warning:  If you change the default prompt format, be sure to change it
; in eval-event-lst, where we print it by hand.

  ":Doc-Section Miscellaneous

  the default ~il[prompt] printed by ~ilc[ld]~/
  ~bv[]
  Example prompt:
  ACL2 p!s>
  ~ev[]
  The ~il[prompt] printed by ACL2 displays the current package, followed by
  a space, followed by zero or more of the three ~il[characters] as
  specified below, followed by the character ~ilc[>] printed one or more
  times, reflecting the number of recursive calls of ~ilc[ld].  The three
  ~il[characters] in the middle are as follows:
  ~bv[]
  p     ; when (default-defun-mode (w state)) is :program
  !     ; when guard checking is on
  s     ; when (ld-skip-proofsp state) is t
  ~ev[]
  ~l[default-defun-mode], ~pl[set-guard-checking], and
  ~pl[ld-skip-proofsp].~/

  Also ~pl[ld-prompt] to see how to install your own ~il[prompt].

  Here are some examples with ~c[ld-skip-proofsp nil].
  ~bv[]
  ACL2 !>    ; logic mode with guard checking on
  ACL2 >     ; logic mode with guard checking off
  ACL2 p!>   ; program mode with guard checking on
  ACL2 p>    ; program mode with guard checking off
  ~ev[]
  Here are some examples with ~ilc[default-defun-mode] of ~c[:]~ilc[logic].
  ~bv[]
  ACL2 >     ; guard checking off, ld-skip-proofsp nil
  ACL2 s>    ; guard checking off, ld-skip-proofsp t
  ACL2 !>    ; guard checking on, ld-skip-proofsp nil
  ACL2 !s>   ; guard checking on, ld-skip-proofsp t
  ~ev[]
  Finally, here is the prompt in raw mode (~pl[set-raw-mode]),
  regardless of the settings above:
  ~bv[]
  ACL2 P>
  ~ev[]~/
  "

  (let ((prompt-memo (and (f-boundp-global 'prompt-memo state)
                          (f-get-global 'prompt-memo state))))
    (cond
     ((and prompt-memo
           (equal (car prompt-memo) (f-get-global 'current-package state))
           (equal (cadr prompt-memo) (f-get-global 'ld-level state))
           (eq (caddr prompt-memo) (f-get-global 'ld-skip-proofsp state))
           (eq (cadddr prompt-memo) (and (not (raw-mode-p state))
                                         (default-defun-mode (w state))))

; In the following, we could use iff instead of eq, because the dependence of
; defun-mode-prompt on (f-get-global 'guard-checking-on state) is restricted to
; whether or not the latter is nil/:none.  But it's cheap to update the
; prompt-memo so we keep the more restrictive eq test for robustness, in case
; the code for defun-mode-prompt changes.

           (eq (car (cddddr prompt-memo))
               (f-get-global 'guard-checking-on state)))
      (fmt1 "~@0~sr ~@1~*2" (cadr (cddddr prompt-memo)) 0 channel state nil))
     (t
      (let ((alist
             (list (cons #\0 (f-get-global 'current-package state))
                   (cons #\1 (defun-mode-prompt-string state))
                   (cons #\2 (list "" ">" ">" ">"
                                   (make-list-ac (f-get-global 'ld-level state)
                                                 nil nil)))
                   (cons #\r
                         #+:non-standard-analysis "(r)"
                         #-:non-standard-analysis ""))))
       (pprogn
        (f-put-global 'prompt-memo
                      (list (f-get-global 'current-package state)
                            (f-get-global 'ld-level state)
                            (f-get-global 'ld-skip-proofsp state)
                            (and (not (raw-mode-p state))
                                 (default-defun-mode (w state)))
                            (not (gc-off state))

; There is no need to memoize the binding of #\r for the purpose of checking if
; the prompt is current, since it never changes during a given session.  Of
; course, #\r is bound in the alist.

                            alist)
                      state)
        (fmt1 "~@0~sr ~@1~*2" alist 0 channel state nil)))))))

(defun print-prompt (prompt output-channel state)
  (with-output-forced
   output-channel
   (col state)
   (let ((prompt-fn (cond ((null prompt) nil)
                          ((eq prompt t)
                           (f-get-global 'prompt-function state))
                          (t prompt))))
     (cond
      ((null prompt-fn) (mv 0 state))
      ((eq prompt-fn 'default-print-prompt)
       (default-print-prompt output-channel state))
      (t (mv-let (erp trans-ans state)
           (trans-eval (list prompt-fn
                             (list 'quote output-channel)
                             'state)
                       'print-prompt state t)

; If erp is non-nil, trans-ans is of the form (stobjs-out . valx).  We
; strongly expect that stobjs-out is (nil state).  (That is true if
; prompt is in fact ld-prompt.)  That being the case, we expect
; valx to be (col replaced-state).

           (cond
            ((or erp
                 (not (and (equal (car trans-ans) '(nil state))
                           (integerp (car (cdr trans-ans))))))
             (fmt1 "~%~%Bad Prompt~%See :DOC ld-prompt>"
                   nil 0 output-channel state nil))
            (t (mv (car (cdr trans-ans)) state)))))))))

(defun initialize-timers (state)
  (pprogn
   (set-timer 'prove-time '(0) state)
   (set-timer 'print-time '(0) state)
   (set-timer 'proof-tree-time '(0) state)
   (set-timer 'other-time '(0) state)))

(defun maybe-add-command-landmark (old-wrld old-default-defun-mode form
                                            trans-ans state)

; Old-wrld is the world before the trans-evaluation of form.  That
; trans-evaluation returned trans-ans, which is thus of the form (stobjs-out
; . valx).  If valx contains a state (then it must in fact contain the state
; state), and the current world of that state is different from old-wrld and
; does not end with a command landmark, we add a command landmark for form.

; We pass in old-default-defun-mode as the default-defun-mode of old-wrld.
; This way, we can compute that value at a time that old-wrld is still
; installed, so that the corresponding getprop will be fast.

  (let ((wrld (w state)))
    (cond ((and (member-eq 'state (car trans-ans))
                (not (and (eq (caar wrld) 'command-landmark)
                          (eq (cadar wrld) 'global-value)))
                (not (equal old-wrld wrld)))
           (er-progn
            (get-and-chk-last-make-event-expansion

; For purposes of tracking make-event, we allow time$ only at the top level.
; If there is user demand, we could consider allowing it in arbitrary positions
; of embedded event forms, though in that case we should be careful to check
; that nested calls work well.  Note that we look for time$, not for
; return-last, because we are looking at a user-supplied form, not its
; macroexpansion.

             (cond ((consp form)
                    (case (car form)
                      (time$ (cadr form))
                      (otherwise form)))
                   (t form))
             wrld 'top-level state
             (primitive-event-macros))
            (pprogn
             (cond ((raw-mode-p state)

; If we are in raw mode, then it is scary to imagine that we have changed the
; logical world.

                    (warning$ 'top-level "Raw"
                              "The ACL2 world is being modified while in raw ~
                               mode.  See :DOC set-raw-mode.  Further ~
                               computation in this ACL2 session may have some ~
                               surprising results."))
                   (t state))
             (set-w 'extension
                    (add-command-landmark
                     old-default-defun-mode
                     form
                     (f-get-global 'connected-book-directory state)
                     (f-get-global 'last-make-event-expansion state)
                     wrld)
                    state)
             (value nil))))
          (t (value nil)))))

(defun replace-last-cdr (x val)
  (cond ((atom x) val)
        ((atom (cdr x)) (cons (car x) val))
        (t (cons (car x) (replace-last-cdr (cdr x) val)))))

(defun ld-standard-oi-missing (val file-name ld-missing-input-ok ctx state)
  (cond ((eq ld-missing-input-ok t)
         (value nil))
        (t (let ((msg (msg "~@0  It is likely that the file you requested, ~
                            ~x1, does not exist."
                           (msg *ld-special-error*
                                'standard-oi val)
                           file-name)))
             (cond (ld-missing-input-ok ; not t, so :warn
                    (pprogn (warning$ ctx "ld-missing-input" "~@0" msg)
                            (value nil)))
                   (t (er soft ctx "~@0" msg)))))))

(defun chk-acceptable-ld-fn1-pair (pair ld-missing-input-ok ctx state
                                        co-string co-channel)

; We check that pair, which is of the form (var . val) where var is a symbolp,
; specifies a legitimate "binding" for the LD special var.  This means that we
; check that var is one of the state globals that LD appears to bind (i.e.,
; push and pop in an unwind-protected way) and that val is a reasonable value
; of that global.  For example, 'standard-oi is an LD special but must be bound
; to a true-list of objects or an open object input channel.

; Co-string and co-channel are here to provide a very subtle feature of LD.  If
; the same string is specified for both standard-co and proofs-co then we open
; one channel and use it in both places.  Our caller, chk-acceptable-ld-fn1, is
; responsible for maintaining these two accumulators as we map down the list of
; pairs.  It puts into co-string and co-channel the string and returned channel
; for the first of standard-co or proofs-co encountered.

  (let ((var (car pair))
        (val (cdr pair)))

; The first three LD specials, namely the three channels, are special because
; we may have to open a channel and create a new pair.  Once we get past those
; three, we can just use the standard checkers and return the existing pair.

    (case var
          (standard-oi
           (cond
            ((and (symbolp val)
                  (open-input-channel-p val :object state))
             (value pair))
            ((true-listp val)
             (value pair))
            ((stringp val)
             (let ((file-name (extend-pathname
                               (f-get-global 'connected-book-directory state)
                               val
                               state)))
               (mv-let (ch state)
                       (open-input-channel file-name :object state)
                       (cond (ch (value (cons 'standard-oi ch)))
                             (t (ld-standard-oi-missing
                                 val file-name ld-missing-input-ok ctx
                                 state))))))
            ((consp val)
             (let ((last-cons (last val)))
               (cond
                ((and (symbolp (cdr last-cons))
                      (open-input-channel-p (cdr last-cons) :object state))
                 (value pair))
                ((stringp (cdr last-cons))
                 (let ((file-name (extend-pathname
                                   (f-get-global 'connected-book-directory
                                                 state)
                                   (cdr last-cons)
                                   state)))
                   (mv-let (ch state)
                           (open-input-channel file-name :object state)
                           (cond
                            (ch (value (cons 'standard-oi
                                             (replace-last-cdr val ch))))
                            (t (ld-standard-oi-missing
                                val file-name ld-missing-input-ok ctx
                                state))))))
                (t (er soft ctx *ld-special-error* 'standard-oi val)))))
            (t (er soft ctx *ld-special-error* 'standard-oi val))))
          (standard-co
           (cond
            ((and (symbolp val)
                  (open-output-channel-p val :character state))
             (value pair))
            ((equal val co-string)
             (value (cons 'standard-co co-channel)))
            ((stringp val)
             (mv-let (ch state)
                     (open-output-channel
                      (extend-pathname
                       (f-get-global 'connected-book-directory state)
                       val
                       state)
                      :character
                      state)
                     (cond (ch (value (cons 'standard-co ch)))
                           (t (er soft ctx *ld-special-error* 'standard-co
                                  val)))))
            (t (er soft ctx *ld-special-error* 'standard-co val))))
          (proofs-co
           (cond
            ((and (symbolp val)
                  (open-output-channel-p val :character state))
             (value pair))
            ((equal val co-string)
             (value (cons 'proofs-co co-channel)))
            ((stringp val)
             (mv-let (ch state)
                     (open-output-channel
                      (extend-pathname
                       (f-get-global 'connected-book-directory state)
                       val
                       state)
                      :character
                      state)
                     (cond (ch (value (cons 'proofs-co ch)))
                           (t (er soft ctx *ld-special-error* 'proofs-co val)))))
            (t (er soft ctx *ld-special-error* 'proofs-co val))))
          (current-package
           (er-progn (chk-current-package val ctx state)
                     (value pair)))
          (ld-skip-proofsp
           (er-progn (chk-ld-skip-proofsp val ctx state)
                     (value pair)))
          (ld-redefinition-action
           (er-progn (chk-ld-redefinition-action val ctx state)
                     (value pair)))
          (ld-prompt
           (er-progn (chk-ld-prompt val ctx state)
                     (value pair)))
          (ld-missing-input-ok
           (er-progn (chk-ld-missing-input-ok val ctx state)
                     (value pair)))
          (ld-pre-eval-filter
           (er-progn (chk-ld-pre-eval-filter val ctx state)
                     (value pair)))
          (ld-pre-eval-print
           (er-progn (chk-ld-pre-eval-print val ctx state)
                     (value pair)))
          (ld-post-eval-print
           (er-progn (chk-ld-post-eval-print val ctx state)
                     (value pair)))
          (ld-evisc-tuple
           (er-progn (chk-evisc-tuple val ctx state)
                     (value pair)))
          (ld-error-triples
           (er-progn (chk-ld-error-triples val ctx state)
                     (value pair)))
          (ld-error-action
           (er-progn (chk-ld-error-action val ctx state)
                     (value pair)))
          (ld-query-control-alist
           (er-progn (chk-ld-query-control-alist val ctx state)
                     (value pair)))
          (ld-verbose
           (er-progn (chk-ld-verbose val ctx state)
                     (value pair)))
          (otherwise
           (er soft ctx
               "The variable ~x0 is not an authorized LD special and ~
                hence cannot be bound by LD."
               var)))))

(defun close-channels (channel-closing-alist state)

; It is necessary to close the channels that we open.  We must in fact
; record them somewhere in state so that if we abort LD with a hard error or
; user interrupt that throws us into the unwind-protect code of LP, they are
; still closed.  To enable such "remote closings" we invent the notion of a
; "channel closing alist" which is an alist that pairs opened channels to
; their "types", where a type is either 'oi (object input) or 'co (character
; output).  Given such an alist we close each channel in it, if the channel
; is in fact open.

  (cond
   ((null channel-closing-alist) state)
   (t (pprogn
       (cond
        ((eq (cdar channel-closing-alist) 'oi)
         (cond
          ((open-input-channel-p (caar channel-closing-alist) :object state)
           (close-input-channel (caar channel-closing-alist) state))
          (t state)))
        ((eq (cdar channel-closing-alist) 'co)
         (cond
          ((open-output-channel-p (caar channel-closing-alist)
                                  :character state)
           (close-output-channel (caar channel-closing-alist) state))
          (t state)))
        (t (let ((temp (er hard 'close-channels
                           "The channel ~x0 was tagged with an unimplemented ~
                            channel type, ~x1."
                           (caar channel-closing-alist)
                           (cdar channel-closing-alist))))
             (declare (ignore temp))
             state)))
       (close-channels (cdr channel-closing-alist) state)))))

(defun chk-acceptable-ld-fn1 (alist ld-missing-input-ok ctx state co-string
                                    co-channel new-alist channel-closing-alist)

; We copy alist (reversing it) onto new-alist, checking that each pair in it
; binds an LD special to a legitimate value.  We open the requested files as we
; go and replace the file names with the open channels.  We also accumulate
; into channel-closing-alist the pairs necessary to close (with close-channels)
; the channels we have opened.  We return a pair consisting of the new-alist
; and the final channel-closing-alist.  See chk-acceptable-ld-fn1-pair for an
; explanation of co-string and co-channel.

; Implementation Note: This odd structure has the single redeeming feature that
; if any given pair of alist causes an error, we have in our hands enough
; information to close any channels we might have opened thus far.  If we get
; all the way down alist without causing an error, the channel-closing-alist
; will be used in the acl2-unwind-protect cleanup form and enable us to "close
; on pop" -- which was its original purpose.  But an earlier coding of this
; function suffered from the problem that we could open several channels and
; then, right here, cause an error (e.g., the proposed 'current-package setting
; is bad).  If that happened, those open channels would never be closed.  It is
; still possible to "lose" an opened channel: abort this function after some
; files have been opened.

; This flaw cannot be fixed, at least with the current set of primitives.  To
; close a channel we must have the channel.  We don't have the channel until
; after we have opened it, i.e., the way we get our hands on a channel in ACL2
; is to open a file, but the way we close a channel is to call
; close-output-channel on the channel object (rather than the file).  Thus,
; there is no way we can unwind protect code that opens a channel so as to
; guarantee to close the channel because we can't get the object we are to
; "cleanup" (the channel) until after we have "modified" (opened) it.  So there
; is a window of vulnerability between the time we open the channel and the
; time we stash it away in some location known to our cleanup form.  During
; that window an abort can cause us to lose a channel in the sense that we do
; not close it.  Now we can make that window much smaller than it is now.  As
; things stand now we are vulnerable to aborts from the time we start
; processing alist here until we finish and enter the acl2-unwind-protect in
; ld-fn that "binds" the ld specials.  But all this vulnerability means is that
; lisp fails to close some opened channels during an abort.  If such a thing
; happens, the user could detect it with some poking around.  For example, he
; could just type

; (open-output-channel-p 'ACL2-OUTPUT-CHANNEL::STANDARD-CHARACTER-OUTPUT-i
;                        :character state)

; for a bunch of i starting at 0 and see if there are some he doesn't know
; about.  This is not a catastrophic error.  It is as though the abort placed
; in the open-output-channels field of the state an additional channel or two.
; The only way, as far as we can see, that this can be a problem is in the
; sense of resource exhaustion: operating systems (and thus lisps) generally
; allow a finite number of open channels.

; If we someday endeavor to plug this hole some additional care must be taken
; because the act of opening an ACL2 channel (in raw lisp) is non-atomic -- we
; have to open the stream, generate a channel symbol, and store some stuff on
; the property list of the symbol.  So an abort there can cause an
; irretrievable loss of an open channel unless the problem is addressed down
; there as well.

; Finally we would just like to note that soft errors are handled perfectly
; here in the sense that if some channels are opened and then we get a soft
; error, we close the channels.  And aborts are handled perfectly once we get
; outside of the window of vulnerability discussed.

  (cond
   ((null alist)
    (let ((new-alist
           (cond ((eq ld-missing-input-ok :missing)
                  (put-assoc-eq 'ld-verbose nil
                                (put-assoc-eq 'ld-prompt nil new-alist)))
                 (t new-alist))))
      (value (cons new-alist channel-closing-alist))))
   (t (mv-let
       (erp pair state)
       (chk-acceptable-ld-fn1-pair (car alist) ld-missing-input-ok ctx state
                                   co-string co-channel)
       (cond
        (erp (pprogn
              (close-channels channel-closing-alist state)
              (mv t nil state)))
        (t
         (mv-let
          (pair ld-missing-input-ok)
          (cond ((null pair)
                 (assert$ (eq  (caar alist) 'standard-oi)
                          (mv (cons 'standard-oi nil) :missing)))
                (t (mv pair ld-missing-input-ok)))
          (chk-acceptable-ld-fn1
           (cdr alist) ld-missing-input-ok ctx state
           (cond ((and (null co-string)
                       (or (eq (car pair) 'standard-co)
                           (eq (car pair) 'proofs-co))
                       (stringp (cdr (car alist))))
                  (extend-pathname
                   (f-get-global 'connected-book-directory state)
                   (cdr (car alist))
                   state))
                 (t co-string))
           (cond ((and (null co-channel)
                       (or (eq (car pair) 'standard-co)
                           (eq (car pair) 'proofs-co))
                       (stringp (cdr (car alist))))
                  (cdr pair))
                 (t co-channel))
           (cons pair new-alist)
           (cond
            ((eq (car pair) 'standard-oi)
             (cond ((stringp (cdr (car alist)))
                    (cons (cons (cdr pair) 'oi) channel-closing-alist))
                   ((and (consp (cdr (car alist)))
                         (stringp (cdr (last (cdr (car alist))))))
                    (cons (cons (cdr (last (cdr pair))) 'oi)
                          channel-closing-alist))
                   (t channel-closing-alist)))
            ((and (or (eq (car pair) 'standard-co)
                      (eq (car pair) 'proofs-co))
                  (stringp (cdr (car alist))))
             (cons (cons (cdr pair) 'co) channel-closing-alist))
            (t channel-closing-alist))))))))))

(defun chk-acceptable-ld-fn (alist state)

; Alist is an alist that pairs LD specials with proposed values.  We check
; that those values are legitimate and that only authorized LD specials are
; bound.  If strings are supplied for the specials standard-oi, standard-co,
; and proofs-co, we open corresponding channels and put those channels in
; for the values in the alist.  We return a pair consisting of the modified
; alist and a channel closing alist that pairs opened channels with the
; type information it takes to close them.

  (let ((ctx 'ld))
    (er-progn
     (cond
      ((or (null (f-boundp-global 'current-acl2-world state))
           (null (w state)))
       (er soft ctx
           "The theorem prover's database has not yet been initialized.  To ~
            initialize ACL2 to its full theory, which currently takes about 3 ~
            minutes on a Sparc 2 (Dec. 1992), invoke (initialize-acl2) from ~
            Common Lisp."))
      (t (value nil)))
     (cond ((symbol-alistp alist) (value nil))
           (t (er soft ctx
                  "The argument to ld-fn must be a symbol-alistp and ~x0 is ~
                   not."
                  alist)))
     (cond ((assoc-eq 'standard-oi alist) (value nil))
           (t (er soft ctx
                  "The alist argument to ld-fn must specify a value ~
                   for 'standard-oi and ~x0 does not."
                  alist)))
     (cond ((not (duplicate-keysp-eq alist)) (value nil))
           (t (er soft ctx
                  "The alist argument to ld-fn must contain no duplications ~
                   among the LD specials to be bound.  Your alist contains ~
                   duplicate values for ~&0."
                  (duplicates (strip-cars alist)))))
     (chk-acceptable-ld-fn1 alist
                            (cdr (assoc-eq 'ld-missing-input-ok alist))
                            ctx state nil nil nil nil))))

(defun f-put-ld-specials (alist state)

; Alist is an alist that pairs LD specials with their new values.  We
; f-put-global each special.  Because f-put-global requires an explicitly
; quoted variable, we case split on the authorized LD-specials.  This is
; easier and safer than making translate give us special treatment.  To add
; a new LD-special you must change this function, as well as
; f-get-ld-specials and the checker chk-acceptable-ld-fn1-pair.

; Warning: Somebody else better have checked that the values assigned are
; legitimate.  For example, we here set 'current-package to whatever we are
; told to set it.  This is not a function the user should call!

  (cond
   ((null alist) state)
   (t (pprogn
       (case
        (caar alist)
        (standard-oi
         (f-put-global 'standard-oi (cdar alist) state))
        (standard-co
         (f-put-global 'standard-co (cdar alist) state))
        (proofs-co
         (f-put-global 'proofs-co (cdar alist) state))
        (current-package
         (f-put-global 'current-package (cdar alist) state))
        (ld-skip-proofsp
         (f-put-global 'ld-skip-proofsp (cdar alist) state))
        (ld-redefinition-action
         (f-put-global 'ld-redefinition-action (cdar alist) state))
        (ld-prompt
         (f-put-global 'ld-prompt (cdar alist) state))
        (ld-missing-input-ok
         (f-put-global 'ld-missing-input-ok (cdar alist) state))
        (ld-pre-eval-filter
         (f-put-global 'ld-pre-eval-filter (cdar alist) state))
        (ld-pre-eval-print
         (f-put-global 'ld-pre-eval-print (cdar alist) state))
        (ld-post-eval-print
         (f-put-global 'ld-post-eval-print (cdar alist) state))
        (ld-evisc-tuple
         (f-put-global 'ld-evisc-tuple (cdar alist) state))
        (ld-error-triples
         (f-put-global 'ld-error-triples (cdar alist) state))
        (ld-error-action
         (f-put-global 'ld-error-action (cdar alist) state))
        (ld-query-control-alist
         (f-put-global 'ld-query-control-alist (cdar alist) state))
        (ld-verbose
         (f-put-global 'ld-verbose (cdar alist) state))
        (otherwise
         (let ((x (er hard 'f-put-ld-specials
                      "Someone is using ~x0 as an unauthorized LD-special."
                      (caar alist))))
           (declare (ignore x))
           state)))
       (f-put-ld-specials (cdr alist) state)))))

(defun f-get-ld-specials (state)

; Make an alist, suitable for giving to f-put-ld-specials, that records the
; current values of all LD-specials.  To add a new LD-special you must
; change this function, f-put-ld-specials, and the checker
; chk-acceptable-ld-fn1-pair.

  (list (cons 'standard-oi
              (f-get-global 'standard-oi state))
        (cons 'standard-co
              (f-get-global 'standard-co state))
        (cons 'proofs-co
              (f-get-global 'proofs-co state))
        (cons 'current-package
              (f-get-global 'current-package state))
        (cons 'ld-skip-proofsp
              (f-get-global 'ld-skip-proofsp state))
        (cons 'ld-redefinition-action
              (f-get-global 'ld-redefinition-action state))
        (cons 'ld-prompt
              (f-get-global 'ld-prompt state))
        (cons 'ld-missing-input-ok
              (f-get-global 'ld-missing-input-ok state))
        (cons 'ld-pre-eval-filter
              (f-get-global 'ld-pre-eval-filter state))
        (cons 'ld-pre-eval-print
              (f-get-global 'ld-pre-eval-print state))
        (cons 'ld-post-eval-print
              (f-get-global 'ld-post-eval-print state))
        (cons 'ld-evisc-tuple
              (f-get-global 'ld-evisc-tuple state))
        (cons 'ld-error-triples
              (f-get-global 'ld-error-triples state))
        (cons 'ld-error-action
              (f-get-global 'ld-error-action state))
        (cons 'ld-query-control-alist
              (f-get-global 'ld-query-control-alist state))
        (cons 'ld-verbose
              (f-get-global 'ld-verbose state))))

(defun ld-read-keyword-command1 (n state)
  (cond
   ((= n 0) (value nil))
   (t (mv-let (eofp obj state)
              (read-standard-oi state)
              (cond
               (eofp (er soft 'ld-read-keyword-command
                         "Unfinished keyword command at eof on (standard-oi ~
                          state)."))
               (t
                (er-let*
                 ((rst (ld-read-keyword-command1 (1- n) state)))

; Note: We take advantage of the fact that this function ALWAYS returns a list
; of quoted objects.  See the call of strip-cadrs in ld-read-keyword-command
; below.  So if you optmize away some of the quotes, beware!

                 (value (cons (list 'quote obj) rst)))))))))

(defun exit-ld (state)

; This is the function most commonly aliased to the keyword command :q.  Its
; evaluation causes LD to terminate immediately.  Any function that returns
; three results, the first of which is nil, the second of which is :q and the
; third of which is STATE will do the same.

  (value :q))

(defun macro-minimal-arity1 (lst)
  (declare (xargs :guard (true-listp lst)))
  (cond ((endp lst) 0)
        ((lambda-keywordp (car lst))
         0)
        (t (1+ (macro-minimal-arity1 (cdr lst))))))

(defun macro-minimal-arity (sym default wrld)
  (let ((args (getprop sym 'macro-args default 'current-acl2-world wrld)))
    (macro-minimal-arity1 (if (eq (car args) '&whole)
                              (cddr args)
                            args))))

(defun ld-read-keyword-command (key state)

; ld supports the convention that when a keyword :key is typed
; as a command and the corresponding symbol in the "ACL2" package,
; ACL2::key is a function or macro of arity n, we read n more
; objects, quote them, and apply the ACL2 function or macro.
; Thus,

; MY-PKG !>:ubt foo

; is the same thing as

; MY-PKG !>(ACL2::UBT 'foo)

; We require that the macro not have any lambda keyword arguments, since
; that makes it hard or impossible to determine how many things we should
; read.

; We also support the convention that if :key is bound on 'ld-keyword-aliases
; in state, say in the entry (:key n fn), we manufacture (fn 'x1 ...  'xn)
; instead of requiring that key be a function and returning (key 'x1 ...  'xn).

; This function returns four results, (mv erp keyp form state).  If erp is t an
; error was caused and the message has been printed.  Otherwise, keyp is
; non-nil or nil according to whether the keyword hack was involved.  Form is
; the parsed form of the command read, e.g., (acl2::ubt 'foo).  If non-nil,
; keyp is the actual list of objects read, e.g., (:ubt foo).

  (let ((temp (assoc-eq key (ld-keyword-aliases state))))
    (cond
     (temp
      (mv-let (erp args state)
              (ld-read-keyword-command1 (cadr temp) state)
              (cond
               (erp (mv t nil nil state))
               (t (mv nil
                      (cons key (strip-cadrs args))
                      (cons (caddr temp) args)
                      state)))))
     ((eq key :q)

; Here is the only place we recognize :q as a special command.  Essentially :q
; is an alias for (exit-ld state) except it is overridden by any other aliases
; for :q.

      (mv nil '(:q) '(exit-ld state) state))
     (t
      (let* ((sym (intern (symbol-name key) "ACL2"))
             (wrld (w state))
             (len (cond ((function-symbolp sym wrld)
                         (length (formals sym wrld)))
                        ((getprop sym 'macro-body nil 'current-acl2-world wrld)
                         (macro-minimal-arity
                          sym
                          `(:error "See LD-READ-KEYWORD-COMMAND.")
                          wrld))
                        (t nil))))
        (cond (len (mv-let (erp args state)
                           (ld-read-keyword-command1 len state)
                           (cond (erp (mv t nil nil state))
                                 (t (mv nil
                                        (cons key (strip-cadrs args))
                                        (cons sym args)
                                        state)))))
              (t (mv-let (erp val state)
                         (er soft 'LD
                             "Unrecognized keyword command ~x0."
                             key)
                         (declare (ignore erp val))
                         (mv t nil nil state)))))))))

(defun ld-read-command (state)

; This function reads an ld command from the standard-oi channel of state and
; returns it.  It implements the keyword command hack.  We return five results:
; (mv eofp erp keyp form state).  Eofp means we exhausted standard-oi.  Erp,
; when t, indicates that an error occurred, e.g., an ill-formed keyword command
; was read.  The error message has been printed.  Keyp, when non-nil, indicates
; that form is the parsed form of a keyword command.  The list of objects
; actually read is the non-nil value of keyp and that list, without the
; enclosing parentheses, should be printed instead of form.  Thus, if :kons is
; an alias for cons, then :kons x y will parse into (cons 'x 'y) and keyp will
; be (:kons x y).

  (mv-let (eofp val state)
          (read-standard-oi state)
          (pprogn
           (cond ((int= (f-get-global 'ld-level state) 1)
                  (let ((last-index (iprint-last-index state)))
                    (cond ((> last-index (iprint-soft-bound state))
                           (rollover-iprint-ar nil last-index state))
                          (t state))))
                 (t state))
           (cond (eofp (mv t nil nil nil state))
                 ((keywordp val)
                  (mv-let (erp keyp form state)
                          (ld-read-keyword-command val state)
                          (mv nil erp keyp form state)))
                 ((stringp val)
                  (let ((upval (string-upcase val)))
                    (cond ((find-non-hidden-package-entry
                            upval
                            (global-val 'known-package-alist (w state)))
                           (mv nil nil nil `(in-package ,upval) state))
                          (t (mv nil nil nil val state)))))
                 (t (mv nil nil nil val state))))))

(deflabel acl2-customization
  :doc
  ":Doc-Section switches-parameters-and-modes

  file of initial commands for ACL2 to run at ~il[startup]~/

  ACL2 provides a mechanism to load automatically a so-called ``ACL2
  customization file,'' via ~ilc[ld], the first time ~ilc[lp] is called in an
  ACL2 session.  ACL2 looks for this file as follows.
  ~bq[]
  o If the host Lisp reads a non-empty value for the system's environment
  variable ~c[ACL2_CUSTOMIZATION], then that string value is used for the
  customization file name.  In this case, if the file does not exist or if the
  string is \"NONE\" then there is no customization file.  Notes.  (1) If the
  customization file name is a relative pathname (~pl[pathname]), then the
  pathname is considered relative to the connected book directory (~pl[cbd]).
  (2) If this variable is not already defined, then its value is set to
  ~c[NONE] when the ACL2 makefile system is invoked (specifically,
  using community books file ~c[books/Makefile-generic]), e.g., for a
  regression.

  o Otherwise (empty environment variable value), file
  ~c[\"acl2-customization.lsp\"] or ~c[\"acl2-customization.lisp\"] on the
  connected book directory (~pl[cbd]), generally the current directory, is the
  customization file (in that order) if either exists.

  o Otherwise file ~c[\"acl2-customization.lsp\"] or
  ~c[\"acl2-customization.lisp\"] on your home directory is the customization
  file (in that order), if either exists (except, this case is skipped on
  Windows operating systems.~eq[]

  Except for the fact that this ~ilc[ld] command is not typed explicitly by
  you, it is a standard ~ilc[ld] command, with one exception: any settings of
  ~ilc[ld] specials are remembered once this call of ~ilc[ld] has completed.
  For example, suppose that you start your customization file with
  ~c[(set-ld-skip-proofsp t state)], so that proofs are skipped as it is loaded
  with ~ilc[ld].  Then the ~ilc[ld] special ~ilc[ld-skip-proofsp] will remain
  ~c[t] after the ~ilc[ld] has completed, causing proofs to be skipped in your
  ACL2 session, unless your customization file sets this variable back to
  ~c[nil], say with ~c[(set-ld-skip-proofsp nil state)].~/

  If the customization file exists, it is loaded with ~ilc[ld] using the
  usual default values for the ~ilc[ld] specials (~pl[ld]).  Thus, if an
  error is encountered, no subsequent forms in the file will be
  evaluated.

  To create a customization file it is recommended that you first give it a
  name other than ~c[\"acl2-customization.lsp\"] or
  ~c[\"acl2-customization.lisp\"] so that ACL2 does not try to include it
  prematurely when you next enter ~ilc[lp].  Then, while in the uncustomized
  ~ilc[lp], explicitly invoke ~ilc[ld] on your evolving (but renamed)
  customization file until all forms are successfully evaluated.  The same
  procedure is recommended if for some reason ACL2 cannot successfully evaluate
  all forms in your customization file: temporarily rename your customization
  file so that ACL2 does not try to ~ilc[ld] it automatically and then debug
  the new file by explicit calls to ~ilc[ld].

  WARNING!  If you certify a book after the (automatic) loading of a
  customization file, the forms in that file will be part of the
  ~il[portcullis] of the ~il[books] you certify!  That is, the forms in your
  customization file at certification time will be loaded whenever anybody uses
  the ~il[books] you are certifying.  Since customization files generally
  contain idiosyncratic ~il[command]s, you may not want yours to be part of the
  ~il[books] you create for others.  Thus, if you have a customization file
  then you may want to invoke ~c[:]~ilc[ubt]~c[ 1] before certifying any
  ~il[books]; alternatively, ~pl[certify-book!] for automatic invocation of
  ~ilc[ubt].

  On the other hand, if you wish to prevent undoing commands from the
  customization file, ~pl[reset-prehistory].

  Finally, we note that except on Windows-based systems, if there is a file
  ~c[acl2-init.lsp] in your home directory, then it will be loaded into raw
  Lisp when ACL2 is invoked.~/

  :cited-by Programming")

(deflabel keyword-commands
  :doc
  ":Doc-Section Miscellaneous

  how keyword commands are processed~/
  ~bv[]
  Examples:
  user type-in                 form evaluated
  :pc 5                        (ACL2::PC '5)
  :pcs app rev                 (ACL2::PCS 'app 'rev)
  :length (1 2 3)              (ACL2::LENGTH '(1 2 3))
  :quit                        (ACL2::QUIT) ; Note: avoid optional argument
  ~ev[]~/

  When a keyword, ~c[:key], is read as a command, ACL2 determines whether the
  symbol with the same name in the ~c[\"ACL2\"] package, ~c[acl2::key], is a
  function or simple macro of n arguments.  If so, ACL2 reads ~c[n] more
  objects, ~c[obj1], ..., ~c[objn], and then acts as though it had read the
  following form (for a given ~c[key]):
  ~bv[]
  (ACL2::key 'obj1 ... 'objn)
  ~ev[]
  Thus, by using the keyword command hack you avoid typing the parentheses, the
  ~c[\"ACL2\"] package name, and the quotation marks.

  ~l[ld-keyword-aliases] for how to customize this behavior.

  Note the generality of this hack.  Any function or macro in the ~c[\"ACL2\"]
  package can be so invoked, not just ``commands.''  Indeed, there is no such
  thing as a distinguished class of commands.  Users may take advantage of the
  keyword command hack by defining functions and macros in the ~c[\"ACL2\"]
  package.

  The one caveat is that when the keyword hack is used to invoke a macro, only
  the required arguments for that macro are read before calling that macro:
  none of the ~c[&optional], ~c[&rest], ~c[&body], or ~c[&key] arguments are
  read for that call.  The macro is thus called with only its required
  arguments.  The following log illustrates this caveat.
  ~bv[]
  ACL2 !>:set-iprint t

  ACL2 Query (:SET-IPRINT):  Action  (T, NIL, RESET, RESET-ENABLE, SAME,
  Q or ?):
  ACL2 Observation in SET-IPRINT:  Iprinting has been enabled.
  ACL2 !>
  ~ev[]
  What happened?  First, the command ~c[:set-iprint] was read.  Since the macro
  ~ilc[set-iprint] has no required arguments, the ACL2 evaluator was then
  called on the form ~c[(set-iprint)], that is, calling the macro on no
  arguments.  ~c[Set-iprint] is defined to query the ACL2 user when its first
  argument is omitted.  The log shows that query, which is set up to read the
  next form from the input stream.  That form was available immediately: the
  form ~c[t] that had been supplied by the user.  So the query returned
  immediately and the ~c[set-iprint] call was completed.~/")

(defun ld-print-command (keyp form col state)
  (with-base-10
   (mv-let (col state)
     (cond
      ((not (eq (ld-pre-eval-print state) t)) (mv col state))
      (keyp
       (fmt1 "~*0~|"
             (list (cons #\0 (list "" "~x*" "~x* " "~x* " keyp)))
             col
             (standard-co state)
             state
             (ld-evisc-tuple state)))
      (t
       (fmt1 "~q0~|"
             (list (cons #\0 form))
             col
             (standard-co state)
             state
             (ld-evisc-tuple state))))
     (declare (ignore col))
     state)))

(defun ld-filter-command (form state)
  (let ((filter (ld-pre-eval-filter state)))
    (cond ((eq filter :all) (value t))
          ((eq filter :query)
           (acl2-query :filter
                       '("~#0~[~Y12?~/Eval?~]"
                         :y t :n nil :r :return :q :error
                         :? ("We are in the LD read-eval-print loop, ~
                              processing the forms in standard-oi.  The ~
                              form printed above is one of those forms.  Do ~
                              you want to evaluate it (Y) or not (N)?   You ~
                              may also answer R, meaning ``return ~
                              immediately from LD (without reading or ~
                              evaluating any more forms)'' or Q meaning ~
                              ``return immediately from LD, signalling an ~
                              error.''"
                             :y t :n nil :r :return :q :error))
                       (list (cons #\0 (if (eq (ld-pre-eval-print state) t) 1 0))
                             (cons #\1 form)
                             (cons #\2 (ld-evisc-tuple state)))
                       state))
          (t (value t)))))

#-acl2-loop-only
(defun-one-output ppr? (x raw-x col channel state)
  (cond
   ((and (raw-mode-p state)
         (bad-lisp-objectp x))
    (if (not (eq channel *standard-co*))
        (error "Attempted to print LD results to other than *standard-co*!"))
    (format t "[Note:  Printing non-ACL2 result.]")
    (terpri)
    (prin1 raw-x)
    state)
   (t
    (ppr x col channel state t))))

(defun ld-print-results (trans-ans state)

; This is the function used by ld to print the results of the
; trans-evaluation of the form read.  Trans-ans is of the form
; (stobjs-out . valx).

; If ld-post-eval-print is nil we print nothing.  If it is t, we
; print with the standard evisceration (ld-evisc-tuple).  If it is
; :command-conventions, we hide error/value/state pairs by just printing
; value and we don't print anyting when the value is :invisible.

  (let ((flg (ld-post-eval-print state))
        (output-channel (standard-co state)))

; In raw mode in Allegro Common Lisp (and not GCL, but perhaps other lisps),
; evaluation of (time ...) causes the result value to be printed at the end of
; a comment line printed by time, which is unfortunate.  This sort of printing
; problem does not seem to have come up in other than raw mode, and besides, we
; do not want to try to model this sort of maybe-newline printing in the
; logic.  So we restrict this solution to raw mode.  Furthermore, the lisps
; listed below do not need this fix, and they all print a newline even with
; "~&" when apparently not necessary, so we exclude them from this fix.

    #-(or acl2-loop-only gcl cmu sbcl lispworks ccl)
    (when (raw-mode-p state)
      (format (get-output-stream-from-channel output-channel) "~&"))
    (cond
     ((null flg) state)
     (t
      (let* ((stobjs-out (car trans-ans))
             (valx (cdr trans-ans))
             (evisc-tuple (ld-evisc-tuple state))
             (evisc-alist (world-evisceration-alist state (car evisc-tuple)))
             (print-level (cadr evisc-tuple))
             (print-length (caddr evisc-tuple)))
        (mv-let
         (eviscerated-valx state)
         (eviscerate-stobjs-top (evisceration-stobj-marks stobjs-out nil)
                                valx
                                print-level print-length evisc-alist
                                (table-alist 'evisc-table (w state))
                                nil
                                state)
         (cond
          ((and (eq flg :command-conventions)
                (ld-error-triples state)
                (equal stobjs-out *error-triple-sig*))

; We get here if we are following command-conventions and the form
; returned triple (mv erp val state).  Note that erp must be a
; non-stobj (typically a Boolean) but that val may be a stobj or not.

           (cond
            ((eq (cadr valx) :invisible)
             state)
            (t
             (pprogn
              (princ$ (if (stringp (f-get-global 'triple-print-prefix state))
                          (f-get-global 'triple-print-prefix state)
                        "")
                      output-channel state)

; The following raw code is identical to the logic code below except that the
; raw code handles infix printing, which is, at the moment, entirely
; extra-logical.

              #-acl2-loop-only
              (let ((col
                     (if (stringp (f-get-global 'triple-print-prefix state))
                         (length (f-get-global 'triple-print-prefix state))
                       0))
                    (evg (cadr eviscerated-valx)))
                (cond
                 ((and (live-state-p state)
                       (output-in-infixp state))
                  (print-infix
                   evg
                   nil
                   (- (fmt-hard-right-margin state) col)
                   0 col
                   (get-output-stream-from-channel output-channel)
                   t)
                  *the-live-state*)
                 (t (ppr? evg (cadr valx) col output-channel state))))
              #+acl2-loop-only
              (ppr (cadr eviscerated-valx)
                   (if (stringp (f-get-global 'triple-print-prefix state))
                       (length (f-get-global 'triple-print-prefix state))
                     0)
                   output-channel state t)
              (newline output-channel state)))))
          (t (pprogn
              #-acl2-loop-only
              (cond
               ((and (live-state-p state)
                     (output-in-infixp state))
                (print-infix
                 eviscerated-valx
                 nil
                 (fmt-hard-right-margin state)
                 0 0
                 (get-output-stream-from-channel output-channel)
                 t)
                *the-live-state*)
               (t (ppr? eviscerated-valx valx 0 output-channel state)))
              #+acl2-loop-only
              (ppr eviscerated-valx 0 output-channel state t)
              (newline output-channel state))))))))))

(defun ld-print-prompt (state)

; Like print-prompt except may print the prompt both to *standard-co*
; and (standard-co state).

  (mv-let (col state)
          (print-prompt (ld-prompt state) (standard-co state) state)
          (cond
           ((and (eq (standard-oi state) *standard-oi*)
                 (not (eq (standard-co state) *standard-co*)))
            (mv-let (irrel-col state)
                    (print-prompt (ld-prompt state) *standard-co* state)
                    (declare (ignore irrel-col))
                    (mv col state)))
           (t (mv col state)))))

(defun good-bye-fn (status)
  (declare (xargs :mode :logic :guard t))
  #-acl2-loop-only
  (exit-lisp (ifix status))
  status)

(defmacro good-bye (&optional (status '0))

  ":Doc-Section Other

  quit entirely out of Lisp~/
  ~bv[]
  Examples:
  ACL2 !>(good-bye)
  ; [ACL2 is exited]

  ACL2 !>(good-bye 3)
  ; [ACL2 is exited with Unix exit status 3]
  ~ev[]
  Note:  Your entire session will disappear forever when you evaluate
  ~c[(good-bye)].~/

  The command ~c[(good-bye)] quits not only out of the ACL2 ~il[command] loop,
  but in fact quits entirely out of the underlying Lisp.  Thus, there
  is no going back!  You will ~st[not] be able to re-enter the ~il[command] loop
  after typing ~c[(good-bye)]!  All your work will be lost!!!

  This command may not work in some underlying Common Lisp implementations.  In
  such cases, there is no harm in trying; ACL2 will let you know how to proceed
  if it cannot exit.

  In some systems, typing ~c[control-d] at the top-level ACL2 prompt
  (~c[control-c control-d] if inside emacs) will call this function.

  If you give ~c[good-bye] an argument, it should be a natural number, and it
  indicates the Unix (Linux) exit status.

  If you merely want to exit the ACL2 ~il[command] loop, use ~c[:q] instead
  (~pl[q])."

  (declare (xargs :guard (natp status)))
  `(good-bye-fn ,status))

(defun ld-return-error (state)
  (let ((action (ld-error-action state)))
    (cond ((eq action :return!)
           (mv :return
               (list :stop-ld (f-get-global 'ld-level state))
               state))
          (t (mv action :error state)))))

(defun initialize-accumulated-warnings ()
  #-acl2-loop-only
  (setq *accumulated-warnings* nil)
  nil)

(defun ld-read-eval-print (state)

; This is LD's read-eval-print step.  We read a form from standard-oi, eval it,
; and print the result to standard-co, will lots of bells and whistles
; controlled by the various LD specials.  The result of this function is a
; triple (mv signal val state), where signal is one of :CONTINUE, :RETURN, or
; :ERROR.  When the signal is :continue or :error, val is irrelevant.  When the
; signal is :return, val is the "reason" we are terminating and is one of
; :exit, :eof, :error, :filter, or (:stop-ld n) where n is the ld-level at the
; time of termination.

  (pprogn
   (cond ((<= (f-get-global 'ld-level state) 1)
          (pprogn (f-put-global 'trace-level 0 state)
                  (print-deferred-ttag-notes-summary state)))
         (t state))
   (mv-let
    (col state)
    (ld-print-prompt state)
    (mv-let
     (eofp erp keyp form state)
     (ld-read-command state)
     (cond
      (eofp (cond ((ld-prompt state)
                   (pprogn (princ$ "Bye." (standard-co state) state)
                           (newline (standard-co state) state)

; In versions before v2-8, typing ctrl-d (ctrl-c ctrl-d in Emacs) did not
; immediately kill the Lisp if the resulting eof condition was detected by BRR
; processing.  The code below fixes that; let's hope it doesn't "fix" anything
; else!

                           (prog2$ (and (equal (standard-oi state) *standard-oi*)
                                        (good-bye))
                                   state)
                           (mv :return :eof state)))
                  (t (mv :return :eof state))))
      (erp (ld-return-error state))
      (t (pprogn
          (ld-print-command keyp form col state)
          (mv-let
           (erp ans state)
           (ld-filter-command form state)
           (cond
            (erp (ld-return-error state))
            ((null ans) (mv :continue nil state))
            ((eq ans :error) (mv :error nil state))
            ((eq ans :return) (mv :return :filter state))
            (t (pprogn
                (cond ((<= (f-get-global 'ld-level state) 1)
                       (prog2$ (initialize-accumulated-warnings)
                               (initialize-timers state)))
                      (t state))
                (f-put-global 'last-make-event-expansion nil state)
                (let* ((old-wrld (w state))
                       (old-default-defun-mode
                        (default-defun-mode old-wrld)))
                  (mv-let
                   (error-flg trans-ans state)
                   (revert-world-on-error
                    (mv-let (error-flg trans-ans state)
                            (if (raw-mode-p state)
                                (acl2-raw-eval form state)
                              (trans-eval form 'top-level state t))

; If error-flg is non-nil, trans-ans is (stobjs-out . valx).

                            (er-progn
                             (chk-absstobj-invariants nil state)
                             (cond
                              (error-flg (mv t nil state))
                              ((and (ld-error-triples state)
                                    (equal (car trans-ans) *error-triple-sig*)
                                    (car (cdr trans-ans)))
                               (mv t nil state))
                              (t (er-progn
                                  (maybe-add-command-landmark
                                   old-wrld
                                   old-default-defun-mode
                                   form
                                   trans-ans state)
                                  (mv nil trans-ans state)))))))

; If error-flg is non-nil, trans-ans is (stobjs-out . valx) and we know
; that valx is not an erroneous error triple if we're paying attention to
; error triples.

; The code inside the revert-world-on-error arranges to revert if either
; trans-eval returns an error, or the value is to be thought of as an
; error triple and it signals an error.  Error-flg, now, is set to t
; iff we reverted.

                   (cond
                    (error-flg (ld-return-error state))
                    ((and (equal (car trans-ans) *error-triple-sig*)
                          (eq (cadr (cdr trans-ans)) :q))
                     (mv :return :exit state))
                    (t (pprogn
                        (ld-print-results trans-ans state)
                        (cond
                         ((and (ld-error-triples state)
                               (not (eq (ld-error-action state) :continue))
                               (equal (car trans-ans) *error-triple-sig*)
                               (let ((val (cadr (cdr trans-ans))))
                                 (and (consp val)
                                      (eq (car val) :stop-ld))))
                          (mv :return
                              (list* :stop-ld
                                    (f-get-global 'ld-level state)
                                    (cdr (cadr (cdr trans-ans))))
                              state))
                         (t

; We make the convention of checking the new-namep filter immediately after
; we have successfully eval'd a form (rather than waiting for the next form)
; so that if the user has set the filter up he gets a satisfyingly
; immediate response when he introduces the name.

                          (let ((filter (ld-pre-eval-filter state)))
                            (cond
                             ((and (not (eq filter :all))
                                   (not (eq filter :query))
                                   (not (new-namep filter
                                                   (w state))))
                              (er-progn

; We reset the filter to :all even though we are about to exit this LD
; with :return.  This just makes things work if "this LD" is the top-level
; one and LP immediately reenters.

                               (set-ld-pre-eval-filter :all state)
                               (mv :return :filter state)))
                             (t (mv :continue nil state))))))))))))))))))))))

(defun ld-loop (state)

; Note: We use a bit of raw lisp to ensure that the ACL2 unwind protect stack
; is properly configured before we execute the prompt for the next command.
; This acl2-unwind can be exercised, we think, by evaluating LD recursively
; and aborting the inferior LD so that it fails to cleanup after itself.

  (mv-let
   (signal val state)
   #+acl2-loop-only (ld-read-eval-print state)
   #-acl2-loop-only (progn (acl2-unwind *ld-level* t)
                           (ld-read-eval-print state))
   (cond ((eq signal :continue)
          (ld-loop state))
         ((eq signal :return)
          (value val))
         (t (mv t nil state)))))

; The following raw lisp special variable controls whether the raw lisp version
; of ld-fn-body, below, prints the header as per ld-verbose or does not.  The
; handling of aborts in ld-fn forces us to call ld-fn-body again after each
; abort and we wish to suppress the header message after all entrances other
; than the first.  This only happens after an abort (all bets are off) and the
; idea is to fool the user into thinking a normal error was signalled.

#-acl2-loop-only
(defvar *first-entry-to-ld-fn-body-flg*)

(defun update-cbd (standard-oi0 state)

; For the case that standard-oi0 is a string (representing a file), we formerly
; used extend-pathname to compute the new cbd from the old cbd and
; standard-oi0.  However, this caused us to follow soft links when that was
; undesirable.  Here is a suitable experiment, after building the nonstd books
; by connecting to books/nonstd/ and running "make clean-nonstd" followed by
; "make all-nonstd".  In this experiment, we had already certified the regular
; books using ACL2(h), and an error occurred because of an attempt to read
; books/arithmetic/equalities.cert, which used a special hons-only format.

; cd /projects/acl2/devel/books/nonstd/arithmetic/
; /projects/acl2/devel/allegro-saved_acl2r
; (ld "top.lisp")

  (let ((old-cbd (f-get-global 'connected-book-directory state)))
    (cond ((and old-cbd
                (stringp standard-oi0)
                (position *directory-separator* standard-oi0))
           (let* ((os (os (w state)))
                  (filename-dir
                   (expand-tilde-to-user-home-dir
                    (concatenate 'string
                                 (remove-after-last-directory-separator
                                  standard-oi0)
                                 *directory-separator-string*)
                    os 'update-cbd state)))
             (f-put-global
              'connected-book-directory
              (if (absolute-pathname-string-p filename-dir nil os)
                  filename-dir
                (our-merge-pathnames old-cbd filename-dir))
              state)))
          (t state))))

(defun ld-fn-body (standard-oi0 new-ld-specials-alist state)

; This function is defined only to make it convenient for ld-fn to execute its
; "body" either inside or outside an acl2-unwind-protect.

; WARNING: Because of the hidden acl2-unwind in the raw code for ld-loop above
; do not try to use acl2-unwind-protect in this function.  The cleanup form for
; it will be executed before the first form is read because ld-loop rolls back
; to the initialized version of the frame.  Furthermore, do not execute
; non-idempotent state changing forms here, i.e., incrementing or decrementing
; some counter in state, because the abort handling may cause this body to be
; reentered after an abort while the logical semantics suggests that it is
; entered only once.  (Of course, aborts mean all bets are off, but the idea is
; to make it seem like they are errors.)  We once incremented and decremented
; ld-level here and found the load level going down every time an abort
; occurred (because its increment was undone by the hidden acl2-unwind in
; ld-loop, mentioned above, and it was decremented at every abort).

  #+(and acl2-par (not acl2-loop-only))
  (when (and (not *wormholep*)

; We do not reset parallelism variables while in a wormhole (say from :brr),
; because that could interfere with a surrounding (outside the wormhole) prover
; call.

; Fortunately, it isn't necessary to do that reset, because there is nothing to
; clean up: we (plan as of Feb. 2011 to) disable entry to the prover from a
; wormhole when parallelism is enabled for the prover.

             (or (eql *ld-level* 1)
                 *reset-parallelism-variables*))

; We claim that the parallelism variables are reset when either (1) we are
; entering the top-level ACL2 loop from raw Lisp, or else (2) a raw Lisp break
; has occurred.  Let's see how the conditions above guarantee that claim.  If
; (1) holds then the initial call of ld-fn-body in ld-fn0 will get us here with
; *ld-level* 1.  When (2) holds then our-abort threw to 'local-top-level after
; setting *reset-parallelism-variables* to t, and the ld-fn-body call in ld-fn0
; is re-entered after that throw is caught, and here we are!

; In rare cases we might get here without (1) or (2) holding -- say, after :a!.
; But it's OK to call reset-all-parallelism-variables in such cases; we simply
; prefer to minimize the frequency of calls, for efficiency.

    (reset-all-parallelism-variables))

  (pprogn
    (f-put-ld-specials new-ld-specials-alist state)
    (update-cbd standard-oi0 state)
    (cond (#+acl2-loop-only (ld-verbose state)
           #-acl2-loop-only (and *first-entry-to-ld-fn-body-flg*
                                  (ld-verbose state))

; We print the file name rather than the channel.

           (cond
            ((eq (ld-verbose state) t)
             (fms (if (eq standard-oi0 *standard-oi*)
                      "ACL2 loading *standard-oi*.~%"
                      "ACL2 loading ~x0.~%")
                  (list (cons #\0 (cond ((consp standard-oi0) (kwote standard-oi0))
                                        (t standard-oi0))))
                  (standard-co state)
                  state
                  (ld-evisc-tuple state)))
            (t (with-base-10
                (fms
                 "~@0"
                 (list (cons #\0 (ld-verbose state))
                       (cons #\v (f-get-global 'acl2-version state))
                       (cons #\l (f-get-global 'ld-level state))
                       (cons #\c (f-get-global 'connected-book-directory
                                               state))
                       (cons #\b (f-get-global 'system-books-dir
                                               state)))
                 (standard-co state)
                 state
                 (ld-evisc-tuple state))))))
          (t state))
    (mv-let
     (erp val state)
     (ld-loop state)
     (pprogn
      (cond ((eq (ld-verbose state) t)
             (fms (if (eq standard-oi0 *standard-oi*)
                      "Finished loading *standard-oi*.~%"
                      "Finished loading ~x0.~%")
                  (list (cons #\0 (cond ((consp standard-oi0) (kwote standard-oi0))
                                        (t standard-oi0))))
                  (standard-co state)
                  state
                  (ld-evisc-tuple state)))
            (t state))
      (mv erp val state)))))

(defun ld-fn1 (standard-oi0 alist state bind-flg)

; If this function weren't defined we would have to duplicate its body twice in
; ld-fn, once in the #+acl2-loop-only section and again in the
; #-acl2-loop-only section in the case where the state is not the live state.
; The reason we grab the old ld-level and use it in the cleanup form rather
; than just decrementing the then current value is that we do not know how many
; times the cleanup form will be tried before it is not interrupted.

  (let* ((old-ld-level (f-get-global 'ld-level state))
         (new-ld-level (1+ old-ld-level))
         (old-cbd (f-get-global 'connected-book-directory state)))
    (er-let*
     ((pair (chk-acceptable-ld-fn alist state)))
     (let ((old-ld-specials-alist (f-get-ld-specials state))
           (new-ld-specials-alist (car pair))
           (channel-closing-alist (cdr pair)))
       (if bind-flg
           (acl2-unwind-protect
            "ld-fn"
            (pprogn
             (f-put-global 'ld-level new-ld-level state)
             (ld-fn-body standard-oi0 new-ld-specials-alist state))
            (pprogn
             (f-put-global 'ld-level old-ld-level state)
             (f-put-global 'connected-book-directory old-cbd state)
             (f-put-ld-specials old-ld-specials-alist state)
             (close-channels channel-closing-alist state))
            (pprogn
             (f-put-global 'ld-level old-ld-level state)
             (f-put-global 'connected-book-directory old-cbd state)
             (f-put-ld-specials old-ld-specials-alist state)
             (close-channels channel-closing-alist state)))
         (acl2-unwind-protect
          "ld-fn"
          (pprogn (f-put-global 'ld-level new-ld-level state)
                  (ld-fn-body standard-oi0 new-ld-specials-alist state))
          (f-put-global 'ld-level old-ld-level state)
          (f-put-global 'ld-level old-ld-level state)))))))

(defun ld-fn-alist (alist state)
  (let ((standard-oi (cdr (assoc 'standard-oi alist)))
        (dir         (cdr (assoc 'dir alist)))
        (ctx         'ld)
        (os (os (w state))))
    (cond ((and (stringp standard-oi)
                dir)
           (let ((standard-oi-expanded
                  (expand-tilde-to-user-home-dir standard-oi os ctx state)))
             (cond ((absolute-pathname-string-p standard-oi-expanded nil os)
                    (er hard ctx
                        "It is illegal to supply a :DIR argument to LD here ~
                         because the supplied filename,~|~%  ~s0,~|~%is an ~
                         absolute pathname (see :DOC pathname), and hence ~
                         there is no reasonable way to merge it with a :DIR ~
                         value."
                        standard-oi))
                   (t
                    (let ((resolve-dir
                           (include-book-dir-with-chk hard 'ld dir)))
                      (cond (resolve-dir
                             (put-assoc-eq 'standard-oi
                                           (our-merge-pathnames
                                            resolve-dir
                                            standard-oi-expanded)
                                           (delete-assoc-eq 'dir alist)))
                            (t alist)))))))
          ((assoc-eq 'dir alist)
           (delete-assoc-eq 'dir alist))
          (t alist))))

#-acl2-loop-only
(defmacro with-interrupts (&rest forms)

; This macro allows, in raw Lisp for underlying Common Lisp implementations
; where we know how to do this, the interrupting of evaluation of any of the
; given forms.  We expect this behavior to take priority over any enclosing
; call of without-interrupts.

  #+ccl
  `(ccl:with-interrupts-enabled ,@forms)
  #+sbcl
  `(sb-sys:with-interrupts ,@forms)
  #+gcl
  `(let ((system:*interrupt-enable* t))
     ,@forms)
  #-(or ccl sbcl gcl)
  `(progn ,@forms))

(defun ld-fn0 (alist state bind-flg)

; We set the ld specials to the values specified in alist and then enter the
; standard ACL2 read-eval-print loop.  If bind-flg is t then the ld specials
; are restored to their pre-call values upon exit or abort.  Otherwise they are
; not.  Another interpretation of the flag is: if bind-flg is t then the load
; specials are merely "bound" locally to the values in alist, otherwise, they
; are globally smashed to values in alist.  If this call is considered the
; "top-level" call of ld-fn, bind-flg ought to be nil: the final values of the
; load specials established during the interaction survive exiting to raw lisp
; and are present when ld-fn is reentered later.  If this call is not
; "top-level" then the values established during interaction are lost on exit.

; Advice: It is best to read this function as though ld-fn1's body were
; substituted below.  Ld-fn1 is just a way to avoid duplication of code and has
; nothing to do with the unwind protection we are really implementing.

  (let ((alist (ld-fn-alist alist state)))

    #+acl2-loop-only
    (ld-fn1 (cdr (assoc-eq 'standard-oi alist)) alist state bind-flg)

; The part in UPPERCASE below is raw lisp that manages the unwind stack and
; *ld-level*.  The part in lowercase is identical to the pure ACL2 in ld-fn1
; above.  It is helpful to split the buffer, put the pure ACL2 in the top
; window and read what follows in the bottom one.  Observe that if the state is
; not live, we just use the pure ACL2.  So start with the PROGN below.

    #-acl2-loop-only
    (COND
     ((LIVE-STATE-P STATE)
      (PROGN
       (ACL2-UNWIND *LD-LEVEL* NIL)
       (PUSH NIL *ACL2-UNWIND-PROTECT-STACK*)
       (LET* ((*LD-LEVEL* (1+ *LD-LEVEL*))
              (*READTABLE* *ACL2-READTABLE*)
              (*FIRST-ENTRY-TO-LD-FN-BODY-FLG* T)
              (ABORT-OBJ (CONS 'ABORT NIL))
              (THROWN-VAL NIL)
              (LD-ERP ABORT-OBJ)
              (LD-VAL NIL)) ; below implies an abort happened
             (let* ((old-ld-level (f-get-global 'ld-level state))
                    (new-ld-level (1+ old-ld-level))
                    (old-cbd (f-get-global 'connected-book-directory state)))
               (MV-LET
                (ERP pair STATE)
                (chk-acceptable-ld-fn alist state)
                (COND
                 (ERP (ACL2-UNWIND (1- *LD-LEVEL*) NIL) (MV ERP PAIR STATE))
                 (T
                  (let ((old-ld-specials-alist (f-get-ld-specials state))
                        (new-ld-specials-alist (car pair))
                        (channel-closing-alist (cdr pair)))
                    (PUSH-CAR
                     (CONS "ld-fn"
                           (IF bind-flg
                               (FUNCTION
                                (LAMBDA
                                 NIL
                                 (pprogn
                                  (f-put-global 'ld-level old-ld-level state)
                                  (f-put-global 'connected-book-directory
                                                old-cbd state)
                                  (f-put-ld-specials old-ld-specials-alist
                                                     state)
                                  (close-channels channel-closing-alist
                                                  state))))
                               (FUNCTION
                                (LAMBDA
                                 NIL
                                 (pprogn
                                  (f-put-global 'ld-level old-ld-level state))))))
                     *ACL2-UNWIND-PROTECT-STACK*
                     'LD-FN)
                    (TAGBODY
                     LOOP
                     (UNWIND-PROTECT
                      (pprogn (f-put-global 'ld-level new-ld-level state)
                              (PROGN
                               (SETQ THROWN-VAL
                                     (CATCH
                                      'LOCAL-TOP-LEVEL
                                      (MV-LET
                                       (ERP VAL STATE)
                                       (ld-fn-body (cdr (assoc-eq 'standard-oi
                                                                  alist))
                                                   new-ld-specials-alist state)
                                       (PROGN
                                        (WHEN bind-flg
                                              (f-put-global
                                               'connected-book-directory
                                               old-cbd
                                               state))
                                        (SETQ LD-ERP ERP)
                                        (SETQ LD-VAL VAL)
                                        NIL))))
                               STATE))
                      (WITH-INTERRUPTS

; We allow interrupts for the cleanup form.  This seems acceptable because of
; how we handle ACL2 unwind-protects, calling ACL2-UNWIND; see The
; Unwind-Protect Essay.  It also seems acceptable because some Lisps don't
; disable interrupts during evaluation of unwind-protect cleanup forms, so we
; expect to allow interrupts anyhow.  And it seems important to do so, in case
; printing the gag-state needs to be interrupted; see the call of
; print-pstack-and-gag-state in prove-loop0.

                       (COND
                        (*ACL2-PANIC-EXIT-STATUS*
                         (exit-lisp *ACL2-PANIC-EXIT-STATUS*))
                        ((EQ LD-ERP ABORT-OBJ)

; We get here if the ld-fn-body failed to terminate normally.  This can happen
; either because lisp caused some error or because we threw to the tag above.
; If we threw to the tag then LD-ERP is ABORT-OBJ (because we didn't get to
; the SETQ above) and THROW-VAL is whatever we threw.  If we did not throw,
; then THROWN-VAL is NIL (because the lisp error prevented us from doing the
; SETQ THROWN-VAL).  We make the convention that we always throw non-nil
; values to the tag so as to distinguish these two cases.

                         #+akcl (si::RESET-STACK-LIMITS)
                         (COND ((EQ THROWN-VAL :ABORT)

; THROWN-VAL is always either NIL (meaning no throw occurred) or else the
; "reason" we threw.  Currently the possibilities are :ABORT (thrown when the
; user types (a!)), :POP (thrown when the user types (p!)) or :WORMHOLE-ER
; (thrown when we tried to make a non-undoable change to state while in a
; wormhole).  We only care about :ABORT.  :WORMHOLE-ER is treated as a "normal"
; lisp error, i.e., we just unwind back to here and continue at this level.
; :ABORT means we are to exit all the way back to *LD-LEVEL* 1.  :POP means
; that we are to pop up one level unless we are already at the top level.

                                (COND ((= *LD-LEVEL* 1)

; At *LD-LEVEL* = 1 we know *standard-co* is *STANDARD-OUTPUT*.

                                       (PRINC "Abort to ACL2 top-level"
                                              *STANDARD-OUTPUT*)
                                       (TERPRI *STANDARD-OUTPUT*))
                                      (T
                                       (THROW 'LOCAL-TOP-LEVEL :ABORT))))
                               ((EQ THROWN-VAL :POP)
                                (COND ((= *LD-LEVEL* 1)
                                       (PRINC "Currently at ACL2 top-level"
                                              *STANDARD-OUTPUT*))
                                      (t
                                       (COND ((= *LD-LEVEL* 2)
                                              (PRINC "Pop up to ACL2 top-level"
                                                     *STANDARD-OUTPUT*))
                                             (t
                                              (PRINC "Pop up one LD level"
                                                     *STANDARD-OUTPUT*)))
                                       (WHEN (NOT (EQ (LD-ERROR-ACTION STATE)
                                                      :ERROR))
                                             (SET-LD-ERROR-ACTION :RETURN!
                                                                  STATE))))
                                (TERPRI *STANDARD-OUTPUT*)))
                         (ACL2-UNWIND *LD-LEVEL* T)
; We first unwind back to the current level so STANDARD-OI and LD-ERROR-ACTION
; are correctly set.
                         (COND ((EQ (LD-ERROR-ACTION STATE) :CONTINUE)
                                (SETQ *FIRST-ENTRY-TO-LD-FN-BODY-FLG*
                                      (COND ((EQ THROWN-VAL :ABORT) T)
                                            (T NIL)))
                                (SETQ NEW-LD-SPECIALS-ALIST NIL)
                                (SETQ THROWN-VAL NIL)
                                (GO LOOP))
                               ((EQ (LD-ERROR-ACTION STATE) :RETURN)
                                (ACL2-UNWIND (1- *LD-LEVEL*) NIL)
                                (RETURN-FROM LD-FN0 (VALUE :ERROR)))
                               ((EQ (LD-ERROR-ACTION STATE) :RETURN!)
                                (ACL2-UNWIND (1- *LD-LEVEL*) NIL)
                                (RETURN-FROM
                                 LD-FN0
                                 (VALUE (LIST :STOP-LD
                                              (F-GET-GLOBAL 'LD-LEVEL
                                                            STATE)))))
                               (T (ACL2-UNWIND (1- *LD-LEVEL*) NIL)
                                  (RETURN-FROM LD-FN0 (MV T NIL STATE)))))
                        (T
                         (ACL2-UNWIND (1- *LD-LEVEL*) NIL)
                         (RETURN-FROM LD-FN0
                                      (MV LD-ERP LD-VAL STATE)))))))))))))))
     (T (ld-fn1 (cdr (assoc-eq 'standard-oi alist)) alist state bind-flg)))))

(defun ld-fn (alist state bind-flg)

; See ld-fn0.  Here, we just provide a little wrapper for top-level calls of
; ld-fn0 in case that there is an interrupt that isn't handled inside ld-fn0.
; To see this issue in action, evaluate the following four forms and interrupt
; the last one twice: once late in the proof attempt and once immediately upon
; printing the checkpoint summary (which is done by a call of acl2-unwind in
; the cleanup form of an unwind-protect, on behalf of a call of
; acl2-unwind-protect inside prove-loop0 that invokes
; print-pstack-and-gag-state upon an error).

; (defun foo (n acc)
;   (if (zp n)
;       acc
;     (foo (1- n)
;          (cons `(equal (nth ,n x) x)
;                acc))))
;
; (defmacro mac (n)
;   (cons 'and (foo n nil)))
;
; (set-rewrite-stack-limit 10000)
;
; (thm
;  (mac 1000)
;  :otf-flg t
;  :hints (("Goal" :do-not '(preprocess))))

  #-acl2-loop-only
  (cond (*load-compiled-stack*
         (error "It is illegal to call LD while loading a compiled book, in ~
                 this case:~%~a .~%See :DOC calling-ld-in-bad-contexts."
                (caar *load-compiled-stack*)))
        ((= *ld-level* 0)
         (return-from
          ld-fn
          (let ((complete-flg nil))
            (unwind-protect
                (mv-let (erp val state)
                        (ld-fn0 alist state bind-flg)
                        (progn (setq complete-flg t)
                               (mv erp val state)))
              (when (and (not complete-flg)
                         (not *acl2-panic-exit-status*))
                (fms "***NOTE***: An interrupt or error has occurred in the ~
                      process of cleaning up from an earlier interrupt or ~
                      error.  This is likely to leave you at the raw Lisp ~
                      prompt after you abort to the top level.  If so, then ~
                      execute ~x0 to re-enter the ACL2 read-eval-print ~
                      loop.~|~%"
                     (list (cons #\0 '(lp)))
                     *standard-co*
                     state
                     nil)))))))
  (cond ((not (f-get-global 'ld-okp state))
         (er soft 'ld
             "It is illegal to call LD in this context.  See DOC ~
              calling-ld-in-bad-contexts."))
        (t (ld-fn0 alist state bind-flg))))

(defmacro ld (standard-oi
              &key
              dir
              (standard-co 'same standard-cop)
              (proofs-co 'same proofs-cop)
              (current-package 'same current-packagep)
              (ld-skip-proofsp 'same ld-skip-proofspp)
              (ld-redefinition-action 'same ld-redefinition-actionp)
              (ld-prompt 'same ld-promptp)
              (ld-missing-input-ok 'same ld-missing-input-okp)
              (ld-pre-eval-filter 'same ld-pre-eval-filterp)
              (ld-pre-eval-print 'same ld-pre-eval-printp)
              (ld-post-eval-print 'same ld-post-eval-printp)
              (ld-evisc-tuple 'same ld-evisc-tuplep)
              (ld-error-triples 'same ld-error-triplesp)
              (ld-error-action ':RETURN!)
              (ld-query-control-alist 'same ld-query-control-alistp)
              (ld-verbose 'same ld-verbosep))

  ":Doc-Section Other

  the ACL2 read-eval-print loop, file loader, and ~il[command] processor~/
  ~bv[]
  Examples:
  (LD \"foo.lisp\")              ; read and evaluate each form in file
                               ; \"foo.lisp\", in order
  (LD \"foo.lisp\" :ld-pre-eval-print t)
                               ; as above, but print each form to standard
                               ; character output just before it is evaluated

  General Form:
  (LD standard-oi                  ; open obj in channel, stringp file name
                                   ; to open and close, or list of forms
                                   ; Optional keyword arguments:
      :dir                ...      ; use this add-include-book-dir directory
      :standard-co        ...      ; open char out or file to open and close
      :proofs-co          ...      ; open char out or file to open and close
      :current-package    ...      ; known package name
      :ld-skip-proofsp    ...      ; nil, 'include-book, or t
                                   ;   (~pl[ld-skip-proofsp])
      :ld-redefinition-action ...  ; nil or '(:a . :b)
      :ld-prompt          ...      ; nil, t, or some prompt printer fn
      :ld-missing-input-ok ...     ; nil, t, :warn, or warning message
      :ld-pre-eval-filter ...      ; :all, :query, or some new name
      :ld-pre-eval-print  ...      ; nil, t, or :never
      :ld-post-eval-print ...      ; nil, t, or :command-conventions
      :ld-evisc-tuple     ...      ; nil or '(alist nil nil level length)
      :ld-error-triples   ...      ; nil or t
      :ld-error-action    ...      ; :return!, :return, :continue or :error
      :ld-query-control-alist ...  ; alist supplying default responses
      :ld-verbose         ...)     ; nil or t~/
  ~ev[]

  ~c[Ld] is the top-level ACL2 read-eval-print loop.  (When you call ~ilc[lp],
  a little initialization is done in raw Common Lisp and then ~c[ld] is
  called.)  ~c[Ld] is also a general-purpose ACL2 file loader and a
  ~il[command] interpreter.  ~c[Ld] is actually a macro that expands to a
  function call involving ~ilc[state].  ~c[Ld] returns an ``error triple''
  ~c[(mv erp val state)] as explained below.  (For much more on error triples,
  ~pl[programming-with-state].)

  ~l[rebuild] for a variant of ~c[ld] that skips proofs.  ~l[output-to-file]
  for examples showing how to redirect output to a file.

  The arguments to ~c[ld], except for ~c[:dir], all happen to be global
  variables in ~ilc[state] (~pl[state] and ~pl[programming-with-state]).  For
  example, ~c[']~ilc[current-package] and ~c[']~ilc[ld-verbose] are global
  variables, which may be accessed via ~c[(@ current-package)] and
  ~c[(@ ld-verbose)].  When ~c[ld] is called, it ``binds'' these variables.  By
  ``binds'' we actually mean the variables are globally set but restored to
  their old values on exit.  Because ~c[ld] provides the illusion of ~il[state]
  global variables being bound, they are called ``~c[ld] specials'' (after the
  Lisp convention of calling a variable ``special'' if it is referenced freely
  after having been bound).

  Note that all arguments but the first are passed via keyword.  Any variable
  not explicitly given a value in a call retains its pre-call value, with the
  exception of ~c[:]~ilc[ld-error-action], which defaults to ~c[:return!] if
  not explicitly specified.

  Just as an example to drive the point home: If ~ilc[current-package] is
  ~c[\"ACL2\"] and you typed
  ~bv[]
  (ld *standard-oi* :current-package \"MY-PKG\")
  ~ev[]
  you would find yourself in (an inner) read-eval-print loop in which the
  ~il[current-package] was ~c[\"MY-PKG\"].  You could operate there as long as
  you wished, changing the current package at will.  But when you typed
  ~c[:]~ilc[q] you would return to the outer read-eval-print loop where the
  current package would still be ~c[\"ACL2\"].

  Roughly speaking, ~c[ld] repeatedly reads a form from ~ilc[standard-oi],
  evaluates it, and prints its result to ~ilc[standard-co].  It does this until
  the form is ~c[:]~ilc[q] or evaluates to an error triple whose value
  component is ~c[:]~ilc[q], or until the input channel or list is emptied.
  However, ~c[ld] has many bells and whistles controlled by the ~c[ld]
  specials.  Each such special is documented individually.  For example, see
  the documentation for ~ilc[standard-oi], ~ilc[current-package],
  ~ilc[ld-pre-eval-print], etc.

  A more precise description of ~c[ld] is as follows.  In the description below
  we use the ~c[ld] specials as variables, e.g., we say ``a form is read from
  ~ilc[standard-oi].''  By this usage we refer to the current value of the
  named ~il[state] global variable, e.g., we mean ``a form is read from the
  current value of ~c[']~ilc[standard-oi].'' This technicality has an important
  implication: If while interacting with ~c[ld] you change the value of one of
  the ~c[ld] specials, e.g., ~c[']~ilc[standard-oi], you will change the
  behavior of ~c[ld], e.g., subsequent input will be taken from the new value.

  Three ~c[ld] specials are treated as channels: ~ilc[standard-oi] is treated
  as an object input channel and is the source of forms evaluated by ~c[ld];
  ~ilc[standard-co] and ~ilc[proofs-co] are treated as character output
  channels and various flavors of output are printed to them.  However, the
  supplied values of these specials need not actually be channels; several
  special cases are recognized.

  If the supplied value of one of these is in fact an open channel of the
  appropriate type, that channel is used and is not closed by ~c[ld].  If the
  supplied value of one of these specials is a string, the string is treated as
  a file name in (essentially) Unix syntax (~pl[pathname]) and a channel of the
  appropriate type is opened to/from that file.  Any channel opened by ~c[ld]
  during the binding of the ~c[ld] specials is automatically closed by ~c[ld]
  upon termination.  If ~ilc[standard-co] and ~ilc[proofs-co] are equal
  strings, only one channel to that file is opened and is used for both.

  As a special convenience, when ~ilc[standard-oi] is a string and the ~c[:dir]
  argument provided and not ~c[nil], we look up ~c[:dir] in the table of
  directories maintained by ~ilc[add-include-book-dir], and prepend this
  directory to ~ilc[standard-oi] to create the filename.  (In this case,
  however, we require that ~c[standard-oi] is a relative pathname, not an
  absolute pathname.)  For example, one can write
  ~c[(ld \"arithmetic/top-with-meta.lisp\" :dir :system)] to ~c[ld] that
  particular community books library.  (Of course, you should almost always
  load books like ~c[arithmetic/top-with-meta] using ~ilc[include-book] instead
  of ~c[ld].)  If ~c[:dir] is not specified, then a relative pathname is
  resolved using the connected book directory; ~pl[cbd].

  Several other alternatives are allowed for ~ilc[standard-oi].  If
  ~ilc[standard-oi] is a true list then it is taken as the list of forms to be
  processed.  If ~ilc[standard-oi] is a list ending in an open channel, then
  ~c[ld] processes the forms in the list and then reads and processes the forms
  from the channel.  Analogously, if ~ilc[standard-oi] is a list ending a
  string, an object input channel from the named file is opened and ~c[ld]
  processes the forms in the list followed by the forms in the file.  That
  channel is closed upon termination of ~c[ld].

  In the cases that a string is to be converted to an object input channel ~-[]
  that is, when ~ilc[standard-oi] is a string or is a list ending in a string
  ~-[] an error occurs by default if the conversion fails, presumably because
  the file named by the string does not exist.  However, if keyword argument
  ~c[:ld-missing-input-ok] is ~c[t], then ~c[ld] immediately returns without
  error in this case, but without reading or executing any forms, as though
  ~c[standard-oi] is ~c[nil] and keyword arguments ~c[:ld-verbose] and
  ~c[ld-prompt] both have value ~c[nil].  The other legal values for
  ~c[:ld-missing-input-ok] are ~c[nil], which gives the default behavior, and
  ~c[:warn], which behaves the same as ~c[t] except that a warning is printed,
  which contains the same information as would be printed for the default error
  described above.

  The remaining ~c[ld] specials are handled more simply and generally have to
  be bound to one of a finite number of tokens described in the ~c[:]~ilc[doc]
  entries for each ~c[ld] special.  Should any ~c[ld] special be supplied an
  inappropriate value, an error message is printed.

  Next, if ~ilc[ld-verbose] is ~c[t], ~c[ld] prints the message ``ACL2 loading
  name'' where ~c[name] names the file or channel from which forms are being
  read.  At the conclusion of ~c[ld], it will print ``Finished loading name''
  if ~ilc[ld-verbose] is ~c[t].

  Finally, ~c[ld] repeatedly executes the ACL2 read-eval-print step, which may
  be described as follows.  A ~il[prompt] is printed to ~ilc[standard-co] if
  ~ilc[ld-prompt] is non-~c[nil].  The format of the ~il[prompt] is determined
  by ~ilc[ld-prompt].  If it is ~c[t], the default ACL2 ~il[prompt] is used.
  If it is any other non-~c[nil] value then it is treated as an ACL2 function
  that will print the desired ~il[prompt].  ~l[ld-prompt].  In the exceptional
  case where ~c[ld]'s input is coming from the terminal ~c[(*standard-oi*)] but
  its output is going to a different sink (i.e., ~ilc[standard-co] is not
  ~ilc[*standard-co*]), we also print the ~il[prompt] to the terminal.

  ~c[Ld] then reads a form from ~ilc[standard-oi].  If the object read is a
  keyword, ~c[ld] constructs a ``keyword command form'' by possibly reading
  several more objects.  ~l[keyword-commands].  This construction process is
  sensitive to the value of ~ilc[ld-keyword-aliases].  ~l[ld-keyword-aliases].
  Otherwise, the object read is treated as the command form.

  ~c[Ld] next decides whether to evaluate or skip this form, depending on
  ~ilc[ld-pre-eval-filter].  Initially, the filter must be either ~c[:all],
  ~c[:query], or a new name.  If it is ~c[:all], it means all forms are
  evaluated.  If it is ~c[:query], it means each form that is read is displayed
  and the user is queried.  Otherwise, the filter is a name and each form that
  is read is evaluated as long as the name remains new, but if the name is ever
  introduced then no more forms are read and ~c[ld] terminates.
  ~l[ld-pre-eval-filter].

  If the form is to be evaluated, then ~c[ld] first prints the form to
  ~ilc[standard-co], if ~ilc[ld-pre-eval-print] is ~c[t].  With this feature,
  ~c[ld] can process an input file or form list and construct a script of the
  session that appears as though each form was typed in.
  ~l[ld-pre-eval-print].

  ~c[Ld] then evaluates the form, with ~ilc[state] bound to the current
  ~il[state].  The result is some list of (multiple) values.  If a ~il[state]
  is among the values, then ~c[ld] uses that ~il[state] as the subsequent
  current ~il[state].

  Depending on ~ilc[ld-error-triples], ~c[ld] may interpret the result as an
  ``error.''  ~l[ld-error-triples].  We first discuss ~c[ld]'s behavior if no
  error signal is detected (either because none was sent or because ~c[ld] is
  ignoring them because ~ilc[ld-error-triples] is ~c[nil]).

  In the case of a non-erroneous result, ~c[ld] does two things: First, if the
  logical ~il[world] in the now current ~il[state] is different than the
  ~il[world] before execution of the form, ~c[ld] adds to the ~il[world] a
  ``~il[command] landmark'' containing the form evaluated.
  ~l[command-descriptor].  Second, ~c[ld] prints the result to
  ~ilc[standard-co], but only if ~ilc[ld-post-eval-print] is not ~c[nil].  The
  result is printed as a list of (multiple) values unless
  ~ilc[ld-post-eval-print] is ~c[:command-conventions],
  ~ilc[ld-error-triples] is ~c[t], and the result is an ``error triple'', i.e.,
  of the form ~c[(mv * * state)] (~pl[error-triples]).  In that case, only the
  non-erroneous ``value'' component of the result is printed.
  ~l[ld-post-eval-print].

  Whenever ~c[ld] prints anything (whether the input form, a query, or
  some results) it ``eviscerates'' it if ~c[ld-evisc-tuple] is non-~c[nil].
  Essentially, evisceration is a generalization of Common Lisp's use
  of ~c[*print-level*] and ~c[*print-length*] to hide large substructures.
  ~l[evisc-tuple] and also ~pl[set-iprint].

  We now return to the case of a form whose evaluation signals an error.  In
  this case, ~c[ld] first restores the ACL2 logical ~il[world] to what it was
  just before the erroneous form was evaluated.  Thus, a form that partially
  changes the ~il[world] (i.e., begins to store properties) and then signals an
  error, has no effect on the ~il[world].  You may see this happen on
  ~il[command]s that execute several ~il[events] (e.g., an ~ilc[encapsulate] or
  a ~ilc[progn] of several ~ilc[defuns]): even though the output makes it
  appear that the initial ~il[events] were executed, if an error is signalled
  by a later event the entire block of ~il[events] is discarded.

  After rolling back, ~c[ld] takes an action determined by
  ~ilc[ld-error-action].  If the action is ~c[:continue], ~c[ld] merely
  iterates the read-eval-print step.  Note that nothing suggestive of the value
  of the ``erroneous'' form is printed.  If the action is ~c[:return], ~c[ld]
  terminates normally; similarly if the action is ~c[:return!], but a special
  value is returned that can cause superior ~c[ld] commands to terminate;
  ~pl[ld-error-action] for details.  If the action is ~c[:error], ~c[ld]
  terminates signalling an error to its caller.  If its caller is in fact
  another instance of ~c[ld] and that instance is watching out for error
  signals, the entire ~il[world] created by the inner ~c[ld] will be discarded
  by the outer ~c[ld] if the inner ~c[ld] terminates with an error.

  ~c[Ld] returns an error triple, ~c[(mv erp val state)].  ~c[Erp] is ~c[t] or
  ~c[nil] indicating whether an error is being signalled.  If no error is
  signalled, ~c[val] is the ``reason'' ~c[ld] terminated and is one of
  ~c[:exit] (meaning ~c[:]~ilc[q] was read), ~c[:eof] (meaning the input source
  was exhausted), ~c[:error] (meaning an error occurred but has been supressed),
  ~c[:filter] (meaning the ~ilc[ld-pre-eval-filter] terminated ~c[ld]), or a
  cons pair whose first component is the symbol ~c[:STOP-LD], which typically
  indicates that an error occurred while the value of variable
  ~c[']~ilc[ld-error-action] was ~c[:RETURN!].  ~l[ld-error-action] for
  details of this last case."

  `(ld-fn
    (list ,@(append
             (list `(cons 'standard-oi ,standard-oi))
             (if dir
                 (list `(cons 'dir ,dir))
                 nil)
             (if standard-cop
                 (list `(cons 'standard-co ,standard-co))
                 nil)
             (if proofs-cop
                 (list `(cons 'proofs-co ,proofs-co))
                 nil)
             (if current-packagep
                 (list `(cons 'current-package ,current-package))
                 nil)
             (if ld-skip-proofspp
                 (list `(cons 'ld-skip-proofsp ,ld-skip-proofsp))
                 nil)
             (if ld-redefinition-actionp
                 (list `(cons 'ld-redefinition-action
                              ,ld-redefinition-action))
                 nil)
             (if ld-promptp
                 (list `(cons 'ld-prompt ,ld-prompt))
                 nil)
             (if ld-missing-input-okp
                 (list `(cons 'ld-missing-input-ok ,ld-missing-input-ok))
               nil)
             (if ld-pre-eval-filterp
                 (list `(cons 'ld-pre-eval-filter ,ld-pre-eval-filter))
                 nil)
             (if ld-pre-eval-printp
                 (list `(cons 'ld-pre-eval-print ,ld-pre-eval-print))
                 nil)
             (if ld-post-eval-printp
                 (list `(cons 'ld-post-eval-print ,ld-post-eval-print))
                 nil)
             (if ld-evisc-tuplep
                 (list `(cons 'ld-evisc-tuple ,ld-evisc-tuple))
                 nil)
             (if ld-error-triplesp
                 (list `(cons 'ld-error-triples ,ld-error-triples))
                 nil)
             (list `(cons 'ld-error-action ,ld-error-action))
             (if ld-query-control-alistp
                 (list `(cons 'ld-query-control-alist ,ld-query-control-alist))
                 nil)
             (if ld-verbosep
                 (list `(cons 'ld-verbose ,ld-verbose))
                 nil)))
    state
    t))

(defdoc calling-ld-in-bad-contexts
  ":Doc-Section ld

  errors caused by calling ~ilc[ld] in inappropriate contexts~/

  The macro ~ilc[ld] was designed to be called directly in the top-level ACL2
  loop, although there may be a few occasions for calling it from functions.
  ACL2 cannot cope with invocations of ~ilc[ld] during the process of loading a
  compiled file for a book, so this is an error.

  To see how that can happen, consider the following book, where file
  ~c[const.lsp] contains the single form ~c[(defconst *foo* '(a b))].
  ~bv[]
    (in-package \"ACL2\")
    (defttag t)
    (progn! (ld \"const.lsp\"))
  ~ev[]
  An attempt to certify this book will cause an error, but that particular
  error can be avoided, as discussed below.  If the book is certified, however,
  with production of a corresponding compiled file (which is the default
  behavior for ~ilc[certify-book]), then any subsequent call of
  ~ilc[include-book] that loads this compiled file will cause an error.
  Again, this error is necessary because of how ACL2 is designed; specifically,
  this ~ilc[ld] call would interfere with tracking of constant definitions when
  loading the compiled file for the book.

  Because including such a book (with a compiled file) causes an error, then as
  a courtesy to the user, ACL2 arranges that the certification will fail (thus
  avoiding a surprise later when trying to include the book).  The error in
  that case will look as follows.
  ~bv[]
    ACL2 Error in LD:  It is illegal to call LD in this context.  See DOC
    calling-ld-in-bad-contexts.
  ~ev[]
  If you really think it is OK to avoid this error, you can get around it by
  setting ~il[state] global variable ~c[ld-okp] to t:  ~c[(assign ld-okp t)].
  You can then certify the book in the example above, but you will still not be
  able to include it with a compiled file.~/~/")

(defmacro quick-test nil

; We might want to add other events to the list below to test a wide variety of
; features.

  '(ld '((defun app (x y)
           (declare (xargs :guard (true-listp x)))
           (if (eq x nil) y (cons (car x) (app (cdr x) y))))
         (defthm true-listp-app
           (implies (true-listp x) (equal (true-listp (app x y)) (true-listp y))))
         :program
         (defun rev (x)
           (declare (xargs :guard (true-listp x)))
           (if (eq x nil) nil (app (rev (cdr x)) (list (car x)))))
         :logic
         (verify-termination rev)
         (verify-guards rev)
         (defthm true-listp-rev
           (implies (true-listp x) (true-listp (rev x)))
           :rule-classes :type-prescription)
         (defthm rev-rev (implies (true-listp x) (equal (rev (rev x)) x))))
       :ld-pre-eval-print t
       :ld-error-action :return))

(defun wormhole-prompt (channel state)
  (fmt1 "Wormhole ~s0~sr ~@1~*2"
        (list (cons #\0 (f-get-global 'current-package state))
              (cons #\1 (defun-mode-prompt-string state))
              (cons #\r
                    #+:non-standard-analysis "(r)"
                    #-:non-standard-analysis "")
              (cons #\2
                    (list "" ">" ">" ">"
                          (make-list-ac (- (f-get-global 'ld-level state) 1) nil nil))))
        0 channel state nil))

(defun reset-ld-specials-fn (reset-channels-flg state)

; We restore all of the ld specials to their initial, top-level
; values, except for the three channels, standard-oi, standard-co, and
; proofs-co, which are not reset unless the reset-channels-flg is t.
; Of course, if this function is called while under a recursive ld,
; then when we pop out of that ld, the reset values will be lost.

  (f-put-ld-specials
   (cond (reset-channels-flg *initial-ld-special-bindings*)
         (t (cdddr *initial-ld-special-bindings*)))
   state))

(defmacro reset-ld-specials (reset-channels-flg)

  ":Doc-Section Other
  restores initial settings of the ~ilc[ld] specials~/
  ~bv[]
  Examples:
  (reset-ld-specials t)
  (reset-ld-specials nil)
  ~ev[]~/

  Roughly speaking, the ~ilc[ld] specials are certain ~il[state] global
  variables, such as ~ilc[current-package], ~ilc[ld-prompt], and
  ~ilc[ld-pre-eval-filter], which are managed by ~ilc[ld] as though they were
  local variables.  These variables determine the channels on which ~ilc[ld]
  reads and prints and control many options of ~ilc[ld].  ~l[ld] for
  the details on what the ~ilc[ld] specials are.

  This function, ~c[reset-ld-specials], takes one Boolean argument, ~c[flg].
  The function resets all of the ~ilc[ld] specials to their initial,
  top-level values, except for the three channel variables,
  ~ilc[standard-oi], ~ilc[standard-co], and ~ilc[proofs-co], which are reset to their
  initial values only if ~c[flg] is non-~c[nil].  Of course, if you are in a
  recursive call of ~ilc[ld], then when you exit that call, the ~ilc[ld] specials
  will be restored to the values they had at the time ~ilc[ld] was called
  recursively.  To see what the initial values are, inspect the value
  of the constant ~c[*initial-ld-special-bindings*]."

  `(reset-ld-specials-fn ,reset-channels-flg state))

(defun maybe-reset-defaults-table1
  (key pre-defaults-tbl post-defaults-tbl state)
  (let* ((pre-val (cdr (assoc-eq key pre-defaults-tbl)))
         (post-val (cdr (assoc-eq key post-defaults-tbl)))
         (cmd `(table acl2-defaults-table ,key ',pre-val)))
    (if (equal pre-val post-val)
        (value nil)
      (er-let*
       ((ans
         (acl2-query
          :ubt-defaults
          '("The default ~s0 was ~x1 before undoing, but will be ~x2 after ~
             undoing unless the command ~x3 is executed.  Do you wish to ~
             re-execute this command after the :ubt?"
            :y t :n nil
            :? ("If you answer in the affirmative, then the command ~X34 will ~
                 be executed on your behalf.  This will make the default ~s0 ~
                 equal to ~x1, which is what it was just before your :ubt ~
                 command was executed.  Otherwise, the default ~s0 will be ~
                 what it is in the world after the undoing, namely ~x2.  See ~
                 also :DOC acl2-defaults-table."
                :y t :n nil))
          (list (cons #\0 (string-downcase (symbol-name key)))
                (cons #\1 pre-val)
                (cons #\2 post-val)
                (cons #\3 cmd)
                (cons #\4 nil))
          state)))
       (if ans
           (ld (list cmd)
               :ld-pre-eval-filter :all
               :ld-pre-eval-print t
               :ld-post-eval-print :command-conventions
               :ld-evisc-tuple (abbrev-evisc-tuple state)
               :ld-error-triples t
               :ld-error-action :return)
         (value nil))))))

(defun maybe-reset-defaults-table2
  (keys pre-defaults-tbl post-defaults-tbl state)
  (if keys
      (er-progn (maybe-reset-defaults-table1
                 (car keys) pre-defaults-tbl post-defaults-tbl state)
                (maybe-reset-defaults-table2
                 (cdr keys) pre-defaults-tbl post-defaults-tbl state))
    (value nil)))

(defun maybe-reset-defaults-table (pre-defaults-tbl post-defaults-tbl state)
  (maybe-reset-defaults-table2 (union-equal (strip-cars pre-defaults-tbl)
                                            (strip-cars post-defaults-tbl))
                               pre-defaults-tbl post-defaults-tbl state))

(defun delete-something (lst)

; Lst must be non-nil.  We return a list that is one shorter than lst by either
; dropping the first nil we find in lst or, if there are no nils, the last
; element.

  (cond ((null (cdr lst)) nil)
        ((null (car lst)) (cdr lst))
        (t (cons (car lst) (delete-something (cdr lst))))))

(defun store-in-kill-ring (x0 ring)

; A "kill ring" is a fancy queue that stores a fixed number, say n, of non-nil
; items in the order in which they were stored.  Only the most recent n non-nil
; items stored are saved.  When a non-nil item is stored and the ring is full,
; the oldest item is dropped out and lost.  So we have described a queue so
; far.  The only other operation on kill rings is "rotate" which selects an
; item from the kill ring but does not remove it.  Given a ring containing n
; items, n+1 rotations will return the each of the items in turn and in the
; reverse order in which they were stored.  More on rotation later.

; Kill rings are just lists of the n items, in order.  The length of the list
; is n but there may be nils in the list.  The initial kill ring of length n
; is just n nils.

  (cond ((or (null x0)          ; item is nil or the size of the
             (null ring))       ; ring is 0.  We store nothing.
         ring)
        (t (cons x0 (delete-something ring)))))

(defun rotate-kill-ring1 (ring xn)
  (cond ((null ring) xn)
        ((car ring) (append ring xn))
        (t (rotate-kill-ring1 (cdr ring) (append xn (list nil))))))

(defun rotate-kill-ring (ring xn)

; See store-in-kill-ring for background on rings.  Xn is an element to add to
; the ring.  We step the ring once, returning (mv item ring'), where item is
; the most recently added item in ring and ring' is the result of removing that
; item and adding xn as the oldest item in the ring.  Thus, a series of
; rotate-kill-ring n+1 long will return us to the original configuration.

  (cond ((null (car ring)) (mv nil ring))
        (t (mv (car ring)
               (rotate-kill-ring1 (cdr ring) (list xn))))))

(defun ubt-ubu-fn1 (kwd wrld pred-wrld state)
  (let ((pre-defaults-table (table-alist 'acl2-defaults-table wrld)))
    (er-let*
     ((redo-cmds (ubt-ubu-query kwd wrld pred-wrld nil
                                nil wrld state nil)))
     (pprogn
      (f-put-global
       'undone-worlds-kill-ring
       (store-in-kill-ring wrld
                           (f-get-global
                            'undone-worlds-kill-ring
                            state))
       state)
      (set-w 'retraction pred-wrld state)
      (let ((redo-cmds (if (eq (car redo-cmds)
                               (default-defun-mode pred-wrld))
                           (cdr redo-cmds)
                         redo-cmds)))
        (er-progn
         (if redo-cmds
             (mv-let (col state)
                     (fmt "Undoing complete.  Redoing started...~%"
                          nil (standard-co state) state nil)
                     (declare (ignore col))
                     (value nil))
           (value nil))
         (if redo-cmds
             (ld redo-cmds
                 :ld-redefinition-action '(:doit! . :overwrite)
                 :ld-pre-eval-filter :all
                 :ld-pre-eval-print t
                 :ld-post-eval-print :command-conventions
                 :ld-evisc-tuple (abbrev-evisc-tuple state)
                 :ld-error-triples t
                 :ld-error-action :return
                 :ld-query-control-alist
                 (cons '(:redef :y)
                       (ld-query-control-alist state)))
           (value nil))
         (if redo-cmds
             (mv-let (col state)
                     (fmt1 "Redoing complete.~%~%"
                           nil 0 (standard-co state) state nil)
                     (declare (ignore col))
                     (value nil))
           (value nil))
         (maybe-reset-defaults-table
          pre-defaults-table
          (table-alist 'acl2-defaults-table (w state))
          state)
         (io? event nil (mv erp val state)
              ()
              (pcs-fn :x :x nil state))
         (value :invisible)))))))

(defun ubt-ubu-fn (kwd cd state)

; Kwd is :ubt or :ubu.

  (let* ((wrld (w state))
         (command-number-baseline
          (access command-number-baseline-info
                  (global-val 'command-number-baseline-info wrld)
                  :current)))
    (er-let* ((cmd-wrld (er-decode-cd cd wrld kwd state)))
             (cond ((if (eq kwd :ubt)
                        (<= (access-command-tuple-number (cddar cmd-wrld))
                            command-number-baseline)
                      (< (access-command-tuple-number (cddar cmd-wrld))
                         command-number-baseline))

; We prevent ubt and ubu from going into prehistory, thus burning users due to
; typos.  But sometimes developers need to do it.  Here is how from within the
; ACL2 loop:

; (set-state-ok t)
; (defun my-ubt-ubu-fn (inclp x state) (declare (xargs :guard t)) (value x))
; :q
; Grab this defun, rename it to my-ubt-ubu-fn, edit out the cond clause
; containing this comment and define my-ubt-ubu-fn in raw lisp.
; (lp)
; (my-ubt-ubu-fn t 'sys-fn state), where sys-fn is the desired target of the
; ubt.

                    (cond
                     ((let ((command-number-baseline-original
                             (access command-number-baseline-info
                                     (global-val 'command-number-baseline-info wrld)
                                     :original)))
                        (if (eq kwd :ubt)
                            (<= (access-command-tuple-number (cddar cmd-wrld))
                                command-number-baseline-original)
                          (< (access-command-tuple-number (cddar cmd-wrld))
                             command-number-baseline-original)))
                      (er soft kwd "Can't undo into system initialization."))
                     (t (er soft kwd
                            "Can't undo into prehistory.  See :DOC ~
                             reset-prehistory."))))
                   ((and (eq kwd :ubu) (equal wrld cmd-wrld))
                    (er soft kwd
                        "Can't undo back to where we already are!"))
                   (t
                    (let ((pred-wrld (if (eq kwd :ubt)
                                         (scan-to-command (cdr cmd-wrld))
                                       cmd-wrld)))
                      (ubt-ubu-fn1 kwd wrld pred-wrld state)))))))

(defun ubt!-ubu!-fn (kwd cd state)

; Kwd is :ubt or :ubu.

  (state-global-let*
   ((ld-query-control-alist
     (list* `(,kwd :n!)
            '(:ubt-defaults :n)
            (@ ld-query-control-alist)))
    (inhibit-output-lst
     (union-equal '(observation warning error)
                  (@ inhibit-output-lst))))
   (mv-let (erp val state)
           (ubt-ubu-fn kwd cd state)
           (declare (ignore erp val))
           (value :invisible))))

(defmacro ubt-prehistory ()

  ":Doc-Section History

  undo the ~il[command]s back through the last ~ilc[reset-prehistory] event~/

  This command is only used to eliminate a ~ilc[reset-prehistory] event.  If
  your most recent ~c[reset-prehistory] event does not have a flag argument
  of ~c[t], then ~c[:ubt-prehistory] undoes all command back through, and
  including, that ~c[reset-prehistory] event.~/~/"

  (list 'ubt-prehistory-fn 'state))

(defun ubt-prehistory-fn (state)
  (let* ((ctx 'ubt-prehistory)
         (wrld (w state))
         (command-number-baseline-info
          (global-val 'command-number-baseline-info wrld))
         (command-number-baseline
          (access command-number-baseline-info
                  command-number-baseline-info
                  :current)))
    (cond ((eql command-number-baseline
                (access command-number-baseline-info
                        command-number-baseline-info
                        :original))
           (er soft ctx
               "There is no reset-prehistory event to undo."))
          ((access command-number-baseline-info
                   command-number-baseline-info
                   :permanent-p)
           (er soft ctx
               "It is illegal to undo a reset-prehistory event that had its ~
                permanent-p flag set to t.  See :DOC reset-prehistory."))
          (t (er-let* ((val (ubt-ubu-fn1
                             :ubt-prehistory
                             wrld
                             (scan-to-command
                              (cdr (lookup-world-index
                                    'command command-number-baseline wrld)))
                             state)))
                      (er-progn
                       (reset-kill-ring t state)
                       (prog2$ #-acl2-loop-only
                               (pop *checkpoint-world-len-and-alist-stack*)
                               #+acl2-loop-only
                               nil
                               (value val))))))))

(defun oops-warning (state)

; If the set of Lisps that compile all functions changes from {sbcl, ccl}, then
; change the #+/#- below accordingly.

  #+(or sbcl ccl)
  (fms "Installing the requested world.~|~%"
       nil (f-get-global 'standard-co state) state nil)
  #-(or sbcl ccl)
  (fms "Installing the requested world.  Note that functions being re-defined ~
        during this procedure will not have compiled definitions, even if ~
        they had compiled definitions before the last :ubt or :u.~|~%"
       nil (f-get-global 'standard-co state) state nil))

(defun oops-fn (state)
  (mv-let (new-wrld new-kill-ring)
          (rotate-kill-ring (f-get-global 'undone-worlds-kill-ring state)
                            (w state))
          (cond ((null new-wrld)
                 (cond ((null (f-get-global 'undone-worlds-kill-ring state))
                        (er soft :oops
                            "Oops has been disabled in this ACL2 session.  ~
                             See :DOC reset-kill-ring"))
                       (t
                        (er soft :oops
                            "ACL2 cannot execute :oops at this time, ~
                             presumably because you have never executed :ubt ~
                             or :u during this ACL2 session (at least not ~
                             since the last invocation of reset-kill-ring)."))))
                (t (er-progn
                    (revert-world-on-error
                     (pprogn
                      (oops-warning state)
                      (set-w! new-wrld state)
                      (er-progn (pcs-fn :x :x nil state)
                                (value nil))))
                    (pprogn
                     (f-put-global 'undone-worlds-kill-ring
                                   new-kill-ring state)
                     (value :invisible)))))))

(defmacro oops nil

  ":Doc-Section History

  undo a ~c[:u] or ~c[:]~ilc[ubt]~/

  The keyword ~il[command] ~c[:oops] will undo the most recent ~c[:]~ilc[ubt] (or ~c[:u],
  which we here consider just another ~c[:]~ilc[ubt]).  A second ~c[:oops] will undo
  the next most recent ~c[:]~ilc[ubt], a third will undo the ~c[:]~ilc[ubt] before that
  one, and a fourth ~c[:oops] will return the logical ~il[world] to its
  configuration before the first ~c[:oops].~/

  Consider the logical world (~pl[world]) that represents the
  current extension of the logic and ACL2's rules for dealing with it.
  The ~c[:]~ilc[ubt] and ~c[:u] ~il[command]s ``roll back'' to some previous ~il[world]
  (~pl[ubt]).  Sometimes these ~il[command]s are used to inadvertently
  undo useful work and user's wish they could ``undo the last undo.''
  That is the function provided by ~c[:oops].

  ~c[:Oops] is best described in terms of an implementation.  Imagine a
  ring of four ~il[world]s and a marker (~c[*]) indicating the current ACL2
  ~il[world]:
  ~bv[]
               *
             w0
           /    \\
         w3      w1
           \\    /
             w2
  ~ev[]
  This is called the ``kill ring'' and it is maintained as follows.
  When you execute an event the current ~il[world] is extended and the kill
  ring is not otherwise affected.  When you execute ~c[:]~ilc[ubt] or ~c[:u], the
  current ~il[world] marker is moved one step counterclockwise and that
  ~il[world] in the ring is replaced by the result, say ~c[w0'], of the ~c[:]~ilc[ubt] or
  ~c[:u].
  ~bv[]
             w0
           /    \\
        *w0'     w1
           \\    /
             w2
  ~ev[]
  If you were to execute ~il[events] at this point, ~c[w0'] would be extended
  and no other changes would occur in the kill ring.

  When you execute ~c[:oops], the marker is moved one step clockwise.
  Thus the kill ring becomes
  ~bv[]
               *
             w0
           /    \\
         w0'     w1
           \\    /
             w2
  ~ev[]
  and the current ACL2 ~il[world] is ~c[w0] once again.  That is, ~c[:oops]
  ``undoes'' the ~c[:]~ilc[ubt] that produced ~c[w0'] from ~c[w0].  Similarly,
  a second ~c[:oops] will move the marker to ~c[w1], undoing the undo that
  produced ~c[w0] from ~c[w1].  A third ~c[:oops] makes ~c[w2] the current
  ~il[world].  Note however that a fourth ~c[:oops] restores us to the
  configuration previously displayed above in which ~c[w0'] has the marker.

  In general, the kill ring contains the current ~il[world] and the three
  most recent ~il[world]s in which a ~c[:]~ilc[ubt] or ~c[:u] were done.

  While ~c[:]~ilc[ubt] may appear to discard the information in the ~il[events]
  undone, we can see that the ~il[world] in which the ~c[:]~ilc[ubt] occurred is
  still available.  No information has been lost about that ~il[world].
  But ~c[:]~ilc[ubt] does discard information!  ~c[:]~ilc[Ubt] discards the information
  necessary to recover from the third most recent ~ilc[ubt]!  An ~c[:oops], on
  the other hand, discards no information, it just selects the next
  available ~il[world] on the kill ring and doing enough ~c[:oops]es will
  return you to your starting point.

  We can put this another way.  You can freely type ~c[:oops] and inspect
  the ~il[world] that you thus obtain with ~c[:]~ilc[pe], ~c[:]~ilc[pc], and other ~il[history]
  ~il[command]s.  You can repeat this as often as you wish without risking
  the permanent loss of any information.  But you must be more careful
  typing ~c[:]~ilc[ubt] or ~c[:u].  While ~c[:oops] makes ~c[:]~ilc[ubt] seem ``safe'' because the
  most recent ~c[:]~ilc[ubt] can always be undone, information is lost when you
  execute ~c[:]~ilc[ubt].

  We note that ~c[:ubt] and ~c[:u] may remove compiled definitions (but note
  that in some Lisps, including CCL (OpenMCL) and SBCL, functions are always
  compiled).  When the original world is restored using ~c[:oops], restored
  functions will not generally be compiled (except for Lisps as above), though
  the user can remedy this situation; ~pl[comp].

  Finally, we note that our implementation of ~c[oops] can use a significant
  amount of memory, because of the saving of old logical ~il[world]s.  Most
  users are unlikely to experience a memory problem, but if you do, then you
  may want to disable ~c[oops] by evaluting ~c[(reset-kill-ring 0 state)];
  ~pl[reset-kill-ring]."

  '(oops-fn state))

(defmacro i-am-here ()

  ":Doc-Section Miscellaneous

  a convenient marker for use with ~ilc[rebuild]~/
  ~bv[]
  Example Input File for Rebuild:
  (defun fn1 (x y) ...)
  (defthm lemma1 ...)
  (defthm lemma2 ...)
  (i-am-here)
  The following lemma won't go through.  I started
  typing the hint but realized I need to prove a
  lemma first.  See the failed proof attempt in foo.bar.
  I'm going to quit for the night now and resume tomorrow
  from home.

  (defthm lemma3 ...
    :hints ((\"Goal\" :use (:instance ???
  ...
  ~ev[]~/

  By putting an ~c[(i-am-here)] form at the ``frontier'' of an evolving file of
  ~il[command]s, you can use ~ilc[rebuild] to load the file up to the
  ~c[(i-am-here)].  ~c[I-am-here] simply returns an error
  triple (~pl[error-triples]) that indicates an error, and any form that
  ``causes an error'' will do the same job.  Note that the text of the file
  after the ~c[(i-am-here)] need not be machine readable."

  '(mv-let (col state)
           (fmt1 "~ I-AM-HERE~|" nil 0 (standard-co state) state nil)
           (declare (ignore col))
           (mv t nil state)))

(defun rebuild-fn-read-filter (file state)
  (state-global-let*
   ((standard-oi *standard-oi*)
    (standard-co *standard-co*))
   (er-let*
     ((ans
       (acl2-query
        :rebuild
        '("How much of ~x0 do you want to process?"
          :t :all :all :all :query :query :until :until
          :? ("If you answer T or ALL, then the entire file will be ~
               loaded.  If you answer QUERY, then you will be asked ~
               about each command in the file.  If you answer UNTIL, ~
               then you should also type some name after the UNTIL ~
               and we will then proceed to process all of the events ~
               in file until that name has been introduced.  Rebuild ~
               automatically stops if any command causes an error.  ~
               When it stops, it leaves the logical world in the ~
               state it was in immediately before the erroneous ~
               command.  Thus, another way to use rebuild is to get ~
               into the habit of planting (i-am-here) -- or any other ~
               form that causes an error when executed -- and then ~
               using the filter T or ALL when you rebuild."
              :t :all :all :all :query :query :until :until))
        (list (cons #\0 file))
        state)))
     (cond ((eq ans :until)
            (state-global-let*
             ((infixp nil))
             (read-object *standard-oi* state)))
           (t (value ans))))))

(defun rebuild-fn (file filter filterp dir state)
  (er-let*
      ((filter
        (if filterp
            (value (if (eq filter t) :all filter))
          (rebuild-fn-read-filter file state))))
    (mv-let (erp val state)
      (ld file
          :dir dir
          :standard-co *standard-co*
          :proofs-co *standard-co*
          :ld-skip-proofsp t
          :ld-prompt nil
          :ld-missing-input-ok nil
          :ld-pre-eval-filter filter
          :ld-pre-eval-print nil
          :ld-post-eval-print :command-conventions
          :ld-evisc-tuple (abbrev-evisc-tuple state)
          :ld-error-triples t
          :ld-error-action :return!
          :ld-query-control-alist '((:filter . nil) . t)
          :ld-verbose t)
      (declare (ignore erp val))
      (value t))))

(defmacro rebuild (file &optional (filter 'nil filterp)
                        &key dir)

  ":Doc-Section Other

  a convenient way to reconstruct your old ~il[state]~/
  ~bv[]
  Examples:
  ACL2 !>(rebuild \"project.lisp\")
  ACL2 !>(rebuild \"project.lisp\" t)
  ACL2 !>(rebuild \"project.lisp\" t :dir :system)
  ACL2 !>(rebuild \"project.lisp\" :all)
  ACL2 !>(rebuild \"project.lisp\" :query)
  ACL2 !>(rebuild \"project.lisp\" 'lemma-22)
  ~ev[]~/

  ~c[Rebuild] allows you to assume all the ~il[command]s in a given file or
  list, supplied in the first argument.  Because ~c[rebuild] processes an
  arbitrary sequence of ~il[command]s with ~ilc[ld-skip-proofsp] ~c[t], it is
  unsound!  However, if each of these ~il[command]s is in fact admissible,
  then processing them with ~c[rebuild] will construct the same logical
  ~il[state] that you would be in if you typed each ~il[command] and waited
  through the proofs again.  Thus, ~c[rebuild] is a way to reconstruct a
  ~il[state] previously obtained by proving your way through the ~il[command]s.

  The second, optional argument to ~c[rebuild] is a ``filter''
  (~pl[ld-pre-eval-filter]) that lets you specify which ~il[command]s
  to process.  You may specify ~c[t], ~c[:all], ~c[:query], or a new logical name.
  ~c[t] and ~c[:all] both mean that you expect the entire file or list to be
  processed.  ~c[:query] means that you will be asked about each ~il[command]
  in turn.  A new name means that all ~il[command]s will be processed as
  long as the name is new, i.e., ~c[rebuild] will stop processing ~il[command]s
  immediately after executing a ~il[command] that introduces name.  ~C[Rebuild]
  will also stop if any ~il[command] causes an error.  You may therefore
  wish to plant an erroneous form in the file, e.g., ~c[(mv t nil state)],
  (~pl[ld-error-triples]), to cause ~c[rebuild] to stop there.  The
  form ~c[(i-am-here)] is such a pre-defined form.  If you do not specify
  a filter, ~c[rebuild] will query you for one.

  Inspection of the definition of ~c[rebuild], e.g., via ~c[:]~ilc[pc] ~c[rebuild-fn],
  will reveal that it is just a glorified call to the function ~ilc[ld].
  ~l[ld] if you find yourself wishing that ~c[rebuild] had additional
  functionality.

  If you supply the above ``filter'' argument, then you may also supply the
  keyword argument ~c[:dir], which is then passed to ~c[ld]; ~pl[ld]."

  `(rebuild-fn ,file ,filter ,filterp ,dir state))

;           The Tall Texas Tale about  BIG-CLOCK

; Like any Lisp system, it may be said, loosely speaking, that ACL2
; typically reads a form, evaluates it in the current state, and
; prints the result.  This read-eval-print activity in ACL2 is done by
; the function ld-fn.  When the user enters ACL2 by invoking (LP),
; ld-fn is called to do the work.

; The read phase of the read-eval-print activity is done with the
; read-object function, which calls the Common Lisp read function.
; This read is influenced by *package*, *readtable*, and *features*,
; as described in acl2.lisp.

; The semantics of an ACL2 read-eval-print cycles is best desribed
; from the logical point of view via the logic programming pradigm, to
; which we degress momentarity.  In the Lisp paradigm, one thinks
; of an interaction as always being something like

; >  (fact 3) = ?

; wherein a variable free term is evaluated to obtain a suitable
; value, say 6.  In logic programming, as in Baroque or Prolog, one
; can ask a question like:

; ? (fact x) = 6

; i.e. does there exist an x whose factorial is 6?  The system then
; attempts to answer the question and may find one or several values for
; x that does the job, e.g. 3.  In fact, one can even imagine asking

; ? (fact x) = y

; to obtain a variety of values of x and y that satisfy the relation.
; Or might might merely be informed that that, yes, there do exist
; values of x and y satisfying the relation, without being given x and
; y explicitly.

; The point of this digression is merely to mention the well-known
; (but non-Lispish) idea that the input to a computation need not
; always be given entirely in advance of the commencement of a
; computation.  In truth, even in regular Common Lisp, the input is not
; really always given entirely in advance because the charcters that
; may appear in *standard-input* or the file system need not be known
; before evaluation commences.  ACL2 employs this ``incompletely
; specified at evaluation commencement'' idea.

; From the logical point of view, an ACL2 ``state'' is any object in
; the logic satifying the state-p predicate, q.v. in axioms.lisp.
; There is a long comment in axioms.lisp under the heading STATE which
; describes the many fields that a state has.

; At the beginning of any interaction with the top-level ACL2 ld-fn,
; there is a ``partial current state'', which may be partially
; perceived, without side-effect, in Common Lisp, but outside of ACL2,
; by invoking (what-is-the-global-state).  This partial current-state
; includes (a) the names, types, and times of the open input and
; output channels (but not the characters read or written to those
; channels), (b) the symbols in the global table, (c) the t-stack, (d)
; the 32-bit stack, and (e) the file clock.  We say that an object o
; satisfying state-p is ``consistent with the current paritial state''
; provided that every fact revealed by (what-is-the-global-state) and
; by examination of the bound globals is true about o.

; In Lisp (as opposed to Prolog) the input form has no explicit free
; variable.  In ACL2, however, one free variable is permitted, and
; this variable, always named STATE, refers, loosely speaking to the
; ``value of the state at the time of input''.  In ACL2, the variable
; STATE includes the input via files and channels.


;                   Common LISP IO

; If we have a Common Lisp system that is connected to an IO system,
; then at each tick of time, the system may (a) print a character,
; byte, or object to any of the open streams, (b) read a character,
; byte, or object from any of the open streams, (c) open a file for
; reading or writing and (c) close an open stream.

; Suppose that old and new are two objects satisfying state-p and that
; we have an implementation of ACL2 in a Common Lisp which is
; connected to an IO system.  We say that old and new are ``IO
; consistent with the Common Lisp IO system's behavior in the time
; period between old and new'' provided that the relationships between
; the various io fields of old and new are just what happened.  For
; example, suppose that old and new are different only in that in new
; on one input character channel one character has been consumed.
; Then that is consistent with a Common Lisp IO system in which the
; stream corresponding to the channel was read to get just one
; character.  As another example, suppose that old and new are
; different only because a file is now on read-files that was not
; there before and file-clock has been ticked twice and the two most
; recent values of the file clock are the open and close time of the
; read file.  Then that is consistent with a Common Lisp IO system in
; which a stream for a file of the read file's name was opened and
; consumed and the characters read were exactly the characters
; associated with the file name in readable-files at the file-clock
; upon open.  This concept needs to be completely and fully spelled
; out, but we believe it is all boring and obvious:  the file clock is
; to keep track of the opening and closing times.  The read-files and
; written-files entries record closing times and contents.  The
; readable-files and input channels entries record characters actually
; consumed.

; In the extremely important degenerate case, old and new are
; consistent with the Common Lisp IO system's behavior over a time
; interval if all the fields of old and new are identical, excepting
; only the global-table, stacks, and big-clock entries, and no IO
; occurred in the time interval.


;                        The ACL2 ld theorem

; Let us suppose, without loss of generality, that run is a function
; of one argument, state, that has been defined by the user, and
; accepted by ACL2.  Let us further suppose that run returns a single
; state value.  (There is no loss of generality here because any
; particular arguments or output value that the user wishes to provide
; or see can be placed in state globals.  For example, one could add
; two to three by defining run as (defun run (state) (f-set-global
; 'foo (+ 2 3)))).  Let us suppose that an ACL2 interaction of the
; form

; ACL2 !> (run state)

; completes.  What is the theorem that describes the relationship
; between the old current partial state and the new current partial
; state?  The theorem is that (a) there exists an object, old, which
; satisfies the predicate statep and an object, new, which also
; satisfies the predicate statep such that old is consistent with the
; partial current state at the time of the input and new is consistent
; with the partial current state at the time of the output (b) new and
; old are IO consistent with the Common Lisp IO system's behavior in
; the time period between the beginning and ending of the evaluation
; (c) new = (trans-eval '(run state) nil old t), and (d) (run old) =
; (trans-eval '(run state) nil old t) except in the big-clock field.

; In the important degenerate case in which no io occurs, this means
; essentially that there exists (in the constructive sense) a
; big-clock entry in old which is ``large enough'' to perform the
; trans-eval of the input form without ``running out of time''.  ACL2
; does not reveal to the user how much ``time'' was required, but
; merely guarantees that there exists a sufficiently large amount of
; time.  In fact, because we ``jump into compiled code'' in
; raw-ev-fncall, we have no way of efficiently keeping track of how
; much time has been used.

; Note that there is no commitment to a uniform value for big-clock
; across all ACL2 interactions.  In particular, there obviously exists
; an infinite sequence of forms, say (fact 1), (fact 2), (fact 3),
; ....  which would require an infinitely increasing series of
; big-clocks.  An ACL2 evaluation effort may fail for a variety of
; reasons, including resource errors and certain design decisions,
; e.g. the detection that a function should not be clobbered because
; there is already a function by that name with a symbol-function
; property.  If evaluation fails, some characters may nevertheless
; have been printed or read and state may have been changed.

(defconst *basic-sweep-error-str*
  "The state back to which we have been asked to roll would contain an ~
   object that is inconsistent with the requested resetting of the ~
   ACL2 known-package-alist.  Logical consistency would be imperiled ~
   if the rollback were undertaken.  Please get rid of pointers to ~
   such objects before attempting such a rollback.~|~%")

(defun sweep-symbol-binding-for-bad-symbol (sym obj deceased-packages state)
  (cond ((symbolp obj)
         (cond ((member-equal (symbol-package-name obj) deceased-packages)
                (er soft "undo consistency check"
                    "~@0In particular, the value of the global ~
                     variable ~x1 contains the symbol ~x2 in package ~
                     ~x3, which we have been asked to remove.   ~
                     Please reset ~x1, as with (assign ~x1 nil)."
                    *basic-sweep-error-str*
                    sym
                    obj
                    (symbol-package-name obj)))
               (t (value nil))))
        ((atom obj) (value nil))
        ((equal obj (w state))
         (value nil))
        (t (er-progn (sweep-symbol-binding-for-bad-symbol
                      sym (car obj)
                      deceased-packages state)
                     (sweep-symbol-binding-for-bad-symbol
                      sym (cdr obj) deceased-packages state)))))

(defun sweep-global-lst (l deceased-packages state)
  (cond ((null l) (value nil))
        (t (er-progn
            (sweep-symbol-binding-for-bad-symbol
             (car l)
             (get-global (car l) state)
             deceased-packages state)
            (sweep-global-lst (cdr l) deceased-packages state)))))

(defun sweep-stack-entry-for-bad-symbol (name i obj deceased-packages state)
  (cond ((symbolp obj)
         (cond ((member-equal (symbol-package-name obj) deceased-packages)
                (er soft "undo consistency check"
                    "~@0In particular, the entry in the ~@1 at ~
                     location ~x2 contains the symbol ~x3 in package ~
                     ~x4, which we have been asked to undo.  Please ~
                     change the ~@1 entry at location ~x2 or ~
                     shrink the ~@1."
                    *basic-sweep-error-str*
                    name
                    i
                    obj
                    (symbol-package-name obj)))
               (t (value nil))))
        ((atom obj) (value nil))
        ((equal obj (w state))
         (value nil))
        (t (er-progn (sweep-stack-entry-for-bad-symbol
                      name i (car obj) deceased-packages state)
                     (sweep-stack-entry-for-bad-symbol
                      name i (cdr obj) deceased-packages state)))))

(defun sweep-t-stack (i deceased-packages state)
  (cond ((> i (t-stack-length state))
         (value nil))
        (t (er-progn
            (sweep-stack-entry-for-bad-symbol
             "t-stack" i (aref-t-stack i state) deceased-packages state)
            (sweep-t-stack (+ 1 i) deceased-packages state)))))

(defun sweep-acl2-oracle (i deceased-packages state)

; A valid measure is (- (len (acl2-oracle state)) if we want to admit this
; function in logic mode, since read-acl2-oracle replaces the acl2-oracle of
; the state with its cdr.

  (mv-let
   (nullp car-oracle state)
   (read-acl2-oracle state)
   (cond (nullp (value nil))
         (t (er-progn
             (sweep-stack-entry-for-bad-symbol
              "acl2-oracle" i car-oracle deceased-packages state)
             (sweep-acl2-oracle (+ 1 i) deceased-packages state))))))

(defun sweep-global-state-for-lisp-objects (deceased-packages state)

; This function sweeps every component of the state represented by
; *the-live-state* to verify that no symbol is contained in a package that we
; are about to delete.  This is sensible before we undo a defpkg, for example,
; which may ``orphan'' some objects held in, say, global variables in the
; state.  We look in the global variables, the t-stack, and acl2-oracle.  If a
; global variable, t-stack entry, or acl2-oracle entry contains such an object,
; we cause an error.  This function is structurally similar to
; what-is-the-global-state in axioms.lisp.

; The components of the state and their disposition are:

; open-input-channels  -  there are no objects in the dynamic channels.
;                         Objects obtained from those channels will be
;                         read into an otherwise ok state.

; open-output-channels -  there are no objects in the dynamic channels

; global-table - the global table is the most likely place we will find
;                bad objects.  However, we know that the value of
;                'current-acl2-world is not bad, because after an undo
;                it is set to a previously approved value.

  (er-progn
   (sweep-global-lst (global-table-cars state) deceased-packages state)


; t-stack - this stack may contain bad objects.

   (sweep-t-stack 0 deceased-packages state)
   (sweep-acl2-oracle 0 deceased-packages state))

; The remaining fields contain no ``static'' objects.  The fields are:
; 32-bit-integer-stack
; big-clock
; idates
; file-clock
; readable-files
; written-files
; read-files
; writeable-files
; list-all-package-names-lst

  )

(deflabel compilation
  :doc
  ":Doc-Section ACL2::Programming

  compiling ACL2 functions~/

  ACL2 has several mechanisms to speed up the evaluation of function calls by
  ~em[compiling] functions: ~pl[comp], ~pl[set-compile-fns], and
  ~pl[certify-book].  The intention is that compilation never changes the
  value returned by a function call, though it could cause the call to succeed
  rather than fail, for example by avoiding a stack overflow.

  The ~ilc[state] global variable ~c['compiler-enabled] is set automatically
  when the system is built, and may depend on the underlying Lisp
  implementation.  (In order to disable the compiler at build time, which will
  defeat the speed-up but usually be pretty harmless when the host Lisp is CCL
  or SBCL, see the discussion of ~c[ACL2_COMPILER_DISABLED] in distributed file
  ~c[GNUmakefile].)  The value of ~c['compiler-enabled], as returned by
  ~c[(@ compiler-enabled)], can be ~c[t], ~c[:books], or ~c[nil].  If the value
  is ~c[nil], then ~ilc[include-book] and ~ilc[certify-book] coerce their
  arguments ~c[:load-compile-file] and ~c[compile-flg] arguments (respectively)
  to ~c[nil].  Otherwise, the value is ~c[:books] or ~c[t] and there is no such
  coercion; but if the value is not ~c[t], then ~ilc[comp] and
  ~ilc[set-compile-fns] are no-ops, which is probably desirable for Lisps such
  as CCL and SBCL that compile on-the-fly even when the compiler is not
  explicitly invoked.

  However, you may have reason to want to change the above (default) behavior.
  To enable compilation by default for ~ilc[certify-book] and
  ~ilc[include-book] but not for ~ilc[comp] or ~ilc[set-compile-fns]:
  ~bv[]
  (set-compiler-enabled :books state)
  ~ev[]
  To enable compilation not only as above but also for ~ilc[comp] and
  ~ilc[set-compile-fns]:
  ~bv[]
  (set-compiler-enabled t state)
  ~ev[]
  To suppress compilation and loading of compiled files by ~ilc[include-book]
  (for example, if you get a raw Lisp error such as ``Wrong FASL version''):
  ~bv[]
  (set-compiler-enabled nil state)
  ~ev[]~/

  ~l[book-compiled-file] for more discussion about compilation and
  ~il[books].~/")

(defdoc book-compiled-file
  ":Doc-Section books

  creating and loading of compiled and expansion files for ~il[books]~/

  An effect of ~il[compilation] is to speed up the execution of the functions
  defined in a book.  Compilation can also remove tail recursion, thus avoiding
  stack overflows.  The presence of compiled code for the functions in the book
  should not otherwise affect the performance of ACL2.  ~l[guard] for a
  discussion; also ~l[compilation].

  By default, the ~ilc[certify-book] command compiles the book that it
  certifies.  ~pl[certify-book] for how to control this behavior.

  By default, the ~ilc[include-book] command loads the compiled file for the
  book.  The details of how this loading works are subtle, and do not need to
  be understood by most users.  The ACL2 source code contains an ``Essay on
  Hash Table Support for Compilation'' that explains such details for those
  interested.  All that users should generally need to know about this is that
  the compiled file is always the result of compiling a so-called ``expansion
  file'', which contains certain additional code besides the book itself.  The
  relevance to users of the expansion file is that it can be loaded if the
  compiled file is missing (except when ~c[:load-compiled-file t] is specified
  by the ~ilc[include-book] form), and its existence is required in order for
  ~ilc[include-book] to create a book's compiled file, as described below.

  Most users can skip the remainder of this documentation topic, which
  addresses the uncommon activity of using ~ilc[include-book] to compile books.

  ~c[Include-book] can be made to compile a book by supplying its keyword
  argument ~c[:load-compiled-file] the value ~c[:comp].  However, a compiled
  file can only be produced if there is already an ~em[expansion file] that is
  at least as recent as the book's ~il[certificate].  Such a file, whose name
  happens to be the result of concatenating the string ~c[\"@expansion.lsp\"]
  to the book name (without the ~c[\".lisp\"] suffix), is created by
  ~ilc[certify-book] when state global variable ~c['save-expansion-file] has a
  non-~c[nil] value.  That will be the case if ACL2 started up when environment
  variable ~c[ACL2_SAVE_EXPANSION] was ~c[t] (or any value that is not the
  empty string and whose ~ilc[string-upcase] is not ~c[\"NIL\"]), until the
  time (if any) that ~c['save-expansion-file] is assigned a different value by
  the user.  In most respects, the ~c[:comp] setting is treated exactly the
  same as ~c[:warn]; but after all events in the book are processed, the
  expansion file is compiled if a compiled file was not loaded, after which the
  resulting compiled file is loaded.

  One can thus, for example, compile books for several different host Lisps
  ~-[] useful when installing ACL2 executables at the same site that are built
  on different host Lisps.  A convenient way to do this in an environment that
  provides Gnu `make' is to certify the community books using the shell
  command ``~c[make regression]'' in the ~c[acl2-sources/] directory, after
  setting environment variable ~c[ACL2_SAVE_EXPANSION] to ~c[t], and then
  moving to the ~c[books] directory and executing the appropriate `make'
  commands to compile the books (targets ~c[fasl], ~c[o], and so on, according
  to the compiled file extension for the host Lisp).

  We conclude by saying more about the ~c[:load-compiled-file] argument of
  ~ilc[include-book].  We assume that ~il[state] global ~c['compiler-enabled]
  has a non-~c[nil] value; otherwise ~c[:load-compiled-file] is always treated
  as ~c[nil].

  We do not consider raw mode below (~pl[set-raw-mode]), which presents a
  special case: ACL2 will attempt to load the book itself whenever it would
  otherwise load the expansion or compiled file, but cannot (either because the
  ~c[:load-compiled-file] argument is ~c[nil], or for each of the expansion and
  compiled files, either it does not exist or it is out of date with respect to
  the ~c[.cert] file).

  The ~c[:load-compiled-file] argument is not recursive: calls of
  ~c[include-book] that are inside the book supplied to ~c[include-book] use
  their own ~c[:load-compiled-file] arguments.  However, those subsidiary
  ~c[include-book] calls can nevertheless be sensitive to the
  ~c[:load-compiled-file] arguments of enclosing ~c[include-book] calls, as
  follows.  If ~c[:load-compiled-file] has value ~c[t], then every subsidiary
  ~c[include-book] is required to load a compiled file.  Moreover, if a book's
  compiled file or expansion file is loaded in raw Lisp, then an attempt will
  be made to load the compiled file or expansion file for any
  ~ilc[include-book] form encountered during that load.  If that attempt fails,
  then that load immediately aborts, as does its parent load, and so on up the
  chain.  If, when going up the chain, an ~ilc[include-book] is aborted for
  which keyword argument ~c[:load-compiled-file] has value ~c[t], then an error
  occurs.

  When loading a book's compiled file or expansion file, ~c[FILE], it is
  possible to encounter an ~ilc[include-book] form for a book that has no
  suitable compiled file or expansion file.  In that case, the load of ~c[FILE]
  is aborted at that point.  Similarly, the load of ~c[FILE] is aborted in the
  case that this ~c[include-book] form has a suitable compiled file or
  expansion file whose load is itself aborted.  Thus, whenever any
  ~c[include-book] aborts, so do all of its parent ~c[include-book]s, up the
  chain.  Such an abort causes an error when the ~c[include-book] form
  specifies a ~c[:load-compiled-file] value of ~c[t].~/~/")

(deflabel escape-to-common-lisp
  :doc
  ":Doc-Section Miscellaneous

  escaping to Common Lisp~/
  ~bv[]
  Example:
  ACL2 !>:Q
  ~ev[]~/

  There is no Common Lisp escape feature in the ~ilc[lp].  This is part of
  the price of purity.  To execute a form in Common Lisp as opposed to
  ACL2, exit ~ilc[lp] with ~c[:]~ilc[q], submit the desired forms to the Common Lisp
  read-eval-print loop, and reenter ACL2 with ~c[(lp)].~/")

(deflabel copyright
  :doc
  ":Doc-Section Miscellaneous

  ACL2 copyright, license, sponsorship~/~/

  ACL2 Version 6.3 -- A Computational Logic for Applicative Common Lisp
  Copyright (C) 2013, Regents of the University of Texas

  This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
  (C) 1997 Computational Logic, Inc.  See the documentation topic NOTE-2-0.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the LICENSE file distributed with ACL2.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  LICENSE for more details.

  Written by:  Matt Kaufmann               and J Strother Moore
  email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
  Department of Computer Science
  University of Texas at Austin
  Austin, TX 78701 U.S.A.

  Please also ~pl[acknowledgments].~/")

(deflabel acknowledgments
  :doc
  ":Doc-Section Miscellaneous

  some contributors to the well-being of ACL2~/~/

  The development of ACL2 was initially made possible by funding from the
  U. S. Department of Defense, including ARPA and ONR.  We thank all the
  organizations that have contributed support, including the following (in
  alphabetical order).~bq[]

  o AMD, for providing significant time over several years for Matt Kaufmann
  to carry out ACL2 research, support, and development~nl[]
  o Computational Logic, Inc. and its president, Don Good, where the first
  eight years of ACL2 development occurred~nl[]
  o Centaur Technology~nl[]
  o DARPA~nl[]
  o Digital Equipment Corporation~nl[]
  o EDS, which provided some time for Matt Kaufmann's ACL2 work 1998-1999~nl[]
  o ForrestHunt and, more generally, Warren A. Hunt, Jr. (see below)~nl[]
  o IBM~nl[]
  o NSF~nl[]
  o ONR~nl[]
  o Rockwell Collins~nl[]
  o SRC~nl[]
  o Sun Microsystems~nl[]
  o University of Texas at Austin (in particular support to J Moore through
  the Admiral B. R.  Inman Chair of Computing Theory)

  ~eq[]We are especially grateful to Warren A. Hunt, Jr. for his unrivaled
  efforts in securing support for the entire ACL2 research group at both
  Computational Logic, Inc., and the University of Texas at Austin.  Without
  his efforts, we would have spent less time working on the system and fewer
  students would have been funded to apply it.

  ACL2 was started in August, 1989 by Boyer and Moore working together.
  They co-authored the first versions of axioms.lisp and basis.lisp, with Boyer
  taking the lead in the formalization of ``~il[state]'' and the most primitive
  ~il[io] functions.  Boyer also had a significant hand in the development of
  the early versions of the files interface-raw.lisp and translate.lisp.  For
  several years, Moore alone was responsible for developing the ACL2 system
  code, though he consulted often with both Boyer and Kaufmann.  In August,
  1993, Kaufmann became jointly responsible with Moore for developing the
  system.  Boyer has continued to provide valuable consulting on an informal
  basis.

  Bishop Brock was the heaviest early user of ACL2, and provided many
  suggestions for improvements.  In particular, the ~c[:cases] and
  ~c[:restrict] ~il[hints] were his idea; he developed an early version of
  congruence-based reasoning for Nqthm; and he helped in the development of
  some early ~il[books] about arithmetic.  In a demonstration of his courage
  and faith in us, he pushed for Computational Logic, Inc., to agree to the
  Motorola CAP contract -- which required formalizing a commercial DSP in the
  untested ACL2 -- and moved to Scottsdale, AZ, to do the work with the
  Motorola design team.  His demonstration of ACL2's utility was an
  inspiration, even to those of us designing ACL2.

  John Cowles also helped in the development of some early ~il[books] about
  arithmetic, and also provided valuable feedback and bug reports.

  Other early users of ACL2 at Computational Logic, Inc. helped influence its
  development.  In particular, Warren Hunt helped with the port to Macintosh
  Common Lisp, and Art Flatau and Mike Smith provided useful general feedback.

  Mike Smith helped develop the Emacs portion of the implementation of proof
  trees.

  Bill Schelter made some enhancements to akcl (now gcl) that helped to enhance
  ACL2 performance in that Common Lisp implementation, and more generally,
  responded helpfully to our bug reports.  Camm Maguire has since provided
  wonderful gcl support, and has created a Debian package for ACL2 built on
  GCL.  We are also grateful to developers of other Common Lisp
  implementations.

  Kent Pitman helped in our interaction with the ANSI Common Lisp
  standardization committee, X3J13.

  John Cowles helped with the port to Windows (98) by answering questions and
  running tests.

  Ruben Gamboa created a modification of ACL2 to allow reasoning about the real
  numbers using non-standard analysis.  His work has been incorporated into the
  ACL2 distribution; ~pl[real].

  Rob Sumners has made numerous useful suggestions.  In particular, he has
  designed and implemented improvements for ~il[stobj]s and been key in our
  development of locally-bound stobjs; ~pl[note-2-6].

  Robert Krug has designed and implemented many changes in the vicinity of the
  linear arithmetic package and its connection to type-set and rewrite.  He was
  also instrumental in the development of ~il[extended-metafunctions].

  Pete Manolios has made numerous useful suggestions.  In particular, Pete
  helped us to organize the first workshop and was a wonderful equal partner
  with the two of us (Kaufmann and Moore) in producing the books that arose
  from that workshop.  Pete and his student, Daron Vroon, provided the current
  implementation of ~il[ordinals].

  Jared Davis and Sol Swords have our gratitude for starting the acl2-books
  repository, ~url[http://acl2-books.googlecode.com/].

  We thank David L. Rager for contributing an initial version of the support
  for ~il[parallelism] in an experimental extension of ACL2.

  Bob Boyer and Warren A. Hunt, Jr. developed a canonical representation for
  ACL2 data objects and a function memoization mechanism to facilitate reuse of
  previously computed results.  We thank them for their extensive efforts for
  the corresponding experimental (as of 2008 and 2009) extension of ACL2;
  ~pl[hons-and-memoization].

  We also thank the contributors to the ACL2 workshops for some suggested
  improvements and for the extensive collection of publicly distributed
  benchmark problems.  And we thank participants at the ACL2 seminar at the
  University of Texas for useful feedback.  More generally, we thank the ACL2
  community for feedback, contributed ~il[books] (~pl[community-books]), and
  their interest in the ACL2 project.

  ~em[Regarding the documentation:]

  ~bq[]
  Bill Young wrote significant portions of the original ~c[acl2-tutorial]
  section of the ACL2 documentation, including what is now called
  ~il[alternative-introduction].  This was an especially important task in the
  early years when there was no guide for how to use ACL2 and we are very
  grateful.  He, Bishop Brock, Rich Cohen, and Noah Friedman read over
  considerable amounts of the documentation, and made many useful comments.
  Others, particularly Bill Bevier and John Cowles, have also made useful
  comments on the ~il[documentation].

  Art Flatau helped develop the ACL2 ~il[markup] language and translators from
  that language to Texinfo and HTML.  Michael ``Bogo'' Bogomolny created a
  search engine, beginning with Version 2.6, and for that purpose modified the
  HTML translator to create one file per topic (a good idea in any case).

  Laura Lawless provided many hours of help in marking up appropriate parts of
  the ~il[documentation] in typewriter font.

  Noah Friedman developed an Emacs tool that helped us insert ``invisible
  links'' into the ~il[documentation], which improve the usability of that
  documentation under HTML readers such as Mosaic.

  Richard Stallman contributed a texinfo patch, to be found in the file
  ~c[doc/texinfo.tex].~eq[]

  ")

(deflabel breaks
  :doc
  ":Doc-Section ACL2::ACL2-built-ins

  Common Lisp breaks~/
  ~bv[]
  Example:
  Broken at PROVE.  Type :H for Help.
  >>:Q

  ACL2 !>
  ~ev[]~/

  You may interrupt the system by typing various control character sequences.
  The precise sequences are determined by the host Lisp and operating system
  environment.  For example, in GCL and Allegro Common Lisp, a console
  interrupt is caused by typing ``~c[ctrl-c]''.  If, however, the GCL or
  Allegro is running in an Emacs shell buffer, one must type ``ctrl-c ctrl-c''.

  If a break occurs, for example because of a bug in ACL2 or a user interrupt,
  the break will run a Common Lisp read-eval-print loop, not an ACL2
  read-eval-print loop.  This may not be obvious if the ~il[prompt]s in the two
  loops are similar.  Because you are typing to a Common Lisp evaluator, you
  must be careful.  It is possible to damage your ACL2 state in irreparable
  ways by executing non-ACL2 Common Lisp.  It is even possible to disrupt and
  render inaccurate the interrupted evaluation of a simple ACL2 expression.

  For ACL2 built on most host Common Lisps, you will see the string
  ~c[[RAW LISP~]] in the ~il[prompt] at a break, to emphasize that one is
  inside a break and hence should quit from the break.  For some host Common
  Lisps, the top-level prompt also contains the string ~c[[RAW LISP~]].
  ~l[prompt] for how to control printing of that string.

  The most reliable way to return to the ACL2 top level is by executing the
  following command: ~c[(]~ilc[abort!]~c[)].  Appropriate cleanup will then be
  done, which should leave you in an appropriate state.

  However, you may be able to quit from the break in the normal Lisp manner (as
  with ~c[:q] in GCL or CCL, ~c[:reset] in Allegro CL, and ~c[q] in CMU CL).
  If this attempt to quit is successful, it will return you to the innermost
  ACL2 read-eval-print loop, with appropriate cleanup performed first.  Note
  that if you are within a ~ilc[brr] environment when the break occurs,
  quitting from the break will only return you to that environment, not to the
  top of ACL2's read-eval-print loop.~/")

(deflabel ordinals
  :doc
  ":Doc-Section Miscellaneous

  ordinals in ACL2~/

  Ordinals are used in ACL2 for proving termination in the admission of
  recursive function definitions.  For a proof that the ACL2 ordinals are
  well-founded, ~pl[proof-of-well-foundedness].

  The representation of ordinals changed in ACL2 Version_2.8, and is due to
  Pete Manolios and Daron Vroon.  They have also defined algorithms for ordinal
  arithmetic, created a library of theorems to reason about ordinal arithmetic,
  and written the rest of this documentation in order to explain this change.
  We thank them for their efforts.  Although they have provided the
  implementation and even modified the community books as needed, we have
  looked over their work and are maintaining it (and this documentation); if
  there are any bugs, they should be considered ours (Matt Kaufmann and J
  Moore).

  A book is included for compatibility with the representation before
  Version_2.8.  For books that contain events relying on the previous ordinal
  implementation, insert the following lines before the first such event:
  ~bv[]
  (include-book \"ordinals/e0-ordinal\" :dir :system)
  (set-well-founded-relation e0-ord-<)
  ~ev[]

  The new ordinal representation is based on a slightly different version of
  Cantor Normal Form than that used by the old ordinals.  An advantage of the
  new representation is that it is exponentially more succinct than the old
  representation.

  While pre-Version_2.8 ACL2 versions provided built-in functions for checking
  if an object is an ordinal and for comparing two ordinals, they did not
  provide support for reasoning about and constructing ordinals.  The community
  books directory ~c[books/ordinals] provides such support.  First, it
  provides efficient algorithms for ordinal arithmetic (including addition,
  subtraction, multiplication, and exponentiation).  The algorithms and their
  complexity are described in the following paper.
  ~bf[]
  Manolios, Panagiotis & Vroon, Daron.
  Algorithms for ordinal arithmetic.
  Baader, Franz (ed),
  19th International Conference on Automated Deduction--CADE-19.
  Pages 243-257 of LNAI, vol. 2741.  Springer-Verlag.
  ~ef[]
  Second, the algorithms are mechanically verified and libraries of theorems
  which can be used to automate reasoning involving the ordinals are provided.
  For details, see the following paper.
  ~bf[]
  Manolios, Panagiotis & Vroon, Daron.
  Ordinal arithmetic in ACL2.
  Kaufmann, Matt, & Moore, J Strother (eds).
  Fourth International Workshop on the ACL2 Theorem
  Prover and Its Applications (ACL2-2003),
  July, 2003.
  See ~url[http://www.cs.utexas.edu/users/moore/acl2/workshop-2003/].
  ~ef[]
  We now describe aspects of the above mentioned books in more detail.

  The new ordering function is ~ilc[o<] and the new ordinal recognizer is
  ~ilc[o-p].  See also ~ilc[natp], ~ilc[posp], ~ilc[o<=], ~ilc[o>], ~ilc[o>=],
  ~ilc[o-first-expt], ~ilc[o-first-coeff], ~ilc[o-rst], ~ilc[make-ord],
  ~ilc[o-finp], and ~ilc[o-infp]. ~/

  The old ordinals were based on the following formulation of Cantor Normal
  Form:

  For any ordinal, ~c[a < epsilon-0], there exist natural numbers ~c[p] and
  ~c[n], and ordinals ~c[a1 >= a2 >= ... >= an > 0] such that ~c[a > a1]
  and ~c[a = w^(a1) + w^(a2) + ... + w^(an) + p].

  Thus, a predicate recognizing ACL2's old ordinals is given by the following
  definition.
  ~bv[]
  (defun e0-ordinalp (x)
    (if (consp x)
        (and (e0-ordinalp (car x))
             (not (equal (car x) 0))
             (e0-ordinalp (cdr x))
             (or (atom (cdr x))
                 (not (e0-ord-< (car x) (cadr x)))))
      (and (integerp x)
           (>= x 0))))
  ~ev[]
  The new representation is based on a corollary to the above theorem, which we
  get by the left distributive property of ordinal multiplication over ordinal
  addition. Thus, ~c[w^a + w^a = (w^a)2], ~c[w^a + w^a + w^a = (w^a)3] and so
  forth. The corollary is as follows:

  For any ordinal, ~c[a < epsilon-0], there exist natural numbers ~c[p]
  and ~c[n], positive integers ~c[x1, x2, ..., xn] and ordinals
  ~c[a1 > a2 > ... > an > 0] such that ~c[a > a1] and
  ~c[a = w^(a1)x1 + w^(a2)x2 + ... + w^(an)xn + p].

  Instead of representing an ordinal as a list of non-increasing ordinals, we
  represent it as a list of exponent-coefficient pairs, such that the exponents
  are strictly decreasing (~pl[o-p]).  Note that this representation is
  exponentially more efficient than the old representation.

  The ordinal arithmetic functions: ~c[o+], ~c[o-], ~c[o*], and ~c[o^] are
  defined in the ordinals library (in the community books directory
  ~c[books/ordinals]). To use them, include the book
  ~c[ordinals-without-arithmetic] or ~c[ordinals], depending on whether you
  want the arithmetic books included or not (~c[ordinals] includes community
  book ~c[books/arithmetic/top-with-meta]). To use the old ordinals, include
  the book ~c[e0-ordinal] and run the command
  ~c[(set-well-founded-relation e0-ord-<)]

  The community book ~c[books/arithmetic/natp-posp] is a book for reasoning
  about ~c[posp] and ~c[natp].  We recommend using this book if you have to
  reason about ~c[posp] and ~c[natp].  It is included in community book
  ~c[books/arithmetic/top], which is included in community book
  ~c[books/arithmetic/top-with-meta], which is included in community book
  ~c[books/ordinals/ordinals].

  If you have a good reason to use the old definitions of the ordinals (e.g.,
  because of legacy code and theorems), then we provide a convenient way to do
  this.  The book ~c[ordinal-isomorphism] proves that the new ordinals are
  order-isomorphic to the old ordinals and thus theorems proved in one context
  can be directly transferred to the other.  For an example of how to do this,
  look at the book ~c[defmul] in the community books directory
  ~c[books/workshops/2000/ruiz/multiset].

  The ordinals books have been used to prove non-trivial theorems.  For a good
  example, see the books in the community books directory
  ~c[books/workshops/2003/sustik/support], where Matyas Sustik proves Dickson's
  lemma.

  Finally, many termination proofs can be carried out with weaker orderings
  than the ordinals up to ~c[epsilon-0].  For example, many inductive theorem
  provers only know that the lexicographic ordering on natural numbers is
  well-founded.  The book ~c[lexicographic-ordering] contains a definition of
  such an ordering ~c[l<] whose arguments are either a list of natural numbers,
  or a natural number.  In the book we prove that ~c[l<] is well-founded (that
  is, we prove a ~c[:well-founded-relation] ~ilc[defthm] and provide a macro
  ~c[llist] to simplify the generation of measure functions.  We also show how
  to use ~c[l<] to prove that the famous Ackermann function terminates.
  Finally, since ~c[l<] does something reasonable with natural numbers, it gets
  along with ~ilc[acl2-count], the default measure chosen by ACL2.")

(defmacro wet (form &rest kwd-args)

  ":Doc-Section Trace

  evaluate a form and print subsequent error trace~/

  The acronym ``wet'' stands for ``with-error-trace''.  ~c[Wet] provides a
  convenient way to obtain a backtrace when evaluation causes a guard violation
  or other error.

  The basic idea is that ~c[(wet form)] evaluates ~c[form] and, if there is an
  error, shows a backtrace of calls that led to that error.  Note however that
  by default only calls of user-defined (not built-in) functions ``supporting''
  ~c[form] in the following sense will show up in the backtrace: those that
  occur in the macroexpansion of ~c[form] or (recursively) support any of those
  functions.  So for example, since ~c[(make-event form)] macroexpands to
  ~c[(make-event-fn (quote form) ...)], calls of functions occurring in
  ~c[form] will likely not show up in the backtrace by default.  The option
  ~c[:fns all] overrides this default, with potential loss of speed; more on
  this below.

  The following example explains the use of ~c[wet].  First, submit the
  following three definitions:
  ~bv[]
  (defun foo (x) (declare (xargs :guard (consp x))) (car x))
  (defun bar (x) (foo (cdr x)))
  (defun g (x) (bar (cdr x)))
  ~ev[]
  Now imagine you have obtained the following guard violation:
  ~bv[]
  ACL2 !>(g '(3 4))


  ACL2 Error in TOP-LEVEL:  The guard for the function call (FOO X),
  which is (CONSP X), is violated by the arguments in the call (FOO NIL).
  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.  See
  :DOC set-guard-checking for information about suppressing this check
  with (set-guard-checking :none), as recommended for new users.

  ACL2 !>
  ~ev[]
  With ~c[wet], you can get a backtrace of user-defined functions.  The
  package prefixes shown below, ~c[ACL2_*1*_], indicate that the
  executable (logical) counterparts of the corresponding raw Lisp functions are
  being called; ~pl[guard].  Don't forget to start with
  ~c[(include-book \"misc/wet\" :dir :system)].
  ~bv[]
  ACL2 !>(wet (g '(3 4)))
  ; Fast loading /projects/acl2/devel/books/misc/wet.fasl

  TTAG NOTE: Adding ttag :TRACE! from the top level loop.


  ACL2 Error in WET:  The guard for the function call (FOO X), which
  is (CONSP X), is violated by the arguments in the call (FOO NIL).
  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.  See
  :DOC set-guard-checking for information about suppressing this check
  with (set-guard-checking :none), as recommended for new users.


  Backtrace stack:
  ----------------
  1. (ACL2_*1*_ACL2::FOO NIL)
  2. (ACL2_*1*_ACL2::BAR (4))
  3. (ACL2_*1*_ACL2::G (3 4))

  ACL2 !>
  ~ev[]
  By default, large structures are hidden during the printing of the backtrace
  stack.  But you can supply a value for keyword argument ~c[:evisc-tuple] to
  modify the printing: ~c[nil] to avoid hiding, else a suitable evisc-tuple, as
  shown below (~pl[evisc-tuple]).
  ~bv[]
  ACL2 !>(wet (g '(3 4)) :evisc-tuple (evisc-tuple 1 1 nil nil))
  ; Fast loading /projects/acl2/devel/books/misc/wet.fasl

  TTAG NOTE: Adding ttag :TRACE! from the top level loop.


  ACL2 Error in WET:  The guard for the function call (FOO X), which
  is (CONSP X), is violated by the arguments in the call (FOO NIL).
  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.  See
  :DOC set-guard-checking for information about suppressing this check
  with (set-guard-checking :none), as recommended for new users.


  Backtrace stack:
  ----------------
  1. (ACL2_*1*_ACL2::FOO ...)
  2. (ACL2_*1*_ACL2::BAR ...)
  3. (ACL2_*1*_ACL2::G ...)

  ACL2 !>
  ~ev[]
  For a backtrace as a data object, evaluate the form ~c[(@ wet-stack)].  But
  note that this object may not be a legal ACL2 value, for example because of
  the ``~c[*1*]'' symbols shown above.~/

  ~bv[]
  General Form:
  (wet form           ; an arbitrary form
       :book bk-form  ; optional, not evaluated
       ;;; the rest are optional and evaluated:
       :evisc-tuple e ; an evisc-tuple
       :fns fns       ; :all, or a list of functions to show in a backtrace
       :compile c     ; :same, t, or nil; default :same (nil if :fns supplied)
  ~ev[]

  ~c[Form] is evaluated.  If there is an error, a backtrace stack is printed to
  the standard output (~ilc[*standard-co*]), containing (by default) the
  user-defined function calls made before the error.  Such printing is
  controlled by the ~c[:evisc-tuple] if supplied; otherwise, hiding of large
  structures will occur.  (Technical detail: by default the global
  abbrev-evisc-tuple is used, if bound; ~pl[set-evisc-tuple].

  The ~c[:fns] option.  As mentioned above, by default the ~c[wet] backtrace
  shows user-defined functions that syntactically ``support'' the form being
  evaluated.  This default can be overridden by supplying an explicit list,
  ~c[fns], of functions, using option ~c[:fns fns]; these will then be the
  functions whose calls are eligible for inclusion in the backtrace.  The
  special value ~c[:fns :all] will allow all user-defined function calls in the
  backtrace.  This value can be useful when using ~ilc[oracle-apply], for
  example, since the function being applied isn't typically included as
  a syntactic supporter of the form being evaluated.

  The ~c[:compile] option.  ~c[Wet] uses the ~ilc[trace$] utility to modify the
  definitions of affected functions so that they can record information for the
  backtrace.  As described above, these affected functions are those that
  syntactically ``support'' the form unless specified by the ~c[:fns] option.
  As is the case for ~c[trace$] ~-[] ~pl[trace$] ~-[] the new definitions of
  these affected functions may or may not be compiled.  For ~c[trace$] and for
  ~c[wet], the default is to compile the new definition if and only if the
  original definition was compiled, except: For ~c[wet], if option
  ~c[:fns :all] is provided, then the default is not to compile the affected
  definitions.  And for ~c[trace$] and ~c[wet], the ~c[:compile] option
  overrides the default, to specify what will be compiled: value ~c[:same] to
  compile each affected function if and only if its original definition was
  compiled, value ~c[t] to compile all affected functions, and value ~c[nil] to
  skip compilation.

  The ~c[:book] option.  ~c[Wet] actually works by temporarily including a
  community book,
  ~bv[]
  (include-book \"misc/wet\" :dir :system)
  ~ev[]
  and then passing its arguments to macro ~c[wet!], defined in that book.
  The keyword argument ~c[:book] allows you to specify a different book that
  defines a macro ~c[wet!] to which to pass its arguments.  If the value of
  ~c[:book] is a string, then the book named by that string is temporarily
  included using ~ilc[include-book]: ~c[(include-book \"bk\")].  Otherwise
  ~c[:book] should be a list of arguments, to be provided (unevaluated) to
  ~ilc[include-book], for example ~c[(\"my-wet\" :dir :my-utils)].  Thus you
  can experiment by copying community book ~c[books/misc/wet.lisp] to your
  own directory and making modifications to the copy.  If you make changes, we
  invite you to share them with the ACL2 community (~pl[books]).  Note that you
  can also supply ~c[:book nil], in which case the definition of ~c[wet!] in
  your current session will be used without including a book.

  Also ~pl[trace$] for a general tracing utility.  As mentioned above, ~c[wet]
  is implemented using ~c[trace$].  ~c[Wet] actually first applies
  ~ilc[untrace$]; upon completion, ~c[wet] then applies ~ilc[trace$] to
  re-trace any functions that it had untraced, using their original trace
  specs.~/"

  (let* ((book-tail (member-eq :book kwd-args))
         (kwd-args (if book-tail (remove-keyword :book kwd-args) kwd-args))
         (book-form (if book-tail
                        (cond ((null book-tail)
                               nil)
                              ((stringp (cadr book-tail))
                               (list 'include-book (cadr book-tail)))
                              (t (cons 'include-book (cadr book-tail))))
                      '(include-book "misc/wet" :dir :system))))
    `(with-output
      :off (summary event)
      (make-event (mv-let (erp val state)
                          (progn
                           ,@(and book-form (list book-form))
                           (wet! ,form ,@kwd-args))
                          (cond (erp (mv "WET failed!" nil state))
                                (t (value `(value-triple ',val)))))))))

(defmacro disassemble$ (fn &rest args
                           &key (recompile ':default)

; And, in case community book books/misc/disassemble.lisp changes between
; releases:

                           &allow-other-keys)

  ":Doc-Section Other

  disassemble a function~/

  The macro ~c[disassemble$] provides a convenient interface to the underlying
  ~c[disassemble] utility of the host Common Lisp implementation, which prints
  assembly code for a given function symbol at the terminal.  It works by
  including the community book ~c[books/misc/disassemble.lisp], which defines
  the supporting function ~c[disassemble$-fn], and then by calling that
  function.  Note that the arguments to ~c[disassemble$] are evaluated.  Also
  note that ~c[disassemble$] is intended as a top-level utility for the ACL2
  loop, not to be called in code; for such a purpose, include the above book
  and call ~c[disassemble$-fn] directly.

  ~bv[]
  Example Forms:

  (disassemble$ 'foo)
  (disassemble$ 'foo :recompile t)~/

  General Forms:
  (disassemble$ form)
  (disassemble$ form :recompile flg)
  ~ev[]
  where ~c[form] evaluates to a function symbol and ~c[flg] evaluates to any
  value.  If ~c[flg] is ~c[nil], then the existing definition of that function
  symbol is disassembled.  But if ~c[flg] is supplied and has a value other
  than ~c[nil] or ~c[:default], and if that function symbol is defined in the
  ACL2 loop (not merely in raw Lisp; for example, ~pl[set-raw-mode]), then the
  disassembly will be based on a recompilation of that ACL2 definition.
  Normally this recompilation is not necessary, but for some host Lisps, it may
  be useful; in particular, for CCL the above book arranges that source code
  information is saved, so that the output is annotated with such information.
  When recompilation takes place, the previous definition is restored after
  disassembly is complete.  Finally, if ~c[flg] is omitted or has the value
  ~c[:default] ~-[] i.e., in the default case ~-[] then recompilation may take
  place or not, depending on the host Lisp.  The values of ~c[(@ host-lisp)]
  for which recompilation takes place by default may be found by looking at the
  above book, or by including it and evaluating the constant
  ~c[*host-lisps-that-recompile-by-default*].  As of this writing, CCL is the
  only such Lisp (because that is the one for which we can obtain source
  annotation in the output by recompiling).~/"

  `(with-ubt!
    (with-output
     :off (event expansion summary proof-tree)
     (progn
       (include-book "misc/disassemble" :dir :system :ttags '(:disassemble$))
       (value-triple (disassemble$-fn ,fn ,recompile (list ,@args)))))))

(deflabel release-notes
  :doc
  ":Doc-Section release-notes

  pointers to what has changed~/~/

  This section of the online ~il[documentation] contains notes on the
  changes that distinguish successive released versions of ACL2.

  The current version of ACL2 is the value of the constant
  ~c[(@ acl2-version)].")

(deflabel note1
  :doc
  ":Doc-Section release-notes

  Acl2 Version 1.1 Notes~/

  The new features are extensively documented.  The relevant topics
  are:~/

  It is especially important to read all of of the ~il[documentation] for
  ~il[books] before trying to use books.  However, the new ~c[:]~ilc[more]
  keyword command is so handy for reading long ~il[documentation] strings
  that we recommend you start with ~c[:]~ilc[doc] more if reading at the
  terminal.  Some documentation has been written for ~il[guard]s which
  you might find interesting.~/

  :cite books
  :cite more
  :cite guard
  :cite redundant-events")

(deflabel note2
  :doc
  ":Doc-Section release-notes

  Acl2 Version 1.2 Notes~/

  Hacker mode has been eliminated and ~il[programming] mode has been added.
  ~il[Programming] mode is unsound but does syntax checking and permits
  redefinitions of names.  See ~c[:]~ilc[doc] ~c[load-mode] and ~c[:]~ilc[doc] ~c[g-mode].

  The arguments to ~ilc[ld] have changed.  ~ilc[Ld] is now much more
  sophisticated.  ~l[ld].

  For those occasions on which you wish to look at a large list
  structure that you are afraid to print, try ~c[(walkabout x state)],
  where ~c[x] is an Acl2 expression that evaluates to the structure in
  question.  I am afraid there is no ~il[documentation] yet, but it is
  similar in spirit to the Interlisp structure editor.  You are
  standing on an object and commands move you around in it.  E.g., 1
  moves you to its first element, 2 to its second, etc.; 0 moves you
  up to its parent; ~c[nx] and ~c[bk] move you to its next sibling and
  previous sibling; ~c[pp] prettyprints it; ~ilc[q] exits returning ~c[nil]; ~ilc[=] exits
  returning the thing you're standing on; ~c[(= symb)] assigns the thing
  you're standing on to the ~il[state] global variable ~c[symb].

  Several new ~il[hints] have been implemented, including ~c[:by] and ~c[:do-not].
  The old ~c[:do-not-generalize] has been scrapped in favor of such new
  ~il[hints] as ~c[:do-not] ~c[(generalize elim)].  ~c[:By] lets you say ``this goal is
  subsumed by'' a given lemma instance.  The ~c[:by] hint also lets you
  say ``this goal can't be proved yet but skip it and see how the rest
  of the proof goes.'' ~l[hints].~/

  ")

(deflabel note3
  :doc
  ":Doc-Section release-notes

  Acl2 Version 1.3 Notes~/

  ~il[Programming] mode has been eliminated.  Instead, all functions have a
  ``color'' which indicates what can be done with the function.  For
  example, ~c[:red] functions can be executed but have no axioms
  describing them.  Thus, ~c[:red] functions can be introduced after
  passing a simple syntactic check and they can be redefined without
  undoing.  But nothing of consequence can be proved about them.  At
  the other extreme are ~c[:gold] functions which can be executed and
  which also have passed both the termination and the ~il[guard]
  verification proofs.  The color of a function can be specified with
  the new ~ilc[xargs] keyword, ~c[:color], which, if omitted defaults to the
  global setting of ~c[ld-color].  ~c[Ld-color] replaces ~c[load-mode].  Setting
  ~c[ld-color] to ~c[:red] causes behavior similar to the old ~c[:g-mode].
  Setting ~c[ld-color] to ~c[:gold] causes behavior similar to the old
  ~c[:v-mode].  It is possible to prototype your system in ~c[:red] and then
  convert ~c[:red] functions to :~c[blue] individually by calling
  ~ilc[verify-termination] on them.  They can then be converted to ~c[:gold]
  with ~ilc[verify-guards].  This allows us to undertake to verify the
  termination and ~il[guard]s of system functions.  See ~c[:]~ilc[doc] color for an
  introduction to the use of colors.

  Type prescription rules have been added.  Recall that in Nqthm, some
  ~ilc[rewrite] rules were actually stored as ``~il[type-prescription]s.''  Such
  rules allow the user to inform Nqthm's primitive type mechanism as
  to the kinds of shells returned by a function.  Earlier versions of
  Acl2 did not have an analogous kind of rule because Acl2's type
  mechanism is complicated by ~il[guard]s.  Version 1.3 supports
  ~ilc[type-prescription] rules.  ~l[type-prescription].

  Three more new ~il[rule-classes] implement congruence-based rewriting.
  It is possible to identify a binary relation as an equivalence
  relation (~pl[equivalence]), to show that one equivalence
  relation refines another (~pl[refinement]) and to show that a
  given equivalence relation is maintained when rewriting a given
  function call, e.g., ~c[(fn ...xk...)], by maintaining another
  equivalence relation while rewriting the ~c[k]th argument
  (~pl[congruence]).  If ~c[r] has been shown to be an ~il[equivalence]
  relation and then ~c[(implies hyps (r (foo x) (bar x)))] is proved as a
  ~c[:]~ilc[rewrite] rule, then instances of ~c[(foo x)] will be replaced by
  corresponding instances of ~c[(bar x)] provided the instance occurs in a
  slot where the maintainence of ~c[r-equivalence] is known to be
  sufficient and ~c[hyps] can be established as usual.

  In Version 1.2, ~il[rule-classes] were simple keywords, e.g., ~c[:]~ilc[rewrite] or
  ~c[:]~ilc[elim].  In Version 1.3, ~il[rule-classes] have been elaborated to allow
  you to specify how the theorem ought to be used as a rule.  That is,
  the new ~il[rule-classes] allows you to separate the mathematical
  statement of the formula from its interpretation as a rule.
  ~l[rule-classes].

  Rules used to be named by symbols, e.g., ~ilc[car] and ~c[car-cons] were the
  names of rules.  Unfortunately, this was ambiguous because there are
  three rules associated with function symbols: the symbolic
  definition, the executable counterpart, and the ~il[type-prescription];
  many different rules might be associated with theorems, depending on
  the rule classes.  In Version 1.3 rules are named by ``~il[rune]s''
  (which is just short hand for ``rule names'').  Example ~il[rune]s are
  ~c[(:definition car)], ~c[(:executable-counterpart car)], and
  ~c[(:type-prescription car . 1)].  Every rule added by an event has a
  different name and you can ~il[enable] and ~il[disable] them independently.
  ~l[rune] and ~pl[theories].

  The identity function ~ilc[force], of one argument, has been added and
  given a special interpretation by the functions responsible for
  establishing hypotheses in backchaining: When the system fails to
  establish some hypothesis of the form ~c[(force term)], it simply
  assumes it is true and goes on, delaying until later the
  establishment of term.  In particular, pushes a new subgoal to prove
  term in the current context.  When that subgoal is attacked, all of
  the resources of the theorem prover, not just rewriting, are brought
  to bear.  Thus, for example, if you wish to prove the rule
  ~c[(implies (good-statep s) (equal (exec s n) s'))] and it is your
  expectation that every time ~c[exec] appears its first argument is a
  ~c[good-statep] then you might write the rule as
  ~c[(implies (force (good-statep s)) (equal (exec s n) s'))].  This
  rule is essentially an unconditional rewrite of ~c[(exec s n)] to
  ~c[s'] that spawns the new goal ~c[(good-statep s)].  ~l[force].
  Because you can now specify independently how a theorem is used as a
  rule, you need not write the ~ilc[force] in the actual theorem proved.
  ~l[rule-classes].

  Version 1.3 supports a facility similar to Nqthm's ~ilc[break-lemma].
  ~l[break-rewrite].  You can install ``~il[monitor]s'' on ~il[rune]s that
  will cause interactive breaks under certain conditions.

  Acl2 also provides ``~il[wormhole]s'' which allow you to write functions
  that cause interaction with the user but which do not require that
  you have access to ~ilc[state].  ~l[wormhole].

  The rewriter now automatically backchains to stronger recognizers.
  There is no user hook to this feature but it may simplify some
  proofs with which older versions of Acl2 had trouble.  For example,
  if the rewriter is trying to prove ~c[(rationalp (foo a b c))] it is now
  smart enough to try lemmas that match with ~c[(integerp (foo a b c))].~/

  ")

(deflabel note4
  :doc
  ":Doc-Section release-notes

  Acl2 Version 1.4 Notes~/

  Once again ~ilc[ld] only takes one required argument, as the ~c[bind-flg] has
  been deleted.

  Three commands have been added in the spirit of ~c[:]~ilc[pe].  ~c[:]~ilc[Pe!] is
  similar to ~c[:]~ilc[pe] but it prints all ~il[events] with the given name, rather
  than just the most recent.  The command ~c[:]~ilc[pf] prints the corollary
  formula corresponding to a name or ~il[rune].  The command ~c[:]~ilc[pl] (print
  lemmas) prints rules whose top function symbol is the given name.
  ~l[pe!], ~pl[pf], and ~pl[pl].

  Book naming conventions have been changed somewhat.  The
  once-required ~c[.lisp] extension is now prohibited!  Directories are
  supported, including a notion of ``connected book directory''.
  ~l[book-name].  Also, the second argument of ~ilc[certify-book] is
  now optional, defaulting to ~c[0].

  ~il[Compilation] is now supported inside the Acl2 loop.  ~l[comp]
  and ~pl[set-compile-fns].

  The default color is now part of the Acl2 ~il[world];
  see ~c[:]~ilc[doc] ~c[default-color].  ~c[Ld-color] is no longer an ~ilc[ld] special.
  Instead, colors are ~il[events]; see the documentation for ~c[red],
  ~c[pink], ~c[blue], and ~c[gold].

  A ~il[table] exists for controlling whether Acl2 prints comments when it
  ~il[force]s hypotheses of rules; see ~c[:]~ilc[doc] ~c[force-table].  Also, it is now
  possible to turn off the forcing of assumptions by disabling the
  definition of ~il[force]; ~pl[force].

  The event ~c[defconstant] is no longer supported, but a very similar
  event, ~ilc[defconst], has been provided in its place.  ~l[defconst].

  The event for defining ~il[congruence] relations is now ~ilc[defcong]
  (formerly, ~c[defcon]).

  Patterns are now allowed in ~c[:expand] ~il[hints].  See the documentation
  for ~c[:expand] inside the documentation for ~il[hints].

  We have improved the way we report rules used by the simplifier.
  All ~il[rune]s of the same type are reported together in the running
  commentary associated with each goal, so that for example,
  executable counterparts are listed separately from definitions, and
  rewrite rules are listed separately from ~il[linear] rules.  The
  preprocessor now mentions ``simple'' rules; ~pl[simple].

  The mechanism for printing warning messages for new rewrite rules,
  related to subsumption, now avoids worrying about nonrecursive
  function symbols when those symbols are ~il[disable]d.  These messages
  have also been eliminated for the case where the old rule is a
  ~c[:]~ilc[definition] rule.

  Backquote has been modified so that it can usually provide
  predictable results when used on the left side of a rewrite rule.

  Time statistics are now printed even when an event fails.

  The Acl2 trace package has been modified so that it prints using the
  values of the Lisp globals ~c[*print-level*] and ~c[*print-length*]
  (respectively).

  ~il[Table] has been modified so that the ~c[:clear] option lets you replace
  the entire ~il[table] with one that satisfies the ~c[val] and key guards (if
  any); ~pl[table].

  We have relaxed the translation rules for ~c[:measure] ~il[hints] to ~ilc[defun],
  so that the the same rules apply to these terms that apply to terms
  in ~ilc[defthm] ~il[events].  In particular, in ~c[:measure] ~il[hints] ~ilc[mv] is treated
  just like ~ilc[list], and ~ilc[state] receives no special handling.

  The ~il[loop-stopper] test has been relaxed.  The old test required that
  every new argument be strictly less than the corresponding old
  argument in a certain ~il[term-order].  The new test uses a lexicographic
  order on term lists instead.  For example, consider the following
  rewrite rule.
  ~bv[]
    (equal
     (variable-update var1
                      val1 (variable-update var2 val2 vs))
     (variable-update var2
                      val2 (variable-update var1 val1 vs)))
  ~ev[]
  This rule is permutative.  Now imagine that we want to apply this
  rule to the term
  ~bv[]
    (variable-update u y (variable-update u x vs)).
  ~ev[]
  Since the actual corresponding to both ~c[var1] and ~c[var2] is ~c[u], which
  is not strictly less than itself in the ~il[term-order], this rule would
  fail to be applied in this situation when using the old test.
  However, since the pair ~c[(u x)] is lexicographically less than the
  pair ~c[(u y)] with respect to our ~il[term-order], the rule is in fact
  applied using our new test.

  Messages about ~il[events] now contain a space after certain left
  parentheses, in order to assist emacs users.  For example, the event
  ~bv[]
    (defthm abc (equal (+ (len x) 0) (len x)))
  ~ev[]
  leads to a summary containing the line
  ~bv[]
    Form:  ( DEFTHM ABC ...)
  ~ev[]
  and hence, if you search backwards for ``~c[(defthm abc]'', you won't
  stop at this message.

  More tautology checking is done during a proof; in fact, no goal
  printed to the screen, except for the results of applying ~c[:use] and
  ~c[:by] ~il[hints] or the top-level goals from an induction proof, are known
  to Acl2 to be tautologies.

  The ~ilc[ld-query-control-alist] may now be used to suppress printing of
  queries; ~pl[ld-query-control-alist].

  Warning messages are printed with short summary strings, for example
  the string ``~c[Use]'' in the following message.
  ~bv[]
    Acl2 Warning [Use] in DEFTHM:  It is unusual to :USE an enabled
    :REWRITE or :DEFINITION rule, so you may want to consider
    disabling FOO.
  ~ev[]
  At the end of the event, just before the time is printed, all such
  summary strings are printed out.

  The keyword command ~c[:u] has been introduced as an abbreviation for
  ~c[:]~ilc[ubt] ~c[:]~ilc[max].  Printing of query messages is suppressed by ~c[:u].

  The keyword ~c[:cheat] is no longer supported by any event form.

  Some irrelevant formals are detected; ~pl[irrelevant-formals].

  A bug in the application of metafunctions was fixed: now if the
  output of a metafunction is equal to its input, the application of
  the metafunction is deemed unsuccessful and the next metafunction is
  tried.

  An example has been added to the documentation for ~il[equivalence]
  to suggest how to make use of ~il[equivalence] relations in rewriting.

  The following Common Lisp functions have been added to Acl2:
  ~ilc[alpha-char-p], ~ilc[upper-case-p], ~ilc[lower-case-p], ~ilc[char-upcase],
  ~ilc[char-downcase], ~ilc[string-downcase], ~ilc[string-upcase], and ~c[digit-charp-p].

  A documentation section called ~ilc[proof-checker] has been added for the
  interactive facility, whose documentation has been slightly
  improved.  See in particular the documentation for
  ~il[proof-checker], ~ilc[verify], and ~il[macro-command].

  A number of ~il[events] that had been inadvertently disallowed in ~il[books]
  are now permitted in ~il[books].  These are:  ~ilc[defcong], ~c[defcor], ~ilc[defequiv],
  ~ilc[defrefinement], ~ilc[defstub], and ~ilc[verify-termination].

  ~/

  ")

(deflabel note5
  :doc
  ":Doc-Section release-notes

  Acl2 Version 1.5 Notes~/

  Acl2 now allows ``complex rationals,'' which are complex numbers
  whose real parts are rationals and whose imaginary parts are
  non-zero rationals.  ~l[complex].

  A new way of handling ~ilc[force]d hypotheses has been implemented.
  Rather than cause a case split at the time the ~ilc[force] occurs, we
  complete the main proof and then embark on one or more ``forcing
  rounds'' in which we try to prove the ~il[force]d hypotheses.
  ~l[forcing-round].  To allow us to compare the new handling of
  ~ilc[force] with the old, Version 1.5 implements both and uses a flag in
  ~ilc[state] to determine which method should be used.  Do
  ~c[(assign old-style-forcing t)] if you want ~ilc[force] to be handled
  as it was in Version 1.4.  However, we expect to eliminate the
  old-style forcing eventually because we think the new style is more
  effective.  To see the difference between the two approaches to
  forcing, try proving the associativity of ~il[append] under both settings
  of ~c[old-style-forcing].  To get the new behavior invoke:
  ~bv[]
  (thm (implies (and (true-listp a) (true-listp b))
                (equal (append (append a b) c)
                       (append a (append b c)))))
  ~ev[]
  Then ~c[(assign old-style-forcing t)] and invoke the ~c[thm] ~il[command] above
  again.

  A new ~c[:cases] ~il[hints] allows proof by cases.  ~l[hints].

  ~ilc[Include-book] and ~ilc[encapsulate] now restore the ~ilc[acl2-defaults-table]
  when they complete.  ~l[include-book] and ~pl[encapsulate].

  The ~il[guard]s on many Acl2 primitives defined in ~c[axioms.lisp] have been
  weakened to permit them to be used in accordance with lisp custom
  and tradition.

  It is possible to attach heuristic filters to ~c[:]~ilc[rewrite] rules to
  limit their applicability.  ~l[syntaxp].

  A tutorial has been added (but as of Version_3.6.1 it has become obsolete).

  ~il[Events] now print the Summary paragraph listing ~il[rune]s used, time,
  etc., whether they succeed or fail.  The format of the ``~il[failure]
  banner'' has been changed but still has multiple asterisks in it.
  ~c[Thm] also prints a Summary, whether it succeeds or fails; but ~c[thm] is
  not an event.

  A new event form ~ilc[skip-proofs] has been added; ~pl[skip-proofs].

  A user-specific customization facility has been added in the form of
  a book that is automatically included, if it exists on the current
  directory.  ~l[acl2-customization].

  A facility for conditional metalemmas has been implemented;
  ~pl[meta].

  The acceptable values for ~ilc[ld-skip-proofsp] have changed.  In the old
  version (Version 1.4), a value of ~c[t] meant that proofs and ~ilc[local]
  ~il[events] are to be skipped.  In Version 1.5, a value of ~c[t] means proofs
  (but not ~ilc[local] ~il[events]) are to be skipped.  A value of ~c[']~ilc[include-book]
  means proofs and ~ilc[local] ~il[events] are to be skipped.  There are two
  other, more obscure, acceptable values.  ~l[ld-skip-proofsp].

  In order to turn off the forcing of assumptions, one should now
  ~il[disable] the ~c[:]~ilc[executable-counterpart] of ~ilc[force] (rather than the
  ~c[:]~ilc[definition] of ~ilc[force], as in the previous release); ~pl[force].

  The macros ~ilc[enable-forcing] and ~ilc[disable-forcing] make it convenient to
  ~il[enable] or ~il[disable] forcing.  ~l[enable-forcing] and
  ~pl[disable-forcing].

  The new commands ~c[:]~ilc[pr] and ~c[:]~ilc[pr!] print the rules created by an event or
  command.  ~l[pr] and ~pl[pr!].

  The new ~il[history] ~il[command]s ~c[:]~ilc[puff] and ~c[:]~ilc[puff*] will replace a compound
  ~il[command] such as an ~ilc[encapsulate] or ~ilc[include-book] by the sequence of
  ~il[events] in it.  That is, they ``~il[puff] up'' or ``lift'' the subevents
  of a ~il[command] to the ~il[command] level, eliminating the formerly superior
  ~il[command] and lengthening the ~il[history].  This is useful if you want to
  ``partially undo'' an ~ilc[encapsulate] or book or other compound ~il[command]
  so you can experiment.  ~l[puff] and ~pl[puff*].

  Theory expressions now are allowed to use the free variable ~ilc[world]
  and prohibited from using the free variable ~ilc[state].
  ~l[theories], although it is essentially the same as before
  except it mentions ~ilc[world] instead of ~ilc[state].  ~l[world] for a
  discussion of the Acl2 logical ~il[world].  Allowing ~ilc[in-theory] ~il[events] to
  be state-sensitive violated an important invariant about how ~il[books]
  behaved.

  ~ilc[Table] keys and values now are allowed to use the free variable ~ilc[world]
  and prohibited from using the free variable ~ilc[state].  See the note
  above about theory expressions for some explanation.

  The macro for minus, ~ilc[-], used to expand ~c[(- x 3)] to ~c[(+ x -3)] and now
  expands it to ~c[(+ -3 x)] instead.  The old macro, if used in the
  left-hand sides of rewrite rules, produced inapplicable rules
  because the constant occurs in the second argument of the ~ilc[+], but
  potential target terms generally had the constant in the first
  argument position because of the effect of ~c[commutativity-of-+].

  A new class of rule, ~c[:linear-alias] rules, allows one to implement
  the nqthm package and similar hacks in which a ~il[disable]d function is
  to be known equivalent to an arithmetic function.

  A new class of rule, ~c[:built-in-clause] rules, allows one to extend
  the set of clauses proved silently by ~ilc[defun] during measure and ~il[guard]
  processing.  ~l[built-in-clause].

  The new command ~ilc[pcb!] is like ~ilc[pcb] but sketches the ~il[command] and then
  prints its subsidiary ~il[events] in full.  ~l[pcb!].

  ~c[:]~ilc[Rewrite] class rules may now specify the ~c[:]~ilc[loop-stopper] field.
  ~l[rule-classes] and ~pl[loop-stopper].

  The rules for how ~il[loop-stopper]s control permutative rewrite rules
  have been changed.  One effect of this change is that now when the
  built-in commutativity rules for ~ilc[+] are used, the terms ~c[a] and ~c[(- a)]
  are permuted into adjacency.  For example, ~c[(+ a b (- a))] is now
  normalized by the commutativity rules to ~c[(+ a (- a) b)]; in Version
  1.4, ~c[b] was considered syntactically smaller than ~c[(- a)] and so
  ~c[(+ a b (- a))] is considered to be in normal form.  Now it is
  possible to arrange for unary functions be be considered
  ``invisible'' when they are used in certain contexts.  By default,
  ~ilc[unary--] is considered invisible when its application appears in
  the argument list of ~ilc[binary-+].  ~l[loop-stopper] and
  see :DOC set-invisible-fns-table.

  Extensive documentation has been provided on the topic of Acl2's
  ``term ordering.''  ~l[term-order].

  Calls of ~ilc[ld] now default ~ilc[ld-error-action] to ~c[:return] rather than to
  the current setting.

  The ~il[command] descriptor ~c[:x] has been introduced and is synonymous with
  ~c[:]~ilc[max], the most recently executed ~il[command].  ~il[History] ~il[command]s such as
  ~c[:]~ilc[pbt] print a ~c[:x] beside the most recent ~il[command], simply to indicate
  that it ~st[is] the most recent one.

  The ~il[command] descriptor ~c[:x-23] is synonymous with ~c[(:x -23)].  More
  generally, every symbol in the keyword package whose first character
  is ~c[#\\x] and whose remaining ~il[characters] parse as a negative integer
  is appropriately understood.  This allows ~c[:]~ilc[pbt] ~c[:x-10] where ~c[:]~ilc[pbt]
  ~c[(:max -10)] or ~c[:]~ilc[pbt] ~c[(:here -10)] were previously used.  The old forms
  are still legal.

  The order of the arguments to ~ilc[defcong] has been changed.

  The simplifier now reports the use of unspecified built-in type
  information about the primitives with the phrase ``primitive type
  reasoning.''  This phrase may sometimes occur in situations where
  ``propositional calculus'' was formerly credited with the proof.

  The function ~ilc[pairlis] has been replaced in the code by a new function
  ~ilc[pairlis$], because Common Lisp does not adequately specify its
  ~ilc[pairlis] function.

  Some new Common Lisp functions have been added, including ~ilc[logtest],
  ~ilc[logcount], ~ilc[integer-length], ~ilc[make-list], ~ilc[remove-duplicates], ~ilc[string], and
  ~ilc[concatenate].  The source file ~c[/slocal/src/acl2/axioms.lisp] is the
  ultimate reference regarding Common Lisp functions in Acl2.

  The functions ~ilc[defuns] and ~ilc[theory-invariant] have been documented.
  ~l[defuns] and ~pl[theory-invariant].

  A few symbols have been added to the list ~c[*acl2-exports*].

  A new key has been implemented for the ~ilc[acl2-defaults-table],
  ~c[:irrelevant-formals-ok].  ~l[set-irrelevant-formals-ok].

  The connected book directory, ~ilc[cbd], must be nonempty and begin and
  end with a slash.  It is set (and displayed) automatically upon your
  first entry to ~ilc[lp].  You may change the setting with ~ilc[set-cbd].
  ~l[cbd].

  ~c[:]~ilc[oops] will undo the last ~c[:]~ilc[ubt].  ~l[oops].

  Documentation has been written about the ordinals.  See :DOC ~c[e0-ordinalp]
  and see :DOC ~c[e0-ord-<].  [Note added later: Starting with Version_2.8,
  instead ~pl[o-p] and ~pl[o<].~/

  The color ~il[events] ~-[] (red), (pink), (blue), and (gold) ~-[] may no
  longer be enclosed inside calls of ~ilc[local], for soundness reasons.  In
  fact, neither may any event that sets the ~ilc[acl2-defaults-table].
  ~l[embedded-event-form].

  ~l[ld-keyword-aliases] for an example of how to change the exit
  keyword from ~c[:]~ilc[q] to something else.

  The attempt to install a ~il[monitor] on ~c[:]~ilc[rewrite] rules stored as simple
  abbreviations now causes an error because the application of
  abbreviations is not tracked.

  A new message is sometimes printed by the theorem prover, indicating
  that a given simplification is ``specious'' because the subgoals it
  produces include the input goal.  In Version 1.4 this was detected
  but not reported, causing behavior some users found bizarre.
  ~l[specious-simplification].

  ~c[:]~ilc[Definition] rules are no longer always required to specify the
  ~c[:clique] and ~c[:controller-alist] fields; those fields can be defaulted
  to system-determined values in many common instances.
  ~l[definition].

  A warning is printed if a macro form with keyword arguments is given
  duplicate keyword values.  Execute ~c[(thm t :doc nil :doc \"ignored\")]
  and read the warning printed.

  A new restriction has been placed on ~ilc[encapsulate].  Non-~ilc[local]
  recursive definitions inside the ~ilc[encapsulate] may not use, in their
  tests and recursive calls, the constrained functions introduced by
  the ~ilc[encapsulate].  ~l[subversive-recursions].  (Note added in
  Version  2.3:  Subversive recursions were first recognized by us here
  in Version 1.5, but our code for recognizing them was faulty and the
  bug was not fixed until Version  2.3.)

  The ~il[events] ~ilc[defequiv], ~ilc[defcong], ~ilc[defrefinement], and ~ilc[defevaluator] have
  been reimplemented so that they are just macros that expand into
  appropriate ~ilc[defthm] or ~ilc[encapsulate] ~il[events]; they are no longer
  primitive ~il[events].  See the ~il[documentation] of each affected event.

  The ~c[defcor] event, which was a shorthand for a ~ilc[defthm] that
  established a ~il[corollary] of a named, previously proved event, has
  been eliminated because its implementation relied on a technique we
  have decided to ban from our code.  If you want the effect of a
  ~c[defcor] in Version 1.5 you must submit the corresponding ~ilc[defthm] with
  a ~c[:by] hint naming the previously proved event.

  Error reporting has been improved for inappropriate ~ilc[in-theory] ~il[hints]
  and ~il[events], and for syntax errors in rule classes, and for
  non-existent filename arguments to ~ilc[ld].

  Technical Note:  We now maintain the Third Invariant on ~c[type-alists],
  as described in the Essay on the Invariants on Type-alists, and
  Canonicality.  This change will affect some proofs, for example, by
  causing a to rewrite more quickly to ~c[c] when ~c[(equiv a b)] and
  ~c[(equiv b c)] are both known and ~c[c] is the canonical
  representative of the three.

  ~/

  ")

(deflabel note6
  :doc
  ":Doc-Section release-notes

  Acl2 Version 1.6 Notes~/

  A new key has been implemented for the ~ilc[acl2-defaults-table],
  ~c[:ignore-ok].  ~l[set-ignore-ok].

  It is now legal to have color ~il[events], such as ~c[(red)], in the
  ~il[portcullis] of a book.  More generally, it is legal to set the
  ~ilc[acl2-defaults-table] in the ~il[portcullis] of a book.  For example, if
  you execute ~c[:red] and then certify a book, the event ~c[(red)] will show
  up in the ~il[portcullis] of that book, and hence the definitions in that
  book will all be red (except when overridden by appropriate
  declarations or ~il[events]).  When that book is included, then as
  always, its ~il[portcullis] must first be ``raised,'' and that will cause
  the default color to become red before the ~il[events] in the book are
  executed.  As always, the value of ~ilc[acl2-defaults-table] immediately
  after execution of an ~ilc[include-book], ~ilc[certify-book], or ~ilc[encapsulate]
  form will be the same as it was immediately before execution (and
  hence, so will the default color).  ~l[portcullis] and, for
  more about books, ~pl[books].

  A theory ~ilc[ground-zero] has been defined to contain exactly those rules
  that are ~il[enable]d when Acl2 starts up.  ~l[ground-zero].

  The function ~ilc[nth] is now ~il[enable]d, correcting an oversight from
  Version 1.5.

  Customization files no longer need to meet the syntactic
  restrictions put on ~il[books]; rather, they can contain arbitrary Acl2
  forms.  ~l[acl2-customization].

  Structured directory names and structured file names are supported;
  see especially the documentation for ~il[pathname], ~il[book-name],
  and ~ilc[cbd].

  Acl2 now works with some Common Lisp implementations other than
  akcl, including Lucid, Allegro, and MCL.

  A facility has been added for displaying proof trees, especially
  using emacs; ~pl[proof-tree].

  There is a considerable amount of new ~il[documentation], in particular
  for the printing functions ~ilc[fmt], ~ilc[fmt1], and ~ilc[fms], and for the notion of
  Acl2 term (~pl[term]).

  It is possible to introduce new well-founded relations, to specify
  which relation should be used by ~ilc[defun], and to set a default
  relation.  ~l[well-founded-relation].

  It is possible to make functions suggest new inductions.
  ~l[induction].

  It is possible to change how Acl2 expresses ~il[type-set] information; in
  particular, this affects what clauses are proved when ~il[force]d
  assumptions are generated.  ~l[type-set-inverter].

  A new restriction has been added to ~ilc[defpkg], having to do with
  undoing.  If you undo a ~ilc[defpkg] and define the same package name
  again, the imports list must be identical to the previous imports or
  else an explanatory error will occur.
  ~l[package-reincarnation-import-restrictions].

  ~ilc[Theory-invariant] and ~ilc[set-irrelevant-formals-ok] are now embedded
  event forms.

  The command ~c[:]~ilc[good-bye] may now be used to quit entirely out of Lisp,
  thus losing your work forever.  This command works in akcl but may
  not work in every Common Lisp.

  A theory ~ilc[ground-zero] has been added that contains exactly the
  ~il[enable]d rules in the ~il[startup] theory.  ~l[ground-zero].

  ~c[Define-pc-macro] and ~c[define-pc-atomic-macro] now automatically define
  ~c[:red] functions.  (It used to be necessary, in general, to change
  color to ~c[:red] before invoking these.)

  ~/

  For a proof of the well-foundedness of ~c[e0-ord-<] on the ~c[e0-ordinalp]s,
  ~pl[proof-of-well-foundedness].  [Note added later: Starting with
  Version_2.8, ~ilc[o<] and ~ilc[o-p] replace ~c[e0-ord-<] and ~c[e0-ordinalp],
  respectively.]

  Free variables are now handled properly for hypotheses of
  ~c[:]~ilc[type-prescription] rules.

  When the system is loaded or saved, ~ilc[state] is now bound to
  ~c[*the-live-state*].

  ~ilc[Certify-book] has been modified so that when it compiles a file, it
  loads that object file.

  ~ilc[Defstub] has been modified so that it works when the color is hot
  (~c[:red] or ~c[:pink]).

  Several basic, but not particularly commonly used, ~il[events] have been
  added or changed.  The obscure axiom ~c[symbol-name-intern] has been
  modified.  The definition of ~c[firstn] has been changed.  ~ilc[Butlast] is
  now defined.  The definition of ~ilc[integer-length] has been modified.
  The left-hand side of the rewrite rule ~c[rational-implies2] has been
  changed from ~c[(* (numerator x) (/ (denominator x)))] to
  ~c[(* (/ (denominator x)) (numerator x))], in order to respect the
  fact that ~ilc[unary-/] is invisible with respect to ~ilc[binary-*].
  ~l[loop-stopper].

  The `preprocess' process in the waterfall (~pl[hints] for a
  discussion of the ~c[:do-not] hint) has been changed so that it works to
  avoid case-splitting.  The `simplify' process refuses to force
  (~pl[force]) when there are ~ilc[if] terms, including ~ilc[and] and ~ilc[or]
  terms, in the goal being simplified.

  The function ~c[apply] is no longer introduced automatically by
  translation of user input to internal form when functions are called
  on inappropriate explicit values, e.g., ~c[(car 3)].

  The choice of which variable to use as the measured variable in a
  recursive definition has been very slightly changed.

  ~/

  ")

(deflabel note7
  :doc
  ":Doc-Section release-notes

  ACL2 Version 1.7 (released October 1994) Notes~/

  ~ilc[Include-book] now takes (optionally) an additional keyword
  argument, indicating whether a compiled file is to be loaded.  The
  default behavior is unchanged, except that a warning is printed when
  a compiled file is not loaded.  ~l[include-book].

  A ~il[markup] language for ~il[documentation] strings has been implemented,
  and many of the source files have been marked up using this language
  (thanks largely to the efforts of Laura Lawless).  ~l[markup].
  Moreover, there are translators that we have used to provide
  versions of the ACL2 ~il[documentation] in info (for use in emacs), html
  (for Mosaic), and tex (for hardcopy) formats.

  A new event ~ilc[defdoc] has been implemented.  It is like ~ilc[deflabel],
  but allows redefinition of ~il[doc] strings and has other advantages.
  ~l[defdoc].

  We used to ignore corollaries when collecting up the axioms
  introduced about constrained functions.  That bug has been fixed.
  We thank John Cowles for bringing this bug to our attention.

  The macro ~ilc[defstub] now allows a ~c[:]~ilc[doc] keyword argument, so that
  ~il[documentation] may be attached to the name being introduced.

  A new command ~ilc[nqthm-to-acl2] has been added to help Nqthm users to
  make the transition to ACL2.  ~l[nqthm-to-acl2], which also
  includes a complete listing of the relevant tables.

  Many function names, especially of the form ``foo~c[-lst]'', have been
  changed in order to support the following convention, for any
  ``foo'':
  ~bf[]
  ~c[(foo-listp lst)] represents the notion ~c[(for x in lst always foop x)].
  ~ef[]
  A complete list of these changes may be found at the end of this
  note.  All of them except ~c[symbolp-listp] and
  ~c[list-of-symbolp-listp] have the string ``~c[-lst]'' in their names.
  Note also that ~c[keyword-listp] has been renamed ~ilc[keyword-value-listp].

  Accumulated persistence has been implemented.  It is not connected
  to ~c[:]~ilc[brr] or rule monitoring.  ~l[accumulated-persistence].

  ~c[:Trigger-terms] has been added for ~c[:]~ilc[linear] rule classes, so you
  can hang a ~il[linear] rule under any addend you want.  ~l[linear],
  which has been improved and expanded.

  ACL2 now accepts ~c[256] ~il[characters] and includes the Common Lisp
  functions ~ilc[code-char] and ~ilc[char-code].  However, ACL2 controls the lisp
  reader so that ~c[#\\c] may only be used when ~c[c] is a single standard
  character or one of ~c[Newline], ~c[Space], ~c[Page], ~c[Rubout], ~c[Tab].  If you want
  to enter other ~il[characters] use ~ilc[code-char], e.g.,
  ~c[(coerce (list (code-char 7) (code-char 240) #\a) 'string)].
  ~l[characters].  Note:  our current handling of ~il[characters]
  makes the set of theorems different under Macintosh Common Lisp
  (MCL) than under other Common Lisps.  We hope to rectify this
  situation before the final release of ACL2.

  A new ~il[table], ~ilc[macro-aliases-table], has been implemented, that
  associates macro names with function names.  So for example, since
  ~ilc[append] is associated with ~ilc[binary-append], the form ~c[(disable append)]
  it is interpreted as though it were ~c[(disable binary-append)].
  ~l[macro-aliases-table], ~pl[add-macro-alias] and
  ~pl[remove-macro-alias].

  The implementation of conditional metalemmas has been modified so
  that the metafunction is applied before the hypothesis metafunction
  is applied.  ~l[meta].

  The Common Lisp functions ~ilc[acons] and ~ilc[endp] have been defined in
  the ACL2 logic.

  We have added the symbol ~ilc[declare] to the list ~c[*acl2-exports*],
  and hence to the package ~c[\"ACL2-USER\"].

  A new hint, ~c[:restrict], has been implemented.  ~l[hints].

  It used to be that if ~c[:]~ilc[ubt] were given a number that is greater
  than the largest current ~il[command] number, it treated that number the
  same as ~c[:]~ilc[max].  Now, an error is caused.

  The ~il[table] ~c[:force-table] has been eliminated.

  A command ~c[:]~ilc[disabledp] (and macro ~ilc[disabledp]) has been added;
  ~pl[disabledp].

  ~il[Compilation] via ~c[:]~ilc[set-compile-fns] is now suppressed during
  ~ilc[include-book].  In fact, whenever the ~il[state] global variable
  ~ilc[ld-skip-proofsp] has value ~c[']~ilc[include-book].

  ~/

  Here are some less important changes, additions, and so on.

  Unlike previous releases, we have not proved all the theorems in
  ~c[axioms.lisp]; instead we have simply assumed them.  We have deferred
  such proofs because we anticipate a fairly major changed in Version
  1.8 in how we deal with ~il[guard]s.

  We used to (accidentally) prohibit the ``redefinition'' of a ~il[table]
  as a function.  That is no longer the case.

  The check for whether a ~il[corollary] follows tautologically has been
  sped up, at the cost of making the check less ``smart'' in the
  following sense:  no longer do we expand primitive functions such as
  ~ilc[implies] before checking this propositional implication.

  The ~il[command] ~ilc[ubt!] has been modified so that it never causes or
  reports an error.  ~l[ubt!].

  ACL2 now works in Harlequin LispWorks.

  The user can now specify the ~c[:trigger-terms] for ~c[:]~ilc[linear] rules.
  ~l[linear].

  The name of the system is now ``ACL2''; no longer is it ``Acl2''.

  The raw lisp counterpart of ~ilc[theory-invariant] is now defined to be a
  no-op as is consistent with the idea that it is just a call of
  ~ilc[table].

  A bug was fixed that caused ~il[proof-checker] ~il[instructions] to be
  executed when ~ilc[ld-skip-proofsp] was ~c[t].

  The function ~ilc[rassoc] has been added, along with a corresponding
  function used in its ~il[guard], ~c[r-eqlable-alistp].

  The ~ilc[in-theory] event and hint now print a warning not only when
  certain ``primitive'' ~c[:]~ilc[definition] rules are ~il[disable]d, but also when
  certain ``primitive'' ~c[:]~ilc[executable-counterpart] rules are ~il[disable]d.

  The modified version of ~c[trace] provided by ACL2, for use in raw
  Lisp, has been modified so that the lisp special variable
  ~c[*trace-alist*] is consulted.  This alist associates, using ~ilc[eq],
  values with their print representations.  For example, initially
  ~c[*trace-alist*] is a one-element list containing the pair
  ~c[(cons state '|*the-live-state*|)].

  The system now prints an observation when a form is skipped because
  the default color is ~c[:red] or ~c[:pink].  (Technically:  ~c[when-cool] has
  been modified.)

  Additional protection exists when you submit a form to raw Common
  Lisp that should only be submitted inside the ACL2 read-eval-print
  loop.

  Here is a complete list of the changes in function names described
  near the top of this note, roughly of the form
  ~bv[]
  foo-lst --> foo-listp
  ~ev[]
  meaning:  the name ``~c[foo-lst]'' has been changed to ``~c[foo-listp].''
  ~bv[]
  symbolp-listp    --> symbol-listp
  list-of-symbolp-listp  --> symbol-list-listp
                         {for consistency with change to symbol-listp}
  rational-lst     --> rational-listp
                       {which in fact was already defined as well}
  integer-lst      --> integer-listp
  character-lst    --> character-listp
  stringp-lst      --> string-listp
  32-bit-integer-lst   --> 32-bit-integer-listp
  typed-io-lst     --> typed-io-listp
  open-channel-lst --> open-channel-listp
  readable-files-lst   --> readable-files-listp
  written-file-lst --> written-file-listp
  read-file-lst    --> read-file-listp
  writeable-file-lst   --> writable-file-listp
                       {note change in spelling of ``writable''}
  writeable-file-lst1  --> writable-file-listp1
  pseudo-termp-lst     --> pseudo-term-listp
  hot-termp-lst --> hot-term-listp {by analogy with pseudo-term-listp}
  weak-termp-lst   --> weak-term-listp
  weak-termp-lst-lst   --> weak-termp-list-listp
  ts-builder-case-lstp -> ts-builder-case-listp
  quotep-lst       --> quote-listp
  termp-lst        --> term-listp
  instr-lst        --> instr-listp
  spliced-instr-lst    --> spliced-instr-listp
  rewrite-fncallp-lst  --> rewrite-fncallp-listp
  every-occurrence-equiv-hittablep1-lst -->
              every-occurrence-equiv-hittablep1-listp
  some-occurrence-equiv-hittablep1-lst  -->
              some-occurrence-equiv-hittablep1-listp
              {by analogy with the preceding, even though it's a
               ``some'' instead of ``all'' predicate]
  almost-quotep1-lst   --> almost-quotep1-listp
  ffnnames-subsetp-lst --> ffnnames-subsetp-listp
  boolean-lstp     --> boolean-listp
  subst-expr1-lst-okp  --> subst-expr1-ok-listp
  ~ev[]
  ~/

  ")

(deflabel note8
  :doc
  ":Doc-Section release-notes

  ACL2 Version 1.8 (May, 1995) Notes~/

  ~l[note8-update] for yet more recent changes.

  ~il[Guard]s have been eliminated from the ACL2 logic.  A summary is
  contained in this brief note.  Also ~pl[defun-mode] and
  ~pl[set-guard-checking].

  ~il[Guard]s may be included in ~il[defuns] as usual but are ignored from the
  perspective of admission to the logic: functions must terminate on
  all arguments.

  As in Nqthm, primitive functions, e.g., ~ilc[+] and ~ilc[car], logically
  default unexpected arguments to convenient values.  Thus, ~c[(+ 'abc 3)]
  is ~c[3] and ~c[(car 'abc)] is ~c[nil].  ~l[programming], and see
  the ~il[documentation] for the individual primitive functions.

  In contrast to earlier versions of ACL2, Version 1.8 logical
  functions are executed at Nqthm speeds even when ~il[guard]s have not
  been verified.  In versions before 1.8, such functions were
  interpreted by ACL2.

  Colors have been eliminated.  Two ``~il[defun-mode]s'' are supported,
  ~c[:]~ilc[program] and ~c[:]~ilc[logic].  Roughly speaking,
  ~c[:]~ilc[program] does what ~c[:red] used to do, namely, allow you to
  prototype functions for execution without any proof burdens.
  ~c[:]~ilc[Logic] mode does what ~c[:blue] used to do, namely, allow you to
  add a new definitional axiom to the logic.  A global ~il[default-defun-mode]
  is comparable to the old default color.  The system comes up in
  ~c[:]~ilc[logic] mode.  To change the global ~il[defun-mode], type
  ~c[:]~ilc[program] or ~c[:]~ilc[logic] at the top-level.  To specify the
  ~il[defun-mode] of a ~ilc[defun] locally use
  ~bv[]
  ~c[(declare (xargs :mode mode))].
  ~ev[]

  The ~il[prompt] has changed.  The initial ~il[prompt], indicating
  ~c[:]~ilc[logic] mode, is
  ~bv[]
  ACL2 !>
  ~ev[]
  If you change to ~c[:]~ilc[program] mode the ~il[prompt] becomes
  ~bv[]
  ACL2 p!>
  ~ev[]

  ~il[Guard]s can be seen as having either of two roles: (a) they are a
  specification device allowing you to characterize the kinds of
  inputs a function ``should'' have, or (b) they are an efficiency
  device allowing logically defined functions to be executed directly
  in Common Lisp.  If a ~il[guard] is specified, as with ~ilc[xargs] ~c[:]~ilc[guard], then
  it is ``verified'' at defun-time (unless you also specify ~ilc[xargs]
  ~c[:verify-guards nil]).  ~il[Guard] verification means what it always has:
  the input ~il[guard] is shown to imply the ~il[guard]s on all subroutines in
  the body.  If the ~il[guard]s of a function are verified, then a call of
  the function on inputs satisfying the ~il[guard] can be computed directly
  by Common Lisp.  Thus, verifying the ~il[guard]s on your functions will
  allow them to execute more efficiently.  But it does not affect
  their logical behavior and since you will automatically get Nqthm
  speeds on unverified logical definitions, most users will probably
  use ~il[guard]s either as a specification device or only use them when
  execution efficiency is extremely important.

  Given the presence of ~il[guard]s in the system, two issues are unavoidable.
  Are ~il[guard]s verified as part of the ~ilc[defun] process?  And are ~il[guard]s checked
  when terms are evaluated?  We answer both of those questions below.

  Roughly speaking, in its initial ~il[state] the system will try to verify
  the ~il[guard]s of a ~ilc[defun] if a ~c[:]~ilc[guard] is supplied in the ~ilc[xargs]
  and will not try otherwise.  However, ~il[guard] verification in ~ilc[defun]
  can be inhibited ``locally'' by supplying the ~ilc[xargs]
  ~c[:]~ilc[verify-guards] ~c[nil].  ``Global'' inhibition can be obtained via
  the ~c[:]~ilc[set-verify-guards-eagerness].  If you do not use the
  ~c[:]~ilc[guard] ~ilc[xargs], you will not need to think about ~il[guard]
  verification.

  We now turn to the evaluation of expressions.  Even if your functions contain
  no ~il[guard]s, the primitive functions do and hence you have the choice: when you
  submit an expression for evaluation do you mean for ~il[guard]s to be checked at
  runtime or not?  Put another way, do you mean for the expression to be
  evaluated in Common Lisp (if possible) or in the logic?  Note: If Common Lisp
  delivers an answer, it will be the same as in the logic, but it might be
  erroneous to execute the form in Common Lisp.  For example, should
  ~c[(car 'abc)] cause a ~il[guard] violation error or return ~c[nil]?

  The top-level ACL2 loop has a variable which controls which sense of
  execution is provided.  To turn ``~il[guard] checking on,'' by which we
  mean that ~il[guard]s are checked at runtime, execute the top-level form
  ~c[:set-guard-checking t].  To turn it off, do ~c[:set-guard-checking nil].
  The status of this variable is reflected in the ~il[prompt].
  ~bv[]
  ACL2 !>
  ~ev[]
  means ~il[guard] checking is on and
  ~bv[]
  ACL2 >
  ~ev[]
  means ~il[guard] checking is off.  The exclamation mark can be thought of
  as ``barring'' certain computations.  The absence of the mark
  suggests the absence of error messages or unbarred access to the
  logical axioms.  Thus, for example
  ~bv[]
  ACL2 !>(car 'abc)
  ~ev[]
  will signal an error, while
  ~bv[]
  ACL2 >(car 'abc)
  ~ev[]
  will return ~c[nil].

  Note that whether or not ~il[guard]s are checked at runtime is
  independent of whether you are operating in ~c[:]~ilc[program] mode or
  ~c[:]~ilc[logic] mode and whether theorems are being proved or not.
  (Although it must be added that functions defined in ~c[:]~ilc[program]
  mode cannot help but check their ~il[guard]s because no logical
  definition exists.)

  Version 1.8 permits the verification of the ~il[guard]s of theorems, thus
  insuring that all instances of the theorem will evaluate without
  error in Common Lisp.  To verify the ~il[guard]s of a theorem named
  ~c[name] execute the event
  ~bv[]
  (verify-guards name).
  ~ev[]
  If a theorem's ~il[guard]s have been verified, the theorem is guaranteed
  to evaluate without error to non-~c[nil] in Common Lisp (provided
  resource errors do not arise).

  Caveat about ~ilc[verify-guards]: ~ilc[implies] is a function symbol, so in the
  term ~c[(implies p q)], ~c[p] cannot be assumed true when ~c[q] is evaluated;
  they are both evaluated ``outside.''  Hence, you cannot generally
  verify the ~il[guard]s on a theorem if ~ilc[implies] is used to state the
  hypotheses.  Use ~ilc[if] instead.  In a future version of ACL2, ~ilc[implies]
  will likely be a macro.

  See sum-list-example.lisp for a nice example of the use of Version
  1.8.  This is roughly the same as the documentation for
  ~il[guard-example].

  We have removed the capability to do ``old-style-forcing'' as
  existed before Version 1.5.  ~l[note5].

  NOTE:  Some low level details have, of course, changed.  One such
  change is that there are no longer two distinct type prescriptions
  stored when a function is admitted with its ~il[guard]s verified.  So for
  example, the type prescription ~il[rune] for ~ilc[binary-append] is now
  ~bv[]
  (:type-prescription binary-append)
  ~ev[]
  while in Versions 1.7 and earlier, there were two such ~il[rune]s:
  ~bv[]
  (:type-prescription binary-append . 1)
  (:type-prescription binary-append . 2)
  ~ev[]

  Nqthm-style forcing on ~il[linear] arithmetic assumptions is no longer
  executed when forcing is ~il[disable]d.

  Functional instantiation now benefits from a trick also used in
  Nqthm:  once a ~il[constraint] generated by a ~c[:functional-instance]
  lemma instance (~pl[lemma-instance]) has been proved on behalf
  of a successful event, it will not have to be re-proved on behalf of
  a later event.

  ~ilc[1+] and ~ilc[1-] are now macros in the logic, not functions.  Hence, for
  example, it is ``safe'' to use them on left-hand sides of rewrite
  rules, without invoking the common warning about the presence of
  nonrecursive function symbols.

  A new ~il[documentation] section ~il[file-reading-example] illustrates how to
  process forms in a file.

  A new ~il[proof-checker] command ~c[forwardchain] has been added;
  ~pl[acl2-pc::forwardchain].

  It is now possible to use quantifiers.  ~l[defun-sk] and
  ~pl[defchoose].

  There is a new event ~ilc[set-inhibit-warnings], which allows the user
  to turn off warnings of various types.
  ~pl[set-inhibit-warnings].

  An unsoundness relating ~ilc[encapsulate] and ~c[:functional-instance]
  ~il[hints] has been remedied, with a few small effects visible at the
  user level.  The main observable effect is that ~ilc[defaxiom] and
  non-local ~ilc[include-book] ~il[events] are no longer allowed in the scope
  of any ~ilc[encapsulate] event that has a non-empty ~il[signature].

  When ~ilc[certify-book] is called, we now require that the default
  ~il[defun-mode] (~pl[default-defun-mode]) be ~c[:]~ilc[logic].  On a related
  note, the default ~il[defun-mode] is irrelevant to ~ilc[include-book]; the
  mode is always set to ~c[:]~ilc[logic] initially, though it may be changed
  within the book and reverts to its original value at the conclusion
  of the ~ilc[include-book].  A bug in ~ilc[include-book] prevented it from
  acting this way even though the ~il[documentation] said otherwise.

  The ~il[documentation] has been substantially improved.  A new
  section ``Programming'' contains ~il[documentation] of many useful
  functions provided by ACL2; ~pl[programming].  Also, the
  ~il[documentation] has been ``marked up'' extensively.  Thus in
  particular, users of Mosaic will find many links in the
  ~il[documentation].

  The symbols ~ilc[force], ~ilc[mv-nth], and ~c[acl2-count] have been added
  to the list ~c[*acl2-exports*].

  We now permit most names from the main Lisp package to be used as
  names, except for names that define functions, macros, or constants.
  ~l[name].

  We have changed the list of imports from the Common Lisp package to
  ACL2, i.e., the list ~c[*common-lisp-symbols-from-main-lisp-package*],
  to be exactly those external symbols of the Common Lisp package as
  specified by the draft Common Lisp standard.  In order to
  accommodate this change, we have renamed some ACL2 functions as
  shown below, but these and other ramifications of this change should
  be transparent to most ACL2 users.
  ~bv[]
  warning      --> warning$
  print-object --> print-object$
  ~ev[]

  Proof trees are no longer enabled by default.  To start them up,
  ~c[:]~ilc[start-proof-tree].

  We have added the capability of building smaller images.  The
  easiest way to do this on a Unix (trademark of AT&T) system is:
  ~c[make small].

  ~/

  Here we will put some less important changes, additions, and so on.

  We have added definitions for the Common Lisp function ~ilc[position]
  (for the test ~ilc[eql]), as well as corresponding versions
  ~ilc[position-equal] and ~ilc[position-eq] that use tests ~ilc[equal] and
  ~ilc[eq], respectively.  ~l[position], ~pl[position-equal],
  and ~pl[position-eq].

  The ~ilc[defthm] event ~c[rational-listp-implies-rationalp-car] no
  longer exists.

  We fixed a bug in the hint mechanism that applied ~c[:by], ~c[:cases], and
  ~c[:use] ~il[hints] to the first induction goal when the prover reverted to
  proving the original goal by induction.

  We fixed a bug in the handling of ~c[(set-irrelevant-formals-ok :warn)].

  In support of removing the old-style forcing capability, we deleted
  the initialization of ~il[state] global ~c[old-style-forcing] and deleted the
  definitions of ~c[recover-assumptions], ~c[recover-assumptions-from-goal],
  ~c[remove-assumptions1], ~c[remove-assumptions], and ~c[split-on-assumptions],
  and we renamed ~c[split-on-assumptions1] to ~c[split-on-assumptions].

  The special value ~c['none] in the ~il[proof-checker] commands ~c[claim] and ~ilc[=]
  has been replaced by ~c[:none].

  A bug in the handling of ~il[hints] by subgoals has been fixed.  For
  example, formerly a ~c[:do-not] hint could be ``erased'' by a ~c[:use] hint
  on a subgoal.  Thanks go to Art Flatau for noticing the bug.

  The functions ~c[weak-termp] and ~c[weak-term-listp] have been
  deleted, and their calls have been replaced by corresponding calls
  of ~ilc[pseudo-termp] and ~c[pseudo-term-listp].  The notion of
  ~ilc[pseudo-termp] has been slightly strenthened by requiring that
  terms of the form ~c[(quote ...)] have length 2.

  Performance has been improved in various ways.  At the prover level,
  backchaining through the recognizer alist has been eliminated in
  order to significantly speed up ACL2's rewriter.  Among the other
  prover changes (of which there are several, all technical):  we no
  longer clausify the input term when a proof is interrupted in favor
  of inducting on the input term.  At the ~il[IO] level, we have improved
  performance somewhat by suitable declarations and proclamations.
  These include technical modifications to the macros ~ilc[mv] and
  ~ilc[mv-let], and introduction of a macro ~c[the-mv] analogous to the
  macro ~ilc[the] but for forms returning multiple values.

  The function ~c[spaces] now takes an extra argument, the current column.

  A bug in the ~il[proof-checker] ~c[equiv] command was fixed.

  The function ~c[intersectp] has been deleted, because it was
  essentially duplicated by the function ~ilc[intersectp-equal].

  We now proclaim functions in AKCL and GCL before compiling ~il[books].
  This should result in somewhat increased speed.

  The function ~c[repeat] has been eliminated; use ~ilc[make-list]
  instead.

  The ~il[proof-checker] command ~c[expand] has been fixed so that it
  eliminates ~ilc[let] (lambda) expressions when one would expect it to.

  A new primitive function, ~ilc[mv-nth], has been introduced.  ~ilc[Mv-nth]
  is equivalent to ~ilc[nth] and is used in place of ~ilc[nth] in the
  translation of ~ilc[mv-let] expressions.  This allows the user to
  control the simplification of ~ilc[mv-let] expressions without
  affecting how ~ilc[nth] is treated.  In that spirit, the rewriter has
  been modified so that certain ~ilc[mv-nth] expressions, namely those
  produced in the translation of ~c[(mv-let (a b c)(mv x y z) p)], are
  given special treatment.

  A minor bug in ~c[untranslate] has been fixed, which for example will
  fix the printing of conjunctions.

  ~c[Translate] now takes a ~c[logicp] argument, which indicates whether it
  enforces the restriction that ~c[:]~ilc[program] mode functions do not occur
  in the result.

  The modified version of ~c[trace] provided by ACL2, for use in raw Lisp,
  has been modified so that the lisp special variable ~c[*trace-alist*]
  has a slightly different functionality.  This alist associates,
  using ~ilc[eq], symbols with the print representations of their values.
  For example, initially ~c[*trace-alist*] is a one-element list
  containing the pair ~c[(cons 'state '|*the-live-state*|)].  Thus, one
  may cons the pair ~c[(cons '*foo* \"It's a FOO!\")] on to ~c[*trace-alist*];
  then until ~c[*foo*] is defined, this change will have no effect, but
  after for example
  ~bv[]
  (defconst *foo* 17)
  ~ev[]
  then ~c[trace] will print ~c[17] as ~c[\"It's a FOO!\"].

  ~c[Trace] also traces the corresponding logic function.

  ~il[Proof-tree] display has been improved slightly in the case of
  successful proofs and certain event failures.

  The function ~c[positive-integer-log2] has been deleted.

  The macro ~ilc[skip-proofs] now prints a warning message when it is
  encountered in the context of an ~ilc[encapsulate] event or a book.
  ~l[skip-proofs].

  Some functions related to ~c[the-fn] and ~c[wormhole1] now have
  ~il[defun-mode] ~c[:]~ilc[program], but this change is almost certain to
  be inconsequential to all users.

  ~/

  ")

(deflabel note8-update
  :doc
  ":Doc-Section release-notes

  ACL2 Version 1.8 (Summer, 1995) Notes~/

  ACL2 can now use Ordered Binary Decision Diagram technology.
  ~l[bdd].  There is also a ~il[proof-checker] ~c[bdd] command.

  ACL2 is now more respectful of the intention of the function
  ~ilc[hide].  In particular, it is more careful not to dive inside any
  call of ~ilc[hide] during equality substitution and case splitting.

  The ~ilc[ld] special (~pl[ld]) ~ilc[ld-pre-eval-print] may now be used
  to turn off printing of input forms during processing of
  ~ilc[encapsulate] and ~ilc[certify-book] forms, by setting it to the value
  ~c[:never], i.e., ~c[(set-ld-pre-eval-print :never state)].
  ~l[ld-pre-eval-print].

  The TUTORIAL documentation section (now obsolete) has, with much help from
  Bill Young, been substantially improved to a bona fide introduction.

  The term pretty-printer has been modified to introduce ~c[(<= X Y)]
  as an abbreviation for ~c[(not (< Y X))].

  Forward chaining and linear arithmetic now both benefit from the
  evaluation of ground subterms.

  A new macro ~ilc[set-inhibit-output-lst] has been defined.  This should
  be used when setting the ~il[state] global ~c[inhibit-output-lst];
  ~pl[set-inhibit-output-lst] and ~pl[proof-tree].

  The test for redundancy in definitions includes the ~il[guard] and type
  declarations.  ~l[redundant-events].

  ~l[generalized-booleans] for a discussion of a potential
  soundness problem for ACL2 related to the question:  Which Common
  Lisp functions are known to return Boolean values?

  ~/

  Here we will put some less important changes, additions, and so on.

  A bug has been fixed so that now, execution of ~c[:comp t]
  (~pl[comp]) correctly handles non-standard characters.

  A bug in ~ilc[digit-char-p] has been fixed, so that the ``default'' is
  ~c[nil] rather than ~c[0].

  ~ilc[True-listp] now tests the final ~ilc[cdr] against ~c[nil] using ~ilc[eq]
  instead of ~ilc[equal], for improved efficiency.  The logical meaning
  is, however, unchanged.

  ~ilc[Put-assoc-equal] has been added to the logic (it used to have
  ~c[:]~ilc[defun-mode] ~c[:]~ilc[program], and has been documented.

  ~/

  ")

(deflabel note9
  :doc
  ":Doc-Section release-notes

  ACL2 Version 1.9 (Fall, 1996) Notes~/

  By default, when the system is started it is illegal to use the
  variable ~ilc[STATE] as a formal parameter of a function definition.
  The aim is to prevent novice users from stumbling into the Byzantine
  syntactic restrictions on that variable symbol.  Use
  ~bv[]
  :set-state-ok t
  ~ev[]
  or, equivalently,
  ~bv[]
  (set-state-ok t)
  ~ev[]
  to switch back to the old default mode.  ~l[set-state-ok]

  ~c[Set-state-ok] is an event that affects the ACL2 defaults table
  (~pl[acl2-defaults-table]).  Recall that when books are
  included, the defaults table is restored to its pre-inclusion state.
  Thus, while a ~c[set-state-ok] form will permit the book to define a
  ~c[state]-using function, it will not permit the user of the book to
  make such a definition.  We recommend putting ~c[(set-state-ok t)] in
  any book that defines a ~c[state] using function.

  Books certified under Version 1.8 must be recertified under Version
  1.9.  See :DOC version.

  The simplifier has been made to look out for built-in clauses,
  whereas in past versions such clauses were only noticed by the
  ``preprocessor'' at the top of the waterfall.  THIS CHANGE MAY
  PREVENT OLD SCRIPTS FROM REPLAYING!  The undesirable side-effect is
  caused by the fact that ~c[:HINTS] require you to refer to clauses by
  their exact name (~pl[goal-spec]) and because the new simplifier
  proves more clauses than before, the goals produced have different
  names.  Thus, if a script uses ~c[:HINTS] that refer to clauses other
  than \"Goal\", e.g., \"Subgoal 1.3\" then the hint may be applied to
  a different subgoal than originally intended.

  The use of built-in-clauses has been made more efficient.  If a set
  of clauses arise often in a piece of work, it might be advantageous
  to build them in even if that results in a large set (hundreds?) of
  built-in clauses.  ~l[built-in-clause]

  Wormholes can now be used in :logic mode functions. ~l[wormhole]

  It is now possible to provide ``computed hints.''  For example, have
  you ever wished to say ``in all goals with a name like this, :use
  that'' or ``if this term is in the subgoal, then :use that''?  Well,
  ~pl[computed-hints] and the extraordinarily long example in
  ~pl[using-computed-hints].

  ~c[Hide] terms may be rewritten with :rewrite rules about ~c[hide].
  ~l[hide], where we also now explain why ~c[hide] terms are sometimes
  introduced into your proof attempts.~/

  A bug that sometimes caused the ``non-lazy IF'' hard error message
  was fixed.

  A bug that sometimes caused a hard error in forward chaining was
  fixed.

  A bug in print-rules (:pr) was fixed.

  We report the use of :executable-counterparts in the evaluation of
  SYNTAXP forms.

  Some documentation errors were fixed.

  A bug in parent-tree tracking in add-literal-and-pt was fixed.

  A bug in ok$, go$ and eval$ was fixed.

  Clausify now optimizes (mv-nth 'k (list x0 ... xk ... xn)) to xk.

  ~/

  ")

(deflabel note-2-0
  :doc
  ":Doc-Section release-notes

  ACL2 Version 2.0 (July, 1997) Notes~/

  This is the first version of ACL2 released under the copyright of
  the University of Texas (UT).  Future releases of ACL2 will be made
  from UT rather than Computational Logic, Inc. (CLI).  Version 2.0 is just
  Version 1.9 as released by CLI, with a few bugs fixed.

  A bug causing an infinite loop was fixed in functional instantiation.
  The bug manifested itself when two conditions occurred simultaneously:
  First, the functional substitution replaces a function symbol, e.g., ~c[FOO],
  with a ~c[LAMBDA] expression containing a free variable (a variable not among
  in the ~c[LAMBDA] formals).  And, second, in one of the constraints being
  instantiated there is a call of the function symbol ~c[FOO] within the scope
  of another ~c[LAMBDA] expression.  Unless you used such a functional
  substitution, this bug fix will not affect you.

  ~/
  Less important notes:

  The implementation of ~c[PRINC$] was changed so that it was no longer
  sensitive to the external setting of ~c[*print-base*] and other Common Lisp
  special variables.

  Typographical errors were fixed in the documentation.

  ~/

  ")

(deflabel note-2-1
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.1 (December, 1997) Notes~/

  The identity function ~ilc[case-split] has been added.  It is similar
  to ~ilc[force] but causes an immediate split of the top-level goal on
  whether the indicated hypothesis is true.

  ~/
  Less important notes:

  Minor bugs in the documentation were fixed.
  ~/

  ")

(deflabel note-2-2
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.2 (August, 1998) Notes~/

  Important changes:

  A bug was fixed in the compile command, ~c[:comp].  The compiled code
  produced by ~c[:comp] in previous versions could be wildly incorrect
  because of a confusion between the printer and the reader regarding
  what was the current Lisp ~c[*package*].  This bug could manifest itself
  only if you used the ~c[:comp] command to compile previously uncompiled
  functions while the current package was different from ~c[\"ACL2\"].
  What happened in that situation depended upon what symbols were
  imported into your current package.  The most likely behavior is
  that the compiler would break or complain or the resulting compiled
  code would call functions that did not exist.

  There have been no other important changes to the code.

  However, this release contains some useful new books, notably those on
  the ~c[books] subdirectories ~c[cli-misc] and ~c[ihs].  Both have
  ~c[README] files.  The ~c[ihs] books provide support for integer
  hardware specifications.  These books were crucial to Bishop Brock's
  successful modeling of the Motorola CAP.  We thank Bishop for producing
  them and we thank all those who worked so hard to get these books released.
  We highly recommend the ~c[ihs] books to those modeling ALUs and other
  arithmetic components of microprocessors or programming languages.

  In previous versions of ACL2, the arithmetic books, found on
  ~c[books/arithmetic/], included the addition of several unproved axioms
  stating properties of the rationals that we believed could be derived from
  our ``official'' axioms but which we had not mechanically proved.  The axioms
  were found in the book ~c[rationals-with-axioms.lisp],
  which was then used in the uppermost arithmetic books ~c[top.lisp] and
  ~c[top-with-meta.lisp].  John Cowles has now provided us with ACL2 proofs
  of those ``axioms'' and so in this release you will find both
  ~c[rationals-with-axioms.lisp] and ~c[rationals-with-axioms-proved.lisp].
  The former is provided for compatibility's sake.  The latter is identical
  but contains ~c[defthm]s where the former contains ~c[defaxiom]s.
  The top-most books have been rebuilt using ``~c[-axioms-proved]'' book.
  Thanks John.

  ~/
  Less important notes:

  Bishop Brock found a bug in ~c[translated-acl2-unwind-protectp4].
  Jun Sawada reported a bug in linear arithmetic that caused us not to
  prove certain trivial theorems concluding with ~c[(not (equal i j))].
  We have fixed both.

  We now prohibit definitions that call certain event commands
  such as ~c[DEFTHM] and ~c[TABLE] because our Common Lisp implementations
  of them differ from their ACL2 meanings (so that compiled books
  can be loaded correctly and efficiently).

  Minor bugs in the documentation were fixed.
  ~/

  ")

(deflabel note-2-3
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.3 (October, 1998) Notes~/

  Important changes:

  Versions of ACL2 preceding this one contain a subtle soundness bug!
  We found a flaw in our detection of ~il[subversive-recursions].  The
  bug allowed some subversive recursions to slip through undetected.

  We believe it would have been difficult to have exploited this flaw
  inadvertently.  In particular, the following five conditions are
  necessary.
  ~nl[]~nl[]
  (1) Introduce a constrained function, say ~c[f], via an ~c[encapsulate].
  ~nl[]~nl[]
  (2) In the same encapsulation, define a clique of mutually
  recursive functions.  This clique must be non-~c[local] and in
  ~c[:logic] mode.
  ~nl[]~nl[]
  (3) In that mutually recursive clique, use the constrained function
  ~c[f] (perhaps indirectly) so that the termination argument for the
  clique depends on properties of the ~i[witness] for ~c[f].  Thus,
  ~c[f] or some other function dependent upon ~c[f], must be used in
  an argument in a recursive call or in a term governing a recursive
  call.  Furthermore, the use of ~c[f] must be such that the
  termination proof cannot be done without exploiting properties of
  the witness for ~c[f].  Other uses of the constrained functions in
  the clique are ok.
  ~nl[]~nl[]
  (4) Fail to include the exploited properties of ~c[f] among the
  constraints of the encapsulation.
  ~nl[]~nl[]
  (5) Later, outside the encapsulation, explicitly use a functional
  instantiation in which ~c[f] is replaced by a function not enjoying
  the crucial properties.
  ~nl[]~nl[]
  See ~il[subversive-recursions] for details.

  ~/

  Less important notes:

  We have begun to write some introductory tutorial material for those
  who wish to learn to program in ACL2.  Most of this material is
  HTML-based.  See the Hyper-Card on the ACL2 home page.

  The documentation of ~ilc[verify-guards] was improved to explain why
  one might wish to verify the ``guards'' of a ~c[defthm] event.  The
  missing documentation was noticed by John Cowles.

  A bug was fixed in cross fertilization.  The bug caused the system to report
  that it had substituted one term for another when in fact no substitution
  occurred.  The bug was noticed by Bill McCune.

  ~/
  ")

(deflabel note-2-4
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.4 (August, 1999) Notes~/

  Important changes:

  We corrected a soundness bug in Version 2.3 related to the handling of
  ~ilc[immediate-force-modep].  The bad behavior was noticed by Robert
  Krug.  Thanks!

  We corrected a bug that permitted ~ilc[verify-guards] to accept a function
  even though a subfunction had not yet had its guards verified.  Thanks to
  John Cowles for noticing this.

  User defined single-threaded objects are now supported.  See ~il[stobj].
  ~/
  Less important notes:

  We corrected a bug that prevented the intended expansion of some recursive
  function calls.

  We changed the handling of the primitive function ~ilc[ILLEGAL], which
  is logically defined to be ~c[nil] but which is programmed to signal an
  error, so that when it is evaluated as part of a proof, it does not signal
  an error.  The old handling of the function prevented some guard proofs
  involving ~ilc[THE] or ~ilc[LET]s with internal declarations.

  We corrected a bug that permitted some ~c[LOCAL] ~c[DEFAXIOM] events to slip
  into certified books.

  We corrected a bug that prevented the correct undoing of certain ~c[DEFPKG]
  forms.

  Changes were made to support CMU Lisp.  Pete Manolios helped with these
  changes.

  Changes were made to make the make files more compatible with Allegro
  Common Lisp.  Jun Sawada, who has been a great help with keeping ACL2
  up and running at UT on various platforms, was especially helpful.
  Thanks Jun.

  ~/
  ")

(deflabel note-2-5
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.5 (June, 2000) Notes~/

  Important Changes:

  Concurrent with the release of ACL2 Version  2.5 is the publication
  of two books about ACL2.  See the ``Books and Papers about ACL2 and Its
  Applications'' on the ACL2 Home Page.

  The ~c[books] subdirectory now contains many new certifiable books,
  including solutions to the exercises in the two published books and
  full scripts for the case studies.  See ~c[books/README.html].

  Improved Unix ~c[Makefile] support for book certification has also been
  written.  See ~c[books/README.html].

  The list of symbols in ~c[*acl2-exports*] has been considerably expanded.
  If you have packages built by importing ~c[*acl2-exports*] you might want
  to look carefully at the new value of that constant.  The new value includes
  all ~c[:logic] mode functions as of Version  2.5, as well as all documented
  macros and all built-in theorem names.

  ~ilc[Include-book] and ~ilc[certify-book] were modified to
  have some additional keyword arguments.  It is possible to
  certify a book containing ~ilc[defaxiom] and/or ~ilc[skip-proofs]
  events and get warning messages or errors signaled, according to
  the settings of these new flags.  In addition, it is possible to
  specify in ~c[include-book] whether the book must be certified
  (under penalty of error if not).  The default values of these new
  arguments cause warnings to be printed rather than errors signaled.

  The above change involved altering the form of certificate files.
  When books certified under previous versions are included, more
  warnings will be generated because these books are considered
  possibly to contain ~c[defaxiom] and/or ~c[skip-proofs] events.

  We anticipate further changes to this aspect of books and consider
  the current mechanisms (for controlling whether warnings or errors
  are signaled) just a prototype.  See also the discussion below of
  ``soundness related'' warnings.  Your suggestions are welcome.

  A discrepancy between ACL2 and Common Lisp was fixed, having to do
  with ~c[declare ignore].  In past versions of ACL2, a formal
  parameter of a ~c[defun] was considered ignored if it was not used
  in the body, the guard or the measure of the ~c[defun].  That meant
  that a variable used only in the guard could not be declared ignored
  in ACL2; but some Common Lisp compilers would complain because the
  variable was not used in the body.  Now, ACL2 considers a variable
  ignored if it is not used in the body.

  ACL2 can now be built in releases 5.0 and later of Allegro Common
  Lisp.  (Other releases of Allegro Common Lisp and of other lisps
  continue to be supported as well.)  This includes Allegro Common
  Lisp running on Windows 98 platforms.  John Cowles helped us do
  some testing and answered questions for us.  Thanks John!

  We incorporated Ruben Gamboa's changes to allow the building of a
  variant, ACL2(r), of ACL2, in which the user can reason about the real
  numbers using non-standard analysis.  ~l[real].  Note that ACL2(r)
  and ACL2 have different underlying theories, and books certified in
  one system may not be included in the other.  For backward
  compatibility and to ensure a smooth transition, ACL2 is built by
  default, not ACL2(r).  This is a compile-time switch; see the
  makefile for instructions.  There should be no changes to ACL2
  resulting from the capability of building ACL2(r) from the same
  sources.  Also ~pl[acknowledgments] for more on the history of
  ACL2(r).

  A large number of bugs (some affecting soundness) were fixed, and
  many small new features were added.  See below.

  ~/
  Less Important Changes:

  Some warnings are now considered ``soundness related,'' namely,
  those that advise you that an uncertified book has been included
  or that a book containing ~c[DEFAXIOM] or ~c[SKIP-PROOFS] events.
  (Technically, ~c[DEFAXIOM]s do not imperil soundness in the proof-
  theoretic sense, though they may imperil the validity of theorems.
  But you sould know when a book has added an axiom to your logic!)  In
  previous versions of ACL2, all warnings were inhibited if the token
  ~c[warning] was included in the argument to
  ~ilc[set-inhibit-output-lst].  Now, soundness related warnings are
  printed even if ~c[warning]s have been inhibited.  To inhibit all
  warnings, supply the token ~c[warning!] to ~c[set-inhibit-output-lst].

  Several bugs in ~ilc[defstobj] were fixed, relating to the
  possibility that some of the subfunctions introduced by the
  ~c[defstobj] were already defined.

  ~c[:]~ilc[Puff] no longer tries to expand ~ilc[defstobj] events.
  Previously, the attempt would cause a hard error.

  A soundness bug was fixed.  The bug might have been exercised if you
  had an alternative definition (implies hyps (equiv (fn ...) body)) in
  which equiv is an equivalence relation other than EQUAL.  In this case,
  calls of fn might have been expanded to body in places that were not
  equiv-hittable.

  An obscure soundness bug was fixed.  The bug was exercised only if
  you had a metafunction with a computed hypothesis (i.e., a ``meta
  hypothesis function''), the hypothesis contained a free variable,
  i.e., a variable not involved in the term being rewritten, and the
  free variable occurred in the output of the metafunction.  The
  possibility of this bug was brought to our attention by Robert Krug.

  We fixed a bug in the handling of ~c[hide] related to the question
  of whether a variable symbol occurs in a term.  The old code did not
  find the variable and could cause the system to throw away a
  hypothesis about it on the grounds that it was never mentioned.  Rob
  Sumners helped discover this problem.

  The handling of ~c[:]~ilc[elim] rules was generalized, permitting arbitrary
  known equivalence relations instead of merely ~c[equal] in the
  concluding equality.

  The printing of runes (rule names; ~pl[rune]) used has been made
  \"deterministic,\" both in proof output and in proof attempt
  summaries, by sorting the runes before printing.

  The handling of free variables has been improved for hypotheses such
  as ~c[(< 0 X)], and more generally, any hypotheses involving a
  comparison with ~c[0] (even for example ~c[(< X 1)] where ~c[X] is known to be
  an integer, which is handled as ~c[(<= X 0)]).  Thanks to Robert Krug
  for bringing relevant examples to our attention.

  A new value, ~c[:comp], has been implemented for the
  ~c[:load-compiled-file] keyword of ~ilc[include-book].  If this
  value is supplied, then a compiled file will always be loaded, even
  if that requires creating the compiled file first.

  The event ~c[include-book] now generates a warning when a compiled
  file is expected but not found (~pl[include-book]).  Formerly,
  it only did so when executed at the top level; it failed to generate
  the warning when executed on behalf of a surrounding ~c[include-book]
  command.

  Certain redefinition warnings generated by Allegro Common Lisp have
  been eliminated.

  A new key has been implemented for the ~ilc[acl2-defaults-table],
  ~c[:bogus-mutual-recursion-ok], set with ~c[:]~ilc[set-bogus-mutual-recursion-ok].
  Thanks to David Russinoff for pointing out the utility of such a key.

  A bug was fixed in ~ilc[defun-sk] that prevented its generated events from
  being accepted when guard verification is being performed.  Thanks
  to Bill Young for bringing this problem to our attention.  A second
  bug was brought to our attention by Pete Manolios, which was causing
  certain ~ilc[defun-sk] events to be rejected.  That problem has been
  fixed, and an \"Infected\" warning has also been eliminated.

  The command ~ilc[good-bye] now works with Allegro Common Lisp.

  A low-level bug was fixed that could, for example, cause an error
  such as \"Error: Expected 5 args but received 4 args\" when
  interrupting a ~c[local] event.

  A bug has been fixed in the ~il[proof-checker] related to definition
  expansion.  Thanks to Pete Manolios for bringing this to our attention with a
  simple example.

  A bug has been fixed related to the ~c[:]~il[bdd] hint in the presence of
  ~il[equivalence] relations.  Thanks to Pete Manolios for bringing this to our
  attention with a simple example.

  The functions ~ilc[position] and ~ilc[position-equal] formerly
  required the second argument to be a true list.  In accordance with
  Common Lisp, we now also allow the second argument to be a string.
  This could cause earlier proofs about these functions to fail unless
  ~ilc[true-listp] is known to hold where necessary.

  Robert Krug wrote a patch, which has been incorporated, to prevent
  certain infinite loops that can arise in linear arithmetic.  Thanks,
  Robert!

  The macro ~ilc[let*] no longer requires the bound variables to be
  distinct.

  An obscure bug was fixed related to congruence rules.  The bug would
  sometimes cause ACL2 to behave as though no rules (other than equality)
  were available for some argument positions.  Thanks to Pete Manolios for
  bringing this bug to our attention.

  Documentation topics have been added for ~ilc[hard-error] and ~ilc[prog2$],
  and the documentation for ~ilc[illegal] has been improved.  Thanks to Rob
  Sumners for a useful suggestion in the examples in documentation for
  ~c[prog2$] and a fix in documentation for ~ilc[sublis].

  The event form ~ilc[certify-book] was made more secure, in that it can now
  catch attempts to write a book to disk during its certification.
  Thanks to Rob Sumners for pointing out the insecurity of the
  existing mechanism.

  A Y2K problem was fixed with our applicative handling of dates.

  Accessors and updaters for ~ilc[stobj]s have been made more efficient when
  the underlying lisp is Allegro Common Lisp, by the use of
  appropriate simple array declarations.

  A raw Lisp break had been possible when a certified book that had no
  guard verification was included in a session after
  ~c[(]~ilc[set-verify-guards-eagerness]~c[ 2)].  This has been fixed.

  The keyword command ~c[:]~ilc[comp] can now be used to compile only raw
  Lisp functions, excluding executable counterparts, by supplying the
  argument ~c[:raw].

  Rewrite rule ~c[nth-of-character-listp] was removed from source file
  axioms.lisp since it is essentially subsumed by ~c[characterp-nth].

  Printing has been sped up.  In one example the improvement was over
  50% in both Allegro and GCL.

  We now allow printing in a \"downcase\" mode, where symbols are
  printed in lower case.  All printing functions except ~c[print-object$]
  now print characters in lower case for a symbol when the ACL2 state
  global variable ~c[print-case] has value ~c[:downcase] and vertical bars are
  not necessary for printing that symbol.  ~l[IO] for a discussion of
  the macros ~c[acl2-print-case] and ~c[set-acl2-print-case].  The default
  printing remains unchanged, i.e., symbols are printed in upper case
  when vertical bars are not required.

  A low-level printing function (~c[prin1$]) was modified so that it is
  not sensitive to various Common Lisp globals related to printing.  So
  for example, the function ~ilc[fmt] is no longer sensitive to the value
  of Common Lisp global ~c[*print-case*].  (The preceding paragraph
  explains how to control the case for printing in ACL2.)

  The definition of ~ilc[array1p] was fixed so that the ~c[:maximum-length] of
  an array must be strictly greater than the number specified in the
  ~c[:dimensions] field; they may no longer be equal.  This was always the
  intention; the documentation (~pl[arrays]) has remained unchanged.
  The corresponding change was also made to ~ilc[array2p].  Allegro Common
  Lisp formerly caused an error when ~ilc[compress1] was called on an array
  where the numbers above were equal; now, we get a guard violation
  instead, which is appropriate.

  In the context of theories, a name now represents not just the
  corresponding ~c[:definition] ~il[rune], as it has done in earlier versions
  of ACL2, but also the corresponding ~c[:]~ilc[induction] rune.
  ~l[theories] for a discussion of runic designators.  Most users
  will rarely, if ever, notice this change.  One situation where this
  change will make a difference is after executing
  ~c[(in-theory (current-theory 'foo))] followed by
  ~c[(in-theory (enable bar))], where function ~c[bar] is introduced after
  event ~c[foo], and ~c[bar] is recursively defined.  The latter ~ilc[in-theory]
  form now enables the rune ~c[(:induction bar)], which implies that the
  prover can use the induction scheme stored at definition time for
  ~c[bar].  Formerly, the rune ~c[(:induction bar)] was not enabled by
  ~c[(in-theory (enable bar))], and hence the induction scheme for ~c[bar] was
  ignored even when explicit ~c[:induct] hints were supplied.

  You may now supply ~ilc[xargs] keyword pair ~c[:normalize nil] in order to
  prevent certain definitions from ``hanging'' when there are many
  ~c[if]-subexpressions.  ~pl[defun].

  We now translate type declarations of ~c[real] into guards, as we have
  already done for other types such as ~c[rational].  For example,
  ~c[(declare (type real x))] generates the ~il[guard] ~c[(rationalp x)].
  ~l[type-spec].

  The theorem prover now behaves reasonably under the combination of
  specifying a value of ~c[t] both for ~c[:]~ilc[otf-flg] and for a hint
  ~c[:do-not-induct].  Previously, it aborted the first time it would have
  otherwise pushed a goal for induction, but now, it will continue and
  wait until all induction subgoals have been pushed before it aborts.

  We changed slightly the definition of ~ilc[round].  However, we believe
  that the new definition is equivalent to the old.

  The definition of Common Lisp function ~ilc[substitute] has been added.

  The following changes have been made in the use of file names within
  ACL2.  We thank Warren Hunt and John Cowles for running some tests
  of these changes on Macintosh and Windows 98 platforms
  (respectively).~bq[]

  (1) Names of directories and files now use a syntax like that used
  for Unix (trademark of AT&T), where directories are separated using
  the ``~c[/]'' character even when the operating system is not Unix or
  Linux.  ~l[pathname].  ACL2 also continues to support its notion
  of ~em[structured pathnames] from Version  2.4 and before, but might not
  do so in future releases and hence no longer documents such syntax.

  (2) The command ~c[:]~ilc[set-cbd] may now take a relative pathname
  as an argument.

  (3) When the macro ~ilc[ld] is given a file name as a value for
  ~ilc[standard-oi], then if that file name is a relative pathname
  it refers to the result of prepending the connected book directory
  (~pl[pathname], ~pl[cbd], and ~pl[set-cbd]) in order to
  obtain an absolute pathname.  Simiarly for the ~c[ld] specials
  ~ilc[standard-co] and ~ilc[proofs-co].

  ~eq[]It is no longer necessary to issue ~c[:]~ilc[set-state-ok]~c[ t] if you
  include a ~il[stobj] declaration for ~ilc[state], for example:
  ~bv[]
  (declare (xargs :stobjs state))
  ~ev[]
  ~l[declare-stobjs].

  The ~il[proof-checker] has been cleaned up a bit, including the
  documentation and the capability (once again) to define pc-macro
  commands (~pl[define-pc-macro]) and proof-checker meta commands
  (~pl[define-pc-meta]).

  Recall that events generate summaries that include a line beginning
  with ``~c[Warnings:]'', which is followed (on the same line) by zero or
  more brief strings that summarize the warnings generated by that
  event.  Formerly, this warnings summary for an ~ilc[encapsulate] or
  ~ilc[include-book] event did not include the summary strings for
  warnings generated by subsidiary events.  This has been fixed.

  Macro ~ilc[cw] has been documented and now expands to a call of
  a ~c[;]~ilc[logic] mode function.  ~l[cw] for a way to print to the screen
  without having to involve the ACL2 ~ilc[state].  Thanks to Rob Sumners
  for suggesting that we document this useful utility.

  Functions ~c[duplicates], ~c[add-to-set-equal], ~c[intersection-eq], ~c[evens], and
  ~c[odds] are now ~c[:]~ilc[logic] mode functions.

  ~/
  ")

; Do not make note-2-5(r) below conditional on #+:non-standard-analysis,
; because we want to make just one version of the documentation.

(deflabel |NOTE-2-5(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.5(r) (June, 2000) Notes~/

  Important changes to non-standard version:

  ~/

  Please ~pl[note-2-5] for changes to Version  2.5 of ACL2.  We
  hope to write more documentation for ACL2(r) in the future.

  ~/
  ")

(deflabel note-2-6
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.6 (November, 2001) Notes~/

  Because of the large number of modifications, we have divided up
  the Version  2.6 notes into the following subtopics.~bq[]

  o New functionality (~pl[note-2-6-new-functionality]):~nl[]
  o Changes in proof engine (~pl[note-2-6-proofs]):~nl[]
  o Changes in rules and definitions (~pl[note-2-6-rules]):~nl[]
  o Guard-related changes (~pl[note-2-6-guards]):~nl[]
  o Proof-checker changes (~pl[note-2-6-proof-checker]):~nl[]
  o System-level changes (~pl[note-2-6-system]):~nl[]
  o Other (minor) changes (~pl[note-2-6-other]):~nl[]

  ~eq[]~/~/")

(deflabel note-2-6-new-functionality
  :doc
  ":Doc-Section note-2-6

  ACL2 Version  2.6 Notes on New Functionality~/

  A fundamental change is the provision of the ``nu-rewriter'' for
  simplifying expressions composed of ~c[NTH], ~c[UPDATE-NTH], and
  ~c[UPDATE-NTH-ARRAY] applications and ~c[LET] expressions and other
  calls of non-recursive functions or ~c[LAMBDA] expressions involving
  those symbols.  The nu-rewriter applies the obvious rewrite rule for
  ~c[(NTH i (UPDATE-NTH j v s))] and the analogous rule for
  ~c[UPDATE-NTH-ARRAY].  ~l[nu-rewriter]  The nu-rewriter can be
  enabled with ~ilc[set-nu-rewriter-mode].

  A new flag has been added to the ~c[xargs] of ~ilc[defun] permitting
  the declaration that the function is ~c[non-executable].  The
  usage is ~c[(declare (xargs :non-executable t))] and the effect is that
  the function has no executable counterpart.  On the positive side:  the
  function is permitted to use single-threaded object names and functions
  arbitrarily, as in theorems rather than as in executable definitions.
  Such functions are not permitted to declare any names ~c[:]~ilc[stobj]~c[s] but
  accessors, etc., may be used, just as in theorems.

  A new flag has been added to permit the system to abbreviate output
  by introducing ~c[LET*] notation identifying common subterms.  The
  formula being proved is not affected; this flag changes its
  displayed form only.  See ~il[set-let*-abstractionp].

  A ``raw mode'' has been added, primarily for faster loading of
  applications.  ~pl[set-raw-mode].

  Functions ~ilc[alphorder] and ~ilc[lexorder] have been put in ~c[:]~ilc[logic] mode.
  ~c[Lexorder] is now a total order ordering of the ACL2 universe, and
  theorems are included to that effect.  Thanks to Pete Manolios for
  suggesting the idea and providing events to use, and to Rob Sumners
  for assistance with some modifications.  See also the new book
  ~c[books/misc/total-order] for an irreflexive total order.

  The ACL2 user can now make system calls to the host operating system.
  ~l[sys-call] and ~pl[sys-call-status].  Thanks to Rob Sumners
  for working out this idea with Pete Manolios and Robert Krug, who we
  also thank, and for working out the implementation with us.

  It is no longer required to use absolute ~il[pathname]s in ~ilc[include-book]
  forms that have been executed before a ~ilc[certify-book].  Any relative
  pathname strings in such contexts will be expanded into absolute
  pathnames before they are saved in the ~ilc[portcullis] of the ~ilc[certificate]
  of the book being certified.

  ACL2 can now be built on top of Allegro Common Lisp 6.0, and also on
  Windows platforms on top of Allegro Common Lisp and GCL.  Thanks to Pete
  Manolios and Vinay K. Siddhavanahalli for their help with Windows.

  Rob Sumners has designed and provided an initial implementation for two
  improvements to ~ilc[defstobj] (also ~pl[stobj]).  First, array fields can
  now be resized.  Resize and length functions are provided for array fields,
  which can be used to resize stobj array fields dynamically.  The recognizers
  for array fields have been simplified to accommodate this change, so that
  they only check that each element of the array field has the specified type.
  Second, performance has been improved for stobjs with a large number of
  fields, by changing their Common Lisp implementation to store the fields in a
  simple vector instead of a list.

  Now ~il[stobj]s may be bound locally; ~pl[with-local-stobj].
  Thanks to Rob Sumners, who encouraged us to implement this
  capability, was an early user of it, and participated usefully in
  discussions on its design.

  New functions ~ilc[fms!], ~ilc[fmt!], and ~ilc[fmt1!] are the same as their respective
  functions without the ``~c[!],'' except that the ``~c[!]'' functions are
  guaranteed to print forms that can be read back in (at a slight
  readability cost).

  We added ~ilc[extended-metafunctions], metafunctions which
  allow ~ilc[state] and context sensitive rewriting to some
  extent.  We thank Robert Krug for pushing for and on this idea.

  The documentation has been improved.  In particular, a new
  documentation topic provides a gentle introduction to ACL2
  ~ilc[arrays] ~-[] ~pl[arrays-example] ~-[] and additional
  documentation has been provided for getting started with proof trees
  in emacs ~-[] ~pl[proof-tree].

  New Makefile targets ~c[fasl] and ~c[o] have been added to the ~c[books/]
  directory of the distribution.  For example, you might first certify
  books using an ACL2 built on top of GCL (which creates compiled
  files with suffix ~c[o]).  Then, when standing in the ~c[books/]
  directory, you might execute the command~bq[]

  make fasl ACL2=my-allegro-acl2

  ~eq[]which will create compiled (~c[.fasl]) files for Allegro Common
  Lisp, assuming that ~c[my-allegro-acl2] starts up an ACL2 built on
  that Common Lisp.

  The macro ~ilc[let*] now allows variables to be declared ignored.
  ~l[let*] and ~pl[let].

  The user may now control backchaining.  This feature was designed and
  primarily implemented by Robert Krug (though the authors
  of ACL2 are resposible for any errors); thanks, Robert!
  ~l[backchain-limit].

  It is now possible to ``slow down'' the rate at which case splits are
  generated by the simplifier.  ~l[set-case-split-limitations].

  Accesses to ~il[stobj]s using ~ilc[nth] or ~ilc[update-nth] are now
  displayed using symbolic constants instead of numeric indices.  For
  example, given the event
  ~bv[]
  (defstobj foo a b :renaming ((b c)))
  ~ev[]
  then the term ~c[(nth 0 foo)] will be displayed (for example, during
  proofs) as ~c[(nth *a* foo)] while ~c[(nth 1 foo)] will be displayed
  as ~c[(nth *c* foo)].  The ~ilc[defstobj] event now correspondingly
  introduces a ~ilc[defconst] event for each field accessor function,
  introducing a constant whose name is obtained from the accessor's
  name by prefixing and suffixin a ``~c[*],'' as in the example above:
  accessor ~c[a] generates ~c[(defconst *a* 0)] and accessor ~c[c]
  generates ~c[(defconst *c* 1)]. ~l[nth-aliases-table] for how to
  extend this feature for alternate names of ~il[stobj]s.

  Computed hints have been improved.  It is now possible to detect
  within a computed hint whether the goal clause is stable under
  simplification; it is also possible for a computed hint to change
  the list of available hints.  ~l[computed-hints].

  It is now possible to provide ``default hints'' that are appended
  to the hints explicitly provided.  ~l[set-default-hints].

  Using computed hints (~pl[computed-hints]) and default hints
  (~pl[set-default-hints]) it is possible to implement a book that
  supports ``priority phased simplification.''  Using this book
  you can assign priorities to your rules and cause the theorem
  prover to simplify each goal maximally under all the rules of
  one priority before enabling rules of the next priority.
  See ~c[books/misc/priorities.lisp].

  The macro ~ilc[defabbrev] has been improved to allow ~ilc[declare] forms and
  documentation strings and to do more error-checking.  Thanks to Rob Sumners
  for designing this enhancement and providing the first implementation.
  ~l[defabbrev].

  Further changes were made to support CMU Lisp.  Wolfhard Buss helped
  with these changes.

  A new table was added that is used when printing proof output, so
  that nests of right-associated calls of a binary function are
  replaced by corresponding macro calls, as has been the case for
  ~ilc[binary-+] and ~ilc[+], ~ilc[binary-append] and ~ilc[append], and so on.
  ~l[add-binop].

  Operators ~ilc[logand], ~ilc[logior], ~ilc[logxor], and ~ilc[logeqv] are now
  macros (formerly, they were functions) that call corresponding
  binary functions (e.g., ~c[binary-logand]) defined in source file
  ~c[\"axioms.lisp\"].  Thanks to Rob Sumners for this enhancement.  Proof
  output will however continue to show calls of ~ilc[logand], ~ilc[logior],
  ~ilc[logxor], and ~ilc[logeqv].

  Function ~c[(]~ilc[allocate-fixnum-range]~c[ fixnum-lo fixnum-hi)] sets aside more
  \"permanent\" fixnums in GCL.

  ACL2 now runs under ~c[CLISP].  Thanks to Wolfhard Buss and Sam
  Steingold for their assistance with the port.

  Michael ``Bogo'' Bogomolny has created a search engine, accessible
  from the ACL2 home page.  For that purpose he modified the HTML
  translator to create one file per topic (a good idea in any case).
  Thanks, Bogo!

  An emacs file of potential (but optional) use for ACL2 users may be
  found in ~c[emacs/emacs-acl2.el].  In particular, this file supports
  the use of proof trees (~pl[proof-tree]).

  Some ~il[books] have been added or modified.  In particular, Robert Krug has
  contributed ~c[books/arithmetic-2/], which provides an alternative to the
  existing collection of books about arithmetic, ~c[books/arithmetic/].  For a
  discussion of the distributed books see the link to ~c[README.html] in the
  installation instructions.

  ~/~/")

(deflabel note-2-6-proofs
  :doc
  ":Doc-Section note-2-6

  ACL2 Version  2.6 Notes on Changes in Proof Engine~/

  Certain optimizations are performed when converting terms to clausal
  form.  For example, ~c[(< 0 1)] is known to be ~c[t],
  ~c[(HARD-ERROR ctx str alist)] is known to be ~c[nil], and
  ~c[(INTEGERP n)] is known to imply ~c[(RATIONALP n)].

  In earlier versions of ACL2, the conversion of a term to clausal
  form expanded ~c[LAMBDA] applications.  That may no longer occur.
  Some proofs may slow down (or fail) because your
  ~c[LAMBDA]-expressions are not expanded away when you ``expected''
  them to be.

  Robert Krug found a soundness bug in our linear arithmetic package.
  The bug was caused by the derivation of an equation from two
  inequalities without taking adequate precautions to ensure that both
  sides of the inequalities were numeric.  Robert also kindly provided
  a fix which we adopted.  Thanks Robert!

  We fixed a bug that could prevent the application of a metatheorem.

  A bug has been fixed that had caused bogus forcing rounds
  (~pl[forcing-round]).  The bug could occur when the hypothesis of
  a rule was forced (~pl[force]) before the prover decided to start
  over and prove the original goal by induction.  Thanks to Rob
  Sumners for drawing our attention to this problem.

  Some low-level fixes have been made that prevent certain infinite
  loops, based on reports by users.  We thank Yunja Choi, Matt
  Wilding, and Pete Manolios for reporting such problems.

  An obscure potential soundness hole has been fixed by redoing the
  way evaluation takes place in the ACL2 loop and during theorem
  proving.  We expect that users will see no difference based on this
  change.  (Those interested in the details can see the long comment
  ``Essay on Evaluation in ACL2'' in source file interface-raw.lisp.)

  A small change was made in computation for a heuristic that controls
  backchaining.  This will speed up proofs dramatically in a very few
  cases but should have a very small impact in general.

  The simplifier has been modified to avoid eliminating hypotheses of
  goals that can be established by contextual (specifically, type-set)
  reasoning alone.  We believe that this change will generally
  strengthen ACL2's reasoning engine, although on rare occasions a
  lemma that formerly was provable may require user assistance.
  Thanks to Robert Krug for suggesting this change and providing its
  implementation.

  Case splits are now limited, by default.  This may allow some proof
  attempts to provide output where previously the prover would appear
  to ``go out to lunch.''  For a more complete discussion, including
  instructions for how users can control case splitting,
  ~pl[set-case-split-limitations].

  A bug has been fixed in the handling of ~c[:]~ilc[type-prescription] rules by
  the ~il[bdd] package.  Thanks to Rob Sumners for discovering this bug
  and supplying a helpful example.

  ACL2 may now use the built-in induction scheme for a function symbol
  even if that function symbol is disabled.  Formerly, if a function
  symbol was disabled then its induction scheme was only considered if
  an explicit induction hint was supplied, other than ~c[:induct t].

  We eliminated the rule-class ~c[linear-alias].  This rule class was seldom
  used and complicated the linear arithmetic decision procedure in ways that
  made it difficult to extend to handle some non-linear special cases.
  The only use of the rule-class that we know of was in our own ~c[nqthm]
  books, which were an attempt to provide an embedding of the Nqthm logic
  and theorem prover into ACL2.  But that facility was also practically
  never used, as far as we know.  So both ~c[linear-alias] rules and the
  ~c[nqthm] books have been eliminated.

  In earlier versions of ACL2, when the ~c[IF]-form of ~c[(AND p q)] was
  assumed true -- as when rewriting the ~c[alpha] expression in
  ~c[(IF (AND p q) alpha beta)] -- the assumption mechanism did not deduce
  that ~c[p] and ~c[q] are true, only that their conjunction, in its
  ~c[IF]-form, is true.  This has long been known as a deficiency in
  both ACL2 and the earlier Nqthm but it was tedious to do better when
  one considered the full range of ~c[IF]-forms one might encounter in the
  test of another ~c[IF].  Rather than code all the cases, we just waited
  until clausification got rid of them.  Robert Krug developed a pretty
  nice treatment of the general case and we added it in this version.
  This also involved a surprising number of changes elsewhere in the system
  because the improved handling of assumptions caused the theorem prover
  often to ``erase'' hypotheses provided by ~c[:use] hints because it could
  simplify them to ~c[t].  Thank you Robert!

  In response to a suggestion from Robert Krug, we added ~c[mfc-ap] so
  that extended metafunctions can take advantage of linear arithmetic.
  ~l[extended-metafunctions].

  There is less delay in printing goals.  In previous versions, a
  goal was not printed until its subgoals were created (or the goal
  was proved).  Now, the goal is printed essentially as soon as it is
  created.

  A small technical change has been made in the function ~ilc[term-order],
  to give priority on the function symbol count over the weighting of
  constants.  So for example, while previously the term ~c[(f)] preceded
  the constant 2, that is no longer the case.  If this change is noticed
  at all, it will probably be noticed in how so-called ~em[permutative]
  rewrite rules are applied; ~pl[loop-stopper].  Thanks to Robert Krug
  for suggesting this improvement and providing part of the
  implemtation.

  ~/~/")

(deflabel note-2-6-rules
  :doc
  ":Doc-Section note-2-6

  ACL2 Version  2.6 Notes on Changes in Rules and Constants~/

  The following symbols have been added to the list constant
  ~c[*common-lisp-specials-and-constants*]: ~c[REPLACE], ~c[FILL], ~c[CHARACTER],
  ~c[=], ~c[BREAK], and ~c[PRIN1].  This was done in support of ports to
  Allegro 6.0 and Windows platforms (~pl[note-2-6-new-functionality]).

  The list of symbols in ~c[*acl2-exports*] has been modified, for
  example to include ~c[show-accumulated-persistence] and the legal
  arguments to ~ilc[set-inhibit-output-lst].

  Functions ~ilc[zp] and ~ilc[zip] are now handled slightly differently.  They are
  are now disabled, but each comes with a ~c[:]~ilc[rewrite] rule that allows
  their expansion on non-variable terms, and also with a
  ~c[:]~ilc[compound-recognizer] rule that avoids the need for opening up these
  functions when applied to variables.  The resulting behavior should
  be very similar to the behavior of previous versions, except that
  case splits will be avoided when these functions are applied to
  variables.

  Function ~ilc[standard-string-alistp] replaces function
  ~c[string-alistp].  For further discussion, ~pl[note-2-6-guards].

  Rules of class ~c[:]~ilc[rewrite] whose conclusion is a term of the form
  ~c[(equal lhs rhs)] have always been stored in the expected way:  ~c[lhs]
  rewrites to ~c[rhs].  This way of storing ~c[:rewrite] rules has been
  extended to allow ~ilc[=], ~ilc[eq], or ~ilc[eql] in place of ~ilc[equal].

  Rewrite rule ~c[nth-update-nth], in source file ~c[axioms.lisp], has been
  strengthened.

  A new rewrite rule ~c[equal-constant-+] has been added to the book
  ~c[arithmetic/equalities].  This should generally be a beneficial
  change, but existing proofs involving the arithmetic books could
  conceivably be affected.

  Function ~ilc[symbol-package-name] and constant ~c[*main-lisp-package-name*]
  have undergone small changes.  This change should rarely be noticed
  by users and is discussed elsewhere; ~pl[note-2-6-system].

  We mention here that proofs involving ~il[stobj]s may need to be modified
  because of changes in auxiliary functions generated by ~ilc[defstobj].
  (These changes were made in support of a new resizing capability,
  mentioned elsewhere in these release notes;
  ~pl[note-2-6-new-functionality].

  In the distributed book directory ~c[books/arithmetic/], the book
  ~c[rationals-with-axioms-proved.lisp] has been renamed ~c[rationals.lisp].

  (ACL2(r) only) Rewrite rules ~c[realp-+], ~c[realp-*], ~c[realp-unary--], and
  ~c[realp-unary-/] have been added in analogy to existing rules
  ~c[rationalp-+], ~c[rationalp-*], ~c[rationalp-unary--], and ~c[rationalp-unary-/].
  Thanks to Jun Sawada for suggesting this change.

  The definition of ~ilc[aref1] has been modified slightly.  Previously, if
  ~c[*my-a*] were an array then ~c[(aref1 'some-name *my-a* :header)]  would
  evaluate to the ~c[cdr] of the ~ilc[header] of ~c[*my-a*] rather than to its
  ~ilc[default].  ~l[arrays].

  Changes have been made in the ~c[ihs] books, based on suggestions from
  Jun Sawada, that support its use with ACL2(r) (~pl[real]).  The
  primary change is to replace calls of ~ilc[rationalp] with calls of
  ~ilc[real/rationalp], which should have no effect on users of standard
  ACL2.

  ~/~/")

(deflabel note-2-6-guards
  :doc
  ":Doc-Section note-2-6

  ACL2 Version  2.6 Notes on Guard-related Changes~/

  When you ~ilc[declare] that a function treats certain formals
  as ~c[:]~ilc[stobj]~c[s], the ~il[guard] of the function is automatically
  extended to include the corresponding stobj-recognizer calls.  For example,
  if a definition includes ~c[(declare (xargs :stobjs (ST)))] then the
  guard of the function is changed by the addition of the conjunct
  ~c[(ST-P ST)].

  One impact of this is that if you use the built-in ACL2 ~ilc[state]
  as a formal parameter of a function, ~c[(STATE-P STATE)] is added to
  the guard.  This may introduce a guard where there was none in
  previous versions of the system.  In older versions, therefore, no
  attempt would be made to ~ilc[verify-guards], while in the new
  version, we would attempt guard verification.  You may wish to add
  ~c[(declare (xargs :verify-guards nil))] to such definitions.

  A related change affects users who do not use stobjs or ~c[state].
  In previous versions of the system ~-[] as now ~-[] a ~c[type]
  declaration extended the guard you provided explicitly.  Thus, if
  you wrote ~c[(declare (type integer n))] then ~c[(INTEGERP n)] was
  added to your guard.  This is still the case and ~c[:stobjs]
  recognizers are similarly added.  But in older versions of the system
  we ``added'' the conjuncts without checking whether they were already
  present in the guard you provided.  This sometimes produced such
  guards as ~c[(and (integerp n) (integerp n))] where the first was
  produced by your ~c[type] declaration and the second was your
  ~c[:guard].  We now eliminate redundant conjuncts; this may rearrange
  the order of the conjuncts.

  The guard conjectures for functions using ~c[stobj]s have been simplified
  somewhat by taking advantage of the syntactic restrictions checked for
  single-threaded objects.

  The following functions have been modified so that character and
  string arguments are restricted to standard characters.
  (~l[standard-char-p] and ~pl[standard-char-listp].)~bq[]

  ~ilc[upper-case-p]
  ~ilc[lower-case-p]
  ~ilc[char-upcase]
  ~ilc[char-downcase]
  ~c[string-downcase1]
  ~ilc[string-downcase]
  ~c[string-upcase1]
  ~ilc[string-upcase]
  ~ilc[char-equal]
  ~c[string-equal1]
  ~ilc[string-equal]

  ~eq[]Also, function ~ilc[standard-string-alistp] replaces function
  ~c[string-alistp], with concomitant changes in the guard to
  ~ilc[assoc-string-equal], and in variable ~c[*acl2-exports*].
  Also, lemma ~c[standard-string-alistp-forward-to-alistp] replaces
  lemma ~c[string-alistp-forward-to-alistp].  There is a new lemma
  ~c[standard-char-p-nth], which has also been added to ~c[*acl2-exports*].

  The guard had been inadvertently omitted from the definition of the
  function ~ilc[substitute] (and its subroutine ~c[substitute-ac]).  This
  omission has been corrected; also, the guard is slightly stronger
  than the documentation had claimed (and that has been corrected).

  ~/~/")

(deflabel note-2-6-proof-checker
  :doc
  ":Doc-Section note-2-6

  ACL2 Version  2.6 Notes on Proof-checker Changes~/

  The proof-checker command ~c[=], when used with no arguments, now
  reports which hypothesis is being used.

  The output from ~il[proof-checker] command ~c[type-alist] has been
  improved.

  A slight change has been made to the ~il[proof-checker] for commands
  ~c[promote], ~c[casesplit], ~c[equiv], and ~c[=], so that terms of the form
  ~c[(if x nil y)] are recognized as conjunctions, ~c[(and (not x) y)].
  Thanks to Pete Manolios for suggesting that we consider such a change.

  There is a new ~il[proof-checker] command ~c[print-all-concs] that prints
  all the conclusions of the unproved goals.

  A new ~ilc[proof-checker] command, ~c[runes], has been added.  It reports
  the ~il[rune]s that have participated in the interactive proof up to the
  current point.

  ~/~/")

(deflabel note-2-6-system
  :doc
  ":Doc-Section note-2-6

  ACL2 Version  2.6 Notes on System-level Changes~/

  We modified the tracking of ~ilc[skip-proofs] events and the use of
  ~ilc[state] global ~c[ld-skip-proofsp] in order to avoid some soundness
  issues.  For example, ~ilc[skip-proofs] events buried in locally-included
  books are now tracked.  The ``Essay on Skip-proofs'' in source file
  ~c[axioms.lisp] gives several examples of dicey behavior that is no
  longer supported.

  We fixed a problem with some of the makefiles, so that recursive invocations
  of `make' now use the version of `make' specified on the command line.

  Files were fixed to help non-Unix/Linux users with book
  certification.  Thanks to John Cowles for finding some problems
  and suggesting fixes to ~c[books/certify-numbers.lisp],
  ~c[books/arithmetic/certify.lsp], and ~c[books/cowles/certify.lsp].
  We thank Scott Burson for noticing and fixing some other such
  problems.  Moreover, a bdd test was being ignored entirely in
  Version  2.5; this problem has been fixed as well.

  A minor change in system function save-acl2-in-allegro will allow
  this function to continue to work in Allegro CL versions starting
  (someday) with 10.0.  Thanks to Art Flatau for suggesting such a
  fix.

  The ~c[books/case-studies/] directory has been removed.  These books are
  in support of the first (1998) ACL2 workshop, and are accessible via the
  ACL2 home page on the Web,
  ~url[http://www.cs.utexas.edu/users/moore/acl2/].  Also, the
  ~c[books/cli-misc] directory has been renamed ~c[books/misc], and the
  ~c[books/nqthm] directory has been removed.

  The notion of ACL2 version has been slightly modified to catch
  unsoundness due to implementation dependencies.  ~l[version].
  Another change to eliminate such unsoundness is that built-in
  symbols now have a ~ilc[symbol-package-name] of ~c[\"COMMON-LISP\"]; formerly,
  this string was ~c[\"LISP\"] for ACL2 images built on GCL.
  ~l[symbol-package-name].  At a low level, the (undocumented) constant
  ~c[*main-lisp-package-name*] is now ~c[\"COMMON-LISP\"]; before, it was
  ~c[\"LISP\"] for GCL.

  ~/~/")

(deflabel note-2-6-other
  :doc
  ":Doc-Section note-2-6

  ACL2 Version  2.6 Notes on Other (Minor) Changes~/

  Warning strings are now case-insensitive.
  ~l[set-inhibit-warnings].

  ACL2 causes a warning when an ~il[in-theory] hint or event causes a 0-ary
  function's definition to be disabled but its ~c[:]~ilc[executable-counterpart]
  to be enabled.

  A minor modification has been made to ~ilc[defstobj] that can have a
  positive impact on performance in Allegro Common Lisp.  (For Lisp
  hackers:  the stobj name was formerly declared special, and that was
  disabling Allegro's tail-merging routing for compilation of some
  recursive functions using stobjs.)  The downside is that stobj names
  can no longer be evaluated in raw Lisp.  However, raw Lisp is not
  the right place to be evaluating ACL2 forms anyhow;
  ~pl[set-raw-mode].  We thank Rob Sumners for bringing this issue
  to our attention.

  Before Version  2.6, there has been the following problem with
  ~ilc[defstub] and ~ilc[encapsulate] in the case that the current package is not
  the ACL2 package.  If a ~il[signature] was specified using the symbol ~c[=>],
  then that symbol had have been imported into the current package
  from the ACL2 package when the current package was defined.  There
  are no longer any package restrictions on the use of ~c[=>].  Thanks to
  John Cowles for bringing this problem to our attention.

  Bugs in ~ilc[defun-sk] have been fixed.  ~c[Defun-sk] forms introducing
  functions of no arguments were failing to be admitted, for example:
  ~c[(defun-sk always-p1 () (forall (x) (p1 x)))].
  Thanks to John Cowles for bringing this problem to our attention.
  Also, ~c[defun-sk] failed on an example in the documentation
  (~pl[tutorial4-defun-sk-example]), as pointed out by Matyas
  Sustik; this bug has been fixed as well.

  The trace mechanism has been fixed to handle ~il[stobj]s, and to
  avoid the printing of so-called ~em[enabled structures].

  The ~ilc[brr] command ~c[:type-alist] now produces more readable output.

  An ~ilc[include-book] of an uncertified book no longer loads an associated
  compiled file.

  We added a few checks to make sure that the underlying lisp is
  suitable, for example checking that the reader is case-insensitive
  and reads in symbols with upper-case names where appropriate.

  We now warn when forcing (~pl[force]) or immediate force mode
  (~pl[immediate-force-modep]) change state between enabled and
  disabled.  Also ~pl[enable-immediate-force-modep] and
  ~pl[disable-immediate-force-modep] for information about these
  new macros, which may be used to control immediate force mode.

  We have eliminated the use of a low-level raw Lisp constant,
  ~c[*most-recent-multiplicity*].  Our test suite saw a speed-up
  of approximately 2% as a result for an ACL2 image built on GCL
  (but no significant speed-up for an ACL2 image built on Allegro
  Common Lisp).  We thank Rob Sumners for suggesting this improvement.

  Fixnum declarations are now realized as ~c[(signed-byte 29)] instead of
  ~c[(signed-byte 27)].  We check that the underlying Common Lisp
  recognizes objects of type ~c[(signed-byte 29)] as fixnums, with
  the exception of CLISP, which is said to have an efficient bignum
  implementation.

  A new documentation topic ~il[functional-instantiation-example]
  illustrates functional instantiation.

  A bug has been fixed in the monitoring of runes (~pl[monitor]).
  Thanks to Dave Greve for sending an example that clearly showed
  the problem.

  A warning is now issued when it is detected that a
  ~c[:]~ilc[type-prescription] rule may not be as strong as it appears because
  it is not sufficient to prove itself by type reasoning.

  An error is caused for rules of class ~c[:]~ilc[meta] when the function symbol
  ~c[IF] is among the ~c[:trigger-fns].  (~c[IF] was ignored anyhow; the point of
  this change is to avoid misleading the user.)

  A minor bug has been fixed in ~c[:]~ilc[pr], evident for example if this
  command was applied to ~c[IF].

  A minor hole in ~c[:]~ilc[set-bogus-mutual-recursion-ok] did not permit the
  acceptance of ~ilc[mutual-recursion] forms that include constant function
  definitions.  This has been fixed.  Thanks to Eric Smith for coming
  up with a simple example illustrating the problem.

  The temporary files \"TMP.lisp\" and \"TMP1.lisp\" written out by ~c[:]~ilc[comp]
  are now written to the connected book directory (~pl[cbd]).

  Previously, the Allegro compiler was not eliminating tail recursion
  for executable counterparts of functions, because of the way one of
  its flags had been set.  As a result, calls of functions whose
  guards had not been verified could run out of stack space when this
  was not necessary.  This situation has been fixed.

  Executable counterparts could have slow array accesses.  This has
  been fixed (specifically, constants are no longer replaced with
  their values in the definitions of executable counterparts).

  Various improvements have been made to the documentation.  Thanks in
  particular to Eric Smith for pointing out a numbers of places where
  fixes were in order.

  File \"mcl-acl2-startup.lisp\" has been updated, thanks to feedback
  from Philippe Georgelin.

  Inefficiencies in GCL fixnum computations were remedied for macros ~c[+f] and
  ~c[*f].  Thanks to Rob Sumners for pointing out this issue.

  ~/~/")

; Do not make note-2-6(r) below conditional on #+:non-standard-analysis,
; because we want to make just one version of the documentation.

(deflabel |NOTE-2-6(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.6(r) (November, 2001) Notes~/

  Important changes to non-standard version:  None since Version  2.5.

  ~/

  Please ~pl[note-2-6] for changes to Version  2.6 of ACL2.  We
  hope to write more documentation for ACL2(r) in the future.

  ~/
  ")

(deflabel note-2-7
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.7 (November, 2002) Notes~/

  The Version_2.7 notes are divided into the subtopics below.  Here we give
  only a brief summary of a few of the changes that seem most likely to impact
  existing proofs.  Not included in this brief summary, but included in the
  subtopics, are descriptions of improvements (including bug fixes and new
  functionality) that should not get in the way of existing proof efforts.

  In particular, please ~pl[note-2-7-new-functionality] for discussion of a
  number of new features that you may find useful.

  Acknowledgements and elaboration, as well as other changes, can be found in
  the subtopics listed below.

  o Bug fixes (~pl[note-2-7-bug-fixes]):~bq[]

  + Three soundness bugs were fixed.  These bugs were probably rarely hit, so
  users may well not notice these changes.

  + ~ilc[Certify-book] now requires ~c[:skip-proofs-ok t] (respectively,
  ~c[:defaxioms-okp t]) if there are ~ilc[skip-proofs] (respectively,
  ~ilc[defaxiom]) events in the book or any included sub-books.

  + When ~c[:by] hints refer to a definition, they now use the original body of
  that definition rather than the simplfied (``normalized'') body.

  + When ~ilc[ld] is applied to a stringp file name, it now temporarily sets the
  connected book directory (~pl[cbd]) to the directory of that file while
  evaluating forms in that file.~eq[]

  o New functionality (~pl[note-2-7-new-functionality]):~bq[]

  + ACL2 now works harder to apply ~c[:]~ilc[rewrite] and ~c[:]~ilc[linear]
  rules with free variables in the hypotheses.  ~l[note-2-7-new-functionality],
  in particular its first two paragraphs, for details.  ~il[Forward-chaining]
  also does more with free variables.~eq[]

  o Changes in proof engine (~pl[note-2-7-proofs]):~bq[]

  + Some prover heuristics have changed slightly.  Among other consequences,
  this can cause subgoal ~il[hints] to change.  For example, suppose that the
  Version_2.6 proof of a particular theorem generated \"Subgoal 2\" and
  \"Subgoal 1\" while Version_2.7 only generates the second of these.  Then a
  subgoal hint attached to \"Subgoal 1\" in Version_2.6 would have to be
  attached to \"Goal'\" in Version_2.7.  (~l[goal-spec].)  The full topic has
  details (~pl[note-2-7-proofs]).~eq[]

  o Changes in rules and definitions (~pl[note-2-7-rules]):~bq[]

  + The package name of a generated variable has changed for ~ilc[defcong].~eq[]

  o Guard-related changes (~pl[note-2-7-guards]):~bq[]

  + ~ilc[Guard] verification formerly succeeded in a few cases where it should
  have failed.

  + Guards generated from type declarations now use functions
  ~c[signed-byte-p] and ~c[unsigned-byte-p], now defined in source file
  ~c[axioms.lisp] and formerly defined rather similarly under ~c[books/ihs/].~eq[]

  o Proof-checker changes (~pl[note-2-7-proof-checker]):~bq[]

  + See the above doc topic.~eq[]

  o System-level changes (~pl[note-2-7-system]):~bq[]

  + See the above doc topic.~eq[]

  o Other changes (~pl[note-2-7-other]):~bq[]

  + A new ~ilc[table], ~ilc[invisible-fns-table], takes the place of the
  handling of invisible functions in the ~ilc[acl2-defaults-table],

  + The ~ilc[theory-invariant] event has been modified so that the default action
  is an error rather than a warning.

  +  Proof output that reports destructor elimination no longer uses the word
  ``generalizing''.~eq[]

  Again, please proceed to the subtopics for more thorough release notes.

  ~/~/")

(deflabel note-2-7-bug-fixes
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on Bug Fixes~/

  Francisco J. Martin-Mateos emailed us a soundness bug (!) in our handling of
  functional instantiation (for example ~pl[functional-instantiation-example]).
  We are grateful for that email, which clearly illustrated the problem.
  It is included just below the definition of ~c[push-clause] in ACL2 source file
  ~c[prove.lisp], where we have fixed the bug.  This bug was fixed in a
  re-release of Version  2.6 in February, 2002.

  Rob Sumners emailed us a soundness bug (!) in function ~c[commutative-p1],
  which is used by the ACL2 ~il[bdd] package.  We are grateful for his help;
  his email gave a proof of nil and also pointed to the problem function.
  This bug was fixed in a re-release of Version  2.6 in February, 2002.

  We discovered and fixed a soundness bug illustrated by the book below, which
  was certifiable in Version  2.6 and ends in a proof of ~c[nil].  The event
  ~c[(verify-guards foo)] should have been rejected, because ~c[foo] calls a
  function whose guards have not been verified, namely, ~c[bar].  However, ACL2
  did not notice the call of function ~c[bar] in the body of ~c[foo] because it
  was looking in the simplified (normalized) body of ~c[foo] rather than in the
  original body of ~c[foo].  During processing of the book below, the logical
  definition of ~c[zp] is used before ~c[(verify-guards foo)], and ~c[(zp -3)]
  reduces to ~c[t] in the logic.  After ~c[(verify-guards foo)], ACL2
  simplifies ~c[(foo -3)] by going into raw Lisp, where ~c[(zp -3)] is
  evaluated and reduces to ~c[nil].
  ~bv[]
    (in-package \"ACL2\")
    (defun bar (x)
      (zp x))
    (defthm zp-false-on-negatives
      (implies (< x 0)
               (bar x))
      :rule-classes :type-prescription)
    (defun foo (x)
      (declare (xargs :guard (rationalp x)
                      :verify-guards nil))
      (if (< x 0)
          (if (bar x) 0 1) ; simplified body reduces this line to 0
        17))
    (defthm foo-of-minus-3-is-0
      (equal (foo -3) 0)
      :rule-classes nil)
    (verify-guards foo)
    (defthm foo-of-minus-3-is-1
      (equal (foo -3) 1)
      :rule-classes nil)
    (defthm bug
      nil
      :rule-classes nil
      :hints ((\"Goal\" :use (foo-of-minus-3-is-0 foo-of-minus-3-is-1))))
  ~ev[]
  The above bug exploited the fact that ~ilc[zp] has a different definition in
  raw Lisp than in the logic for arguments that violate its guard).  The
  following example caused a hard error in raw Lisp, though not a soundness
  error.
  ~bv[]
    (in-package \"ACL2\")
    (defun bar (x)
      (cons (car x) (car x)))
    (defun foo (x)
      (declare (xargs :guard t
                      :verify-guards nil))
      (if (bar x) x nil))
    (verify-guards foo)
    (defthm bug
      (equal (foo 3) t)
      :rule-classes nil)
  ~ev[]
  We have made a minor change to the notion of the ~em[formula] of a function
  symbol, related to the change above, which however is unlikely to be
  noticeable.

  In order to make it harder to hit problems like the guard problem above, we
  have slighly modified the raw Lisp definition of ~ilc[zp].

  A ~ilc[break-rewrite] command, ~c[:ancestors], was broken, but has been
  fixed.  Thanks to Eric Smith for bringing the problem to our attention, and
  to Robert Krug for supplying the final part of the fix.

  Some ~il[proof-checker] commands caused errors when all goals have already
  been proved.  This has been fixed.  Thanks to Matt Wilding for reporting this
  bug.

  Fixed a bug in ~c[:]~ilc[comp].  When compiling uncompiled functions with
  very large definitions, ACL2 was inserted a backslash (~c[\\]) character into
  generated files.

  Fixed the ~c[:type-alist] ~c[:]~ilc[brr] command (~pl[brr-commands]), whose
  output was difficult to read when typed after an ~c[:eval]..

  Fixed some clumsy handling of errors when including an uncertified book, for
  example, with the error message when including an uncertified book with a bad
  ~ilc[deftheory] event.  Thanks to Eric Smith for pointing out this problem.

  Two modifications to ~ilc[certify-book] now cause it to reflect natural
  expectations with respect to soundness.  First, it now has default values of
  ~c[nil] instead of ~c[t] for keyword arguments ~c[:skip-proofs-okp] and
  ~c[:defaxioms-okp].  Thanks to Robert Krug for suggesting this change and the
  ACL2 seminar at the University of Texas for discussing it.  Second, when
  ~c[:skip-proofs-okp] (respectively, ~c[:defaxioms-okp]) is ~c[nil], either
  explicitly or by default, then ~ilc[skip-proofs] commands (respectively,
  ~ilc[defaxiom] events) are disallowed inside any included books, regardless
  of the keyword parameters passed to ~ilc[include-book].  This had not been
  the case for previous versions of ACL2, regardless of the values of
  ~c[:skip-proofs-okp] or ~c[:defaxioms-okp] passed to ~ilc[include-book].

  Improved warnings and errors for ~ilc[certify-book] and ~ilc[include-book] to
  mention the ~il[portcullis] as a possible source of ~ilc[skip-proofs] and
  ~ilc[defaxiom]s.

  ACL2 formerly caused an error when ~il[hints] in a ~c[:]~ilc[corollary] were
  not well-formed.  This situation could arise as follows when certifying a
  book.  A lemma FOO is proved ~ilc[LOCAL]ly to the book (or, is present in a
  sub-book that is included locally).  The ~c[:corollary] of a subsequent
  theorem, BAR, disables that rule in a hint.  When BAR is proved, this is not
  a problem.  But ~ilc[certify-book] makes a second pass after processing the
  events in a book: it essentially does an ~ilc[include-book].  During the
  ~c[include-book] pass, FOO is not known (because it was ~ilc[local]), and
  therefore ACL2 fails to process the ~ilc[disable] of FOO in an
  ~ilc[in-theory] hint.  The fix is that during ~ilc[include-book], ~il[hints]
  are ignored in corollaries just as they have been for the main theorem (or
  definition).

  It was possible for guard verification to succeed where it should have
  failed.  We have fixed the bug (which was in source function (ironically
  named!) ~c[fcons-term-smart]).  Thanks to Robert Krug for sending us an
  example of bungled guard verification.  It turns out that this bug was also
  present in Version_2.6.

  The ~il[proof-checker] command ~c[=] has been improved.  Formerly, it could
  fail to apply when certain ~ilc[implies] terms were in the context.  Thanks
  to Pete Manolios for bringing this problem to our attention.

  The command ~ilc[add-binop] failed to work.  This has been fixed.  Thanks to
  Rob Sumners for pointing out this problem.  Also ~pl[note-2-7-other] for a
  discussion of how this and another ~il[table] are no longer part of the
  ~ilc[acl2-defaults-table].

  Book certification could cause a segmentation fault in cases where the
  certification world (~pl[certify-book]) has a very large number of events.
  This has been fixed.

  We now allow empty ~c[:use] ~il[hints] and empty hints, as requested by Eric
  Smith.  Examples:
  ~bv[]
  (\"Goal\" :use ())
  (\"Goal\")
  ~ev[]

  A large ~ilc[mutual-recursion] nest could cause a stack overflow when
  executing either ~c[:pr FN], ~c[:pr! FN], or ~c[:monitor (:definition FN) t],
  where ~c[FN] is in that large mutual recursion nest.  This has been fixed
  (implementation detail:  function ~c[actual-props] has been made
  tail-recursive).  NOTE:  If you just want the definition of ~c[FN],
  ~c[:]~ilc[pf]~c[ FN] can be much faster than ~c[:]~ilc[pr]~c[ FN] if ~c[FN]
  is in a large ~ilc[mutual-recursion].

  Hard Lisp errors could occur when including uncertified books.  This has been
  fixed; ACL2 now does syntax-checking formerly omitted when including
  uncertified books.

  Previously, the evaluation of ~ilc[defstobj] and ~ilc[mutual-recursion] forms
  could cause ``undefined'' warnings when the form was compiled.  This has been
  fixed.  Thanks to Eric Smith for bring a ~c[mutual-recursion] example to our
  attention.

  A bug has been fixed in the syntactic check for valid ~c[:]~ilc[loop-stopper]
  values.  Formerly, valid ~c[:loop-stopper] values were erroneously restricted
  to lists of length at most 2 (a minor problem, since these lists typically
  have length 1), and the function symbol(s) need not have been defined in the
  current ACL2 ~il[world].  Thanks to Eric Smith for sending an example to
  demonstrate the latter problem.

  Functions definitions that are ~c[:non-executable] (~pl[xargs]) had never
  been recognized as redundant, but this has been fixed.  Thanks to Vernon
  Austel for pointing out this problem.

  Compilation using ~c[:]~ilc[comp] now compiles user-defined
  ~c[:]~ilc[program] mode functions.  Formerly only ~c[:]~ilc[logic] mode
  functions could be compiled using ~c[:comp].

  Handling of ~c[:by] hints has been improved in essentially three ways.  The
  primary change is that now, when the current goal exactly matches the
  supplied lemma instance, the subsumption test will always succeeds
  (~pl[hints], in particular the discussion of ~c[:by]).  Second, certain proof
  failures involving ~c[:by] ~il[hints] were failing silently, with duplicate
  messages ``As indicated by the hint, this goal is subsumed by....''  This
  could happen when the original goal was among the goals generated by applying
  the hint.  This problem has been fixed by no longer considering this proof
  step to be specious (~pl[specious-simplification]).  Third and finally, when
  the ~il[lemma-instance] refers to a definition, the original body of that
  definition is used rather than the simplfied (``normalized'') body.

  In addition to the obove, we now recognize more cases of specious
  simplification (~pl[specious-simplification]).  Thanks to Eric Smith for
  bringing this issue to our attention.

  Fixed building of ACL2 under CLISP so that (1) the appropriate ACL2 startup
  message is printed out when ACL2 starts up, and (2) the lisp process supplied
  to make, e.g., LISP=/usr/bin/clisp, is the one written out to the saved ACL2
  file.  Thanks to Dave Greve and Noah Friedman for suggesting (2).  Also, ACL2
  now works with CLISP 2.30.  We have accommodated a change in CLISP's handling
  of streams and its package-locking mechanism, as well as certain non-standard
  characters that formerly could cause CLISP 2.30 to break, even when those
  characters are in comments.

  Eliminated compiler warnings for CMU Lisp.

  Fixed an incorrect error supplied when book certification proceeded so
  quickly that the file write dates of the book (~c[.lisp] file) and the
  corresponding compiled file are equal.  Now that error only occurs if the
  compiled file has a strictly earlier write date, which probably should never
  happen.

  Fixed an infinite loop when executing ~c[make clean-books] (and hence `make'
  with targets that call ~c[clean-books], namely, ~c[certify-books-fresh],
  ~c[regression-fresh], and ~c[regression-nonstd-fresh]), which could occur
  when any subdirectories of ~c[books/] are missing ~-[] even ~c[workshops/],
  which is intended to be optional.  Thanks to Pete Manolios for pointing out
  this bug.

  The ~ilc[include-book] command now works properly even when filenames, or
  their directories or parent directories (etc.) are links.  Thanks to Matt
  Wilding for pointing out this problem.

  The commands ~c[:]~ilc[puff] ~c[:]~ilc[puff*] have been fixed.  Formerly,
  there was a bug when ~c[:puff] or ~c[:puff*] caused the execution of an
  ~ilc[include-book] for an absolute ~il[pathname], ~c[P], that was other than
  the current connected book directory (~pl[cbd]).  When including ~c[P], any
  subsidiary ~ilc[include-book] with a relative pathname would be erroneously
  considered relative to the current ~ilc[cbd] rather than relative to the
  directory of ~c[P].  Thanks to Pete Manolios and Matt Wilding for pointing
  out this problem.

  It had been possible in a ``large'' ACL2 image to call
  ~ilc[verify-termination] successfully on built-in function ~ilc[sys-call],
  with undesirable results.  This hole has been plugged.  Thanks to Rob Sumners
  for pointing out this problem.  The new function ~ilc[gc$] must also stay in
  ~c[:]~ilc[program] mode.

  ACL2 no longer warns when certifying a book based on ~ilc[local] functions
  whose ~il[guard]s have not yet been verified.  Thanks to Pete Manolios for
  pointing out this issue.

  An occasional ``slow array warning'' had been possible during proofs.  The
  following sequence shows how to evoke that warning in previous versions.
  ~bv[]
  (in-theory (disable binary-append))
  (in-theory (enable binary-append))
  (in-theory (disable binary-append))
  (ubt 2)
  (thm (equal (car (cons x y)) x))
  ~ev[]
  (~l[note-2-7-other] for a discussion of a change to ~ilc[compress1] in
  support of this fix; however, users should not need to read that discussion.)

  The raw Lisp code for ~ilc[defchoose] had a small bug, which was only
  evidenced in CLISP implementations as far as we know.  It has been fixed.

  When ~ilc[ld] is applied to a stringp file name, it now temporarily sets the
  connected book directory (~pl[cbd]) to the directory of that file while
  evaluating forms in that file.  To see the effect of this change, imagine a
  subdirectory ~c[\"sub\"] of the current directory, and imagine executing
  ~c[(ld \"sub/foo.lisp\")], where file ~c[foo.lisp] contains the form
  ~c[(include-book \"bar\")].  Presumably the intention was to consider the
  file ~c[bar.lisp] in the same directory, ~c[sub/], as ~c[foo.lisp].  ~c[Ld]
  now honors that intention, but in previous versions ~c[\"bar.lisp\"] would
  have been a reference to a file in the current directory, not in ~c[sub/].

  For users of ~c[run-acl2] [perhaps there are none!]: A fix has been provided
  by a Debian user via Camm Maguire so that acl2-mode anyone using that?] will
  work in Xemacs, which apparently uses variable ~c[lisp-mode-shared-map] rather
  than ~c[shared-lisp-mode-map].

  ACL2 has, for a long time (always?), had a mechanism for avoiding re-proving
  ~il[constraint]s generated by ~c[:functional-instance] ~il[lemma-instance]s
  in ~c[:use] and ~c[:by] hints.  But this mechanism had not applied to defined
  (as opposed to constrained) functions.  This has been fixed.  Thanks to
  Francisco J. Martin-Mateos (ChesKo) for pointing out this problem by sending
  a clear example.

  ~/~/")

(deflabel note-2-7-new-functionality
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on New Functionality~/

  ACL2 now has a more powerful technique for relieving a ~c[:]~ilc[rewrite] or
  ~c[:]~ilc[linear] rule's hypothesis that contains free variables.  A new
  ~il[documentation] section has been written describing the handling free
  variables in rules; ~pl[free-variables].  In brief, the primary change is
  that when a free-variable match for the current hypothesis fails to allow
  subsequent hypotheses to be relieved, then additional matches may be
  attempted until they have all been tried.  Also ~pl[rule-classes] (discussion
  of ~c[:match-free]).  Also ~pl[set-match-free-error],
  ~pl[set-match-free-default], and ~pl[add-match-free-override] for interfaces
  provided to the user for controlling the way ACL2 deals with free variables
  in hypotheses.  We thank Rob Sumners for several helpful discussions about
  the designs of those interfaces, as well as Eric Smith and Robert Krug for
  helpful related discussions.  Robert Krug also found a performance bug in a
  preliminary version, for which we are grateful.

  WARNING: Book certification attempts may take much longer now that, by
  default, ACL2 looks for more free variable matches (see paragraph just
  above).  You can get the old behavior by inserting the form
  ~bv[]
  (set-match-free-default :once)
  ~ev[]
  just after the initial ~ilc[in-package] form.  However, rules from included
  books that have free variables can still slow down certification.  This can
  be fixed by inserting
  ~bv[]
  (add-match-free-override :once t)
  ~ev[]
  before the first event in the file that generates a proof.

  ~il[Forward-chaining] has been made more powerful in the presence of free
  variables (~pl[free-variables]), thanks to a contribution by Erik Reeber.
  Both before and now, when an attempt is made to relieve (prove) a hypothesis
  of a ~c[:forward-chaining] rule in the case that at least one variable in
  that hypothesis is not yet bound, ACL2 looks in the current context for an
  instance of that hypothesis.  If it finds one, then it binds the unbound
  variables and continues to the next hyopothesis.  What is new is that ACL2
  can now looks for multiple instances of that hypothesis.  Consider the
  following example; an explanation is below.
  ~bv[]
  (encapsulate (((op * *) => *))
               (local (defun op (x y) (< x y)))
               (defthm transitivity-of-op
                 (implies (and (op x y) (op y z)) (op x z))
                 :rule-classes :forward-chaining))

  ; fails in Version_2.6; succeeds in in Version_2.7
  (thm (implies (and (op a b) (op b c) (op b e)) (op a c)))
  ~ev[]
  Before Version_2.7, the proof of the ~c[thm] above fails.  When the
  ~c[:forward-chaining] rule ~c[transitivity-of-op] binds ~c[x] to ~c[a] and
  ~c[y] to ~c[b], it then looks for an instance of ~c[(op y z)] in the current
  context, with ~c[y] bound to ~c[b] but ~c[z] unbound.  It happens to find
  ~c[(op b e)] before ~c[(op b c)], and it then adds ~c[(op a e)] to the
  context.  But starting with Version_2.7, it continues to look for additional
  instances and finds ~c[(op b c)] in the context as well, chaining forward to
  ~c[(op a c)] and thus proving the theorem.

  A new macro, ~ilc[bind-free], provides a simple way to get much or most of
  the power of ~il[meta]functions.  Thanks to Eric Smith for coming up with the
  idea and to Robert Krug for providing an implementation (which we modified
  only very slightly) and documentation. ~l[bind-free] and
  ~pl[bind-free-examples].

  With the addition of ~ilc[bind-free] (mentioned above), ~ilc[syntaxp] has
  become a macro, although that change should be transparent to the user.  More
  importantly, the argument of ~c[syntaxp] may now refer to variables ~c[mfc]
  and ~c[state], giving ~c[syntaxp] some of the power of extended metafunctions;
  ~pl[syntaxp] and ~pl[extended-metafunctions].  Thanks to Robert Krug for
  implementing that extension.  Also, the argument of ~ilc[syntaxp] may now
  include calls of ~c[:]~ilc[program] mode functions.  ~l[syntaxp] and
  ~pl[syntaxp-examples] (thanks to Robert Krug for updating the former and
  creating the latter documentation).

  The linear-arithmetic decision procedure (~pl[linear-arithmetic]) has now
  been extended so that ACL2 can reason about non-linear arithmetic as well
  (~pl[non-linear-arithmetic] for how to turn on this feature).  We thank
  Robert Krug for the initial implementation of this, and Eric Smith for finding
  a couple of bugs in it.

  Some ~ilc[trace] utilities have been made available in the ACL2 loop.~bq[]

  o Function ~ilc[trace$] (and also ~ilc[untrace$]) calls the corresponding
  underlying Lisp routine ~c[trace] (and ~c[untrace]), which however continues
  (as it has for some time) to be enhanced for GCL and Allegro CL.

  o Macro ~ilc[open-trace-file] causes trace output to go to a specified
  file.  Macro ~ilc[close-trace-file] causes trace output to go to the
  screen (which is the default).

  o Macro ~c[with-error-trace] (or, ~c[wet] for short) causes a backtrace
  to be written out for many failures, including guard violations.  ~l[trace],
  ~pl[trace$], and see :DOC ~c[wet] [** NOTE: eliminated after Version  3.3].

  ~eq[]

  A new ~ilc[theory], ~ilc[minimal-theory] has been provided (~pl[theories]).
  It can be particularly useful for speeding up proofs involving ~c[:use]
  ~il[hints].

  New ~ilc[events] ~ilc[defund] and ~ilc[defthmd] behave exactly like
  ~ilc[defun] and ~ilc[defthm], respectively, except that these new events
  disable the new name.

  The new macro ~ilc[with-output] can be used to suppress output that would
  normally result from evaluation of a form.

  The form ~c[(]~ilc[pstack]~c[)] can give the user an idea of what the
  prover has been up to during a proof, or after a user-aborted proof.
  Moreover, by evaluating ~c[(verbose-pstack t)] (~pl[verbose-pstack])
  one can get ~il[trace]-like information about prover functions, including
  time summaries, printed to the screen during a proof.  Thanks to Bill Legato
  and Robert Krug for initiating this work and to Robert for providing some
  initial implementation.

  The new command ~c[:]~ilc[comp-gcl] is identical in functionality, except
  that it always leaves ~c[.c] and ~c[.h] files when compiling in GCL.  Thanks
  to Rob Sumners and Vernon Austel for suggesting such a capability.

  The macro ~ilc[e/d] provides a convenient way to ~ilc[enable] some rules and
  ~ilc[disable] others.  It was formerly in a book supplied with the
  distribution, ~c[books/ihs/ihs-init.lisp], written by Bishop Brock (who we
  thank for providing this useful macro).

  New distributed books include those in ~c[books/ordinals/],
  ~c[books/rtl/rel3/], and ~c[books/misc/simplify-defuns.lisp] (which is
  documented in ~c[books/misc/simplify-defuns.txt]).

  The ~c[:expand] hint now accepts a special value, ~c[:LAMBDAS], that tells
  the ACL2 rewriter to expand all lambda applications (~ilc[let] expressions).
  ~l[hints].

  A new function ~ilc[zpf] has been added as fast test against 0 for
  nonnegative fixnums.

  A new macro ~ilc[gc$] allows the user to call the garbage collector of the
  underlying Common Lisp.  Thanks to Rob Sumners for suggesting this feature.

  It is now possible to ~ilc[monitor] ~il[simple] (abbreviation) rules.
  However, as a warning explains, they are still not considered monitored
  during preprocessing; ~pl[monitor].  Thanks to Robert Krug for providing this
  improvement.

  The second argument of ~ilc[certify-book], if supplied, formerly had to be
  either ~c[t] or a non-negative integer.  Now it can be the symbol ~c[?], in
  the ~c[ACL2] package, indicating that the usual check should be suppressed on
  the number of commands that have been executed to create the world in which
  ~ilc[certify-book] was called.

  ~/~/")

(deflabel note-2-7-proofs
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on Changes in Proof Engine~/

  An improvement in the linear arithmetic heuristics has been provided
  by Robert Krug.  For information about this change, search for the
  comment in ~c[add-linear-lemma] (file ~c[rewrite.lisp]) that begins
  as follows.
  ~bv[]
  ; Previous to Version_2.7, we just went ahead and used the result of
  ~ev[]
  Thanks, Robert!  Also thanks to Eric Smith for providing a
  motivating example.

  The non-linear-arithmetic addition (~pl[non-linear-arithmetic]) led to
  several small changes in the linear-arithmetic decision procedure
  (~pl[linear-arithmetic]).  Two of these changes could affect existing
  proofs.~bq[]

  First, when we are setting up the initial arithmetic database (which we call
  the ``pot-lst''), we have always scanned it to see if there were any pairs of
  inequalities from which we could derive a previously unknown equality.  In
  some cases we added this equality to the clause and in others we used it to
  rewrite the clause, substituting one side of the equality for the other
  throughout the clause.  Previously, the heuristics that we used to determine
  whether we performed the substitution differed from those used in several
  other places in the code.  This has now been regularized, and similar
  heuristics are now used throughout the code.

  The second change to the linear-arithmetic decision procedure is
  that we now explicitly add inequalities derived from type reasoning
  to the pot-lst.  Previously, we performed cancellations against these
  inequalities without adding them to the pot-lst.  This change results
  in there being more inequalities in the pot-lst than before, and
  so more chances for there to be a pair of inequalities from which an
  equality can be derived.  In effect, certain simple consequences of
  the current goal (~pl[type-set]) may now be added as hypotheses of
  the goal or used to peform equality substitutions.

  ~eq[]

  A slight improvement has been made to the way certain rewrite rules are
  stored.  It was already the case that a rewrite rule rule whose conclusion
  ~c[C] is not a call of a known equivalence relation (or ~ilc[eq], ~ilc[eql],
  or ~ilc[=]) is stored as ~c[(iff C t)], except that if ACL2 can determine
  (using its ~ilc[type-set] mechanism) that ~c[C] is Boolean, then the rule is
  stored as ~c[(equal C t)].  The iprovement is that if ~c[C] and ~c[C'] are
  Boolean, then a rule stated as ~c[(iff C C')] is stored as ~c[(equal C C')].
  Thanks to Pete Manolios for providing an example that led us to consider this
  improvement.

  The heuristic use of equalities (fertilization) has been modified.
  Previously, ACL2 would sometimes substitute using an equality but keep the
  equality, and then undo the substitution by using the equality again.  Now,
  when ACL2 keeps an equality after using it, it puts the equality inside a
  call of ~ilc[hide].  Descendents of that goal that are unchanged by
  simplification will have this call of ~ilc[hide] removed so that the equality
  can once again contribute to the proof.  This change can cause some proofs to
  succeed that otherwise would fail.  In the unlikely event that a proof fails
  that formerly succeeded, the following hint on \"Goal\" may fix the problem
  (~pl[hints]):
  ~bv[]
  :expand ((:free (x) (hide x)))
  ~ev[]

  We have refined the heuristics employed when an ~ilc[IF] form is assumed true
  or false.  Our previous attempt (see ~il[note-2-6-proofs] for the original
  announcement) was not as general as we had believed.  We have also improved
  some low-level code responsible for rewriting ~c[IF] expressions.  In
  earlier versions of ACL2, it was possible to have the truth or falsity
  of an ~c[IF] expression explicitly recorded in the type-alist, and yet
  not use this information during rewriting.  This problem has been corrected.
  Thanks to Robert Krug for noticing this problem and implementing the fix.

  We have sped up the rewriter in some cases where there are large collections
  of mutually-recursive functions (~pl[mutual-recursion]).  (Implementation
  notes: technically, we have modified the way function ~c[being-openedp]
  operates on the ~c[fnstack], and we have modified
  ~c[*current-acl2-world-key-ordering*] as described in the essay above its
  definition.)

  ~il[Forward-chaining] is now done in the preprocessing phase of proof
  attempts (see the discussion of ~c[:DO-NOT] ~-[] ~pl[hints]).  This is part
  of a technical change, made in support of translation of type declarations to
  ~il[guard]s (~pl[note-2-7-guards]).  Previously, whenever ACL2 checked for
  ~il[built-in-clause]s, it then looked for a contradiction using
  ~ilc[type-set] reasoning if it did not find a suitable built-in clause.  The
  change is to perform forward-chaining in such cases (i.e., when a built-in
  clause is not found).

  A couple of changes have been made in the generation of goals for
  ~il[forcing-round]s.  Thanks to Eric Smith for bringing issues to our
  attention that led to these changes.  For one, ~il[guard]s are no longer
  relevant in such goal generation.  Formerly, the addition of a guard could
  make a proof fail that otherwise succeeded.  Secondly, contextual information
  is now always kept when it involves a constrained constant, i.e., a zero-ary
  function introduced in the signature of an ~ilc[encapsulate].

  ~/~/")

(deflabel note-2-7-rules
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on Changes in Rules and Constants~/

  The ~ilc[defcong] macro has been slightly changed.  The difference is that
  the variable generated with suffix ~c[-EQUIV] will now be in the same package
  as the name of the variable from which it is generated, rather than always
  belonging to the ACL2 package.  Thanks to Hanbing Liu for suggesting this
  change.  (Note that a couple of books have been modified to accommodate this
  change, e.g., ~c[books/finite-set-theory/set-theory].)

  In Version_2.6, a change was made for rules of class ~c[:]~ilc[rewrite] whose
  conclusion is a term of the form ~c[(EQV lhs rhs)], where ~c[EQV] is ~ilc[=],
  ~ilc[eq], or ~ilc[eql]: the rule was stored as though ~c[EQV] were
  ~ilc[equal].  (~l[note-2-6-rules].)  This change has been extended to rules
  of class ~c[:]~ilc[definition].

  ~/~/")

(deflabel note-2-7-guards
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on Guard-related Changes~/

  It was possible for guard verification to succeed where it should have
  failed.  See the discussion under ~il[note-2-7-bug-fixes].

  There have been changes in the guards generated from type declarations for
  the following cases.  Thanks to Dave Greve and Matt Wilding for suggesting
  such changes.
  ~bv[]
  (type (signed-byte n) val)
  (type (unsigned-byte n) val)
  (type (integer m n) val)
  ~ev[]
  The following examples illustrate the changes.
  ~bv[]
  (type (signed-byte 4) x)
  ==> [old] (AND (INTEGERP X) (<= -8 X) (<= X 7))
  ==> [new] (SIGNED-BYTE-P 4 X)

  (type (unsigned-byte 4) x)
  ==> [old] (AND (INTEGERP X) (<= 0 X) (<= X 15))
  ==> [new] (UNSIGNED-BYTE-P 4 X)
  ~ev[]

  ~/~/")

(deflabel note-2-7-proof-checker
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on Proof-checker Changes~/

  Output from the ~il[proof-checker] can now be inhibited by supplying the
  symbol ~c[proof-checker] in the list given to ~il[set-inhibit-output-lst].

  ~/~/")

(deflabel note-2-7-system
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on System-level Changes~/

  ACL2 now runs (once again) under LispWorks, specifically, LispWorks 4.2.0.
  However, we needed a patch, which presumably will be unnecessary after 4.2.7.
  From LispWorks support:
  ~bq[]
  Users with LispWorks4.2.7 should ask us at lisp-support@xanalys.com
  for the transform-if-node patch. It will be helpful if they quote
  (Lisp Support Call #11372) when doing so. Also, they must send a bug
  form generated from their LispWorks image: instructions at
  http://www.lispworks.com/support/bug-report.html.
  ~eq[]

  File ~c[books/Makefile-generic] has been improved so that failed attempts to
  certify a book will cause the `make' to fail.  Previously, an existing
  ~c[.cert] file was left in place, and that sufficed for the `make' to be
  considered a success.  Now, the old ~c[.cert] file is first removed when
  recertification is found to be necessary.

  A change has been made to source file ~c[acl2.lisp] to accommodate GCL 2.4.3.
  (ACL2 Version  2.6 does not work with some versions of GCL 2.4.3.)

  The error message has been improved when certain forms are typed to raw Lisp
  and the ACL2 loop has never been entered (with ~c[(]~ilc[LP]~c[)]).

  The following symbols in the ACL2 package have been made untouchable, meaning
  that they are not available to the user: ~c[ev-fncall], ~c[ev], ~c[ev-lst],
  ~c[ev-acl2-unwind-protect], ~c[ev-fncall!], and ~c[user-stobj-alist-safe].
  The reason is that these functions can not be called safely except under
  certain restrictions.  If you want to call the ACL2 evaluator, consider using
  the built-in system functions ~c[trans-eval] or simple-translate-and-eval.

  CLISP Version_2.30 implements a notion of ``locking'' the \"LISP\" package
  that is incompatible with building ACL2.  (CLISP Version_2.27 does not appear
  to have had this feature.)  We have gotten around this problem by unlocking
  the \"LISP\" package in ACL2 images built on such CLISPs.

  Automatic proclaiming for GCL, which has (for a long time) been done for
  functions in compiled books, has been improved.  Formerly, the only time a
  non-trivial output type (i.e., other than ~c[t]) was inferred was when
  macroexpansion produced an explicit call of ~ilc[the].  Now, ~ilc[if]
  expressions can also generate non-~c[t] output types.  Consider the following
  example.
  ~bv[]
  (defmacro the-fixnum (n)
    (list 'the '(signed-byte 29) n))
  (defmacro 1+f (x)
    (list 'the-fixnum
          (list '1+ (list 'the-fixnum x))))
  (defun foo (x)
    (declare (type (unsigned-byte 27) x))
    (if (zp x)
        0
      (1+f (foo (1-f x)))))
  ~ev[]
  Formerly, the ~c[proclaim] forms for ~c[foo], before and after this
  improvement, are as shown below.
  ~bv[]
  (PROCLAIM '(FTYPE (FUNCTION ((UNSIGNED-BYTE 27)) T) FOO))                ;old
  (PROCLAIM '(FTYPE (FUNCTION ((UNSIGNED-BYTE 27)) (SIGNED-BYTE 29)) FOO)) ;new
  ~ev[]

  Compiler info messages sent to error stream were eliminated for CMUCL.

  ~/~/")

(deflabel note-2-7-other
  :doc
  ":Doc-Section note-2-7

  ACL2 Version  2.7 Notes on Miscellaneous Changes~/

  Made several minor ~il[documentation] improvements.  We are grateful to Eric
  Smith for suggesting (most of) these.

  Improved ~c[(show-bdd)] (~pl[bdd]) to give more useful feedback when there
  are ``leaf'' terms not known to be Boolean.

  Sped up processing of large mutual-recursion nests.  In one large example the
  speedup was roughly two orders of magnitude.

  Modified event printing so that if both ~c['prove] and ~c['event]
  are inhibited, then events are no longer printed on behalf of
  ~ilc[certify-book], ~ilc[encapsulate], or ~ilc[defstobj].  Thanks
  to Eric Smith for prompting consideration of such a change.

  The following technical change was made to support ~c[with-error-trace] and
  ~c[wet] (~pl[note-2-7-new-functionality]), but may be of interest to those
  who do low-level programming using the ACL2 logical ~ilc[world].  The
  ~c['unnormalized-body] property is now stored not only for functions defined
  in ~c[:]~ilc[logic] mode, but also for functions defined by the user in
  ~c[:]~ilc[program] mode.  (~c[:Program] mode Functions built into ACL2 still
  have their ~c['unnormalized-body] property omitted, in order to save space.)

  The handling of ``invisible'' functions for purposes of controlling rewriting
  (~pl[loop-stopper]) has been moved to a new table; ~pl[invisible-fns-table].
  Macros that access and modify this table are called
  ``~c[...-invisible-fns-table]'' in place of their former names,
  ``~c[...-invisible-fns-alist].''  This feature was formerly implemented in
  the ~ilc[acl2-defaults-table], which prevented a book from exporting lists of
  invisible functions intended to work with the ~il[rewrite] rules developed in
  the book.  Thanks to Eric Smith and Rob Sumners for suggesting this change.
  ~l[set-invisible-fns-table] (formerly ~c[set-invisible-fns-alist]), and also
  ~pl[add-invisible-fns] and ~pl[remove-invisible-fns], which provides ways to
  incrementally add to and remove from this table, respectively.  The handling
  of printing binary function call nests using macros
  (~l[add-binop]) has also been moved out of the ~ilc[acl2-defaults-table] as
  suggested by Eric and Rob, but this feature didn't work anyhow
  (~pl[note-2-7-bug-fixes]).  Incidentally, the symbols ~c[binop-table],
  ~ilc[add-binop], and ~ilc[remove-binop] have all been added to the list
  ~c[*acl2-exports*] (~pl[acl2-user]), ~ilc[add-invisible-fns] and
  ~ilc[remove-invisible-fns] have been added to that list, and
  ~c[set-invisible-fns-alist] has been replaced in that list by
  ~ilc[set-invisible-fns-table].  Function ~c[invisible-fns-alistp] is no
  longer defined and has been removed from ~c[*acl2-exports*].

  We now enforce the stated restriction on the pairings in
  ~c[macro-aliases-table] (~pl[macro-aliases-table]), namely, that it
  associates names of macros with names of funcions (with respect to the
  current ACL2 logical ~il[world]).  We make a similar requirement on
  ~ilc[invisible-fns-table].

  The ~ilc[theory-invariant] event has been modified so that the default action
  is an error rather than a warning.  Thanks to Eric Smith for suggesting this
  change.  Also, the value returned upon successful execution of a
  ~ilc[theory-invariant] event is now the key.

  Proof output that reports destructor elimination no longer uses the word
  ``generalizing''.  This small change may help in browsing proof output, since
  now ``generaliz'' takes you to true uses of generalization.  Thanks to Matyas
  Sustik for suggesting such a change.

  The command ~c[:]~ilc[pl] now prints an abbreviated controller-alist for
  ~c[;]~ilc[definition] rules.  Formerly the output from ~c[:pl] could be
  overwhelming when the supplied function was part of a large
  ~ilc[mutual-recursion] nest.

  The defaults for keyword parameters of ~ilc[certify-book] have changed.
  ~l[note-2-7-bug-fixes], in particular, the discussion there of two
  modifications to ~c[certify-book].

  Technical changes have been made to ~ilc[compress1] and ~ilc[compress2] that
  should usually be invisible to users.  The next paragraph describes them in
  detail, only for competeness (i.e., that description can be ignored by most
  users).  But first, here is an example showing an effect on users.  The slow
  array warning was not there previously.  Notice that the warning only arises
  if the event form is changed.  The solution is to be sure that redundant
  ~ilc[defconst] forms are syntactically identical.
  ~bv[]
  ACL2 !>(defconst *a* (compress1 'demo
                                  '((:header :dimensions (5)
                                             :maximum-length 15
                                             :default uninitialized
                                             :name demo)
                                    (1 . one)
                                    (0 . zero))))

  Summary
  Form:  ( DEFCONST *A* ...)
  Rules: NIL
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
   *A*
  ACL2 !>(aref1 'demo *a* 0)
  ZERO
  ACL2 !>(defconst *a* (compress1 'demo
                                  '((:header :dimensions (5)
                                             :maximum-length 15
                                             :default uninitialized
                                             :name demo)
                                    (1 . one)
                                    (0 . zero))))

  This event is redundant.  See :DOC redundant-events.

  Summary
  Form:  ( DEFCONST *A* ...)
  Rules: NIL
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
   :REDUNDANT
  ACL2 !>(aref1 'demo *a* 0)
  ZERO
  ACL2 !>(defconst *a* (compress1 'demo
                                  '((:header :dimensions (5)
                                             :maximum-length 15
                                             :default uninitialized
                                             :name demo)
                                    (0 . zero)
                                    (1 . one))))

  This event is redundant.  See :DOC redundant-events.

  Summary
  Form:  ( DEFCONST *A* ...)
  Rules: NIL
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
   :REDUNDANT
  ACL2 !>(aref1 'demo *a* 0)


  **********************************************************
  Slow Array Access!  A call of AREF1 on an array named
  DEMO is being executed slowly.  See :DOC slow-array-warning
  **********************************************************

  ZERO
  ACL2 !>
  ~ev[]

  As before, the von Neumann structure stored in the ~c['acl2-array] property
  of the array name contains the array list object in its ~ilc[car].  However,
  previously it was the case that ~c[compress1] and ~c[compress2] did not
  update that ~c[car] when its new value would be equal to its old value.  This
  was done largely in support of some type-set tables defined using
  ~ilc[defconst] in ~c[type-set-b.lisp].  The new versions of ~ilc[compress1]
  and ~ilc[compress2] are simpler in that no such exception is made in the case
  of equal lists, although instead the entire compression process is
  short-circuited when the input array list object is ~ilc[eq] to the ~c[car]
  of the ~c['acl2-array] property.  This change was made because the equality
  test was causing a ``slow array access'' warning to be printed in rare cases
  during proofs, as described elswhere (~pl[note-2-7-bug-fixes]).

  We no longer distribute documentation specific to Lucid Emacs.  The Info
  documentation in directory ~c[doc/EMACS/] works well both for Gnu Emacs and
  XEmacs.

  A little-advertised macro, ~c[value], has long been allowed for top-level
  forms in ~il[books]; ~pl[embedded-event-form].  This has been replaced by a
  new macro, ~c[value-triple].  The two have the same semantics at the
  top-level of books, where ~ilc[state] is ``live''.  However, ~c[value-triple]
  should be used at the top-level of a book, while ~c[value] should be used in
  function definitions (as before).  This change eliminates a warning put out
  by the Allegro Common Lisp compiler for top-level ~c[value] forms in
  ~il[books].

  ~/~/")

(deflabel |NOTE-2-7(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.7(r) (November, 2002) Notes~/

  In source file ~c[axioms.lisp], in order for proofs to succeed,
  (~c[make proofs]), the definitions of ~ilc[acl2-count] and ~c[explode-atom]
  have been modified slightly, and lemma ~c[standard-numberp-one] [modified
  after Version_3.4 to become ~c[standardp-one]] has been given
  ~c[:rule-classes nil].

  All ~ilc[skip-proofs] forms have been eliminated from the nonstd books, thanks
  to Ruben Gamboa.

  The directory ~c[books/sqrt/], which was intended for ACL2(r), has been moved
  to ~c[books/nonstd/sqrt/] and added as appropriate to
  ~c[books/nonstd/Makefile].

  ~/

  Please ~pl[note-2-7] for changes to Version_2.7 of ACL2.
  ~/
  ")

(deflabel note-2-8
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.8 (March, 2004) Notes~/

  BRIEF SUMMARY.

  The Version_2.8 notes are divided into the indicated subtopics.  Here we give
  only a brief summary of just a few of the major new features and changes that
  seem most likely to impact existing proofs.  Not included in this brief
  summary, but included in the subtopics, are descriptions of many improvements
  (including bug fixes and new functionality) that should not get in the way of
  existing proof efforts.  In the description below we also omit discussion of
  changes that will become clear by way of error messages if they affect you.

  In particular, please ~pl[note-2-8-new-functionality] for discussion of a
  number of new features that you may find useful.

  Acknowledgements and elaboration, as well as other changes, can be found in
  the subtopics listed below.

  o Some of the bug fixes (~pl[note-2-8-bug-fixes]):~bq[]

  + Some soundness bugs were fixed.

  + The handling of free variables in hypotheses (~pl[free-variables]) of
  rewrite and linear rules had a bug that prevented some proofs from going
  through.  Now that this bug has been fixed, you may find some proofs running
  much more slowly than before.  You can use ~ilc[accumulated-persistence] and
  ~ilc[add-match-free-override] to remedy this situation;
  ~pl[note-2-8-bug-fixes] for details.

  + The ~il[default-hints] in the current logical ~il[world] are no longer
  ignored by ~ilc[verify-guards].

  + Forms violating guard-checking such as ~c[(defconst *silly* (car 3))] are
  now allowed in ~il[books].~eq[]

  o Some of the new functionality (~pl[note-2-8-new-functionality]):~bq[]

  + WARNING: You may find that ~c[control-d] (in emacs,
  ~c[control-c control-d]) can throw you completely out of Lisp where it had
  not formerly done so.

  + ACL2 now starts up inside the ACL2 loop ~-[] that is, ~c[(]~ilc[LP]~c[)] is
  executed automatically ~-[] when built on CLISP or Allegro CL.  This was
  already the case for GCL and CMUCL, and it still is not true for LispWorks.

  + ~l[note-2-8-ordinals] for a discussion of a significant change in ordinal
  represtation, and in particular, for how to preserve existing proofs that
  depend on the previous ordinal representation.

  + Macros ~ilc[mbe] (``must be equal''), ~ilc[mbt] (``must be true''), and
  ~ilc[defexec] have been introduced, which allow the user to attach
  alternate executable definitions to functions.

  + The user can now control multiple matching for free variables in hypotheses
  for ~c[:]~ilc[forward-chaining] rules, as has already been supported for
  ~c[:]~ilc[rewrite] and ~c[:]~ilc[linear] rules.

  + It is no longer necessary to specify ~c[(set-match-free-error nil)] in
  order to avoid errors when a rule with free variables in its hypotheses is
  missing the ~c[:match-free] field.

  + The form ~c[(break-on-error)] causes, at least for most Lisps, entry into
  the Lisp debugger whenever ACL2 causes an error.

  + A new ~ilc[table] has been provided so that advanced users can override the
  built-in ~c[untranslate] functionality.  ~l[user-defined-functions-table].

  + The ~ilc[pstack] (`process [prover] stack'') mechanism, formerly denoted
  ~c[checkpoints], has been improved.  One of these improvements is to show
  actual parameters with ~c[(pstack t)] rather than formals.

  + The ~ilc[defstobj] event is now allowed to take an ~c[:inline] argument,
  which can speed up execution.

  + Macro ~ilc[cw-gstack] no longer takes arguments for the ~c[gstack] or
  ~ilc[state].  To print terms in full rather than abbreviated:
  ~c[(cw-gstack :evisc-tuple nil)].

  + The ~ilc[include-book] event now has an additional (optional) keyword,
  ~c[:dir].  In particular, ~c[(include-book \"foo/bar\" :dir :system)] will
  include the indicated book after prepending the path of the built-in
  ~c[books/] directory.  You will probably not find ~c[:dir :system] to be
  useful if you move the executable image or distributed books;
  ~pl[include-book], in particular its ``soundness warning''.

  + The printing of results in raw mode (~pl[set-raw-mode]) may now be
  partially controlled by the user:  ~pl[add-raw-arity].

  + For those using Unix/Linux `make':  A ~c[cert.acl2] file can contain
  forms to be evaluated before an appropriate ~ilc[certify-book] command is
  invoked automatically (not included in ~c[cert.acl2]).~eq[]

  o Some of the changes in the proof engine (~pl[note-2-8-proofs]):~bq[]

  + ACL2 now prevents certain rewriting loops; ~pl[rewrite-stack-limit].

  + Small changes have been made to heuristics for controlling rewriting during
  proofs by induction and in handling certain ``weak'' ~il[compound-recognizer]
  rules.

  + The handling of free variables in a hypothesis of a ~il[rewrite] rule
  (~pl[free-variables]) has been improved in the case that the hypothesis is of
  the form ~c[(equiv x y)], where ~c[equiv] is a known equivalence relation
  (~pl[equivalence]).

  + We have modified how the ACL2 simplifier handles the application of a
  defined function symbol to constant arguments, by avoiding the introduction
  of ~il[hide] when evaluation fails if the term can be rewritten.

  + The generation of \"Goal\" for recursive (and mutually-recursive) definitions
  now uses the subsumption/replacement limitation (default 500).
  ~l[case-split-limitations].

  + Default hints now apply to hints given in definitions, not just theorems.
  ~l[default-hints].

  + Linear arithmetic now uses the conclusions of ~ilc[forward-chaining] rules,
  and ~ilc[type-set] now uses a small amount of linear reasoning when deciding
  inequalities.~eq[]

  o Some of the changes in rules, definitions, and constants
  (~pl[note-2-8-rules]):~bq[]

  + See the above doc topic.~eq[]

  o Guard-related changes are described in ~pl[note-2-8-bug-fixes].

  o Some of the proof-checker changes (~pl[note-2-8-proof-checker]):~bq[]

  + Added new ~il[proof-checker] commands ~c[wrap1], ~c[wrap], and
  ~c[wrap-induct], to combine multiple conjuncts or goals.

  + The ~c[type-alist] command now takes optional arguments that control
  whether or not the governors and/or conclusion are used in computing the
  context.~eq[]

  o Some of the system-level changes (~pl[note-2-8-system]):~bq[]

  + ACL2 now runs on OpenMCL and on MCL 5.0.~eq[]

  o Some of the other changes (~pl[note-2-8-other]):~bq[]

  + Emacs file ~c[emacs/emacs-acl2.el] has been updated (~pl[note-2-8-other]
  for details).

  + When ~c[:pl] is given a term other than a symbol, it will print all rewrite
  rules that match that term.

  + A new function, ~ilc[pkg-witness], returns a symbol in the given package.

  + The list constant ~c[*acl2-exports*] has been extended.

  + A new release of the rtl library has been included: ~c[books/rtl/rel4/].
  See the ~c[README] file in that directory.~eq[]

  Again, please proceed to the subtopics for more thorough release notes.

  ~/~/")

(deflabel note-2-8-bug-fixes
  :doc

; Fixes not included below, and other notes:

; The tautology checker bug mentioned in the :doc below was in call-stack.

; A bug was fixed in assign-wormhole-output [renamed assign-wormhole-status
; after v3-6-1]: er-progn replaces pprogn.

; It is no longer legal for user code to call include-book (this is disallowed
; in translate11).  See the comment in *inside-include-book-fn*.

; The missing argument in the first (er hard ...) in rewrite-fncall has been
; supplied.

; Subterm-one-way-unify has been modified in order to avoid any possibility of
; calling fargs on a quotep.

; Here is a way to exhibit the proof-checker expand bug described in the first
; paragraph of the documentation below:

;  (in-package "ACL2")
;
;  (encapsulate
;   (((foo *) => *)
;    ((bar *) => *))
;
;   (local (defun foo (x) x))
;   (local (defun bar (x) (not x)))
;
;   (defthm foo-open
;     (equal (foo x) x)
;     :rule-classes :definition)
;
;   (defthm bar-not-foo
;     (equal (bar x) (not (foo x)))
;     :rule-classes :definition))
;
;  (defthm bad (equal (foo x) (bar x))
;    :rule-classes nil
;    :instructions
;    ((:dv 1) :expand :nx :expand :top :s))
;
;  (defthm contradiction
;    nil
;    :rule-classes nil
;    :hints (("Goal" :use bad)))

; The second proof-checker bug mentioned below can be exhibited as follows:

;  (encapsulate
;   ()
;   (local
;    (defthm bug-lemma (if x (if x t nil) nil)
;      :rule-classes nil
;      :instructions ((dive 2 3) :s)))
;   (defthm bug nil
;     :rule-classes nil
;     :hints (("Goal" :use ((:instance bug-lemma (x nil)))))))

; The function ev-acl2-unwind-protect was fixed to incorporate a change made
; long ago, by J, to acl2-unwind-protect.  This function was subsequently
; replaced by ev-w-acl2-unwind-protect.

; Without the new fix based on Matyas's suggestion (see paragraph on tautology
; checker below and Qiang's example), we had to change subgoal numbers in hints
; in the following books:
; direct-incorporation-sound-iff in
;   workshops/2003/matlin-mccune/support/simp.lisp
; graph-equivp1-load-graph1 in
;   workshops/2003/greve-wilding_mbe/support/run-fpst.lisp

; Regarding the soundness bug about type-prescription rules, whose description
; below mentions local-incompatibility and refers to an example in that :doc
; topic: The full example is below.  It actually proved in every GCL image of
; ACL2 from v2-7 back to at least v2-4.  The problem goes back at least through
; v2-3 as well; evaluate :PROPS FOO to see a type-prescription record with a
; :basic-ts of nil.  Our fix is to modify add-type-prescription-rule to cause a
; hard error when destructure-type-prescription fails; previously we had
; ignored the erp return value from destructure-type-prescription.

;   (in-package "ACL2")
;
;   (defun my-natp (x)
;     (declare (xargs :guard t))
;     (and (integerp x)
;          (<= 0 x)))
;
;   (defun foo (x)
;     (nfix x))
;
;   (in-theory (disable foo (:type-prescription foo)))
;
;   (encapsulate
;    ()
;    (local (defthm my-natp-cr
;             (equal (my-natp x)
;                    (and (integerp x)
;                         (<= 0 x)))
;             :rule-classes :compound-recognizer))
;    (defthm foo-type-prescription
;      (my-natp (foo x))
;      :hints (("Goal" :in-theory (enable foo)))
;      :rule-classes ((:type-prescription :typed-term (foo x)))))
;
;   (defthm rationalp-foo
;     (rationalp (foo x))
;     :hints (("Goal" :in-theory (enable foo)))
;     :rule-classes :type-prescription)
;
;   (defthm bad-lemma
;     (equal (foo x) 1)
;     :rule-classes nil)
;
;   (defthm bad
;     nil
;     :rule-classes nil
;     :hints (("Goal" :use ((:instance bad-lemma (x 1))))))

  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on Bug Fixes~/

  We have fixed a soundness bug in the tautology checker's handling of
  expressions of the form ~c[(not (not x))].  This bug has gone back at least
  as far as Version_2.4.  All of the regression tests passed after the fix,
  without modification.  So we hope that this bug has rarely bitten anyone.
  Thanks to Qiang Zhang for sending us a proof of ~c[nil] that led us to this
  fix: ~c[(thm (equal (and p q) (not (or (not p) (not q)))))].  And thanks to
  Matyas Sustik for an observation that led to an improvement of our initial
  fix.

  The preceding version (2.7) introduced a soundness bug in handling of ACL2
  ~il[arrays], in which functions ~ilc[compress1] and ~ilc[compress2] were
  returning the input alist rather than compressing it appropriately.  Here is
  a proof of ~c[nil] that no longer succeeds, based on a bug report from Warren
  Hunt, who we thank for bringing this problem to our atttention.
  ~bv[]
  (defthm bad
    (not (let* ((ar2 (aset1 'my-array ar1 3 10))
                (ar3 (compress1 'my-array ar2))
                (ar4 (reverse (reverse ar2)))
                (ar5 (compress1 'my-array ar4)))
           (and (equal ar2 ar4)
                (not (equal ar3 ar5)))))
    :rule-classes nil)
  (defthm contradiction
    nil
    :rule-classes nil
    :hints ((\"Goal\" :use
             ((:instance bad
                         (ar1 (compress1 'my-array
                                         '((3 . 5)
                                           (:HEADER :DIMENSIONS (5)
                                                    :MAXIMUM-LENGTH 6
                                                    :DEFAULT 0
                                                    :NAME MY-ARRAY)))))))))
  ~ev[]
  On a related note, a new function ~ilc[flush-compress] can be used for subtle
  control of under-the-hood raw Lisp support for fast array access, although we
  expect it to be very rare that users need this extra support.

  Previous versions have had two soundness bugs that can occur when using the
  ~il[proof-checker]:
  ~bq[]
  o The first bug pertains to the ~c[expand] command, and hence ~c[x] and
  ~c[x-dumb] commands (which call ~c[expand]); ~pl[proof-checker-commands].
  The bug can occur when applying the above commands when the current term is a
  call of a constrained function symbol for which there is a
  ~c[:]~ilc[definition] rule.  Now, the ~c[expand] command will succeed only
  when the function symbol of the current term is a defined function symbol, in
  which case the original definition is always used, in analogy to how the
  ~c[:expand] hint works in the prover; ~pl[hints].  Thanks to John Erickson
  for sending an example that led us to wonder if there might be a soundness
  problem.

  o The second bug pertains to the ~c[s] command (and commands that call it,
  e.g., ~c[s-prop]).  The proof-checker forms a context out of the top-level
  hypotheses and the ~c[if]-terms governing the current term.  If there is a
  contradiction in the top-level hypotheses, the proof-checker can
  appropriately consider the goal to be proved, and it does so.  But formerly,
  the criterion was weaker:  the contradiction could involve the combination of
  the top-level hypotheses and ~c[if]-term governors.  Thanks to Rob Sumners
  for noticing this bug.~eq[]

  A soundness bug could be provoked in some Lisps by applying ~ilc[defpkg] to
  the empty string.  This has been disallowed.

  We fixed a soundness bug related to packages caused by a failure to track
  axioms introduced ~ilc[local]ly on behalf of ~ilc[defpkg] events.
  ~l[hidden-death-package].

  We fixed a soundness bug caused by a failure to check that a
  ~c[:]~ilc[type-prescription] rule can be processed when proofs are skipped or
  under a ~ilc[defequiv] event.  The former case can occur when processing an
  ~ilc[encapsulate] or ~ilc[include-book] event, where the rule could depend
  on a ~ilc[local] ~c[:]~ilc[compound-recognizer] rule preceding the proposed
  ~c[:]~ilc[type-prescription] rule under the same ~ilc[encapsulate] or
  ~ilc[include-book] event.  ~l[local-incompatibility] for such an example.

  We fixed a potential soundness bug relating to reclassifying a
  ~c[:program] mode function to ~c[:logic] mode (as done by
  ~ilc[verify-termination] or the submission of an appropriate ``redundant''
  definition) without adequate checking that ~ilc[stobj] usage was identical.
  Allegedly redundant definitions must now preserve the ~c[stobjs] declaration
  as well as the formals, body, guard and type declarations.  We thank
  Vernon Austel for pointing out this problem.

  It was possible to get a raw Lisp error by introducing a ~ilc[local]ly defined
  function with ~il[guard] verification inhibited and then subsequently
  introducing the same definition non-locally without that inhibition.  The
  following example will clarify.
  ~bv[]
  (encapsulate nil
    (local
      (defun foo (x) (declare (xargs :guard t :verify-guards nil)) (car x)))
    (defun foo (x) (declare (xargs :guard t)) (car x)))
  ; The following causes a raw lisp error because ACL2 runs the Common Lisp
  ; definition of foo, because it thinks that foo's guard of t was verified.
  (thm (equal (foo 3) xxx))
  ~ev[]
  Thanks to Jared Davis for bringing this problem to our attention.  We are
  particularly grateful to Jared because his example exploited this bug by
  applying it to a function defined using ~ilc[mbe] (introduced in this same
  version, 2.8), in order to prove ~c[nil]!

  The sort of error message shown below can legitimately occur when certifying
  a book in a certification world where there was an ~ilc[include-book] command
  with a relative pathname (~pl[pathname]).  However, it was occurring more
  often than necessary.  This has been fixed.
  ~bq[]
  ACL2 Error in (CERTIFY-BOOK \"foo\" ...): The certification world has
  include-book commands for book \"bar\" that correspond to different full
  pathnames, namely \"/u/dir1/bar\" and \"/u/dir2/bar\".  ACL2 cannot currently
  certify a book in such a world.  To work around this problem, use an absolute
  pathname for at least one of these books (see :DOC pathname).~eq[]

  Bugs were fixed in ~ilc[with-output], in particular related to the use of
  values ~c[:all].  Also, documentation for ~c[with-output] has been improved.
  Thanks to Vernon Austel for pointing out the bugs.

  Fixed a lisp error occurring when ~c[bash] proof-checker command was given
  illegal syntax, e.g., ~c[(bash ((\"Goal\" :in-theory (enable binary-append))))]
  instead of ~c[(bash (\"Goal\" :in-theory (enable binary-append)))].

  We added an appropriate guard to ~ilc[find-rules-of-rune], which will avoid
  hard lisp errors when this function is called on non-~il[rune] arguments.
  Thanks to Eric Smith for pointing out this issue.

  It was possible for a redundant ~ilc[include-book] form
  (~pl[redundant-events]) to leave a ~il[command] in the ACL2 logical
  ~il[world] and to cause (re-)loading of a compiled file.  These behaviors
  have been fixed.  In particular, if ~c[book1] has already been included in
  the current ACL2 ~il[world] and ~c[(include-book \"book1\")] occurs in
  ~c[book2], then the compiled file for ~c[book1] will not be loaded again when
  ~c[book2] is included.  Thanks to Dave Greve for bringing our attention to
  these problems, and to Eric Smith for bringing up a special case earlier
  (where \"//\" occurred in the book name).

  The summary printed at the end of a proof had not listed ~c[:]~ilc[induction]
  rules used in a proof.  This has been corrected.

  The use of proof trees in emacs redefined `~c[control-c control-c]' in such a
  way that in telnet mode, the telnet session was interrupted and perhaps could
  not be continued.  This has been fixed.

  Source function ~c[load-theory-into-enabled-structure] contained a
  guard-violating call of ~ilc[compress1].  Thanks to Vernon Austel for
  bringing this problem to our attention; even though this bug was benign
  (as he pointed out), we like keeping the source code free of guard
  violations.

  A number of proof-checker atomic macros caused a hard error when all goals
  have already been proved.  This has been fixed.  Thanks to John Erickson for
  sending an example of the issue.

  A bug has been fixed in ~ilc[add-match-free-override].  Formerly, a
  ~ilc[table] ~il[guard] violation occurred when calling
  ~ilc[add-match-free-override] more than once with first argument other than
  ~c[:clear].

  Defininitions of functions involving large constants could cause stack
  overflows.  This has been fixed, at least in some of the most egregious
  cases (by making a source function ~c[fn-count-evg] tail-recursive).  Thanks
  to Jared Davis for bringing this problem to our attention.

  Evaluation of computed hints could cause stack overflows.  This has been
  fixed.  Thanks to Eric Smith for bringing this problem to our attention.

  Evaluation of ~c[:]~ilc[monitor] on ~c[:]~ilc[definition] ~il[rune]s is now
  fast even if the specified function is part of a very large
  ~ilc[mutual-recursion] nest.  Thanks to Eric Smith for sending an example
  showing that this wasn't always the case.

  Fixed a bug in ~c[books/bdd/cbf.lisp] that was causing certification of
  distributed bdd books to fail when the connected book directory (~pl[cbd])
  differs from the current working directory.  Thanks to Scott Guthery for
  bringing this bug to our attention and supplying a helpful log.

  Duplicate rule names have been eliminated from warnings generated upon the
  use of enabled ~c[:]~ilc[rewrite] or ~c[:]~ilc[definition] rules.  Thanks to
  Eric Smith for pointing out this problem.

  The trace utilities (~pl[trace]), as modified for GCL and Allegro Common
  Lisp, had failed to show more than the first return value for so-called
  ``~c[*1*]'' functions (essentially, ~il[executable-counterpart] functions)
  when they were returning multiple values (via ~il[mv]).  This has been fixed.
  Thanks to Erik Reeber for pointing out this problem.
  Also, it is now possible to refer to ~c[arglist] in ~il[trace$] forms when
  ACL2 is built on GCL, not just when ACL2 is built on Allegro Common Lisp.

  Uses of ~ilc[hide] introduced during proofs by failed attempts to evaluate
  constrained functions (~pl[hide]) are now tracked, so that the ~il[rune]
  ~c[(:DEFINITION HIDE)] will show up in the summary.

  The following bug, introduced back in Version  2.7, has been fixed.  The bug
  applied only to GCL and may well not have affected anyone.  But the function
  proclamation computed by ACL2 for compilation usually had an output type of
  ~c[nil] where it should have been ~c[t].

  The macro ~ilc[gc$] had a bug exhibited when it was supplied one or more
  arguments.  This has been fixed.

  The macro ~ilc[defabbrev] broke when supplied a string and no documentation,
  e.g., ~c[(defabbrev foo () \"\")].  Thanks to Rob Sumners for noticing this
  problem and providing a fix, which we have incorporated.

  For ACL2 executables built on Allegro Common Lisp, a Lisp error occurred when
  ~ilc[trace$] was called on other than a defined function symbol.  Now ACL2
  prints a more useful error message.

  The proof-checker no longer accepts a ~c[(]~ilc[verify]~c[)] command when
  some function symbol in the original goal no longer exists in the current
  ACL2 logical ~il[world].  Thanks to John Erickson for bringing this issue
  to our attention.

  The function ~c[ld-redefinition-action] may now be called by the user.
  Thanks to Vernon Austel for suggesting that we remove this symbol from
  the list of so-called untouchables.

  The handling of free variables in hypotheses (~pl[free-variables]) of rewrite
  and linear rules had a bug that prevented some proofs from going through.
  Here is a simple example, essentially provided by Diana Moisuc, who we thank
  for bringing this issue to our attention.  The proof of the ~ilc[thm] below
  had failed, but now will succeed.  This particular bug prevented, for
  example, the ~c[:all] behavior from occurring when the first hypothesis of
  the rule does not have free variables.  NOTE:  Now that this bug has been
  fixed, you may find some proofs running much more slowly than before.  You
  can use ~ilc[accumulated-persistence] to locate rules that are slowing down
  your proofs because of excessive attention to free variables, and then
  execute ~ilc[add-match-free-override] for those rules (or, just change the
  rules themselves to specify ~c[:once] in the ~c[:]~ilc[rule-classes]).
  ~bv[]
  (defstub foo1 (* ) => *)
  (skip-proofs
   (defthm aux-foo1
     (implies (and (integerp a)
                   (integerp i)
                   (equal (foo1 0)  (list 0 i)))
              (equal (foo1 a) (list 0 (+ a i))))
     :rule-classes ((:rewrite :match-free :all))))
  (thm
   (implies (and (integerp i)
                 (integerp a)
                 (equal (foo1 0) (list 0 i)))
            (equal (foo1 a) (list 0 (+ a i)))))
  ~ev[]

  Formerly, creation of large arrays could cause an error in the underlying
  Common Lisp implementation without helpful messages for the user.  Now, we
  check Common Lisp restrictions on arrays and print a helpful error message if
  they are violated, namely: each dimension must be less than the value of
  Common Lisp constant ~c[array-dimension-limit], and the product of the
  dimensions must be less than the value of Common Lisp constant
  ~c[array-total-size-limit].  Thanks to Warren Hunt for bringing this issue to
  our attention.  Note:  this change also removes a former restriction of
  ~ilc[stobj] array fields to size smaller than 2^28-1, provided the underlying
  Lisp can support larger arrays.

  The ~il[default-hints] in the current logical ~il[world] were ignored by
  ~ilc[verify-guards].  This has been fixed.  Thanks to Jared Davis for
  pointing out this bug and sending a helpful example.

  The ~ilc[brr] mechanism has been cleaned up in order to avoid hard errors and
  infinite loops that can arrive when typing interrupts (~c[control-c]) or
  end-of-files (~c[control-d]) inside the ~ilc[brr] loop.  Thanks to Dave
  Greve, Olga Matlin, Eric Smith, and Serita Van Groningen for bringing this
  issue to our attention.  As a byproduct, if you type ~c[control-d] (or if
  inside emacs, ~c[control-c control-d]), you may now quit entirely out of ACL2
  and lisp (~pl[good-bye]) in some cases where you formerly would not have, for
  example when sitting at the ACL2 prompt (which formerly, in Allegro Common
  Lisp for example, would merely take you into raw Lisp rather than quitting
  everything).

  We have eliminated structural flaws in the HTML documentation pages that
  could make them unreadable in some browsers.  Thanks to Bill Young for
  bringing this issue to our attention and to Joe Hendrix for diagnosing the
  problem.

  The ~il[proof-checker] could run very slowly after many instructions in a
  given session.  This has been fixed; thanks to Art Flatau for bringing this
  problem to our attention.  (Implementation detail: We now keep tag-trees
  duplicate-free when we accumulate them into state.  This change could have
  minor speed advantages for some top-level proofs too, not just in the
  proof-checker.)

  The printing of accesses to stobjs using nth or update-nth has been done
  using symbolic constants since ACL2 Version_2.6.  However, there was a bug
  that prevented this feature from working for ~ilc[update-nth] except at a
  top-level call.  This has been fixed.  Thanks to Julien Schmaltz for bringing
  this problem to our attention.  For example, consider these events:
  ~bv[]
  (defstobj st field0 field1)
  (thm (equal (nth 1 (update-nth 0 17 st)) (car (cons xxx yyy)))
       :hints ((\"Goal\" :in-theory (disable nth update-nth))))
  ~ev[]
  Before the fix, the proof attempt of the above silly thm printed the
  following.
  ~bv[]
  (NTH 1 (UPDATE-NTH *FIELD0* 17 ST))
  ~ev[]
  After the fix, we instead see the following.
  ~bv[]
  (NTH *FIELD1* (UPDATE-NTH *FIELD0* 17 ST))
  ~ev[]

  It is now possible to certify and subsequently include ~il[books] that
  require guard-checking to be off.  For example, the book can contain the form
  ~c[(defconst *silly* (car 3))] even though ~c[3] fails to satisfy the guard
  of ~ilc[car].  Formerly, it was necessary to execute
  ~c[:]~ilc[set-guard-checking]~c[ nil] before a ~ilc[certify-book] or
  ~ilc[include-book] in order for such a form to be handled without error.
  Thanks to Hanbing Liu for bringing this problem to our attention.

  Fixed a ~il[proof-checker] bug that could cause probably cause strange error,
``Attempt to access the plist field''.  Thanks to Bill Young for bringing this
  problem to our attention.

  Fixed a ~il[proof-checker] bug that was failing to record applications of
  rewrite rules using the proof-checker's ~c[:rewrite] command, causing the
  proof summary to omit mention of that rule (for example, when using the
  proof-checker's ~c[:exit] command to generate an ~c[:instructions] hint).
  Thanks to Bill Young for pointing out this bug.

  Modernized some of the proof-tree emacs and infix printing stuff, thanks to
  suggestions made by Camm Maguire.

  ~/~/")

(deflabel note-2-8-new-functionality

; Not mentioned in detail below:

; Rob's stobj :inline mod moves the live stobj tests from the raw Lisp
; definitions to the *1* code.  The idea is not even to call the raw Lisp code
; with non-live stobjs.

  :doc
  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on New Functionality~/

  WARNING: You may find that ~c[control-d] (in emacs, ~c[control-c control-d])
  can throw you completely out of Lisp where it had not formerly done so.

  (CLISP and Allegro CL only) ACL2 now starts up inside the ACL2 loop ~-[] that
  is, ~c[(]~ilc[LP]~c[)] is executed automatically ~-[] when built on CLISP or
  Allegro CL.  This was already the case for GCL and CMUCL, and it still is not
  true for LispWorks.  Thanks to Joe Corneli for bringing the CLISP
  command-line option ~c[\"-i\"] to our attention, which led to this CLISP
  change and inspired reconsideration of how to do this for Allegro CL.

  Pete Manolios and Daron Vroon have changed the representation of ordinals in
  ACL2, defined algorithms for ordinal arithmetic, and created a library of
  theorems to reason about ordinal arithmetic.  We thank them for these nice
  contributions.  ~l[note-2-8-ordinals] for details, in particular, for how to
  preserve existing proofs that depend on the previous ordinal representation.

  Sometimes users create rules of class ~c[:]~ilc[rewrite] that cause an
  infinite loop in the ACL2 rewriter.  This has lead to Lisp stack overflows
  and even segmentation faults.  Now, the depth of calls of functions in the
  ACL2 rewriter is limited, and under user control.  ~l[rewrite-stack-limit].

  Macros ~ilc[mbe] (``must be equal'') and ~ilc[mbt] (``must be true'') have
  been introduced, which allow the user to attach fast executable definitions
  to (presumably slower) ~c[:]~ilc[logic] mode functions.  Thanks to Vernon
  Austel for a key idea.  Also provided is a macro ~ilc[defexec], which employs
  ~ilc[mbe] but enforces the requirement that the executable definition also
  terminates.  Thanks to Jose Luis Ruiz Reina for collaborating in the design
  and development of ~ilc[defexec], and for useful comments from a number of
  others as well in the development of ~c[mbe] including Joe Hendrix and Rob
  Sumners.

  Definitions have been added for functions ~ilc[rassoc-eq] and
  ~ilc[rassoc-equal], which are like ~ilc[rassoc] but use different tests
  and have different guards.  (Compare ~ilc[assoc-eq] and ~ilc[assoc-equal],
  which are in similar relation to ~ilc[assoc].)

  The user can now control multiple matching for free variables in hypotheses
  for ~c[:]~ilc[forward-chaining] rules, as has already been supported for
  ~c[:]~ilc[rewrite] and ~c[:]~ilc[linear] rules.  For ~c[:forward-chaining]
  rules, ``free variables'' are those in the hypotheses not bound by a given
  trigger term.  As for ~c[:rewrite] and ~c[:linear] rules, free-variable
  matching may be limited to the first successful attempt by specifying
  ~c[:match-free :once] with ~c[:forward-chaining] in the
  ~c[:]~ilc[rule-classes], and ~ilc[add-match-free-override] may be used to
  modify the behavior of an existing rule.  Thanks to Erik Reeber for most of
  the implementation of these new capabilities, as well as significant
  assistance with a corresponding new documentation topic
  (~pl[free-variables-examples-forward-chaining]).

  It is no longer necessary to specify ~c[(set-match-free-error nil)] in order
  to avoid errors when a rule with free variables in its hypotheses is missing
  the ~c[:match-free] field.  (This was already true during book certification,
  but now it is the case in interactive sessions as well.)

  The form ~c[(break-on-error)] causes, at least for most Lisps, entry into
  the Lisp debugger whenever ACL2 causes an error.  ~l[break-on-error].  Thanks
  to John Erickson for providing encouragement to provide this feature.

  A new ~ilc[table] has been provided so that advanced users can override the
  built-in ~c[untranslate] functionality.  ~l[user-defined-functions-table].

  The ~ilc[pstack] mechanism (formerly denoted ~c[checkpoints]) has been
  improved.  The ``process [prover] stack,'' or pstack, is automatically
  printed when proofs abort.  Evaluation of function calls on explicit
  arguments during proofs is now tracked.  Actual parameters are shown with
  ~c[(pstack t)] rather than formals.  Thanks to Bill Legato for
  suggesting the first two of these improvements and, in general, encouraging
  changes that make ACL2 easier to use.

  The ~ilc[defstobj] event is now allowed to take an ~c[:inline] argument,
  which can speed up execution.  Thanks to Rob Sumners for suggesting and
  implementing this new feature.

  Macro ~ilc[assert$] has been added in order to make it easy to write
  assertions in one's code.  Semantically, ~c[(assert$ test form)] is the same
  as ~c[form], but it causes a hard error (using ~ilc[illegal]) if ~c[test]
  evaluates to ~c[nil].

  Macro ~ilc[cw-gstack] no longer takes arguments for the gstack or ~ilc[state].
  However, it now takes a keyword argument (which is optional),
  ~c[:evisc-tuple], that can be used to control how it prints terms.  In
  particular, ~c[cw-gstack] abbreviates large terms by default, but
  ~c[(cw-gstack :evisc-tuple nil)] causes terms to be printed in full.
  Thanks to Robert Krug and Eric Smith for requesting this improvement.

  The advanced user now has more control over the evisceration of terms.
  ~l[ld-evisc-tuple], in particular the new paragraph on ``The printing of
  error messages and warnings.''

  The ~ilc[include-book] event now has an additional (optional) keyword,
  ~c[:dir].  The value of ~c[:dir] should be a keyword that is associated with
  an absolute directory pathname to be used in place of the current book
  directory (~pl[cbd]) for resolving the first argument of ~c[include-book] to
  an absolute pathname.  At start-up, the only such keyword is ~c[:system], so
  that for example ~c[(include-book \"arithmetic/top\" :dir :system)] will
  include the book ~c[\"arithmetic/top\"] under the ~c[\"books/\"] directory of
  your ACL2 installation.  But you can associate ``projects'' with keywords
  using ~ilc[add-include-book-dir], e.g.,
  ~c[(add-include-book-dir :my-project \"/u/smith/project0/\")].
  ~l[add-include-book-dir] and also ~pl[delete-include-book-dir] and
  ~pl[include-book].  Note: You will probably not find ~c[:dir :system] to be
  useful if the distributed books are not placed in the path of their original
  location, pointed to by ~c[:dir :system], which will often happen if the
  executable image is obtained from another site.  Also ~pl[include-book], in
  particular its ``soundness warning''.

  The printing of results in raw mode (~pl[set-raw-mode]) may now be partially
  controlled by the user:  ~pl[add-raw-arity].  Also, newlines are printed when
  necessary before the value is printed.

  For those using Unix/Linux `make': A ~c[cert.acl2] file can contain forms
  to be evaluated before an appropriate ~ilc[certify-book] command is invoked
  automatically (not included in ~c[cert.acl2]).

  Jared Davis has contributed a new set of books for ordered finite set theory
  to the standard distribution, ~c[books/finite-set-theory/osets-0.81/].  See
  the ~c[README] file in that directory.  Thanks, Jared.

  Robert Krug has contributed two related changes (thanks, Robert!) in support
  of stronger arithmetic reasoning.  First, one can now enable and disable
  nonlinear arithmetic with a ~c[:nonlinearp] hint, which will override the
  default provided by ~ilc[set-non-linearp] (initially, ~c[nil]).  ~l[hints].
  Second, ~il[computed-hints] can now have access to the ~c[HISTORY], ~c[PSPV],
  and ~c[CTX] variables of the waterfall, which (for example) allows the
  writing of a hint which will enable nonlinear arithmetic on precisely those
  goals that are ~c[stable-under-simplificationp].  ~l[computed-hints].

  Robert Krug has contributed a new set of arithmetic books to the standard
  distribution, ~c[books/arithmetic-3/].  See the ~c[README] file in that
  directory.  Thanks, Robert.

  ~/~/")

(deflabel note-2-8-proofs
  :doc
  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on Changes in Proof Engine~/

  ACL2 now prevents certain rewriting loops; ~pl[rewrite-stack-limit].

  During the computation of ~ilc[constraint]s for functional instantiation,
  ~c[(prog2$ term1 term2)] and ~c[(the type term2)] are now treated as
  ~c[term2].

  A change has been made in heuristics for controlling rewriting during proofs
  by induction.  Formerly, during induction proofs, ACL2 suppressed rewriting
  of certain ``induction hypothesis'' terms, and forced expansion of certain
  ``induction conclusion'' terms, until rewriting had stabilized.  This
  meddling with the rewriter is still turned off when rewriting has stabilized,
  but it is now turned off earlier once an ancestor has been through the
  rewriter and the current goal is free of ``induction conclusion'' terms.
  Thanks to Dave Greve and Matt Wilding for providing an example and associated
  analysis that led us to look for a heuristic modification.

  A change has been made in the heuristics for handling certain ``weak''
  ~il[compound-recognizer] rules when building contexts.  Those who want to dig
  deeply into this change are welcome to look at the code following the call of
  ~c[most-recent-enabled-recog-tuple] in the code for function
  ~c[assume-true-false] in the ACL2 sources.

  The handling of free variables in a hypothesis of a ~il[rewrite] rule
  (~pl[free-variables]) has been improved in the case that the hypothesis is of
  the form ~c[(equiv x y)], where ~c[equiv] is a known equivalence relation
  (~pl[equivalence]).  Previously, if the rewriter was attempting to rewrite
  the hypothesis ~c[(equiv x y)] of a rewrite rule, in a context where ~c[x']
  is an instance of ~c[x], then the rewriter could fail to notice a term
  ~c[(equiv x' y')] true in the current context where ~c[y'] is an instance of
  ~c[y], in the case that ~c[x'] precedes ~c[y'] in the ~ilc[term-order].  This
  has been remedied.  This improvement applies regardless of whether ~c[x],
  ~c[y], or (we believe) both are already fully instantiated in the present
  context.  Thanks to Joe Hendrix for bringing up an example and to Vernon
  Austel for providing another, simple example.

  A very minor change has been made to the rewriter in the case that an
  equality appears on the left-hand side of a ~c[:]~ilc[rewrite] rule.
  Formerly, when such an equality ~c[(equal x y)] was commuted to
  ~c[(equal y x)] in order for the rule to match the current term, then all
  equalities on the instantiated right-hand side of the rule were commuted,
  except for those occurring inside another equality.  The instantiated
  right-hand side is no longer modified.  It seems very unlikely that this
  change will cause proofs to fail, though we cannot completely rule out that
  possibility.

  We have modified how the ACL2 simplifier handles the application of a defined
  function symbol to constant arguments in certain cases, which we now
  describe.  As before, ACL2 attempts to simplify such a function application
  by evaluation, provided the ~c[:]~ilc[executable-counterpart] of the function
  is enabled.  And as before, if that evaluation fails due to a subroutine call
  of a constrained function (introduced by ~ilc[encapsulate]), ACL2 may wrap a
  call of ~c[hide] around this function application.  (~l[hide].)  But now,
  ACL2 attempts to apply definitions and rewrite rules in the case that this
  evaluation fails, and only if the resulting term is unchanged does ACL2 wrap
  ~ilc[hide] around this function application.  Thanks to Matt Wilding for
  bringing up the idea of this modification.

  The generation of \"Goal\" for recursive (and mutually-recursive) definitions
  now uses the subsumption/replacement limitation (default 500).
  ~l[case-split-limitations].

  Default hints now apply to hints given in definitions, not just theorems.
  ~l[default-hints].

  Thanks to Robert Krug for implementing the following two improvements
  involving linear arithmetic reasoning: linear arithmetic now uses the
  conclusions of ~ilc[forward-chaining] rules, and ~ilc[type-set] now uses a
  small amount of linear reasoning when deciding inequalities.

  ~/~/")

(deflabel note-2-8-rules
  :doc
  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on Changes in Rules, Definitions, and Constants~/

  The ~il[theory] ~ilc[minimal-theory] has been changed by adding the
  ~il[definition] ~il[rune] for ~ilc[mv-nth] to the theory.  A corresponding
  change has been made to the theory warning mechanism, which was failing to
  warn if the definition of ~c[mv-nth] is disabled, even though calls of
  ~c[mv-nth] can be expanded by special-purpose code in the rewriter.  Thanks
  to Serita Van Groningen for pointing out this problem with the theory warning
  mechanism.

  The ~ilc[defevaluator] event has been modified so that in the body of the
  evaluator function, to add a new case ~c[(ATOM X)] (returning ~c[nil]) has
  been inserted immediately after the case ~c[(EQ (CAR X) 'QUOTE)].  This is a
  no-op semantically but may speed up proofs.  Thanks to Warren Hunt for
  suggesting this change.

  A new form of ~c[:]~ilc[compound-recognizer] rule is now allowed:
  ~bv[]
  (if (fn x) concl1 concl2)
  ~ev[]
  This is equivalent to an existing form:
  ~bv[]
  (and (implies (fn x) concl1)
       (implies (not (fn x)) concl2))
  ~ev[]
  Thanks to Josh Purinton for bringing this to our attention.

  Rewrite rules ~c[realpart-+] and ~c[imagpart-+] have been added in order
  to simplify the ~ilc[realpart] and ~ilc[imagpart] (respectively) of a sum.
  They follow from a theorem ~c[add-def-complex] that equates a sum with
  the complex number formed by adding real and imaginary parts.  All three
  of these theorems may be found in source file ~c[axioms.lisp].  Thanks to
  Eric Smith for raising a question leading to these additions, as well as
  to Joe Hendrix and Vernon Austel for helpful suggestions.

  ~/~/")

(deflabel note-2-8-guards
  :doc
  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on Guard-related Changes~/

  All the guard-related changes may be found elsewhere; in particular,
  ~pl[note-2-8-bug-fixes].

  ~/~/")

(deflabel note-2-8-proof-checker
  :doc
  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on Proof-checker Changes~/

  Added new ~il[proof-checker] commands ~c[wrap1], ~c[wrap], and
  ~c[wrap-induct].  ~c[Wrap] replaces multiple goals by their conjunction:
  ~c[(wrap instr1 instr2 ...)] employs ~c[wrap1] so that the indicated
  instructions create only at most one new goal.  ~c[Wrap-induct] is a simple
  example of the use of ~c[wrap], so that induction creates only one goal (the
  conjunction of the base and induction steps).  ~c[Wrap1] can be used
  immediately after a prover call (~c[bash], ~c[prove], ~c[reduce], ~c[bdd], or
  ~c[induct]) to collapse the new goals into one.  ~l[proof-checker-commands].

  The ~il[proof-checker] command ~c[=] failed to work as expected when a
  governing ~c[IF]-test of the current term is T.  This has been fixed (by
  fixing source function ~c[conjuncts-of]).  Thanks to Yoann Padioleau for
  bringing this problem to our attention.

  The ~c[type-alist] command now takes optional arguments that control whether
  or not the governors and/or conclusion are used in computing the context
  that is printed (~pl[proof-checker-commands], specifically subtopic
  ~c[type-alist]).  Thanks to Rob Sumners for suggesting this improvement.

  The macro ~ilc[toggle-pc-macro] has always taken an optional second argument
  of ~c[atomic-macro] or ~c[macro].  However, this was not clearly documented,
  and those two symbols had to be in the ~c[ACL2] package.  Both of these
  problems have been remedied.  Thanks to John Erickson for bringing the lack
  of documentation of the second argument to our attention.

  ~/~/")

(deflabel note-2-8-system
  :doc

; Modified compile-uncompiled-defuns/compile-uncompiled-*1*-defuns to write
; comment to the top of TMP.lisp/TMP1.lisp saying that file can be deleted.
; (Matyas Sustik had asked about these files.)

  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on System-level Changes~/

  ACL2 now runs on OpenMCL, ``an opensourced Common Lisp implementation,
  derived from Digitool's Macintosh Common Lisp product.''  Thanks to Greg
  Wright and Robert Krug for doing most of the work for this port.

  When ~c[(]~ilc[LP]~c[)] is first executed, the underlying raw Lisp package
  will change to ~c[\"ACL2\"] (if that is not already the current package in
  raw  Lisp).  This is a minor change that will probably not be noticed, since
  up to now it has probably been the case that the ACL2 executable starts up
  with ~c[\"ACL2\"] as the underlying raw Lisp package.  But this change was
  made because we have been informed that ACL2 executables based on OpenMCL
  need not start up with ~c[\"ACL2\"] as the underlying raw Lisp package.

  ACL2 now runs on MCL 5.0.  Thanks to Pascal Costanza for updates to the
  instructions in file ~c[mcl-acl2-startup.lisp] and for an update to the
  ACL2 sources (parameter ~c[*compiled-file-extension*]).

  ~/~/")

(deflabel note-2-8-ordinals
  :doc
  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on Changes to the Ordinals~/

  Please ~pl[ordinals].~/~/")

(deflabel note-2-8-other

; Changed ev-fncall, ev, etc. so that we could have versions that take the
; world (see note below about user-defined-functions-table).

; Used memoization to speed up global-symbol and *1*-symbol by avoiding string
; concatentation.

; Eliminated all-new-flg/all-new-flag from the following, as it always had
; value nil (Robert Krug agrees with this change).
;   new-vars-in-pot-lst (formerly arg 3)
;   expanded-new-vars-in-pot-lst (formerly arg 3)
;   add-polys-and-lemmas1-nl (formerly arg 2)
;   add-polys-and-lemmas (formerly arg 2)
;   add-disjunct-polys-and-lemmas (formerly arg 3)
;   add-disjuncts-polys-and-lemmas (formerly arg 2)
;   add-terms-and-lemmas (formerly arg 4)

  :doc
  ":Doc-Section note-2-8

  ACL2 Version  2.8 Notes on Miscellaneous Changes~/

  Execution of ~ilc[table] events has been sped up in many cases by avoiding
  excessive consing.

  ACL2 now warns if ~c[:]~ilc[rewrite] (or ~c[:]~ilc[definition]) rules contain
  free variables on the right-hand side.  Thanks to Dave Greve for raising this
  issue.

  Emacs file ~c[emacs/emacs-acl2.el] has been updated to better comprehend the
  notion of the ``ACL2 shell'', which is the buffer to which ACL2 forms are
  written by commands defined in the above file.  Thus, command ~c[control-t e]
  has been modified always to write to the ACL2 shell (which is ~c[\"*shell*\"]
  by default), and the following new commands have been defined.
  ~bq[]
  o control-t c~nl[]
    Set the ACL2 shell to the current buffer.
  o control-t b~nl[]
    Change to the ACL2 shell.
  ~eq[]

  The commands ~c[:]~ilc[pl] and ~c[:]~ilc[pr] may now be given a macro name
  that corresponds via the ~c[macro-aliases-table] to a function name, so that
  for example ~c[:pl append] is treated the same as ~c[:pl binary-append].  A
  more interesting improvement, for ~c[:pl] only, is that ~c[:pl] may now take
  any term.  When ~c[:pl] is given a term other than a symbol, it will print
  all rewrite rules that match that term.  Thanks to David Russinoff, Robert
  Krug, and Bill Legato for getting this going.

  A new function, ~ilc[pkg-witness], returns a symbol in the given package.

  The installation instructions have been updated, for example to give more
  guidance on obtaining Lisp implementations and to mention the acl2-help
  mailing list.

  Jared Davis has suggested some symbols to be added to ~c[*acl2-exports*],
  and we have done so.  Thanks, Jared.
  ~bq[]
  o ~c[MFC] (used in ~ilc[syntaxp] and ~ilc[extended-metafunctions]; thanks
  also to Robert Krug for this one)
  o ~c[ID], ~c[CLAUSE], ~c[WORLD], and ~c[STABLE-UNDER-SIMPLIFICATIONP] (used
    in ~ilc[computed-hints])
  o ~ilc[SET-DEFAULT-HINTS]
  ~eq[]

  The command ~c[:]~ilc[pe] has been improved so that when the event is inside
  an included book, the path of included books (from the top-level book down to
  the one containing the event) is shown.  Thanks to Eric Smith (perhaps among
  others) for pointing out the utility of this improvement.

  A new release of the rtl library has been included: ~c[books/rtl/rel4/].
  See the ~c[README] file in that directory.

  ~/~/")

(deflabel |NOTE-2-8(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.8(r) (March, 2003) Notes~/

  The ~c[Makefile] has been modified by adding a new target, ~c[clean-links].
  This can be used in order to remove all soft links, which is useful if the
  directory is copied or moved to a new location or if there are file system
  changes that cause problems with link pathnames.

  ~/

  Please also ~pl[note-2-8] for changes to Version_2.8 of ACL2.
  ~/
  ")

(deflabel note-2-9

; BUG FIXES:
; Example for the soundness bug involving *1* package names:

; ============================== sub.lisp ==============================
;
; #|
; (defpkg "ACL2_*1*_MYPKG" ())
; (certify-book "sub" 1)
; |#
;
; (in-package "ACL2_*1*_MYPKG")
;
; (acl2::defun foo (x)
;              (acl2::declare (acl2::xargs :verify-guards acl2::t))
;              x)
;
; ============================== top.lisp ==============================
;
; #|
; (include-book "sub")
; (defpkg "MYPKG" ())
; (certify-book "top" 2)
; |#
;
; (in-package "ACL2")
;
; (defthm lemma1
;   (equal (acl2_*1*_mypkg::foo 3) 3)
;   :rule-classes nil)
;
; (defun mypkg::foo (x)
;   (cons x x))
;
; (defthm lemma2
;   (equal (acl2_*1*_mypkg::foo 3) '(3 . 3))
;   :rule-classes nil)
;
; (defthm ouch
;   nil
;   :rule-classes nil
;   :hints (("Goal" :use (lemma1 lemma2))))

; End of example related to *1* package names.

; Example related to soundness bug on local and redundancy checking:

; (encapsulate
;  ()
;  (defun foo (x)
;    (declare (xargs :mode :program))
;    (zp x))
;  (local (verify-termination foo))
;  (defun bar (x)
;    (foo x))
;  (defthm thm-1
;    (bar -1)
;    :rule-classes nil))
;
; (defthm thm-2
;   (not (bar -1))
;   :rule-classes nil)
;
; (defthm bad
;   nil
;   :rule-classes nil
;   :hints (("Goal" :use (thm-1 thm-2))))

; Here's a related example, showing how to get a world where a
; :common-lisp-compliant function, bar, calls an :ideal function, foo.

; ============================== abc.lisp ==============================
;
; (in-package "ACL2")
;
; (defun foo (x)
;   (declare (xargs :guard (consp x)))
;   (car x))
;
; ============================== abc-top.lisp ==============================
;
; (in-package "ACL2")
;
; (local (include-book "abc"))
;
; (defun foo (x)
;   (declare (xargs :guard (consp x) :verify-guards nil))
;   (car x))
;
; (defun bar (x)
;   (declare (xargs :guard (consp x)))
;   (foo x))

; End of example related to soundness bug on local and redundancy checking.

; Example related to soundness bug pertaining to safe-mode.

; ============================== bad1.lisp ==============================
;
; (in-package "ACL2")
;
; (defconst *c* '(((a b))))
;
; (defconst *d* *c*)
;
; (defmacro bad-macro ()
;   (list 'quote (union-eq-cars (list *c* *d*))))
;
; (defthm thm1
;   (equal (bad-macro)
;          '((a b)))
;   :rule-classes nil)
;
; ============================== bad2.lisp ==============================
;
; (in-package "ACL2")
;
; (defconst *c* '(((a b))))
;
; (defconst *d* '(((a b))))
;
; (defmacro bad-macro ()
;   (list 'quote (union-eq-cars (list *c* *d*))))
;
; (defthm thm2
;   (equal (bad-macro)
;          '((a b) (a b)))
;   :rule-classes nil)
;
; ============================== bad.lisp ==============================
;
; (in-package "ACL2")
;
; (include-book "bad1" :load-compiled-file nil)
;
; (include-book "bad2" :load-compiled-file nil)
;
; (defthm ouch
;   nil
;   :hints (("Goal" :use (thm1 thm2)))
;   :rule-classes nil)
;
; ======================================================================

; End of example related to soundness bug pertaining to safe-mode.

  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.9 (October, 2004) Notes~/

  ~bf[]
  ~st[TABLE OF CONTENTS.]~nl[]
  ==============================
  BUG FIXES.
  NEW FUNCTIONALITY.
  CHANGES IN PROOF ENGINE.
  GUARD-RELATED CHANGES.
  PROOF-CHECKER CHANGES.
  SYSTEM-LEVEL CHANGES.
  BOOK CHANGES.
  MISCELLANEOUS CHANGES.
  ==============================
  ~ef[]

  ~st[BUG FIXES.]

  We fixed a soundness bug due to a conflict between user-supplied package
  names and internal package names (obtained by prepending a Lisp constant,
  ~c[*1*-package-prefix*]) and user-supplied package names.  For example, the
  form ~c[(defpkg \"ACL2_*1*_MYPKG\" ())] is no longer legal.  Thanks to Robert
  Krug for asking a question that led directly to the discovery of this bug.

  We fixed a soundness bug that allows ~c[:]~ilc[logic] mode functions to call
  ~c[:]~ilc[program] mode functions.  The fix furthermore prevents functions
  with ~il[guard]s verified from calling functions with guards not verified.
  We had thought we already prevented all this, but there was a problem with
  the interaction of ~ilc[local] definitions and redundancy checking
  (~pl[redundant-events]).

  We fixed a soundness bug that could occur when built-in functions were called
  during macroexpansion (a hole in so-called ``safe-mode'').

  Fixed a minor bug in system functions ~c[genvar1] and ~c[genvar], where
  ~ilc[eq] had been used in place of ~ilc[eql].  This bug was discovered during
  the plugging of a hole in safe-mode, mentioned just above.

  We fixed handling of the ~c[:inline] keyword for ~ilc[defstobj], which
  previously could cause raw Lisp errors in OpenMCL and CMU Common Lisp.
  Thanks to John Matthews for bringing this issue to our attention.

  Calls of ~ilc[include-book] could result in a state for which some function
  definitions were not compiled that should have been.  The result could be
  performance degradation or even stack overflows.  This situation could arise
  in the following two ways.~bq[]

  o Inclusion of a book with an absolute pathname that differs from the
  absolute pathname at certification time, presumably because of the use of
  soft links.  Thanks to Bob Boyer and Warren Hunt for bringing a stack
  overflow to our attention that led us to this fix.

  o Large ~ilc[mutual-recursion] nests (more than 20 functions) are executed
  in a superior book.

  ~eq[]
  We fixed some performance bugs that can increase the speed of
  ~ilc[include-book] calls by a factor of close to 2.  Thanks to Eric Smith for
  asking if we could speed up ~ilc[include-book] processing; we have done so in
  the past, but primarily focusing on large ~ilc[mutual-recursion] nests (which
  have nothing in particular to do with the current improvements).  Also,
  thanks to Rob Sumners for a very useful remark early in the process that kept
  us from drawing an incorrect conclusion at that point.

  We fixed ~c[:]~ilc[pl] so that it can be run on a form that returns multiple
  values, which it could not do previouslly.  Thanks to Eric Smith for pointing
  out this problem.

  Fixed a bug in the Allegro ACL2 trace utility (~pl[trace$]) that was causing
  ``~c[10>]'' to be printed as ``~c[9>]'', ``~c[11>]'' to be printed as
  ``~c[10 >]'', ``~c[12>]'' to be printed as ``~c[11 >]'', and so on.

  Fixed a ~il[proof-checker] bug that was preventing the use of the ~c[DV]
  command (or a numerical command) on ~ilc[let] expressions.  Thanks to Bill
  Young for pointing out this problem.

  Fixed a bug in a comment on how to set ~c[ACL2_BOOKS_DIR] in the makefile.
  Thanks to Dave Greve for pointing out this problem.

  Fixed a potential soundness bug in the linear arithmetic routines.  Thanks to
  Jared Davis for noticing this problem and to Robert Krug for implementing the
  fix.  (Technical details: We had been assuming that polynomials were being
  normalized -- see the definition of good-polyp in linear-a.lisp -- but this
  assumption was false.)

  When the macro ~ilc[open-trace-file] is opened twice in succession, it now
  automatically closes the first trace output channel before opening another.

  It is now possible to use `make' to build ACL2 on Windows systems that
  support `make'.  Thanks to Pete Manolios and Mike Thomas for pointing out
  the problem, to Jared Davis and Mike for helping us to analyze the problem,
  and to Jared for testing the fix.

  Fixed a bug in the ~il[guard] of ~ilc[with-output] that was causing a
  needless guard violation.

  ~st[NEW FUNCTIONALITY.]

  The new events ~ilc[add-default-hints] and ~ilc[remove-default-hints] allow
  one to append to or subtract from the current list of default hints.  The
  event ~ilc[set-default-hints] continues to set the list of default hints,
  discarding the previous value of the ~ilc[default-hints].  Note that
  ~ilc[set-default-hints] is still ~ilc[local] to the ~ilc[encapsulate] or book
  in which it occurs, and ~ilc[add-default-hints] has the same property,
  although neither is implemented any longer using the
  ~ilc[acl2-defaults-table].  New events ~ilc[add-default-hints!],
  ~ilc[remove-default-hints!], and ~ilc[set-default-hints!] are the same as
  ~ilc[add-default-hints], ~ilc[remove-default-hints], and
  ~ilc[set-default-hints], respectively, except that the former three events
  are not ~ilc[local] to their enclosing ~ilc[encapsulate] or book.  Thanks to
  Jared Davis for suggesting these enhancements.

  OpenMCL's tracing routines have been modified in a similar manner as those
  of Allegro CL.  Thanks to Robert Krug for providing this enhancement.

  Guard-checking can now be caused to happen on recursive calls.
  See ``GUARD-RELATED CHANGES'' below for details.

  Advanced users can now inhibit compilation of so-called ``*1* functions''
  with the ~c[:comp] command; ~pl[comp].  Thanks to Rob Sumners for suggesting
  this enhancement.

  Added new legal argument ~c[hard?] for the ~ilc[er] macro, which is now
  documented.  ~l[er].  Thanks to Rob Sumners for a discussion leading to this
  change.  Also, the three legal first arguments to ~ilc[er] ~-[] ~c[hard],
  ~c[hard?], and ~c[soft] ~-[] may now be in any package (thanks to Jared Davis
  for bringing this issue to our attention).

  We have removed the requirement that for a rule's hypothesis
  ~c[(bind-free term var-list)], at least one variable must occur free in
  ~c[term].  For example, the expression ~c[(bind-free (bind-divisor a b) (x))]
  was legal because ~c[a] and ~c[b] occur free in ~c[(bind-divisor a b)]; but
  ~c[(bind-free (foo (bar)) (x))] was not legal.  The latter is no longer
  disallowed. (Technical note: this allows ~ilc[bind-free] to be used to create
  explicit substitutions in metafunction hypotheses.)

  The following two enhancements have been implemented for rules of class
  ~c[:]~ilc[meta].  Thanks to Eric Smith for requesting more control of
  reasoning with ~c[:]~ilc[meta] rules, which led to these enhancements, and to
  him and Robert Krug for helpful discussions.~bq[]

  o It is now possible to control backchaining in rules of class
  ~c[:]~ilc[meta] by providing a ~c[:backchain-limit-lst] argument, as was
  already allowed for rules of class ~c[:]~ilc[rewrite] and ~c[:]~ilc[linear].
  ~l[rule-classes].  However, unlike those other two rule classes, the value
  for ~c[:backchain-limit-lst] is prohibited from being a non-empty list; it
  must be either ~c[nil] or a non-negative integer.

  o (For advanced users.) It is now legal for hypothesis metafunctions to
  generate, in essense, calls of ~ilc[syntaxp] and ~ilc[bind-free], handled
  essentially as they are handled in hypotheses of ~c[:]~ilc[rewrite] and
  ~c[:]~ilc[linear] rules.  We say ``essentially'' primarily because both
  ~ilc[syntaxp] and ~ilc[bind-free] are actually macros, but hypothesis
  metafunctions must generate translated terms (~pl[term]).  The enterprising
  advanced user can call ~c[:]~ilc[trans] to see examples of translated terms
  corresponding to calls of ~ilc[syntaxp] and ~ilc[bind-free].

  ~eq[]A new command ~c[:]~ilc[trans!] has been added, which is like
  ~c[:]~ilc[trans] except that ~c[:]~ilc[trans!] ignored issues of
  single-threadedness.  ~l[trans!].  Thanks to Eric Smith for suggesting this
  addition.

  The ~c[:]~ilc[pf] command now works when the argument is the name of a macro
  associated with a function by ~il[macro-aliases-table].

  ~st[CHANGES IN PROOF ENGINE.]

  The simplifier has been changed slightly in order to avoid using
  ~il[forward-chaining] facts derived from a literal (essentially, a top-level
  hypothesis or conclusion) that has been rewritten.  As a practical matter,
  this may mean that the user should not expect forward-chaining to take place
  on a term that can be rewritten for any reason (generally function expansion
  or application of rewrite rules).  Formerly, the restriction was less severe:
  forward-chaining facts from a hypothesis could be used as long as the
  hypothesis was not rewritten to ~c[t].  Thanks to Art Flatau for providing an
  example that led us to make this change; see the comments in source function
  ~c[rewrite-clause] for details.

  The rewriter has been modified to work slightly harder in relieving
  hypotheses.  Thanks to Eric Smith for providing an example that inspired the
  following, which illustrates the issue.  Suppose we introduce functions
  ~c[foo] and ~c[bar] with the (non-~ilc[local]) properties shown below.
  ~bv[]
   (encapsulate
    (((foo *) => *)
     ((bar *) => *))

    (local (defun foo (x) (declare (ignore x)) t))
    (local (defun bar (x) (declare (ignore x)) t))

    (defthm foo-holds
      (implies x
               (equal (foo x) t)))
    (defthm bar-holds-propositionally
      (iff (bar x) t)))
  ~ev[]
  Consider what happens when ACL2's rewriter is used to prove the following
  theorem.
  ~bv[]
  (thm (foo (bar y)))
  ~ev[]
  With ACL2's inside-out rewriting, ~c[(bar y)] is first considered, but
  rewrite rule ~c[bar-holds-propositionally] does not apply because the context
  requires preserving equality, not mere Boolean (~c[iff]) equivalence.  Then
  the rewriter moves its attention outward and sees the term ~c[(foo (bar y))].
  It attempts to apply the rule ~c[foo-holds], in a context created by binding
  its variable ~c[x] to the term ~c[(bar y)].  It then attempts to relieve the
  hypothesis ~c[x] of rule ~c[foo-holds] in that context.  Before this change,
  ACL2 basically assumed that since rewriting was inside out, then ~c[(bar y)]
  had already been rewritten as much as possible, so the rewrite of ~c[x] in
  the aforementioned context (binding ~c[x] to ~c[(bar y)]) simply returned
  ~c[(bar y)], and the attempt to relieve the hypothesis of ~c[foo-holds]
  failed.  The change is essentially for ACL2's rewriter to make a second pass
  through the rewriter when the attempt fails to rewrite a variable to ~c[t],
  this time using the fact that we are in a Boolean context.  (We mention that
  source function ~c[rewrite-solidify-plus] implements this idea, for those who
  want to dig deeply into this issue.)  In our example, that means that the
  rewriter considers ~c[(bar y)] in a Boolean context, where it may apply the
  rule ~c[bar-holds-propositionally] to relieve the hypothesis successfully.

  When ~c[(]~ilc[set-non-linearp]~c[ t)] has been executed,
  ~il[non-linear-arithmetic] can now be applied in some cases for which it
  previously was not.  Thanks to Robert Krug for supplying this modification
  and to Julien Schmaltz for providing a motivating example.

  We modified the rewriter to avoid certain infinite loops caused by an
  interaction of the opening of recursive functions with equality reasoning.
  (This change is documented in detail in the source code, in particular
  functions ~c[rewrite-fncall] and ~c[fnstack-term-member].)  Thanks to Fares
  Fraij for sending us an example that led us to make this change.

  The ~c[:]~ilc[executable-counterpart] of function ~ilc[hide] is now disabled
  when ACL2 starts up.  This removes an anomoly, for example that
  ~bv[]
  (thm (not (equal 1 (* (hide 0) a))))
  ~ev[]
  succeeded while
  ~bv[]
  (thm (equal (foo (equal 1 (* (hide 0) a))) (foo nil)))
  ~ev[]
  failed.  Now both fail.

  The theory ~c[*s-prop-theory*] is no longer used by the ~i[proof-checker];
  it has been replaced by ~c[(theory ']~ilc[minimal-theory].  We have left
  the constant ~c[*s-prop-theory*] defined in the source code in support of
  existing books, however.  This change eliminates annoying theory warnings
  printed upon invocation of ~il[proof-checker] commands ~c[s-prop], ~c[sl],
  and ~c[split].

  Terms are now kept in an internal form that avoids calls of primitive
  functions (built-ins without explicit definitions; see code for ~c[cons-term]
  for details), in favor of the constants that result from evlaluating those
  calls.  So for example, the internal form for ~c[(cons 1 2)] is
  ~c[(quote (1 . 2))].  This change was made at around the same time as changes
  in support of ~ilc[bind-free]; see above.  One consequence is that the
  splitting of goals into cases (technically, source function ~c[clausify] and
  even more technically, source function ~c[call-stack]) has been modified,
  which can cause subgoal numbers to change.

  ~st[GUARD-RELATED CHANGES.]

  Guard-checking can now be caused to happen on recursive calls, where this was
  formerly not the case for ~c[:]~ilc[program] mode functions and, perhaps more
  important, for ~c[:]~ilc[logic] mode functions whose ~il[guard]s have not
  been verified.  Moreover, a warning is printed when ACL2 does not rule out
  the exclusion of guard-checking on recursive calls.  ~l[set-guard-checking].
  Thanks to David Rager for bringing this issue to our attention, and to Rob
  Sumners and the Univ. of Texas ACL2 seminar in general for their feedback.

  Guard violations are reported with less of the offending term hidden.  Thanks
  to Jared Davis for suggesting that we look at this issue.

  ~st[PROOF-CHECKER CHANGES.]

  We fixed the ~il[proof-checker] so that diving works as you might expect for
  a macro call ~c[(op a b c)] representing ~c[(binary-op a (binary-op b c))].
  In the past, if the current term was of the form ~c[(append t1 t2 t3)], then
  ~c[(DV 3)] (and ~c[3]) would dive to ~c[t3] even though the corresponding
  function call is ~c[(binary-append t1 (binary-append t2 t3))].  This is still
  the case, but now this behavior holds for any macro associated with a
  function in ~c[binop-table] (~pl[add-binop]).  Moreover, users can now
  write customized diving functions; ~pl[dive-into-macros-table], and also see
  ~c[books/misc/rtl-untranslate.lisp] for example calls to
  ~ilc[add-dive-into-macro].  Of course, the old behavior can still be obtained
  using the ~il[proof-checker]'s ~c[DIVE] command; ~pl[proof-checker-commands].

  The ~c[runes] command in the ~il[proof-checker] now shows only the ~il[rune]s
  used by the most recent primitive or macro command (as shown by ~c[:comm]),
  unless it is given a non-~c[nil] argument.  Also, ~il[proof-checker] command
  ~c[lemmas-used] has been added as, in essence, an alias for ~c[runes].

  (The following two items are also mentioned above under ``BUG FIXES.'')

  Fixed a ~il[proof-checker] bug that was preventing the use of the ~c[DV]
  command (or a numerical command) on ~ilc[let] expressions.  Thanks to Bill
  Young for pointing out this problem.

  The theory ~c[*s-prop-theory*] is no longer used by the ~i[proof-checker];
  it has been replaced by ~c[(theory ']~ilc[minimal-theory].  We have left
  the constant ~c[*s-prop-theory*] defined in the source code in support of
  existing books, however.  This change eliminates annoying theory warnings
  printed upon invocation of ~il[proof-checker] commands ~c[s-prop], ~c[sl],
  and ~c[split].

  ~st[SYSTEM-LEVEL CHANGES.]

  Fixed a problem with building ACL2 on CMUCL in some systems (source function
  ~c[save-acl2-in-cmulisp]).  Thanks to Bill Pase for bringing this to our
  attention.

  The installation instructions have been extended to include instructions for
  building on GCL in Mac OS X.  Thanks to Jun Sawada and Camm Maguire.

  Initial pre-allocation of space has been updated for GCL to reflect more
  current GCL executables (we considered GCL 2.6.1-38).  This can help avoid
  running out of memory for large ACL2 sessions.

  The main ~c[Makefile] has been replaced by ~c[GNUmakefile], in order to
  enforce the use of GNU `make'.  If you use another `make' program, you'll
  get an error message that may help you proceed.

  (GCL only) SGC is no longer turned on for GCL 2.6 sub-versions through 2.6.3
  if ~c[si::*optimize-maximum-pages*] is bound to ~c[T], due to an apparent
  issue with their interaction in those sub-versions.  Also, we have eliminated
  preallocation for all versions after 2.6.1 because GCL doesn't need it
  (comments are in source function ~c[save-acl2-in-akcl]).  Thanks to Camm
  Maguire for excellent GCL help and guidance, and to Camm and Bob Boyer for
  useful discussions.

  We have removed support for so-called ``small'' images.  Thus,
  ~c[:]~ilc[doc], ~c[:]~ilc[pe] and ~c[:]~ilc[pc], ~ilc[verify-termination],
  and other commands are fully supported in ACL2 saved images.  Because of this
  and other changes in the generation of the so-called ``*1*'' logical
  functions, related to guards (as described above in -GUARD-RELATED CHANGES'',
  and related to the discussion of safe-mode in ``BUG FIXES'' above), image
  sizes have increased substantially.

  We no longer ~c[time] or run ``~c[nice]'' the certification of individual
  books.  The file ~c[books/Makefile-generic] had done these by default, and
  some individual distributed and workshop book directories had ~c[Makefile]s
  that did so as well.  Thanks to Mike Thomas, who pointed out the lack of
  ~c[nice] on some Windows systems (and we decided on this simple solution).
  Overall targets in ~c[books/Makefile] still ~c[time] their runs by default,
  and the partiular ~c[time] program is now controlled by a ~c[Makefile]
  variable.

  Failures during ~c[make certify-books] or ~c[make regression] now show up
  in the log as ``~c[**CERTIFICATION FAILED**]'', regardless of the operating
  system (as long as it supports `make').  Formerly, one searched for
  ``~c[**]'' but this did not appear in openMCL runs.

  We have eliminated ``Undefined function'' warnings that could occur in
  OpenMCL.

  ~st[BOOK CHANGES.]

  Reconciled the definitions of ~c[firstn] in ~c[book/misc/csort.lisp],
  ~c[books/bdd/bdd-primitives.lisp],
  ~c[books/ordinals/ordinal-definitions.lisp], and
  ~c[books/data-structures/list-defuns.lisp].  Thanks to Ray Richards for
  bringing this issue to our attention.

  Distributed book ~c[books/misc/defpun] now can handle ~il[stobj]s where it
  did not previously.  Thanks to John Matthews for bringing this issue to our
  attention.

  The \"make\" variable ~c[COMPILE_FLG] in file ~c[books/Makefile-generic]
  formerly only had an effect if there was a ~c[cert.acl2] file present.  That
  oversight has been remedied.

  File ~c[\"books/arithmetic/certify.lsp\"] was missing a ~ilc[certify-book]
  command for ~c[\"natp-posp\"].  Thanks to John Cowles for noticing this
  deficiency and supplying a fix.  (This file is of use to those who want to
  certify the ~c[\"books/arithmetic/\"] books without using ~c[\"make\"].)

  A few small changes have been made to ~c[\"books/rtl/rel4\"].

  Small changes were made to books ~c[misc/symbol-btree] and
  ~c[misc/rtl-untranslate].  In particular, the definition of ~c[symbol-btreep]
  was strengthened.

  We made a minor fix to ~c[books/ordinals/e0-ordinal.lisp], adding
  ~c[(verify-guards ob+)] and hence ~c[(verify-guards ocmp)] as well.  This was
  necessitated by the fix prohibiting functions with guards verified from
  calling functions with guards not verified (see also the related discussion
  under ``BUG FIXES'' above).

  ~st[MISCELLANEOUS CHANGES.]

  Further sped up processing of large ~ilc[mutual-recursion] nests (extending
  what was done for Version_2.7), perhaps by a factor of two in some cases.

  As promised in Version_2.5 (~pl[note-2-5]), structured pathnames are no
  longer supported.  So for example, the argument to ~ilc[include-book] must
  now be a string constant.

  Some documentation has been improved, for ~il[stobj]s thanks to suggestions
  by John Matthews and much of the rest thanks to feedback from Eric Smith.

  The function ~ilc[current-package] is now available to users (it has been
  taken off the list of so-called ``untouchables'').  Thanks to Jared Davis for
  bringing this issue to our attention.

  The documentation for topic ~il[using-computed-hints-7] has been improved.
  Thanks to Doug Harper and Eric Smith for inspiring this improvement.

  We added several symbols to ~c[*acl2-exports*]: ~ilc[cw], ~ilc[er],
  ~ilc[intern$], ~ilc[set-case-split-limitations], and ~ilc[set-difference-eq].
  Thanks to Jared Davis for suggesting most of these.

  Now, a ~ilc[table] event that sets the value for a key,
  ~c[(table tbl key val :put)], is redundant (~pl[redundant-events]) when it
  does not change the value associated with an existing key of the table.  In
  particular, ~ilc[define-pc-macro] is now fully redundant when it does not
  change an existing ~il[proof-checker] macro-command definition.  Thanks to
  Bill Young for bringing the latter issue to our attention.

  The definitions of unused system functions ~c[ev-w] and ~c[ev-w-lst] have
  been deleted.

  ACL2 now prints a warning if a ~ilc[defpkg] event introduces a package name
  with lower-case letters, since there is opportunity for later confusion in
  that case.  Thanks to Frederic Peschanski for bringing this problem to our
  attention and Sandip Ray for encouragement.

  ACL2 now works in Version 19 of CMU Common Lisp.

  The function ~ilc[sys-call] has been modified so that for ACL2 built on
  Allegro Common Lisp in Unix or Linux, the existing environment is used.
  Thanks to Erik Reeber for bringing this issue to our attention.

  The function ~ilc[disabledp] can now be given a macro name that has a
  corresponding function; ~pl[macro-aliases-table].  Also, ~ilc[disabledp] now
  has a ~il[guard] of ~c[t] but causes a hard error on an inappropriate
  argument.

  ~/~/")

(deflabel |NOTE-2-9(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  2.9(r) (October, 2004) Notes~/

  No changes have been made for support of non-standard analysis, other than
  a minor modification or two in ~c[books/nonstd/] books.

  ~/

  Please also ~pl[note-2-9] for changes to Version_2.9 of ACL2.
  ~/
  ")

(deflabel note-2-9-1

; Changes that are too low-level for the user documentation:

; The name *soundness-related-warning-summaries* has been changed to
; *uninhibited-warning-summaries*, and "Compiled file" has been added to this
; list, along with "Tainted".  When we added "Tainted", we found that a
; "Compiled file" warning could be made without the "Compiled file" showing up
; in the event summary.  That appears to be because we circumvented the basic
; design of warnings in function unfound-compiled-file by locally binding state
; global 'ld-skip-proofsp to nil.

; Here is an example of how to exploit the soundness bug, mentioned below, in
; symbol-package-name.
;
; ++++++++++++++++++++++++++++++ bad-1.lisp ++++++++++++++++++++++++++++++
;
; (in-package "ACL2")
;
; (defthm lemma-1
;   (equal (symbol-package-name (intern (car (cons "X-EQUIV" xxx)) "COMMON-LISP"))
;          "LISP")
;   :rule-classes nil)
;
; ++++++++++++++++++++++++++++++ bad-2.lisp ++++++++++++++++++++++++++++++
;
; (in-package "ACL2")
;
; (defthm lemma-2
;   (equal (symbol-package-name 'COMMON-LISP::X-EQUIV)
;          "COMMON-LISP")
;   :rule-classes nil)
;
; ++++++++++++++++++++++++++++++ bad.lisp ++++++++++++++++++++++++++++++
;
; (in-package "ACL2")
;
; (include-book "bad-1")
;
; (include-book "bad-2")
;
; (defthm bad
;   nil
;   :hints (("Goal" :use lemma-1))
;   :rule-classes nil)
;
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

  :doc
  ":Doc-Section release-notes


  ACL2 Version  2.9.1 (December, 2004) Notes~/

  (GCL only) A bug in ~ilc[symbol-package-name] has been fixed that could be
  exploited to prove ~c[nil], and hence is a soundness bug.  Thanks to Dave
  Greve for sending us an example of a problem with ~ilc[defcong] (see below)
  that led us to this discovery.

  ACL2 now warns when ~ilc[defcong] specifies ~ilc[equal] as the first
  equivalence relation, e.g., ~c[(defcong equal iff (member x y) 2)].  The
  warning says that the rule has no effect because ~ilc[equal] already refines
  all other equivalence relations.  Formerly, this caused an error unless
  ~c[:event-name] was supplied (~pl[defcong]), and in fact the error was a
  nasty raw Lisp error on GCL platforms due to some mishandling of packages by
  ACL2 that has been fixed (see the paragraph about ~ilc[symbol-package-name]
  above).  Thanks to Dave Greve for sending a helpful example in his report of
  this problem.

  (GCL only) The build process was broken for GCL 2.6.0 (and perhaps some
  earlier versions), and has been fixed.  Thanks to Jose Luis Ruiz-Reyna for
  bringing this problem to our attention.

  (GCL only) We have increased the hole size to at least 20% of max-pages,
  which may eliminate some garbage collection at the expense of larger virtual
  memory (not larger resident memory or larger image).  Thanks to Camm Maguire
  for helpful explanations on this topic.

  We have clarified the ~il[guard] warning message that is printed during
  evaluation of recursively-defined functions whose ~il[guard]s have not been
  verified, for example:
  ~bv[]
    ACL2 Warning [Guards] in TOP-LEVEL:  Guard-checking may be inhibited
    on some recursive calls of executable counterparts (i.e., in the ACL2
    logic), including perhaps EVENLP.  To check guards on all recursive
    calls:
      (set-guard-checking :all)
    To leave behavior unchanged except for inhibiting this message:
      (set-guard-checking :nowarn)
  ~ev[]
  And, ACL2 no longer prints that message when the ~il[guard] was
  unspecified for the function or was specified as ~c[T].  Thanks to Serita
  Nelesen for bringing the latter issue to our attention.  Finally, ACL2 now
  prints such a warning at most once during the evaluation of any top-level
  form; thanks to Bill Young for pointing out this issue.

  The function ~ilc[verbose-pstack] has been enhanced to allow specified prover
  functions ~st[not] to be traced.  ~l[verbose-pstack].

  Added ~ilc[lp], ~c[wet], and ~ilc[set-non-linearp] to ~c[*acl2-exports*],
  and hence to the ~c[\"]~ilc[ACL2-USER]~c[\"] package.

  The distributed book
  ~c[books/arithmetic-3/bind-free/integerp.lisp] has been modified in order to prevent
  potential looping; specifically, the definition of function
  ~c[reduce-integerp-+-fn-1].  Thanks to Robert Krug for providing this change.

  A small improvement was made in the ~c[wet] failure message when the error
  occurs during translation to internal form.  Thanks to Jared Davis for
  pointing out the obscurity of some ~c[wet] error messages.

  We have improved ACL2's evaluation mechanism for the function ~c[bad-atom<=],
  which now is specified to return ~c[nil] if neither argument is a so-called
  ``bad atom'' (as recognized by function ~c[bad-atom]).  The following events
  had caused a hard error, for example.  (We're sorry that ~c[bad-atom] and
  ~c[bad-atom<=] are not documented, but we also consider it unlikely that
  anyone needs such documentation; otherwise, please contact the implementors.)
  ~bv[]
  (defun foo (x y) (declare (xargs :guard t)) (bad-atom<= x y))
  (defun bar (x y) (declare (xargs :guard t)) (foo x y))
  (thm (equal (bar 3 4) 7))
  ~ev[]
  We have also changed the guard on ~ilc[alphorder] to require both arguments
  to be atoms.

  For forms ~c[(local x)] that are skipped during ~ilc[include-book], or during
  the second pass of ~ilc[certify-book] or ~ilc[encapsulate], ACL2 had
  nevertheless checked that ~c[x] is a legal event form.  This is no longer the
  case.

  The ~il[proof-checker] now does non-linear arithmetic when appropriate.  It
  had formerly ignored ~ilc[set-non-linearp] executed in the ACL2 command loop.

  Incremental releases are now supported.  ~l[version] and {obsolete after
  Version  4.3} set-tainted-okp.  Thanks to Hanbing Liu for discovering a flaw
  in our original design.

  The pattern-matching algorithm for ~c[:]~ilc[rewrite] rules has been made
  slightly more restrictive, thanks to a suggestion and examples from Robert
  Krug.  For example, previously one could get an infinite loop as follows.
  ~bv[]
  (defstub foo (x) t)
  (defaxiom foo-axiom
    (equal (foo (+ 1 x))
           (foo x)))
  (thm (foo 0)) ; or replace 0 by any integer!
  ~ev[]
  That is because the term ~c[(foo 0)] was considered to match against the
  pattern ~c[(foo (+ 1 x))], with ~c[x] bound to ~c[-1].  While such matching
  is sound, it leads to an infinite loop since it allows ~c[foo-axiom] to
  rewrite ~c[(foo 0)] to ~c[(foo -1)], and then ~c[(foo -1)] to ~c[(foo -2)],
  and so on.  The fix is to insist that the new value, in this case ~c[-1], is
  no larger in size according to ~ilc[acl2-count] than the old value, in this
  case ~c[0].  Since that test fails, the match is considered to fail and the
  loop no longer occurs.  An analogous fix has been made for multiplication,
  where now we only match when the new term is still a non-zero integer.  That
  change avoids a loop here.
  ~bv[]
  (defstub foo (x) t)
  (defaxiom foo-axiom
    (equal (foo (* 2 x))
           (foo x)))
  (thm (foo 0)) ; or try (thm (foo 4))
  ~ev[]

  Added macro ~c[find-lemmas] in ~c[books/misc/find-lemmas.lisp] (see
  brief documentation there) for finding all lemmas that mention all function
  symbols in a given list.

  ~c[:Restrict] ~il[hints] now work for ~c[:]~ilc[definition] rules, though
  they continue to be ignored by the preprocessor and hence you may want to use
  ~c[:do-not '(preprocess)] with any restrict hints.  Thanks to John Matthews
  for pointing out the lack of support for ~c[:definition] rules in
  ~c[:restrict] hints.

  Some books have been updated.  In particular, there is a new directory
  ~c[books/workshops/2004/] in workshops distribution, for the 2004 ACL2
  workshop.  There is also a new version of Jared Davis's ordered sets library,
  formerly in ~c[books/finite-set-theory/osets-0.81/] but now in
  ~c[books/finite-set-theory/osets/].

  Fixed a bug in the (under-the-hood) raw Lisp definition of ~ilc[defchoose],
  which had been causing a warning in CMU Common Lisp.

  [Technical improvements related to the use of ``~c[make dependencies]'' for
  certifying distributed books:]~nl[]File ~c[books/Makefile-generic] now does a
  better job with ``~c[make dependencies],'' specifically with respect to
  handling ~c[*.acl2] files and handling ~ilc[include-book] commands with
  ~c[:dir :system].  Regarding the latter, suppose for example that book
  ~c[basic.lisp] contains the line:
  ~bv[]
  (include-book \"arithmetic/top-with-meta\" :dir :system)
  ~ev[]
  Then ~c[make dependencies] would generate the following line:
  ~bv[]
  basic.cert: $(ACL2_SRC_BOOKS)/arithmetic/top-with-meta.cert
  ~ev[]
  Thus, if ~c[:dir :system] is used with ~ilc[include-book], the corresponding
  ~c[Makefile] should define the variable ~c[ACL2_SRC_BOOKS].  A standard
  ~c[Makefile] header for a books directory could thus be as follows.
  ~bv[]
  # The following variable should represent the ACL2 source directory.  It is the
  # only variable in this Makefile that may need to be edited.
  ACL2_SRC = ../../../../../..

  ACL2_SRC_BOOKS = $(ACL2_SRC)/books
  include $(ACL2_SRC_BOOKS)/Makefile-generic
  ACL2 = $(ACL2_SRC)/saved_acl2
  ~ev[]
  Finally, the ``~c[-s]'' flag may now be omitted when running
  ``~c[make dependencies].''

  ~/~/")

(deflabel note-2-9-2

; Example for item below,
; "Guard-related warnings could be printed during proofs.  These warnings have
;  been eliminated."
;
;(defun my-test1 (expr)
;  (declare (xargs :guard (true-listp expr)
;                  :verify-guards nil))
;  (if (atom expr)
;      expr
;    (cons (my-test1 (car expr))
;          (my-test1 (cdr expr)))))
;
;(defun foo (n)
;  (declare (xargs :measure (acl2-count n)))
;  (if (zp n) n (foo (equal (list (my-test1 '(a b c)) (my-test1 '(a b c))) 17)) ))
;
;(thm (equal (list (my-test1 '(a b c)) (my-test1 '(a b c))) 17))

  :doc
  ":Doc-Section release-notes


  ACL2 Version  2.9.2 (April, 2005) Notes~/

  Also ~pl[note-2-9-1] for other changes since the last non-incremental release
  (Version_2.9).

  There was a bug in non-linear arithmetic (~pl[non-linear-arithmetic]) that
  caused the following error:
  ~bv[]
  ACL2 !>(include-book \"rtl/rel4/lib/top\" :dir :system)
  ....
  ACL2 !>(set-non-linearp t)
   T
  ACL2 !>(thm
   (implies (and (bvecp a 77)
                 (bvecp b 50))
            (bvecp (fl (/ (* a b) (expt 2 23)))
                   104))
   :hints ((\"Goal\" :in-theory (enable bvecp))))

  [Note:  A hint was supplied for our processing of the goal above.
  Thanks!]

  By the simple :definition BVECP, the :executable-counterparts of EXPT
  and UNARY-/ and the simple :rewrite rule ASSOCIATIVITY-OF-* we reduce
  the conjecture to

  Goal'
  (IMPLIES (AND (INTEGERP A)
                (<= 0 A)
                (< A 151115727451828646838272)
                (INTEGERP B)
                (<= 0 B)
                (< B 1125899906842624))
           (BVECP (FL (* A B 1/8388608)) 104)).


  HARD ACL2 ERROR in VARIFY:  This should not have happened.  The supposed
  variable, '1/8388608, is instead a constant.

  ACL2 !>
  ~ev[]
  Thanks to Robert Krug for providing a fix for the above error.

  Guard-checking was being inhibited (since v2-9) for calls of built-in
  primitives on explicit values, e.g., ~c[(car 3)].  This has been fixed.

  Guard-related warnings could be printed during proofs (this bug was
  introduced in Version_2.9.1).  These warnings have been eliminated.

  Compound-recognizer rules ~c[natp-compound-recognizer] and
  ~c[posp-compound-recognizer] are now built into ACL2 for predicates
  ~ilc[natp] and ~ilc[posp], and hence have been deleted from book
  ~c[natp-posp.lisp] (where they were called ~c[natp-cr] and ~c[posp-cr],
  respectively).

  The function ~c[file-clock-p], which recognizes a component of the ACL2
  ~ilc[state], is now defined using ~ilc[natp] instead of ~ilc[integerp].
  Thanks to Jared Davis for suggesting this change.  (Technical explanation
  about functions in ACL2 source file ~c[axioms.lisp]: With a ~c[file-clock] of
  -1, the call of ~c[make-input-channel] in ~c[open-input-channel] will create
  a channel that can't be closed; see the guard of ~c[close-input-channel].)

  (Allegro CL users only) Support is now provided for building an Allegro CL
  application, provided you have an Allegro CL dynamic runtime license.  (Our
  belief is that with such a license, many users can use the same application,
  rather than each user needing a separate license.)  See new GNUmakefile
  target ~c[allegro-app] and file ~c[build-allegro-exe.cl] for more
  information.

  The new home page now contains a link to a new page ~c[other-releases.html],
  which contains information about other ACL2 releases.  (This is in one's
  local home page, but may not show up on the central ACL2 home page until the
  next non-incremental release.)  Thanks to Warren Hunt for suggesting this
  addition.

  We thank Erik Reeber for suggesting a solution to output redirection using
  ~ilc[sys-call], which we have described at the end of its documentation.

  A new documentation topic fixes the flawed argument for conservativity of the
  ~ilc[defchoose] event that appears in Appendix B of Kaufmann and Moore's
  paper, ``Structured Theory Development for a Mechanized Logic'' (Journal of
  Automated Reasoning 26, no. 2 (2001), pp. 161-203).
  ~l[conservativity-of-defchoose].  Thanks to John Cowles and Ruben Gamboa for
  helpful feedback on drafts of this note.

  The solution to exercise 6.15 in ~c[books/textbook/chap6/solutions.txt] has
  been fixed.  Thanks to Aaron Smith for pointing out the problem.

  A new documentation topic ~il[defun-sk-example] gives a little more help in
  using ~ilc[defun-sk] effectively.  Thanks to Julien Schmaltz for presenting
  this example as a challenge.

  (GCL only) There is now a way to speed up GCL builds of ACL2, at the cost of
  perhaps a percent or so in performance of the resulting image.  Using
  `make' one supplies the following.
  ~bv[]
  LISP='gcl -eval \"(defparameter user::*fast-acl2-gcl-build* t)\"
  ~ev[]

  Various makefiles have been improved in several ways.
  ~bq[]
  (1) Parallel book certification, using GNU make's ~c[-j] option, can be used.

  (2) Book certifications now stops at the first failure if ~c[books/Makefile]
  or ~c[books/Makefile-generic] is used, and returns non-zero exit status.
  However, the various make targets in the ACL2 source directory
  (~c[regression], ~c[certify-books], etc.) still continue past failures unless
  you provide ~c[ACL2_IGNORE=' '] on the `make' command line.

  (3) The build process has been modified (file ~c[GNUmakefile]) so that it
  stops upon a failed compile or a failed initialization.

  (4) The automatic dependency generation (from ``~c[make dependencies]'' has
  been improved so that commands of the form ~c[(ld \"my-book.lisp\")] in
  ~c[.acl2] files cause the appropriate depedencies to be generated.~eq[]
  Thanks to comments from several users that led to the above Makefile
  improvements: Ray Richards, Doug Harper, and the Rockwell ACL2 users for (1)
  and (2) (and inspiring (4)), and David Rager for (2) and (3).  In particular,
  Doug Harper sent a replacement for the ~c[.date] mechanism, which was
  interfering with ~c[make -n]; so, these files are no longer written.

  A mechanism has been added for saving output.  In particular, you can now
  call ~ilc[ld] on a file with output turned off, for efficiency, and yet when
  a proof fails you can then display the proof attempt for the failed (last)
  event.  ~l[set-saved-output].  Another new command ~-[]
  ~pl[set-print-clause-ids] ~-[] causes subgoal numbers to be printed during
  proof attempts when output is inhibited.

  Documentation has been added for using ACL2's makefile support to automate
  the certification of collections of books.  ~l[books-certification-classic].

  Fixed a bug in ~ilc[sys-call-status] that was causing hard Lisp errors.

  Improved ~ilc[cw-gstack] to allow a ~c[:frames] argument to specify a range
  of one or more frames to be printed.  ~pl[cw-gstack].

  Fixed a bug in ~il[proof-checker] command ~c[forwardchain].  Thanks to
  Ming-Hsiu Wang for bringing this bug to our attention.

  We have provided a mechanism for saving an executable image.
  ~l[saving-and-restoring] and ~pl[save-exec].  We have eliminated obsolete
  functions ~c[note-lib] and ~c[make-lib].

  Modified the ~ilc[ground-zero] ~il[theory] so that it contains all of the
  built-in rules (in ACL2 source file ~c[axioms.lisp]).  It had formerly failed
  to include rules from some definitions and theorems near the end of
  ~c[axioms.lisp].

  A new event, ~ilc[set-enforce-redundancy], allows the enforcement of
  ~ilc[defthm], ~ilc[defun], and most other events during book development.
  ~l[set-enforce-redundancy].

  A bug has been fixed that had allowed ~ilc[deftheory] ~il[events] to cause a
  hard Lisp error when calling ~ilc[union-theories] on ill-formed theories
  after, for example:
  ~bv[]
  :set-guard-checking nil
  (in-theory (union-theories '((:rewrite no-such-rule))
                             (current-theory 'ground-zero)))
  ~ev[]
  The handling of ~il[guard] checking has been modified somewhat in a way that
  should only very rarely affect users.  (An ``Essay on Guard Checking'' in the
  ACL2 source code explains this point to anyone interested in implementation
  details.)

  (GCL ONLY) Removed the -dir setting in the ACL2 wrapper script for GCL.  This
  should generally have no effect for most users, but it eliminates a potential
  source of error down the road.

  Several interesting new definitions and lemmas have been added to the rtl
  library developed at AMD, and incorporated into ~c[books/rtl/rel4/lib/].
  Other book changes include a change to lemma ~c[truncate-rem-elim] in
  ~c[books/ihs/quotient-remainder-lemmas.lisp], as suggested by Jared Davis.

  The macro ~ilc[real/rationalp] may now be referred to in ~ilc[in-theory]
  ~il[events] and ~il[hints], thanks to a new ~ilc[add-macro-alias] event.
  Thanks to Jared Davis for this suggestion.

  ACL2 terms of the form ~c[(if p 'nil 't)] are now printed as ~c[(not p)],
  where in some setting they had been printed as ~c[(and (not p) t)].  Thanks
  to Robert Krug for this improvement.

  (GCL ONLY) Added profiling support, based heavily on code supplied by Camm
  Maguire.  See file ~c[save-gprof.lsp] for instructions.  Thanks to Camm, and
  also to David Hardin for inspiring this addition.

  Added support for preprocessing before printing (untranslating) a term.
  ~l[user-defined-functions-table], in particular the discussion of
  ~c[untranslate-preprocess].  Thanks to Jared Davis for inspiring this
  addition, and for providing a book that takes advantage of it
  (~c[books/misc/untranslate-patterns.lisp]).

  The documentation has been improved for explaining how ~il[rune]s are
  assigned; ~pl[rune].  Thanks to Robert Krug for pointing out inaccuracies in
  the existing documentation.

  ~/~/")

(deflabel note-2-9-3
  :doc

; Things that seem too minor to mention:

; (Intern (coerce (list #\Page) 'string) "ACL2") was printing as control-L
; without surrounding |..|, which cannot be read back in (at least in GCL).  A
; similar problem occurs with (Intern (coerce (list #\A #\Page #\B) 'string)
; "ACL2") So, added #\Page to *slashable-chars*.

; Eliminated a warning in CMUCL 19b due to missing arguments in error cases for
; source functions find-alternative-start1 and find-alternative-stop.

  ":Doc-Section release-notes

  ACL2 Version  2.9.3 (August, 2005) Notes~/

  Also ~pl[note-2-9-1] and ~pl[note-2-9-2] for other changes since the last
  non-incremental release (Version_2.9).

  We fixed a soundness bug that exploited the ability to define
  ~c[:]~ilc[program] mode functions that are improperly guarded, and then to
  use those functions in ~ilc[defconst] forms.  The fix is to evaluate
  ~ilc[defconst] forms using the same ``safe-mode'' that is used in
  macroexpansion (~pl[guards-and-evaluation]).  Here is a proof of ~c[nil] that
  succeeded in Allegro Common Lisp (but not, for example, GCL).  See also a
  long comment in source function ~c[defconst-fn] for an example that does not
  require the use of ~c[:set-guard-checking].
  ~bv[]
  :set-guard-checking nil ; execute before certifying the book below

  (in-package \"ACL2\")

  (encapsulate
   ()
   (local (defun f1 ()
            (declare (xargs :mode :program))
            (char-upcase (code-char 224))))
   (local (defconst *b* (f1)))
   (defun f1 ()
     (char-upcase (code-char 224)))
   (defconst *b* (f1))
   (defthm bad
     (not (equal *b* (code-char 224)))
     :rule-classes nil))

  (defthm ouch
    nil
    :hints ((\"Goal\" :use bad))
    :rule-classes nil)
  ~ev[]

  We fixed a soundness hole due to the fact that the \"LISP\" package does not
  exist in OpenMCL.  We now explicitly disallow this package name as an
  argument to ~ilc[defpkg].  Thanks to Bob Boyer and Warren Hunt for bringing
  an issue to our attention that led to this fix.

  ACL2 now requires all package names to consist of standard characters
  (~pl[standard-char-p], none of which is lower case.  The reason is that we
  have seen at least one lisp implementation that does not handle lower case
  package names correctly.  Consider for example the following raw lisp log
  (some newlines omitted).
  ~bv[]
  >(make-package \"foo\")
  #<\"foo\" package>
  >(package-name (symbol-package 'FOO::A))
  \"foo\"
  >(package-name (symbol-package '|FOO|::A))
  \"foo\"
  >
  ~ev[]
  Distributed book ~c[books/textbook/chap10/compiler], as well as workshop
  books in directory ~c[books/workshops/2004/cowles-gamboa/support/], were
  modified to accommodate the above change.

  Added ~c[newline], ~ilc[add-to-set-eql], ~c[the-fixnum], and ~c[the-fixnum!]
  to ~c[*acl2-exports*].  Thanks to Jared Davis for bringing these to our
  attention.

  Added a line to ~c[acl2.lisp] to support CMUCL running on Mac OSX, thanks to
  a suggestion from Fabricio Chalub Barbosa do Rosario.

  The executable scripts for saved ACL2 images now include ~c[$*], so that
  command-line arguments will be passed along.

  (For GCL profiling only) Fixed a colon (~c[:]) that should have been a
  semicolon (~c[;]) in file ~c[save-gprof.lsp].  Thanks to David Hardin for
  pointing out this bug.

  The documentation for ~c[:]~ilc[elim] rules has been expanded and improved,
  thanks to useful feedback from Hanbing Liu.

  Fixed a bug in the guard for function ~c[include-book-dir].

  For those who want to experiment with an alternate implementation of ~ilc[mv]
  and ~ilc[mv-let], there is now support for under-the-hood implementation of
  these in terms of raw Lisp functions ~c[values] and ~c[multiple-value-bind],
  respectively.  The regression suite has seen about a 10% speed-up in Allegro
  CL and about an 8% slowdown in GCL for builds with this change.  See the
  makefile (~c[GNUmakefile]) for examples of how to build ACL2 by including the
  feature, ~c[:acl2-mv-as-values].  Source file ~c[init.lsp] has been renamed
  to ~c[init.lisp] in support of this change (technical detail: otherwise GCL
  loads the init file too soon, before its ~c[-eval] argument is evaluated).
  Thanks to David Rager for inspiring this change, by pointing out the
  problematic use of globals by the existing ~ilc[mv] implementation from the
  standpoint of supporting parallel evaluation.  This capability is
  experimental: there is likely to be some remaining work to be done on it.

  A change related to the one just above is that we now limit the maximum
  number of arguments to any call of ~ilc[mv] to 32.  Thanks to Bob Boyer for
  raising a question that lead to this change.

  Eliminated some compiler warnings in OpenMCL.

  In the rtl library (~c[books/rtl/rel4/]), functions ~c[bits] and ~c[setbits]
  have had their ~il[guard]s improved (as they had been too restrictive,
  especially for ~c[setbits]).

  A new function ~ilc[time$] permits timing of forms, by using (under the hood)
  the host Common Lisp's ~c[time] utility.

  We fixed an infinite loop that could occur during destructor elimination
  (~pl[elim]).  Thanks to Sol Swords to bringing this to our attention and
  sending a nice example, and to Doug Harper for sending a second example that
  we also found useful.

  The method of speeding up GCL-based builds (~pl[note-2-9-2]) has changed
  slightly from Version_2.9.2.  Now, in the `make' command:
  ~bv[]
    LISP='gcl -eval \"(defparameter user::*fast-acl2-gcl-build* t)\"
  ~ev[]

  We improved the pretty-printer's handling of keywords.  For example, before
  this change one might see the following printed by ACL2.
  ~bv[]
  (MODIFY TH S :KEY1 VAL1 :KEY2
          (IF (IF X Y Z) AAAAAAAAAA BBBBBBB))
  ~ev[]
  Now, the above might print as follows.  Notice that we have avoided breaking
  after a keyword (~pl[keywordp]) that is preceded by other forms on the same
  line.
  ~bv[]
  (MODIFY TH S
          :KEY1 VAL1
          :KEY2 (IF (IF X Y Z) AAAAAAAAAA BBBBBBB))
  ~ev[]
  ~l[note-2-9-3-ppr-change] for a detailed discussion of this change.

  (GCL ONLY) Evaluation in a break is no longer inhibited by ACL2 when built on
  top of GCL, so GCL now matches other Common Lisps in this respect.

  For ACL2 built on most host Common Lisps, you will see the string
  ~c[[RAW LISP~]] in the prompt, at least at a break, to emphasize
  that one is inside a break and hence should probably quit from the
  break.  ~l[breaks].

  Jared Davis suggested improvements to lemmas ~c[len-update-nth] (in source
  file ~c[axioms.lisp]) and ~c[append-true-listp-type-prescription] (in
  ~c[books/meta/term-defuns.lisp]), which have been incorporated.  The former
  required a change in ~c[books/workshops] book
  ~c[2004/ruiz-et-al/support/q-dag-unification.cert], which has been made.

  The ~il[proof-checker] command ~c[rewrite] allows further binding of free
  variables in hypotheses, with new optional argument ~c[instantiate-free].
  Proof-checker command ~c[show-rewrites] (~c[sr]) gives corresponding
  additional information.  Documentation for these commands has been improved;
  ~pl[proof-checker-commands].  Thanks to John Matthews and Bill Young for
  suggestions and feedback leading to these improvements.

  Fixed downcase printing so that the package name of a symbol is also
  downcased.  For example, after execution of ~c[(defpkg \"FOO\" nil)] and
  ~c[(set-acl2-print-case :downcase)], ~c['foo::ab] will print back as the
  same, rather than as ~c['FOO::ab].

  It is now possible to control the output so that numbers are printed in
  binary, octal, or hex, though the default is still radix 10.
  ~l[set-print-base].  Note that in support of this change, built-in
  functions ~ilc[explode-nonnegative-integer] and ~c[explode-atom] now take an
  extra ~c[print-base] argument.  Different support for radix conversion may be
  found in a book newly contributed by Jun Sawada, ~c[books/misc/radix.lisp].

  Built-in axiom ~c[car-cdr-elim] is now only an ~c[:]~ilc[elim] rule.  It was
  formerly both an ~c[:elim] rule and a ~c[:]~ilc[rewrite] rule.  A new rule,
  ~c[cons-car-cdr], takes the place of the old ~c[:rewrite] rule, but is
  instead a hypothesis-free rule that can cause a case split (see source file
  ~c[axioms.lisp]).  Thanks to Jared Davis for suggesting this change.

  Lemmas about ~ilc[alphorder] (~c[alphorder-reflexive], ~c[alphorder-transitive],
  ~c[alphorder-anti-symmetric], and ~c[alphorder-total]) are now available.
  (They had been ~ilc[local] in source file ~c[axioms.lisp].)  Thanks to Serita
  Nelesen for bringing this issue to our attention.

  ACL2 has, for some time, printed a space in the event summary after the open
  parenthesis for a ~ilc[defthm] event, in order to ease backward searching for
  the original form, for example ~c[(defthm bar ...)]:
  ~bv[]
  Form:  ( DEFTHM BAR ...)
  ~ev[]
  The intention was that this extra space should be printed for every event
  form; but it was missing in some cases, for example, for ~ilc[verify-guards].
  This has been fixed.

  In analogy to ~ilc[include-book], now ~ilc[ld] takes the (optional) keyword
  argument ~c[:dir].  Thanks to Jared Davis for providing an implementation of
  this feature and to Eric Smith and Jeff Marshall for requesting this feature.

  We fixed a bug in ~ilc[include-book] that could cause an error when
  redefinition is on, for example:
  ~bv[]
  (set-ld-redefinition-action '(:warn! . :overwrite) state)
  (include-book \"/u/acl2/books/arithmetic/top\")
  ~ev[]

  The behavior of ~ilc[include-book] now matches the documentation: handling of
  compiled files for uncertified ~il[books] will follow the same rules as for
  certified books.  In particular, if you create an object file in raw Lisp for
  some book, then including that book will load that object file.  Thanks to
  Jared Davis for bringing this issue to our attention.

  New documentation explains the interaction of redefinition and redundancy.
  ~l[redundant-events] ~-[] the ``Note About Unfortunate Redundancies'' is new.
  Thanks to Grant Passmore for providing examples that led us to write this
  additional documentation.

  Solutions to exercises in ``How To Prove Theorems Formally''
  (~url[http://www.cs.utexas.edu/users/moore/publications/how-to-prove-thms])
  are now available in distributed book ~c[books/misc/how-to-prove-thms.lisp].
  Also in that directory may be found a new book ~c[hanoi.lisp] that contains a
  solution to the Towers of Hanoi problem.

  ~/~/")

(deflabel note-2-9-3-ppr-change
  :doc
  ":Doc-Section note-2-9-3

  change in pretty-printing for ACL2 Version_2.9.3~/

  We have improved pretty-printing in ACL2 Version_2.9.3 to handle keywords a
  little differently.  To see a discussion of the basics of this change,
  ~pl[note-2-9-3].  In this note we describe it in considerable detail.~/

  Those who wish to understand the ACL2 pretty-printer's implementation can now
  find considerably more comments on it in the source code.  In this note, we
  do not focus on the implementation.  Rather, we motivate the change and show
  how the improved prettyprinter performs.

  Why do we want better keyword handling?  Imagine a macro that builds a new
  state from an old state by changing the values in the affected fields,
  leaving everything else unchanged.  One could write
  ~bv[]
  (modify th s :key1 val1 :key2 val2 :key3 val3)
  ~ev[]
  where the three keys identify fields in the state.

  To make it easier to read new concrete states, we may have a function that
  prints them ``relative'' to a given base state, expressing the new state as a
  modification of the given base state.  So we may find ourselves
  prettyprinting modify forms like that above.

  The previous prettyprinter will sometimes print the form above as follows.
  ~bv[]
  (modify th s :key1
          val1
          :key2 val2 :key3 val3)
  ~ev[]
  This can be unpleasant to read, because of the way ~c[:key1] and ~c[val1] are
  separated.  Here is an example of the old prettyprinter and the new one, both
  printing an expression from the ACL2 source code in a width of 40:
  ~bv[]
  Old:
  (ADD-TO-TAG-TREE
   'ASSUMPTION
   (MAKE
    ASSUMPTION :TYPE-ALIST TYPE-ALIST
    :TERM TERM :REWRITTENP REWRITTENP
    :IMMEDIATEP IMMEDIATEP :ASSUMNOTES
    (LIST
     (MAKE
          ASSUMNOTE :CL-ID
          NIL :RUNE RUNE :TARGET TARGET)))
   TTREE)

  New:
  (ADD-TO-TAG-TREE
       'ASSUMPTION
       (MAKE ASSUMPTION
             :TYPE-ALIST TYPE-ALIST
             :TERM TERM
             :REWRITTENP REWRITTENP
             :IMMEDIATEP IMMEDIATEP
             :ASSUMNOTES
             (LIST (MAKE ASSUMNOTE
                         :CL-ID NIL
                         :RUNE RUNE
                         :TARGET TARGET)))
       TTREE)
  ~ev[]
  Basically the change we made forces the prettyprinter to print each ~c[:key]
  on a new line unless they all fit on a single line.  So we would now get
  either
  ~bv[]
  (modify th s :key1 val1 :key2 :val2 :key3 val3)
  ~ev[]
  or
  ~bv[]
  (modify th s
          :key1 val1
          :key2 val2
          :key3 val3)
  ~ev[]
  Furthermore, we fixed it so that if ~c[val1] (say) is a big s-expression we
  may still print it on the same line as its key.  The old prettyprinter
  enforced the rule that if you wanted to print ~c[(foo a b)] and ~c[b] gets
  broken up into several lines, then it has to start on a new line.  Thus,
  we'd never print
  ~bv[]
  (foo a (bbb
          (mum x)))
  ~ev[]
  but would print instead
  ~bv[]
  (foo a
       (bbb
        (mum x)))
  ~ev[]
  Now, if a is a keyword, we can print the first way.

  So here are some nice examples of prettyprinted keyword forms.  All of these
  are printed for a page of width 40.
  ~bv[]
  <--            40 chars               ->
  xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

  (MODIFY TH S :KEY1 V1 :KEY2 V2)

  (MODIFY TH S :KEY1 V1 :KEY2 V2 :KEY3 V3)

  (MODIFY TH S1                               ; Because of the extra char
          :KEY1 V1                            ; in S1 the flat size exceeds
          :KEY2 V2                            ; 40 and we break it.
          :KEY3 V3)
  ~ev[]
  The old ppr would have printed this as:
  ~bv[]
  (MODIFY
       TH S1 :KEY1 V1 :KEY2 V2 :KEY3 V3)
  ~ev[]
  Returning to new examples:
  ~bv[]
  <--            40 chars               ->
  xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

  (MODIFY TH S
          :KEY1 (IF (IF X Y Z) AAAA BBBB)
          :KEY2 VAL2
          :KEY3 VAL3)
  ~ev[]
  Now we extend ~c[AAAA] and ~c[BBBB] by one char each, so it would overflow
  the right margin if printed as above, and we get:
  ~bv[]
  <--            40 chars               ->
  xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

  (MODIFY
       TH S
       :KEY1 (IF (IF X Y Z) AAAAX BBBBX)
       :KEY2 VAL2
       :KEY3 VAL3)
  ~ev[]
  If we make these names even longer we force the value off the line containing
  ~c[:key1]:
  ~bv[]
  (MODIFY
       TH S
       :KEY1
       (IF (IF X Y Z) AAAAXXXXX BBBBXXXXX)
       :KEY2 VAL2
       :KEY3 VAL3)
  ~ev[]
  Here are some examples from the ACL2 source code, printed in 40 characters:
  ~bv[]
  (DEFTHM
   ALPHORDER-ANTI-SYMMETRIC
   (IMPLIES (AND (NOT (CONSP X))
                 (NOT (CONSP Y))
                 (ALPHORDER X Y)
                 (ALPHORDER Y X))
            (EQUAL X Y))
   :HINTS
   ((\"Goal\"
     :IN-THEORY
     (UNION-THEORIES
      '(STRING< SYMBOL-<)
      (DISABLE
         CODE-CHAR-CHAR-CODE-IS-IDENTITY))
     :USE
     ((:INSTANCE SYMBOL-EQUALITY (S1 X)
                 (S2 Y))
      (:INSTANCE BAD-ATOM<=-ANTISYMMETRIC)
      (:INSTANCE
           CODE-CHAR-CHAR-CODE-IS-IDENTITY
           (C Y))
      (:INSTANCE
           CODE-CHAR-CHAR-CODE-IS-IDENTITY
           (C X)))))
   :RULE-CLASSES
   ((:FORWARD-CHAINING
     :COROLLARY
     (IMPLIES
        (AND (ALPHORDER X Y)
             (NOT (CONSP X))
             (NOT (CONSP Y)))
        (IFF (ALPHORDER Y X) (EQUAL X Y)))
     :HINTS
     ((\"Goal\"
       :IN-THEORY (DISABLE ALPHORDER))))))
  ~ev[]
  Here is that same one, printed in a width of 60.
  ~bv[]
  (DEFTHM
   ALPHORDER-ANTI-SYMMETRIC
   (IMPLIES (AND (NOT (CONSP X))
                 (NOT (CONSP Y))
                 (ALPHORDER X Y)
                 (ALPHORDER Y X))
            (EQUAL X Y))
   :HINTS
   ((\"Goal\"
       :IN-THEORY
       (UNION-THEORIES
            '(STRING< SYMBOL-<)
            (DISABLE CODE-CHAR-CHAR-CODE-IS-IDENTITY))
       :USE ((:INSTANCE SYMBOL-EQUALITY (S1 X)
                        (S2 Y))
             (:INSTANCE BAD-ATOM<=-ANTISYMMETRIC)
             (:INSTANCE CODE-CHAR-CHAR-CODE-IS-IDENTITY (C Y))
             (:INSTANCE CODE-CHAR-CHAR-CODE-IS-IDENTITY
                        (C X)))))
   :RULE-CLASSES
   ((:FORWARD-CHAINING
        :COROLLARY (IMPLIES (AND (ALPHORDER X Y)
                                 (NOT (CONSP X))
                                 (NOT (CONSP Y)))
                            (IFF (ALPHORDER Y X) (EQUAL X Y)))
        :HINTS ((\"Goal\" :IN-THEORY (DISABLE ALPHORDER))))))
  ~ev[]
  Just for comparison, here is the above printed in 60 columns by the old
  prettyprinter.
  ~bv[]
  (DEFTHM
   ALPHORDER-ANTI-SYMMETRIC
   (IMPLIES (AND (NOT (CONSP X))
                 (NOT (CONSP Y))
                 (ALPHORDER X Y)
                 (ALPHORDER Y X))
            (EQUAL X Y))
   :HINTS
   ((\"Goal\" :IN-THEORY
            (UNION-THEORIES
                 '(STRING< SYMBOL-<)
                 (DISABLE CODE-CHAR-CHAR-CODE-IS-IDENTITY))
            :USE
            ((:INSTANCE SYMBOL-EQUALITY (S1 X)
                        (S2 Y))
             (:INSTANCE BAD-ATOM<=-ANTISYMMETRIC)
             (:INSTANCE CODE-CHAR-CHAR-CODE-IS-IDENTITY (C Y))
             (:INSTANCE CODE-CHAR-CHAR-CODE-IS-IDENTITY
                        (C X)))))
   :RULE-CLASSES
   ((:FORWARD-CHAINING
         :COROLLARY
         (IMPLIES (AND (ALPHORDER X Y)
                       (NOT (CONSP X))
                       (NOT (CONSP Y)))
                  (IFF (ALPHORDER Y X) (EQUAL X Y)))
         :HINTS
         ((\"Goal\" :IN-THEORY (DISABLE ALPHORDER))))))
  ~ev[]

  Of course, given that you cannot tell for sure whether the keywords you're
  seeing are part of a keyword/value parameter list or part of some constant
  containing random keywords, the prettyprinter can't solve the problem
  perfectly.  We just tried to make it work nicely on well-formed keyword/value
  parameter lists.

  For example, here is a form printed by the each prettyprinter.
  ~bv[]
  Old:
  (MEMBER
   X
   '(:MONDAY
        :MON :TUESDAY :TUES :WEDNESDAY
        :WED :THURSDAY :THURS :FRIDAY
        :FRI :SATURDAY :SAT :SUNDAY :SUN))

  New:
  (MEMBER X
          '(:MONDAY :MON
                    :TUESDAY :TUES
                    :WEDNESDAY :WED
                    :THURSDAY :THURS
                    :FRIDAY :FRI
                    :SATURDAY :SAT
                    :SUNDAY :SUN))
  ~ev[]

  The new way is not how one would print it by hand!  But then, neither is the
  old way.~/")

(deflabel note-2-9-4

; The soundness bug "due to inadequate checking of :meta rules" is documented
; in chk-acceptable-rules.

; Things that seem too minor to mention:

; Added a force-output call to print-prompt by way of new macro
; with-output-forced.  This seemed necessary for SBCL.

; In load-acl2, replaced (eval '(setq state *the-live-state*)) with what we
; used to do only for cmulisp, namely (set 'state *the-live-state*).

; Made small fix to translate11's handling of synp, to avoid requiring
; quote-normal form but to insist that the QUOTE calls have one argument.

; The error message for guard violations was cleaned up: added a sentence for
; new users, added ev-fncall-guard-er-msg for code sharing, and added initial
; part for new users to :doc set-guard-checking.

; To see the change mentioned below for set-let*-abstractionp, try this:
; (defpkg "FOO" (union-eq *acl2-exports*
;                         *common-lisp-symbols-from-main-lisp-package*))
; (in-package "FOO")
; (defstub foo (x) t)
; (defstub bar (x) t)
; :set-let*-abstractionp t
; (thm (equal (foo (append x y)) (bar (append x y))))

; Added hard! severity value for er (not advertised).

; Replaced conjuncts-of by flatten-ands-in-lit.

; Modified books/workshops/2004/sumners-ray/support/Makefile to avoid having to
; remake success.txt when that is unnecessary.

; Regarding this item below:
;
; We fixed an inefficiency that could cause an ~ilc[ld] command to seem to hang
; at its conclusion.  Thanks to Sandip Ray for pointing out this problem.
;
; Here is a way to see the problem.  The change was to maybe-add-command-landmark.
; (include-book "arithmetic/top-with-meta" :dir :system)
; (ld '((u) (include-book "arithmetic/top-with-meta" :dir :system)))

; Added (type (signed-byte 29) col) declarations for parameter col in fmt0 and
; fmt1.

; Modified many warnings to say something like "A rule generated by FOO"
; instead of "The rule generated by FOO", in response to an email from John
; Cowles.  This may not really deal with the objection that it appears that
; more than one copy of the same warning can appear for a single event (for the
; case of more than one rule), but at least the new wording is more accurate,
; and much more of a change could require considerable effort.

  :doc

  ":Doc-Section release-notes

  ACL2 Version  2.9.4 (February, 2006) Notes~/

  Also ~pl[note-2-9-1], ~pl[note-2-9-2], and ~pl[note-2-9-3] for other changes
  since the last non-incremental release (Version_2.9).

  A soundness bug has been fixed that was due to inadequate checking of
  ~c[:]~ilc[meta] rules in the presence of ~ilc[local] ~il[events].
  Specifically, a ~c[local] ~ilc[defevaluator] event is insufficient for
  supporting a ~c[:meta] rule (an example is shown in source function
  ~c[chk-acceptable-rules]).  Thanks to Dave Greve and Jared Davis for bringing
  this bug to our attention, by sending a proof of ~c[nil] that exploited this
  bug.  The fix is to check legality of ~c[:meta] rules even when skipping
  proofs during an ~ilc[include-book] event or the second pass of an
  ~ilc[encapsulate] event.

  Fixed problem with parallel make for workshop books by adding a dependency
  line to ~c[books/workshops/2003/Makefile].

  Default hints (~pl[set-default-hints]) no longer prevent the use of
  ~c[:INSTRUCTIONS] (~pl[proof-checker]).  Thanks to Jared Davis for pointing
  out this problem.

  New functions ~ilc[remove-eq] and ~ilc[remove-equal] have been defined, in
  analogy to ~ilc[remove].  These two symbols have also been added to
  ~c[*acl2-exports*].  Thanks to David Rager for pointing out that
  ~c[remove-equal] was missing.  Moreover, the definitions of ~c[delete1-eq]
  and ~c[delete1-equal] have been eliminated.  Function ~c[remove1-eq], now in
  ~c[:]~ilc[logic] mode in source file ~c[axioms.lisp], serves in place of
  ~c[delete1-eq], with corresponding new function definitions for ~ilc[remove1]
  and ~ilc[remove1-equal].

  The symbol ~ilc[assert$] has been added to ~c[*acl2-exports*].  Thanks to
  Jared Davis for the suggestion.

  Added SBCL support.  Thanks to Juho Snellman for significant assistance with
  the port.  Thanks to Bob Boyer for suggesting the use of feature
  ~c[:acl2-mv-as-values] with SBCL, which can allow thread-level parallelism
  in the underlying lisp; we have done so when feature ~c[:sb-thread] is
  present.

  We have continued to incorporate suggestions for wording improvements in
  documentation and error messages.  Thanks to all who send these suggestions,
  especially to Eric Smith, who has suggested the vast majority of them.

  Made a small improvement to errors and warnings caused on behalf of
  ~il[set-enforce-redundancy], to indicate when an event of the same name
  already exists.

  Fixed a bug in ~c[books/misc/rtl-untranslate.lisp] that was causing a guard
  violation when adding a new entry for an existing key.

  Fixed a bug in translation to internal form that caused ~ilc[defun-sk] and
  ~ilc[defchoose] to have difficulties handling ignored variables in ~ilc[let]
  forms.  Thanks to Sandip Ray to bringing this issue to our attention with a
  helpful example.

  The form ~c[(push :acl2-mv-as-values *features*)] has been added in source
  file ~c[acl2-init.lisp] for SBCL and OpenMCL only, in order to support
  parallel execution (looking to the future...).

  Default-hints (~pl[set-default-hints]) were being ignored inside the
  ~il[proof-checker], but no longer.  Thanks to John Erickson for bringing this
  problem to our attention and providing a simple example of it.

  Modified the ~c[TAGS] ~c[\"make\"] target (specifically, function
  ~c[make-tags]) so that it is gracefully skipped if the ~c[etags] program is
  not found.  Thanks to David Rager for pointing out this issue.

  Sandip Ray has re-worked the supporting materials for his ACL2 Workshop 2003
  talk (originally with John Matthews and Mark Tuttle), to run in a few
  minutes.  The result is in ~c[workshops/2003/ray-matthews-tuttle/support/]
  and is included in the full ACL2 regression suite.  Thanks, Sandip.

  Debian releases of ACL2 had created superfluous ~c[.cert.final] files when
  certifying books.  This has been fixed; thanks to Jared Davis for noticing
  this problem.

  Jared Davis has pointed out that ``If you add a ~c[:backchain-limit-lst 0] to
  a rewrite rule whose hypotheses are all forced, then ACL2 `really assumes them'
  without trying to relieve them right there through rewriting.''  Relevant
  documentation has been added for ~c[:backchain-limit-lst]; ~pl[rule-classes].

  A new version of the rtl library has been included in ~c[books/rtl/rel5/].
  Thanks to David Russinoff for contributing hand proofs for the new lemmas,
  and to Matt Kaufmann for carrying out their mechanization.

  Fixed a bug in ~ilc[save-exec] that was failing to set the initial ~c[cbd]
  according to the current directory when starting up ACL2.  Thanks to Camm
  Maguire for bringing our attention to this problem.

  Variables introduced during ~c[let*] abstraction are now in the current
  package.  Thanks to Jared Davis for suggesting such a change.
  ~l[set-let*-abstractionp].

  It is now allowed for two definitions to be considered the same from the
  standpoint of redundancy (~pl[redundant-events]) when one specifies a
  ~c[:]~ilc[guard] of ~c[t] and the other has no explicit ~c[:guard] (hence,
  the guard is implicitly ~c[t]).  Thanks to Jared Davis for bringing this
  issue to our attention.

  (For users of ~c[emacs/emacs-acl2.el]) There have been a few enhancements to
  distributed file ~c[emacs/emacs-acl2. el] (skip this paragraph if you don't
  use that file):
  ~bq[]
  o ~c[Control-t q] continues to compare windows ignoring whitespace, but now,
     a prefix argument can be given to control case is also ignored (ignore case if
     positive, else use case).

  o ~c[Control-t Control-l] has been defined to be similar to ~c[Control-t l],
     except that proofs are skipped and output is suppressed.

  o ~c[Control-t u] has been defined to print, to the shell buffer, a
    ~c[:]~ilc[ubt!] form for the command containing the cursor.

  o Control-t Control-f buries the current buffer.

  o ~c[Meta-x new-shell] now puts the new shell buffer in ~c[shell-mode]
    (thanks to David Rager for noticing this issue).~eq[]

  Linear arithmetic has been modified so that we do not generate the equality
  ~c[(equal term1 term2)] from the pair of inequalities ~c[(<= term1 term2)]
  and ~c[(<= term2 term1)] in the case that we would have to ~ilc[force] both
  ~c[term1] and ~c[term2] to be ~ilc[acl2-numberp]s.  Thanks to Dave Greve for
  providing a motivating example and to Robert Krug for providing a fix.

  The event ~ilc[delete-include-book-dir] had not been allowed inside
  ~il[books] and ~ilc[encapsulate] forms.  This was an oversight, and has been
  fixed.

  Sandip Ray has contributed a new library of books to support proofs of
  partial and total correctness of sequential programs based on assertional
  reasoning, in ~c[books/symbolic/].  This work is based on the paper
  J. Matthews, J S. Moore, S. Ray, and D. Vroon, ``A Symbolic Simulation
  Approach to Assertional Program Verification,'' currently in draft form.
  In particular, the books include the macro ~c[defsimulate], which
  automatically transforms inductive assertion proofs of correctness of
  sequential programs to the corresponding interpreter proofs.  See the
  ~c[README] in that directory.

  We have changed the implementation of ~c[:dir :system] for ~ilc[ld] and
  ~ilc[include-book].  This change will not affect you if you build an ACL2
  executable in the normal manner, leaving in place the ~c[books/] subdirectory
  of the source directory; nor will it affect you if you download a GCL Debian
  binary distribution.  The change is that if environment variable
  ~c[ACL2_SYSTEM_BOOKS] is set, then it specifies the distributed books
  directory, i.e., the directory determined by ~c[:dir :system].  You may find
  it convenient to set this variable in your ACL2 script file (typically,
  ~c[saved_acl2]).  If it is set when you build ACL2, the generated script for
  running ACL2 will begin by setting ~c[ACL2_SYSTEM_BOOKS] to that value.
  Thanks to various people who have discussed this issue, in particular Jared
  Davis who sent an email suggesting consideration of the use of an environment
  variable, and to Eric Smith who helped construct this paragraph.  (Note that
  this use of ~c[ACL2_SYSTEM_BOOKS] replaces the use of ~c[ACL2_SRC_BOOKS]
  described previously; ~pl[note-2-9-1].)

  ACL2 now automatically deletes files ~c[TMP*.lisp] created during the build
  process and created by ~c[:]~ilc[comp].  If you want these to be saved,
  evaluate ~c[(assign keep-tmp-files t)] in the ACL2 loop or in raw Lisp.  The
  ~c[clean] target for the standard `make' process for certifying books
  (~pl[books-certification-classic]) will however delete all files ~c[TMP*.*].

  The ~c[TMP] files discussed just above now generally include the current process
  ID in their names, e.g., ~c[TMP@16388@1.lisp] instead of ~c[TMP1.lisp].
  Thanks to Bob Boyer for suggesting this measure, which will reduce the
  possibility that two different processes will attempt to access the same
  temporary file.

  Now, ~c[:]~ilc[pe] will print the information formerly printed by ~c[:pe!],
  slightly enhanced to work for logical names that are strings, not just
  symbols.  Thanks to Warren Hunt for leading us to this change by suggesting
  that ~c[:pe nth] print the definition of ~ilc[nth].

  We eliminated spurious warnings that could occur in raw mode in OpenMCL or
  CMUCL when ~il[stobj]s are present.  We thank Juho Snellman for pointing out
  the relevant bug and appropriate fix.

  ~c[Mfc-rw] now takes a third argument that can specify an arbitrary known
  equivalence relation; ~pl[extended-metafunctions].  Thanks to Dave Greve for
  discussions suggesting this improvement.

  A small modification to a symbol-reading function allows documentation string
  processing on Windows systems that use CR/LF for line breaks.  Thanks to
  William Cook for bringing this issue to our attention.

  The documentation has been improved on how to control the printing of ACL2
  terms.  ~l[user-defined-functions-table].  Thanks to Sandip Ray for asking a
  question that led to the example presented there.

  We fixed an inefficiency that could cause an ~ilc[ld] command to seem to hang
  at its conclusion.  Thanks to Sandip Ray for pointing out this problem.

  We checked that ACL2 runs under LispWorks 4.4.5, and have inhibited
  redefinition warnings.

  Two changes have been made on behalf of congruence-based reasoning.  Thanks
  to Dave Greve for examples and discussions that have led to these changes,
  and to Eric Smith and Vernon Austel, who also sent relevant examples.
  ~bq[]
  o When a call of the new unary function ~ilc[double-rewrite] is encountered
  by the rewriter, its argument will be rewritten twice.  This solves certain
  problems encountered in congruence-based rewriting.  Warnings for
  ~c[:]~ilc[rewrite] and ~c[:]~ilc[linear] rules will suggest when calls of
  ~ilc[double-rewrite] on variables in hypotheses are likely to be a good idea.
  ~l[double-rewrite].

  o Hypotheses of the form ~c[(equiv var (double-rewrite term))], where
  ~c[equiv] is a known ~il[equivalence] relation and ~c[var] is a free variable
  (~pl[free-variables]), will bind ~c[var] to the result of rewriting ~c[term]
  twice.  Previously, hypotheses of the form ~c[(equal var term)] would bind a
  free variable ~c[var], but the call had to be of ~c[equal] rather than of an
  arbitrary known equivalence relation.~eq[]

  The following improvements were made in support of ACL2 on top of OpenMCL.
  ~bq[]

  o New versions of OpenMCL that do not have ~c[:mcl] in Lisp variable
  ~c[*features*] will now work with ACL2.  Thanks to David Rager for bringing
  this issue to our attention.

  o Added support for OpenMCL 1.0 for 64-bit DarwinPPC/MacOS X, thanks to
  Robert Krug.

  o Fixed tracing in OpenMCL so that the level is reset to 1 even if there has
  been an abort.

  o Added support in OpenMCL for ~c[WET].

  o Incorporated suggestions from Gary Byers for printing the ``Welcome to
  OpenMCL'' prompt before initially entering the ACL2 loop and, and for setting
  useful environment variable ~c[CCL_DEFAULT_DIRECTORY] in the ACL2
  script.~eq[]

  Fixed a long-standing bug in forward-chaining, where variable-free hypotheses
  were being evaluated even if the ~il[executable-counterpart]s of their
  function symbols had been disabled.  Thanks to Eric Smith for bringing this
  bug to our attention by sending a simple example that exhibited the problem.

  Improved reporting by the ~il[break-rewrite] utility upon failure to relieve
  hypotheses in the presence of free variables, so that information is shown
  about the attempting bindings.  ~l[free-variables-examples-rewrite].  Thanks
  to Eric Smith for requesting this improvement.  Also improved the
  ~il[break-rewrite] loop so that terms, in particular from unifying
  substitutions, are printed without hiding subterms by default.  The user can
  control such hiding (``evisceration''); see :DOC
  ~c[set-brr-term-evisc-tuple].

  A new directory ~c[books/defexec/] contains books that illustrate the use of
  ~ilc[mbe] and ~ilc[defexec].  Thanks to the contributors of those books (see
  the ~c[README] file in that directory).

  The directories ~c[books/rtl/rel2] and ~c[books/rtl/rel3] are no longer
  distributed.  They are still available by email request.  (Subdirectory
  ~c[rel1/] supports some of the optional ~c[workshop/] books, so it is still
  distributed.)

  Added book ~c[books/misc/sticky-disable.lisp] to manage ~il[theories] that
  might otherwise be modified adversely by ~ilc[include-book].  Thanks to Ray
  Richards for a query that led to our development of this tool.

  The commands ~c[(exit)] and ~c[(quit)] may now be used to quit ACL2 and Lisp
  completely; in fact they macroexpand to calls of the same function as does
  ~ilc[good-bye] (which is now a macro).  Thanks to Jared Davis for suggesting
  the new aliases.  (OpenMCL-only comment:) These all work for OpenMCL even
  inside the ACL2 loop.

  The macro ~c[wet] now hides structure by default on large expressions.
  However, a new optional argument controls this behavior, for example avoiding
  such hiding if that argument is ~c[nil].  Thanks to Hanbing Liu for
  pointing out that ~c[wet] was not helpful for very large terms.

  We have fixed a bug in the forward-chaining mechanism that, very rarely,
  could cause a proof to be aborted needlessly with an obscure error message.
  Thanks to Jared Davis for sending us an example that evoked this bug.

  Fixed a bug that was causing proof output on behalf of
  ~c[:functional-instance] to be confusing, because it failed to mention that
  the number of constraints may be different from the number of subgoals
  generated.  Thanks to Robert Krug for pointing out this confusing output.
  The fix also causes the reporting of rules used when silently simplifying the
  constraints to create the subgoals.

  Fixed a bug in handling of leading ~c[./] in pathnames, as in:
  ~c[(include-book \"./foo\")].  Thanks to Jared Davis for bringing this bug to
  our attention.

  Made a small fix for handling of free variables of ~c[:]~il[forward-chaining]
  rules, which had erroneously acted as though a hypothesis
  ~c[(equal var term)] can bind the variable ~c[var].

  A small change has been made for ~c[:]~ilc[type-prescription] rules for
  hypotheses of the form ~c[(equal var term)], where ~c[var] is a free variable
  and no variable of ~c[term] is free (~pl[free-variables]).  As with
  ~c[:]~ilc[rewrite] and ~c[:]~ilc[linear] rules, we now bind ~c[var] to
  ~c[term] even if ~c[(equal u term)] happens to be known in the current
  context for some term ~c[u].  Also as with ~c[:rewrite] and ~c[:linear]
  rules, similar handling is given to hypotheses
  ~c[(equiv var (double-rewrite term))] where ~c[equiv] is a known
  ~il[equivalence] relation.

  We changed the handling of free variables in hypotheses of ~c[:]~ilc[rewrite]
  rules being handled by the ~il[proof-checker]'s ~c[rewrite] (~c[r]) command,
  in complete analogy to the change described just above for
  ~c[:]~ilc[type-prescription] rules.

  The installation instructions have been updated for obtaining GCL on a
  Macintosh.  Thanks to Robert Krug for supplying this information and to Camm
  Maguire for simplifying the process by eliminating the ~c[gettext]
  dependency.

  The macro ~ilc[comp] is now an event, so it may be placed in ~il[books].

  Previously, a ~ilc[save-exec] call could fail because of file permission
  issues, yet ACL2 (and the underlying Lisp) would quit anyhow.  This has been
  fixed.  Thanks to Peter Dillinger for bringing this problem to our attention.

  Jared Davis, with assistance from David Rager, has updated his ordered sets
  library, ~c[books/finite-set-theory/osets/].  See file ~c[CHANGES.html] in
  that directory.

  A new function, ~ilc[reset-kill-ring], has been provided for the rare user
  who encounters memory limitations.  ~l[reset-kill-ring].

  ~/~/")

(deflabel note-2-9-5

; Things that seem too minor to mention:

; Modified translate11 to complain about LOCAL in code.  Before this fix, the
; definition
; (defun foo (x state) (local (value x)))
; caused (foo 3 state) to cause a hard error in getprop.

; Removed "time nice" from books/defexec/dag-unification/Makefile (doesn't work
; in some environments, as I recall).

; Added targets d64fsl and all-d64fsl to books/Makefile*, to support
; compilation of already-certified books on 64-bit OpenMCL.

; Added initial bindings of acl2-raw-mode-p and raw-mode-restore-lst to nil (in
; *initial-global-table*), as suggested by Peter Dillinger.

; Fixed warning stack issue with print-summary, which caused new
; theory-invariant implementation to avoid popping warning stacks because of
; with-output calls.

; Inlined call of equivalence-relation-to-geneqv with something more
; appropriate, and elminated equivalence-relation-to-geneqv.

; Changed rewrite-with-lemmas so that :expand hints (from user or from
; induction) are always followed, without the being-openedp call that used to
; be there.

; The change for bodies (new 'def-bodies property) causes the following
; to use the latest definition for the body (e.g., affecting :expand
; hints).
; - non-rec-defun, called by deref for nu-rewriter
; - recursivep, called in rewrite-with-lemma and bdd-rules-alist1
; - expand-and-or, used by preprocess
; - non-recursive-fnnames, used only in warnings
; - proof-checker's expand command (hence also x and x-dumb commands)
; - induction heuristics that use controller-alist

; Modified extend-type-alist to add check for (not (fquotep term)), which had
; been assumed but was not necessarily true because of the call of
; extend-type-alist from extend-type-alist-with-bindings.

; Updated *current-acl2-world-key-ordering*, and comments about it.

; Replaced the use of proclaim by the use of declaim (thanks to Bob Boyer for
; the suggestion to consider this).

; Fixed typo in uncovered-equivs-alist, 'f instead of 'if, that could cause
; incorrect double-rewrite warnings.  But went beyond this and gave more
; appropriate special handling for 'if.

; Fixed obscure bug in maybe-push-undo-stack, which could happen with:
;
; :redef!
; (defun exit-boot-strap-mode () t)
;
; The problem was that we assumed that a *1* function is defined when its raw
; Lisp counterpart is defined.

; Added missing dependencies in books/ihs/Makefile.

; Changed chk-embedded-event-form so that it returns the expansion rather than
; the original form, in support of the implementation of make-event.

; Eliminated get-check-sums-lst and check-sum-file (dead code).

; Acl2-compile-file now loads the compiled file.

; Simplified *compiled-file-extension* according to a suggestion from Gary
; Byers.

; With Robert Krug's help, including arithmetic-3/bind-free/top will now print
; a message about turning on non-linear.

; Made slight improvement in ev-fncall-guard-er-msg so that we don't suggest
; set-guard-checking nil when the problem is safe-mode or stobjs.

; Documentation and comments have been changed, for the most part, so that the
; word ``multiple'' is used correctly.  Thanks to David Russinoff for pointing
; out this issue.

; Fixed getpid$ for clisp to use 'system::process-id rather than 'process-id.
; This puts the process id into the name of TMP* files produced by :comp t,
; which hadn't been done for clisp because of the getpid$ bug.

; Changed definition of macro state-global-let* (specifically, changed
; definition of supporting function state-global-let*-cleanup) to produce much
; more compact code for large numbers of bindings as in
; protect-system-state-globals.  Without that change, lispworks
; reported:
; **++++ Error in ACL2::PROTECTED-EVAL-WITH-PROOFS:
;   Function size 87365 is too large.

; Eliminated compiler note "Ignoring free ignore declaration" for mbe, for
; CMUCL and SBCL (had already done so for OpenMCL).

; Fixed a bug in make-include-books-absolute that caused an "Implementation
; error" when with-output occurs in a progn with an include-book in the
; certification world, as in:
; (progn (include-book "cowles/acl2-asg" :dir :system)
;        (with-output :off summary (defun abc (x) x)))

; Added a call to the garbage collector before saving in Allegro CL, CMUCL,
; SBCL, CLISP, and OpenMCL.  (There was already such a call in GCL and
; LispWorks.)  We saw a little performance increase and significant shrinkage
; of the saved image when we did this for Allegro CL.

  :doc
  ":Doc-Section release-notes

  Changes in Version  3.0 since Version  2.9.4~/

  Fixed a bug in ~ilc[cw-gstack] that was causing a hard error when attempting
  to report on a forced assumption.  Thanks to Jared Davis for pointing this
  out and sending an example that helped us to determine a fix.

  Added ~ilc[set-backchain-limit] to the set of legal ~il[events] that can be
  placed in ~ilc[encapsulate] forms and ~il[books].  Thanks to John Cowles for
  bringing this issue to our attention.

  Fixed a bug that broke ~c[wet].  Thanks to David Rager for bringing this
  bug to our attention.

  Guard verification now evaluates ground subexpressions (those with no free
  variables) when computing the guard conjecture for the body of a function.
  Thanks to Jared Davis for useful conversations leading to this change.
  ~l[verify-guards], in particular its ``Note on computation of guard
  conjectures and evaluation'' near the end of that topic, for more details.

  Added a warning when a ~ilc[theory-invariant] is redefined.  Thanks to Jared
  Davis for suggesting a warning in this case and providing an informative
  example.  Also, ~ilc[theory-invariant]s are now maintained more completely,
  as they are checked at the end of every event except for events executed on
  behalf of an ~ilc[include-book] or the second pass of an
  ~ilc[encapsulate].

  Fixed the handling of runic designators to match their specification
  (~pl[theories]), so that disabling the name of a ~ilc[defthm] event
  ~ilc[disable]s all rules generated for that event.

  (For those who do numerous builds using feature ~c[:acl2-mv-as-values],
  currently only OpenMCL and multi-threaded SBCL by default:) You can speed up
  builds by adding the following parameter to `make', under conditions
  described in ~c[GNUmakefile]: ~c[USE_ACL2_PROCLAIMS=:REUSE].

  Arranged that traced functions (~pl[trace$]) are automatically untraced when
  events are undone (for example ~pl[ubt]), at least for most underlying Common
  Lisp implementations.

  The macro ~ilc[defun-sk] now creates non-executable functions, which allows
  ~ilc[stobj]s to be used where they had previously been prohibited.  More
  generally, the user now has control over ~ilc[declare] forms to be used by
  the underlying ~ilc[defun]'d function; ~pl[defun-sk].  Thanks to Sandip Ray
  for pointing out the need for such a modification.

  ~c[:]~ilc[Definition] rules are now treated, at least by default, as truly
  first-class definitions.  In particular, ~c[:expand] ~il[hints] use the
  latest ~c[:]~ilc[definition] rule by default.  You may specify
  ~c[:install-body nil] to get the previous behavior of ~c[:definition] rules;
  ~l[definition], and you may choose a previously-installed ~c[:definition]
  rule to provide the current body; ~pl[set-body].  Also ~pl[rule-classes] for
  details of the ~c[:install-body] field, and ~pl[hints] to see a new ~c[:with]
  directive for controlling expansion.  The ~c[:with] directive for ~c[:expand]
  hints can even direct the use of a ~c[:]~ilc[rewrite] rule for expansion!
  Thanks to various people, including Sandip Ray and Rob Sumners, for
  discussions on the issue of the applicability of ~c[:definition] rules for
  ~c[:expand] ~il[hints].

  ~il[Constraint]s for functional instantiation now use the original definition
  rather than a simplified (``normalized'') version of it.

  Fixed a bug that caused the prompt to stay the same when guard-checking is
  off (~pl[set-guard-checking]) and raw-mode is changed (~pl[set-raw-mode]).

  Lemma names in directory ~c[books/ordinals] have been changed by replacing
  ~c[/\\] with ~c[&] and replacing ~c[\\/] with ~c[V].  We made this change
  because backslash is an escape character and hence disappears unless it is
  itself escaped.

  Fixed ~il[proof-tree] output so that failed non-proof events do not cause the
  proof-tree to be re-printed.  Thus for example, if you have already advanced
  the checkpoint marker, it will not be reset by subequent failed non-proof
  events.  Thanks to Pete Manolios and Peter Dillinger for bringing this bug to
  our attention.

  Fixed a bug that was preventing the printing of ~il[stobj] fields as
  constants instead of numbers in certain cases.  (Note that this bug only
  affected printing, not soundness.) Thanks to Eric Smith for bringing this
  problem to our attention and providing the following example (which now works
  fine).
  ~bv[]
  (defstobj st fld1 fld2)
  (in-theory (disable update-nth))
  (defund run (st)
    (declare (xargs :stobjs (st))) ;adding this didn't seem to help..
    st)

  ;works great; *fld1* prints as  *fld1*
  (thm (equal (update-nth *fld1* 'abc st)
              (car (cons x y))))

  ;*fld1* gets printed as 0, presumably because the call to run intervenes.
  (thm (equal (update-nth *fld1* 'abc (run st))
              (car (cons x y))))
  ~ev[]

  The macro ~ilc[progn] now allows the use of macros defined within its bodies
  even when at the event level, as illustrated by the following example.
  ~bv[]
  (progn (defmacro my-defun (&rest args)
           `(defun ,@args))
         (my-defun g (x) x))
  ~ev[]
  Thanks to Anna Slobodova for bringing this issue to our attention.  A related
  change is that all arguments of ~ilc[progn] must now be embedded event forms
  (~pl[embedded-event-form]), so use ~ilc[er-progn] instead if this is not the
  case.

  The change to ~ilc[progn] mentioned above also fixes a bug in handling
  ~il[local] events inside a ~ilc[progn] that is inside an ~ilc[encapsulate] or
  in a book.  For example, the following form formerly caused an error.
  ~bv[]
  (encapsulate
   ()
   (defun foo (x) x)
   (progn (local (defun bar (x) x))
          (defun abc (x) x)))
  ~ev[]

  We fixed two bugs in ~c[:]~ilc[puff] and ~c[:]~ilc[puff*].  The first,
  brought to our attention by Eric Smith (who we thank), caused a cryptic error
  message when puffing a command with no subsidiary stored events; try, for
  example, ~c[(encapsulate () (value-triple 3))].  The second was due to a
  failure to restore the ~ilc[acl2-defaults-table].  Suppose for example that
  we have certified the book ~c[foo.lisp], which contains
  ~c[(]~ilc[program]~c[)] followed by some definitions and/or theorems.  Now
  suppose we start ACL2 and execute the following.
  ~bv[]
  (include-book \"foo\")
  (defthm test-thm
    (equal x x)
    :rule-classes nil)
  ~ev[]
  If we now execute ~c[:]~ilc[puff]~c[ 1], ACL2 will roll back the world to
  before the ~ilc[include-book]; then ``puff'' the include-book, which will
  leave us in ~c[:]~ilc[program] mode; and finally skip re-execution of the
  ~ilc[defthm] because such ~il[events] are skipped in ~c[:]~ilc[program] mode.
  The fix is to re-install the ~ilc[acl2-defaults-table] immediately after the
  ~ilc[include-book] to its pre-~ilc[include-book] value.

  A new event, ~ilc[make-event], provides something like macros that take
  state.  For example, one can use it to put tests into certified books, do
  proof search, and generate new function names.  Many examples appear in
  directory ~c[books/make-event/].  ~l[make-event].  Thanks to Bob Boyer and
  Jared Davis for useful feedback and to Warren Hunt, David Rager, and Sandip
  Ray for helpful discussions leading to some of the examples in directory
  ~c[books/make-event/].

  In support of ~ilc[make-event], which is described in the preceding
  paragraph, ~c[certify-book] has a new keyword argument, ~c[:save-expansion],
  that controls whether the result of expanding ~ilc[make-event] forms is
  written out to a file.  ~l[certify-book]; and for a discussion of book
  expansion files, ~pl[make-event].

  We fixed a soundness bug that did not correctly detect ~ilc[local] events.
  For example, the following event was admitted.
  ~bv[]
  (encapsulate
   ()
   (local
    (encapsulate
     ()
     (local (progn (program))) ; or, (local (with-output :off summary (program)))
     (set-irrelevant-formals-ok t)
     (defun foo (x)
       (declare (xargs :measure (acl2-count x)))
       (1+ (foo x)))))
   (defthm inconsistent
     nil
     :hints ((\"Goal\" :use foo))
     :rule-classes nil))
  ~ev[]

  A new value for ~il[guard] checking, ~c[:none], is now allowed.  If you
  execute ~c[:]~ilc[set-guard-checking]~c[ :none], then no guard checking will
  take place (but raw Lisp code will not be executed in this case).  As a
  result, you should never see a guard violation, even for calls of
  ~c[:]~c[program] mode functions.  We thank Pete Manolios, who has long wanted
  this feature, and also Peter Dillinger, for asking for it.  New documentation
  explains the interaction between the ~il[defun-mode] and the value supplied
  to ~c[:]~ilc[set-guard-checking].  ~l[guard-evaluation-table],
  ~pl[guard-evaluation-examples-script], and
  ~pl[guard-evaluation-examples-log].

  In the course of adding the ~il[guard]-checking value ~c[:none] described in
  the paragraph above, we eliminated an optimization that eliminated guard
  checking for some recursive calls of ~c[:]~il[logic] mode mutually-recursive
  functions that have not had their guards verified.  But we doubt that this
  change will be noticed by any users!)

  The ACL2 hyper-card has been enhanced, thanks to David Rager, with a listing
  of ``Useful EMACS Commands'' to match comments in ~c[emacs/emacs-acl2.el].

  Users contributed books following the ~c[Readme.lsp] methodology:
  ~c[data-structures/memories] and ~c[unicode] (Jared Davis), ~c[proofstyles]
  (Sandip Ray and J Moore).

  Made some improvements to ~c[books/Makefile-generic] (a file discussed
  elsewhere; ~pl[books-certification-classic]).  In particular, improved
  handling of ~c[.acl2] files for ~c[dependencies] target.

  (Only OpenMCL and, with feature ~c[:acl2-mv-as-values], GCL) Fixed a bug that
  was causing proclaiming to fail when definitions are submitted interactively.

  The default stack size has been increased for several lisps.

  (Very technical) A restriction has been weakened on the use of ~ilc[local]
  ~il[stobj]s under a call of an ACL2 evaluator (~c[trans-eval] or
  ~c[simple-translate-and-eval]).  Now, the error can only take place for
  ~ilc[stobj] names that occur in the term being evaluated.  Thanks to Erik
  Reeber for bringing this issue to our attention.

  The notion of ``ancestor'' has been changed slightly.  This notion is used by
  extended metafunctions and ~il[break-rewrite] (~pl[extended-metafunctions]
  and ~pl[brr@]), and also with backchain limits (~pl[backchain-limit] and
  ~pl[set-backchain-limit]).  Basically, each time a hypothesis is encountered
  during application of a ~il[rewrite] rule, that hypothesis is pushed (after
  instantiating and negating) onto the current list of ancestors before it is
  rewritten.  However, hypotheses of the form ~c[(equal var term)], where
  ~c[var] is free (~pl[free-variables]), had not been included in the ancestors
  (similarly for ~c[(equiv var (double-rewrite term))] where ~c[equiv] is a
  known ~il[equivalence] relation).  Now such ``binding hypotheses'' are
  included in a special way in ancestors data structures.  In particular,
  ~c[(null (mfc-ancestors mfc))] will now be true if and only if the term being
  rewritten is part of the current goal as opposed to a hypothesis from a rule
  encountered during backchaining, even if that hypothesis is a binding
  hypothesis.  Thanks to Dave Greve for bringing this issue to our attention.

  Termination and induction analysis now continue through both arguments of
  ~ilc[prog2$], not just the second.  (Normally, the gathering up of ~ilc[if]
  tests stops at function calls; but it continued through the second argument
  of ~ilc[prog2$], and now it will continue through both arguments.)  Thanks to
  Sol Swords for discussion leading to this change.

  The ACL2 distribution is now kept on the http server rather than the ftp
  server (but the home page has not been moved).  Thanks to Robert Krug for
  letting us know that some ACL2 users have found it inconvenient to fetch ACL2
  using ftp.

  The file ~c[books/README.html] has been renamed to ~c[books/Readme.html],
  since some browsers don't show the former in the directory listing.

  ~/~/")

(deflabel note-3-0

; See note-2-9-5 for some comments enumerating changes not in the :doc since
; v2-9-4.

  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.0 (June, 2006) Notes~/

  Please ~pl[note-2-9-5] for a description of changes since Version  2.9.4.
  These include the new ~ilc[make-event] feature, a soundness bug fix, an
  improvement for ~c[:expand] ~il[hints], evaluation in the logic by way of
  ~c[:]~ilc[set-guard-checking]~c[ :none], and many other improvements.

  More generally, there have been several incremental releases since
  Version  2.9: ~pl[note-2-9-1], ~pl[note-2-9-2], ~pl[note-2-9-3],
  ~pl[note-2-9-4], and ~pl[note-2-9-5].

  A very few users have contributed books following the instructions on the
  web.  We expect that when more contributions come in, we will give more
  attention to the question of how to organize the distributed and workshop
  books.  For now, we have simply added the new contributions according to the
  old-style distribution methodology.

  ~/~/")

(deflabel |NOTE-3-0(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.0(r) (June, 2006) Notes~/

  No significant changes have been made since Version  2.9 for support of
  non-standard analysis in particular.

  ~/

  Please also ~pl[note-3-0] for changes to Version  3.0 of ACL2.
  ~/
  ")

(deflabel note-3-0-1

; Things that seem too minor to mention:

; Modified proclaiming to understand (declare (type fixnum ...)) for function
; arguments.

; Regarding the subsumption item below: We still do the old subsumption check
; for chk-evaluator and processing of :by hints, because the user has some idea
; in these cases of what is going on (plus, the former will probably always be
; fast).  But we do the restricted check in the clause-set-subsumes call of
; chk-acceptable-equivalence-rule and (for induction) in
; some-pool-member-subsumes because those are kind of subtle and it's OK if
; they fail.

; The new requirement on the argument of satisfies resulted in an extra wrld
; argument for translate-declaration-to-guard (and other source functions),
; which is used in several books.

; Took suggestion from Bob Boyer to arrange to throw out extra values returned
; by intern, for efficiency (see intern-in-package-of-symbol and comments "See
; comment in intern-in-package-of-symbol").

; Fixed *initial-global-table* to satisfy ordered-symbol-alistp, and added a
; check for this just after the definition of *initial-global-table*.

; The improvement in handling of theories included an extra argument in
; set-difference-theories-fn and in union-theories-fn, lst1-known-to-be-runic.
; Users who are sophisticated and brave enough to use these unadvertised
; internal functions should be able to figure out this change, so we choose not
; to confuse other users by mentioning that in this :doc topic.

; Regarding the item below on hard lisp error and safe mode: We added a number
; of function/macro symbols to the list in *oneify-primitives* and to avoid
; oneifying, in order to support the use of safe-mode with make-event (which
; however we have abandoned for now because it seems difficult to ensure that
; we are dealing properly with *1* functions with corresponding hand-coded raw
; Lisp definitions).  We also fixed bugs discovered as we attempted to certify
; the books/make-event books:
; - state-global-let*-get-globals to use f-boundp-global in place of
;   boundp-global;
; - the call of fmt1 in print-warnings-summary;
; - the call (eq new-type-alist type-alist) in type-alist-clause-finish; and
; - the calls of eq in changed-pot-vars.

; Oneify-cltl-code has been slightly optimized for the case that
; 'guard-checking-on is :none, by eliminating some code that has no effect.

; The event comp now compiles properly for lists of functions even when some
; are in the boot-strap world.  In general, the code for
; compile-uncompiled-defuns and compile-uncompiled-*1*-defuns was cleaned up
; considerably, including the removal of proclaiming since add-trip is
; responsible for that.

; Deleted now-dead code collect-ideal-user-defuns and
; collect-ideal-user-defuns1.

; Among the "miscellaneous efficiency improvements not listed above" is an
; improvement to assume-true-false, which now takes an ignore argument that
; avoids some computation of the true-type-alist or false-type-alist when these
; are to be thrown away anyhow.

; Improved the "Guards" warning from certify-book to avoid rather odd mention
; of loading into raw Lisp.  Thanks to Sandip Ray and Robert Krug for helpful
; discussions.

; Modified documentation for topics ``~ilc[brr]'' and ``~ilc[breaks]'' to
; clarify that if you are at a raw Lisp break, then ~c[(]~ilc[abort!]~c[)] will
; get you back to the ACL2 top level.  Thanks to Dave Greve and Eric Smith for
; bringing this issue to our attention.

; Made minor mods, e.g. to avoid an eq test of n against header in the raw Lisp
; code for aref1.

; Improved support for window interfaces, in particular with new state globals
; window-interface-prelude and window-interface-postlude, thanks to
; contributions from Peter Dillinger.

  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.0.1 (August, 2006) Notes~/

  NOTE!  New users can ignore these release notes, because the documentation
  has been updated to reflect all changes that are recorded here.

  Fixed a soundness bug, introduced in the previous release, due to a failure
  to disallow ~ilc[table] ~il[events] that set the ~ilc[acl2-defaults-table] in
  a ~ilc[local] context.  Here is a proof of ~c[nil] that exploits the bug.
  ~bv[]
  (encapsulate ()
   (local (program))
   (defun foo ()
     (declare (xargs :measure 17))
     (+ 1 (foo))))
  (thm
   nil
   :hints ((\"Goal\" :in-theory (disable foo (foo)) :use foo)))
  ~ev[]

  Fixed a bug in the alternatives to ~ilc[good-bye], which are the ~ilc[exit]
  and ~ilc[quit] commands.  Thanks to Jared Davis and Peter Dillinger for
  pointing this out right away.

  The definition of ~ilc[len] has been highly optimized in raw Lisp.  Thanks to
  Bob Boyer and Warren Hunt for suggesting such an improvement and providing
  a lot of help in coming up with the current implementation.

  The clause subsumption algorithms have been improved, both to improve
  efficiency during warnings for ~c[:]~ilc[rewrite] rules and to punt when the
  subsumption computation for induction appears to be blowing up.  Thanks to
  Robert Krug for bringing this issue to our attention and supplying a useful
  example.

  A bug has been fixed that prevented ~ilc[time$] from working properly in
  OpenMCL and multi-threaded SBCL (actually, in any ACL2 image where feature
  ~c[:acl2-mv-as-values] is present).  Thanks to Sol Swords for bringing this
  problem to our attention.

  A ~il[type-spec] of the form ~c[(satisfies pred)] carries the requirement
  that ~c[pred] be a unary function symbol in the current ACL2 ~il[world];
  otherwise, it is illegal.  Thanks to Dave Greve for pointing out that Common
  Lisp has this requirement.

  Installed a fix provided by Gary Byers (for ACL2 source function
  ~c[install-new-raw-prompt]), for OpenMCL, that fixes an issue exposed in some
  versions of OpenMCL when compiler optimization is off.

  Fixed a bug in contributed book ~c[misc/untranslate-patterns.lisp] that was
  causing calls of ~c[add-untranslate-pattern] to be rejected in ~il[books].
  Thanks to Ray Richards for pointing out this bug and to Jared Davis for
  assisting in the fix.

  Fixed a bug in ~ilc[defstobj] when keywords ~c[:initially] and ~c[:resizable]
  are both supplied.  In this case, the definition of the resizing function
  mistakenly failed to quote the ~c[:initially] value, even though this value
  is not to be evaluated.  One could even get an error in this case, as in the
  following example supplied by Erik Reeber, whom we thank for bringing this
  bug to our attention:
  ~bv[]
    (defstobj $test
      (test-x :type (array t (5)) :initially (0) :resizable t))
  ~ev[]

  A new feature, ~ilc[with-prover-time-limit], allows the setting of time
  limits during proofs.  This is ~st[not] a general-purpose time-limit utility,
  nor is it guaranteed to implement a strict bound; it only attempts to limit
  time approximately during proofs.  Thanks to Pete Manolios and Daron Vroon,
  who made the most recent request for such a feature, and to Robert Krug for a
  helpful discussion.

  (GCL only) Fixed a bug in the procedure for building a profiling image.
  Thanks to Sol Swords for bringing this bug to our attention and to Eric Smith
  for bringing a subsequent problem to our attention.

  Handling of ~il[theories] can now use significantly less time and space.  A
  regression suite run took about 25% longer before this change than it did
  after making this change (and also the ones in the next two paragraphs).
  Thanks to Vernon Austel for bringing this issue to our attention and for
  supplying code, quite some time ago, that provided detailed, useful
  implementation suggestions.  Also thanks to the folks at Rockwell Collins,
  Inc. for pushing the limits of the existing implementation, thus encouraging
  this improvement.

  Fixed a performance bug in obtaining executable counterpart symbols.

  We now avoid certain computations made on behalf of warnings, when such
  warnings are disabled.

  We have relaxed the checks made when including an uncertified book, to
  match the checks made when including a certified book.  Thanks to Eric Smith
  for suggesting this change.

  Fixed a bug in ~c[:]~ilc[pso] (~pl[set-saved-output]) that caused an error
  when printing the time summary.

  Made fixes to avoid potential hard Lisp errors caused by the use of
  ~c[:]~ilc[program] mode functions.  The fix was to use a ``safe mode,''
  already in use to prevent such errors during macroexpansion;
  ~pl[guards-and-evaluation].  However, such errors were possible during
  evaluation of macro ~il[guard]s, for example as follows:
  ~bv[]
  (defun foo (x)
    (declare (xargs :mode :program))
    (car x))
  (defmacro mac (x)
    (declare (xargs :guard (foo 3)))
    x)
  (defun g (x)
    (mac x))
  ~ev[]
  A similar issue existed for calls of ~ilc[defpkg], ~ilc[in-theory],
  ~ilc[table], ~ilc[make-event], and ~c[value-triple], but has been fixed for
  all but ~c[in-theory] and ~c[make-event], where technical issues have caused
  us to defer this change.

  Fixed a bug in ~c[wet] that caused problems in OpenMCL, and perhaps other
  Lisp implementations, when the argument to ~c[wet] calls, or depends on,
  certain built-ins including ~ilc[prog2$], ~ilc[time$], ~ilc[mbe], and
  ~ilc[must-be-equal].  Thanks to David Rager for bringing this problem to our
  attention.

  The file ~c[books/Makefile-generic] has been improved so that when book
  certification fails with `make', the failure message contains the book
  filename.

  Documentation has been written to explain how to avoid an expensive immediate
  rewrite of the result of applying a ~c[:]~ilc[rewrite] or ~c[:]~ilc[meta]
  rule.  ~l[meta].  Thanks to Robert Krug for supplying this trick, and to Eric
  Smith and Dave Greve for useful discussions.

  (OpenMCL only) OpenMCL-based ACL2 image names formerly had extension
  ~c[\".dppccl\"], which was correct only for some platforms (including 32-bit
  Darwin PPC).  That has been fixed, thanks to a suggestion from Gary Byers.

  It is now legal to attach both a ~c[:use] and a ~c[:cases] hint at the same
  goal.  Thanks to Eric Smith for (most recently) requesting this feature.

  It is now permissible to include the same symbol more than once in the
  imports list of a ~ilc[defpkg] form (i.e., its second argument).  Also, the
  evaluation of ~ilc[defpkg] forms with long import lists now uses a reasonably
  efficient sorting routine to check for two different symbols with the same
  name (see also ~c[books/misc/sort-symbols.lisp]).  If you currently call a
  function like ~c[remove-duplicates-eql] for your imports list, as had been
  suggested by a ~ilc[defpkg] error message, then you may experience some
  speed-up by removing that call.  Thanks to Eric Smith for helping to discover
  this issue through profiling.

  Made miscellaneous efficiency improvements not listed above (for example,
  following a suggestion of Eric Smith to avoid checking for so-called ``bad
  Lisp objects'' during ~ilc[include-book], which saved almost 3% in time on
  one large example).

  Modified the notion of ``untouchable'' to separate the notion of untouchable
  functions and macros from the notion of untouchable state global variables.
  ~l[push-untouchable].  Thanks to Bob Boyer for sending an example,
  ~c[(put-global 'ld-evisc-tuple t state)], that suggested to us the need for
  more restrictive handling of untouchables.  In particular, many ~c[ld]
  specials (~pl[ld]) are now untouchable.  You may be able to work around this
  restriction by calling ~ilc[ld]; see for example the change to
  ~c[books/misc/expander.lisp].  But please contact the ACL2 implementors if
  this sort of workaround does not appear to be sufficient for your purposes.

  Fixed a bug in function ~c[set-standard-oi] (~pl[standard-oi]).

  Fixed a bug in the use of ~ilc[ld-evisc-tuple].  The bad behavior was an
  improper use of the print-level and print-length components of the tuple
  (specifically, taking its ~ilc[caddr] and ~ilc[cadddr] instead of taking its
  ~ilc[cadr] and ~ilc[caddr]).  Thanks to Bob Boyer for bringing this bug to
  our attention.

  A new argument to the ~c[compile-flg] argument of ~ilc[certify-book],
  ~c[:all], causes creation of a file to be compiled in place of the given
  book, where that file contains not only a copy of the book (with
  ~ilc[make-event] forms expanded) but also contains definitions of the
  so-called ``executable counterparts'' of the functions defined in the book.
  Then, functions defined in the book will be run compiled when including the
  book, even for functions whose ~il[guard]s have not been verified, or are in
  ~c[:program] mode and running in so-called ``safe mode''
  (for example, during expansion of macros).  The default behavior, value ~c[t]
  of ~c[compile-flg], is unchanged.  Moreover, a new ~c[:comp!] argument of
  ~ilc[include-book] now compiles the executable counterparts when creating the
  book's compiled file, and unlike ~c[:comp], always deletes the old compiled
  file first so that one always gets a fresh compile.

  Now, ~ilc[certify-book] gives a \"Guards\" warning only for ~c[:]~ilc[logic]
  mode functions that are defined in the given book but have not had their
  guards verified.  Previously, it also warned about such functions that were
  defined in the certification world or in sub-books.

  A new command, ~ilc[redo-flat], facilitates the debugging of failed
  ~ilc[encapsulate] and ~ilc[progn] forms by evaluating preliminary forms in
  order to leave one at the point of failure.  ~l[redo-flat].  Thanks to
  Ray Richards and others for asking for such a utility, and to Sandip Ray
  for useful discussions.

  We have changed the automatic declaration of of function types (still done in
  GCL and OpenMCL only, for now).  Our motivation was to avoid the assumption
  that Common Lisp functions return one value when ACL2 says that they do;
  thanks to Bob Boyer for bringing this issue to our attention with the example
  of defining ~c[(foo x y)] to be ~c[(floor x y)].  ACL2 was saying that
  ~c[foo] returns a single value, but because ~c[floor] returns two values in
  raw Lisp, so does ~c[foo].  Other changes to automatic declaration include
  comprehending ~ilc[defund], not just ~ilc[defun].

  A new function, ~ilc[mod-expt], computes ~c[(mod (expt base exp) m)], and
  does so efficiently in some implementations (currently only in GCL 2.7.0,
  which is not yet released).  Thanks to Warren Hunt for suggesting such an
  addition.

  New functions ~ilc[getenv$] and ~ilc[setenv$] have been made available for
  reading and writing environment variables.  Thanks to Jun Sawada for
  requesting these utilities.

  The query utility ~c[:]~ilc[pl] has been improved in several ways.  As
  before, ~c[:]~ilc[meta] rules are only printed if the argument is a symbol;
  but the information printed for them is now more appropriate.  The following
  are changes for the case that the argument is not a symbol, but rather, a
  term.  (1) Rules are displayed that have ~il[equivalence] relations other
  than ~ilc[equal].  (2) All matching ~c[:]~ilc[definition] rules are
  displayed, where previously ~c[:definition] rules were only shown if they
  were ``simple'' rules (sometimes known as ``abbreviations''); ~pl[simple].
  (3) The ``Equiv'' field is printed for terms, not just symbols.  (4) The
  substitution is shown that, when applied to the left-hand side of the rule,
  will yield the specified term.  Thanks to Eric Smith for suggesting these
  changes.

  The ~il[proof-checker] command ~c[;show-rewrites] has been improved to match
  the changes described above for ~c[:]~ilc[pl].  In particular,
  ~c[:]~ilc[definition] rules that are not ``~il[simple]'' are now displayed by
  the ~il[proof-checker]'s ~c[show-rewrites] (and ~c[sr]) command, and the
  ~il[proof-checker]'s ~c[rewrite] command has been correspondingly modified to
  accept these ~c[:definition] rules.

  Fixed `make' targets ~c[copy-distribution], ~c[copy-workshops], and
  ~c[copy-nonstd] so that they should also work for non-developers.

  Fixed a bug that was causing ~c[:]~ilc[pr] to display ~ilc[syntaxp]
  hypotheses oddly in some cases, in particular ~c[(syntaxp (let ...))].
  (The problem was in the ``untranslate'' display of the internal form of
  ~c[syntaxp] calls.)  Thanks to Robert Krug for bringing this problem to our
  attention.  We also removed the restriction on ~ilc[bind-free] that its
  argument could not be a variable, a constant, or (more interestingly) a
  ~ilc[lambda] application (i.e., a ~ilc[let] or ~ilc[mv-let] expression).

  ~/~/")

(deflabel |NOTE-3-0-1(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.0.1(r) (August, 2006) Notes~/

  No significant changes have been made since Version  3.0 for support of
  non-standard analysis in particular.

  ~/

  Please also ~pl[note-3-0-1] for changes to Version  3.0.1 of ACL2.
  ~/
  ")

(deflabel note-3-0-2

; Things that seem too minor to mention to users:

; Added ev-w and ev-w-lst.

; The "soundness bug in linear arithmetic" mentioned below was confined to
; linearize1.  Robert provided the fix and we checked it.  Below is an example
; from Robert that proves nil in ACL2 Version_3.0.1 but fails after the patch.

;  (defun id (x) x)
;
;  (defthm id-rationalp
;    (implies (force (rationalp x))
;             (rationalp (id x)))
;    :rule-classes :type-prescription)
;
;  (in-theory (disable id))
;
;  (defun id2 (x y)
;    (if (zp x)
;        y
;      (id2 (+ -1 x) y)))
;
;  (in-theory (disable (:type-prescription id2)))
;
;  (defthm bad
;    (implies (and (not (equal (id x) (id2 y z)))
;                  (acl2-numberp y)
;                  (integerp z)
;                  (<= 0 z))
;             (or (< (id x) (id2 y z))
;                 (< (id2 y z) (id x))))
;    :hints (("[1]Goal" :in-theory (enable (:type-prescription id2))))
;    :rule-classes nil)
;
;  (set-guard-checking :none)
;
;  (let ((x "foo")
;        (y 0)
;        (z 0))
;    (implies (and (not (equal (id x) (id2 y z)))
;                  (acl2-numberp y)
;                  (integerp z)
;                  (<= 0 z))
;             (or (< (id x) (id2 y z))
;                 (< (id2 y z) (id x)))))
;
;  (thm
;   nil
;   :hints (("Goal" :use (:instance bad (x "foo") (y 0) (z 0))))) ; |

; Added type declaration in ts-subsetp (but seemed not to make a measurable
; difference in time, at least for fast GCL build).

; Here is evidence for the bug in symbol-package-name-pkg-witness-name:

;  (defthm contradiction
;    nil
;    :hints (("Goal"
;             :use ((:instance symbol-package-name-pkg-witness-name
;                              (pkg-name ""))
;                   (:instance intern-in-package-of-symbol-symbol-name
;                              (x (pkg-witness ""))
;                              (y 3)))
;             :in-theory (disable (pkg-witness)
;                                 intern-in-package-of-symbol-symbol-name)))
;    :rule-classes nil)

; Implementation note: for reset-prehistory, the key idea is to manipulate
; world global 'command-number-baseline-info.

; Set ld-skip-proofsp to 'include-book during loading of compiled file by
; include-book.

; The "what could be considered a soundness hole" could be exploited as
; follows.

;     (in-package "ACL2")
;
;     ; Portcullis commands:
;     #|
;     (set-ld-redefinition-action '(:warn! . :overwrite) state)
;
;     (encapsulate
;      ()
;      (defun foo () t)
;      (local (defun foo () nil))
;      (defthm foo-prop
;        (equal (foo) nil)
;        :rule-classes nil))
;     |# ; |
;
;     ; NOTE: After the above commands:
;     ; ACL2 !>(redefined-names state)
;     ; NIL
;     ; ACL2 !>
;
;     ; Now execute:
;
;     ; (certify-book "bad" 1)
;
;     (defthm contradiction
;       nil
;       :hints (("Goal" :use foo-prop))
;       :rule-classes nil)

; After certifying the book we can do this:

; (include-book "bad")
; (thm nil :hints (("Goal" :use contradiction)))

; The documentation can now be built on Linux and probably Mac OS (seems that
; Linux texinfo issues have been solved).

; Made a correction in :doc guard-evaluation-table for built-in :program mode
; functions.

; Improved error message when compound event (including certify-book and
; include-book) has sub-event that is not an embedded event form.  If the
; sub-event is a macro call, then then "Note" at the end will give the original
; form as the macro call, not as the compound event.

; Improved max-output-type-for-declare-form-lst to add missing arguments in
; error message, as suggested by Robert.  Also improved
; max-output-type-for-declare-form to check for true lists before calling
; max-output-type-for-declare-form-lst.

; Improved translate error message from mutual-recursion so that the ctx
; identifies the problem function.  Thanks to Robert Krug for pointing out the
; value of making such an identification.

; Expanded *initial-global-table* to include all built-in state globals, so as
; do allow a more inclusive value of *protected-state-globals-for-make-event*
; (called *protected-system-state-globals* starting with v3-2).  Thanks to
; Peter Dillinger for pointing out that
; *protected-state-globals-for-make-event* was incomplete.

; In the course of implementing ttags for include-book-fn, we noticed that we
; were accessing a world with global-val (in particular on property
; 'boot-strap-flg) after calling chk-certificate-file, which calls
; chk-raise-portcullis, which can extend the world.  We fixed this to call
; global-val on an installed world instead.  It seems possiblee that for large
; proof efforts, this change might provide performance improvements.

; Moved Boyer's scary sys-call example from :doc make-event to :doc sys-call,
; since make-event isn't the issue.

; Here are the books changed to accommodate the fix for the soundness bug in
; the redundancy criterion for defun events.  It's possible that a few of these
; changes are no longer necessary now that we ignore the measure for redundancy
; when including books.
;
; workshops/2000/moore-manolios/partial-functions/report.lisp
; rtl/rel1/lib1/float.lisp
; rtl/rel1/lib1/round.lisp
; rtl/rel1/lib3/float.lisp
; rtl/rel1/lib3/round.lisp
; rtl/rel2/lib/bits.lisp
; rtl/rel2/lib/float.lisp
; rtl/rel3/lib/bits.lisp
; rtl/rel3/lib/float.lisp
;
; Problem with rtl/rel3/lib/top.lisp: can't have :? before real measure..
; Solution: Modify the following to use :? in expo:
; rtl/rel3/support/bvecp-lemmas.lisp
; rtl/rel4/support/bvecp-lemmas.lisp
; (Might not be necessary with the skip-proofs change, but it's a nice change
; anyhow.)
; Then also needed (even after skip-proofs mod) to make such a change in:
; rtl/rel4/lib/bvecp-raw-helpers.lisp
;
; rtl/rel4/lib/bits.lisp
; rtl/rel4/lib/float.lisp
; rtl/rel4/lib/fadd.lisp
; rtl/rel5/lib/log.lisp
; rtl/rel5/lib/float.lisp
; rtl/rel5/lib/add.lisp
; finite-set-theory/osets-0.9/sets.lisp
; finite-set-theory/osets/sets.lisp
; workshops/2004/davis/support/sets.lisp
; workshops/2000/russinoff-short/summary.lisp

; Improved the error message for a bad bind-free alist result, based on an
; example sent by Serita Nelesen to the acl2-help mailing list.

; Directory books/defexec/chapter3 has been renamed books/defexec/other-apps/
; because the records stuff is part of Section 4 of the corresponding paper,
; not Section 3.

  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.0.2 (December, 2006) Notes~/

  NOTE!  New users can ignore these release notes, because the documentation
  has been updated to reflect all changes that are recorded here.

  Fixed soundness bugs in the handling of primitive function ~ilc[pkg-witness],
  and improved its documentation.  (The executable counterpart returned an
  incorrect default value, and the axiom
  ~c[symbol-package-name-pkg-witness-name] needed ~c[pkg-name] to be other than
  ~c[\"\"] in order to avoid the default value of \"ACL2\".)  As fallout, a new
  built-in ~c[:]~ilc[forward-chaining] rule,
  ~c[symbol-package-name-of-symbol-is-not-empty-string], now asserts that the
  ~ilc[symbol-package-name] of a symbol is never ~c[\"\"].  Thanks to Mike
  Gordon for bringing these soundness bugs to our attention by attempting to
  prove translations of ACL2 axioms in HOL4.

  Fixed a soundness bug in linear arithmetic, due to incomplete tracking of
  forced assumptions while deriving inequalities.  Thanks to Robert Krug for
  providing a fix and a proof of ~c[nil] before the fix.

  Fixed a soundness bug in the redundancy criterion for ~ilc[defun] events,
  which has been modified; ~pl[redundant-events].  This bug is illustrated
  below.  Thanks to Peter Dillinger and Jared Davis for contributions to an
  email thread that led us to discover this bug.  The solution is that for a
  definition to be redundant with an earlier definition, ACL2 no longer ignores
  ~c[:]~ilc[measure] ~ilc[xargs] except when skipping proofs (e.g., during
  ~ilc[include-book]).  However, a new ``measure'', ~c[(:? v1 ... vk)], is
  supported, for specifying a measured subset of the set of formals, i.e., a
  set of formals that serves as the set of parameters for some valid measure.
  ~bv[]
  (encapsulate
   ()

   (local (defun foo (x y)
            (declare (xargs :measure (acl2-count y)))
            (if (and (consp x) (consp y))
                (foo (cons x x) (cdr y))
              y)))

  ; So the following is redundant -- but it guesses a measure
  ; of (acl2-count x), which isn't right!
   (defun foo (x y)
     (if (and (consp x) (consp y))
         (foo (cons x x) (cdr y))
       y)))

  ; end of encapsulate

  ; Now we prove a non-theorem by exploiting the bug above,
  ; erroneously replacing formal y by a constant in the induction
  ; scheme hinted below.  (This should not be allowed, as y should be
  ; labeled as a measured formal.)

  (defthm bad
    (atom x)
    :rule-classes nil
    :hints ((\"Goal\" :induct (foo x '(3)))))

  ; It's easy to get a contradiction by instantiating the
  ; non-theorem just above.
  (defthm contradiction
    nil
    :rule-classes nil
    :hints ((\"Goal\" :use ((:instance bad (x '(7)))))))
  ~ev[]

  Fixed a bug in ~c[:]~ilc[pl] and the ~il[proof-checker]'s ~c[show-rewrites]
  (~c[sr]) command that was causing a Lisp break.  For ~c[:]~ilc[pl], also
  improved the display of unifying substitutions, modified output to take
  binding hypotheses ~c[(equal var term)] into account properly, and arranged
  for inclusion of ~il[meta] rules that modify the given term.  Thanks to Eric
  Smith for bringing these issues to our attention.

  Introduced new utilities for undoing ~il[command]s, ~c[:]~ilc[ubu] and
  ~c[:]~ilc[ubu!], which are analogous to ~c[:]~ilc[ubt] and ~c[:]~ilc[ubt!]
  (respectively) except that they only undo back up to, but not including, the
  indicated command.

  Fixed a performance bug, pointed out by Eric Smith, that was negating efforts
  made for the preceding release to avoid computation for disabled warnings.

  Added ~ilc[time$] and ~c[value-triple] to ~c[*acl2-exports*].  Thanks to Bob
  Boyer and Erik Reeber (respectively) for bringing these issues to our
  attention.

  Improved the automatic proclaiming of function types for GCL and OpenMCL,
  specifically to use an output format consistent with the Common Lisp spec.
  Thanks to Bob Boyer for bringing this issue to our attention.

  Added ~c[books/misc/transfinite.lisp], which deals with transfinite induction
  in ACL2.  Thanks to Eric Smith for contributing this book.

  Added ~c[books/misc/process-book-readme.lisp] to the distribution.  Thanks to
  Sandip Ray for pointing out its omission.

  Added contributions ~c[books/concurrent-programs/bakery/] and
  ~c[books/concurrent-programs/german-protocol/].  These contributions can be
  used as tutorials, especially by new ACL2 users, for learning how to model
  concurrent protocols in ACL2 and the steps involved in reasoning about their
  correctness.  Thanks to Sandip Ray for these contributions.  See the
  ~c[Readme.lsp] files in these directories.

  Theory invariants may now involve the variable ~c[ENS] instead of the
  variable ~c[THEORY].  The practical effect of this change is that any
  expression of the form ~c[(MEMBER-EQUAL rune THEORY)] occurring in a
  ~ilc[theory-invariant] expression should be replaced by
  ~c[(ACTIVE-RUNEP rune)].  ~l[theory-invariant].  Thanks to Eric Smith and
  Dave Greve for pointing out an inefficiency in the handling of theory
  invariants that led to this change, which can speed up their handling by
  orders of magnitude on large examples, and to Eric for testing this change
  and pointing out problems with an early implementation of it.

  Theory invariants (~pl[theory-invariant]) are no longer checked on theories
  defined by ~ilc[deftheory] ~il[events].  After all, one can define a theory
  with ~c[deftheory] that is not intended to be used as the current theory, but
  rather is intended to be combined with other ~il[theories]
  (~pl[theory-functions]).  Thanks to Eric Smith for bringing this issue to our
  attention.

  ~ilc[Theory-invariant] errors had been reported with very little detail when
  warnings were inhibited.  This problem has been fixed; thanks to Eric Smith
  for bringing it to our attention and providing an example.  We have also
  improved the handling of redundancy for ~ilc[theory-invariant] ~il[events].

  The macro ~ilc[defun-sk] now has a new optional keyword, ~c[rewrite], that
  can be used to change the form of the ~c[:]~ilc[rewrite] rule generated when
  the quantifier is ~ilc[forall].  Thanks to Eric Smith and Sandip Ray for
  useful discussions on this topic.  We have also slightly modified the
  ~il[hints] for the ~ilc[defthm] event underneath a ~c[defun-sk] in order to
  make the proof more reliably efficient.

  A new event, ~ilc[reset-prehistory], allows setting of a barrier before which
  undoing is illegal.  An argument to this macro allows the barrier to be made
  permanent; otherwise, it can be removed with ~c[:]~ilc[ubt-prehistory].
  Thanks to Peter Dillinger for useful conversations leading to the addition of
  ~ilc[reset-prehistory].

  A new query, ~c[(]~ilc[wormhole-p]~c[ ]~ilc[state]~c[)], allows users to
  determine whether or not they are in a ~ilc[wormhole].  Thanks to Peter
  Dillinger for providing this utility.

  ~c[Value-triple] no longer evaluates its form during ~ilc[include-book], and
  in raw Lisp its calls trivially macroexpand to ~c[nil], without any
  consideration of its argument.  This change avoids errors and warnings when
  ~il[stobj] names occur in the argument.

  We fixed what could be considered a soundness hole that could occur by
  exploiting redefinition in a particular way.  Thanks to Peter Dillinger for
  raising a question that led to discovery of this hole.

  A bug has been fixed in handling of illegal ~il[theory] expressions.  Thanks
  to Eric Smith, who reported this problem and provided the example
  ~c[(in-theory '((:definition natp) (:rewrite doesntexist)))]
  to show how a hard error could occur.

  Improved error reporting by ~ilc[certify-book] when the certification
  ~il[world] contains inadmissible forms.

  Modified ~ilc[defchoose] to add two new keyword arguments.  There is now a
  ~c[:doc] keyword argument; previously, an optional documentation string
  (~pl[doc-string]) was to be placed just before the body, without a keyword.
  There is also a ~c[:strengthen] argument that strengthens the axiom added,
  which allows for the definition of ``fixing'' functions for equivalence
  relations that choose canonical representatives of equivalence classes.
  ~l[defchoose].  Thanks for Dave Greve for useful discussions that led us to
  this ~c[:strengthen] enhancement.

  Added ~c[books/misc/bash.lisp], which provides utilities for simplifying a
  goal into a list of subgoals (as documented at the top of that file).  Thanks
  to Dave Greve for requesting this utility and suggesting refinements to its
  functionality, which have been incorporated.

  (For Emacs users only) The command ~c[meta-x new-shell] provided by file
  ~c[emacs/emacs-acl2.el] now puts you in shell-mode, which for example
  supports directory tracking.  Thanks to Jared Davis for suggesting this
  change.

  Fixed some mishandling of ~il[stobj]s by ~ilc[make-event] expansion.

  Introduced a new event, ~ilc[defttag], that introduces a ``trust tag''
  (``ttag'') allowing for extensions of ACL2 and for the use of generally
  unsafe ACL2 constructs.  Thanks to Peter Dillinger, Sandip Ray, and Erik
  Reeber for useful discussions on ~c[defttag] and the following related
  items.
  ~bq[]

  A new event, ~ilc[remove-untouchable], can be used to give users access to
  system functions and data structures.  We also fixed a bug in
  ~ilc[push-untouchable]; and, it no longer is a no-op in ~c[:]~ilc[program]
  mode.  Thanks to Peter Dillinger for proposing ~ilc[remove-untouchable] and
  suggesting that it and ~ilc[push-untouchable] be functional in
  ~c[:]~ilc[program] mode.

  Raw-mode (~pl[set-raw-mode]) no longer disables ~ilc[certify-book].  However,
  ~ilc[set-raw-mode] is now disallowed unless there is an active ttag
  (~pl[defttag]).  If you want to execute ~c[(]~ilc[set-raw-mode]~c[ t)] and
  there is no active ttag, consider executing ~c[(]~ilc[set-raw-mode-on!]~c[)]
  instead.

  Redefinition of system functions is disallowed unless there is an active
  ttag.  However, ~ilc[redef!] now introduces ~c[(defttag :redef!)] in order to
  allow redefinition of system functions.

  A new event, ~ilc[progn!], is a legal embedded event form that can go in
  ~il[books] and both ~ilc[encapsulate] and ~ilc[progn] forms
  (~pl[embedded-event-form]), and is similar to ~ilc[progn] except that it
  allows arbitrary forms.  Thus, a ~ilc[progn!] form is potentially dangerous
  and can only be evaluated if there is an active ttag.

  ~l[ttags-seen] for information about how to find the ttags known in the
  current ACL2 ~il[world], and for related caveats.

  A new book created with Peter Dillinger, ~c[books/misc/hacker.lisp] (added
  after Version_3.3: now ~c[books/hacking/hacker.lisp]), uses ~ilc[progn!] to
  define utiliities ~c[with-raw-mode] and ~c[with-redef-allowed], which
  respectively allow raw Lisp evaluation and redefinition to take place within
  a certifiable book (!).~eq[]

  Macro ~ilc[with-output] is no longer allowed in function bodies because it
  does not have (and has never had) any effect in raw Lisp.  ~l[with-output]
  for a workaround.

  Fixed a bug in redundancy of ~ilc[defstobj] in raw Lisp, which caused an
  error when certifying a book with a redundant ~ilc[defstobj] event whose
  ~ilc[stobj] had already been modified.  Here is an example:
  ~bv[]
  (defstobj st fld)
  (update-fld 3 st)
  (certify-book \"foo\" 1) ; where foo.lisp contains (defstobj st fld)
  ~ev[]

  New books illustrating ~ilc[make-event] have been contributed in directory
  ~c[books/make-event/]: ~c[dotimes.lisp] (David Rager), ~c[stobj-test.lisp],
  and ~c[logical-tangent.lisp] (Peter Dillinger).

  Modified ~c[print-object$] (~pl[io]) so that it no longer prints an extra
  space at the end.

  Replaced the ``draconian restriction to avoid capture'' that had prevented
  some ~c[:functional-instance] ~il[hints] from being legal.  The corresponding
  check now only requires that no variable free in the functional substitution
  is captured by a ~ilc[let] or ~ilc[mv-let] (or ~ilc[lambda]) binding.
  ~l[lemma-instance].

  Added new extended metafunction, ~c[mfc-rw+], which is equivalent to
  ~c[mfc-rw] except that it takes an alist argument, which may be useful for
  efficiency.  ~l[extended-metafunctions].  Thanks to Robert Krug for
  suggesting this more efficient variant of ~c[mfc-rw].

  Added support for the ~c[ignorable] ~ilc[declare] form.

  We now cause an error on a call of ~c[open-input-channel] (~pl[io]) with an
  argument string whose first character is the ~c[|] character.  Thanks to Bob
  Boyer for providing an example (several months ago) showing the danger of
  such calls, namely that the following command would log you out and kill all
  of your processes when running on top of GCL in Linux:~nl[]
  ~c[(open-input-channel \"|kill -9 -1\" :object state)]

  Restricted the use of ~ilc[make-event] to contexts in which it can be tracked
  properly, under legal ~il[events] (~pl[embedded-event-form]).  Thanks to
  Peter Dillinger for bringing an example to our attention that led to this
  fix.

  Fixed a bug that was avoiding ~il[guard]-checking for the functions
  ~ilc[compress1] and ~ilc[compress2].  Thanks to David Rager for bringing this
  bug to our attention.

  Added an error message when a ~ilc[defun] or ~ilc[mutual-recursion] event
  fails, to clarify whether failure is for the ~il[measure] conjecture or for
  the ~il[guard] conjecture.  Thanks to David Rager for requesting
  clarification for such failures.

  Fixed a bug in reporting of ~il[guard] violations (hard Lisp error) when certain
  macros (for example, ~ilc[cond]) are used in the ~il[guard].  Thanks to Jared
  Davis for bringing this problem to our attention and providing assistance
  with the solution, in particular by providing a helpful example.

  Grant Passmore has contributed a resolution/paramodulation prover written in
  ACL2, in directory ~c[books/deduction/passmore/].  Thanks, Grant.

  Improved the error message when illegal theories are encountered.

  Improved the suppression of output for ~c[inhibit-output] arguments of
  routines in the book ~c[books/misc/expander.lisp].  Thanks to Qiang Zhang for
  pointing out the possibility for improvement here.

  Added a new directory ~c[books/arithmetic-3/extra/] that extends
  ~c[books/arithmetic-3] with additional rules, contributed by Alex
  Spiridonov with guidance from Robert Krug.  WARNING: This directory is under
  development.  It may undergo large changes in future releases, so please
  consider it experimental and subject to change.  Feedback is welcomed.

  As part of the work mentioned just above, Robert Krug and Alex Spiridonov
  contributed improvements to ~c[books/arithmetic-3/]:
  ~bq[]

  o A new rule ~c[|(* (/ x) (/ (expt x n)))|] in ~c[bind-free/collect.lisp],
  which is important for reducing ~c[collect-*] expressions though it slowed
  down one proof (see comment above this rule in ~c[bind-free/collect.lisp]).

  o Slight improvements of rules ~c[integerp-mod] and ~c[rationalp-mod] in
  ~c[floor-mod/floor-mod.lisp].

  o To avoid conflict with ~c[books/rtl/rel6/arithmetic/], renamed rule
  ~c[mod-minus] to ~c[mod-neg] in ~c[floor-mod/floor-mod.lisp], and renamed
  ~c[integerp-+-reduce-leading-constant] to
  ~c[integerp-+-reduce-leading-rational-constant] in
  ~c[bind-free/integerp.lisp].~eq[]

  (GCL on Windows only) Made a low-level change to avoid multiplying stacks for
  GCL on Windows, since GCL 2.6.6 broke while doing this.

  Fixed bugs in linear arithmetic (rarely evidenced, it seems) involving using
  ~c[<] to compare complex rational constants.  Thanks to Robert Krug for
  helping with the fixes.

  Added a new event, ~ilc[assert-event], for checking that forms evaluate to
  non-~c[nil] values.  Thanks to Peter Dillinger for suggesting and
  collaborating on this addition.

  ~/~/")

(deflabel note-3-1

  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.1 (December, 2006) Notes~/

  NOTE!  New users can ignore these release notes, because the documentation
  has been updated to reflect all changes that are recorded here.

  Please ~pl[note-3-0-2] for a description of changes since Version  3.0.1, and
  also ~pl[note-3-0-1] for additional changes since Version  3.0.

  ~/~/")

(deflabel |NOTE-3-1(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.1(r) (December, 2006) Notes~/

  No significant changes have been made since Version  3.0 for support of
  non-standard analysis in particular.

  ~/

  Please also ~pl[note-3-1] for changes to Version  3.1 of ACL2.

  ~/
  ")

(deflabel note-3-2

; The "soundness bug that was allowing unknown packages" mentioned below was
; due to binding *inside-include-book-fn* to t in certify-book-fn, which was
; causing avoidance of the chk-bad-lisp-object check done by read-object.  Here
; is a proof of nil in Version_3.1.  Each book is to be certified in a fresh
; session (fresh for each book).

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ----------------
; Book sub.lisp:
; ----------------
; ; Portcullis command:
; ; (make-event (er-progn (defpkg "FOO" nil) (value '(value-triple nil))))
;
; (in-package "ACL2")
;
; (defconst *c* 'foo::a)
;
; (defthm thm1
;   (equal (symbol-package-name *c*) "FOO")
;   :rule-classes nil)
;
; ----------------
; Book top.lisp:
; ----------------
; ; Portcullis command:
; ; (defpkg "FOO" '(a))
;
; (in-package "ACL2")
;
; (include-book "sub")
;
; (defthm contradiction
;   nil
;   :hints (("Goal" :use thm1))
;   :rule-classes nil)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Here is Peter Dillinger's original example related to package stuff.  After
; certifying "pkg" (after executing the given defpkg) and then (after :ubt! 1)
; "use", then (include-book "use") will break (unknown package) in a new
; session when an attempt is made to read use.cert.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; File pkg.lisp:

; #| (defpkg "FOO" ()) |#
;
; (in-package "ACL2")
;
; (defun foo::foo () 42)
;
; (defun bar () '(value-triple (foo::foo)))

;;; File use.lisp:

; (in-package "ACL2")
;
; (include-book "pkg")
;
; (make-event (bar))
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; We came up with the following example when dealing with the issues exposed in
; the two examples above.  If you certify "book1", "book2", and then "top",
; with portcullis commands as shown, then (include-book "top") will break
; (unknown package) when run in a new session, when reading top.cert.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; File book1.lisp:

; ; (defpkg "P1" nil)
; ; (certify-book "book1" 1)
;
; (in-package "ACL2")
;
; (defun f1 (x) x)

;;; File book2.lisp:

; ; (defpkg "P1" nil)
; ; (defmacro my-id (x) x)
; ; (defpkg "P2" (my-id '(p1::a p1::b x y)))
; ; (certify-book "book2" 3)
;
; (in-package "ACL2")
;
; (defun f2 (x) x)

;;; File top.lisp:

; (in-package "ACL2")
;
; (include-book "book1")
;
; (local (include-book "book2"))
;
; (defun f3 (x) x)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Fixed a bug in books/arithmetic-3/extra/Makefile discovered by Gary Byers
; (our fault, not Alex Spiridonov's).  Also disabled two rules in ext.lilsp in
; that directory, in part following suggestion of Robert Krug.

; Following a suggestion from Peter Dillinger, makunbound-global and put-global
; can now take a first argument that is not a quoted symbol, provided there are
; no untouchables.  The latter can be accomplished by, for example:
; (defttag t)
; (make-event
;  (let ((names (global-val 'untouchable-vars (w state))))
;    (value `(remove-untouchable (,@names) nil))))

; Made minor efficiency improvements to termination-machine.

; An expansion file is no longer written by certify-book if :save-expansion and
; the compile-flg are nil.

; Fixed a bug in ffnnames-subsetp that made it true too often when lambda terms
; are involved, and hence could allow probably-not-validp to be true too
; often.  This in turn could lead irrelevant-lits to return too large a list,
; which could lead eliminate-irrelevance-clause to strengthen the clause too
; much.  So this was a (probably rare) heuristic issue but not a soundness
; issue.

; Improved with-warnings-suppressed for Allegro and CLISP, and removed
; corresponding suppressions elsewhere since lp calls ld-fn under
; with-warnings-suppressed already.

; Installed fix from Robert Krug to
; books/arithmetic-3/bind-free/default-hint.lisp that prints messages when
; enabling or disabling nonlinear arithmetic using a particular computed hint.

; For the "subtle soundness bug related to ~c[:]~ilc[meta] rules" item below,
; the developer-only directory tests/meta/ has some relevant tests.

; Updated files books/Makefile* to comprehend OpenMCL compiled files on
; 64-bit Linux and Intel-based Macs.  Thanks to Robert Krug for pointing out
; that the ``clean'' target wasn't cleaning up such files.  Note however that
; we no longer create compiled files for OpenMCL anyhow.

; The raw version of mfc-rw appeared to be missing a throw in the case that the
; obj is not a member of '(t nil ?).  This has been fixed.

; Added a missing unsigned-byte (fixnum) declaration to rewrite-solidify-plus.

; Modified state-global-let* to take an optional ``setter'' function with each
; binding.

; For GCL, at Peter Dillinger's request, read-object now binds si::*notify-gbc*
; to nil (but only when reading from *standard-ci* or *standard-oi*).

; Fixed failure to lay down a command landmark when value :q is returned, e.g.:
; (er-progn (defun foo (x) x) (value :q))

; In maybe-add-command-landmark, eliminated redundant (eq wrld old-wrld) test
; that had been conjoined with (raw-mode-p state) check.

; Fixed complex comment about make-event in encapsulate-pass-2.

; Disallowed theory-invariant in code just as we do for table, in translate11.
; Also cleaned up theory-invariant a bit by separating out its raw Lisp version
; and removing its name from *macros-for-nonexpansion-in-raw-lisp*.

; Modified handling of state global 'inhibit-output-lst, so that if it's set
; during the expansion phase of a make-event, then it will persist.  Thanks to
; Jared Davis for asking about allowing the setting inhibit-output-lst as an
; embedded event form.

; Regarding "The rewriter has been tweaked to run faster": Here is the example
; from Eric (and Jared).  Note that we get a slow-down by a factor of about 10
; even without set-nu-rewriter-mode in versions before we added the following
; test near the end of rewrite, in v3-2:
; (not (member-eq (nu-rewriter-mode wrld) '(nil :literals)))

;  (defun repeat (n v)
;    (declare (xargs :guard (natp n)))
;    (if (zp n)
;        nil
;      (cons v (repeat (1- n) v))))
;
;  (defthmd len-of-cons
;    (equal (len (cons a x))
;           (+ 1 (len x))))
;
;  (in-theory (disable len))
;
;  (set-rewrite-stack-limit nil)
;
;  (set-nu-rewriter-mode t) ; slows things down by a factor of almost 20
;
;  (thm (equal (len (repeat 7000 a)) 7000)
;       :hints (("goal'" :in-theory (enable len-of-cons))
;               ("Goal"
;                :do-not '(preprocess)
;                :expand (:free (a x) (repeat a x)))))

; Regarding the slow array warning related to wormholes mentioned below: here
; is a way to cause that problem before Version_3.2.
;   (wormhole t 'interactive-break nil '(value 'hi!))
;   (in-theory (disable binary-append))
;   :a!
;   (thm (equal (car (cons x y)) x))

; The modification for tag-trees caused about a 1% slowdown.  However, a
; trivial modification to ancestors-check, which avoids recursion if ancestors
; is nil, caused about a 1.5% speedup.

; Here is how to cause the slow-array-warning mentioned below in Version_3.1
; and before.
;
; (wormhole t 'interactive-break nil '(value 'hi!))
; (in-theory (disable binary-append))
; :a!
; (thm (equal (car (cons x y)) x))

; Fixed books/Makefile-generic to use :ttags :all for include-book commands
; that support compilation (targets all-o, all-fasl, etc.).

; Fixed include-book with :load-compiled-file :comp (or :comp!) so that it
; always deletes a new expansion file.

; Built-in state global variables may no longer be made unbound with
; makunbound-global.  See always-boundp-global.

; Removed some noise printed even using with-output, e.g.:
; (with-output :off :all (encapsulate nil (program)))
; Thanks to Peter D. for requesting such a change.

; Make-event can now be used inside state-global-let* when there is an active
; trust tag.  See the discussion in :doc make-event of "the one exception to
; this restriction".  Also, we have eliminated compiler warnings for calls of
; progn!.  The problem was that state was unbound at a top-level call of
; progn!, but when we modified the raw Lisp code for progn! to bind state to
; *the-live-state*, we found that (pprogn .... (progn! ....)) was giving us a
; compiler warning since state no longer occurred free in the last form!  So we
; fixed pprogn as well to declare state ignorable above the last form.

; Fixed spacing after a period that was resulting in output such as the
; following after a successful termination proof:
;
; "admit this function under the principle of definition.We could deduce"

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  3.2 (April, 2007) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Before this release, a raw Lisp error could put the ACL2 user into the
  debugger of the host Common Lisp.  Now, such cases will generally put the
  user back at the top-level loop after an informative message.  For details,
  ~pl[set-debugger-enable]; also ~pl[break$].

  Fixed a soundness bug that was allowing unknown packages to sneak into a book
  and wreak havoc.  Thanks to Peter Dillinger for sending an interesting
  example that led us to an exploration resulting in finding this bug.  (A
  comment in the source code for ~c[note-3-2] shows such an example.)  That
  example led us to fix a couple of other bugs related to packages.
  ~l[hidden-death-package] if you are generally interested in such issues, and
  for associated examples, see comments in ~c[note-3-2] in the ACL2 source
  code.

  Fixed subtle soundness bugs related to ~c[:]~ilc[meta] rules by restricting
  evaluators (~pl[defevaluator]), as discussed in a new documentation topic:
  ~pl[evaluator-restrictions].

  Fixed a soundness bug that was allowing redefinition from ~c[:]~ilc[logic] to
  ~c[:]~ilc[program] mode.  This prohibition had been in ACL2 for awhile but
  was accidentally removed in the preceding version.

  Fixed a soundness bug related to ~ilc[trace$].  Thanks to Peter Dillinger for
  bringing it to our attention and for useful discussions, and providing a
  proof of ~c[nil], the essence of which is illustrated as follows:
  ~bv[]
  (value-triple (trace$ (bar :entry (defun foo () 17))))
  ~ev[]
  Thus, ~ilc[trace$] could be used to cause destructive raw Lisp behavior.
  Now, ~c[trace$] fails unless it is either given a list of symbols or else
  there is an active trust tag (~pl[defttag]); otherwise, consider using
  ~c[trace!] instead.

  Closed a loophole that could be viewed as compromising soundness.  It was
  possible to write files during book certification by exploiting
  ~ilc[make-event] expansion, but that is no longer the case by default.  A new
  function ~ilc[open-output-channel!] is identical as a function to
  ~c[open-output-channel], except that the new function may be called even
  during ~ilc[make-event] expansion and ~ilc[clause-processor] ~il[hints], but
  requires that there is an active trust tag (~pl[defttag]).  Thanks to Peter
  Dillinger for producing a convincing example (forging a ~il[certificate]
  during book certification; ~pl[open-output-channel!]) and to him,
  Sandip Ray, and Jared Davis for useful discussions on the topic.

  Added book ~c[books/defexec/reflexive/reflexive.lisp] to illustrate reflexive
  functions.

  ACL2 now generate scripts that invoke the saved image with ~c[exec].
  (Previously this was only done for GCL and CLISP.)  The benefit of this
  change can be to avoid the lingering of ACL2 processes after enclosing
  processes have exited.  Thanks to Peter Dillinger for pointing out this
  issue.

  ACL2 has a better implementation of ~c[(]~ilc[good-bye]~c[)] (hence of
  synonyms ~c[(]~ilc[quit]~c[)] and ~c[(]~ilc[exit]~c[)]).  As a result, you
  should now be able to exit ACL2 and Lisp from within the ACL2 read-eval-print
  loop with any of the above; formerly, this was not supported for some Lisp
  implementations, and was slow in OpenMCL.  Thanks to SBCL developer Harald
  Hanche-Olsen for useful advice.

  Fixed a bug in raw-mode (~pl[set-raw-mode]) that was causing hard errors when
  evaluating calls of ~ilc[er-progn], or of macros expanding to such calls.

  Fixed a few Makefile dependencies, necessary only for parallel `make'.

  A new book, ~c[misc/defpun-exec-domain-example.lisp], provides an example
  showing how partial functions which return a unique value for arguments in a
  specified domain can be efficiently executed with ACL2.  Execution is
  achieved using the ~ilc[mbe] construct.  Thanks to Sandip Ray for providing
  this example.

  Existing function ~ilc[mod-expt] computes ~c[(mod (expt base exp) mod)] with
  great efficiency in GCL, but not in other Lisps.  Now, the book
  ~c[arithmetic-3/floor-mod/mod-expt-fast.lisp] defines a function
  ~c[mod-expt-fast] that should provide significantly improved performance for
  such expressions in other Lisps as well, though still probably not as fast as
  when using ~c[mod-expt] in GCL.  Thanks to Warren Hunt, with contributions
  from Robert Krug, for providing this book,

  Modified macro ~il[break-on-error] to print of an error message before
  entering a break, and to cause a hard error if the underlying Lisp cannot
  handle it (formerly, a raw Lisp break would occur).  Thanks to Bob Boyer for
  bringing these issues to our attention.

  The book ~c[books/misc/defpun.lisp], as well as other books related to the
  ~c[defpun] macro, has been modified to avoid namespace collisions by
  prefixing function symbol names with ~c[\"DEFPUN-\"]; for example ~c[base] has
  been replaced by ~c[defpun-base].  Thanks to Dave Greve for providing a first
  version of this update to ~c[defpun.lisp].

  A theory, ~c[base], in ~c[books/arithmetic-3/bind-free/top.lisp], has been
  renamed ~c[arithmetic-3-bind-free-base], to avoid potential name conflicts.

  Fixed ~c[books/arithmetic-3/bind-free/banner.lisp] to print (as before) a
  message about how to turn on non-linear arithmetic, by modifying the call of
  ~c[value-triple] to use ~c[:on-skip-proofs t].  Thanks to Robert Krug for
  bringing this issue to our attention.

  Modified ~c[books/Makefile-subdirs] and ~c[books/Makefile-psubdirs] so that
  they can be used with ~c[books/Makefile-generic].  Thus, one can set things
  up so that `make' can be used to certify books both in the current
  directory and subdirectories, for example as follows.
  ~bv[]
    ACL2 = ../../saved_acl2

    arith-top: top all
    all: top

    DIRS = pass1 bind-free floor-mod
    include ../Makefile-subdirs
    include ../Makefile-generic

    top.cert: top.lisp
    top.cert: bind-free/top.cert
    top.cert: floor-mod/floor-mod.cert
    top.cert: floor-mod/mod-expt-fast.cert
  ~ev[]

  An experimental extension of ACL2 is under development by Bob Boyer and
  Warren Hunt to support function memoization, hash conses, and an applicative
  version of hash tables.  The default build of ACL2 does not include this
  extension, other than simple logic definitions of functions in new source
  file ~c[hons.lisp].  Future versions of ACL2 may fully incorporate this
  experimental extension.

  The ~ilc[defevaluator] event macro has been modified primarily by adding a
  new constraint as follows, where ~c[evl] is the evaluator.  The idea is that
  for the evaluation of a function call, one may replace each argument by the
  quotation of its evaluation and then also replace the alist environment with
  ~c[nil].
  ~bv[]
  (DEFTHMD UNHIDE-evl-CONSTRAINT-0
    (IMPLIES (AND (CONSP X)
                  (SYNTAXP (NOT (EQUAL A ''NIL)))
                  (NOT (EQUAL (CAR X) 'QUOTE)))
             (EQUAL (evl X A)
                    (evl (CONS (CAR X)
                               (KWOTE-LST (UNHIDE-evl-LIST (CDR X) A)))
                         NIL))))
  ~ev[]
  In order to support this change, there is another change: an evaluator maps
  ~c[nil] to ~c[nil] (note ~c[(AND X (CDR (ASSOC-EQ X A)))] in place of
  ~c[(CDR (ASSOC-EQ X A))] below).
  ~bv[]
  (DEFTHM UNHIDE-evl-CONSTRAINT-1
    (IMPLIES (SYMBOLP X)
             (EQUAL (UNHIDE-evl X A)
                    (AND X (CDR (ASSOC-EQ X A))))))
  ~ev[]
  With the new ~ilc[defevaluator], Dave Greve has been able to do a proof about
  beta reduction that seemed impossible before (see
  ~c[books/misc/beta-reduce.lisp]).  Thanks to Dave for suggesting an
  initial version of this change.

  Explicit compilation is now avoided for OpenMCL, resulting in fewer files to
  manage (no more files resulting from compilation) and, according to some
  tests, slightly faster run times.  ~l[compilation].  Thanks to Bob Boyer and
  Warren Hunt for suggesting this possibility.

  Now, the ~c[term-evisc-tuple] (~pl[ld-evisc-tuple]) is overridden by state
  global ~c[user-term-evisc-tuple] in all cases.  Formerly, this was only the
  case when ~c[term-evisc-tuple] was called with non-~c[nil] first argument.

  Symbols with the dot (~c[.]) character are generally no longer printed with
  vertical bars.  For example, before this change:
  ~bv[]
    ACL2 !>'ab.c
    |AB.C|
    ACL2 !>
  ~ev[]
  After this change:
  ~bv[]
    ACL2 !>'ab.c
    AB.C
    ACL2 !>
  ~ev[]
  Thanks to Jared Davis for suggesting this improvement.

  Fixed bugs in ~c[guard] verification for theorems.  The following examples
  illustrate these bugs.  If either theorem's body is executed in raw Lisp
  there is likely to be a hard Lisp error, even though ~ilc[verify-guards] was
  supposed to ensure against that behavior.
  ~bv[]
  ; Example: Verify-guards failed to check that all functions in the theorem
  ; had already been guard-verified.
  (defun my-car (x) (car x))
  (defthm my-car-compute-example (equal (my-car 3) (my-car 3)))
  (verify-guards my-car-compute-example)

  ; Example: Verify guards of a theorem whose body uses state improperly.
  (defthm bad-state-handler
    (if (state-p state)
        (equal (car state) (car state))
      t)
    :rule-classes nil)
  (verify-guards bad-state-handler)
  ~ev[]

  ~l[GCL] for an example, developed with Warren Hunt and Serita Nelesen,
  that shows how to get fast fixnum (small integer) arithmetic operations in GCL.

  Fixnum declarations are now realized as ~c[(signed-byte 30)] and
  ~c[(unsigned-byte 29)] instead of what was generally ~c[(signed-byte 29)] and
  ~c[(unsigned-byte 28)].  MCL users may thus find better performance if they
  switch to OpenMCL.  Note that some definitions have changed correspondingly;
  for example, ~ilc[zpf] now ~ilc[declare]s its argument to be of type
  ~c[(unsigned-byte 29)] instead of ~c[(unsigned-byte 28)].  A few ~il[books]
  may thus need to be adjusted; for example, changes were made to
  books in ~c[books/data-structures/memories/].

  ACL2's rewriter now avoids removing certain true hypotheses and false
  conclusions.  When a hypothesis rewrites to true or a conclusion rewrites to
  false, ACL2 formerly removed that hypothesis or conclusion.  Now, it only
  does such removal when the hypothesis or conclusion is either a call of
  ~ilc[equal] or an equivalence relation (~pl[equivalence]), or else is
  sufficiently trivial (roughly, either redundant with another hypothesis or
  conclusion or else trivially true without considering the rest of the goal).
  A specific example may be found in source file ~c[simplify.lisp]; search for
  ``; But we need to do even more work''.  Thanks to Robert Krug for providing
  the idea for this improvement and its initial implementation.  As is common
  with heuristic changes, you may find it necessary on occasion to rename some
  subgoals in your ~il[hints].  And in this case, you might also find it
  necessary on rare occasions to add ~c[:do-not '(generalize)] ~il[hints].

  A new function, ~c[mfc-relieve-hyp], allows (for example) for more powerful
  ~ilc[bind-free] hypotheses, by providing an interface to the rewriter's
  routine for relieving hypotheses.  ~l[extended-metafunctions].  Thanks to
  Robert Krug for providing the idea for this feature and its initial
  implementation.

  Two improvements have been made to non-linear arithmetic
  (~pl[non-linear-arithmetic]).  One allows for deducing strict inequality
  (~c[<]) for the result of certain polynomial multiplications, where
  previously only non-strict inequality (~c[<=]) was deduced.  A second allows
  the use of the product of two polynomials when at least one of them is known
  to be rational.  We had previously restricted the use of the product to the
  case where both were known to be rational.  Thanks to Robert Krug for these
  improvements.

  (OpenMCL and Allegro CL only) Fixed ACL2's redefinitions of raw Lisp
  ~c[trace] and ~c[untrace] in OpenMCL and Allegro CL so that when given no
  arguments, they return the list of traced functions.  For ~c[trace], this is
  an ANSI spec requirement.  Note that ~ilc[trace$] and ~ilc[untrace$] continue
  to return ~c[nil] in the ACL2 loop.

  Fixed a bug that was allowing the symbol ~c[&whole] to appear in other than
  the first argument position for a ~ilc[defmacro] event, in violation of the
  Common Lisp spec (and leading to potentially surprising behavior).  Thanks to
  Peter Dillinger for bringing this bug to our attention.

  It had been illegal to use ~ilc[make-event] under some calls of ~ilc[ld].
  This has been fixed.  Thanks to Jared Davis for bringing this issue to our
  attention with a simple example, in essence:
  ~bv[]
  (ld '((defmacro my-defun (&rest args) `(make-event '(defun ,@args)))
        (my-defun f (x) x)))
  ~ev[]

  ACL2 no longer prohibits certain ~ilc[make-event] forms when including
  uncertified ~il[books].  Thanks to Peter Dillinger for first bringing this
  issue to our attention.

  Hard errors arose when using ~il[break-rewrite] stack display commands, in
  particular ~c[:path] and ~c[:frame], from inside the ~il[proof-checker].
  This has been fixed.

  Fixed a bug that could cause functions that call system built-ins
  ~c[f-put-global], ~c[f-get-global], or ~c[f-boundp-global] to cause a raw
  Lisp error even when proving theorems.  Thanks to Peter Dillinger, for
  reporting such a failure for the form ~c[(thm (w '(1 2 3)))].

  Renamed the formal parameters of function ~c[set-equal] in distributed book
  ~c[books/arithmetic-3/bind-free/normalize.lisp] so that more distributed
  books can be included together in the same session.  In particular books
  ~c[books/data-structures/set-theory] and ~c[books/arithmetic-3/extra/top-ext]
  can now be included together.  Thanks to Carl Eastlund for bringing this
  problem to our attention and to Robert Krug for suggesting the formals
  renaming as a fix.

  Metafunctions must now be executable.  ~l[meta].

  New utilities allow for user-defined simplifiers at the goal level, both
  verified and unverified (``trusted''), where the latter can even be defined
  by programs outside ACL2.  ~l[clause-processor], which points to a new
  directory ~c[books/clause-processors/] that contains examples of these new
  utilities, including for example a system (``SULFA'') contributed by Erik
  Reeber that implements a decision procedure (thanks, Erik).  Also
  ~pl[proof-checker-commands] for the new ~il[proof-checker] command
  ~c[clause-processor] (or for short, ~c[cl-proc]).

  The rewriter has been tweaked to run faster in some cases involving very
  large terms.  Thanks to Eric Smith and Jared Davis for providing a helpful
  example that helped us locate the source of this inefficiency.

  Added ~c[books/make-event/defspec.lisp].  This book shows how one can mimic
  certain limited forms of higher-order statements in ACL2 by use of macros,
  ~ilc[make-event], and ~ilc[table] events.  Thanks to Sandip Ray for his
  contribution.

  A new release of the RTL library, ~c[books/rtl/rel7/], replaces the previous
  version, ~c[books/rtl/rel6/].  Thanks to Hanbing Liu and David Russinoff for
  providing this new version.

  We thank David Russinoff for providing a proof of the law of quadratic
  reciprocity.  See ~c[books/quadratic-reciprocity/Readme.lsp].

  Eliminated a slow array warning (~pl[slow-array-warning]) that could occur
  when exiting a ~il[wormhole] after executing an ~ilc[in-theory] event in that
  wormhole.  Thanks to Dave Greve for bringing this problem to our attention.

  A new accessor, ~c[(mfc-rdepth mfc)], provides a new field, the remaining
  rewrite stack depth, which has been added to metafunction context structures;
  ~pl[extended-metafunctions].  Thanks to Eric Smith for suggesting this
  addition.

  The algorithms were modified for collecting up rule names and other
  information used in proofs, into so-called ``tag-trees''.  Tag-trees are now
  free of duplicate objects, and this change can dramatically speed up some
  proofs that involve many different rules.  Thanks to Eric Smith for doing
  some profiling that brought this issue to our attention, and for reporting
  that this change reduced proof time on an example by about 47% (from 3681.46
  reported seconds down to 1954.69).

  All legal ~c[xargs] keywords may now be used in ~ilc[verify-termination]
  ~il[events].  In particular, this is the case for ~c[:normalize].

  (SBCL and CMUCL only) Fixed a problem with stobj array resizing functions
  that was causing a hard error in ACL2 images built on SBCL or CMUCL.

  A new ~il[table], ~ilc[evisc-table], allows you to introduce print
  abbreviations, for example for large constants.  Moreover, a new reader macro
  ~-[] ~c[#,] ~-[] makes it convenient to reference constants even inside a
  quote.  ~l[evisc-table].  Thanks to Bob Boyer and Warren Hunt for useful
  discussions leading to this feature.

  The macros in ~c[books/misc/expander.lisp] now have a new keyword argument,
  ~c[:simplify-hyps-p].  The default behavior is as before, but now case
  splitting from hypothesis simplification can be avoided.  For details,
  evaluate ~c[(include-book \"misc/expander\" :dir :system)] and then
  ~c[:doc!  defthm?] and ~c[:doc! symsym].  Thanks to Daron Vroon for sending a
  question that prompted this additional functionality.

  ACL2 failed to apply ~c[:]~ilc[restrict] hints to rules of class
  ~c[:]~ilc[definition], except for the simplest sorts (~pl[simple]).  This has
  been fixed.  Thanks to Jared Davis for pointing out this bug by sending a
  small example.

  Added a new ~c[:msg] argument to ~c[assert-event]; ~pl[assert-event].  The
  implementation of ~c[value-triple] has been modified to support this change.

  Fixed a bug in macro ~c[io?] that now allows the ~c[commentp] argument to be
  ~c[t].  This provides a way other than ~c[cw] to print without modifying
  state, for example as follows.  (Warning: Certain errors may leave you in a
  ~il[wormhole], in which case use ~c[:a!] to abort.)
  ~bv[]
  ACL2 !>(prog2$ (io? event t state
                      ()
                      (fms \"Howdy~~%\" nil *standard-co* state nil))
                 (+ 3 4))

  Howdy
  7
  ACL2 !>:set-inhibit-output-lst (proof-tree event)
   (PROOF-TREE EVENT)
  ACL2 !>(prog2$ (io? event t state
                      ()
                      (fms \"Howdy~~%\" nil *standard-co* state nil))
                 (+ 3 4))
  7
  ACL2 !>
  ~ev[]

  ACL2 now disallows calls of ~ilc[progn!] inside function bodies, just as it
  already disallowed such calls of ~ilc[progn], since in both cases the Common
  Lisp meaning differs from the ACL2 meaning.

  Redefinition of system functions now always requires an active trust tag
  (~pl[defttag]).  This restriction was intended before, but there was a hole
  that allowed a second redefinition without an active trust tag.  Thanks to
  Peter Dillinger for pointing out this bug.

  ~ilc[Verify-termination] has been disabled for a few more built-in functions
  that are in ~c[:]~ilc[program] mode.  (If you are curious about which ones
  they are, evaluate ~c[(f-get-global 'built-in-program-mode-fns state)].)
  [Note added for Version_3.4: This state global has been changed to
  'program-fns-with-raw-code.]  Moreover, such functions now will execute only
  their raw Lisp code, so for example they cannot be called during
  macroexpansion.  Thanks to Peter Dillinger and Sandip Ray for useful
  discussions on details of the implementation of this restriction.

  New untouchable state global variables, ~c[temp-touchable-vars] and
  ~c[temp-touchable-fns], can control the enforcement of untouchability.
  ~l[remove-untouchable].  Thanks to Peter Dillinger for suggesting these
  features.

  The ``TTAG NOTE'' string was being printed by ~ilc[encapsulate] events
  whenever an active trust tag was already in effect (~pl[defttag]), even if
  the encapsulate event contained no ~ilc[defttag] event.  This has been
  fixed.  Thanks to Peter Dillinger for a query leading to this fix.

  Fixed a bug in ~ilc[progn!] that could leave the user in raw-mode
  (~pl[set-raw-mode]).  This could occur when certifying a book with a
  ~c[compile-flg] value of ~c[t] (~pl[certify-book]), when that book contained
  a ~ilc[progn!] event setting raw-mode to ~c[t] without setting raw-mode back
  to ~c[nil]:
  ~bv[]
  (progn! (set-raw-mode t) ...)
  ~ev[]

  ~/~/")

(deflabel |NOTE-3-2(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.2(r) (April, 2007) Notes~/

  Changed the default distributed ~il[books] directory for ACL2(r) from
  ~c[books/] to ~c[books/nonstd/].  ~l[include-book], in particular the
  discussion of ``Distributed Books Directory''.

  Added directory ~c[books/arithmetic-3/] and its subdirectories to
  ~c[books/nonstd/].  (But a chunk of theorems from
  ~c[arithmetic-3/extra/ext.lisp] are ``commented out'' using
  #-:non-standard-analysis because they depend on ~c[books/rtl/rel7/], which is
  not yet in ~c[books/nonstd/]; feel free to volunteer to remedy this!)

  Incorporated changes from Ruben Gamboa to some (linear and non-linear)
  arithmetic routines in the theorem prover, to comprehend the reals rather
  than only the rationals.

  ~/

  Please also ~pl[note-3-2] for changes to Version  3.2 of ACL2.

  ~/
  ")

(deflabel note-3-2-1

; Here is the proof of nil from Sol Swords referenced in the mbe bug discussion
; below.

;  (defun foo (a b)
;    (mbe :logic (mv a b)
;         :exec (mv a b)))
;
;  ;; (foo 'a 'b) returns (A NIL);
;  ;; should return (A B)
;
;  (defthm foo-1-nil
;    (equal (mv-nth 1 (foo 'a 'b)) nil)
;    :rule-classes nil)
;
;  (verify-guards foo)
;  ;; now the correct behavior returns:
;  ;; (foo 'a 'b) returns (A B).
;
;  (defthm foo-1-b
;    (equal (mv-nth 1 (foo 'a 'b)) 'b)
;    :rule-classes nil)
;
;  (thm
;   nil
;   :hints (("Goal" :use (foo-1-nil foo-1-b))))

; Modified warning$ a bit to avoid what appears to be a GCL compilation bug.

; Fixed *hons-primitives*, and hence *acl2-exports*, to be independent of
; whether or not #+hons.  This should make it possible to certify a book in the
; hons version and include it in the non-hons version, or vice-versa.

; Added patch for non-ANSI GCL to compiler::wrap-literals.

; Modified rune-< to comprehend the cddr of a rune.

; Added books/misc/misc2/ and put new book misc.lisp in it.

; Changed handling of built-in-program-mode-fns [called
; program-fns-with-raw-code starting in Version_3.4] outside boot-strap so that
; with guard-checking-on equal to :all or :none and safe-mode off, they behave
; the same as other functions.  Thanks to Peter Dillinger for this change.
; Also introduced two similar state globals -- logic-fns-with-raw-code and
; macros-with-raw-code -- and corresponding defconsts for the initial values of
; all three.  And, used the two *fns-with-raw-code state globals in place of
; *non-exec-fns* in compile-uncompiled-defuns, and eliminated *non-exec-fns*.

; Improved redefinition error message to show book in which redefinition is
; being attempted, in addition to already-shown book in which previous
; definition resides.

; Fixed books/clause-processors/Makefile so that "make clean" will also clean
; directories SULFA/c-files/ and SULFA/scripts/.

; Fixed error output redirection in books/Makefile-generic.  Now, for example,
; various tests under clause-processors/SULFA/ will now direct all output to
; .out files as expected, rather than producing some at the terminal.

; Here is the example promised in the release note below about "handling of
; computed hints related to the stable-under-simplificationp parameter".
; Consider a computed hint like the following (taken from an actual example).
;
; `(:computed-hint-replacement
;   ((adviser-default-hint id clause world stable-under-simplificationp))
;   :use ,(build-hints literals rules hyps nil)
;   :expand ,(build-expand-hint literals))
;
; Here is a new version, which after the fix causes the previous behavior.  The
; idea is that the bug caused ACL2 to go to the preprocess-clause ledge of the
; waterfall, rather than to the top where :use hints are processed.  Thus, the
; :expand hint was applied first, and then after a return to the top of the
; waterfall, the :use hint was applied.  With the new version just below, that
; behavior is encoded explicitly.
;
; `(:computed-hint-replacement
;   ('(:computed-hint-replacement
;      ((adviser-default-hint id clause world stable-under-simplificationp))
;      :use ,(build-hints literals rules hyps nil)))
;   :expand ,(build-expand-hint literals))

; Sped up checksums by perhaps 5% in GCL by avoiding some arithmetic and
; perhaps indirection or boxing in function 32-bit-integerp.

; Made trivial mod to books/Makefile-generic, thanks to Rockwell Collins, that
; could avoid problems when cert.acl2 ends with a line containing `;' and no
; newline.

; In a test of v3-2 vs. the development version (both GCL), with efficiency
; mods (those below through the add-include-book-dir mod), cut the time by 57%
; for this include-book (could be representative):
;  (time$ (include-book "rtl/rel7/support/support/top" :dir :system))
;  8.74 s vs. 3.73 s (3 run average, run-gbc + run time)

; Eliminated FAILED target from books/Makefile-generic, inlining its function
; instead.

; Added dependency to books/make-event/Makefile:
; defspec.cert: eval.cert

; Improved (not yet advertised) feature rewrite-equiv to avoid some loops.
; Thanks to Dave Greve for bringing this to our attention with a very helpful
; example, which has been incorporated (slightly modified) as a comment, with
; the fix, into rewrite-solidify-rec.

; Modified certify-book-fn to do a full GC (garbage collection) in GCL after
; world rollback, just before the include-book phase.  We have some hope that
; this could help reclaim the space taken by .o files from subsidiary books
; that are about to be re-included.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  3.2.1 (June, 2007) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  (OpenMCL and multi-threaded SBCL only) Fixed a soundness bug in the
  evaluation of ~ilc[mbe] forms in the presence of Lisp feature
  ~c[:acl2-mv-as-values].  Thanks to Sol Swords for reporting this problem and
  sending a simple proof of ~c[nil] (which can be found in a comment in the
  ACL2 sources, in ~c[(deflabel note-3-2-1 ...)]).

  Added a new utility, ~ilc[dmr] (Dynamicaly Monitor Rewrites), for watching
  the activity of the rewriter and some other proof processes.  ~l[dmr].  We
  thank Robert Krug for useful contributions.

  Fixed a bug in evaluation of calls of ~ilc[with-prover-time-limit].

  Fixed the writing of executable scripts when building ACL2, so that the
  build-time value of environment variable ~c[ACL2_SYSTEM_BOOKS] is no longer
  written there.  Thanks to Dave Greve for discussing this change.

  Fixed bugs in ~c[:]~ilc[pl] (which are similarly present in the
  ~il[proof-checker]'s ~c[sr] (~c[show-rewrites]) command.  The first bug was
  evident from the following forms sent by Robert Krug, which caused an error.
  ~bv[]
  (include-book \"arithmetic-3/floor-mod/floor-mod\" :dir :system)
  :pl (mod (+ 1 x) n)
  ~ev[]
  The second bug was due to a failure to take note of which rules are disabled,
  and could be seen by executing the following (very slow!).
  ~bv[]
  (defstub modulus () t)
  (include-book \"arithmetic-3/floor-mod/floor-mod\" :dir :system)
  :pl (mod (+ x y) (modulus))
  ~ev[]

  Modified ~ilc[certify-book] so that by default, all executable-counterpart
  functions (sometimes called ``*1* functions'') are compiled.  This is the
  behavior that was already supported with a ~c[compile-flg] argument of
  ~c[:all]; the change is that argument ~c[t] now has this behavior as well
  (and ~c[:all] is supported only for legacy purposes).  A new value for
  ~c[compile-flg], ~c[:raw], gives the behavior formerly produced by value
  ~c[t], namely where executable-counterpart functions are not compiled.  The
  above changes are irrelevant if compilation is suppressed; ~pl[compilation].
  Finally, if environment variable ~c[ACL2_COMPILE_FLG] is set, then after
  converting to upper-case this environment variable's value of ~c[\"T\"],
  ~c[\"NIL\"], or ~c[\":RAW\"] will determine the value of the optional
  ~c[compile-flg] argument to be ~c[t], ~c[nil], or ~c[:raw], respectively,
  when this argument is not explicitly supplied.

  Modified ~ilc[include-book] so that ~c[:comp] argument now acts like
  ~c[:comp!], i.e., compiling a file that includes the file together with all
  executable counterpart (so-called ``*1*'') functions.  A new argument,
  ~c[:comp-raw], has the behavior that ~c[:comp] had formerly, i.e., compiling
  the actual book only.

  The function ~ilc[nonnegative-integer-quotient] is now computed in raw Lisp
  by calling ~ilc[floor] on its arguments.  This change was suggested by Peter
  Dillinger, in order to avoid stack overflows such as reported by Daron
  Vroon.  A new book, ~c[books/misc/misc2/misc.lisp], contains a proof of
  equivalence of ~ilc[nonnegative-integer-quotient] and ~ilc[floor], and serves
  as a repository for other miscellaeous proofs, including those justifying
  ACL2 modifications such as this one.

  Enhanced ~ilc[accumulated-persistence] to break down results by useful vs.
  useless rule applications.  In particular, this provides information about
  which rules were ever applied successfully, as requested by Bill Young.

  Added coverage of ~c[:]~ilc[meta] rules to the ~ilc[accumulated-persistence]
  statistics.

  Fixed a bug that was causing a ~c[:]~ilc[clause-processor] hint to fire on a
  subgoal of the goal to which it was attached, when the original application
  didn't change the clause.  Thanks to Dave Greve for pointing out this bug and
  providing a useful example.

  Fixed a bug in handling of computed ~il[hints] related to the
  ~c[stable-under-simplificationp] parameter (~pl[computed-hints]).  There were
  actually two bugs.  A minor but confusing bug was that the same goal was
  printed twice upon application of such a hint.  The major bug was that
  ~c[:use] ~il[hints] (as well as other ``top'' hints: ~c[:by], ~c[:cases], and
  ~c[:clause-processor]) were not being applied properly.  Thanks to Jared
  Davis for sending an example some time ago that showed the duplicate
  printing, and to Dave Greve for sending an example showing mis-application of
  ~c[:]~ilc[clause-processor] ~il[hints].  Note that you may find that existing
  computed hints using the ~c[stable-under-simplificationp] parameter no longer
  have the same behavior; see a comment about computed hints in ~c[note-3-2-1],
  ACL2 source file ~c[ld.lisp], for an example of how you might want to fix
  such computed hints.

  David Russinoff has contributed an updated version of
  ~c[books/quadratic-reciprocity/] including minor modifications of the
  treatment of prime numbers and a proof that there exist infinitely many
  primes.  Thanks to David for contributing this work, and to Jose Luis
  Ruiz-Reina for posing the challenge.

  Reduced the sizes of some ~il[certificate] (~c[.cert]) files by relaxing the
  test that allows original ~ilc[defpkg] ~il[events] to be placed there, rather
  than evaluating the import list term into an explicit list of symbols.

  Improved execution efficiency slightly for function ~c[rcdp] in file
  ~c[books/misc/records.lisp], by using ~ilc[mbe] to introduce a tail-recursive
  body.

  The executable script generated by ~ilc[save-exec] (and by the normal build
  process) now includes a time stamp as a comment.  Thanks to Jared Davis for
  suggesting this change in order to support his use of ~c[omake].  In the
  process, we also arranged that the startup banner for an executable created
  by ~ilc[save-exec] shows all of the build (save) times, not just the one for
  the original image.

  Sped up most redundant ~ilc[defpkg] ~il[events] by avoiding evaluation and sorting
  of the imports list in the case of identical event forms.  And, for
  ~ilc[defpkg] events that are not redundant, sped up their processing in Allegro
  CL (and perhaps other Lisps, but apparently not GCL) by using our own
  ~c[import] function.

  Modified ~ilc[add-include-book-dir] so that it refuses to associate a keyword
  with a different directory string than one it is already bound to.
  ~l[delete-include-book-dir] for how to remove the existing binding first.
  Thanks to Robert Krug for pointing out that without this change, one can find
  it difficult to debug a failure due to rebinding a keyword with
  ~ilc[add-include-book-dir].

  Added a new value for the ~c[:do-not-induct] hint (~pl[hints]),
  ~c[:otf-flg-override], which causes ACL2 to ignore the ~c[:]~ilc[otf-flg]
  when considering whether to abort the proof because of a ~c[:do-not-induct]
  hint.  Thanks to Daron Vroon for suggesting such an addition.

  Modified the printing of messages for entering and exiting raw mode
  (~pl[set-raw-mode]), so that in particular they are inhibited during
  ~ilc[include-book] or whenever ~c[observation]s are inhibited
  (~pl[set-inhibit-output-lst]).  Thanks to Peter Dillinger for suggesting such
  a change.

  (For system hackers only.)  The handling of ~il[events] of the form
  ~c[(progn! (state-global-let* ...))] had a bug that was causing bindings to
  be evaluated twice.  Moreover, the use of system function
  ~ilc[state-global-let*] is suspect in raw Lisp.  We have eliminated special
  treatment of ~c[state-global-let*] by ~c[progn!] in favor of a new keyword
  argument, ~c[state-global-bindings], that provides the intended
  functionality.  ~l[progn!].  Moreover, special handling that allowed
  ~ilc[make-event] forms under ~c[state-global-let*] has been removed; the
  desired effect can be obtained using (progn! :state-global-bindings ...).
  Thanks to Peter Dillinger for pointing out the above bug and collaborating on
  these changes.

  Incorporated backward-compatible enhancements to ~c[books/misc/expander.lisp]
  from Daron Vroon (documented near the top of that file).

  The specification of ~c[:backchain-limit-lst] had required that only a single
  (~c[:]~ilc[rewrite], ~c[:]~ilc[linear], or ~c[:]~ilc[meta]) rule be
  generated.  We have weakened this restriction to allow more than one rule
  provided that each rule has the same list of hypotheses.  For example, the
  rule class ~c[(:rewrite :backchain-limit-lst 1)] is now legal for the
  corollary formula ~c[(implies (f x) (and (g x) (h x)))], where this was not
  formerly the case.  Thanks to Dave Greve for bringing this issue to our
  attention.

  ~/~/")

(deflabel |NOTE-3-2-1(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.2.1(r) (June, 2007) Notes~/

  ~/

  Please also ~pl[note-3-2-1] for changes to Version  3.2.1 of ACL2.

  ~/
  ")

(deflabel note-3-3

; Modified function error-trace-suggestion to suggest WET only for lisps that
; support it.

; Fixed emacs/log-emacs-code.el (not distributed) to try to be
; platform-independent.  Modified the following makefiles in support of that:
;   books/workshops/2003/hbl/support/Makefile
;   books/workshops/2004/sumners-ray/support/Makefile
;   books/workshops/2003/kaufmann/support/rtl/Makefile
;   books/workshops/1999/multiplier/Makefile
;   books/Makefile

; Book defpun-exec-domain-example.lisp was moved from misc to misc2, so that a
; form depending on make-event could be uncommented.

; Fixed spacing issue in defun messages: too many spaces before
; type-prescription message.

; Modified books/data-structures/structures.lisp to eliminate compiler warnings
; for ignored variables.

; Improved a heretofore misleading error message produced by guess-measure when
; no controller-alist is supplied with a :definition rule.  The message had
; made it appear (erroneously) that the function's original definition doesn't
; have a controller-alist.  Thanks to John Cowles for bringing this to our
; attention.

; Eliminated (defmacro checkpoints ...) and (defun print-checkpoints ...),
; which were defined merely to cause errors for those who used pstack features
; back in Version_2.7.

; Fixed a bug in parse-primes: eq was called for character comparison instead
; of eql, causing a violation during macroexpansion (because of safe-mode).

; Modified some command printing (see print-ldd-full-or-sketch) to recognize
; defund and defthmd.

; Fixed books/misc/bash.lisp so that state global 'guard-checking-on is bound
; to nil, as required by pc-prove (and now documented there).

; Modified message about abandoning induction when a "formula is subsumed by
; one of its parents", following a conversation initiated by Bill Legato.

; Changed name of function print-goal to print-pc-goal.

; Fixed dependencies in Makefiles under
; books/workshops/2003/kaufmann/support/, which were causing problems with
; "make -j 8".

; Fixed problem with error message in check-exact-free-vars.

; Changed calls of len to calls of length in newly-defined-top-level-fns.

; Modified and-list and or-list in books/clause-processors/equality.lisp in
; order to be a bit more efficient using guards.

; Added a type (essentially, fixnum) declaration for a return case in fmt0 (but
; it's not clear how much GCL boxing that will avoid), and added calls of
; the-fixnum in enabled-numep and enabled-arith-numep.

; Improved error message on encapsulate or include-book failure to give
; information when encountering a record-expansion form, to help in debugging
; problems related to user's make-event forms.

; Changed io? type in print-rule-storage-dependencies from 'prove to 'event.

; Error reporting has been improved slightly when mv is called with the wrong
; number of arguments in a function body.  Thanks to Warren Hunt for suggesting
; that the error message be improved.

; Changed names mod* and floor* to mod-induct-fn and floor-induct-fn,
; respectively, in books/arithmetic-3/floor-mod/floor-mod.lisp.  This change
; probably won't affect anyone, except to make arithmetic-3 more compatiible
; with the rtl libraries (which define mod*), since these functions are only
; used to set up inductions.

; Eliminated an error when using state-global-let* with an empty binding list,
; e.g., (progn! :state-global-bindings nil).

; Fixed warning:
; ignoring old commands for target `clean-more'
; with 'make clean' in books/nonstd/workshops/1999/calculus/.

; Here are examples that illustrate bugs (1) through (3) mentioned in the
; paragraph on "Fixed the following issues with packages and book
; certificates".  Try these in Version_3.2.1 and you will see the bugs; but,
; these work fine in Version_3.3.  In the following, foo.lisp has just the
; forms (in-package "ACL2") and (defun foo (x) x).
;
; (1) After doing the following, foo.cert has needless defpkg forms among the
; portcullis commands in Version_3.2.1 but not in Version_3.3.
;
; (include-book "arithmetic/top" :dir :system)
; (certify-book "foo" 1)
;
; (2) Version_3.2.1 causes errors when following the directions below, but
; Version_3.3 does not.  Here, foo.lisp is as above and bar.lisp has only the
; forms (in-package "ACL2") and (defun bar (x) x).
;
; (defpkg "FOO" nil)
; (defconst *a* 'foo::a)
; (certify-book "foo" 2)
;
; [Restart ACL2]
; (encapsulate () (local (include-book "foo")) (defun h (x) x))
; (certify-book "bar" 1)
;
; (3) Watch (include-book "bar") fail at the end of the following instructions
; in Version_3.2.1, but not in Version_3.3.
;
; (set-ignore-ok :warn)
; (defpkg "FOO" (let ((x 3)) '(a b)))
; (certify-book "foo" 2)
;
; [Restart ACL2]
; (encapsulate () (local (include-book "foo")) (defun h (x) x))
; (certify-book "bar" 1)
;
; [Restart ACL2]
; (include-book "bar")

; Fixed books/clause-processors/Makefile to work on Sun/Solaris.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  3.3 (November, 2007) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  3.2.1 into new features,
  bug fixes, prover algorithm enhancements, and miscellaneous.  Also
  ~pl[note-3-2-1] for other changes since Version  3.2.

  ~st[NEW FEATURES]

  A new ``gag-mode'' provides less verbose, more helpful feedback
  from the theorem prover, in support of The Method
  (~pl[the-method]).  ~l[set-gag-mode].  We recommend the use of gag-mode,
  which may become the default in future ACL2 releases, and we welcome
  suggestions for improvement.  We thank Robert Krug and Sandip Ray for helpful
  feedback in the design of ~il[gag-mode].  Note that when proofs fail, then
  even without gag-mode and even if proof output is inhibited, the summary will
  contain a useful listing of so-called ``key checkpoints''
  (~pl[set-gag-mode]).

  Added support for a leading `~c[~~]' in filenames.  Thanks to Bob Boyer for
  suggesting this enhancement.  Note that since `~c[~~/]' depends on the user,
  it is disallowed in ~il[books] to be certified (~pl[certify-book]), since
  otherwise an ~ilc[include-book] form in a book, ~c[b], could have a different
  meaning at certification time than at the time ~ilc[include-book] is later
  executed on book ~c[b].

  Made a change to allow ~c[(time$ FORM)] and
  ~c[(with-prover-time-limit TIME FORM)] when ~c[FORM] includes
  ~ilc[make-event] calls that change the ACL2 ~il[world].  Thanks to Jared
  Davis for requesting such support for ~ilc[time$].

  Computed ~il[hints] (~pl[computed-hints]) may now produce a so-called ``error
  triple'', i.e., a result of the form ~c[(mv erp val state)], where a
  non-~c[nil] ~c[erp] causes an error, and otherwise ~c[val] is the value of
  the hint.  It remains legal for a computed hint to return a single ordinary
  value; indeed, the symbol form of a computed hint must still be a function
  that returns an ordinary single value.

  New hints provide additional control of the theorem prover, as follows.
  ~l[hints] for more details, and see new distributed book directory
  ~c[books/hints/] for examples, in particular file ~c[basic-tests.lisp] in
  that directory for simple examples.
  ~bq[]

  o The hint ~c[:OR (hints-1 ... hints-k)] causes an attempt to prove the
  specified goal using each ~c[hints-i] in turn, until the first of these
  succeeds.  If none succeeds, then the prover proceeds after heuristically
  choosing the ``best'' result, taking into account the goals pushed in each
  case for proof by induction.

  o A custom hint is a keyword that the user associates with a corresponding
  hint-generating function by invoking ~ilc[add-custom-keyword-hint].  Thus, a
  custom hint may be viewed as a convenient sort of computed hint.

  o A custom hint, ~c[:MERGE], is implemented in distributed book
  ~c[books/hints/merge.lisp].  It is useful for combining hints.

  o A sophisticated yet useful custom hint is the ~c[:CONSIDER] hint
  implemented in distributed book ~c[books/hints/consider-hint.lisp].  With
  this hint, you can for example give the equivalent of a ~c[:USE] hint without
  the need to supply an instantiation.  Include that book in order to see
  documentation online with ~c[:doc consideration], and see the book
  ~c[books/hints/consider-hint-tests.lisp] for examples.~eq[]

  A new hint, ~c[:]~ilc[reorder], allows the specification of which subgoals
  are to be considered first.  Thanks to Sandip Ray for putting forward this
  idea.

  Enhanced ~ilc[set-saved-output] by supporting a second argument of ~c[:same],
  which avoids changing which output is inhibited.

  Added macros ~c[thm?] and ~c[not-thm?] to distributed book
  ~c[books/make-event/eval.lisp], so that it's easy to test within a certified
  book that a proof attempt succeeds or that it fails.

  Added printing function ~ilc[cw!], which is analogous to ~ilc[cw] just as
  ~ilc[fmt!] is to ~ilc[fmt], i.e., printing so that the result can be read
  back in.  Thanks to Jared Davis for suggesting this enhancement (after doing
  his own implementation).

  The ACL2 customization file can now be specified using environment variable
  ~c[ACL2-CUSTOMIZATION] [note: starting with Version_4.0,
  ~c[ACL2_CUSTOMIZATION]].  ~l[acl2-customization].  Thanks to Peter Dillinger
  for requesting this feature.

  Added new emacs capabilities for proof trees (all documented in emacs):
  ~bq[]

  o New function start-proof-tree-noninteractive, for example~nl[]
    (start-proof-tree-noninteractive \"*shell*\")

  o C-z o      Switch to another frame

  o C-z b      Switch to prooftree buffer

  o C-z B      Switch to prooftree buffer in \"prooftree-frame\" frame~eq[]

  Added Common Lisp function, ~c[search], as a macro in ~ilc[logic] mode, with
  limited support for keyword arguments.  Thanks to Warren Hunt for requesting
  this addition.

  Sandip Ray has contributed a book, ~c[books/make-event/defrefine.lisp], that
  provides a collection of macros to aid in reasoning about ACL2 functions via
  refinement.

  Wrote and incorporated new utility for listing all the theorems in an
  included book.  See ~c[books/misc/book-thms.lisp].  Thanks to Jared Davis for
  requesting this functionality.

  The new distributed book ~c[misc/defp.lisp] generalizes the ~ilc[defpun]
  macro to allow more general forms of tail recursion.

  (Low-level printing improvement) A new function,
  ~c[set-ppr-flat-right-margin], allows the right margin for certain kinds of
  ``flat'' printing to exceed column 40.  Thanks to Jared Davis for pointing
  out that state global variables ~c['fmt-hard-right-margin] and
  ~c['fmt-soft-right-margin] are not alone sufficient to extend the right
  margin in all cases.

  The event ~ilc[add-include-book-dir] can now take a relative pathname as an
  argument.  Formerly, it required an absolute pathname.

  A new book, ~c[books/misc/defopener.lisp], provides a utility creating a
  theorem that equates a term with its simplification.

  ACL2 now provides limited support for the Common Lisp primitive ~c[FLET],
  which supports local function bindings.  ~l[flet].  Thanks to Warren Hunt for
  requesting this feature.

  Added a definition of ~ilc[boole$], a close analogue of Common Lisp function
  ~c[boole].  Thanks to Bob Boyer for providing an initial implementation.

  ~st[BUG FIXES]

  Fixed ~ilc[defstobj] to inhibit a potentially useless theory warning.

  Fixed a bug in the application of ~ilc[certify-book] to relative pathnames
  for files in other than the current directory.  Thanks to Amr Helmy for
  bringing this bug to our attention.

  Fixed a bug in ~c[:]~ilc[pl] and ~c[:]~ilc[pr] for displaying rules of class
  ~c[:]~ilc[meta].  Thanks to Jared Davis for finding this bug and providing a
  fix.

  Formerly, ~ilc[set-default-backchain-limit] was not a legal event form for
  ~ilc[encapsulate] forms and ~il[books].  This has been fixed.  Thanks to
  Robert Krug and Sandip Ray for bringing this bug to our attention.

  Fixed the handling of ~il[hints] in ~il[proof-checker] commands for the
  prover, such as ~c[bash] ~-[] ~pl[proof-checker-commands] ~-[] so that the
  user can override the default settings of hints, in particular of ~c[:do-not]
  and ~c[:do-not-induct] hints attached to ~c[\"Goal\"].  This fix also applies
  to the distributed book ~c[misc/bash.lisp], where Robert Krug noticed that he
  got an error with ~c[:hints ((\"Goal\" :do-not '(preprocess)))]; we thank
  Robert for pointing out this problem.

  Fixed a bug in handling of ~il[stobj]s occurring in guards of functions whose
  ~il[guard]s have been verified.  In such cases, a raw Lisp error was possible
  when proving theorems about non-''live'' ~il[stobj]s.  We thank Daron Vroon
  for sending us an example that highlighted this bug.  The following (simpler)
  example causes such an error in previous versions of ACL2.
  ~bv[]
  (defstobj st fld)
  (defun foo (st)
    (declare (xargs :stobjs st :guard (fld st)))
    st)
  (thm (equal (foo '(3))
              '(3)))
  ~ev[]

  The ~il[dmr] feature for dynamic monitoring of rewrites had a bug, where the
  file used for communicating with emacs was the same for all users, based on
  who built the ACL2 executable image.  This has been fixed.  Thanks to Robert
  Krug for bringing this bug to our attention.

  Fixed a bug in some warnings, in particular the warning for including an
  uncertified book, that was giving an incorrect warning summary string.

  Inclusion of uncertified books erroneously re-used ~ilc[make-event]
  expansions that were stored in stale ~il[certificate]s.  This is no longer
  the case.  Thanks to Jared Davis for bringing this bug to our attention.

  Fixed a bug that was disallowing calls of ~ilc[with-output] in ~il[events]
  that were executing before calling ~ilc[certify-book].

  Modified the functionality of ~c[binop-table] so other than binary function
  symbols are properly supported (hence with no action based on
  right-associated arguments).  ~l[add-binop].

  Fixed small ~il[proof-checker] issues related to packages.  Emacs commands
  ~c[ctrl-t d] and ~c[ctrl-t ctrl-d] now work properly with colon
  (`~c[:]') and certain other punctuation characters.  The ~c[p-top] command
  now prints ``~c[***]'' regardless of the current package.

  Fixed a bug that allowed ~ilc[certify-book] to succeed without specifying
  value ~c[t] for keyword argument ~c[:skip-proofs-okp], even with
  ~ilc[include-book] ~il[events] in the certification ~il[world] depending on
  events executed under ~ilc[skip-proofs].

  Improved ~ilc[show-accumulated-persistence] in the following two ways.
  Thanks to Robert Krug and Bill Young for requesting these improvements and
  for providing useful feedback.
  ~bq[]

  o It can provide more complete information when aborting a proof.

  o The ~c[:frames] reported for a rule are categorized as ``useful'' and
  ``useless'' according to whether they support ``useful'' or ``useless''
  ~c[:tries] of that rule, respectively.  ~l[accumulated-persistence] for
  further explanation.~eq[]

  Modified ~ilc[make-event] so that the reported time and warnings include
  those from the expansion phase.  In analogy with ~ilc[encapsulate] and
  ~ilc[progn], the rules reported still do not include those from subsidiary
  events (including the expansion phase).  A related change to ~ilc[ld] avoids
  resetting summary information (time, warnings) with each top-level form
  evaluation; ~il[events] already handle this information themselves.

  Fixed ~ilc[set-inhibit-output-lst] so that all warnings are inhibited when
  ~c[warning!] but not ~c[warning] is included in the list.  Formerly, only
  soundness-related warnings were inhibited in this case.  Thanks to Eric Smith
  for bringing this bug to our attention.

  Distributed directory ~c[doc/HTML/] now again includes installation instructions
  (which was missing in Version_3.2.1), in
  ~c[doc/HTML/installation/installation.html].

  Some fixes have been made for ~il[proof-tree] support.
  ~bq[]

  o ~il[Proof-tree] output is no longer inhibited automatically during
  ~ilc[certify-book], though it continues to be inhibited by default (i.e.,
  ACL2 continues to start up as though ~ilc[set-inhibit-output-lst] has been
  called with argument ~c['(proof-tree)]).

  o Fixed a bug in Xemacs support for ~il[proof-tree] help keys ~c[C-z h] and
  ~c[C-z ?].

  o Fixed a bug in ~il[proof-tree]s that was failing to deal with the case that
  a goal pushed for proof by induction is subsumed by such a goal to be proved
  later.  Now, the proof-tree display regards such subsumed goals as proved, as
  is reported in the theorem prover's output.~eq[]

  Fixed a bug that was disallowing ~ilc[value-triple] forms inside
  ~ilc[encapsulate] forms in a certification ~il[world] (~pl[portcullis]).

  If the ~c[:load-compiled-file] argument of a call of ~ilc[include-book] is
  ~c[:comp], then an existing compiled file will be loaded, provided it is more
  recent than the corresponding book (i.e., ~c[.lisp] file).  A bug was causing
  the compiled file to be deleted and then reconstructed in the case of
  ~c[:comp], where this behavior was intended only for ~c[:comp!].

  Fixed a bug that was avoiding compilation of some executable counterparts
  (sometimes called ``*1* functions'') during ~ilc[certify-book], and also
  during ~ilc[include-book] with ~c[:load-compiled-file] value of ~c[:comp] or
  ~c[:comp!]).  Thanks to Eric Smith for sending a small example to bring this
  bug to our attention.

  Incorporated a fix from Eric Smith for a typo (source function
  ~c[ancestors-check1]) that could cause hard Lisp errors.  Thanks, Eric!

  Fixed the following issues with packages and book ~il[certificate]s.
  ~l[hidden-death-package] if you are generally interested in such issues, and
  for associated examples, see comments on ``Fixed the following issues with
  packages'' in ~c[note-3-3] in the ACL2 source code.
  ~bq[]

  o Reduced the size of ~c[.cert] files by eliminating some unnecessary
  ~ilc[defpkg] events generated for the ~il[portcullis].

  o Fixed a bug that has caused errors when reading symbols from a
  ~il[portcullis] that are in undefined packages defined in locally included
  books.

  o Fixed a bug that could lead to failure of ~ilc[include-book] caused by a
  subtle interaction between ~ilc[set-ignore-ok] and ~ilc[defpkg] events
  generated for the ~il[portcullis] of a ~il[certificate].~eq[]

  ~st[PROVER ALGORITHM ENHANCEMENTS]

  Non-linear arithmetic (~pl[non-linear-arithmetic]) has been improved to be
  more efficient or more powerful in some cases.  Thanks to Robert Krug for
  contributing these improvements.

  Improved certain (so-called ``~ilc[type-set]'') reasoning about whether or
  not expressions denote integers.  Thanks to Robert Krug for contributing code
  to implement this change, along with examples illustrating its power that are
  now distributed in the book ~c[books/misc/integer-type-set-test.lisp].

  Improved ACL2's heuristics for relieving hypotheses, primarily to use linear
  reasoning on conjuncts and disjuncts of the test of an ~ilc[if] expression.
  For example, given a hypothesis of the form ~c[(if (or term1 term2) ...)],
  ACL2 will now use linear reasoning to attempt to prove both ~c[term1] and
  ~c[term2], not merely for ~c[term2].  Thanks to Robert Krug for supplying
  examples illustrating the desirability of such an improvement and for useful
  discussions about the fix.

  Made a slight heuristic change, so that when a hypothesis with ~ilc[let] or
  ~ilc[mv-let] subterms (i.e. ~ilc[lambda] subterms) rewrites to ~c[t], then
  that hypothesis is necessarily eliminated.  Thanks to Jared Davis for sending
  an example that led us to develop this change, and thanks to Robert Krug for
  a helpful discussion.

  ~st[MISCELLANEOUS]

  Added documentation on how to use ~ilc[make-event] to avoid duplicating
  expensive computations, thanks to Jared Davis.
  ~l[using-tables-efficiently].

  Modified the error message for calls of undefined functions to show the
  arguments.  Thanks to Bob Boyer for requesting this enhancement.

  Modified utilies ~c[:]~ilc[pr], ~c[:]~ilc[pr!], ~c[:]~ilc[pl], and
  ~c[:]~ilc[show-bodies] to incorporate code contributed by Jared Davis.  That
  code defines low-level source functions ~c[info-for-xxx] that collect
  information about rules, which is thus available to advanced users.

  Dynamic monitoring of rewrites (~pl[dmr]) has been improved in the following
  ways, as suggested by Robert Krug.
  ~bq[]

  o Some stale entries from the rewrite stack are no longer printed, in
  particular above ~c[ADD-POLYNOMIAL-INEQUALITIES].

  o An additional rewrite stack entry is made when entering non-linear
  arithmetic (~pl[non-linear-arithmetic]).

  o An ~c[ADD-POLYNOMIAL-INEQUALITIES] entry is printed with a counter, to show
  how often this process is called.~eq[]

  Modified ~ilc[save-exec] so that the newly-saved image will have the same raw
  Lisp package as the existing saved image.  This is a very technical change
  that will likely not impact most users; for example, the package in the ACL2
  read-eval-print loop (~pl[lp]) had already persisted from the original to
  newly-saved image.  Thanks to Jared Davis for suggesting this change.

  Changed ~ilc[make-event] expansion so that changes to ~ilc[set-saved-output],
  ~ilc[set-print-clause-ids], ~c[set-fmt-soft-right-margin], and
  ~c[set-fmt-hard-right-margin] will persist after being evaluated during
  ~c[make-event] expansion.  (Specifically,
  ~c[*protected-system-state-globals*] has been modified;
  ~pl[make-event-details].)  Thanks to Jared Davis for bringing this issue to
  our attention.

  Output from the ~il[proof-checker] is now always enabled when invoking
  ~ilc[verify], even if it is globally inhibited
  (~pl[set-inhibit-output-lst]).

  Improved the message printed when an ~c[:induct] hint fails, to give more
  information in some cases.  Thanks to Rex Page for suggesting where an
  improvement could be made and providing useful feedback on an initial
  improvement.

  Added a warning for ~il[congruence] rules (~pl[defcong]) that specify
  ~ilc[iff] as the second equivalence relation when ~ilc[equal] can be used
  instead.  Those who heed these warnings can eliminate certain subsequent
  ~ilc[double-rewrite] warnings for ~il[rewrite] rules with conclusions of the
  form ~c[(iff term1 term2)], and hence implicitly for Boolean conclusions
  ~c[term1] that are interpreted as ~c[(iff term1 t)].  Thanks to Sarah
  Weissman for sending us an example that highlighted the need for such a
  warning.

  Modified macro ~c[:]~ilc[redef!] (which is for system implementors) so that
  it eliminates untouchables.

  Several improvements have been made to the experimental hons/memoization
  version of ACL2.  ~l[hons-and-memoization].

  The distributed books directory, ~c[(@ distributed-books-dir)], is now
  printed in the start-up message.

  ~/~/")

(deflabel |NOTE-3-3(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.3(r) (November, 2007) Notes~/

  ~/

  Please also ~pl[note-3-3] for changes to Version  3.3 of ACL2.

  ~/
  ")

(deflabel note-3-4

; Here is the example promised in :doc note-3-4 for a soundness bug, namely a
; proof of nil based on the use of mbe inside an encapsulate event.  The proof
; goes through for this example in Version_3.3 but not in Version_3.4.

;  (encapsulate
;   ((f (x) t))
;
;   (local (defun f (x)
;            (declare (xargs :guard t))
;            x))
;
;   (defun g (x)
;     (declare (xargs :guard t))
;     (mbe :logic (f x)
;          :exec  x)))
;
;  (defthm g-3-is-3
;    (equal (g 3) 3)
;    :rule-classes nil)
;
;  (defthm f-3-is-3
;    (equal (f 3) 3)
;    :hints (("Goal"
;             :in-theory (disable (g))
;             :use g-3-is-3)))
;
;  (defun foo (x)
;    (declare (ignore x))
;    4)
;
;  (defthm contradiction
;    nil
;    :hints (("Goal" :use ((:functional-instance f-3-is-3
;                                                (f foo)))))
;    :rule-classes nil)

; Fixed a hard error during an error message caused by the use of the wrong ~@
; argument in unknown-binding-msg.  Thanks to Eric Smith for pointing this out.

; The source function for "Fixed bugs in" proof-checker code was
; pc-relieve-hyp.

; The following events prove in GCL and Allegro builds of ACL2  3.3.  Both the
; new trace and fixes to the existing replacements for trace (i.e., now using
; :native t) will fix this.
;
;     (defun f (x y)
;       (mv x y))
;
;     (defthm fact-about-f
;       (equal (f 3 4)
;              (mv 3 4))
;       :rule-classes nil)
;
;     (defttag :trace)
;
;     (trace$ (f :exit
;                (mv-let (u v)
;                        (mv 7 8)
;                        (equal u v))))
;
;     ; But the call of mv in the :exit condition messes up multiple-value
;     ; handling by ACL2, and now (f 3 4) is (mv 3 8).
;
;     (defthm contradiction
;       nil
;       :hints (("Goal" :use fact-about-f))
;       :rule-classes nil)

; Fixed error message at the end of translate-hint-expression (thanks to Robert
; Krug for the fix) to acknowledge that STATE can be used.

; Here is the example Peter Dillinger sent for the bug in the interaction of
; proof-trees with :induct hint value :otf-flg-override.  The fix was to
; consider do-not-induct-otf-flg-override in the definition of aborting-p in
; the definition of increment-proof-tree.

;  (start-proof-tree)
;
;  (defun limit-induction-hint-fn (i)
;    `(or (and (length-exceedsp (car id) ,i)
;              (endp (cdadr id))
;              (eq (cddr id) 0)
;              '(:computed-hint-replacement
;                t
;                :do-not-induct :otf-flg-override))
;         (and (> (cddr id) 20)
;              '(:computed-hint-replacement
;                t
;                :do-not (eliminate-destructors eliminate-irrelevance
;                                               generalize fertilize)
;                :in-theory nil))))
;
;  (add-default-hints
;   `(,(limit-induction-hint-fn 1)))
;
;  (defun sum-from (i lst)
;    (if (and (natp i)
;             (< i (len lst)))
;      (+ (nth i lst) (sum-from (1+ i) lst))
;      0))

; Fixed warning messages (error messages too actually) for computed hints, so
; that mere warnings don't say that a value is illegal.  Thanks to Robert Krug
; for pointing out this problem and sending the following example:

;  (in-theory (disable mv-nth eq))
;
;  (thm
;   (equal (* x y) (+ x y))
;   :hints ((if stable-under-simplificationp
;               '(:in-theory (enable eq))
;             nil)))

; Changed constant *primitive-event-macros* into zero-ary function
; primitive-event-macros, in support of Peter Dillinger's desire to be able to
; change this function with ttags.

; Modified GNUmakefile to use ACL2_SUFFIX for saved_acl2 suffix.

; ACL2 source macro defrec can now be used in :logic mode even when the cheap
; flag is nil.  Formerly, the presence of record-error made that impossible,
; but now record-error is in :logic mode, and with guards verified by using (er
; hard? ...) instead of (er hard ...).

; Strengthened the guard for er-let*.

; Fixed walking and flying tour pages to put the gif linking to the next page
; at the top too, not just the bottom of the page.

; Sped up macroexpansion of with-output forms by changing how OR is called in
; with-output in the loop.

; Fixed a bug in :pso, specifically in its "So we now return to" messages,
; which were printing a goal that had just been proved for induction.  The fix
; involved handling of of jppl-flg values in functions pop-clause-msg1 and
; pop-clause-msg.

; The bug that could cause certify-book to fail with local make-event forms was
; fixed by changing eval-event-lst so that it always concludes by setting
; state-global 'last-make-event-expansion to nil.  Before that, certification
; failed for a book whose only form (other than the leading in-package) is
; (local (make-event '(defun foo (x) x))).

; Modified expansion-alist-pkg-names slightly; memory escapes just why, but
; notes suggest that it was to fix a bug reported by Eric Smith.

; Fixed a bug in the guard for find-clauses1.

; Modified acl2-traced-fns to work in versions of CCL after June 2008.

; Following Bob Boyer, added (debug 0) to the optimize form for CCL (OpenMCL)
; in acl2.lisp.

; The soundness bug in value-triple-fn could be exploited not only to prove nil
; but also to get a raw Lisp error, by way of these events at the top level or
; in a book:
;
; (defun foo (x)
;   (declare (xargs :mode :program :guard t))
;   (car x))
; (value-triple (foo 3))

; Moved #-acl2-loop-only code in getprop-default into the logic.

; Stopped creating io-record forms whose :io-marker is other than ctx or nil;
; see comment in io-record.

; Fixed get-hints1, get-guard-hints1, and get-std-hints1 to avoid raw Lisp
; errors for :hints values not satisfying true-listp.  So now we get a cleaner
; error, for example, on the following:
; (defun foo (x) (declare (xargs :hints 3)) x)

; For developers, "make" variable ACL2_SAFETY can be set in order to change the
; optimization safety setting from its default of 0; see GNUmakefile.  The
; OPTIMIZE form has been changed from a DECLAIM to a PROCLAIM to support this
; change.

; Edited :doc |Other Requirements| to remove MCL from the list of supported
; Lisps.  It was already the case that MCL was not included in
; installation/requirements.html.  If ACL2 works in MCL, it's just luck at this
; point.

; Fixed warning "Unable to determine USER environment variable for
; dmr-file-name", which has been seen on Windows, so that it only occurs at the
; first time in a session that dmr-start is run.  Also printed what to do in
; Emacs when that happens.

; Following an observation of Robert Krug, now a message "*** Note: No
; checkpoints to print. ***" is printed when a proof fails but no checkpoints
; have been generated.

; ACL2 source function trace-multiplicity now returns nil instead of 1 for
; functions not known to the ACL2 loop.

; In gag-mode, when a proof fails to produce checkpoints, we now see "*** Note:
; No checkpoints to print. ***".  Thanks to Robert Krug for requesting this
; additional output.  To get an infinite loop without checkpoints, to check
; this fix:
;
; (defstub foo (x) t)
; (defaxiom ax1
;   (implies (syntaxp (variablep x))
;            (equal (foo x) (foo (cons 3 x)))))
; (defaxiom ax2 (equal (foo (cons x y)) (foo y)))
; (thm (foo a)
;      :hints ((and stable-under-simplificationp
;                   '(:computed-hint-replacement t :in-theory (enable ax2)))
;              (and (not stable-under-simplificationp)
;                   '(:computed-hint-replacement t :in-theory (disable ax2)))))

; Jared's email leading to the expansion of *acl2-exports*, as noted below also
; led to fixing of our-abort to print in the ACL2 current-package.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  3.4 (August, 2008) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  3.3 into changes to
  existing features, new features, bug fixes, new and updated books, and Emacs
  support.  Each change is described just once, though of course many changes
  could be placed in more than one category.

  ~st[CHANGES TO EXISTING FEATURES]

  Fixed a long-standing potential infinite loop in the rewriter.  Thanks to Sol
  Swords for sending a concise example illustrating the looping behavior.
  (Those interested in details are welcome to look at the comment about loop
  behavior in source function ~c[rewrite-equal].)

  Incorporated a slight strengthening of non-linear arithmetic contributed by
  Robert Krug (thanks, Robert).  With non-linear arithmetic enabled, the
  problem was essentially that ACL2 made the following ``optimization'': given
  inequalities ~c[(< a u)] and ~c[(< b v)], for positive rational constants
  ~c[a] and ~c[b] terms ~c[u] and ~c[v] of which at least one is known to be
  rational, infer ~c[(< (* a b) (* u v))].  Without this optimization, however,
  ACL2 now infers the stronger inequality obtained by direct multiplication of
  the two given inequalities.  To see the effect of this change, submit the
  event ~c[(set-non-linearp t)] followed by:
  ~bv[]
  (thm (implies (and (rationalp x) (< 3 x)
                     (rationalp y) (< 4 y))
                (< 0 (+ 12 (* -4 x) (* -3 y) (* x y)))))
  ~ev[]

  The utility ~ilc[set-checkpoint-summary-limit] has been modified in several
  ways: it now takes a single argument (no longer takes ~ilc[state] as an
  argument); a natural number ~c[n] abbreviates the pair ~c[(n . n)]; the
  argument is no longer evaluated, but it still optionally may be quoted; and a
  new value, ~c[t], suppresses all printing of the checkpoint summary.  Thanks
  to Jared Davis for suggesting most of these improvements.

  There was formerly a restriction on ~ilc[mbe] that the ~c[:exec] argument may
  not contain a call of ~ilc[mbe].  This restriction has been removed, thanks
  to a request from Jared Davis and Sol Swords.  Thanks also to Sandip Ray, who
  pointed out that this restriction may have been in place in order that
  ~ilc[defexec] can guarantee termination using the ~c[:exec] code; its
  ~il[documentation] has therefore been updated to clarify this situation.

  Rules of class ~c[:]~ilc[rewrite] are now stored by performing certain
  logical simplifications on the left side of the conclusion: ~c[(prog2$ X Y)]
  is replaced by ~c[Y], ~c[(mbe :logic X :exec Y)] is replaced by ~c[X] (more
  precisely, the analogous change is made to the generated call of
  ~ilc[must-be-equal]); and ~c[(the TYPE X)] is replaced by ~c[X] (again, the
  change is actually made on the macroexpanded form).  Thanks to Jared Davis
  and Sol Swords for requesting this change.  An analogous change has also been
  made for rules of class ~c[:]~ilc[forward-chaining].

  The ~ilc[trace$] utility has been reimplemented to work independently of the
  underlying Lisp trace.  It thus works the same for every host Lisp, except as
  provided by an interface to the underlying host Lisp trace (the ~c[:native]
  option).  Note that the host Lisp trace continues to be modified for GCL,
  Allegro CL, and CCL (OpenMCL); ~pl[trace].  ~l[trace$] for updated detailed
  documentation on tracing options, many of which are new, for example an
  ~c[:evisc-tuple] option that can be set to ~c[:no-print] if you want the
  function traced without the usual entry and exit printing.  The previous
  ~ilc[trace$] had some issues, including the following, which have all been
  fixed.  Thanks to Peter Dillinger for assistance in determining desired
  functionality of the new ~ilc[trace$] and for helping to test it.
  ~bq[]

  Recursive calls were not always shown in the trace for two reasons.
  (1) Compiler inlining could prevent recursive calls from being shown in the
  trace, in particular in CCL (OpenMCL).  Thanks to Jared Davis and Warren Hunt
  for pointing out this issue and requesting a fix, and to Bob Boyer and Gary
  Byers for relevant helpful discussions.  (2) ACL2's algorithm for producing
  executable counterparts prevented tracing of recursive calls even after
  ~c[(set-guard-checking :none)].  Thanks to Peter Dillinger for requesting a
  fix.

  It was possible to exploit a bug in the interaction of multiple values and
  trace to prove a contradiction.  An example is in a comment in
  ~c[(deflabel note-3-4 ...)] in the ACL2 source code.

  Certain large structures could cause expensive computations for printing even
  when a ~c[:cond] condition was specified and evaluated to ~c[nil].

  ~ilc[Trace!] now suppresses printing of the event summary, and returns the
  value that would be returned (if there is an active trust tag) by the
  corresponding call of ~ilc[trace$].

  Some bugs have been fixed in the underlying native trace installed by ACL2
  for GCL, Allegro CL, and CCL (OpenMCL), including the following.  In GCL it
  had been impossible to use the variable ~c[ARGLIST] in a ~c[:cond]
  expression.  In Allegro CL and CCL, a ~ilc[trace$] bug mishandled tracing
  non-ACL2 functions when directives such as ~c[:entry] and ~c[:exit] were
  supplied.  GCL trace now hides the world even when tracing non-ACL2
  functions.  Tracing in CCL no longer causes a Lisp error when untracing a
  traced function defined outside the ACL2 loop; for example ~c[(trace$ len1)]
  followed by ~c[(untrace$ len1)] no longer causes an error.~eq[]

  The macro ~c[wet] has been changed, for the better we think.  ~pl[wet].

  The generation of goals for ~il[forcing-round]s has been changed to avoid
  dropping assumptions formerly deemed ``irrelevant''.  (A simple example may
  be found in a comment in source function ~c[unencumber-assumption], source
  file ~c[prove.lisp].)  Thanks to Jared Davis for sending us an example that
  led us to make this change.

  Modified the implementation of ~ilc[make-event] so that in the ~il[certificate]
  of a book, ~ilc[local] events arising from ~ilc[make-event] forms are
  elided.  For example, if ~c[(make-event <form>)] expands to
  ~c[(local <expanded-form>)], then where the latter had been stored in the
  certificate, now instead ~c[(local (value-triple :ELIDED))] will be stored.
  Thanks to Eric Smith for requesting this improvement.  He has reported that a
  preliminary version of this improvement shrunk a couple of his ~c[.cert]
  files from perhaps 40MB each to about 140K each.

  Now, a ~ilc[table] event that sets the entire table,
  ~c[(table tbl nil alist :clear)], is redundant (~pl[redundant-events]) when
  the supplied ~c[alist] is equal to the current value of the table.  Thanks to
  Peter Dillinger for requesting this change.

  The event constructor ~ilc[progn!] now returns the value that is returned by
  evaluation of its final form if no error occurs, except that it still returns
  ~c[nil] if the that final evaluation leaves ACL2 in raw-mode.

  ~c[:]~ilc[Pso] and ~c[:]~ilc[psog] have been improved so that they show the
  key checkpoint summary at the end of a failed proof.
  (For a discussion of key checkpoints, ~pl[set-gag-mode].)  As a result, a
  call of ~ilc[set-checkpoint-summary-limit] now affects subsequent evaluation
  of ~c[:]~ilc[pso] and ~c[:]~ilc[psog].  In particular, you no longer need to
  reconstruct a proof (by calling ~ilc[thm] or ~ilc[defthm]) in order to see
  key checkpoints that were omitted due to the limit; just call
  ~ilc[set-checkpoint-summary-limit] and then run ~c[:pso] or ~c[:psog].

  The ~il[proof-checker] behaves a little differently under ~il[gag-mode].
  Now, proof-checker commands that call the theorem prover to create new
  proof-checker goals, such as ~c[bash] and ~c[induct]
  (~pl[proof-checker-commands]), will show key checkpoints when in
  ~il[gag-mode].  As before, proof-checker commands ~c[pso] and ~c[pso!] (and
  now, also ~c[psog]) ~-[] ~pl[pso], ~pl[psog], and ~pl[pso!] ~-[] can then
  show the unedited proof log.  However, unlike proof attempts done in the ACL2
  loop, such proof attempts will not show a summary of key checkpoints at the
  end, because from a prover perspective, all such goals were considered to be
  temporarily ``proved'' by giving them ``byes'', to be dispatched by later
  proof-checker commands.

  A little-known feature had been that a ~il[measure] of ~c[0] was treated as
  though no measure was given.  This has been changed so that now, a
  ~il[measure] of ~c[nil] is treated as though no measure was given.

  Expanded ~c[*acl2-exports*] to include every documented symbol whose name
  starts with ~c[\"SET-\"].  Thanks to Jared Davis for remarking that
  ~ilc[set-debugger-enable] was omitted from ~c[*acl2-exports*], which led to
  this change.

  The ~il[trace] mechanism has been improved so that the ~c[:native] and
  ~c[:multiplicity] options can be used together for Lisps that support
  the trace ~c[:exit] keyword.  These Lisps include GCL and Allegro CL, whose
  native trace utilities have been modified for ACL2.  For SBCL and CCL
  (OpenMCL), which use the built-in Lisp mechanism for returning multiple
  values in ACL2 (~pl[mv]), the use of ~c[:multiplicity] with ~c[:native]
  remains unnecessary and will be ignored.  In support of this change, the
  modification of native Allegro CL tracing for ACL2 was fixed to handle
  ~c[:exit] forms correctly that involve ~ilc[mv].

  ~st[NEW FEATURES]

  The command ~c[:]~ilc[redef!] is just like ~c[:]~ilc[redef], but prints a
  warning rather than doing a query.  The old version of ~c[:redef!] was for
  system hackers and has been renamed to ~c[:]~ilc[redef+].

  Introduced a new utility for evaluating a function call using the so-called
  executable counterpart ~-[] that is, executing the call in the logic rather
  than in raw Lisp.  ~l[ec-call].  Thanks to Sol Swords for requesting this
  utility and participating in its high-level design.

  ~l[print-gv] for a new utility that assists with debugging guard
  violations. Thanks to Jared Davis for requesting more tool assistance for
  debugging guard violations.

  Improved the guard violation error message to show the positions of the
  formals, following to a suggestion of Peter Dillinger.

  Added new ~ilc[guard-debug] capability to assist in debugging failed attempts
  at ~il[guard] verification.  ~l[guard-debug].  Thanks to Jared Davis for
  requesting a tool to assist in such debugging and to him, Robert Krug, and
  Sandip Ray for useful discussions.

  New utilities provide the formula to be proved by ~ilc[verify-guards].
  ~l[verify-guards-formula] and ~pl[guard-obligation], Thanks to Mark Reitblatt
  for making a request leading to these utilities.  These utilities can be
  applied to a term, not just an event name; thanks to Peter Dillinger for
  correspondence that led to this extension.

  A new utility causes runes to be printed as lists in proof output from
  simplification, as is done already in proof summaries.
  ~l[set-raw-proof-format].  Thanks to Jared Davis for requesting this
  utility.

  An experimental capability allows for parallel evaluation.  ~l[parallelism].
  Thanks to David Rager for providing an initial implementation of this
  capability.

  Defined ~ilc[xor] in analogy to ~ilc[iff].  Thanks to Bob Boyer, Warren Hunt,
  and Sol Swords for providing this definition.

  Improved distributed file ~c[doc/write-acl2-html.lisp] so that it can now be
  used to build HTML documentation files for ~il[documentation] strings in user
  ~il[books].  See the comment in the definition of macro
  ~c[acl2::write-html-file] at the end of that file.  Thanks to Dave Greve and
  John Powell for requesting this improvement.

  It is now possible to specify ~c[:]~ilc[hints] for non-recursive function
  definitions (which can be useful when definitions are automatically
  generated).  ~l[set-bogus-defun-hints-ok].  Thanks to Sol Swords for
  requesting such a capability.

  Keyword argument ~c[:dir] is now supported for ~ilc[rebuild] just as it has
  been for ~ilc[ld].

  We relaxed the criteria for functional substitutions, so that a function
  symbol can be bound to a macro symbol that corresponds to a function symbol
  in the sense of ~ilc[macro-aliases-table].  So for example, a functional
  substitution can now contain the doublet ~c[(f +)], where previously it
  would have been required instead to contain ~c[(f binary-+)].

  We now allow arbitrary packages in raw mode (~pl[set-raw-mode]) ~-[] thanks
  to Jared Davis for requesting this enhancement ~-[] and more than that, we
  allow arbitrary Common Lisp in raw mode.  Note however that for arbitrary
  packages, you need to be in raw mode when the input is read, not just when
  the input form is evaluated.

  Two new keywords are supported by the ~ilc[with-output] macro.  A
  ~c[:gag-mode] keyword argument suppresses some prover output as is done by
  ~ilc[set-gag-mode].  Thanks to Jared Davis for asking for a convenient way to
  set ~il[gag-mode] inside a book, in particular perhaps for a single theorem;
  this keyword provides that capability.  A ~c[:stack] keyword allows
  sub-~il[events] of ~ilc[progn] or ~ilc[encapsulate] to ``pop'' the effect of
  a superior ~ilc[with-output] call.  Thanks to Peter Dillinger for requesting
  such a feature.  ~l[with-output].

  The command ~ilc[good-bye] and its aliases ~ilc[exit] and ~ilc[quit] now all
  take an optional status argument, which provides the Unix exit status for the
  underlying process.  Thanks to Florian Haftmann for sending a query to the
  ACL2 email list that led to this enhancement.

  Keyword commands now work for macros whose argument lists have lambda list
  keywords.  For a macro with a ~c[lambda] list keyword in its argument list,
  the corresponding keyword command reads only the minimum number of required
  arguments.  ~l[keyword-commands].

  It is now legal to ~ilc[declare] variables ~c[ignorable] in ~ilc[let*] forms,
  as in ~c[(let* ((x (+ a b)) ...) (declare (ignorable x)) ...)].  Thanks to
  Jared Davis for requesting this enhancement.

  Added a warning when more than one hint is supplied explicitly for the same
  goal.  It continues to be the case that only the first hint applicable to a
  given goal will be applied, as specified in the user-supplied list of
  ~c[:hints] followed by the ~ilc[default-hints-table].  Thanks to Mark
  Reitblatt for sending a question that led both to adding this clarification
  to the ~il[documentation] and to adding this warning.

  You may now use ~ilc[set-non-linear], ~ilc[set-let*-abstraction],
  ~c[set-tainted-ok], and ~ilc[set-ld-skip-proofs] in place of their versions
  ending in ``~c[p]''.  Thanks to Jared Davis for suggesting consideration of
  such a change.  All ``~c[set-]'' utilites now have a version without the
  final ``~c[p]'' (and most do not have a version with the final ``~c[p]'').

  Added a \"Loop-Stopper\" warning when a ~c[:]~ilc[rewrite] rule is specified
  with a ~c[:]~ilc[loop-stopper] field that contains illegal entries that
  will be ignored.  Thanks to Jared Davis for recommending such a warning.

  Added a substantial documentation topic that provides a beginner's guide to
  the use of quantification with ~ilc[defun-sk] in ACL2.  Thanks to Sandip Ray
  for contributing this guide, to which we have made only very small
  modifications.  ~l[quantifier-tutorial].

  ~ilc[Defun-sk] now allows the keyword option ~c[:strengthen t], which
  will generate the extra constraint that that is generated for the
  corresponding ~c[defchoose] event; ~pl[defchoose].  Thanks to Dave Greve for
  suggesting this feature.

  ~st[BUG FIXES]

  Fixed a soundness bug related to the use of ~ilc[mbe] inside
  ~ilc[encapsulate] events.  An example proof of ~c[nil] (before the fix) is in
  a comment in ~c[(deflabel note-3-4 ...)] in the ACL2 source code.  We
  therefore no longer allow calls of ~ilc[mbe] inside ~ilc[encapsulate] events
  that have non-empty ~il[signature]s.

  Fixed a bug related to the definition of a function supporting the
  macro ~ilc[value-triple].  Although this bug was very unlikely to affect any
  user, it could be carefully exploited to make ACL2 unsound:
  ~bv[]
  (defthm fact
    (equal (caadr (caddr (value-triple-fn '(foo 3) nil nil)))
           'value) ; but it's state-global-let* in the logic
    :rule-classes nil)
  (defthm contradiction
    nil
    :hints ((\"Goal\" :use fact :in-theory (disable (value-triple-fn))))
    :rule-classes nil)
  ~ev[]

  Non-~ilc[LOCAL] definitions of functions or macros are no longer considered
  redundant with built-ins when the built-ins have special raw Lisp code,
  because ACL2 was unsound without this restriction!  A comment about redundant
  definitions in source function ~c[chk-acceptable-defuns] shows how one could
  prove ~c[nil] without this new restriction.  Note that system utility
  ~c[:]~ilc[redef+] removes this restriction.

  Although ACL2 already prohibited the use of certain built-in
  ~c[:]~ilc[program] mode functions for ~ilc[verify-termination] and during
  macroexpansion, we have computed a much more complete list of functions that
  need such restrictions, the value of constant
  ~c[*primitive-program-fns-with-raw-code*].

  Modified what is printed when a proof fails, to indicate more clearly which
  event failed.

  Fixed a problem with ~ilc[dmr] in CCL (OpenMCL) that was causing a raw Lisp
  break after an interrupt in some cases.  Thanks to Gary Byers for a
  suggestion leading to this fix.

  Fixed bugs in ~il[proof-checker] code for dealing with free variables in
  hypotheses.

  Upon an abort, the printing of ~ilc[pstack] and ~il[gag-mode] summary
  information for other than GCL was avoided when inside a call of ~ilc[ld].
  This has been fixed.

  (Windows only) Fixed bugs for ACL2 built on SBCL on Windows, including one
  that prevented ~ilc[include-book] parameters ~c[:dir :system] from working,
  and one that prevented certain compilation.  Thanks to Peter Dillinger for
  bringing these to our attention and supplying a fix for the second.  Thanks
  also to Andrew Gacek for bringing ~ilc[include-book] issues to our attention.
  Also, fixed writing of file ~c[saved_acl2] at build time so that for Windows,
  Unix-style pathnames are used.

  Fixed a hard Lisp error that could occur with keywords as ~ilc[table] names,
  e.g., ~c[(table :a :a nil :put)].  Thanks to Dave Greve for bringing this
  problem to our attention and providing this example.

  Fixed handling of ~c[:OR] ~il[hints] so that proof attempts under an ~c[:OR]
  hint do not abort (reverting to induction on the original input conjecture)
  prematurely.  Thanks to Robert Krug for pointing out this problem and
  pointing to a possible initial fix.

  (SBCL and CLISP only) It is now possible to read symbols in the
  ~c[\"COMMON-LISP\"] package inside the ACL2 command loop (~pl[lp]).  This
  could cause a raw Lisp error in previous versions of ACL2 whose host Common
  Lisp was SBCL or CLISP.  Thanks to Peter Dillinger for bringing this issue to
  our attention.

  Fixed a bug that was preventing certain ~il[hints], such as ~c[:do-not]
  hints, from being used after the application of an ~c[:or] hint.  Thanks to
  Robert Krug for bringing this bug to our attention.

  (Hons version only) Fixed a bug in the interaction of ~ilc[memoize]
  (~ilc[hons] version only) with event processing, specifically in interaction
  with failures inside a call of ~ilc[progn] or ~ilc[encapsulate].  Thanks to
  Jared Davis for bringing this bug to our attention and sending an example.  A
  simplified example may be found in a comment in source function
  ~c[table-cltl-cmd], source file ~c[history-management.lisp]; search for
  ``Version_3.3'' there.

  Fixed ~ilc[cw-gstack] so that its ~c[:evisc-tuple] is applied to the top
  clause, instead of using ~c[(4 5 nil nil)] in all cases.  If no
  ~c[:evisc-tuple] is supplied then ~c[(term-evisc-tuple t state)] is used for
  the top clause, as it is already used for the rest of the stack.

  Fixed a bug in the interaction of ~il[proof-tree]s with ~c[:induct] hint
  value ~c[:otf-flg-override].  Thanks to Peter Dillinger for reporting this
  bug and sending an example that evokes it.

  Fixed bugs in ~c[:]~ilc[pr] and ~ilc[find-rules-of-rune] for the case of rule
  class ~c[:]~ilc[elim].  Thanks to Robert Krug and Jared Davis for bringing
  these related bugs to our attention.

  Improved failure messages so that the key checkpoints are printed only once
  when there is a proof failure.  Formerly, a proof failure would cause the key
  checkpoints to be printed for every ~ilc[encapsulate] or ~ilc[certify-book]
  superior to the proof attempt.

  Fixed a bug in generation of ~il[guard]s for calls of ~ilc[pkg-witness].
  Thanks to Mark Reitblatt for sending an example showing this bug.
  The bug can be in play when you see the message:  ``HARD ACL2 ERROR in
  MAKE-LAMBDA-APPLICATION: Unexpected unbound vars (\"\")''.  A
  distillation of Mark's example that causes this hard error is as follows.
  ~bv[]
  (defun foo (x)
    (declare (xargs :guard t))
    (let ((y x)) (pkg-witness y)))
  ~ev[]

  The ~ilc[cond] macro now accepts test/value pairs of the form ~c[(T val)] in
  other than the last position, such as the first such pair in
  ~c[(cond (t 1) (nil 2) (t 3))].  Thanks to Jared Davis for sending this
  example and pointing out that ACL2 was sometimes printing goals that have
  such a form, and hence cannot be submitted back to ACL2.  A few macros
  corresponding to ~ilc[cond] in some books under ~c[books/rtl] and
  ~c[books/bdd] were similarly modified.  (A second change will probably not be
  noticeable, because it doesn't affect the final result: singleton ~ilc[cond]
  clauses now generate a call of ~ilc[or] in a single step of macroexpansion,
  not of ~ilc[if].  For example, ~c[(cond (a) (b x) (t y))] now expands to
  ~c[(OR A (IF B X Y))] instead of ~c[(IF A A (IF B X Y))].  See the source
  code for ~c[cond-macro] for a comment about this change.)

  Fixed a bug in the interaction of ~il[proof-checker] command ~c[DV],
  including numeric (``diving'') commands, with the ~ilc[add-binop] event.
  Specifically, if you executed ~c[(add-binop mac fn)] with ~c[fn] having arity
  other than 2, a ~il[proof-checker] command such as 3 or ~c[(dv 3)] at a call
  of ~c[mac] could have the wrong effect.  We also fixed a bug in diving with
  ~c[DV] into certain ~c[AND] and ~c[OR] calls.  Thanks for Mark Reitblatt for
  bringing these problems to our attention with helpful examples.

  Fixed a couple of bugs that were causing an error, ``HARD ACL2 ERROR in
  RENEW-NAME/OVERWRITE''.  Thanks to Sol Swords for bringing the first of these
  bugs to our attention.

  Fixed a bug that could cause ~ilc[certify-book] to fail in certain cases
  where there are ~ilc[local] ~ilc[make-event] forms.

  Fixed a bug in ~ilc[start-proof-tree] that could cause Lisp to hang or
  produce an error.  Thanks to Carl Eastlund for sending an example to bring
  this bug to our attention.

  Fixed a bug in the proof output, which was failing to report cases where the
  current goal simplifies to itself or to a set including itself
  (~pl[specious-simplification]).

  Fixed a bug in ~ilc[with-prover-time-limit] that was causing a raw Lisp error
  for a bad first argument.  Thanks to Peter Dillinger for pointing out this
  bug.

  The following was claimed in ~c[:doc] ~ilc[note-3-3], but was not fixed until
  the present release:~nl[]
  Distributed directory ~c[doc/HTML/] now again includes installation
  instructions, in ~c[doc/HTML/installation/installation.html].

  In certain Common Lisp implementations ~-[] CCL (OpenMCL) and LispWorks, at
  least ~-[] an interrupt could leave you in a break such that quitting the
  break would not show the usual summary of key checkpoints.  This has been
  fixed.

  ~st[NEW AND UPDATED BOOKS]

  Updated ~c[books/clause-processors/SULFA/] with a new version from Erik
  Reeber; thanks, Erik.

  Added new books directory ~c[tools/] from Sol Swords.  See
  ~c[books/tools/Readme.lsp] for a summary of what these books provide.

  The distributed book ~c[books/misc/file-io.lisp] includes a new utility,
  ~c[write-list!], which is like ~c[write-list] except that it calls
  ~ilc[open-output-channel!] instead of ~ilc[open-output-channel].  Thanks to
  Sandip Ray for requesting this utility and assisting with its
  implementation.

  Added ~c[record-update] macro supplied by Sandip Ray to distributed book
  ~c[books/misc/records.lisp].

  Sandip Ray has contributed books that prove soundness and completeness of
  different proof strategies used in sequential program verification.
  Distributed directory ~c[books/proofstyles/] has three new directories
  comprising that contribution: ~c[soundness/], ~c[completeness/], and
  ~c[counterexamples/].  The existing ~c[books/proofstyles/] directory has been
  moved to its subdirectory ~c[invclock/].

  Jared Davis has contributed a profiling utility for ACL2 built on CCL
  (OpenMCL).  See ~c[books/misc/oprof.lisp].  Thanks, Jared.

  ACL2 utilities ~ilc[getprop] and ~ilc[putprop] take advantage of
  under-the-hood Lisp (hashed) property lists.  The new book
  ~c[books/misc/getprop.lisp] contains an example showing how this works.

  Added the following new book directories: ~c[books/paco/], which includes a
  small ACL2-like prover; and ~c[books/models/jvm/m5], which contains the
  definition of one of the more elaborate JVM models, M5, along with other
  files including JVM program correctness proofs.  See files ~c[Readme.lsp] in
  these directories, and file ~c[README] in the latter.

  Added ~i[books] about sorting in ~c[books/sorting].  See ~c[Readme.lsp] in
  that directory for documentation.

  Added book ~c[books/misc/computed-hint-rewrite.lisp] to provide an interface
  to the rewriter for use in computed hints.  Thanks to Jared Davis for
  requesting this feature.

  Jared Davis has provided a pseudorandom number generator, in
  ~c[books/misc/random.lisp].

  Robert Krug has contributed a new library, ~c[books/arithmetic-4/], for
  reasoning about arithmetic.  He characterizes it as being more powerful than
  its predecessor, ~c[books/arithmetic-3/], and without its predecessor's
  rewriting loops, but significantly slower than its predecessor on some
  theorems.

  Incorporated changes from Peter Dillinger to verify guards for functions in
  ~c[books/ordinals/lexicographic-ordering.lisp] (and one in
  ~c[ordinal-definitions.lisp] in that directory).

  A new directory, ~c[books/hacking/], contains a library for those who wish to
  use trust tags to modify or extend core ACL2 behavior.  Thanks to Peter
  Dillinger for contributing this library.  Obsolete version
  ~c[books/misc/hacker.lisp] has been deleted.  Workshop contribution
  ~c[books/workshops/2007/dillinger-et-al/code/] is still included with the
  workshops/ tar file, but should be considered deprecated.

  In ~c[books/make-event/assert.lisp], changed ~c[assert!] and
  ~c[assert!-stobj] to return ~c[(value-triple :success)] upon success instead
  of ~c[(value-triple nil)], following a suggestion from Jared Davis.

  ~st[EMACS SUPPORT]

  Changed ~c[emacs/emacs-acl2.el] so that the fill column default (for the
  right margin) is only set (still to 79) in lisp-mode.

  Modified Emacs support in file ~c[emacs/emacs-acl2.el] so that names of
  events are highlighted just as ~ilc[defun] has been highlighted when it is
  called.  Search in the above file for ~c[font-lock-add-keywords] for
  instructions on how to eliminate this change.

  The name of the temporary file used by some Emacs utilities defined in file
  ~c[emacs/emacs-acl2.el] has been changed to have extension ~c[.lsp] instead
  of ~c[.lisp]; thus it is now ~c[temp-emacs-file.lsp].  Also, `make'
  commands to `clean' books will delete such files (specifically,
  ~c[books/Makefile-generic] has been changed to delete
  ~c[temp-emacs-file.lsp]).

  ~/~/")

(deflabel |NOTE-3-4(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.4(r) (August, 2008) Notes~/

  ~/

  Please also ~pl[note-3-4] for changes to Version  3.4 of ACL2.

  Fixed makefiles, ~c[books/nonstd/Makefile] and ~c[GNUmakefile].  The old
  set-up seemed to work fine as long as all books certified, but it was really
  broken, for example only certifying some of the books in
  ~c[books/nonstd/nsa/], and then only when required by books in other
  directories.  Also fixed the ``~c[clean]'' target to clean links rather than
  to make links.

  ~/
  ")

(deflabel note-3-5

; See the long comment just after this (deflabel note-3-5 ...) for a discussion
; of changes made March 9-16 for more efficient handling of certificates, etc.

; Added ignorable declaration to (expansion of) er-progn, anticipating
; optimizations in GCL 2.7.0 when it is ready.  (Camm Maguire reported warnings
; that er-progn-not-to-be-used-elsewhere-val is unused.)

; Replaced sloop::sloop by loop.

; :Doc at the terminal had sometimes suggested using :more when there's nothing
; more to print.  This has been fixed.  Thanks to Bob Boyer for bringing this
; issue to our attention.

; Regarding "~il[Gag-mode] now suppresses some messages" below: Search for the
; following comment to find code to modify if we want to undo this change:
;
; Suppress printing for :OR splits; see also other comments with this header.

; Regarding "sped up ~il[guard] generation for some functions", the change was
; in built-in-clausep to delay the call to trivial-clause-p.

; Note that we changed set-debugger-enable-fn so that it no longer sets
; *debugger-hook*.  Instead, our-abort is always called (in other than non-ansi
; GCL) but if the debugger is enabled, then only print-proof-tree-finish (along
; perhaps with print-call-history) is invoked.

; For hons version: Fixed typos 'current-acl2-wrld in function
; memoize-table-chk (should have been 'current-acl2-world), which has no
; functional impact but might improve performance of memoize commands.

; The item "Added a restriction in the linear arithmetic procedure for
; deleting" relates to a change to poly-weakerp, which necessitated (for
; non-linear arithmetic) a corresponding change to bounds-poly-with-var
; (formerly called bounds-polys-with-var1).

; The item "Fixed a bug that could cause simplifications to fail" pertains to a
; change to process-equational-polys.

; Fixed an apparent bug in translate-expand-term1, where after translating a
; form we tested if the UNtranslated form was a variablep or fquotep.

; Regarding "Fixed a bug in processing of ~c[:]~ilc[type-set-inverter] rules",
; below:  Changed call of if-tautologyp to tautologyp in
; chk-acceptable-type-set-inverter-rule.  Examples that now work and didn't
; before:
; (defun my-consp (x) (consp x))
; (defthm foo (equal (my-consp x) (consp x))
;   :rule-classes :type-set-inverter)
; Alternatively (but disable foo before trying this):
; (defthm bar (equal (my-consp x) (consp x))
;   :rule-classes ((:type-set-inverter
;                   :corollary (equal (my-consp x) (consp x))
;                   :type-set 1536)))

; Modified save-acl2-in-sbcl-aux and save-acl2-in-cmulisp-aux to store an
; absolute pathname for the underlying Lisp, after John Erickson reported
; problems with making an sbcl-based ACL2 image using a relative pathname for
; LISP= in the ACL2 make.

; Followed Jared Davis's suggestion of fixing message "To enable breaks into
; the debugger..." to print the suggested (SET-DEBUGGER-ENABLE T) correctly for
; the current package.

; Dropped the deceased-packages (second) argument of bad-lisp-objectp and
; chk-bad-lisp-object.

; Eliminated *read-object-eof* in favor of a dynamic-extent cons (in
; read-object), thanks to Bob Boyer.

; Eliminated constant *acl2-files* from the logic; it is now a constant defined
; in acl2.lisp.

; Regarding "Fixed a soundness bug in the handling of inequalities by the
; ~il[type-set] mechanism": The fix is in the handling of lambda applications
; inside type-set (see the comment "The pot-lst is not valid....").
; Here is how to see that bug in action.  The following theorem proves "using
; trivial observations", but the instance of it below evaluates to nil.

;  (thm
;   (IMPLIES (AND (< (CAR Y) (CADR X))
;                 (TRUE-LISTP X)
;                 (TRUE-LISTP Y)
;                 (< (CAR X) (CAR Y)))
;            (LET ((X Y) (Y X))
;                 (AND (TRUE-LISTP X)
;                      (TRUE-LISTP Y)
;                      (< (CAR X) (CAR Y))))))
;
;  (let ((x '(1 7 3)) (y '(4 5 6)))
;    (IMPLIES (AND (< (CAR Y) (CADR X))
;                  (TRUE-LISTP X)
;                  (TRUE-LISTP Y)
;                  (< (CAR X) (CAR Y)))
;             (LET ((X Y) (Y X))
;                  (AND (TRUE-LISTP X)
;                       (TRUE-LISTP Y)
;                       (< (CAR X) (CAR Y))))))

; A trace of type-set before the fix shows some odd behavior, a simpler version
; of which is below.  The pot-lst represents (< (car x) (car y)), which should
; be irrelevant inside the body of the lambda, but (erroneously) is used to
; return a type-set of *ts-t*.

;  (TYPE-SET '((LAMBDA (X Y)
;                      (< (CAR X) (CAR Y)))
;              Y X)
;            NIL NIL
;            '((Y 576) (X 512))
;            NIL (ENS STATE) (W STATE) NIL
;            '(((0)
;               (CAR Y)
;               ((((((CAR Y) . 1) ((CAR X) . -1))
;                  (3)
;                  (PT . 3))
;                 0 < NIL))))
;            NIL)
; ; From (trace$ type-set), notice the use of the pot-lst inside the body of the
; ; lambda:
;   2> (TYPE-SET ((LAMBDA (X Y) (< (CAR X) (CAR Y))) Y X)
;                NIL NIL ((Y 576) (X 512))
;                NIL |some-enabled-structure|
;                |current-acl2-world| NIL
;                (((0)
;                  (CAR Y)
;                  ((((((CAR Y) . 1) ((CAR X) . -1))
;                     (3)
;                     (PT . 3))
;                    0 < NIL))))
;                NIL)
;  ...
;     3> (TYPE-SET (< (CAR X) (CAR Y))
;                  NIL NIL ((X 576) (Y 512))
;                  T |some-enabled-structure|
;                  |current-acl2-world| NIL
;                  (((0)
;                    (CAR Y)
;                    ((((((CAR Y) . 1) ((CAR X) . -1))
;                       (3)
;                       (PT . 3))
;                      0 < NIL))))
;                  NIL)

; Changed fixnum declarations in some array functions to (signed-byte 32) and
; (unsigned-byte 31) declarations.

; The :multiplicity trace$ option was added in order to support break-on-error,
; so that raw-ev-fncall could be traced without using :native, so that the
; :cond option could be used.

; The fourth argument of observation1 is now a Boolean rather than an
; evisc-tuple.  We don't advertise this mod since observation1 is not
; advertised.

; Added suitable environment variable settings to ACL2 scripts saved for CCL
; (OpenMCL) and SBCL, thanks (respectively) to assistance from Gary Byers and
; Nikodemus Siivola.  In particular, profiling is now available for ACL2 in
; CCL using CHUD (see
; http://ccl.clozure.com/ccl-documentation.html#Profiling-Using-Apples-CHUD-metering-tools)
; and in SBCL, at least for SBCL 1.0.23 and later, e.g.:
;
; :q
; (require :sb-sprof)
; (sb-sprof:with-profiling (:max-samples 100000
;                           :alloc-interval 1
;                           :mode :alloc
;                           :sample-interval .00001
;                           :report :graph
;                           :loop nil
;                           :threads :all
;                           :show-progress nil)
;            (ld '((mini-proveall))))

; The modification pertaining to the mention of term-order below was carried
; out by eliminating code for variants of var-fn-count/var-fn-count-lst and
; arith-term-order, by generalizing the term-order algorithm so that
; arith-term-order is a special case.  See term-order1.  The main change to
; arith-term-order (which is used in linear arithmetic) is that it is now
; sensitive to the "pseudo function count", which (roughly speaking) is the
; number of implicit constructor function calls in constants.  These counts are
; compared for two terms after their variable counts and function call counts,
; but before the terms are compared lexicographically.

; Made tiny change to get-os in case :unix and :apple features (for example)
; are both present.

; Moved state-global-let* bindings of guard-checking-on and in-prove-flg down
; from prove to prove-loop0.  Note that proof-checker calls prove-loop; a
; comment in pc-prove in fact clarified that guard-checking-on is already bound
; to nil by pc-single-step-primitive, but this new way seems a safer way to go
; (maybe others will call prove-loop), and besides, now in-prove-flg is set,
; which is better for break-on-error.

; Added to GNUmakefile ACL2_SIZE for ACL2 package size, implementing an idea
; from Jared Davis.

; Eliminated a skip-proofs warning caused by chk-embedded-event-form when in a
; make-event but not an encapsulate, include-book, or certify-book.

; Note that the new ruler-extenders feature handles prog2$ slightly differently
; than before, since merging of base cases can now take place under a prog2$
; where before it did not.  But don't really say that in the :doc string below,
; because it seems too subtle to be worth the distraction.  Also notice that
; although ec-call is now a default ruler-extender where before it was not
; (unlike prog2$), nevertheless, since ec-call was likely not applied to calls
; of if, lambdas, or prog2$, it seems unlikely that the inclusion of ec-call as
; a default ruler-extender should have any effect.

; Regarding the "subtle soundness bug in the forming of constraints from
; deduced type prescriptions": This simplified iteratively-grow-constraint1,
; but the following is even more significant.  Now, the constraints are based
; entirely on the lexical content of the events, not on which runes are enabled
; in the current world -- see constraints-introduced and
; definitional-constraints.  This change may address other lurking soundness
; bugs, evoked using functional instantiation on the same encapsulated
; functions but for different worlds, of which we haven't even been aware.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  3.5 (May, 2009) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  3.4 into the following
  categories: changes to existing features, new features, heuristic
  improvements, bug fixes, new and updated books, Emacs support, and
  experimental ~ilc[hons] version.  Each change is described in just one
  category, though of course many changes could be placed in more than one
  category.

  ~st[CHANGES TO EXISTING FEATURES]

  Many improvements have been made to ACL2's ``evisceration'' mechanism for
  hiding substructures of objects before they are printed, and to related
  documentation:
  ~bq[]

  o A new documentation topic explains evisc-tuples.  ~l[evisc-tuple].

  o A new interface, ~ilc[set-evisc-tuple], has been provided for setting the
  four global evisc-tuples.  ~l[set-evisc-tuple].

  o A new mode, ``iprinting'', allows eviscerated output to be read back in.
  ~l[set-iprint].

  o Function ~c[default-evisc-tuple] has been deprecated and will probably be
  eliminated in future releases; use ~c[abbrev-evisc-tuple] instead.  Also
  eliminated is the brr-term-evisc-tuple (also the user-brr-term-evisc-tuple).
  The term-evisc-tuple controls printing formerly controlled by the
  brr-term-evisc-tuple or user-brr-term-evisc-tuple.

  o ACL2 output is done in a more consistent manner, respecting the intention
  of those four global evisc-tuples.  In particular, more proof output is
  sensitive to the term-evisc-tuple.  Again, ~pl[set-evisc-tuple].

  o A special value, ~c[:DEFAULT], may be provided to ~ilc[set-evisc-tuple] in
  order to restore these ~il[evisc-tuple]s to their original settings.

  o (Details for heavy users of the evisc-tuple mechanism) (1) There are no
  longer ~il[state] globals named ~c[user-term-evisc-tuple] or
  ~c[user-default-evisc-tuple].  (2) Because of the above-mentioned
  ~c[:DEFAULT], if you have referenced state globals directly, you should use
  accessors instead, for example ~c[(abbrev-evisc-tuple state)] instead of
  ~c[(@ abbrev-evisc-tuple)].  (3) For uniformity, ~ilc[set-trace-evisc-tuple]
  now takes a second argument, ~c[state].~eq[]

  Improved ~ilc[break-on-error] in several ways.  First, it breaks earlier in a
  more appropriate place.  Thanks to Dave Greve for highlighting this problem
  with the existing implementation.  Also, ~ilc[break-on-error] now breaks on
  hard errors, not only soft errors (~pl[er], options ~c[hard] and ~c[hard?]).
  Thanks to Warren Hunt and Anna Slobodova for sending an example that showed a
  flaw in an initial improvement.  Finally, new options cause printing of the
  call stack for some host Common Lisps.  ~l[break-on-error].  Thanks to Bob
  Boyer for requesting this feature.

  ~ilc[Trace!] may now be used in raw Lisp (though please note that all
  soundness claims are off any time you evaluate forms in raw Lisp!).  Thanks
  to Bob Boyer for feedback that led to this enhancement.

  ACL2 now searches for file ~c[acl2-customization.lsp] in addition to (and
  just before) its existing search for ~c[acl2-customization.lisp];
  ~l[acl2-customization].  Thanks to Jared Davis for suggesting this change,
  which supports the methodology that files with a ~c[.lisp] extension are
  certifiable books (thus avoiding the need to set the ~c[BOOKS] variable in
  makefiles; ~pl[books-certification-classic]).

  Improved the error message for illegal ~ilc[declare] forms of the form
  ~c[(type (satisfies ...))].  Thanks to Dave Greve for sending an example
  highlighting the issue.

  If trace output is going to a file (because ~ilc[open-trace-file] has been
  executed), then a note will be printed to that effect at the time that
  a call of ~ilc[trace$] or ~ilc[trace!] is applied to one or more ~il[trace]
  specs.

  The notion of redundancy (~pl[redundant-events]) has been made more
  restrictive for ~ilc[mutual-recursion] events.  Now, if either the old or new
  event is a ~ilc[mutual-recursion] event, then redundancy requires that both
  are ~ilc[mutual-recursion] events that define the same set of function
  symbols.  Although we are not aware of any soundness bugs fixed by this
  modification, nevertheless we believe that it reduces the risk of soundness
  bugs in the future.

  The definition of ~c[trace*] has been moved to a book, ~c[misc/trace1.lisp].
  A new version, used in ACL2s, is in book ~c[misc/trace-star.lisp].
  ~il[Trace] utilities ~ilc[trace$] and ~ilc[trace!] are still built into
  ACL2.  [Note: File ~c[misc/trace1.lisp] was deleted after Version  4.2.]

  Certain ~il[certificate] files will now be much smaller, by printing in a way
  that takes advantage of structure sharing.  Certifying the following example
  produces a ~c[.cert] file of over 3M before this change, but less than 1K
  after the change.
  ~bv[]
  (defun nest (i)
    ;; Makes an exponentially-sized nest of conses i deep.
    (if (zp i)
        nil
      (let ((next (nest (1- i))))
        (cons next next))))

  (make-event
   `(defconst *big* ',(nest 20)))
  ~ev[]
  Thanks to Sol Swords for providing the above example and to him as well as to
  Bob Boyer, Jared Davis, and Warren Hunt for encouraging development of this
  improvement.  We have also applied this improvement to the printing of
  function definitions to files on behalf of ~ilc[certify-book] and
  ~ilc[comp].

  Names of symbols are now printed with vertical bars according to the Common
  Lisp spec.  Formerly, if the first character of a symbol name could be the
  first character of the print representation of a number, then the symbol was
  printed using vertical bars (~c[|..|]) around its name.  Now, a much more
  restrictive test for ``potential numbers'' is used, which can result in fewer
  such vertical bars.  Base 16 is now carefully considered as well;
  ~pl[set-print-base].  Thanks to Bob Boyer for requesting this improvement.
  Note that macros ~c[set-acl2-print-base] and ~c[set-acl2-print-case] have
  been replaced by functions; ~pl[set-print-base] and ~pl[set-print-case].

  The ACL2 reader now supports `~c[#.]' syntax in place of the `~c[#,] syntax
  formerly supported.  Thanks to Bob Boyer for requesting this change.
  ~l[sharp-dot-reader].  NOTE that because of this change, `~c[#.]' no longer
  causes an abort; instead please use ~c[(a!)] or optionally, if in the ACL2
  loop, ~c[:a!]; ~pl[a!].

  Some small changes have been made related to ~il[gag-mode]:
  ~bq[]

  o ~il[Gag-mode] now suppresses some messages that were being printed upon
  encountering disjunctive splits from ~c[:OR] ~il[hints].  Thanks to Sol
  Swords for suggesting this improvement.

  o ACL2 had printed ``~c[Q.E.D.]'' with all output suppressed and
  ~ilc[gag-mode] enabled.  Now, ``~c[Q.E.D.]'' will be suppressed when
  ~c[PROVE] and ~c[SUMMARY] output are suppressed, even if ~c[gag-mode] is
  enabled.

  o The use of ~ilc[set-gag-mode] had drastic effects on the inhibited output
  (~pl[set-inhibit-output-lst]), basically inhibiting nearly all output (even
  most warnings) when turning on gag-mode and enabling all output except
  ~c[proof-tree] output when turning off gag-mode.  Now, ~ilc[set-gag-mode]
  only inhibits or enables proof (~c[PROVE]) output, according to whether
  gag-mode is being turned on or off (respectively).  The related utility
  ~ilc[set-saved-output] has also been modified, basically to eliminate
  ~c[:all] as a first argument and to allow ~c[t] and ~c[:all] as second
  arguments, for inhibiting prover output or virtually all output, respectively
  (~pl[set-saved-output]).~eq[]

  A ~ilc[defstub] event ~il[signature] specifying output of the form
  ~c[(mv ...)] now introduces a ~c[:]~ilc[type-prescription] rule asserting
  that the new function returns a ~ilc[true-listp] result.  Thanks to Bob Boyer
  for sending the following example, which motivated this change.
  ~bv[]
  (defstub census (*) => (mv * *))
  (defn foo (x)
    (mv-let (a1 a2)
            (census x)
            (list a1 a2)))
  ~ev[]

  Improved the efficiency of ~ilc[string-append] so that in raw Lisp, it calls
  ~ilc[concatenate].  Thanks to Jared Davis for suggesting this change,
  including the use of ~ilc[mbe].  A minor change was made to the definition of
  ~ilc[concatenate] to support this change, and the lemma ~c[append-to-nil] was
  added (see below).

  The checksum algorithm used for ~il[certificate] files of ~il[books] has
  been changed.  Thanks to Jared Davis for contributing the new code.  This
  change will likely not be noticed unless one is using the experimental
  ~ilc[hons] version of ACL2, where it can greatly speed up book certification
  and inclusion because of function memoization (of source function
  ~c[fchecksum-obj]).

  Fewer calls are made to the checksum algorithm on behalf of
  ~ilc[certify-book] and a few other operations.  Thanks to Jared Davis for
  providing data that helped lead us to these changes.

  Formatted printing directives ~c[~~p], ~c[~~q], ~c[~~P], and ~c[~~Q] are
  deprecated, though still supported.  ~l[fmt].  Instead, please use ~c[~~x],
  ~c[~~y], ~c[~~X], and ~c[~~Y] (respectively).  As a by-product, rule names in
  proof output are no longer hyphenated.

  A new keyword, ~c[:multiplicity], is available for tracing raw Lisp functions
  using the ACL2 ~il[trace] utility.  ~l[trace$].

  Users may now control whether or not a slow array access results in a warning
  printed to the screen (which is the default, as before), and if so, whether
  or not the warning is followed by a break.  ~l[slow-array-warning].

  On linux-like systems (including Mac OS X and SunOS), ~c[:]~ilc[comp] will
  now write its temporary files into the ~c[\"/tmp\"] directory, which is the
  value of ~ilc[state] global ~c['tmp-dir].  You can change that directory with
  ~c[(assign tmp-dir \"<your_temp_directory_path>\")].

  The messages printed for uncertified books have been enhanced.  Thanks to
  Jared Davis for requesting such an improvement.

  A function definition that would be redundant if in ~c[:]~ilc[logic] mode is
  now considered redundant even if it (the new definition) is in
  ~c[:]~ilc[program] mode.  That is, if a definition is ``downgraded'' from
  ~c[:logic] to ~c[:program] mode, the latter (~c[:program] mode) definition is
  considered redundant.  Previously, such redundancy was disallowed, but we
  have relaxed that restriction because of a scenario brought to our attention
  by Jared Davis: include a book with the ~c[:logic] mode definition, and then
  include a book with the ~c[:program] mode definition followed by
  ~ilc[verify-termination].  Thanks, Jared.

  The ACL2 reader no longer accepts characters other than those recognized by
  ~ilc[standard-char-p] except for ~c[#\\Tab], ~c[#\\Page], and ~c[#\\Rubout]
  (though it still accepts strings containing such characters).  As a result,
  no ~ilc[make-event] expansion is allowed to contain any such unacceptable
  character or string.  Thanks to Sol Swords for sending an example that led us
  to make this restriction.  A simple example is the following book:
  ~bv[]
  (in-package \"ACL2\")
  (defconst *my-null* (code-char 0))
  (make-event `(defconst *new-null* ,*my-null*))
  ~ev[]
  For this book, a call of ~ilc[certify-book] formerly broke during the
  compilation phase, but if there was no compilation, then a call of
  ~ilc[include-book] broke.  Now, the error occurs upon evaluation of the
  ~ilc[make-event] form.

  ACL2 now collects up ~il[guard]s from ~ilc[declare] forms more as a user
  might expect, without introducing an unexpected ordering of conjuncts.  We
  thank Jared Davis for sending us the following illustrative example,
  explained below.
  ~bv[]
  (defun f (x n)
    (declare (xargs :guard (and (stringp x)
                                (natp n)
                                (= (length x) n)))
             (type string x)
             (ignore x n))
    t)
  ~ev[]
  Formerly, a guard was generated for this example by unioning the conjuncts
  from the ~c[:guard] onto a list containing the term ~c[(string x)] generated
  from the ~c[type] declaration, resulting in an effective guard of:
  ~bv[]
  (and (natp n)
       (= (length x) n)
       (stringp x))
  ~ev[]
  The guard of this guard failed to be verified because ~c[(stringp x))] now
  comes after the call ~c[(length x)].  With the fix, contributions to the
  guards are collected up in the order in which they appear.  So in the above
  example, the effective guard is the specified ~c[:guard]; the contribution
  ~c[(stringp x)] comes later, and is thus dropped since it is redundant.
  NOTE by the way that if ~c[:guard] and ~c[:stobjs] are specified in the same
  ~ilc[xargs] form, then for purposes of collecting up the effective guard as
  described above, ~c[:stobjs] will be treated as through it comes before the
  ~c[:guard].

  Modified ~ilc[close-output-channel] to try to do a better job flushing
  buffers.  Thanks to Bob Boyer for helpful correspondence.

  The notion of ``subversive recursion'' has been modified so that some
  functions are no longer marked as subversive.  ~l[subversive-recursions], in
  particular the discussion elaborating on the notion of ``involved in the
  termination argument'' at the end of that ~il[documentation] topic.

  Formerly, ~c[:]~ilc[type-prescription] rules for new definitions inside
  ~ilc[encapsulate] forms were sometimes added as ~il[constraint]s.  This is no
  longer the case.  See also discussion of the ``soundness bug in the forming
  of constraints'', which is related.

  ~st[NEW FEATURES]

  It is now possible to affect ACL2's termination analysis (and resulting
  induction analysis).  Thanks to Peter Dillinger for requesting this feature.
  The default behavior is essentially unchanged.  But for example, the
  following definition is accepted by ACL2 because of the use of the new
  ~c[:ruler-extenders] features; ~l[ruler-extenders].
  ~bv[]
    (defun f (x)
      (declare (xargs :ruler-extenders :all))
      (cons 3
            (if (consp x)
                (f (cdr x))
              nil)))
  ~ev[]

  The following lemma was added in support of the improvement to
  ~ilc[string-append] described above:
  ~bv[]
  (defthm append-to-nil
    (implies (true-listp x)
             (equal (append x nil)
                    x)))
  ~ev[]

  A mechanism has been provided for users to contribute documentation.
  ~l[managing-acl2-packages] for an example, which contains a link to an
  external web page on effective use of ACL2 packages, kindly provided by Jared
  Davis.  ACL2 ~il[documentation] strings may now link to external web pages
  using the new symbol, ~c[~~url]; ~pl[markup].  Of course, those links appear
  in the web version of the documentation, but you made need to take a bit of
  action in order for these to appear as links in the Emacs Info version;
  ~pl[documentation].

  Added ~ilc[intersectp] (similar to ~ilc[intersectp-eq] and
  ~ilc[intersectp-equal]).

  The user now has more control over how ACL2 prints forms; ~l[print-control].
  Thanks to Bob Boyer for useful discussions leading to this enhancement.

  Some Common Lisp implementations only allow the syntax
  ~c[pkg-name::expression] when ~c[expression] is a symbol.  The ACL2 reader
  has been modified to support a package prefix for arbitrary expressions;
  ~pl[sharp-bang-reader].  Thanks to Hanbing Liu for a query that led to this
  feature and to Pascal J. Bourguignon for suggesting an implmentation.

  Ill-formed ~il[documentation] strings need not cause an error.
  ~l[set-ignore-doc-string-error].  Thanks to Bob Boyer for requesting this
  feature.

  Type declarations are now permitted in ~c[let*] forms; ~pl[let*],
  ~pl[declare], and ~pl[type-spec].

  (For Lisp programmers) Macro ~c[with-live-state] has been provided for
  programmers who refer to free variable ~c[STATE], for example with macros
  that generate uses of ~c[STATE], and want to avoid compiler warnings when
  evaluating in raw Lisp.  For example, the following form can be submitted
  either inside or outside the ACL2 loop to get the desired effect
  (~pl[doc-string]):
  ~c[(with-live-state (f-put-global 'doc-prefix \" \" state))].
  For another example use of this macro, see the definition
  of ~c[trace$] (ACL2 source file ~c[other-events.lisp]).

  (System hackers only) Added ~c[:]~ilc[redef-] to undo the effect of
  ~c[:]~ilc[redef+].  ~l[redef-].

  Function ~ilc[random$] is a built-in random number generator.  ~l[random$].
  Thanks to Sol Swords for requesting this feature and providing an initial
  implementation.

  ~st[HEURISTIC IMPROVEMENTS]

  Sped up ~il[guard] generation for some functions with large if-then-else
  structures in their bodies.  Thanks to Sol Swords for sending an illustrative
  example.

  Sped up ~il[guard] generation in some cases by evaluating ground
  (variable-free) subexpressions.  Thanks to Bob Boyer for sending a motivating
  example:
  ~c[(defn foo (x) (case x ((1 2) 1) ((3 4) 3) ... ((999 1000) 999)))].

  Modified slightly a heuristic association of ``size'' with constants, which
  can result in significant speed-ups in proofs involving constants that are
  very large ~c[cons] trees.

  Added a restriction in the linear arithmetic procedure for deleting
  polynomial inequalities from the linear database.  Formerly, an inequality
  could be deleted if it was implied by another inequality.  Now, such deletion
  requires that certain heuristic ``parent tree'' information is at least as
  restrictive for the weaker inequality as for the stronger.  Thanks to Dave
  Greve for bringing a relevant example to our attention and working with us to
  figure out some surprising behavior, and to Robert Krug for making a key
  observation leading to the fix.

  (GCL especially) Improved compiled code slightly by communicating to raw Lisp
  the output type when a function body is of the form ~c[(the character ...)].
  This tiny improvement will probably only be observed in GCL, if at all.

  Applied a correction suggested by Robert Krug to the variant of
  ~ilc[term-order] used in parts of ACL2's arithmetic reasoning.

  ~st[BUG FIXES]

  Fixed bugs in the handling of ~ilc[flet] expressions, one of which had the
  capability of rendering ACL2 unsound.  Thanks to Sol Swords for pointing out
  two issues and sending examples.  One example illustrated how ACL2 was in
  essence throwing away outer ~ilc[flet] bindings when processing an inner
  ~c[flet].  We have exploited that example to prove a contradiction, as
  follows: this book was certifiable before this fix.
  ~bv[]
  (in-package \"ACL2\")

  (defun a (x)
    (list 'c x))

  ; Example from Sol Swords, which failed to be admitted (claiming that
  ; function A is undefined) without the above definition of A.
  (defun foo1 (x y)
     (flet ((a (x) (list 'a x)))
       (flet ((b (y) (list 'b y)))
         (b (a (list x y))))))

  (defthm not-true
    (equal (foo1 3 4)
           '(b (c (3 4))))
    :hints ((\"Goal\"
             :in-theory
             (disable (:executable-counterpart foo1))))
    :rule-classes nil)

  (defthm contradiction
    nil
    :hints ((\"Goal\" :use not-true))
    :rule-classes nil)
  ~ev[]
  Sol's second example, below, pointed to a second bug related to computing
  output signatures in the presence of nested ~c[flet] expressions, which we
  have also fixed: this form failed before the fix.
  ~bv[]
  :trans (flet ((foo (a) (list (flet ((bar (b) b)) a)))) x)
  ~ev[]

  Fixed a subtle soundness bug in the forming of constraints from deduced type
  prescriptions.  As a result, when ACL2 prints a warning message labeling
  encapsulated functions as ``subversive'', ACL2 will no longer deduce
  ~c[:]~ilc[type-prescription] rules for those functions.  Examples that
  exploit the bug in ACL2 Version_3.4 may be found in comments in ACL2 source
  function ~c[convert-type-set-to-term] (file ~c[other-processes.lisp]) and
  especially in function ~c[putprop-type-prescription-lst] (file
  ~c[defuns.lisp]).  For more on the general issue of ``subversive
  recursions,'' ~pl[subversive-recursions].)

  Fixed a soundness bug in the handling of inequalities by the ~il[type-set]
  mechanism, which was using the inequality database inside the body of a
  ~ilc[lambda].

  Fixed a long-standing soundness bug in ~ilc[compress1] and ~ilc[compress2],
  whose raw Lisp code gave the logically incorrect result in the case of a
  single entry other than the ~ilc[header], where that entry mapped an index to
  the ~ilc[default] value.  Also fixed soundness bugs in ~ilc[compress1], in
  the case of ~c[:order >], where the raw Lisp code could drop the ~ilc[header]
  from the result or, when the input alist had entries in ascending order,
  fail to return an alist in descending order.  For example, the following book
  certified successfully.
  ~bv[]
  (in-package \"ACL2\")
  (defthm true-formula-1
    (equal (compress1 'a '((:HEADER :DIMENSIONS (4) :MAXIMUM-LENGTH 5
                                    :DEFAULT 1 :NAME A :ORDER <)
                           (1 . 1)))
           '((:HEADER :DIMENSIONS (4) :MAXIMUM-LENGTH 5
                      :DEFAULT 1 :NAME A :ORDER <)))
    :hints ((\"Goal\" :in-theory (disable (compress1))))
    :rule-classes nil)
  (defthm ouch-1
    nil
    :hints ((\"Goal\" :use true-formula-1))
    :rule-classes nil)
  (defthm true-formula-2
    (equal (compress1 'a '((:HEADER :DIMENSIONS (4) :MAXIMUM-LENGTH 5
                                    :DEFAULT NIL :NAME A :ORDER >)
                           (1 . 1)
                           (2 . 2)))
           '((:HEADER :DIMENSIONS (4) :MAXIMUM-LENGTH 5
                      :DEFAULT NIL :NAME A :ORDER >)
             (2 . 2)
             (1 . 1)))
    :hints ((\"Goal\" :in-theory (disable (compress1))))
    :rule-classes nil)
  (defthm ouch-2
    nil
    :hints ((\"Goal\" :use true-formula-2))
    :rule-classes nil)
  (defthm true-formula-3
    (equal (compress1 'a '((:HEADER :DIMENSIONS (3) :MAXIMUM-LENGTH 4
                                    :NAME A :ORDER >)
                           (1 . B)
                           (0 . A)))
           '((:HEADER :DIMENSIONS (3) :MAXIMUM-LENGTH 4
                      :NAME A :ORDER >)
             (1 . B)
             (0 . A)))
    :hints ((\"Goal\" :in-theory (disable (compress1))))
    :rule-classes nil)
  (defthm ouch-3
    nil
    :hints ((\"Goal\" :use true-formula-3))
    :rule-classes nil)
  ~ev[]

  Fixed a soundness bug involving measured subsets and ~ilc[verify-termination],
  by changing ~ilc[verify-termination] so that it uses ~ilc[make-event].
  ~l[verify-termination], in particular the discussion about ~ilc[make-event]
  near the end of that ~il[documentation] topic.  Peter Dillinger first raised
  the idea to us of making such a change; when we found this soundness bug, we
  were certainly happy to do so!

  Fixed a bug that could cause a hard Lisp error but not, apparently,
  unsoundness.  The bug was in the lack of attention to the order of guard and
  type declarations when checking for redundancy.  In the following example,
  the second definition was redundant during the first pass of the
  ~ilc[encapsulate] form.  The second definition, however, was stored on the
  second pass with a guard of ~c[(and (consp (car x)) (consp x))], which caused
  a hard Lisp error when evaluating ~c[(foo 3)] due to a misguided attempt to
  evaluate ~c[(car 3)] in raw Lisp.  The fix is to restrict redundancy of
  definitions so that the guard and type declarations must be in the same order
  for the two definitions.
  ~bv[]
  (encapsulate
   ()
   (local (defun foo (x)
            (declare (xargs :guard (consp x)))
            (declare (xargs :guard (consp (car x))))
            x))
   (defun foo (x)
     (declare (xargs :guard (consp (car x))))
     (declare (xargs :guard (consp x)))
     x))
  ; Now we get a hard Lisp error from evaluation of the guard of foo:
  (foo 3)
  ~ev[]

  Fixed a bug in the guard violation report for function
  ~ilc[intern-in-package-of-symbol].  Thanks to Dave Greve for bringing this
  bug to our attention.

  Made a change to allow certain ~il[hints], in particular certain
  ~c[:]~ilc[clause-processor] hints, that had previously caused errors during
  termination proofs by viewing the function being defined as not yet existing.
  Thanks to Sol Swords for bringing this issue to our attention with a nice
  example.

  ACL2 now properly handles interrupts (via control-c) issued during printing
  of the checkpoint summary.  Previously it was possible on some platforms to
  make ACL2 hang when interrupting both during a proof and during the ensuing
  printing of the checkpoint summary.  Thanks to Jared Davis and Sol Swords for
  bringing this problem to our attention.

  Fixed a bug that was preventing, inside some book ~c[\"b\"], the use of a
  ~c[:dir] argument to ~ilc[include-book] that refers to a directory defined
  using ~ilc[add-include-book-dir] earlier in the book ~c[\"b\"].  (We found
  this ourselves, but we thank John Cowles for observing it independently and
  sending us a nice example.)

  (GCL and CCL only) Fixed a bug in certain under-the-hood type inferencing.
  Thanks to Sol Swords for sending an example using ~il[stobj]s defined with
  the ~c[:inline] keyword, along with a helpful backtrace in CCL, and to Gary
  Byers for his debugging help.

  Fixed a bug in ~ilc[print-gv], which was mishandling calls of functions with
  more than one argument.

  Fixed a bug in the handling of ~ilc[AND] and ~ilc[OR] terms by the
  ~il[proof-checker] command ~c[DV], including numeric (``diving'') commands.
  Thanks for Mark Reitblatt for bringing this problems to our attention with a
  helpful example.

  Fixed printing of goal names resulting from the application of ~c[:OR]
  ~il[hints] so that they aren't ugly when working in other than the
  ~c[\"ACL2\"] package.  Thanks to Sol Swords for bringing this issue to our
  attention.

  Fixed ~il[proof-tree] printing so that interrupts will not cause problems
  with hiding ordinary output because of incomplete proof-tree output.  Thanks
  to Peter Dillinger for pointing out this issue.

  Fixed a hard error that could be caused by mishandling a ~ilc[force]d
  hypothesis during ~il[forward-chaining].  Thanks to Peter Dillinger for
  bringing this bug to our attention by sending a useful example.

  Fixed a bug that could cause simplifications to fail because of alleged
  ``specious simplification.''  This bug could appear when deriving an equality
  from the linear arithmetic database, and then attempting to add this equality
  to the current goal's hypotheses when it was already present.  Thanks to Eric
  Smith for sending a helpful example (in July 2005!) that helped us debug this
  issue.

  Fixed a bug in processing of ~c[:]~ilc[type-set-inverter] rules.

  Fixed a bug that was causing an error, at least for an underlying Lisp of CCL
  (OpenMCL), when ~ilc[ec-call] was applied to a term returning multiple
  values.  Thanks to Sol Swords for sending an example that brought this bug to
  our attention.

  Fixed handling of array orders to treat keyword value ~c[:order :none]
  correctly from an array's ~ilc[header].  Previously, there were two problems.
  One problem was that ~c[:order :none] was treated like the default for
  ~c[:order], ~c[<], while ~c[:order nil] was treated in the manner specified
  by ~c[:order :none] (~pl[arrays]).  Now, both ~c[:order :none] and
  ~c[:order nil] are treated as ~c[:order nil] had been treated, i.e., so that
  there is no reordering of the alist by ~ilc[compress1].  The other problem
  with this case of ~c[:order] was that the ~c[:maximum-length] field of the
  ~ilc[header] was not being respected: the length could grow without bound.
  Now, as previously explained (but not previously implemented) ~-[]
  ~pl[arrays] ~-[] a ~ilc[compress1] call made on behalf of ~c[aset1] causes a
  hard error if the header of the supplied array specifies an ~c[:order] of
  ~c[:none] or ~c[nil].

  An ~c[ignorable] ~ilc[declare] form had caused an error in some contexts when
  it should have been allowed.  In particular, this problem could arise when
  using an ~c[ignorable] declaration at the top level in a ~ilc[defabbrev]
  form.  It could also arise upon calling ~ilc[verify-termination] when the
  corresponding ~ilc[defun] form contained an ~c[ignorable] declaration at the
  top level.  These bugs have been fixed.

  Contrary to existing documentation (~pl[make-event-details]), the value of
  ``~ilc[ld] special variable'' ~ilc[ld-skip-proofsp] was always set to ~c[nil]
  during ~ilc[make-event] expansion, not merely when the ~c[make-event] form
  has a non-~c[nil] value for keyword parameter ~c[:check-expansion].  This has
  been fixed.  Thanks to Sol Swords for bringing this issue to our attention.

  We have disallowed the certification of a book when not at the top-level,
  either directly in the top-level loop or at the top level of ~ilc[ld].
  Before this restriction, the following would certify a book with a definition
  such as ~c[(defun foo (x) (h x))] that calls function ~c[h] before defining
  it, if the certification was by way of the form such as:
  ~bv[]
  (er-progn (defun h (x) x) (certify-book \"my-book\"))
  ~ev[]
  But a subsequent ~ilc[include-book] of ~c[\"my-book\"] would then fail,
  because ~c[h] is not defined at the top level.

  Printing with ~ilc[fmt] directive ~c[~~c] now works properly even when the
  print-base is other than 10.  Thanks to Sol Swords for reporting this bug and
  providing a fix for it.

  (SBCL, CMUCL, and CCL only) Fixed a bug in ~ilc[sys-call-status] in the case
  that the underlying Common Lisp is SBCL, CMUCL, or CCL (OpenMCL).  Thanks to
  Jun Sawada for bringing this bug to our attention and providing a fix.

  Fixed a bug that was preventing ~ilc[local] ~ilc[defstobj] events in
  ~ilc[encapsulate] events.  Thanks to Jared Davis for bringing this bug to our
  attention.

  Fixed a bug evidenced by error message ``Unexpected form in certification
  world'', which could result from attempting to certify a book after
  evaluating an ~ilc[encapsulate] form with a local ~ilc[defmacro].  Thanks to
  Jared Davis for pointing out this bug and sending the example:
  ~bv[]
  (encapsulate
   ()
   (local (defmacro foo (x) `(table foo 'bar ,x)))
   (local (foo 3)))
  ~ev[]

  Formerly, evaluating a ~ilc[trace$] form inside a ~il[wormhole] such as the
  ~il[break-rewrite] loop could leave the user in a bad state after returning
  to the top level, in which that function could not be untraced.  This has
  been fixed.  Note however that when you proceed from a break in the
  ~il[break-rewrite] loop, the tracing state will be the same as it was when
  you entered that break: all effects of calling ~ilc[trace$] and
  ~ilc[untrace$] are erased when you proceed from the break.

  A ~c[:]~ilc[guard] of ~c[(and)] is no longer ignored.  Thanks to Sol Swords
  for bringing this bug to our attention.

  A bug has been fixed that could result in needlessly weak induction schemes
  in the case that a recursive call is made in the first argument of
  ~ilc[prog2$].  This has been fixed by including ~ilc[prog2$] as a default
  ruler-extender in the new ruler-extenders feature (see above, and
  ~pl[ruler-extenders]).  For details on this bug see Example 11 in distributed
  book ~c[books/misc/misc2/ruler-extenders-tests.lisp].

  (For CCL/OpenMCL on Windows) ACL2 should now build on CCL (OpenMCL) on
  Windows.  Thanks to David Rager for bringing this issue to our attention and
  helping with a fix that worked for CCL 1.2, and to the CCL team for improving
  handling of Windows-style filenames in CCL 1.3.

  ~st[NEW AND UPDATED BOOKS]

  See ~url[http://code.google.com/p/acl2-books/wiki/BooksSince34] for a list
  of books in Version  3.5 of ACL2 but not Version  3.4.

  Run the shell command
  ~bv[]
  ~c[svn log -r 94:HEAD]
  ~ev[]
  to see all changes to ~c[books/] since the release of Version  3.4.

  Here are just a few highlights.

  Thanks largely to Jared Davis, many ~c[Makefile]s have been improved to do
  automatic dependency analysis.  ~l[books-certification-classic] for how to
  get your own ~c[Makefile]s to do this by adding a line:
  ~c[-include Makefile-deps].

  Libraries ~c[books/arithmetic-4/] and ~c[books/rtl/rel7/] have been
  eliminated from the default book certification (~c[make regression]), in
  favor of new libraries ~c[books/arithmetic-5/] and ~c[books/rtl/rel8/]
  contributed by Robert Krug and Hanbing Liu, respectively.  They and Jun
  Sawada have arranged the compatibility of these libraries; i.e., it is
  possible to evaluate both of the following in the same session:
  ~bv[]
  (include-book \"arithmetic-5/top\" :dir :system)
  (include-book \"rtl/rel8/lib/top\" :dir :system)
  ~ev[]

  Library ~c[books/rtl/rel1/] is no longer certified by default (though it is
  still distributed in support of ACL2(r); ~pl[real]).

  ~st[EMACS SUPPORT]

  Slightly modified ~c[Control-t e] (defined in ~c[emacs/emacs-acl2.el]) to
  comprehend the notion of an ``ACL2 scope'', and added ~c[Control-t o] to
  insert a superior ~ilc[encapsulate] defining such a scope.  See the Emacs
  documentation for ~c[Control-t e] (generally obtained after typing
  ~c[Control-h k]).

  Modified distributed file ~c[emacs/emacs-acl2.el] so that if you put the
  following two forms in your ~c[~~/.emacs] file above the form that loads
  ~c[emacs/emacs-acl2.el], then Emacs will not start up a shell.  Thanks to
  Terry Parks for leading us to this modification.
  ~bv[]
  (defvar acl2-skip-shell nil)
  (setq acl2-skip-shell t)
  ~ev[]

  ~st[EXPERIMENTAL HONS VERSION]

  Bob Boyer and others have contributed numerous changes for the experimental
  ``~c[hons]'' version of ACL2 (~pl[hons-and-memoization]).

  The ACL2 ~ilc[state] can now be queried with ~c[(@ hons-enabled)] so that a
  result of ~c[t] says that one is in the experimental ~c[hons] version, while
  ~c[nil] says the opposite.

  ~/~/")

; Changes made March 9-16, 2009 (after v3-4), for more efficient handling of
; certificates, etc.:

; The Essay on Skip-proofs was redone, and a new Essay on Soundness Threats was
; added, that explain current handling of skip-proofs, redef, etc.  The basic
; idea is that we have eliminated state globals 'skipped-proofsp and
; include-book-alist-state, instead tracking more in the world
; (e.g. include-book-alist-all).  See in particular install-event, which
; handles such matters, and note that maybe-add-command-landmark no longer does
; so.  (We also changed include-book-fn and encapsulate-fn for this purpose.)
; We added state global 'skip-proofs-by-system to help (again, see
; install-event).

; Of course, there were miscellaneous supporting changes, some in comments.  In
; particular, we (about a month later) eliminated chk-certification-worldxxx.
; Also, eval-event-lst now returns an extra element, which can be a natural
; number we can supply to nthcdr to eliminate some expense from our call of
; expansion-alist-pkg-names in certify-book-fn.  This value is passed to
; process-embedded-events, and back from it in the case that the caller is
; 'certify-book.

; We also changed the use of check-sum so that we don't include the
; expansion-alist with the events from the actual book.  For calls of
; check-sum-obj on event lists that support the handling of certificates, we
; now use only the events from the book ev-lst and no longer include events in
; the expansion-alist.  Instead, we rely on the check-sum of the cert-obj,
; which is still incorporated in the certificate, for ensuring that we have the
; right expansion-alist.

(deflabel |NOTE-3-5(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.5(r) (May, 2009) Notes~/

  ~/

  Please also ~pl[note-3-5] for changes in Version  3.5 of ACL2.

  This release incorporates improvements from Ruben Gamboa in support for
  non-standard analysis in ACL2(r), in the following ways:

  ACL2(r) now supports non-classical objects that are not also numeric, e.g.,
  non-classical cons pairs.  Consequently, the built-in ~c[standard-numberp]
  has been replaced with ~ilc[standardp].

  If ~c[f] is a classical function, the value ~c[(f x1 ... xn)] is guaranteed
  to be standard when the ~c[xi] are standard.  ACL2(r) can now recognize this
  fact automatically, using ~c[defun-std].  For example, the following can be
  used to assert that the square root of 2 is a standard value.
  ~bv[]
  (defthm-std sqrt-2-rational
    (standardp (acl2-sqrt 2)))
  ~ev[]
  More generally, the expression ~c[(f x1 ... xn)] can contain free variables,
  but the result is guaranteed to be standard when the variables take on
  standard variables, as in the following:
  ~bv[]
  (defthm-std sqrt-x-rational
    (implies (standardp x)
             (standardp (acl2-sqrt x))))
  ~ev[]

  A potential soundness bug in ~ilc[encapsulate] was fixed.  Specifically, when
  a classical, constrained function is instantiated with a lambda expression
  containing free variables, it may produce non-standard values depending on
  the values of the free variables.  This means that the functional
  instantiation cannot be used to justify a non-classical theorem.  For
  example, consider the following sequence:
  ~bv[]
  (encapsulate
    ((f (x) t))
    (local (defun f (x) x)))
  (defthm-std f-x-standard
    (implies (standardp x)
             (standardp (f x))))
  (defthm plus-x-standard
    (implies (standardp x)
             (standardp (+ x y)))
    :hints ((\"Goal\"
             :use ((:functional-instance f-x-standard
                                         (f (lambda (x) (+ x y))))))))
  (defthm plus-x-eps-not-standard
    (implies (standardp x)
             (not (standardp (+ x (/ (i-large-integer)))))))
  (defthm nil-iff-t
    nil
    :hints ((\"Goal\"
            :use ((:instance plus-x-standard
                             (y (/ (i-large-integer))))
                  (:instance plus-x-eps-not-standard)))))
  ~ev[]

  ACL2(r) also supports the introduction of non-classical functions with
  ~ilc[defchoose].  These behave just as normal functions introduced with
  ~ilc[defchoose], but they have a non-classical choice property.

  Finally, ACL2(r) now comes with a revamped library supporting non-standard
  analysis, still distributed separately as ~c[books/nonstd/].  The new library
  uses ~ilc[defchoose] to state more natural and useful versions of the IVT,
  MVT, etc.  It also supports the introduction of inverse functions, e.g.,
  logarithms.  Finally, the library has much more extensive support for
  differentiation.

  ~/
  ")

(deflabel note-3-6

; Introduced qfuncall to avoid CCL warnings.

; Regarding "Fixed a bug in reporting failures when monitoring rewrite rules
; with free variables in the hypotheses": the bug was in
; tilde-@-failure-reason-free-phrase, taking a car before checking we had a
; consp.  However, we also found a bug (by inspection, never evoking it) in
; relieve-hyps1-free-2, and we added comments to relieve-hyps1-free-1 and
; relieve-hyps1-free-2 to explain better what we are returning.

; Some code has been refactored in the following functions in support of Daron
; Vroon's work on CCG analysis: put-induction-info, chk-acceptable-defuns,
; defuns-fn0, and defuns-fn.

; Delete dead code, redundant-event-tuplep and (now dead) good-defun-mode-p.

; Here is the proof of nil (using ACL2 Version  3.5) promised below regarding
; the soundness bug related to redundancy", as a certifiable book.
;;;;; start proof of nil ;;;;;
; (in-package "ACL2")
;
; (encapsulate
;  ()
;
;  (set-ruler-extenders :all)
;
;  (local
;   (encapsulate
;    ()
;    (defun f (x)
;      (if (equal x 17)
;          x
;        (cons (if (consp x) (f (cdr x)) x)
;              10)))))
;
;  (set-ruler-extenders :basic)
;
;  (encapsulate
;   ()
;   (defun f (x)
;     (if (equal x 17)
;         x
;       (cons (if (consp x) (f (cdr x)) x)
;             10)))))
;
; (defthm bad
;   (implies (true-listp x)
;            (consp x))
;   :hints (("Goal" :induct (f x))))
;
; (defthm contradiction
;   nil
;   :hints (("Goal" :use ((:instance bad (x nil)))))
;   :rule-classes nil)
;;;;; end proof of nil ;;;;;

; Here are the proofs of nil (using ACL2 Version  3.5) promised below regarding
; the "soundness bugs related to the handling of ~il[subversive-recursion]", as
; certifiable books.  The first exploits the bug in subversive-cliquep, while
; the second exploits the bug in subversivep.

;;;;; start proof of nil ;;;;;
; (in-package "ACL2")
;
; (encapsulate
;  ((my-test (x) t))
;  (local (defun my-test (x)
;           (declare (ignore x))
;           nil))
;  (set-bogus-mutual-recursion-ok t)
;  (mutual-recursion (defun f (x)
;                      (if (consp x)
;                          (f (cdr x))
;                        x))
;                    (defun g (x)
;                      (declare (xargs :measure (acl2-count x)))
;                      (if (my-test x)
;                          (not (g x))
;                        x))))
;
; (defthm my-test-is-nil
;   (null (my-test x))
;   :hints (("Goal" :use g))
;   :rule-classes nil)
;
; (defthm contradiction
;   nil
;   :hints (("Goal" :use ((:instance (:functional-instance my-test-is-nil
;                                                          (my-test identity))
;                                    (x t)))))
;   :rule-classes nil)
;;;;; end proof of nil ;;;;;

; As promised above, the following exploits the bug in subversivep in ACL2
; Version  3.5.
;;;;; start proof of nil ;;;;;
; (in-package "ACL2")
;
; (encapsulate
;  ((my-test (x) t))
;  (local (defun my-test (x)
;           (declare (ignore x))
;           nil))
;  (defun g (x)
;    (declare (xargs :measure (acl2-count x)))
;    (if (equal x (cons x x))
;        (g x)
;      (if (my-test x)
;          (not (g x))
;        x))))
;
; (defthm my-test-is-nil
;   (null (my-test x))
;   :hints (("Goal" :use g))
;   :rule-classes nil)
;
; (defthm contradiction
;   nil
;   :hints (("Goal" :use ((:instance (:functional-instance my-test-is-nil
;                                                          (my-test identity))
;                                    (x t)))))
;   :rule-classes nil)
;;;;; end proof of nil ;;;;;

; Peter Dillinger contributed a small refactoring of the code for relocating
; certificate files, including function make-certificate-file-relocated.

; Made a couple of changes in messages printed by our-abort, which is
; responsible for throwing one back to the top-level ACL2 loop in the event of
; a hard Lisp error.  We incorporated Peter Dillinger's request that the
; set-debugger-enable suggested be inhibited if set-debugger-enable-fn is
; untouchable.  Also, we no longer print "(Use (a!) ..."; see the comment
; there.

; Commented out a dynamic-extent declaration in read-object in case it is
; responsible for random Allegro CL failures related to reading certificates.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  3.6 (August, 2009) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  3.5 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, distributed books, Emacs support, and experimental
  ~il[hons-and-memoization] version.  Each change is described in just one
  category, though of course many changes could be placed in more than one
  category.

  Note that (starting with ACL2 Version  3.5) LispWorks is no longer supported
  as a platform for ACL2, as the LispWorks compiler could not handle the ACL2
  sources; see comments in the ACL2 sources about ``function size'' being ``too
  large''.

  ~st[CHANGES TO EXISTING FEATURES]

  In the ~ilc[xargs] ~ilc[declare] form, the function symbol (or symbols,
  plural, in the case of ~ilc[mutual-recursion]) being defined may now be used
  in the specified ~c[:measure] and ~c[:]~ilc[guard] terms.  Note that, the
  definition(s) of the new symbol(s) are still not used in the termination
  proof.  Thanks to Daron Vroon for discussions leading to this addition for
  the measure and to Dave Greve for requesting such an enhancement for the
  guard.

  Processing of the ~c[:guard-hints] in an ~ilc[xargs] ~ilc[declare] form is
  now delayed until the start of ~il[guard] verification.  As a result, the
  function symbol(s) being defined may now be used as one might expect in such
  hints, for example in an ~c[:in-theory] form.  Thanks to Jared Davis for
  suggesting that we make such an improvement and providing an example.

  Made a low-level change to ~ilc[make-event] in support of the ACL2s utility
  ``~c[dynamic-make-event]''.  Thanks to Peter Dillinger for explaining the
  issue leading to this change.

  Modified the failure message printed when a measure conjecture fails to be
  proved, to indicate whether or not the hint ~c[:ruler-extenders :all] would
  create a different measure conjecture.  Thanks to Peter Dillinger for
  suggesting such a modification.

  A call of ~ilc[add-default-hints] had added hints to the end of the current
  list of default hints.  Now, it adds them to the beginning of that list, so
  that they are tried first.  However, ~ilc[add-default-hints] now takes a
  keyword argument, ~c[:at-end].  If that argument is supplied and evaluates to
  other than ~c[nil], then the previous behavior occurs.

  When ~ilc[save-exec] is used to save ACL2 images, the build dates are now
  printed on separate lines at startup and in the executable script.  Thanks
  To Bob Boyer for requesting some newlines.

  ~ilc[Forward-chaining] rules are now generated so that every ~ilc[let] (also
  ~ilc[let*] and ~ilc[lambda]) expression is expanded away, as is every
  call of a so-called ``guard holder'' (~ilc[must-be-equal], ~ilc[prog2$],
  ~ilc[ec-call], ~ilc[the]).  These were formerly only expanded away in the
  conclusion.  Thanks to Konrad Slind for a helpful email leading to this
  change.

  ~ilc[Current-theory] now causes a hard error when applied to a name not found
  in the current ACL2 logical ~il[world], rather than simply returning ~c[nil].

  When the underlying Common Lisp is GCL, ACL2 no longer uses the ~c[#n=]
  reader macro when writing out certain files, including ~il[certificate]
  files.  In all other Lisps, it now uses the ~c[#n=] ``print-circle'' feature
  not only for ~il[certificate] files and ``~c[expansion.lsp]'' files written
  for example in support of ~ilc[make-event], but also for files written in
  support of ~c[:]~ilc[comp].  This is all managed with new ~il[state] global
  variable ~c[print-circle-files]; ~pl[print-control].  Thanks to Dave Greve
  for pointing out that GCL is limited by default to 1024 indices for ~c[#n=].

  ~st[NEW FEATURES]

  A documentation topic explains in some detail how ~il[hints] work with the
  ACL2 proof ``waterfall'': ~pl[hints-and-the-waterfall].  This topic may be
  useful to those who write computed hints (~pl[computed-hints]) or other
  advanced hints.

  Added a new hint keyword, ~c[:no-thanks], that avoids printing the usual
  ``Thanks'' message for ~il[hints].  Thanks to Peter Dillinger for requesting
  this feature.

  Added a new hint keyword, ~c[:backtrack], that checks the goals produced by
  processing a goal, and can cause the goal to be re-processed using a new
  hint.  ~l[hints].  Thanks to Pete Manolios for a conversation that led to the
  idea of this hint.

  Added a new class of hints, ~il[override-hints], that is similar to
  ~il[default-hints], except that override-hints are always applied, even if
  the user has supplied a hint explicitly for the goal.  ~l[override-hints].
  Thanks to Pete Manolios and Harsh Raju Chamarthi for useful discussions on
  this topic, including its application to testing.

  When a goal ready to be pushed for proof by induction is given the new hint
  ``~c[:do-not-induct :otf]'', it is indeed pushed for proof by induction,
  rather than causing immediate failure as in the case of the hint
  ``~c[:do-not-induct t]''.  Instead, the proof fails when the goal is later
  picked to be proved by induction.  Thanks to Peter Dillinger for discussions
  leading to this feature.

  Related to computed hints only: Each history entry in the list stored in
  variable ~c[HIST] (~pl[computed-hints]) now has a ~c[:CLAUSE] field, which
  provide's access to a goal's parent, parent's parent, and so on (within the
  same induction and forcing round only).

  It is now possible to inhibit warnings produced by ~ilc[in-theory]
  ~il[events] and ~il[hints] that occur when certain built-in definitions and
  executable-counterparts are disabled: just evaluate
  ~c[(assign verbose-theory-warning nil)].  Thanks to Jared Davis (and probably
  others in the past) for requesting such a mechanism.

  ~st[HEURISTIC IMPROVEMENTS]

  A source function (~c[linearize-lst]) was replaced by tail-recursive code,
  which can avoid a stack overflow.  Thanks to Dave Greve for sending a helpful
  example.

  The heuristics for limiting ~il[forward-chaining] have been slightly relaxed,
  to allow derivations based on the occurrence of all arguments of the
  forward-chaining rule's conclusion in the goal (after stripping leading calls
  of ~c[NOT]).  Thanks to Dave Greve for contributing this improvement and
  providing a motivating example.

  We simplified induction schemes by eliminating each hypothesis of the form
  ~c[(not (equal term (quote const)))] for which some other hypothesis in the
  same case equates term with some (presumably other) quoted constant.  Thanks
  to Dave Greve for requesting an improvement of this sort.

  ~st[BUG FIXES]

  Fixed a soundness bug related to redundancy of ~ilc[encapsulate] events
  (~pl[redundant-events]) and ~il[ruler-extenders].  A proof of ~c[nil] in ACL2
  Version  3.5 appears in a comment in ~c[(deflabel note-3-6 ...)] in the ACL2
  source code.  The fix is to insist that in order for one ~ilc[encapsulate]
  event to be redundant with another, they must be evaluated with the same
  ~il[default-ruler-extenders].  Analogous to this issue of
  ~il[default-ruler-extenders] for ~ilc[encapsulate]s is an issue of
  the ~il[default-verify-guards-eagerness], which has similarly been fixed.

  Fixed soundness bugs related to the handling of ~il[subversive-recursions]
  for ~il[constraint]s.  Proofs of ~c[nil] in ACL2 Version  3.5 appear in a
  comment in ~c[(deflabel note-3-6 ...)] in the ACL2 source code.

  Fixed a bug that could cause the following error during calls of
  ~ilc[certify-book] in the presence of calls of ~ilc[skip-proofs] in the book:
  ~bv[]
     ACL2 Warning [Skip-proofs] in

     HARD ACL2 ERROR in FMT0: Illegal Fmt Syntax.  The tilde-@ directive at
     position 0 of the string below is illegal because its variable evaluated
     to 0, which is neither a string nor a list.

     \"~~@0\"
  ~ev[]
  Thanks to Dave Greve for reporting this bug and making available a very
  helpful test case.

  The ~c[:corollary] of a rule (~pl[rule-classes]) failed to use the
  ~ilc[default-hints] of the logical ~il[world].  This has been fixed.

  (CCL only) We removed a call, for CCL 1.3 (and beyond) only, of foreign
  function ~c[sync] in the closing of output channels.  Thanks to Daron Vroon
  for reporting issues with such a call on a non-Intel platform.

  Fixed a bug in reporting failures when monitoring rewrite rules with free
  variables in the hypotheses, that could cause a hard Lisp error (from which
  ACL2 continues, however).  Thanks to Eric Smith for sending a very helpful
  example with his bug report.

  Fixed the handling of ~c[:induct] ~il[hints], which had thrown away hint
  information from parent goals.  For example, the ~ilc[thm] form below failed
  to prove even though the second hint is in some sense superfluous; induction
  occurs automatically at ~c[\"Goal'\"] even without the hint on that.  The
  failure was due to discarding the hint information on ~c[\"Goal\"].
  ~bv[]
  (in-theory (disable append))
  (thm (equal (cdr (cons a (append (append x y) z))) (append x y z))
       :hints
       ((\"Goal\" :in-theory (enable append))
        (\"Goal'\" :induct t) ; failed unless this line is commented out
        ))
  ~ev[]

  Fixed a bug in the ~ilc[args] command that was failing to show the formals of
  primitives (built-in functions like ~c[consp] that do not come with explicit
  definitions).  Thanks to John Cowles for pointing out this bug.  (At a lower
  level, the bug was that primitives failed to have ~c['stobjs-in] or
  ~c['stobjs-out] properties.)

  Fixed bugs in the utility supporting moving directories of certified books,
  sometimes used in Debian builds (as described in source function
  ~c[make-certificate-file]).  Thanks to Alan Dunn for pointing out such a bug,
  in paths associated with ~ilc[defpkg] events in ~il[portcullis] commands in
  ~il[certificate]s (which are used for error reporting).  There were also
  bugs, now fixed, that prevented renaming some book paths.  Please note that
  this mechanism is not guaranteed to be sound; in particular, it can probably
  misbehave when macros are used to generate portcullis events.  However, it
  seems likely that such issues will be very rare.

  Eliminated warnings that could arise when tracing a function with
  ~ilc[trace$].  Now, when ~ilc[trace$] is applied to a function without option
  ~c[:native], that function's declarations and documentation are discarded.

  Fixed a bug that could cause a failure when building an executable image
  using SBCL as the underlying Common Lisp.  Thanks to Jun Sawada for reporting
  this failure.  We made a similar fix for CMUCL.

  Fixed a bug in ~ilc[save-exec] in the case that an absolute pathnanme is
  supplied.  Thanks to Jared Davis for bringing this bug to our attention.

  Fixed a bug in the use of ~ilc[trace$] with the ~c[:native] and
  ~c[:multiplicity] options that caused hard errors for some underlying Lisps.

  Fixed a bug in the interaction of ~ilc[trace$] and ~c[:]~ilc[comp], which
  caused error as ~ilc[comp] tried to re-trace functions that it temporarily
  untraced.

  ~st[NEW AND UPDATED BOOKS AND RELATED INFRASTRUCTURE]

  See ~url[http://code.google.com/p/acl2-books/wiki/BooksSince35] for a list of
  books in Version  3.6 of ACL2 but not Version  3.5.  For ~em[changes] to
  existing books rather than ~em[additions], see the log entries at
  ~url[http://code.google.com/p/acl2-books/source/list] starting with revision
  r286 up through revision r329.

  It is no longer required to specify a value for environment (or `make')
  variable ~c[ACL2_SYSTEM_BOOKS] when running `make' in the distributed book
  directory, even when that directory is not under the directory containing the
  ACL2 executable.  Thanks to Peter Dillinger for providing this improvement,
  by modifying ~c[books/Makefile-generic] and, as needed, distributed book
  Makefiles.

  Thanks to Peter Dillinger, some books in support of the ACL2 Sedan (ACL2s)
  are more easily available for ACL2 users who do not use ACL2s.
  ~l[acl2-sedan].

  ~st[EMACS SUPPORT]

  If the following form is evaluated before the load of
  ~c[emacs/emacs-acl2.el], then variables are now set to reflect the directory
  containing that file.
  ~bv[]
  (if (boundp '*acl2-sources-dir*)
      (makunbound '*acl2-sources-dir*))
  ~ev[]

  Fixed info-dir-entry line in generated info file (by patching
  ~c[doc/write-acl2-texinfo.lisp]).  Thanks to Alan Dunn for providing this
  patch.

  ~st[EXPERIMENTAL HONS VERSION]

  Bob Boyer and others have contributed numerous changes for the experimental
  ``~c[hons]'' version of ACL2 (~pl[hons-and-memoization]).  A number of these
  have been crafted to work specifically with CCL (Clozure Common Lisp,
  formerly OpenMCL), which is now required as the underlying Lisp for the
  ``~c[hons]'' version of ACL2.

  A heuristic (source function ~c[too-many-ifs] has been made more scalable
  (for the non-HONS version as well, in fact), but with no functional change.
  Thanks to Jared Davis for noticing performance issues and suggesting fixes.

  Other changes including the following, quoting Bob Boyer:

  ~bq[]
  The CCL CASE macro now does better than a dumb linear
  search in some CASEes.

  SH and CSH are functions to talk to the underlying
  Gnu-Linux from CCL.  Good for repeated calling when
  you simply cannot afford the copying cost of a FORK
  because you are using, say, a dozen gigabytes.

  Added CCL compiler-macros for IF and OR, to support
  some 'coverage' analysis, cf. IF-REPORT, extending
  the profiling.

  Introduced the type 'mfixnum' so that things like
  counting honses and counting calls to memoized or
  profiled functions can run fast in CCL 64 bits and
  yet still run at all under 32 bits.

  Moved all HONSes to CCL's newish static space,
  which permits the address of a cons to be used as a
  hash key, as in most Lisps.  (CCL moves most conses
  and most everything when it does a compacting-gc.)

  Quite a few changes in the memoize-fn reporting.

  Added a timer facility, cf. call-with-timeout.
  Good for running under throttle some gross thoughts
  like 'Is it true that you can't fit 12 pigeons into
  11 holes' on some propositional calculus
  systems/functions.

  Added rwx-size, pid-owner-cmdlines, rss, and
  proc-stat to help see what is really going on
  virtually in Gnu-Linux.

  Fixed at least one bug in compact-print-file and
  helped make its integration into ACL2's
  read-object$ a little more sound.  Still worried
  some about *print-readably* vs. readtable-case.
  Does anyone else stay awake late at night worrying
  about readtable-case?

  Revised how the *watch-dog-process* interrupts the
  main process for the thousandth time, cf. watch.
  Haven't changed it in weeks, which means that (a)
  it is getting tolerable or (b) I've run out of gas.
  ~eq[]

  ~/~/")

(deflabel |NOTE-3-6(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  3.6(r) (August, 2009) Notes~/

  ~/

  Please also ~pl[note-3-6] for changes in Version  3.6 of ACL2.

  ~/
  ")

(deflabel note-3-6-1

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  3.6.1 (September, 2009) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  The essential changes to ACL2 since Version  3.6 are the two bug fixes
  described below.  There was also some functionality-neutral code refactoring,
  as requested by Daron Vroon in support of the ACL2 Sedan (~pl[acl2-sedan]).
  Also see ~url[http://code.google.com/p/acl2-books/wiki/BooksSince36] for a
  list of books in Version  3.6.1 of ACL2 but not Version  3.6.  For
  ~em[changes] to existing books rather than ~em[additions], see the log
  entries at ~url[http://code.google.com/p/acl2-books/source/list] starting
  with revision r329 up through revision 350.

  Fixed a soundness bug in the handling of ~il[ruler-extenders], specifically
  in the handling of ~ilc[LET]-expressions.  Thanks to Pete Manolios, who sent
  us a proof of ~c[nil], essentially as follows.  In the termination proof for
  ~c[foo] below, the binding of ~c[x] to ~c[(cons t x)] was not substituted
  into the recursive call of ~c[foo] for purposes of the termination proof.
  ~bv[]
  (defun foo (x)
    (declare (xargs :ruler-extenders :all))
    (let ((x (cons t x)))
      (if (endp (cdr x))
          x
        (cons t (foo (cdr x))))))

  (defthm foo-bad
    nil
    :hints ((\"Goal\"
             :use ((:instance foo (x '(3))))
             :in-theory (disable foo (foo))))
    :rule-classes nil)
  ~ev[]

  Fixed a typo in code supporting ~il[ruler-extenders] (specifically, swapped
  arguments in a recursive call of ACL2 source function
  ~c[get-ruler-extenders2], which could cause problems for functions defined
  using ~ilc[mutual-recursion]).  Thanks to Daron Vroon for bringing this bug
  to our attention, pointing out the swapped arguments.

  ~/~/")

(deflabel note-4-0

; Thanks to Alan Dunn for contributing improvements to :doc markup and :doc
; truncate.

; Regarding "Fixed a bug in ~c[:]~ilc[pso] during the printing of failure
; messages for termination proofs.": The bug was the duplication of NAMES in
; the VARS for an IO? form in prove-termination.  The macro IO? now has a guard
; prohibiting such duplicates.

; Modified *1*-symbol (also, but probably less important, *1*-symbol? and
; global-symbol) to use find-package-fast, resulting in about 2% performance
; improvement for a hons-based include-book that can take a couple of minutes.

; Regarding the change in the too-many-ifs heuristic: See the Essay on
; Too-many-ifs.

; We improved the error provided when certify-book or include-book fails
; because :skip-proofs-okp or other keywords need to be non-nil.

; We improved The error provided when computing with a live
; stobj with :guard-checking :none, to suggest (set-guard-checking nil).
; Thanks to Harsh Raju Chamarthi for related helpful conversations.

; In support of the new :case-split-limitations hint (see :doc below), the the
; rewrite-constant record was modified, and clausify now takes an sr-limit
; instead of the world.

; In support of the change allowing user control over compilation of books by
; default, raw Lisp variable *suppress-compile* has been renamed
; *suppress-compile-build-time* and state global 'suppress-compile has been
; replaced by state global 'compiler-enabled, which has the opposite parity.

; Removed the prohibition of eval-when for #+cmu; not an issue at least as far
; back as CMUCL 19e.

; Used with-standard-io-syntax for ACL2 source macro write-exec-file so that
; executable scripts can be printed correctly even when implementation-specific
; print variables, such as ccl::*print-string-length*, are set.  Thanks to Bob
; Boyer for suggesting that some such change might be a good idea.

; Added a missing return-from in the raw Lisp GCL code for mod-expt.

; Fixed a bug that was leaving documentation strings in raw Lisp for functions
; that have ACL2 documentation strings.

; Fixed bug in two attempts to coerce a character (instead of a list of one
; character) to a string in bad-lisp-objectp.  Thanks to David Rager for
; pointing out a compiler warning printed by SBCL 1.0.34.2.

; Load-compiled now takes an optional verbosity argument, which is occasionally
; used to print a message saying that a compiled file is being loaded.

; Some changes to include-book should be invisible to the user, but include a
; default for :load-compiled-file of :default instead of :warn.  See :doc
; include-book, which has been improved.

; Weakened the definition of legal-acl2-character-p, in response to a helpful
; email from Jared Davis.

; Eliminated any checks for integers in bad-lisp-objectp, as explained in a
; comment there.

; We now print symbols in suitable packages for some :brr messages, thanks to
; feedback from Jared Davis.

; The directory stored for :dir :system is now always an absolute pathname
; (Unix style).

; In support of using portcullis commands in check-sum computations,
; eliminated the final check-sum-alist message from certify-book,
; which didn't seem appropriate anyhow.

; To prove nil in ACL2 Version_3.6.1 by exploiting the fact that portcullis
; commands were not included in check-sum computations, evaluate (ld
; "script.lsp") in a directory with contents shown below.

;;; % cat sub1.lisp
;;; (in-package "ACL2")
;;;
;;; ; Possible portcullis command:
;;; ; (defun f () 1)
;;;
;;; (defun g () 0)
;;;
;;; % cat sub2.lisp
;;; (in-package "ACL2")
;;;
;;; ; Possible portcullis command:
;;; ; (defun f () 2)
;;;
;;; (defun h () 0)
;;; % cat top1.lisp
;;; (in-package "ACL2")
;;;
;;; (include-book "sub1")
;;; (include-book "sub2")
;;;
;;; ; Certify with sub1's portcullis command.
;;; (defthm f-val-1
;;;   (equal (f) 1))
;;; % cat top2.lisp
;;; (in-package "ACL2")
;;;
;;; (include-book "sub1")
;;; (include-book "sub2")
;;;
;;; ; Certify with sub2's portcullis command.
;;; (defthm f-val-2
;;;   (equal (f) 2))
;;; % cat top.lisp
;;; (in-package "ACL2")
;;;
;;; (include-book "top1")
;;; (include-book "top2")
;;;
;;; (defthm contradiction
;;;   nil
;;;   :rule-classes nil
;;;   :hints (("Goal" :use (f-val-1 f-val-2)
;;;            :in-theory (theory 'ground-zero))))
;;; % cat script.lsp
;;; ; (ld "script.lsp")
;;;
;;; (defun f () 1)
;;; (certify-book "sub1" 1)
;;; (u)
;;; (u)
;;; (certify-book "sub2")
;;; (u)
;;; (certify-book "top1")
;;; (u)
;;;
;;; (defun f () 2)
;;; (certify-book "sub2" 1)
;;; (u)
;;; (u)
;;; (certify-book "sub1")
;;; (u)
;;; (certify-book "top2")
;;; (u)
;;;
;;; (certify-book "top")
;;; %

; Added .NOTPARALLEL to the dependencies of GNUmakefile target, large.  We hope
; this eliminates some build errors when using the -j option.

; Fixed a bug in "make copy-distribution" in GCL builds, where the presence of
; a filename that extends the designated directory string caused an error,
; erroneously stating that the directory already exists.

; Regarding "Fixed a bug that could make ~c[:]~ilc[oops] cause an error.": The
; fix is to use equal instead of eq in retract-world1; see the comment there.
; We similarly modified extend-world1.

; Fixed replace-colons so that something like ~ilc[:oops] in a :doc string
; doesn't cause an error when calling write-texinfo-file.

; Added small optimization in relevant-constraints1, to avoid a call of
; event-responsible-for-proved-constraint.

; Made small changes in doc/write-acl2-texinfo.lisp to avoid warnings when
; building the documentation.

; Improved make clean-doc.

; Eliminated unbound-macro-function and *unbound-macro-function*.  We now check
; in fmakunbound! that fmakunbound really does work on macros.

; Eliminated certify-book-disabledp.

; Fixed a small output bug, by modifying tilde-@-lmi-phrase to account properly
; for the case that functional instances were previously proved only by
; nameless events.  Here is an example:

;   (encapsulate
;    ((f1 (x) t)
;     (f2 (x) t))
;    (local (defun f1 (x) x))
;    (local (defun f2 (x) x))
;    (defthm f1-f2
;      (equal (f1 x) (f2 x))))
;
;   (encapsulate
;    ((g1 (x) t)
;     (g2 (x) t))
;    (local (defun g1 (x) x))
;    (local (defun g2 (x) x))
;    (defthm g1-g2
;      (equal (g1 x) (g2 x))))
;
;   (defthm f-thm1
;     (equal (f1 (cons x x))
;            (f2 (cons x x))))
;
;   (defun k1 (x)
;     (declare (xargs :guard (equal (g1 (cons x x))
;                                   (g2 (cons x x)))))
;     x)
;
;   (defun k2 (x)
;     (k1 x))
;
;   (verify-guards k2
;                  :hints (("Goal" :by (:functional-instance f-thm1
;                                                            (f1 g1)
;                                                            (f2 g2)))))

; Added Essay on the proved-functional-instances-alist, to clarify the life
; cycle of entries in world global 'proved-functional-instances-alist
; (supporting the caching of proved functional instances).

; There is a new source file, multi-threading-raw.lisp, which is loaded only
; when feature :acl2-par is present (see acl2::*acl2-files*).  It was
; contributed by David Rager to support the version of ACL2 that allows
; parallel evaluation (see :doc parallelism).

; Made a small change to the code for exit-lisp.

; In CCL, set ccl::*record-source-file* to nil after discussion with Gary
; Byers, to avoid paying the cost of recording source file information that can
; cause slowdown and isn't typically exploited by ACL2 users.  The HONS version
; still sets this variable to T.

; We now clean up stale values of 'redundant-raw-lisp-discriminator property
; when undoing a defconst or defstobj event, which could free up memory.

; Eliminated the notion of "era" in the ACL2 sources.

; Improved the message after the Q.E.D. for hints :by nil, because the old
; message was a bit misleading in the case of defun (where the goals may not
; actually pass translate).  Thanks to Jared Davis for pointing out this
; issue.

; The function iteratively-grow-constraint was significantly rewritten, and
; since its algorithm was no longer described by that name, we renamed it to
; encapsulate-constraint.

; Changed set-w! to call find-longest-common-retraction with event-p = t in all
; cases, not just when certifying or including books.  Thanks to Sol Swords for
; sending a relevant failure that led us to this change; see the comments in
; set-w!.

; Substantially improved the Essay on Correctness of Meta Reasoning.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  4.0 (July, 2010) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  3.6.1 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, distributed books, Emacs support, and experimental
  versions.  Each change is described in just one category, though of course
  many changes could be placed in more than one category.  Also ~pl[note-3-6-1]
  for other changes since Version  3.6.

  ~st[CHANGES TO EXISTING FEATURES]

  There have been extensive changes to the handling of compiled files for
  books, which may generally be invisible to most users.  The basic idea is
  that when compiled files are loaded on behalf of ~ilc[include-book], they are
  now loaded before events in the book are processed, not afterwards.  This can
  speed up calls of ~c[include-book], especially if the underlying Lisp
  compiles all definitions on the fly (as is the case for CCL and SBCL).  One
  functional change is that for keyword argument ~c[:load-compiled-file] of
  ~ilc[include-book], the values ~c[:comp-raw], ~c[:try], and ~c[:comp!] are no
  longer legal.  (Note that the handling of ~c[:comp-raw] was actually broken,
  so it seems that this value wasn't actually used by anyone; also, the
  handling of ~c[:comp!] formerly could cause an error in some Lisp platforms,
  including SBCL.)  Another change is that if ~ilc[include-book] is called with
  ~c[:load-compiled-file t], then each sub-book must have a compiled file or a
  so-called ``expansion file''; ~pl[book-compiled-file].  In the unlikely event
  that this presents a problem, the makefile provides a way to build with
  compilation disabled; ~pl[compilation].  Users of raw mode
  (~pl[set-raw-mode]) will be happy to know that ~ilc[include-book] now works
  if there an up-to-date compiled file for the book, since ~ilc[portcullis]
  commands are now incorporated into that compiled file.  The mechanism for
  saving expansion files has changed, and the ~c[:save-expansion] argument of
  ~ilc[certify-book] has been eliminated; ~pl[certify-book].  More discussion
  of ACL2's new handling of book compilation is described in a new
  documentation topic; ~pl[book-compiled-file].

  It was possible to get a hard Lisp error when certifying a book with a
  redundant ~ilc[defconst] form whose term is not identical to the existing
  ~ilc[defconst] form for the same name.  The following example illustrates the
  problem, which has been fixed (as part of the change in handling of compiled
  files for books, described above).  Imagine that after the initial
  ~c[(in-package \"ACL2\")] form, file ~c[foo.lisp] has just the form
  ~c[(defconst *a* (append nil nil))].  Then before the fix, we could have:
  ~bv[]
    ACL2 !>(defconst *a* nil)
    [[output omitted]]
    ACL2 !>(certify-book \"foo\" 1)
    [[initial output omitted]
    * Step 5:  Compile the functions defined in \"/v/joe/foo.lisp\".
    Compiling /v/joe/foo.lisp.
    End of Pass 1.
    End of Pass 2.
    OPTIMIZE levels: Safety=0 (No runtime error checking), Space=0, Speed=3
    Finished compiling /vjoe/foo.lisp.
    Loading /v/joe/foo.lisp

    Error: Illegal attempt to redeclare the constant *A*.
  ~ev[]

  The ~ilc[wormhole] facility has been changed to repair a bug that allowed
  guard violations to go undetected.  The major change has to do with the
  treatment of what used to be called the ``pseudo-flag'' argument which has
  been replaced by a quoted lambda expression.  ~l[note-4-0-wormhole-changes]
  for help in converting calls of ~ilc[wormhole].  Also see ~pl[wormhole] and
  ~pl[wormhole-eval].

  The function ~c[assign-wormhole-output] has been eliminated but its
  functionality can be provided by ~ilc[wormhole-eval].

  The ACL2 tutorial has been greatly expanded, for example to include a
  self-contained discussion of rewriting.  ~l[acl2-tutorial].

  Formerly, the ~ilc[mbe] macro and ~ilc[must-be-equal] function were
  disallowed in any definition within an encapsulate having a non-empty
  signature.  Now, these are allowed provided the definition has been declared
  to be non-executable (~pl[defun-nx]).  As a result, ~ilc[defevaluator]
  ~il[events] may now include ~ilc[must-be-equal] among the function symbols
  known by the evaluator; this had not previously been allowed.  Thanks to Sol
  Swords for discussions leading to this relaxation for ~ilc[defevaluator].

  ~ilc[Princ$] now prints strings more efficiently.  Thanks to Jared Davis for
  suggesting the improvements to ~c[princ$].

  The use of ~ilc[xargs] declaration ~c[:non-executable t] no longer requires
  the absence of ~ilc[state] or declared ~ilc[stobj]s among the formal
  parameters of a function definition.  As before, the use of
  ~c[:non-executable t] turns off single-threadedness checking for the body,
  and also as before, attempts to execute the function will fail.  Thanks to
  Sol Swords for requesting this relaxation (for automatic generation of
  so-called ``flag functions'' for definitions using ~ilc[mutual-recursion]).

  The ~il[documentation] has been improved for explaining to advanced users the
  details of the ACL2 hint mechanism; ~pl[hints-and-the-waterfall], and see the
  example about ~c[nonlinearp-default-hint] in distributed book
  ~c[books/hints/basic-tests.lisp].  Thanks to Robert Krug for useful
  discussions, in particular suggesting the above example as one to be
  explained with the documentation.

  The ~c[time$] macro has been enhanced to allow user control of the timing
  message that is printed, and of when it is printed.  ~l[time$].  Thanks to
  Jared Davis for providing the essential design, helpful documentation
  (largely incorporated), and an initial implementation for raw Lisp.

  The ~c[:ttags] argument to ~ilc[include-book] had been required when
  including a certified book that uses trust tags.  This is no longer the case:
  essentially, ~c[:ttags] defaults to ~c[:all] except that warnings will be
  printed.  Thanks to Jared Davis for requesting such a relaxation, and to him
  and Sandip Ray for useful discussions.

  The definition of ~ilc[mv-let] has been modified so that the single-step
  macroexpansion (~pl[trans1]) of its calls can be evaluated.  Thanks to Pete
  Manolios for bringing this evaluation issue to our attention and ensuing
  discussions.

  All calls of so-called ``guard-holders'' ~-[] ~ilc[prog2$],
  ~ilc[must-be-equal] (from calls of ~pl[mbe]), ~ilc[ec-call], and
  ~ilc[mv-list] ~-[] are now removed before storing hypotheses of rules of
  class ~c[:]~ilc[rewrite] or ~c[:]~ilc[linear].  Thanks to Sol Swords for
  requesting this enhancement and sending the following example in the case of
  ~il[rewrite] rules.
  ~bv[]
  (defthm foo
    (prog2$ (cw \"asdf\")
            (and (equal (car (cons x y)) x)
                 (equal (cdr (cons x y)) y))))
  ~ev[]

  The handling of ~ilc[fmt] directive ~c[~~s] has been modified so that if the
  argument is a symbol that would normally be printed using vertical bars
  (~c[|]), then that symbol is printed as with ~c[~~f].  Thanks to Jared Davis
  for providing the following example showing that treatment of ~c[~~s] was a
  bit unexpected: ~c[(cw \"~~s0.~~%\" '|fo\\|o|)].

  Error messages have been improved for ill-formed terms (in ACL2's so-called
  ``translation'' routines).  Thanks to Jared Davis for requesting such an
  enhancement.

  Modified ~ilc[defun-sk] so that it executes in ~c[:]~ilc[logic] mode.
  Previously, evaluation of a ~ilc[defun-sk] event in ~c[:]~ilc[program] mode
  caused a somewhat inscrutable error, but now, ~c[:]~ilc[program] mode is
  treated the same as ~c[:]~ilc[logic] mode for purposes of ~ilc[defun-sk].

  The ``system hacker'' commands ~c[(]~ilc[redef+]~c[)] and
  ~c[(]~ilc[redef-]~c[)] are now embedded event forms
  (~pl[embedded-event-form]), hence may be used in ~il[books] as well as in
  ~ilc[progn] and ~ilc[encapsulate] ~il[events].  Also, these two commands are
  now no-ops in raw Lisp.

  The function symbol ~c[worldp] (in the ~c[\"ACL2\"] package) has been renamed
  to ~c[plist-worldp].

  The function ~c[gc$-fn] (resulting from macroexpansion of ~ilc[gc$]) is now
  in ~c[:]~ilc[logic] mode.  Thanks to Jared Davis for requesting this change.

  The user now has control over whether compilation is used, for example
  whether or not ~ilc[certify-book] compiles by default, using function
  ~c[set-compiler-enabled].  ~l[compilation].

  Modified the conversion of relative to absolute pathnames in ~il[portcullis]
  ~il[command]s for book certification.  Now, more pathnames remain as relative
  pathnames.

  The ~c[\"Ttags\"] warning that can be printed by ~ilc[include-book] is now
  given even if ~ilc[set-inhibit-output-lst] has specified `~c[warning]'.  To
  suppress it, specify ~c[warning!] instead, for example,
  ~c[(set-inhibit-output-lst '(acl2::warning! acl2::proof-tree))].

  On occasion, ACL2 prints the message ``Flushing current installed world'' as
  it cleans up when certain actions (installing a ~il[world]) are interrupted.
  This operation has been sped up considerably.  If your session includes many
  ~il[events], you can probably speed up any such operation further by invoking
  ~ilc[reset-prehistory].  Thanks to Jared Davis for sending a query that led
  us to make this improvement.

  Calls of the form ~c[(ec-call (must-be-equal logic exec))] are no longer
  allowed, since we do not have confidence that they would be handled
  correctly.

  The underlying function for ~ilc[good-bye] (and hence for ~ilc[exit] and
  ~ilc[quit]) is now in ~c[:]~ilc[logic] mode.  Thanks to Jared Davis for
  requesting this enhancement.

  We now require that every function symbol in the ~il[signature] of an
  ~ilc[encapsulate] event have a ~c[:]~ilc[logic] mode definition at the end of
  the first pass, not merely a ~c[:]~ilc[program] mode definition (which
  formerly was sufficient).  You can still define such a function in
  ~c[:program] mode, provided it is followed by a ~c[:logic] mode definition
  (where of course both definitions are ~il[local], since we are discussing
  functions is introduced in the ~il[signature]).  Thanks to Carl Eastlund for
  bringing this issue to our attention.  (Note: An analogous modification has
  been made for ~c[:]~ilc[bdd] ~il[hints] as well.)

  The following functions now have raw Lisp implementations that may run faster
  than their ACL2 definitions: ~ilc[assoc-eq], ~ilc[assoc-equal],
  ~ilc[member-eq], ~ilc[member-equal], ~c[subsetp-eq], ~ilc[subsetp-equal],
  ~ilc[remove-eq], ~ilc[remove-equal], ~ilc[position-eq], and
  ~ilc[position-equal].  Thanks to Jared Davis for suggesting that we consider
  such an improvement.

  We now avoid infinite loops caused when tracing functions that implement
  ~ilc[trace$].  Thanks to Rob Sumners and Eric Smith for useful discussions.

  The implementation of ~ilc[trace!] has been modified slightly, to accommodate
  the fix for ``some holes in the handling of trust tags'' described later,
  below.

  This item applies unless the host Lisp is GCL.  An interrupt (control-c) will
  now cause a proof to exit normally in most cases, by simulating a timeout, as
  though ~ilc[with-prover-time-limit] had been called with a time-limit of 0.
  If the first interrupt doesn't terminate the proof, a second one should do so
  (because a different, more ``severe'' mechanism is used after the first
  attempt).  As a result, ~ilc[redo-flat] should work as one might expect even
  if a proof is interrupted.  Thanks to Dave Greve for requesting this
  enhancement to ~ilc[redo-flat].  Technical note: for reasons related to this
  change, time-limits are no longer checked in evaluator functions
  (~c[ev-fncall], ~c[ev], ~c[ev-lst], ~c[ev-fncall-w], ~c[ev-w], and
  ~c[ev-w-lst]).

  It is now legal for ~il[proof-checker] ~il[macro-command]s to appear in
  ~c[:]~ilc[instructions] that are used in place of ~c[:]~ilc[hints].  Thanks
  to Sandip Ray for (most recently) requesting this feature.

  The value of ~c[:command-conventions] for ~ilc[ld] special variable
  ~c[ld-post-eval-print] is now treated as though it were ~c[t] if the value
  ~ilc[ld] special variable ~c[ld-error-triples] is ~c[nil].  The following
  example illustrates this change.
  ~bv[]
  ACL2 !>(ld-post-eval-print state) ; default
  :COMMAND-CONVENTIONS
  ACL2 !>(ld-error-triples state) ; default
  T
  ACL2 !>(set-ld-error-triples nil state)
  *** Then, before the change:
  ACL2 !>(mv t 3 state)
   3
  *** Instead, after the change:
  ACL2 !>(mv t 3 state)
  (T 3 <state>)
  ~ev[]

  The default behavior of ~ilc[ld] has been changed.  Formerly when an error
  occurred that halted a subsidiary call of ~c[ld], then the parent ~c[ld]
  would continue.  That is no longer the case.  Consider the following
  example.
  ~bv[]
  (ld '((ld '((defun f (x) x)
              (defun bad (x)) ; ERROR -- missing the body
              ))
        (defun g (x) x)))
  ~ev[]
  Formerly, ~c[g] would be defined in the resulting logical ~il[world].  Now,
  the error halts not only the inner ~c[ld] but also the outer ~c[ld].  ~l[ld],
  and for details of the new default value for ~c[:ld-error-action],
  ~c[:RETURN!], see ~pl[ld-error-action].  Also see the paragraph below about
  a new utility, ~c[:]~ilc[p!].  Thanks to Robert Krug and Sandip Ray for
  helpful discussions.

  Environment variable ~c[ACL2-CUSTOMIZATION] has been replaced by
  ~c[ACL2_CUSTOMIZATION] ~-[] that is, the hyphen has been replaced by an
  underscore ~-[] so that it can be set conveniently in the ~c[bash] shell.
  ~l[acl2-customization].

  The ``~c[Warnings:]'' summary is now omitted when there are no warnings,
  where formerly ``~c[Warnings:  None]'' was printed.  Thanks to Jared Davis
  for suggesting this change.

  We have modified the generation of ~c[constraint]s for ~ilc[encapsulate]
  ~il[events] in two primary ways, neither of them likely to affect many users.
  One change is that the virtual movement of definitions and theorems to in
  front of an ~ilc[encapsulate] event, or of definitions to behind that event,
  is no longer inhibited in the case of nested encapsulates with non-empty
  ~il[signature]s.  The following example illustrates the other change, as
  discussed below.
  ~bv[]
  (encapsulate
   ((f (x) t))
   (local (defun f (x) x))
   (defun g (x) (cons x (f x)))
   (defun h (x) (g x))
   (defthm h-is-f (equal (car (h x)) x)))
  ~ev[]
  Previously, the ~il[constraint] on ~c[f] and ~c[h] was essentially the
  conjunction of the definition of ~c[h] and the theorem ~c[h-is-f].  Now, the
  definition of ~c[g] is conjoined as well; moreover, ~c[g] receives the same
  ~il[constraint] as do ~c[f] and ~c[h], where previously ~c[g] was only
  constrained by its definition.  While we are not aware of a soundness bug
  caused by the previous approach, the new approach follows more precisely
  the intended notion of ~il[constraint].

  The use of ~ilc[trace$] (or ~ilc[trace!]) option ~c[:multiplicity] had been
  required when option ~c[:native] was supplied.  This is no longer the case.
  Also, a bug has been fixed that had prevented ~c[:multiplicity] from working
  properly in GCL and Allegro CL.

  Several errors have been eliminated that formerly occurred when the
  constraints for a function symbol were unknown because it was constrained
  using a dependent clause-processor (~pl[define-trusted-clause-processor].
  Now, it is assumed that the ~c[supporters] argument in a
  ~ilc[define-trusted-clause-processor] event is such that every ancestor of
  any function symbol constrained by the ``promised encapsulate'' of that event
  among, or ancestral in, those ~c[supporters].  Thanks to Sol Swords, Sandip
  Ray, and Jared Davis for helpful discussions.

  The notion of ~il[constraint] for functions introduced by ~ilc[defun] has
  been modified slightly.  No longer do we remove from the body of the
  definition calls of so-called ``guard-holders'': ~ilc[prog2$],
  ~ilc[must-be-equal], ~il[ec-call], and ~il[mv-list], and uses of
  ~c[the-error] generated by ~ilc[the].  Also, we now expand calls of
  ~c[the-error] with the same aggressive heuristics applied to a number of
  other functions (technically, adding it to the list
  ~c[*expandable-boot-strap-non-rec-fns*]).

  ~st[NEW FEATURES]

  A new event, ~ilc[defattach], allows evaluation of calls of constrained
  (~ilc[encapsulate]d) functions.  In particular, users can now, in principle,
  soundly modify ACL2 source code; please feel free to contact the ACL2
  implementors if you are interested in doing so.  ~l[defattach].

  Eric Smith has noticed that if you exit the ~il[break-rewrite] loop using
  ~c[:]~ilc[a!] during an ~ilc[ld] of a file, then all changes to the logical
  ~il[world] are discarded that were made during that call of ~ilc[ld].  A new
  utility, ~c[:]~ilc[p!], pops just one level instead, and avoids discarding
  that work.  (This change is related to an item above, ``The default behavior
  of ~ilc[ld] has been changed.'')  Thanks to Eric for pointing out this issue.

  New function ~ilc[mv-list] is the identity function logically, but converts
  multiple values to lists.  The first argument is the number of values, so an
  example form is as follows, where ~c[foo] returns three values:
  ~c[(mv-list 3 (foo x y))].  Thanks to Sol Swords for requesting this
  feature and for reporting a soundness bug in one of our preliminary
  implementations.

  A new ~ilc[state] global variable, ~c[host-lisp], has as its value a keyword
  whose value depends on the underlying Common Lisp implementation.  Use
  ~c[(@ host-lisp)] to see its value.

  It is now possible to write ~il[documentation] for HTML without error when
  there are links to nonexistent documentation topics.  See the comments in
  macro ~c[acl2::write-html-file] at the end of file
  ~c[doc/write-acl2-html.lisp].  When there are such errors, they should be
  easier to understand than previously.  Thanks to Alan Dunn for providing the
  initial modifications.

  It is now possible to inhibit specified parts of the Summary printed at the
  conclusion of an event.  ~l[set-inhibited-summary-types].  Also
  ~pl[with-output], in particular the discussion of the new ~c[:summary]
  keyword.  Thanks to Sol Swords for requesting more control over the Summary.

  A new ~c[:]~ilc[hints] keyword, ~c[:case-split-limitations], can override the
  default case-split-limitations settings (~pl[set-case-split-limitations]) in
  the simplifier.  Thanks to Ian Johnson for requesting this addition and
  providing an initial implementation.

  It is now possible to defer and avoid some ttag-related output;
  ~pl[set-deferred-ttag-notes].  Thanks to Jared Davis for requesting less
  verbosity from ttag-related output.

  A new ~il[command], ~c[:]~ilc[pl2], allows you to restrict the rewrite rules
  printed that apply to a given term.  ~l[pl2].  Thanks to Robert Krug for
  requesting such a capability.

  ACL2 now provides a utility for canonicalizing filenames, so that soft links
  are resolved; ~pl[canonical-pathname].  Moreover, ACL2 uses this utility in
  its own sources, which can eliminate some issues.  In particular,
  ~ilc[include-book] with argument ~c[:ttags :all] no longer breaks when given
  a book name differing from the book name that was used at certification time;
  thanks to Sol Swords for reporting that problem.  Also, certain errors
  have been eliminated involving the combination of packages in the
  certification world and trust tags; thanks to Jared Davis for sending an
  example of that problem.

  You can now suppress or enable guard-checking for an individual form;
  ~pl[with-guard-checking].  Thanks to Sol Swords for requesting this feature.

  The ~ilc[walkabout] utility has been documented (thanks to Rob Sumners for
  suggesting this documentation).  This utility can make it easy to explore a
  large ~c[cons] tree.  New interactive commands ~c[(pp n)] and
  ~c[(pp print-level print-length)] have been added to restrict how much of the
  current object is displayed.  ~l[walkabout].

  Rules of class ~c[:]~ilc[type-prescription] may now be provided a
  ~c[:backchain-limit-lst] keyword.  The default behavior is unchanged, but now
  ~ilc[type-set] is sensitive not only to the new
  ~c[:backchain-limit-lst] of a ~c[:]~ilc[type-prescription] rule (if
  supplied) but to the ~ilc[default-backchain-limit] of the current logical
  ~il[world].  Setting of backchain-limits can now specify either the new
  (type-set) limit or the old limit (for rewriting);
  ~pl[set-default-backchain-limit] and ~pl[set-backchain-limit].  Moreover, the
  functions ~ilc[default-backchain-limit] and ~ilc[backchain-limit] now take a
  second argument of ~c[:ts] or ~c[:rewrite] to specify which backchain-limit
  is desired.

  ~st[HEURISTIC IMPROVEMENTS]

  The so-called ``too-many-ifs'' heuristic has been modified.  Such a heuristic
  has been employed in ACL2 (and previous Boyer-Moore provers) for many years,
  in order to limit the introduction of calls of ~ilc[IF] by non-recursive
  functions.  Most users need not be concerned with this change, but two proofs
  in the regression suite (out of thousands) needed trivial adjustment, so user
  proofs could need tweaking.  In one application, this modification sped up
  proofs by 15%; but the change in runtime for the regression suite is
  negligible, so such speedups may vary.  Thanks to Sol Swords for providing a
  test from ACL2 runs at Centaur Technology, which was useful in re-tuning this
  heuristic.

  Guard proof obligations could have size quadratic in the number of clauses in
  a ~ilc[case] statement.  This inefficiency has been removed with a change
  that eliminates a hypothesis of the form ~c[(not (eql term constant))] when
  there is already a stronger hypothesis, equating the same term with a
  different constant.  Thanks to Sol Swords for bringing this problem to our
  attention and suggesting an alternate approach to solving it, which we may
  consider in the future if related efficiency problems persist.

  We adjusted the heuristics for determining induction schemes in the presence
  of ~il[ruler-extenders], when handling calls of a function symbol that is a
  ruler-extender, in either of two cases: either the function takes only one
  argument; or the function is ~ilc[prog2$] or ~ilc[ec-call], and the first
  argument contains no recursive call.  These cases are treated more directly
  as though the ruler-extender call is replaced by the unique (in the case of
  ~ilc[prog2$] and ~ilc[ec-call], the second) argument.

  A new ~c[:]~ilc[type-prescription] rule, ~c[true-listp-append], has been
  added:
  ~bv[]
  (implies (true-listp b)
           (true-listp (append a b)))
  ~ev[]
  If you are interested in the motivation for adding this rule, see comments in
  ~c[true-listp-append] in ACL2 source file ~c[axioms.lisp].

  The use of ~c[:forward-chaining] lemmas has been improved slightly.  In
  previous versions, a conclusion derived by forward chaining was discarded if
  it was derivable by type-set reasoning, since it was ``already provable.''
  But this heuristic prevented the conclusion from triggering further forward
  chaining.  This has been fixed.  Thanks to Dave Greve for pointing out this
  problem.

  The fundamental utility that turns an ~c[IF] expression into a set of clauses
  has been optimized to better handle tests of the form ~c[(equal x 'constant)]
  and their negations.  This eliminates an exponential explosion in large case
  analyses.  But it comes at the inconveience of sometimes reordering the
  clauses produced.  The latter aspect of this change may require you to change
  some Subgoal numbers in proof hints.  We apologize for the inconvenience.

  Certification can now run faster (specifically, the compilation phase) for
  books with very large structures generated by ~ilc[make-event], when there is
  significant sharing of substructure, because of a custom optimization of the
  Lisp reader.  Thanks to Sol Swords for bringing this efficiency issue to our
  attention.

  Jared Davis reported inefficiency in certain ~ilc[make-event] evaluation due
  to a potentially expensive ``bad lisp object'' check on the expansion
  produced by the ~c[make-event].  This check has been eliminated except in the
  case that the expansion introduces packages (for example, by including a book
  during the expansion phase that introduces packages).  Thanks to Jared for
  providing a helpful example.

  The application of rules of class ~c[:]~ilc[induction] had the potential to
  loop (as commented in ACL2 source function ~c[apply-induction-rule]).  This
  has been fixed.  Thanks to Daron Vroon and Pete Manolios for sending nice
  examples causing the loop.

  Heuristics have been tweaked so that false goals may be simplified to ~c[nil]
  that had formerly been left unchanged by simplification, perhaps resulting in
  useless and distracting proofs by induction.  Thanks to Pete Manolios for
  pointing out this issue by sending the following example:
  ~c[(thm (<= (+ 1 (acl2-count x)) 0))].  (Technical explanation: When every
  literal in a clause simplifies to ~c[nil], even though we might not normally
  delete one or more such literals, we will replace the entire clause by the
  false clause.)

  Improved the efficiency of the built-in function, ~ilc[take].  Thanks to Bob
  Boyer for suggesting this improvement.

  ACL2 can now use evaluation to relieve hypotheses when applying
  ~c[:]~ilc[type-prescription] rules.  Thanks to Peter Dillinger and Dave Greve
  for requesting this enhancement, and to Robert Krug for a relevant discussion
  long ago.

  Evaluation has been sped up during theorems for calls of ~ilc[mv-let], by
  avoiding repeated evaluation of the expression to which its variables are
  bound.  Thanks to Sol Swords for requesting this improvement and sending an
  illustrative example.

  Modified a heuristic to avoid the opening up non-recursive function calls on
  calls of ~ilc[hide] involving ~ilc[if]-expressions.  For example, the
  ~ilc[thm] form below is now admitted
  ~bv[]
  (defun bar (x)
    (cons x x))
  (thm (equal (bar (hide (if a b c)))
       (cons (hide (if a b c)) (hide (if a b c)))))
  ~ev[]

  ~st[BUG FIXES]

  Fixed a soundness bug in destructor elimination, which was preventing some
  cases from being generated.  Thanks to Eric Smith for reporting this bug and
  sending a helpful example.  (Technical detail: the fixes were in ACL2 source
  functions ~c[apply-instantiated-elim-rule] and
  ~c[eliminate-destructors-clause1], and comments in the former contain Eric's
  example.)

  Fixed a bug that supported a proof of ~c[nil] by exploiting the fact that
  ~il[portcullis] ~il[command]s were not included in check-sum computations in
  a book's ~il[certificate].  For such a proof of ~c[nil], see the relevant
  comment in the ACL2 source file ~c[ld.lisp] under
  ~c[(deflabel note-4-0 ...)].

  Changed the implementation of ~ilc[add-include-book-dir].  The previous
  implementation could allow relative pathnames to be stored in the
  ~il[portcullis] ~il[command]s of ~il[certificate]s of ~il[books], which
  perhaps could lead to unsoundness (though we did not try to exploit this to
  prove ~c[nil]).  Thanks to Jared Davis for reporting a bug in our first new
  implementation.  An additional change to both ~ilc[add-include-book-dir] and
  ~ilc[delete-include-book-dir] is that these now work in raw-mode
  (~pl[set-raw-mode]).  (Thanks to Dave Greve for suggesting a reduction in the
  warnings we produced related to raw-mode.)  Note that it is no longer
  permitted to make a direct call of the form
  ~c[(table acl2-defaults-table :include-book-dir-alist ...)]; use
  ~ilc[add-include-book-dir] instead.

  Fixed a soundness bug related to ~ilc[xargs] keyword ~c[:non-executable].
  New macros, ~c[defun-nx] and ~c[defund-nx], have been provided for declaring
  functions to be non-executable; ~pl[defun-nx].  While we expect this bug to
  occur only rarely if at all in practice, the following example shows how it
  could be evoked.
  ~bv[]
    ;;;;;;;;;;;;;;;;;;;;
    ;;; Book sub.lisp
    ;;;;;;;;;;;;;;;;;;;;
    (in-package \"ACL2\")
    (defun f ()
      (declare (xargs :guard t
                      :non-executable t))
      (mv-let (a b c)
              (mv 3 4)
              (declare (ignore a b))
              c))
    (defun g ()
      (declare (xargs :guard t))
      (prog2$ (mv-let (x y z)
                      (mv 2 3 4)
                      (declare (ignore x y z))
                      nil)
              (f)))
    (defthm g-nil
      (equal (g) nil)
      :hints ((\"Goal\" :in-theory (disable (f))))
      :rule-classes nil)
    ;;;;;;;;;;;;;;;;;;;;
    ;;; Book top.lisp
    ;;;;;;;;;;;;;;;;;;;;
    (in-package \"ACL2\")
    (include-book \"sub\")
    (defthm contradiction
      nil
      :hints ((\"Goal\" :use g-nil))
      :rule-classes nil)
  ~ev[]

  The modification described above pertaining to ~ilc[defun-nx] also prevents
  execution of non-executable functions that have been ~il[trace]d.  The
  following example illustrates the problem; now, the following ~ilc[defun] of
  ~c[g] is illegal, and the problem disappears if ~ilc[defun-nx] is used
  instead.
  ~bv[]
  (defun g (x) ; Use defun-nx to avoid an error after Version_3.6.1.
    (declare (xargs :guard t :non-executable t))
    x)
  (g 3) ; causes error, as expected
  (trace$ g)
  (g 3) ; returned 3 before the bug fix; after fix, causes error as expected
  ~ev[]

  A hard error was possible when attempting to include an uncertified book
  containing ~il[events] of the form ~c[(make-event '(local ...))].  This has
  been fixed.  Thanks to Sol Swords for bringing this issue to our attention.

  Fixed a bug in the heuristic improvement described for Version_3.6
  (~pl[note-3-6]) as ``We simplified induction schemes....''  The bug had
  prevented, in unusual cases such as the following (notice the impossible
  case), a proof by induction.
  ~bv[]
  (defun foo (a x)
    (and (consp x)
         (case a
           (0 (foo (car x) (cdr x)))
           (1 (foo (cdr x) (car x)))
           (0 (foo a (cons x x))))))
  (in-theory (disable (:type-prescription foo)))
  (thm (atom (foo a x)))
  ~ev[]

  Macro ~ilc[cw-gstack] did not work with an ~c[:evisc-tuple] argument.  This
  has been fixed by changing ~c[cw-gstack] so that it now evaluates its
  arguments.  Thanks to Sol Swords for bringing this bug to our attention.

  Fixed a bug in ~c[:]~ilc[pso] during the printing of failure messages for
  termination proofs.

  Fixed a bug in the handling of ~c[#.] (~pl[sharp-dot-reader]).  Thanks to Bob
  Boyer for bringing this bug to our attention.

  Replaced a hard Lisp error with a clean error, in certain cases that a
  ~c[:]~ilc[hints] value is erroneously supplied as a non-~c[nil] atom.
  Example: ~c[(thm (equal x x) :hints 3)].

  Fixed a bug in the interaction of function tracing with conversion of a
  function from ~c[:]~ilc[program] to ~c[:]~ilc[logic] mode.  The following
  example illustrates what had been wrong.
  ~bv[]
  (defun f (x)
    (declare (xargs :mode :program))
    (car x))
  (f 3) ; raw Lisp hard error
  (trace$ f)
  (f 3) ; raw Lisp hard error (still)
  (defun f (x) (car x)) ; upgrade f to :logic mode
  (f 3) ; clean guard violation; f is no longer traced
  (trace$) ; uh oh - f is shown as traced
  (untrace$ f)
  (f 3) ; OUCH: hard Lisp error because old :program mode definition of
        ; the executable counterpart (sometimes called *1*f) was restored!
  ~ev[]

  Made a fix so that when building ACL2 with `make' option ~c[ACL2_SAFETY=3],
  there will no longer be any safety-0 compiled code generated.  Thanks to Gary
  Byers for bringing this bug to our attention.

  Fixed a bug in the handling of ~il[override-hints] that generate custom
  keyword hints (~pl[custom-keyword-hints]) involving the variable
  ~c[stable-under-simplificationp].  Thanks to Ian Johnson for bringing this
  bug to our attention with explanation that included a helpful example,
  included as comment in the ACL2 source code for function
  ~c[apply-override-hint].

  The ~c[saved_acl2] script in CLISP could contain unexpected characters where
  simple newlines were expected.  Dave Greve found this in a Cygwin environment
  on Windows.  Thanks to Dave for reporting this bug and experimenting with a
  fix, and thanks to the CLISP folks for providing helpful information.

  Fixed a bug that could make ~c[:]~ilc[oops] cause an error.  Also, the
  ~ilc[oops] command can no longer take you back before a
  ~ilc[reset-prehistory] event.

  (GCL only) Fixed a bug that could occur when calling ~c[trace] in raw Lisp in
  GCL.

  Proof summaries have been improved, so that they account for ~il[rune]s used
  in rewriting that takes place when generating goals to be proved in a forcing
  round.  Thanks to Jared Davis for sending us an example illustrating this
  issue.

  Fixed a bug that (at least in CCL) could put extra backslashes (`~c[\\]') in
  a pathname that ACL2 writes out to the executable script created by a build.
  Thanks to Gary Byers for explaining that the CCL behavior is legal (for our
  previous use of Common Lisp function ~c[merge-pathnames]).

  We closed some holes in the handling of trust tags (also known as ``ttags'';
  ~pl[defttag]) by ~ilc[include-book].  The following example illustrates this
  rather subtle situation.  Consider the following book.
  ~bv[]
  (in-package \"ACL2\")
  (make-event
   (er-progn
    (encapsulate
     ()
     (defttag :foo)
     (value-triple \"Imagine something bad here!\"))
    (value '(value-triple :some-value)))
   :check-expansion t)
  ~ev[]
  Formerly, the following commands succeeded.
  ~bv[]
  (certify-book \"test3\" 0 t :ttags :all)
  :u
  (include-book \"test3\" :ttags nil)
  ~ev[]
  But because of ~ilc[make-event] keyword argument ~c[:check-expansion t], we
  know that the event ~c[(defttag :foo)] is evaluated by the above
  ~ilc[include-book] form, and hence the ~c[:ttags] argument of
  ~c[include-book], above, should have specified ~c[:foo].  The problem was
  that ~ilc[defttag] forms evaluated during ~ilc[make-event] expansion did not
  contribute to the trust tag information stored in the book's
  ~il[certificate].  Note: Because of this change, one should avoid using
  ~ilc[make-event] with ~c[:check-expansion t] when the expansion would
  introduce a ~ilc[defttag] event during ~ilc[include-book] but not
  ~ilc[certify-book] time.  For an example illustrating this issue,
  ~pl[make-event-details], specifically the new version of the section labeled
  ``A note on ttags'' at the end of that ~il[documentation] topic.

  Closed a small loophole that had the potential, in rare circumstances, to
  violate atomicity of under-the-hood updates for ACL2 ~il[arrays].

  The following example was formerly allowed, but resulted in a guard-verified
  function (here, ~c[g]) whose guard proof obligation is not a theorem outside
  the ~ilc[encapsulate] event.  We now disallow ~il[guard] verification for
  functions introduced non-~il[local]ly inside an ~ilc[encapsulate] event unless
  we determine that the proof obligations hold outside the ~ilc[encapsulate]
  event as well.
  ~bv[]
  (encapsulate
   ((f (x) t))
   (local (defun f (x) (declare (xargs :guard t)) (consp x)))
   ;; ERROR!
   (defun g (x)
     (declare (xargs :guard (f x)))
     (car x)))
  ~ev[]

  The use of ~c[:]~ilc[comp] on ~il[stobj] functions had potentially caused a
  hard Lisp error; for example, this could occur when ~c[(defstobj foo fld)]
  was followed by ~c[:comp foop].  This has been fixed.

  Fixed a bug that could cause a raw Lisp error when the first argument of
  ~ilc[with-local-stobj] is not a symbol.

  It had been possible to use the reserved keyword :computed-hints-replacement
  as the name of a custom keyword hint (~pl[custom-keyword-hints]).  This has
  been fixed.  Thanks to Dave Greve, who pointed out a confusing hint error
  message (which has also been fixed) that led us to this issue.

  Fixed a bug that could cause a hard Lisp error, instead of a graceful ACL2
  error, if keyword ~c[:backchain-limit-lst] in a rule class is given a cons
  that is not a true list, such as ~c[(1 . 1)].

  Eliminated an error that could occur when redefining a function as a macro
  and then compiling, as in the example below.
  ~bv[]
  (defun foo (x) x)
  :redef!
  (defmacro foo (x) x)
  :comp t
  ~ev[]
  Thanks to Eric Smith for sending the above example in his bug report.

  Fixed a bug that could result in an assertion when a ~il[clause-processor]
  causes an error.

  ~st[NEW AND UPDATED BOOKS AND RELATED INFRASTRUCTURE]

  See ~url[http://code.google.com/p/acl2-books/source/list] for a record of
  books changed or added since the preceding release, with log entries.

  We note in particular the new ~c[system/] directory, which begins to specify
  ACL2 system code in anticipation of opening the architecture of ACL2
  (~pl[defattach] for a relevant tool).  Some system functions were changed
  slightly (but with the expectation of not generally affecting ACL2 behavior)
  in support of the development of this directory.  Those interested in
  contributing to further such efforts are invited to contact the ACL2
  implementors.

  New utilities have been provided for certifying most of the distributed books
  with more `make'-level parallelism.  For example, we have obtained close to
  a 12x reduction in time by using `~c[make -j 24 regression-fast]' on a
  24-processor machine.  For more information see ~c[books/make-targets], or to
  include the ~c[books/workshops] in the regression run, see
  ~c[books/regression-targets].  Thanks to Sol Swords for providing these nice
  utilities.

  The top-level makefile, ~c[GNUmakefile], has been fixed so that the build
  processes (which are inherently sequential) will ignore the ~c[-j] option of
  `make'.  Note that regressions can still, however, be done in parallel, as
  the ~c[-j] option will be passed automatically to the appropriate `make'
  command.

  ~st[EMACS SUPPORT]

  ~st[EXPERIMENTAL VERSIONS]

  The HONS version, supported primarily by Bob Boyer and Warren Hunt
  (~pl[hons-and-memoization]), has undergone numerous improvements.  For
  example, keyword argument ~c[:FORGET] is now supported when calling
  ~ilc[memoize] from within the ACL2 loop, and system function ~c[worse-than]
  is ~il[memoize]d with the ~c[:condition] that both terms are function
  applications (clearing the memo-table after each prover invocation).  Thanks
  to Jared Davis and Sol Swords for investigating the memoization of
  ~c[worse-than], and with suitable ~c[condition].  Thanks also to Jared Davis
  for contributing structural modifications to the implementation of
  ~ilc[hons].

  David Rager contributed modifications to the parallel version
  (~pl[parallelism]), which include taking advantage of atomic increments
  available at least since Version 1.0.21 of SBCL and Version 1.3 of CCL.

  ~/~/")

(deflabel |NOTE-4-0(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  4.0(r) (July, 2010) Notes~/

  ~/

  Please ~pl[note-4-0] for changes in Version  4.0 of ACL2.

  ~/
  ")

(deflabel note-4-0-wormhole-changes
  :Doc
  ":Doc-Section note-4-0

  how to convert calls of wormhole for Version  4.0~/

  Here we describe how to convert an ``old-style'' call of ~ilc[wormhole] ~-[]
  that is, a call suitable for ACL2 versions preceding  4.0 ~-[] in which the
  ~c[pseudo-flag] was ~c[t].  In order to convert such a call
  ~bv[]
  (wormhole t 'name input form ...)
  ~ev[]
  to a new-style call, the following steps must be carried out.  Note that the
  wormhole name must always be quoted now.

  First, eliminate the first argument, ~c[t], and add a new second argument
  that is the quoted lambda expression
  ~bv[]
  '(lambda (whs) (set-wormhole-entry-code whs :ENTER))
  ~ev[]
  Setting the entry code to ~c[:ENTER] is not necessary if you maintain the
  invariant (after initialization) that it is always ~c[:ENTER].  In that case,
  the simpler quoted lambda will suffice:
  ~bv[]
  '(lambda (whs) whs)
  ~ev[]

  Second, change the ~c[form] argument so that instead of talking about the
  state-global variable ~c[wormhole-output] it talks about the state-global
  variable ~c[wormhole-status].  Look for
  ~c[(@ wormhole-output)], ~c[(assign wormhole-output ...)],
  ~c[(f-get-global 'wormhole-output ...)] and
  ~c[(f-put-global 'wormhole-output ...)] in ~c[form] and replace them with
  expressions involving ~c[wormhole-status].

  However, remember that the old data stored in ~c[wormhole-output] is now
  in the ~c[wormhole-data] component of the ~c[wormhole-status].  Thus, for
  example, an old use of ~c[(@ wormhole-output)] will typically be replaced
  by ~c[(wormhole-data (@ wormhole-status))] and an old use of
  ~c[(assign wormhole-output ...)] will typically be replaced by
  ~bv[]
  (assign wormhole-status (set-wormhole-data (@ wormhole-status) ...))
  ~ev[]

  In summary, an old-style call like
  ~bv[]
  (wormhole t 'name
            input
            '(...1 (@ wormhole-output) ...2
              ...3 (assign wormhole-output ...4) ...5)
            ...6)
  ~ev[]
  can become
  ~bv[]
  (wormhole 'name
            '(lambda (whs) (set-wormhole-entry-code whs :ENTER))
            input
            '(...1 (wormhole-data (@ wormhole-status)) ...2
              ...3 (assign wormhole-status
                          (set-wormhole-data (@ wormhole-status)
                                             ...4) ...5)
            ...6)
  ~ev[]

  In any case, and especially if your ~c[wormhole] call had a ~c[pseudo-flag]
  other than ~c[t], we recommend that you ~pl[wormhole].~/~/")

(deflabel note-4-1

; Eliminated state global 'translate-error-depth (unused, as noticed by David
; Rager).

; Fixed the definition of gv to pass the appropriate stobjs-in to
; throw-raw-ev-fncall.  Before this fix, the following caused a guard violation
; for remove-stobjs-in-by-position.  (After this fix, a different error
; occurred because of safe-mode, but that is irrelevant to this discussion.)
;   (assign safe-mode 3)
;   (car 3)
;   (print-gv)

; Made trivial syntactic simplification in #-acl2-loop-only definition of
; with-prover-time-limit.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  4.1 (September, 2010) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  4.1 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, distributed books, Emacs support, and experimental
  versions.  Each change is described in just one category, though of course
  many changes could be placed in more than one category.

  ~st[CHANGES TO EXISTING FEATURES]

  The ~il[guard] associated with calls of the macro, ~ilc[search], has been
  weakened so that now, given strings are no longer restricted to contain only
  standard characters unless the ~c[:test] argument is ~ilc[char-equal].

  Modified the writing of ``hidden ~ilc[defpkg]'' forms into ~il[certificate]
  files (~pl[hidden-defpkg]), to support moving certificate files for
  distributed books, as is done by ACL2s (~pl[acl2-sedan]) and Debian releases
  of ACL2.  Thanks to Camm Maguire for reporting a problem with Debian releases
  of ACL2 that led to this change.

  Expanded the constant ~c[*acl2-exports*] by adding ~c[intersection-equal] to
  the list.  Thanks to Jared Davis for requesting this change.

  The ~c[:]~ilc[comp] utility now compiles functions that have code
  conditionalized for raw Lisp only (presumably because a trust tag was active
  when they were defined).  Previously, this was not the case when ~c[:comp]
  was applied to more than a single function symbol.

  ~st[NEW FEATURES]

  A new macro, ~ilc[top-level], allows evaluation directly in the top level
  loop for forms that normally need to be evaluated inside function bodies,
  such as ~ilc[with-local-stobj].  ~l[top-level].  Thanks to Jared Davis for
  requesting such a utility.

  Added ~ilc[count], a Common Lisp function, to ACL2.  In support of that
  addition, also added rewrite rule ~c[eqlablep-nth].

  ~st[HEURISTIC IMPROVEMENTS]

  [None this time.]

  ~st[BUG FIXES]

  We fixed a soundness bug that could occur when a function that returns
  multiple values that is called in its own guard.  Thanks to Sol Swords for
  reporting this bug and sending a small self-contained example, which is
  included in a comment in the function ~c[chk-acceptable-defuns1] in ACL2
  source file ~c[defuns.lisp].

  It was possible to cause an error when giving theory hints during
  redefinition of functions.  This has been fixed.  Thanks to Ian Johnson for
  sending an example that nicely illustrated this problem.

  Fixed system function ~c[io?] for the case that formal parameter ~c[commentp]
  is ~c[t] and ~c[vars] is non-empty.  Thanks to David Rager for bringing to
  our attention the fact that ~c[io?] was broken for such a combination of
  parameters.

  Not exactly a bug fix, but: ~ilc[defun-sk] was breaking when a
  ~c[:]~ilc[guard] is specified, so we have improved the documentation
  (~pl[defun-sk]) to explain how to provide verified guards for a function
  introduced by ~ilc[defun-sk].  Thanks to Jared Davis for bringing this issue
  to our attention.

  Made a fix to the handling of interrupts, which in rare cases might have left
  one in a state where all subsequent proof attempts were labeled as
  ``Aborting due to an interrupt''.

  Fixed ~c[:]~ilc[pso] and related utilities, so that when proof output is
  redirected to a file, all summary output goes to that file rather than to the
  terminal.

  (GCL on Windows only) Removed an inappropriate check, resulting in an error
  about ``pathname-device,'' that could prevent Windows GCL builds of ACL2.
  Thanks to Camm Maguire for reporting this problem and a helpful discussion.

  (Windows only) Modified the computation of canonical pathnames to avoid
  issues of case-insensitivity, in particular for the drive (e.g., ~c[\"C:\"]
  vs. ~c[\"c:\"]).  Thanks to Harsh Raju Chamarthi for reporting this issue and
  helping with its debugging.

  (Windows only) The value of ~c[(@ distributed-books-dir)] no longer will be
  missing the Windows drive prefix, for example, ~c[\"C:\"].  Thanks to Harsh
  Raju Chamarthi for reporting this issue and helping with its debugging.

  ~st[NEW AND UPDATED BOOKS AND RELATED INFRASTRUCTURE]

  See ~url[http://code.google.com/p/acl2-books/source/list] for a record of
  books changed or added since the preceding release, with log entries.

  Modified ~c[books/Makefile-generic] by adding a new ~c[BOOKS_SKIP_COMP]
  variable, which is used in ~c[Makefile]s in some subdirectories of
  ~c[books/], in order to avoid errors when compiling certified books for
  multiple Lisps.

  ~st[EMACS SUPPORT]

  Distributed file ~c[emacs/emacs-acl2.el] has been modified so that the forms
  ~c[control-t e] and ~c[control-t control-e] now pick up package markers
  (~pl[sharp-bang-reader]), in the following sense: if the top-level form is
  preceded by a line starting with ~c[#!], then that line is included in the
  inserted string.  Thanks to Jared Davis for suggesting this enhancement and
  providing a preliminary implementation.

  ~st[EXPERIMENTAL VERSIONS]

  For the ~c[HONS] version there have been some changes to ~ilc[memoize]:
  ~bq[]
  ~ilc[Memoize] accepts a new keyword, ~c[:recursive], that is a synonym for
  the existing keyword ~c[:inline].  Thanks to Sol Swords for requesting this
  addition.  Moreover, it is now enforced that these keywords have Boolean
  values.

  ~ilc[Memoize] may now be called on ~c[:]~ilc[program] mode functions.  Thanks
  to Sol Swords for requesting this enhancement.

  A bug has been fixed.  Now, if ~ilc[memoize] is called with a
  ~c[:condition-fn] (with value other than ~c[nil] or ~c[t]), then the
  ~il[guard] of the memoized function and the ~c[:condition-fn] must be the
  same.  Previously, one could exploit the lack of such a check to get a hard
  Lisp error, for example as follows.
  ~bv[]
       (defun f (x) (declare (xargs :guard t)) x)
       (defun cf (x) (declare (xargs :guard (consp x))) (car x))
       (memoize 'f :condition-fn 'cf)
       (f 3)
  ~ev[]

  Memoization is now illegal for built-in functions that use underlying raw
  Lisp in their implementations.  To see why, consider the form ~c[(gc$)],
  which is a macro call expanding to ~c[(gc$-fn nil)].  Previously, after
  evaluation of ~c[(memoize 'gc$-fn)], a call of ~c[gc$] would no longer call
  the garbage collector, which had been invoked by raw Lisp code.  Now,
  evaluation of ~c[(memoize 'gc$-fn)] causes an error.~eq[]
  ~/~/")

(deflabel |NOTE-4-1(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  4.1(r) (September, 2010) Notes~/

  ~/

  Please ~pl[note-4-1] for changes in Version  4.1 of ACL2.

  ~/
  ")

(deflabel note-4-2

; Fixed a bug in ld-fn0 that was passing NIL as the second argument of a
; pprogn, and in that same area of code inserted a progn to obey the intended
; syntax of mv-let.  It's not clear that any of this actually caused problems,
; however.

; Modified (in function tilde-*-book-check-sums-phrase1) the printing of
; include-book errors due to mismatch of sub-book's certificate with parent
; books' certificate, so that a full-book-name is used instead of a
; familiar-name.

; Added brief documentation for quote, following email from Sandip Ray.

; Eliminated inclp argument of functions in the translate11 nest.

; Made minor changes in include-book-fn1 that could conceivably affect handling
; of uncertified books with stale certificates.  But since this may be rare,
; and it doesn't seem important to invest time to come up with an example
; illustrating such a change in behavior, we merely leave this comment rather
; than adding to the :doc string below.

; Modified the implementation of print-indented-list by having it call new
; function print-indented-list-msg, which in turn is used directly in the new
; "error message for free variables" described in this :doc topic (in
; tilde-@-free-vars-phrase, called by chk-free-vars).

; Fixed bugs in :doc set-backchain-limit.

; A potential soundness hole was plugged in the proof-checker by making
; variable pc-assign untouchable.  But we don't mention this in the release
; notes proper because we have not been able to exploit this potential hole.

; Changed fmt-abbrev1 to print the message about :DOC set-iprint in column 0
; after a newline, because otherwise the new message printed immediately after
; evaluating (retrieve ...) looks odd.

; Introduced new function our-truename, which we use in place of truename.
; This was done in support of the item below mentioning "truename", about
; "certain errors in including books".

; Fixed documentation and error message for the case that only some functions
; in a mutual-recursion are non-executable.

; Improved error message for forms such as (defattach f g :hints ...), in which
; the first argument is a symbol but there are more than two arguments.  A lot
; of defattach code was changed to support the use of :program mode functions
; and a few other small changes were made in the process, e.g., world global
; 'proved-functional-instances-alist isn't (needlessly) set to its existing
; value.

; Changed the names of the arrays stored in enabled-structures that are created
; by hints during the waterfall.  We did this with David Rager, in order to
; support parallel evaluation for the waterfall.

; Added source file boot-strap-pass-2.lisp, processed only during pass 2 of the
; boot-strap, which is useful for defattach.  Deleted obsolete function
; load-acl2-execution-environment, rather than figure out whether it should
; also load this new file (and perhaps other pass-2 files as well).

; Split GNUmakefile target DOC into HTML, EMACS, and TEX targets (where EMACS
; replaces the old TEXINFO target).  Users should not generally need to build
; the :doc themselves, but with this split we make that possible even if they
; are missing the texi2dvi and dvips programs needed for building a .ps file
; (as provided by the TEX target).

; Function elide-locals-rec had an odd case for time$1, which we have replaced
; there by time$ along with a comment that this case seems irrelevant anyhow.

; Improved note-fns-in-form to do a more thorough check, for example diving
; into skip-proofs forms (e.g., open-output-channel had been missing from
; *primitive-logic-fns-with-raw-code*, but we hadn't caught that).

; Here is the example promised in the item below labeled: "Fixed a bug in which
; the wrong attachment could be made....".  A more subtle example is described
; in function install-for-add-trip-hcomp-build.
;
; ;;;;; file sub.lisp ;;;;;
; (in-package "ACL2")
; (defun sub-fn (x) x)
; ;;;;;;;;;;;;;;;;;;;;;;;;;
;
; ;;;;; file foo.lisp (note command for certification world) ;;;;;
; ; Portcullis command:
; ; (progn (defstub f (x) t) (defattach f identity))

; (in-package "ACL2")
; (include-book "sub")
; (defun g (x)
;   (declare (xargs :guard t))
;   (cons x x))
; (defattach f g)
; ;;;;;;;;;;;;;;;;;;;;;;;;;
;
; First certify both books, first submitting the above portcullis command in
; the case of foo.lisp; then delete the compiled file for sub.lisp; then start
; ACL2 and evaluate (include-book "foo"); and finally, evaluate the form (f 3).
; The result was 3 where it should have been (3 . 3).

; Functions print-call-history and maybe-print-call-history are now in :logic
; mode, guard-verified.

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  4.2 (January, 2011) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  4.2 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, distributed books, Emacs support, and experimental
  versions.  Each change is described in just one category, though of course
  many changes could be placed in more than one category.

  ~st[CHANGES TO EXISTING FEATURES]

  The ~ilc[accumulated-persistence] utility can now do finer-grained tracking,
  providing data for individual hypotheses and the conclusion of a rule.
  ~l[accumulated-persistence].  To try this out, evaluate the form
  ~c[(accumulated-persistence :all)]; then ~pl[accumulated-persistence] for a
  discussion of display options using ~c[show-accumulated-persistence].  Thanks
  to Dave Greve for suggesting this new capability and collaborating on its
  design and implementation.

  The ~ilc[defattach] utility now permits the use of ~c[:]~ilc[program] mode
  functions, though this requires the use of a trust tag (~pl[defttag]).
  ~l[defattach] and for discussion of the new capability, ~pl[defproxy],
  which explains how part of this change involves allowing ~c[:]~ilc[program]
  mode functions to be declared ~il[non-executable].

  Redefinition (~pl[ld-redefinition-action]) is no longer permitted for
  functions that have attachments (~pl[defattach]).  In such cases, the
  attachment must be removed first, e.g. with ~c[(defattach foo nil)].

  Made small changes to ~ilc[mv-nth] and ~ilc[defun-sk] in order to permit
  guard verification of functions introduced with more than one quantified
  variable in a ~ilc[defun-sk] form.  The change to ~ilc[mv-nth] is to weaken
  the ~il[guard] by eliminating the requirement that the second argument
  satisfy ~ilc[true-listp], and replacing the call of ~ilc[endp] in the
  definition body by a corresponding call of ~ilc[atom].  The new definition of
  ~ilc[mv-nth] is thus logically equivalent to the old definition, but with a
  weaker guard.  Thanks to Sol Swords for sending the following example, for
  which the final ~ilc[verify-guards] form had failed but now succeeds.
  ~bv[]
  (defstub foo (a b c) nil)
  (defun-sk forall-a-b-foo (c)
     (forall (a b) (foo a b c))
     :witness-dcls ((declare (Xargs :guard t
                                    :verify-guards nil))))
  (verify-guards forall-a-b-foo)
  ~ev[]

  The implementations of ~ilc[prog2$], ~ilc[time$],
  ~ilc[with-prover-time-limit], ~ilc[with-guard-checking], ~ilc[mbe] (and
  ~ilc[must-be-equal]), and ~ilc[ec-call] have changed.  See the discussion
  below of the new utility, ~ilc[return-last].  A consequence is that
  ~ilc[trace$] is explicitly disallowed for these and related symbols, which
  formerly could cause hard Lisp errors, because they are now macros.  Tracing
  of return-last is also disallowed.  Another consequence is that time$ now
  prints a more abbreviated message by default, but a version of the old
  behavior can be obtained with ~c[:mintime nil].

  The following utilities no longer print an observation about raw-mode
  transitions: ~c[set-raw-mode-on], ~ilc[set-raw-mode-on!], ~ilc[set-raw-mode],
  and ~c[set-raw-mode-off].  Thanks to Jared Davis for suggestion this change
  in the case of ~ilc[include-book] (which proved awkward to restrict to that
  case).

  The system function ~c[translate-and-test] now permits its ~c[LAMBDA] form to
  refer to the variable ~c[WORLD], which is bound to the current ACL2 logical
  ~il[world].

  Modified abort handling to avoid talking about an interrupt when the error
  was caused by a Lisp error rather than an interrupt.

  The value of the constant ~c[*acl2-exports*], which is still a list, has been
  extended significantly, though only with the addition of symbols that one
  might reasonably have expected all along to belong to this list.  A new
  distributed book, ~c[books/misc/check-acl2-exports.lisp], checks (at
  certification time) that no documented constant, macro, or function symbol in
  the ~c[\"ACL2\"] package has been accidentally omitted from
  ~c[*acl2-exports*].  Thanks to Dave Greve for helpful discussions related to
  this change.

  Improved the built-in `~c[untranslate]' functions to produce ~c[let*]
  expressions when appropriate (more to help with tools that call
  ~c[untranslate] and the like, than to help with proof output).

  The utility ~ilc[redo-flat] now works for ~ilc[certify-book] failures, just
  as it continues to work for failures of ~ilc[encapsulate] and ~ilc[progn].

  The following only affects users who use trust tags to add to list values of
  either of the ~ilc[state] global variables ~c[program-fns-with-raw-code] or
  ~c[logic-fns-with-raw-code].  For functions that belong to either of the
  above two lists, ~c[trace$] will supply a default value of ~c[:fncall] to
  keyword ~c[:notinline], to avoid discarding raw-Lisp code for the function.

  The ~il[guard] of the macro ~ilc[intern] has been strengthened so that its
  second argument may no longer be either the symbol
  ~c[*main-lisp-package-name*] or the string ~c[\"COMMON-LISP\"].  That change
  supports another change, namely that the following symbols in the
  ~c[\"COMMON-LISP\"] package are no longer allowed into ACL2: symbols in that
  package that are not members of the list constant
  ~c[*common-lisp-symbols-from-main-lisp-package*] yet are imported into the
  ~c[\"COMMON-LISP\"] package from another package.  ~l[pkg-imports] and
  ~pl[symbol-package-name].  To see why we made that change, consider for
  example the following theorem, which ACL2 was able to prove when the host
  Lisp is GCL.
  ~bv[]
  (let ((x \"ALLOCATE\") (y 'car))
    (implies (and (stringp x)
                  (symbolp y)
                  (equal (symbol-package-name y)
                         \"COMMON-LISP\"))
             (equal (symbol-package-name (intern-in-package-of-symbol x y))
                    \"SYSTEM\")))
  ~ev[]
  Now suppose that one includes a book with this theorem (with
  ~c[:]~ilc[rule-classes] ~c[nil]), using an ACL2 built on top of a different
  host Lisp, say CCL, that does not import the symbol ~c[SYSTEM::ALLOCATE] into
  the ~c[\"COMMON-LISP\"] package.  Then then one can prove ~c[nil] by giving
  this theorem as a ~c[:use] hint.

  The axioms introduced by ~ilc[defpkg] have changed.  See the discussion of
  ~ilc[pkg-imports] under ``NEW FEATURES'' below.

  The error message for free variables (e.g., in definition bodies and guards)
  now supplies additional information when there are governing IF conditions.
  Thanks to Jared Davis for requesting this enhancement and collaborating in
  its design.

  The command ~c[:]~ilc[redef-] now turns off redefinition.

  Improved proof output in the case of a ~c[:]~ilc[clause-processor] hint that
  proves the goal, so that the clause-processor function name is printed.

  The ~ilc[proof-checker] command `~c[then]' now stops at the first failure (if
  any).

  It is no longer permitted to submit definitions in ~c[:logic] mode for merely
  part of an existing ~ilc[mutual-recursion] event.  Such an action left the
  user in an odd state and seemed a potential soundness hole.

  The function ~ilc[break$] is now in ~c[:]~ilc[logic] mode.  Thanks to Jared
  Davis for requesting this enhancement.

  The macro ~ilc[verify-termination] now provides clearer output in the case
  that it is redundant.  More important perhaps, as a courtesy it now causes an
  error when applied to a constrained function, since presumably such an
  application was unintended (as the constrained function could never have been
  in ~c[:]~ilc[program] mode).  Note that if one desires different behavior,
  one can create one's own version of ~ilc[verify-termination] (but with a
  different name).

  Improved the ~il[guard]s for the following functions, often weakening them,
  to reflect more precisely the requirements for calling ~ilc[eq]:
  ~c[alist-difference-eq], ~c[intersection-eq], ~c[intersection1-eq],
  ~ilc[intersectp-eq], ~c[not-in-domain-eq], ~c[set-difference-assoc-eq],
  ~c[set-equalp-eq], and ~ilc[union-eq].  Thanks to Jared Davis for pointing
  out this issue for ~ilc[intersectp-eq].

  (CCL only) Made a change that can reduce the size of a compiled file produced
  by ~ilc[certify-book] when the host Lisp is CCL, by discarding source
  information (for example, discarding ~ilc[local] events).

  ~st[NEW FEATURES]

  See the discussion above about new statistics that can be gathered by the
  ~ilc[accumulated-persistence] utility.

  A new hint, ~c[:]~ilc[instructions], allows use of the ~il[proof-checker] at
  the level of ~il[hints] to the prover.  Thanks to Pete Manolios for
  requesting this feature (in 2001!).  ~l[instructions].

  (For system hackers) There are new versions of system functions
  ~c[translate1] and ~c[translate], namely ~c[translate1-cmp] and
  ~c[translate-cmp] respectively, that do not take or return ~ilc[state].  See
  the Essay on Context-message Pairs for relevant information.  Thanks to David
  Rager for collaborating on this enhancement.

  A new utility, ~ilc[return-last], is now the unique ACL2 function that can
  pass back a multiple value result from one of its arguments.  Thus, now the
  following are macros whose calls ultimately expand to calls of
  ~ilc[return-last]: ~ilc[prog2$], ~ilc[time$], ~ilc[with-prover-time-limit],
  ~ilc[with-guard-checking], ~ilc[mbe] (and ~ilc[must-be-equal]), and
  ~ilc[ec-call].  With an active trust tag, an advanced user can now write code
  that has side effects in raw Lisp; ~pl[return-last].  Thanks to Jared Davis
  for requesting this feature.

  A new function, ~ilc[pkg-imports], specifies the list of symbols imported
  into a given package.  The axioms for ~ilc[defpkg] have been strengthened,
  taking advantage of this function.  Now one can prove theorems using ACL2
  that we believe could not previously be proved using ACL2, for example the
  following.
  ~bv[]
  (equal (symbol-package-name (intern-in-package-of-symbol str t))
         (symbol-package-name (intern-in-package-of-symbol str nil)))
  ~ev[]
  Thanks to Sol Swords for a helpful report, which included the example above.
  ~l[pkg-imports] and ~pl[defpkg].

  Added function ~ilc[no-duplicatesp-eq].

  Added a new hint keyword, ~c[:]~ilc[backchain-limit-rw], to control the level
  of backchaining for ~il[rewrite], ~il[meta], and ~il[linear] rules.  This
  overrides, for the current goal and (as with ~c[:]~ilc[in-theory] hints)
  descendent goals, the default ~il[backchain-limit]
  (~pl[set-backchain-limit]).  Thanks to Jared Davis for requesting this
  feature.

  Support is now provided for creating and certifying books that do not depend
  on trust tags, in the case that the only use of trust tags is during
  ~ilc[make-event] expansion.  ~l[set-write-acl2x].  Thanks to Sol Swords for
  reporting a couple of bugs in a preliminary implementation.

  Function ~c[(file-write-date$ filename state)] has been added, giving the
  write date of the given file.

  ~l[forward-chaining-reports] for how to get new reports on the forward
  chaining activity occurring in your proof attempts.  Thanks to Dave Greve for
  inspiring the addition of this utility.

  It is now possible to use ACL2's printing utilities to return strings, by
  opening output channels to the keyword ~c[:STRING] rather than to filenames.
  ~l[io].  Thanks to Jared Davis for a helpful conversation that led us to add
  this feature.

  ~st[HEURISTIC IMPROVEMENTS]

  We have slightly improved the handling of ~c[:]~il[forward-chaining]
  rules that contain free variables.  Formerly, such rules might fire only
  once, when the first match for a free variable is discovered, and would
  not fire again even if subsequent forward chaining made available another
  match.  This made it difficult to predict whether a rule with free
  variables would fire or not, depending as it did on the order in which
  newly derived conclusions were added.  The new handling is a little
  slower but more predictable.  Thanks to Dave Greve for sending a helpful
  example that led us to consider making such an improvement.

  We have slightly improved the so-called ``~il[type-set]'' heuristics to work
  a bit harder on terms of the form ~c[(rec term)], where ~c[rec] is a
  so-called ``compound-recognizer'' function, that is, a function with a
  corresponding enabled ~c[:]~ilc[compound-recognizer] rule.  Thanks to Jared
  Davis for sending a helpful example (found, in essence, in the modified
  function ~c[type-set-rec], source file ~c[type-set-b.lisp]).

  We made three heuristic improvements in the way contexts (so-called
  ``type-alists'') are computed from goals (``clauses'').  Although these
  changes did not noticeably affect timing results for the ACL2 regression
  suite, they can be very helpful for goals with many hypotheses.  Thanks to
  Dave Greve for sending a useful example (one where we found a goal with 233
  hypotheses!).

  The algorithm for substituting alists into terms was modified.  This change
  is unlikely to affect many users, but in one example it resulted in a
  speed-up of about 21%.  Thanks to Dave Greve for supplying that example.

  Sped up ~ilc[include-book] a bit by memoizing checksums of symbols.  (This
  change pertains to ``normal'' ACL2 only, not the ~ilc[hons] version
  (~pl[hons-and-memoization], where such memoization already occurred.)  We
  found about a 23% speed-up on an example from Dave Greve.

  Made a small change to the algorithm used to prove hypotheses of
  ~c[:]~ilc[type-prescription] rules (ACL2 source function
  ~c[type-set-relieve-hyps]).  One change avoids a linear walk through the context
  (the ``type-alist'' structure), while the other could avoid storing
  unnecessary ~ilc[force]d assumptions (into the so-called ``tag-tree'').

  ~st[BUG FIXES]

  Fixed a long-standing soundness bug caused by the interaction of ~ilc[force]d
  hypotheses with destructor ~il[elim]ination.  The fix was to avoid using
  forcing when building the context (so-called ``type-alist'') when the goal is
  considered for destructor elimination; those who are interested can see a
  discussion in source function ~c[eliminate-destructors-clause1], which
  includes a proof of ~c[nil] that no longer succeeds.  A similar fix was made
  for generalization, though we have not exploited the previous code to prove
  ~c[nil] in that case.

  Fixed a bug that allowed book certification to ignore ~c[skip-proofs] around
  ~ilc[encapsulate] ~il[events].  Thus, a book could contain an event of the
  form ~c[(skip-proofs (encapsulate ...))], and a call of ~ilc[certify-book] on
  that book could succeed even without supplying keyword
  ~c[:skip-proofs-okp t].  This bug was introduced in Version  3.5 (May,
  2009).

  Fixed a bug that could occur when including a book that attempts to redefine
  a function as a macro, or vice-versa.  (For details of the issue, see the
  comment in the definition of variable ~c[*hcomp-fn-macro-restore-ht*] in
  source file ~c[other-events.lisp].)

  (Windows only) Fixed handling of the Windows drive so that an executable
  image saved on one machine can be used on another, even with a different
  drive.  Thanks to Harsh Raju Chamarthi for reporting this issue and doing a
  lot of testing and collaboration to help us get this right.

  Made a change to avoid possible low-level errors, such as bus errors, when
  quitting ACL2 by calling ~ilc[good-bye] or its synonyms.  This was occurring
  in CCL, and we are grateful to Gary Byers for helping us find the source of
  those errors (which basically was that ACL2 was attempting to quit while
  already in the process of quitting).

  Fixed a bug in ~ilc[with-guard-checking], which was being ignored in function
  bodies.

  Fixed a bug in ~ilc[top-level], which was not reverting the logical
  ~il[world] when an error resulted from evaluation of the given form.  Thanks
  to Jared Davis for bringing this bug to our attention.

  Fixed a long-standing bug (back through Version  2.7) that was discarding
  changes to the connected book directory (~pl[cbd]) when exiting and then
  re-entering the top-level ACL2 loop (with ~ilc[lp]).

  In some host Lisps, it has been possible to be in a situation where it is
  impossible to interrupt checkpoint printing during the summary.  We had
  thought this solved when the host Lisp was CCL, but Sol Swords sent us an
  example (for which we are grateful) illustrating that this behavior could
  occur.  This has been fixed.

  Fixed a bug in a proof obligation generated for ~c[:]~ilc[meta] and
  ~c[:]~ilc[clause-processor] rules, that the ~il[guard] on the metafunction or
  clause-processor function, ~c[fn], holds under suitable assumptions.  Those
  assumptions include not only that the first argument of ~c[fn] satisfies
  ~ilc[pseudo-termp], but also that all ~il[stobj] inputs satisfy the
  corresponding stobj recognizer predicates.  We had erroneously considered
  stobj outputs of ~c[fn] instead of stobj inputs.  Thanks to Sol Swords for
  bringing this bug to our attention with a simple example, and correctly
  pointing us to the bug in our code.

  Fixed the following bugs in ~ilc[defattach].  We hadn't always been applying
  the full functional substitution when generating guard proof obligations.  We
  had been able to hit an assertion when reattaching to more than one function.
  Attachment was permitted in the case of an untouchable function
  (~pl[remove-untouchable]).  Finally, the guard proof obligation could fail in
  the case that the two functions have different formal parameter lists, as in
  the following example.
  ~bv[]
  (encapsulate
   ((foo (x) x :guard (symbolp x)))
   (local (defun foo (x) x)))
  (defun bar (x2)
    (declare (xargs :guard (symbolp x2)))
    x2)
  (defattach foo bar)
  ~ev[]

  Fixed a raw Lisp error that could be caused by including a book using
  ~ilc[make-event] to define a function symbol in a locally-introduced package.
  An example appears in a comment in ACL2 source function
  ~c[write-expansion-file].

  Made a change that can prevent an error near the end of book certification
  when the underlying Host Lisp is Allegro Common Lisp, in the case that
  environment variable ~c[ACL2_SYSTEM_BOOKS] has been set to the name of a
  directory with a parent that is a soft link.  Thanks to Dave Greve for
  supplying an example to led us to this fix, which involves avoiding Allegro
  CL's implementation of the Common Lisp function, ~c[truename].

  Fixed a bug that was failing to substitute fully using bindings of free
  variables in ~ilc[force]d hypotheses.  A related change is that instead of
  binding such a free variable to a new variable of the form ~c[???-Y], the new
  variable is now of the form ~c[UNBOUND-FREE-Y].

  Fixed a bug that could inhibit the printing of certain theory warnings (and
  probably, in the other direction, cause inappropriate such printing).

  We eliminated excessive ~c[\"Raw-mode\"] warnings about
  ~ilc[add-include-book-dir] that could be generated by the use of raw-mode
  during ~ilc[include-book].  Thanks to Dave Greve for bringing this issue to
  our attention.

  Fixed the printing of results from forms within an ~ilc[encapsulate], so that
  they are abbreviated according to the ~ilc[ld-evisc-tuple].

  It is now possible to evaluate ~il[stobj]-related forms after evaluating
  ~c[:]~ilc[set-guard-checking] ~c[:none] or ~c[:]~ilc[set-guard-checking]
  ~c[nil], even in cases where such evaluation formerly caused a guard
  violation due to a bug in ACL2.  Here is an example of an error that no
  longer occurs.
  ~bv[]
  ACL2 !>(defstobj st fld)

  Summary
  Form:  ( DEFSTOBJ ST ...)
  Rules: NIL
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
   ST
  ACL2 !>(set-guard-checking :none)

  Turning off guard checking entirely.  To allow execution in raw Lisp
  for functions with guards other than T, while continuing to mask guard
  violations, :SET-GUARD-CHECKING NIL.  See :DOC set-guard-checking.

  ACL2 >(fld st)


  ACL2 Error in TOP-LEVEL:  The guard for the function call (FLD ST),
  which is (STP ST), is violated by the arguments in the call (FLD ST).
  [... etc. ...]
  ~ev[]
  You can understand how things now work by imagining that when a function
  introduced by ~ilc[defstobj] is called, ~ilc[guard]-checking values of
  ~c[:none] or ~c[nil] are temporarily converted to ~c[t].  Thanks to Pete
  Manolios, Ian Johnson, and Harsh Raju Chamarthi for requesting this
  improvement.

  Fixed a bug in which the wrong attachment could be made when the same
  function has an attachment in a book and another in the certification world
  of that book (possibly even built into ACL2), if the load of a compiled file
  is aborted because a sub-book's compiled file is missing.  The bug has been
  present since the time that ~ilc[defattach] was added (Version_4.0).  An
  example may be found in a comment in the ~ilc[deflabel] for ~c[note-4-2]
  (ACL2 source file ~c[ld.lisp]).

  The ~c[:]~ilc[doc] and related utilities now cause a clean error when
  provided other than a symbol.  Thanks to Jared Davis for pointing out the raw
  Lisp error that had occurred in such cases.

  It had been the case that in raw-mode (~pl[set-raw-mode]), it was possible to
  confuse ~ilc[include-book] when including a book in a directory different
  from the current directory.  This has been fixed.  Thanks to Hanbing Liu for
  bringing this problem to our attention with a small example.

  ~st[NEW AND UPDATED BOOKS AND RELATED INFRASTRUCTURE]

  Many changes have been made to the distributed books, thanks to an active
  ACL2 community.  You can contribute books and obtain updates between ACL2
  releases by visiting the ~c[acl2-books] project web page,
  ~url[http://acl2-books.googlecode.com/].

  There is new ~c[Makefile] support for certifying just some of the distributed
  books.  ~l[books-certification-classic], in particular discussion of the
  variable ~c[ACL2_BOOK_DIRS].  Thanks to Sandip Ray for requesting this
  enhancement.

  The ~il[documentation] for ~ilc[make-event] now points to a new book,
  ~c[books/make-event/defrule.lisp], that shows how ~c[make-event] can be used to
  do macroexpansion before generating ~il[events].  Thanks to Carl Eastlund for
  useful interaction on the acl2-help mailing list that led us to add this
  example.

  ~st[EMACS SUPPORT]

  Incorporated a version of changes from Jared Davis to the ~c[control-t f]
  emacs utility (distributed file ~c[emacs/emacs-acl2.el]), so that one can
  fill a format string from anywhere within the string.

  ~st[EXPERIMENTAL VERSIONS]

  We refrain from listing changes here to experimental versions, other than an
  enhancement to the ~il[HONS] version that can reduce sizes of
  ~il[certificate] files, by applying ~ilc[hons-copy] to introduce structure
  sharing (ACL2 source function ~c[make-certificate-file1]).

  ~/~/")

(deflabel |NOTE-4-2(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  4.2(r) (January, 2011) Notes~/

  ~/

  Please ~pl[note-4-2] for changes in Version  4.2 of ACL2.

  ~/
  ")

(deflabel note-4-3

; The following example illustrates the soundness bug related to mbe that is
; mentioned in :doc note-4-3.  To prove nil:

; (certify-book "sub")
; :u
; (certify-book "mid")
; :u
; (certify-book "top")

; The problem is that the macro mac expands differently in the ACL2 loop from
; how it expands in raw Lisp.  The bug fix is to check equality of the values
; of the :logic and :exec forms when in safe-mode.

;;;;;;;;;;;;;;;
; Book sub.lisp
;;;;;;;;;;;;;;;
;   (in-package "ACL2")
;
;   (defmacro mac ()
;     (mbe :logic ''logic
;          :exec ''exec))
;
;   (defconst *a*
;     (mac))

;;;;;;;;;;;;;;;
; Book mid.lisp
;;;;;;;;;;;;;;;
;   (in-package "ACL2")
;
;   (local (include-book "sub"))
;
;   (defmacro mac ()
;     (mbe :logic ''logic
;          :exec ''exec))
;
;   (defconst *a*
;     (mac))
;
;   (defthm got-exec
;     (equal *a* 'exec)
;     :rule-classes nil)

;;;;;;;;;;;;;;;
; Book top.lisp
;;;;;;;;;;;;;;;
;   (in-package "ACL2")
;
;   (defmacro mac ()
;     (mbe :logic ''logic
;          :exec ''exec))
;
;   (defconst *a*
;     (mac))
;
;   (defthm got-logic
;     (equal *a* 'logic)
;     :rule-classes nil)
;
;   (include-book "mid")
;
;   (defthm contradiction
;     nil
;     :hints (("Goal" :use (got-exec got-logic)))
;     :rule-classes nil)

;;;;;;;;;;;;;;; end of example

; Modified some doc-printing functions in support of translation to xdoc.  the
; process, the HTML output has become prettier; see item about HTML, below.

; Added new macro mv-to-state.

; Changed the name mv-let? to bdd-mv-let to avoid potential confusion with
; mv?-let.

; Functions enable-iprint-ar and disable-iprint-ar now return two values
; instead of three.

; Here is the example promised in :doc note-4-3 to illustrate a bug in the
; loop-checking done on behalf of defattach.  The bug was fixed in function
; update-attachment-records1.

;;;;;;;;;;;;;;; start example
;   (progn
;     (encapsulate
;      ((f1 (x) t)
;       (f2 (x) t))
;      (local (defun f1 (x) x))
;      (local (defun f2 (x) x)))
;
;     (encapsulate
;      ((g1 (x) t)
;       (g2 (x) t))
;      (local (defun g1 (x) x))
;      (local (defun g2 (x) x))
;      (defthm g1-f1
;        (equal (g1 (f1 x))
;               (f1 x))))
;
;     (encapsulate
;      ((h1 (x) t)
;       (h2 (x) t))
;      (local (defun h1 (x) x))
;      (local (defun h2 (x) x))))
;
;   (defattach f2 h1)
;
;   ; The following should cause the following loop to be reported, but didn't:
;       G1 is an extended ancestor of H1.
;       H1 is an extended ancestor of F1.
;       F1 is an extended ancestor of G1.
;   (defattach h2 g2)
;;;;;;;;;;;;;;; end of example

; Replaced a couple of calls of 1+ by 1+f in fn-count-evg-rec, after Robert
; Krug brought these to our attention.

; Improved the error message when discovering during load of compiled or
; expansion file that a defconst is not redundant.

; Fixed macro io? so that we are not left in a wormhole when there is an error
; (as happened previously when the commentp argument of io? was t).

; Regarding "Fixed a bug in detection of package redefinition.": The use of
; member-equal instead of assoc-equal in maybe-introduce-empty-pkg-2 allows (at
; least on quick analysis) every package with empty imports to be considered a
; "virgin" package, which may have allowed (again, on quick analysis) illegal
; package redefinition to occur.

; Made efficiency improvement in check-vars-not-free, which is minor but
; perhaps worth a couple percent since we have added equality-variants.

; We no longer make some duplicate cons-tag-trees calls in defaxiom-fn and
; defthm-fn1.

; Removed needless (and confusing) #+ansi-cl in handler-bind call for sbcl in
; with-warnings-suppressed.

; Modified the handling of package definitions in expansion files (macro
; maybe-introduce-empty-pkg-1).

; We took preliminary steps towards removing uses of the big-clock field of
; state.

; Modified deletion of compiled file of acl2-fns.lisp to occur at the Lisp
; level instead of using GNUmakefile.

; Deleted delete-pair and remove-first-pair, which each duplicated the
; functionality of delete-assoc-eq.

; Eliminated cons-into-ttree in favor of cons-tag-trees.

; Moved assert$ to the right place in cmp-to-error-triple (thanks to David
; Rager for correcting an error in our initial change).

; Below we show how to obtain a hard Lisp error in Version_4.2, when, as
; mentioned in the :doc string below, "including books with hidden packages".
; The problem was that write-expansion-file was deciding whether to push
; maybe-introduce-empty-pkg-1 and maybe-introduce-empty-pkg-2 forms based on
; the known-package-alist just before pass 1 of certify-book, rather than just
; after it.  But a second defpkg form can use symbols defined in the first, so
; since it's pretty cheap just to lay down all such forms, that's what we do.

;   sloth:~/temp> cat sub.lisp
;   ; First execute these two commands in the certification world:
;   ; (defpkg "FOO" '(a))
;   ; (defpkg "BAR" '(foo::b))
;
;   ; Then:
;   ; (certify-book "sub" 2)
;
;   (in-package "ACL2")
;
;   (defun g (x) x)
;   sloth:~/temp> cat top.lisp
;   ; Just do this:
;   ; (certify-book "top")
;
;   (in-package "ACL2")
;
;   (local (include-book "sub"))
;
;   (defun h (x) x)
;   sloth:~/temp>
;
;   ACL2 !>(include-book "top")
;
;   ***********************************************
;   ************ ABORTING from raw Lisp ***********
;   Error:  There is no package named "FOO" .
;   ***********************************************

  :Doc
  ":Doc-Section release-notes

  ACL2 Version  4.3 (July, 2011) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  4.2 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, changes at the system level and to distributed
  books, Emacs support, and experimental versions.  Each change is described in
  just one category, though of course many changes could be placed in more than
  one category.

  ~st[CHANGES TO EXISTING FEATURES]

  Significant changes have been made for list-processing primitives such as
  ~ilc[member] and ~ilc[assoc]; ~pl[equality-variants].  In summary: instead of
  separate functions based on ~ilc[eq], ~ilc[eql], and ~ilc[equal], there is
  essentially just one function, which is based on ~ilc[equal]; the ~ilc[eq]
  and ~ilc[eql] variants are logically just the ~ilc[equal] variant.  For
  example, ~ilc[member-eq] and ~ilc[member] are macros that generate
  corresponding calls of ~ilc[member-equal] in the logic, although in raw Lisp
  they will execute using tests ~ilc[eq] and ~ilc[eql], respectively.
  References to any of these in logical contexts such as ~il[theories] are now
  references to the function based on ~ilc[equal]; for example, the hint
  ~c[:in-theory (disable member)] is completely equivalent to the hint
  ~c[:in-theory (disable member-equal)].  Distributed books have been
  modified as necessary to accommodate this change.  While the need for such
  changes was relatively infrequent, changes were for example needed in
  contexts where terms are manipulated directly; for example,
  ~ilc[defevaluator] needs to mention ~ilc[member-equal] rather than
  ~ilc[member], just as it was already the case to mention, say,
  ~ilc[binary-append] rather than ~ilc[append].  Again, ~pl[equality-variants]
  for more information about equality variants.

  A few improvements were made in support of the modified treatment of equality
  variants discussed above.  The changes include the following.~bq[]

  o We now allow the use of macro aliases (~pl[macro-aliases-table]
  in ~c[:trigger-fns] of rules (~pl[rule-classes]).

  o We now remove so-called ``guard holders'' (including calls of
 ~ilc[return-last], hence of ~ilc[mbe]) in ~c[:trigger-terms] of rules.

  o We also remove guard holders in formulas of ~c[:]~ilc[congruence] and
  ~ilc[type-prescription] rules.

  o Macros ~c[union-eq] and ~c[intersection-eq] can now take any positive
  number of arguments, and ~ilc[union-eq] can take zero arguments.  (Thanks to
  Jared Davis for requesting this enhancement.)  The same can be said for new
  macros ~ilc[union$] and ~ilc[intersection$], respectively.

  o A few changes were made to built-in theorems from source file
  ~c[axioms.lisp], in particular disabling ~c[:]~ilc[type-prescription] rule
  ~c[consp-assoc-equal] (formerly two enabled rules, ~c[consp-assoc-eq] and
  ~c[consp-assoc]) but adding this theorem as a ~c[:forward-chaining] rule, and
  similarly for ~c[true-list-listp-forward-to-true-listp-assoc-equal] (and
  eliminating rule ~c[true-list-listp-forward-to-true-listp-assoc-eq]; and
  disabling rule ~c[true-listp-cadr-assoc-eq-for-open-channels-p].  Also,
  theorem ~c[all-boundp-preserves-assoc] has been renamed to
  ~c[all-boundp-preserves-assoc-equal] and also strengthened.

  o Some ~il[guard]s were slightly improved (logically weaker or the
  same).~eq[]

  Improved ~c[get-output-stream-string$] to allow for a context and to do
  genuine error printing instead of using ~ilc[cw].  ~l[io].

  Added the symbols ~ilc[flet] and ~ilc[with-local-stobj] to
  ~c[*acl2-exports*].

  A small change was made to the processing of more than one ~c[:]~ilc[guard]
  declaration for the same function.  In particular, a guard of ~c[t] is
  essentially ignored.

  Attachments are now allowed during evaluation of the first argument of
  ~ilc[prog2$] even in contexts (such as proofs) during which the use of
  attachments is normally prohibited.  More generally, the second of the three
  arguments of ~ilc[return-last] has this property, except in the case of
  ~ilc[mbe] (or related macros like ~ilc[mbe1]), where the ~c[exec] argument
  may provide the value.  Thanks to Sol Swords for useful discussions leading
  us to implement this enhancement.

  The restriction has been removed that prohibited the use of ~ilc[mbe] inside
  ~ilc[encapsulate] ~il[events] with a non-empty ~il[signature].  This
  restriction was introduced in Version  3.4, but has not been necessary since
  Version  4.0, when we first started disallowing ~il[guard] verification for
  functions introduced non-~il[local]ly inside such ~ilc[encapsulate] events.

  We weakened the checks involving common ancestors for evaluator and
  ~il[meta] (and ~il[clause-processor]) functions (~pl[evaluator-restrictions])
  so that except in the ~ilc[mbe] case, the next-to-last argument of
  ~ilc[return-last] is not considered.  Thanks to Sol Swords for bringing this
  issue to our attention.

  The macro ~ilc[append] no longer requires at least two arguments.  Thanks to
  Dave Greve for requesting this enhancement.

  (Mostly for system hackers) Now, ~ilc[break-on-error] breaks at a more
  appropriate (earlier) time for certain system functions that do not return
  state, such as ~c[translate11].  Thanks to David Rager for requesting this
  improvement.

  ~ilc[Show-accumulated-persistence] may take a new argument, ~c[:runes], which
  simply causes an alphabetical list of ~il[rune]s to be printed out.

  Improved ~ilc[trace$] so that ~c[:entry], ~c[:exit], and ~c[:cond] forms may
  reference ~c[state] even if the function being traced does not include
  ~c[state] as a formal.

  The system function ~c[acl2x-expansion-alist] now takes a second argument,
  namely ~ilc[state].  This change allows for more flexibility in the sorts of
  attachments that can be made to this function (~pl[defattach]).  Thanks to
  Jared Davis and Sol Swords for requesting this enhancement and providing a
  preliminary implementation.

  An obscure ~il[proof-checker] change, unlikely to affect users, replaces the
  state global variables ~c[erp], ~c[val], ~c[print-macroexpansion-flg], and
  ~c[print-prompt-and-instr-flg] by ~c[pc-erp], ~c[pc-val],
  ~c[pc-print-macroexpansion-flg], and ~c[pc-print-prompt-and-instr-flg],
  respectively.

  ~il[State] globals ~c[fmt-hard-right-margin] and ~c[fmt-soft-right-margin]
  are now untouchable (~pl[set-fmt-hard-right-margin] and
  ~pl[push-untouchable]).  If you bind these ~c[state] globals with
  ~ilc[state-global-let*], then you will need to do so with appropriate
  setters to restore their values, for example as follows.
  ~bv[]
    (state-global-let*
     ((fmt-hard-right-margin 500 set-fmt-hard-right-margin)
      (fmt-soft-right-margin 480 set-fmt-soft-right-margin))
     ...)
  ~ev[]

  The error message has been improved for the case of evaluating an undefined
  function that has an attachment (~pl[defattach]).  Thanks to Jared Davis for
  sending the following example, which illustrates the additional part of the
  message.
  ~bv[]
    ACL2 !>(defstub foo (x) t)
    [[... output omitted ...]]
    ACL2 !>(defattach foo identity)
    [[... output omitted ...]]
    ACL2 !>(defconst *x* (foo 3))


    ACL2 Error in ( DEFCONST *X* ...):  ACL2 cannot ev the call of undefined
    function FOO on argument list:

    (3)

    Note that because of logical considerations, attachments (including
    IDENTITY) must not be called in this context.

    [[... additional output omitted ...]]
  ~ev[]

  The directory string supplied to ~ilc[add-include-book-dir] no longer must
  terminate with the `~c[/]' character, as had been required in some Lisp
  implementations.  Thanks to Sol Swords for bringing this issue to our
  attention.

  We no longer print induction schemes with ~il[gag-mode]; use ~c[:]~ilc[pso]
  if you want to see them.  Thanks to Dave Greve for this suggestion.

  It is now legal to supply a constant for a ~il[stobj] array dimension.
  ~l[defstobj].  Thanks to Warren Hunt for requesting this enhancement.

  We cleaned up a few issues with ~ilc[defpkg].~bq[]

  o It is no longer illegal to submit a ~ilc[defpkg] form in raw-mode
  (~pl[set-raw-mode]).  Thanks to Jun Sawada for reporting an example in which
  an ~ilc[include-book] form submitted in raw-mode caused an error because of a
  `hidden' ~ilc[defpkg] form (~pl[hidden-defpkg]).  There will no longer be an
  error in such cases.

  o It had been the case that ~il[local]ly including a book could make it
  possible to use a package defined by that book.  Consider for example the
  following book, ~c[foo.lisp].
  ~bv[]
  (in-package \"ACL2\")
  (local (include-book \"arithmetic/top\" :dir :system))
  ~ev[]
  After certifying this book, it had been possible to admit the following
  events in a new session.
  ~bv[]
  (include-book \"foo\")
  (defconst acl2-asg::*foo* 3)
  (defconst *c* 'acl2-asg::xyz)
  ~ev[]
  In Version_4.3, neither of these ~ilc[defconst] events is admitted.

  o A hard Lisp error is now avoided that had been possible in rare cases when
  including books with hidden packages (~pl[hidden-defpkg]). An example may be
  found in a comment in the ~ilc[deflabel] for ~c[note-4-3] (in ACL2 source
  file ~c[ld.lisp]).~eq[]

  The undocumented (but sometimes useful) functions ~c[packn1] and ~c[packn]
  are now ~il[guard]-verified ~c[:]~ilc[logic] mode functions.  Thanks to
  Sandip Ray for requesting this enhancement.

  It had been the case that when including a book, functions defined in the
  book's certification ~il[world] (that is, in its ~il[portcullis]
  ~il[command]s) were typically not given compiled code.  That has been fixed.

  The commands ~c[:]~ilc[pl] and ~c[:]~ilc[pl2] have been improved, primarily
  by printing information for more rule classes.  ~l[pl] and ~pl[pl2].  See
  also the item below about the new ~il[proof-checker] command,
  ~c[show-type-prescriptions].

  ~st[NEW FEATURES]

  New macros ~ilc[mv?-let] and ~ilc[mv?] extend the funtionality of
  ~ilc[mv-let] and ~ilc[mv] (respectively) to the case of a single value.

  Macro ~il[with-local-state] is available for system programmers who wish bind
  ~ilc[state] locally, essentially using ~il[with-local-stobj].  But this
  should only be done with extreme care, and it requires an active trust tag;
  ~pl[with-local-state].

  Formatted printing functions now have analogues that print to strings and do
  not take an output channel or ~ilc[state] as arguments.
  ~l[printing-to-strings].

  The system function ~c[ancestors-check] is now available for verified
  modification by users, i.e., attachment using
  ~c[(defattach ancestors-check <your_function>)].  Thanks to Robert Krug for
  providing the necessary proof support, which we modified only in small ways.

  New macros, ~c[observation-cw] and ~c[warning$-cw], provide formatted
  printing of ~ilc[observation]s and warnings (respectively) without
  ~ilc[state].  Thanks to Harsh Raju Chamarthi and David Rager for requests
  leading to these utilities.  ~c[Observation-cw] is now used in some of the
  distributed books (thanks to Robert Krug for useful interaction for that).

  The ~il[proof-checker] command ~c[type-alist] (~pl[proof-checker-commands])
  now takes an optional third argument that causes the production of
  forward-chaining reports (~pl[forward-chaining-reports]).  Thanks to Dave
  Greve for requesting such an enhancement.

  The reports generated by forward-chaining, ~il[forward-chaining-reports],
  have been changed to indicate when a conclusion reached by forward chaining
  is ~c[REDUNDANT] with respect to the type information already known.  Thanks
  to Dave Greve for suggesting this enhancement.

  The utility ~ilc[with-prover-time-limit] is now legal for ~il[events]
  (~pl[embedded-event-form]).  For example, the following is now legal.
  ~bv[]
  (encapsulate
   ()
   (with-prover-time-limit
    2
    (defthm append-assoc
      (equal (append (append x y) z)
             (append x (append y z))))))
  ~ev[]

  The new utility ~ilc[with-prover-step-limit] is analogous to the utility
  ~ilc[with-prover-time-limit], but counts ``prover steps'' rather than
  checking for time elapsed.  ~l[with-prover-step-limit].  Also
  ~pl[set-prover-step-limit] to provide a default step-limit.  Note that just
  as ~ilc[with-prover-time-limit] may now be used to create ~il[events], as
  discussed just above, ~ilc[with-prover-step-limit] may also be used to create
  ~il[events].  Thanks to Carl Eastlund for requesting support for step-limits.

  The macro ~ilc[progn$] is analogous to ~ilc[prog2$], but allows an arbitrary
  number of arguments.  For example:
  ~bv[]
  ACL2 !>:trans1 (progn$ (f1 x) (f2 x) (f3 x))
   (PROG2$ (F1 X) (PROG2$ (F2 X) (F3 X)))
  ACL2 !>
  ~ev[]
  Thanks to David Rager for contributing this macro.

  The macro ~ilc[defattach] may now be supplied the argument
  ~c[:skip-checks :cycles].  In this case, as with argument ~c[:skip-checks t],
  a trust tag is reuired (~pl[defttag]), and no logical claims are made.  The
  effect is to avoid the usual check that the extended ancestor relation has no
  cycles (~pl[defattach]).  Thanks to Dave Greve for requesting this feature.

  You can now limit the printing of subgoal names when using
  ~c[:]~ilc[set-gag-mode]~c[ :goals].  ~l[set-print-clause-ids].  Thanks to
  Karl Hoech for a suggestion leading to this enhancement.

  A new ~il[proof-checker] command, ~c[show-type-prescriptions], or ~c[st] for
  short, provides information about ~c[:]~ilc[type-prescription] rules that
  match a given term.  Thanks to Dave Greve for requesting this enhancement.
  See also the item above about related improvements to commands ~c[:]~ilc[pl]
  and ~c[:]~ilc[pl2].

  ~st[HEURISTIC IMPROVEMENTS]

  ACL2 now avoids some repeated attempts to rewrite hypotheses of rewrite
  rules.  ~l[set-rw-cache-state] for a discussion of this behavior and how to
  avoid it.  The default behavior has been observed to reduce by 11% the
  overall time required to complete a regression.  Here are the directories
  that had the top three time decreases and top three time increases, shown in
  seconds.
  ~bv[]
    -368 coi/gacc (1064 down to 696: decrease of 35%)
    -220 workshops/1999/ste (664 down to 444: decrease of 33%)
    -148 unicode (331 down to 183: decrease of 45%)
    ....
      +7 workshops/2002/cowles-flat/support (229 up to 236: increase of 3%)
      +8 workshops/1999/ivy/ivy-v2/ivy-sources (508 up to 516: increase of 2%)
     +12 workshops/2009/hardin/deque-stobj (78 up to 91: increase of 17%)
  ~ev[]

  The so-called ``ancestors check,'' which is used to limit backchaining, has
  been strengthened so that two calls of ~ilc[equal] are considered the same
  even if their arguments appear in the opposite order.  Thanks to Robert Krug
  for providing an implementation and a useful discussion.

  The check for ~il[irrelevant-formals] in processing of ~ilc[defun]s has been
  made more efficient.  Thanks to Eric Smith for reporting this issue in 2001
  (!) and thanks to Warren Hunt for recently sending an example.  For that
  example, we have seen the time for the ~il[irrelevant-formals] check reduced
  from about 10 seconds to about 0.04 seconds.

  (GCL only) The macro ~ilc[mv] has been modified so that certain fixnum boxing
  can be avoided.

  (Allegro CL only) We have set to ~c[nil] four Allegro CL variables that
  otherwise enable storing of certain source information (for details, see the
  discussion of ``cross-referencing'' in ACL2 source file ~c[acl2-init.lisp]).
  As a result of this change we have about a 6% speedup on the regression
  suite, but a 27% time reduction on an example that includes a lot of books.

  Exhaustive matching for the case of ~il[free-variables] has been extended to
  ~il[type-prescription] rules, in analogy to the default setting
  ~c[:match-free :all] already in place for ~il[rewrite], ~il[linear], and
  ~il[forward-chaining] rules.  ~l[free-variables-type-prescription].  Thanks
  to Dave Greve for requesting this enhancement.

  ~st[BUG FIXES]

  A soundness bug was fixed in some raw-Lisp code implementing the function,
  ~ilc[take].  Thanks to Sol Swords for pointing out this bug with
  (essentially) the following proof of ~c[nil].
  ~bv[]
  (defthmd take-1-nil-logic
    (equal (take 1 nil) '(nil))
    :hints((\"Goal\" :in-theory (disable (take)))))
  (thm nil :hints ((\"Goal\" :use take-1-nil-logic)))
  ~ev[]

  Calls of ~ilc[mbe] in ``safe-mode'' situations ~-[] i.e., during evaluation
  of ~ilc[defconst], ~ilc[value-triple], and ~ilc[defpkg] forms, and during
  macroexpansion ~-[] are now guard-checked.  Thus, in these situations both
  the ~c[:logic] and ~c[:exec] forms will be evaluated, with an error if the
  results are not equal.  Formerly, only the ~c[:logic] form was evaluated,
  which was a soundness bug that could be exploited to prove ~c[nil].  For a
  such a proof and a bit of further explanation, see the example at the top of
  the comments for ~c[(deflabel note-4-3 ..)] in ACL2 source file ~c[ld.lisp].

  It had been possible to prove ~c[nil] by proving the following
  theorem using ACL2 built on CCL and then proving its negation using
  ACL2 built on a different host Lisp.
  ~bv[]
  (defthm host-lisp-is-ccl
    (equal (cdr (assoc 'host-lisp *initial-global-table*))
           :ccl)
    :rule-classes nil)
  ~ev[]
  This hole has been plugged by moving the setting of ~c['host-lisp] out
  of the constant ~c[*initial-global-table*].

  Fixed ~ilc[trace$] for arguments that are ~il[stobj] accessors or updaters.
  It also gives an informative error in this case when the accessor or updater
  is a macro (because the introducing ~ilc[defstobj] event specified
  ~c[:inline t]).

  Avoided a potential error that could occur when no user home directory is
  located.  Our previous solution for Windows simply avoided looking for ACL2
  customization files (~pl[acl2-customization]) and ~c[acl2-init.lsp] files in
  a user's home directory.  With this change, we handle such files the same for
  Windows as for non-Windows systems: we always look for ACL2 customization
  files (~pl[acl2-customization]) and ~c[acl2-init.lsp] files in a user's home
  directory, but only if such a directory exists.  Thanks to Hanbing Liu for
  reporting this issue.

  (GCL only) Fixed a bug that prevented the use of
  ~ilc[get-output-stream-string$] when the host Lisp is GCL.

  Fixed ~ilc[with-live-state] to work properly for executable
  counterparts (so-called ``*1*'' functions).

  Fixed a bug in the error message caused by violating the ~il[guard] of a
  macro call.

  Fixed a bug in an error message that one could get when calling
  ~ilc[defattach] with argument ~c[:skip-checks t] to attach to a
  ~c[:]~ilc[program] mode function symbol that was introduced with
  ~ilc[defun].  (This is indeed an error, but the message was confusing.)
  Thanks to Robert Krug for bringing this bug to our attention.

  Fixed a bug in the loop-checking done on behalf of ~ilc[defattach], which
  could miss a loop.  For an example, see the comment about loop-checking in
  the comments for ~c[(deflabel note-4-3 ..)] in ACL2 source file ~c[ld.lisp].

  Terms of the form ~c[(hide <term>)] without free variables could be
  simplified, contrary to the purpose of ~ilc[hide].  This is no longer the
  case,  Thanks to Dave Greve for reporting this issue.

  An infinite loop could occur when an error was encountered in a call of
  ~ilc[wormhole-eval], for example with the following form, and this has been
  fixed.
  ~bv[]
  (wormhole-eval 'demo
                 '(lambda ()
                    (er hard 'my-top \"Got an error!\"))
                 nil)
  ~ev[]

  Fixed a bug in detection of package redefinition.  While we have no example
  demonstrating this as a soundness bug, we cannot rule it out.

  Fixed a bug in the message produced by an erroneous call of ~ilc[flet].
  Thanks to Jared Davis for reporting this bug and sending a helpful example.

  For a failed ~ilc[defaxiom] or ~ilc[defthm] event, we now avoid printing
  ~il[rune]s that are used only in processing proposed rules to be stored, but
  not in the proof itself.  Thanks to Dave Greve for sending us an example that
  led us to make this fix.

  ACL2 did not reliably enforce the restriction against non-~ilc[local]
  ~ilc[include-book] ~il[events] inside ~ilc[encapsulate] events, as
  illustrated by the following examples.
  ~bv[]
  ; not permitted (as expected)
  (encapsulate () (include-book \"foo\"))

  ; permitted (as expected)
  (encapsulate () (local (include-book \"foo\")))

  ; formerly permitted (surprisingly); now, not permitted
  (local (encapsulate () (include-book \"foo\")))
  ~ev[]
  Moreover, the corresponding error message has been fixed.  Thanks to Jared
  Davis and Sandip Ray for relevant discussions.

  When ~ilc[include-book] is given a first argument that is not a string, a
  more graceful error now occurs, where previously an ugly raw Lisp error had
  occurred.  Thanks to Eric Smith for bringing this bug to our attention.

  Fixed a bug in an error message that was printed when an unexpected
  expression has occurred where a ~ilc[declare] form is expected.

  (Since all functions are compiled when the host Lisp is CCL or SBCL, the
  following bug fix did not occur for those host Lisps.)  After evaluation of
  ~c[(]~ilc[set-compile-fns]~c[ t)], all defined functions are expected to run
  with compiled code; but this was not the case for functions exported from an
  ~ilc[encapsulate] event.  This has been fixed.

  It had been the case that the ~c[:]~ilc[puff] command was broken for
  ~ilc[include-book] form whose book had been certified in a world with an
  ~ilc[add-include-book-dir] event.  This has been fixed.

  Evaluation of ~il[stobj] updaters (~pl[defstobj]) may no longer use
  attachments (~pl[defattach]).  This is a subtle point that will likely not
  affect many users.  Thanks to Jared Davis for bringing this issue to our
  attention; a slight variant of his example appears in a comment in ACL2
  source function ~c[oneify-cltl-code].

  It had been the case that even when a ~il[stobj] creator function was
  declared to be untouchable (~pl[push-untouchable]), a ~ilc[with-local-stobj]
  form based on that same stobj was permitted.  Now, such forms are not
  admitted.  Thanks to Jared Davis for a query leading to this fix.

  Fixed a buggy message upon ~il[guard] violations, which was suggesting the
  use of ~c[(set-guard-checking :none)] in some cases when guard-checking was
  already set to ~c[:none].

  It had been possible to get a hard Lisp error when computing with
  ~ilc[ec-call] in ~il[books].  The following is an example of such a book,
  whose certification no longer causes an error.
  ~bv[]
  (in-package \"ACL2\")
  (defun f (x) x)
  (defconst *c* (ec-call (f 3)))
  (defun g (x) (cons x x))
  ~ev[]

  The command ~c[:]~ilc[pl2], and also the ~il[proof-checker] commands
  ~c[rewrite] and ~c[show-rewrites] (and hence their respective aliases ~c[r]
  and ~c[sr]), now take rule-id arguments that can be ~c[:]~ilc[definition]
  ~il[rune]s.  These commands dealt with definition rules already, e.g.
  ~bv[]
  :pl2 (append x y) binary-append
  ~ev[]
  but they did not allow explicit specification of ~c[:definition] runes, e.g.:
  ~bv[]
  :pl2 (append x y) (:definition binary-append)
  ~ev[]

  The following example illustrates a bug in the processing of (admittedly
  obscure) ~il[hints] of the form ~c[:do-not-induct name], where ~c[name] is
  not ~c[t], ~c[:otf-flg-override], ~c[:otf], or ~c[nil].  In this example,
  ACL2 had essentially ignored the hint and reverted to prove the original goal
  by induction, rather than to skip the goal temporarily as is expected for
  such hints.  Thanks to David Rager for a helpful discussion.
  ~bv[]
  (thm (and (equal (append (append x y) z) (append x y z))
            (equal (append (append x2 y2) z2) (append x2 y2 z2)))
       :hints ((\"Subgoal 1\" :do-not-induct some-name)))
  ~ev[]

  Fixed a slight bug in the definitions of built-in ~ilc[theories].  For
  example, in a fresh ACL2 session the value of the following form is ~c[nil],
  but formerly included several ~c[:]~ilc[definition] ~il[rune]s.
  ~bv[]
  (let ((world (w state)))
    (set-difference-theories (function-theory :here)
                             (function-theory 'ground-zero)))
  ~ev[]

  ~st[CHANGES AT THE SYSTEM LEVEL AND TO DISTRIBUTED BOOKS]

  Many changes have been made to the distributed books, as recorded in svn logs
  under the `Source' and 'Updates' links at
  ~url[http://acl2-books.googlecode.com/].  Here we list some of the more
  significant changes.~bq[]

  o A large library has been graciously contributed by the formal verification
  group at Centaur Technology.  See ~c[books/centaur/] and, in particular, file
  ~c[books/centaur/README], which explains how the library depends on the
  experimental HONS extension (~pl[hons-and-memoization]).

  o Among the new books is an illustration of ~ilc[defattach],
  ~c[books/misc/defattach-example.lisp], as well as a variant of defattach that
  avoids the need for ~il[guard] verification,
  ~c[books/misc/defattach-bang.lisp].

  o Distributed book ~c[books/misc/trace1.lisp] has been deleted.  It had
  provided slightly more friendly ~il[trace] output for new users, but
  distributed book ~c[books/misc/trace-star.lisp] may be better suited for that
  purpose.~eq[]

  ACL2 can once again be built on LispWorks (i.e., as the host Lisp), at least
  with LispWorks 6.0.  Thanks to David Rager for useful conversations.
  Several changes have been made from previous LispWorks-based ACL2
  executables:~nl[]
  o ACL2 now starts up in its read-eval-print loop.~nl[]
  o You can save an image with ~ilc[save-exec].~nl[]
  o Multiprocessing is not enabled.~nl[]
  o The stack size is managed using a LispWorks variable that causes the stack
  to grow as needed.~nl[]
  o When ACL2 is built a script file is written, as is done for other host
  Lisps.  Thus, (assuming that no ~c[PREFIX] is specified), ~c[saved_acl2] is
  just a small text file that invokes a binary executable, which for Lispworks
  is ~c[saved_acl2.lw].

  The HTML documentation no longer has extra newlines in <pre> environments.

  Statistics on ACL2 code size may be found in distributed file
  ~c[doc/acl2-code-size.txt].  This file and other information can be found in
  a new ~il[documentation] topic, ~il[about-acl2].

  Fixed the build process to pay attention to environment variable
  ~c[ACL2_SYSTEM_BOOKS] (which may be supplied as a command-line argument to
  `make').  An ACL2 executable can thus now be built even when there is no
  ~c[books/] subdirectory if a suitable replacement directory is supplied.

  Some warnings from the host Lisp are now suppressed that could formerly
  appear.  For example, the warnings shown below occurs in Version  4.2 using
  Allegro CL, but not in Version  4.3.
  ~bv[]
  ACL2 !>(progn (set-ignore-ok t)
                (set-irrelevant-formals-ok t)
                (defun bar (x y)
                  x))
  [[.. output omitted ..]]
   BAR
  ACL2 !>:comp bar
  ; While compiling BAR:
  Warning: Variable Y is never used.
  ; While compiling (LABELS ACL2_*1*_ACL2::BAR ACL2_*1*_ACL2::BAR):
  Warning: Variable Y is never used.
   BAR
  ACL2 !>
  ~ev[]

  ~st[EMACS SUPPORT]

  The distributed Emacs file ~c[emacs/emacs-acl2.el] now indents calls of
  ~c[er@par] and ~c[warning$@par] the same way that calls of ~c[defun] are
  indented.

  ~st[EXPERIMENTAL VERSIONS]

  The parallel version (~pl[parallelism]) now supports parallel evaluation of
  the ``waterfall'' part of the ACL2 prover; ~pl[set-waterfall-parallelism].
  Thanks to David Rager for doing the primary design and implementation work.

  A new macro, ~ilc[spec-mv-let], supports speculative and parallel execution
  in the parallel version, courtesy of David Rager.

  Among the enhancements for the HONS version (~pl[hons-and-memoization]) are
  the following.
  ~bq[]
  ~ilc[Memoize]d functions may now be traced (~pl[trace$]).  Thanks to Sol
  Swords for requesting this enhancement.

  ~ilc[Memoize-summary] and ~ilc[clear-memoize-statistics] are now
  ~c[:]~ilc[logic] mode functions that return ~c[nil].  Thanks to Sol Swords
  for this enhancement.

  ~ilc[Memoize] is now explicitly illegal for constrained functions.  (Already
  such memoization was ineffective.)

  A new keyword argument, ~c[:AOKP], controls whether or not to allow
  memoization to take advantage of attachments; ~pl[memoize] and for relevant
  background, ~pl[defattach].

  ~ilc[Memoize] is now illegal by default for ~c[:]~ilc[logic] mode functions
  that have not had their guards verified.  ~l[memoize] (keyword
  ~c[:ideal-okp]) and ~pl[acl2-defaults-table] (key ~c[:memoize-ideal-okp]) for
  and explanation of this restriction and how to avoid it.

  ~il[History] commands such as ~c[:]~ilc[pe] and ~c[:]~ilc[pbt] now display
  ``~c[M]'' or ``~c[m]'' to indicate memoized functions.  ~l[pc].
  ~eq[]

  ~/~/")

(deflabel |NOTE-4-3(R)|
  :doc
  ":Doc-Section release-notes

  ACL2 Version  4.3(r) (July, 2011) Notes~/

  ~/

  Please ~pl[note-4-3] for changes in Version  4.3 of ACL2.

  ~/
  ")

(deflabel note-5-0

; Total release note items: 125.

; Improved comments about step-limits.

; Here is a slightly simplified version of the example sent to us by Warren
; Hunt for the defstobj bug involving, quoting the :doc below, "excessively
; restrictive type declarations".

;   (defstobj x86-32
;     (mem :type (array (unsigned-byte 8) (4294967296)) ;; 2^32
;          :initially 0
;          :resizable nil)
;     :inline t)

; It has for some time been illegal to do guard verification inside a
; non-trivial encapsulate for an exported function whose body or guard depends
; on signature functions; see function bogus-exported-compliants.  However, we
; made exceptions both for constrained functions and for non-executable
; functions.  The latter exception has been removed; after all, if the guard
; obligations aren't theorems of the post-encapsulate theory, it's a bit odd to
; allow them.

; For more about the increase in speed for ACL2 array reads, see the Essay on
; Array Caching.

; Regarding the fix to getenv$ etc. mentioned below: we also changed
; throw-nonexec-error, get-output-stream-string$-fn, and set-debugger-enable-fn
; just a bit, just to be safe, though we're not aware of a soundness bug for
; those functions.

; We removed a #-acl2-loop-only shortcut in plist-worldp, which remains as a
; comment in that function.

; Modified the check in acl2-check.lisp related to
; *common-lisp-specials-and-constants*, so that it no longer causes an error
; when attempting to build with ECL (which bound the symbol
; COMMON-LISP:FUNCTION), in case this is something we try in the future.

; The soundness bug for with-live-state has been recorded in a comment where
; that macro is defined.

; (Lispworks mods courtesy of Martin Simmons) Changed calls of
; special-form-or-op-p to expand to calls of special-operator-p.  Changed
; 'cl::getenv to 'hcl::getenv and changed 'cl::setenv to 'hcl::setenv.
; Changed lisp::quit to lispworks:quit in acl2.lisp.

; Modified the DOC targets (and HTML, etc. targets under it) so that one can
; specify ACL2=<your_acl2> on the command line with the make command or as an
; environment variable.  If ACL2 is not thus specified, then PREFIX and
; ACL2_SUFFIX will be used, as before.

; The time-reporting bug was in initialize-summary-accumulators, which had
; called main-timer without accumulating times.  The following simple example
; clearly illustrates the bug, as the summary time should match the runtime
; reported by time$.

; (defun fib (n) (if (or (zp n) (eql n 1)) 1 (+ (fib (- n 1)) (fib (- n 2)))))
; (time$ (progn (defun f1 (x) x)
;               (value-triple (length (make-list 10000000)))
;               (defun f2 (x) x)))

; Cleaned up special-form-or-op-p.

; Modified script saved for sbcl to double the control-stack-size, which
; allowed "make DOC" to complete.

; Fixed the function INDUCT so that the second argument of the inform-simplify
; call is less likely to have duplicates.

; Here we say a bit more about the "logical modeling of the ACL2 evaluator"
; mentioned below.  The function ev-fncall-rec is defined in the logic to be
; ev-fncall-rec-logical, but they did not actually match up.  For example, the
; result differed if ev-fncall-rec below was changed to ev-fncall-rec-logical.
;   (defstub foo () t)
;   (defttag t)
;   (remove-untouchable ev-fncall-rec t)
;   (ev-fncall-rec 'foo nil (w state) 100000 nil nil nil nil t)
; We found several such discrepancies and have fixed them.

; After remarks from Gary Byers, improved fgetprop a bit by using defconstant
; to introduce *current-acl2-world-key* and by using symbol-value for
; 'ACL2_GLOBAL_ACL2::CURRENT-ACL2-WORLD.

; Here is an example showing the "security hole" mentioned in these release
; notes, below.

;   ;;;;; File foo.lisp ;;;;;
;   (in-package "ACL2")
;
;   (defun foo (x)
;     x)
;
;   ;;;;; File foo.acl2x ;;;;;
;   ((1 . (defun foo (x) (cons x x))))
;
;   ;;;;; File top.lisp ;;;;;
;   (in-package "ACL2")
;
;   (include-book "foo")
;
;   (defthm foo-cons
;     (equal (foo x)
;            (cons x x)))
;
;   ;;;;; File top2.lisp ;;;;;
;   (in-package "ACL2")
;
;   (include-book "top")
;
;   (defthm ouch
;     nil
;     :hints (("Goal" :in-theory (disable foo-cons)
;              :use foo-cons))
;     :rule-classes nil)
;
;   ;;;;; File cert.lsp ;;;;;
;
;   (certify-book "foo" 0 t :acl2x t)
;   (u)
;   (certify-book "top")
;   (u)
;   (certify-book "foo" 0 t)
;   (u)
;   (certify-book "top2")
;
;   ;;;;; Now evaluate (ld "cert.lsp") ;;;;;

; We improved the function chk-ld-skip-proofsp to cause a soft error instead of
; a hard error.  For this purpose, we moved set-ld-skip-proofsp and replaced
; its use in axioms.lisp with an f-put-global,

; Modified observation-cw for ACL2(p).  Also, in support of ACL2(p) but also of
; general applicability, we modified io? in the commentp=t case to check that
; the form can be translated, at least by default (see new argument
; chk-translatable).

; The previous definition of cons-term1 has been eliminated, and cons-term2 has
; been renamed to cons-term1.  Thanks to Harsh Raju Chamarthi for pointing out
; the dead code that led to this change.

; We tweaked the implementation of defconst to support fast-alists, based on
; discussions with David Rager and Jared Davis.  To see relevant code, search
; for "Remark on Fast-alists" and also see the new call (remprop k
; 'redundant-raw-lisp-discriminator) in the const-restore-ht case of function
; hcomp-restore-defs.

; The new macro ill-formed-certificate-er is called to provide much more
; information than had been provided by use of the constant,
; *ill-formed-certificate-msg*.

; The change "All trust tags are now in the ~il[keyword] package" was made in
; support of provisional certification.  Sol Swords sent an example in which
; the Complete operation caused an error, the reason being that an unknown
; package was being used in the post-alist in the certificate file.

; Here is the example promised in :doc note-5-0 and in a comment about
; :SKIPPED-PROOFSP in certify-book-fn, regarding "Fixed a soundness bug based
; on the use of ~ilc[skip-proofs] ...."  First consider the following two
; books.

;   -- foo.lisp --

;   (in-package "ACL2")
;   (defun f1 (x) x)
;   ; (defthm bad nil :rule-classes nil)

;   -- bar.lisp --

;   (in-package "ACL2")
;   (local (include-book "foo"))
;   (defthm bad nil :rule-classes nil)

;   --

; Notice that foo.lisp ends in a commented-out form.  Now proceed as follows.
; If you prefer, you can eliminate the two calls of set-ld-skip-proofsp by,
; instead, wrapping a skip-proofs around the defthm.

;  (set-ld-skip-proofsp t state)
;  (defthm bad nil :rule-classes nil)
;  (set-ld-skip-proofsp nil state)
;  (certify-book "foo" 1 t :skip-proofs-okp t)
;  Uncomment out the defthm form in foo.lisp.
;  (ubt! 1)
;  (certify-book "foo" t)
;  (ubt! 1)
;  (certify-book "bar")

; You will see no warnings when certifying bar, and its certificate will show
; no trace of skip-proofs.

; Among the changes made to support congruent stobjs are latching of stobjs in
; raw-ev-fncall; the use of FLET in :print-gv output; and a change to
; processing of rewrite rules such that interpret-term-as-rewrite-rule1 now has
; the previous functionality of interpret-term-as-rewrite-rule, except for
; removing lambdas.

; Improved the :use hint warning by adding the goal name and pointing to a new
; :doc topic, using-enabled-rules.  Thanks to David Rager for pointing out how
; the existing warning could be improved.

; To support guard verification (in new books distributed in books/system/):
; Updated several guards and termination conditions (to endp).  Functions
; affected include sublis-var and sublis-var-lst and some ancestors; subst-var
; and subst-var-lst; and subst-expr1 and subst-expr1-lst and subst-expr.
; Thanks to David Rager for his part in this effort, including his addition of
; books that verify guards for these functions.

; Fixed ill-guarded calls of eq and union-eq in non-recursive-fnnames and
; non-recursive-fnnames-lst, respectively.

; Here is an example of the proof-checker "bug that could result in duplicate
; goal names in the case of forced hypotheses".  The log shown was created by
; using Version_4.3 to execute the six events just below.

;   (defstub f (x) t)
;   (defstub g (x) t)
;   (defstub h (x) t)
;   (defstub k (x) t)
;
;   (defaxiom prop
;     (implies (and (f x) (g x))
;              (h x)))
;
;   (defaxiom prop-ts
;     (implies (force (k x)) (f x))
;     :rule-classes :type-prescription)
;
; The log promised above is then as follows.
;
;   ACL2 !>(verify (h x))
;   ->: (r 1)
;   Rewriting with PROP.
;   --NOTE-- Using the following runes in addition to the indicated rule:
;     ((:TYPE-PRESCRIPTION PROP-TS)).
;   NOTE (forcing):  Creating one new goal due to forcing assumptions.
;
;   Creating two new goals:  (MAIN . 1) and (MAIN . 1).
;
;   The proof of the current goal, MAIN, has been completed.  However,
;   the following subgoals remain to be proved:
;     (MAIN . 1) and (MAIN . 1).
;   Now proving (MAIN . 1).
;   ->: goals
;
;   (MAIN . 1)
;   (MAIN . 1)
;   ->:

; Made a small change to newline printing in ubt-ubu-query and ubt-ubu-fn1 in
; support of the change to top-level (to avoid a bogus newline).

; While fixing the cons-term bug, we improved the efficiency of kwote a bit,
; avoiding some unnecessary consing.

; Improved several :doc topics by clarifying the role of type declarations in
; specifying the guard of a function.

; We eliminated some needless property names from renew-name/overwrite, and
; added a comment clarifying why it is only called on function symbols.

; Added new severity option HARD?! for er, so that the guard of theory-fn can
; be t (avoiding the expense of doing the theory-namep check twice).

; We now print notes to use :a! and see :DOC set-evisc-tuple when evaluating
; :brr t.  Thanks to Robert Krug for suggesting this improvement.

; Eliminated warnings at build time for CLISP due to two definitions of the
; same function from both defproxy and encapsulate, by marking those
; encapsulate forms (in boot-strap-pass-2.lisp) as #+acl2-loop-only.

  :doc
  ":Doc-Section release-notes

  ACL2 Version  5.0 (August, 2012) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  4.3 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, changes at the system level and to distributed
  books, Emacs support, and experimental versions.  Each change is described in
  just one category, though of course many changes could be placed in more than
  one category.

  NOTE: ACL2 is now distributed under Version 2 of the GNU General Public
  License.  [Added later: The license has changed since Version_5.0.  See
  LICENSE.]  Formerly, any later version had been acceptable.  Moreover, books
  are no longer distributed from a University of Texas website, but rather,
  from Google Code at ~url[http://code.google.com/p/acl2-books/downloads/].

  ~st[CHANGES TO EXISTING FEATURES]

  A fatal error now occurs if environment variable ~c[ACL2_CUSTOMIZATION] has a
  value other than ~c[NONE] or the empty string, but is not the name of an
  existing file.  Thanks to Harsh Raju Chamarthi for requesting such a change.

  Functions ~c[read-acl2-oracle] (and ~c[read-acl2-oracle@par]),
  ~c[read-run-time], and ~c[main-timer] are no longer untouchable
  (~pl[remove-untouchable]).

  We now avoid certain duplicate conjuncts in the ~il[constraint] stored for
  ~ilc[encapsulate] ~il[events].  For example, the constraint stored for the
  following event formerly included ~c[(EQUAL (FOOP (CONS X Y)) (FOOP Y))] and
  ~c[(BOOLEANP (FOOP X))] twice each, but no more.
  ~bv[]
  (encapsulate
   ((foop (x) t))
   (local (defun foop (x) (declare (ignore x)) t))
   (defthm foop-constraints
     (and (booleanp (foop x))
          (equal (foop (cons x y)) (foop y)))
     :rule-classes
     ((:type-prescription :corollary (booleanp (foop x)))
      (:rewrite :corollary (equal (foop (cons x y)) (foop y))))))
  ~ev[]

  The ~c[:]~ilc[guard] for a constrained function (~pl[signature]) may now
  mention function symbols introduced in the same ~ilc[encapsulate] event that
  introduces that function.  Thanks to Nathan Wetzler for a helpful discussion
  leading to this improvement.

  The test for redundancy (~pl[redundant-events]) of ~ilc[encapsulate]
  ~il[events] has been improved in cases involving redefinition
  (~pl[ld-redefinition-action]).  Thanks to Jared Davis for providing the
  following example, which illustrates the problem.
  ~bv[]
    (redef!)

    (encapsulate ()
      (defun g (x)
        (+ 3 x)))

    (g 0) ; 3, as expected

    (encapsulate ()
      (defun g (x)
        (+ 4 x)))

    (g 0) ; 4, as expected

    ; Unfortunately, the following was flagged as redundant because it agreed
    ; with the first encapsulate above.  That has been fixed; now, it is
    ; recognized as not being redundant.
    (encapsulate ()
      (defun g (x)
        (+ 3 x)))
  ~ev[]

  The test for redundancy of ~ilc[defun] and ~ilc[defconst] events has been
  improved in the case that redefinition is active.  In that case, redundancy
  now additionally requires that the ``translated'' body is unchanged, i.e.,
  even after expanding macro calls and replacing constants (defined by
  ~ilc[defconst]) with their values.  Thanks to Sol Swords for requesting this
  enhancement, and to Jared Davis for pointing out a bug in a preliminary
  change.  ~l[redundant-events], in particular the ``Note About Unfortunate
  Redundancies''.  Note that this additional requirement was already in force
  for redundancy of ~ilc[defmacro] events.

  The macro ~ilc[defmacro-last] and the ~il[table] ~ilc[return-last-table] have
  been modified so that when they give special treatment to a macro ~c[mac] and
  its raw Lisp counterpart ~c[mac-raw], a call ~c[(return-last 'mac-raw ...)]
  can be made illegal when encountered directly in the top level loop, as
  opposed to inside a function body.  ~l[return-last].  Thanks to Harsh Raju
  Chamarthi for showing us an example that led us to make this improvement.

  We removed a barrier to admitting function definitions, as we explain using
  the following example.
  ~bv[]
  (defun foo (m state)
    (declare (xargs :stobjs state))
    (if (consp m)
        (let ((state (f-put-global 'last-m m state)))
          (foo (cdr m) state))
      state))
  ~ev[]
  Previously, ACL2 complained that it could not determine the outputs of the
  ~ilc[LET] form, as is necessary in order to ensure that ~ilc[STATE] is
  returned by it.  ACL2 now works harder to solve this problem as well as the
  analogous problem for ~ilc[MV-LET] and, more generally for
  ~ilc[mutual-recursion].  (The main idea is to reverse the order of processing
  the ~ilc[IF] branches if necessary.)  We thank Sol Swords for contributing a
  version of the above example and requesting this improvement.

  It is no longer the case that ~ilc[break-on-error] causes a Lisp break when
  encountering an error during translation of user input into internal
  (translated) form (~pl[term]).  The reason is that an improvement to the
  translation process, specifically the one described in the preceding
  paragraph, allows certain backtracking from ``errors'', which are intended to
  be silent rather than causing breaks into raw Lisp.  Thanks to Jared Davis
  for sending an example leading to this change.

  (CCL and SBCL only) When the host Lisp is CCL or SBCL, then since all
  functions are compiled, a ~ilc[certify-book] command will no longer load the
  newly-compiled file (and similarly for ~ilc[include-book] with argument
  ~c[:load-compiled-file :comp]).

  ~ilc[Set-write-acl2x] now returns an error triple and can take more values,
  some of which automatically allow including uncertified books when
  ~ilc[certify-book] is called with argument :acl2x t.

  The environment variable ~c[COMPILE_FLG] has been renamed
  ~c[ACL2_COMPILE_FLG]; ~pl[certify-book].

  The macros ~ilc[defthmd] and ~ilc[defund] no longer return an error triple
  with value ~c[:SKIPPED] when proofs are being skipped.  Rather, the value
  returned is the same as would be returned on success when proofs are not
  skipped.

  For those who use ~ilc[set-write-acl2x]: now, when ~ilc[certify-book] is
  called without a ~c[:ttagsx] argument supplied, then the value of ~c[:ttagsx]
  defaults to the (explicit or default) value of the ~c[:ttags] argument.

  The ~c[:]~ilc[pl] and ~c[:]~ilc[pl2] ~il[command]s can now accept ~il[term]s
  that had previously been rejected.  For example, the command
  ~c[:pl (member a (append x y))] had caused an error, but now it works as one
  might reasonably expect, treating ~ilc[member] as ~ilc[member-equal]
  (~pl[equality-variants] for relevant background).  Thanks to Jared Davis for
  reporting this problem by sending the above example.

  We have eliminated some hypotheses in built-in ~il[rewrite] rules
  ~c[characterp-nth] and ~c[ordered-symbol-alistp-delete-assoc-eq].

  Added the symbols ~ilc[f-get-global], ~ilc[f-put-global], and
  ~ilc[state-global-let*] to ~c[*acl2-exports*].

  Added to the ~il[guard]s of ~ilc[push-untouchable] and
  ~ilc[remove-untouchable] the requirement that the second argument must be a
  Boolean.  Thanks to Jared Davis for sending an example that led to this
  change.

  The built-in function ~c[string-for-tilde-@-clause-id-phrase] has been put
  into ~c[:]~ilc[logic] mode and had its guards verified, as have some
  subsidiary functions.  A few new rules have been added in support of this
  work; search for ~c[string-for-tilde-@-clause-id-phrase] in ACL2 source file
  ~c[boot-strap-pass-2.lisp] if interested.  Thanks to David Rager for
  contributing an initial version of this improvement.

  All trust tags are now in the ~il[keyword] package.  The ~ilc[defttag] event
  may still take a symbol in an arbitrary package, but the trust tag created
  will be in the keyword package (with the same ~ilc[symbol-name] as the symbol
  provided).  Similarly, non-~c[nil] symbols occurring in the ~c[:ttags]
  argument of an ~ilc[include-book] or ~ilc[certify-book] command will be
  converted to corresponding keywords.  ~l[defttag].

  There have been several changes to ~il[gag-mode].  It is now is initially set
  to ~c[:goals], suppressing most proof commentary other than key checkpoints;
  ~pl[set-gag-mode].  (As before, ~pl[pso] for how to recover the proof
  output.)  Also, top-level induction schemes are once again printed when
  gag-mode is on, though these as well as printing of guard conjectures can be
  abbreviated (``eviscerated'') with a new ~il[evisc-tuple];
  ~pl[set-evisc-tuple], in particular the discussion there of ~c[:GAG-MODE].
  Finally, the commentary printed within ~il[gag-mode] that is related to
  ~il[forcing-round]s is now less verbose.  Thanks to Dave Greve and David
  Rager for discussions leading to the change in the printing of induction
  schemes under gag-mode; thanks to Warren Hunt for an email that led us to
  similar handling for printing of guard conjectures; and thanks to Robert Krug
  for a suggestion that led us to restore, in abbreviated form, important
  information about the sources of forcing round goals.

  An error now occurs if ~ilc[ld] is called while loading a compiled book.
  ~l[calling-ld-in-bad-contexts].  Thanks to David Rager for reporting a
  low-level assertion failure that led us to make this change.

  The ~il[proof-checker] interactive loop is more robust: most errors will
  leave you in that loop, rather than kicking you out of the proof-checker and
  thus back to the main ACL2 read-eval-print loop.  Thanks to David Hardin for
  suggesting this improvement in the case of errors arising from extra right
  parentheses.

  The summary at the end of a proof now prints the following note when
  appropriate:
  ~bv[]
  [NOTE: A goal of NIL was generated.  See :DOC nil-goal.]
  ~ev[]
  ~l[nil-goal].

  Improved ~ilc[dmr] to show the function being called in the case of
  explicit evaluation: ``~c[(EV-FNCALL function-being-called)]''.

  It is now permitted to bind any number of ~il[stobjs] to themselves in the
  bindings of a ~ilc[LET] expression.  But if any stobj is bound to other than
  itself in ~ilc[LET] bindings, then there still must be only one binding in
  that ~c[LET] expression.  The analogous relaxation holds for ~ilc[LAMBDA]
  expressions.  Thanks to Sol Swords for requesting such a change, which was
  needed for some code generated by macro calls.

  The macro ~ilc[top-level] now returns without error; ~l[top-level].
  Formerly, this macro always returned an error triple ~c[(mv t .. state)],
  which meant that normal calls of ~ilc[ld] would stop after encountering a
  call of ~c[top-level].  Thanks to Jared Davis for bringing this issue to our
  attention.

  It is no longer the case that when you specify ~ilc[xargs] keyword
  ~c[:non-executable t] in a ~ilc[defun] form rather than using ~ilc[defun-nx],
  then the form of the body need match only the shape
  ~c[(prog2$ (throw-nonexec-error ... ...) ...)].  We now require that the body
  of the definition of a function symbol, ~c[fn], with formals ~c[(x1 ... xk)],
  be of the form ~c[(prog2$ (throw-nonexec-error 'fn (list x1 ... xk)) ...)].
  This fixes the following odd behavior, which could be considered a bug.
  Consider a book that contains the following two events.
  ~bv[]
  (defun foo (x)
    (declare (xargs :guard t :non-executable t :mode :logic))
    (prog2$ (throw-nonexec-error 'bar (list x))
            (cons 3 x)))
  (defn h (x)
    (foo x))
  ~ev[]
  After certifying this book and then including it in a new session, the
  behavior occurred that is displayed below; notice the mention of ~c[BAR].
  However, if the two forms were submitted directly in the loop, then the error
  message had mentioned ~c[FOO] instead of ~c[BAR].  This discrepancy has been
  eliminated, by rejecting the proposed definition of ~c[foo] because the name
  in the first argument of ~c[throw-nonexec-error] was ~c['bar] where now it
  must be ~c['foo].
  ~bv[]
  ACL2 !>(h 3)


  ACL2 Error in TOP-LEVEL:  ACL2 cannot ev the call of undefined function
  BAR on argument list:

  (3)

  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.

  ACL2 !>
  ~ev[]

  A tautology checker used in the ACL2 sources (function ~c[if-tautologyp]) has
  been limited somewhat in the effort it makes to recognize a tautology.  While
  we expect it to be rare for the effect of this change to be noticeable, we
  thank Sol Swords for sending us an example that motivated this change: a
  ~il[guard] verification that took about 5 seconds in Version_4.3 now takes,
  on the same machine, about 0.07 seconds.

  The behavior of backquote (~c[`]) has been changed slightly to be compatible
  with its behavior in raw Lisp.  The change is to allow the use of
  comma-atsign (~c[,@]) at the end of a list, as in the following example.
  ~bv[]
  (let ((x 3) (y 2) (z 7)) `(,x ,y ,@z))
  ~ev[]
  Formerly, evaluation of this form had caused a guard violation in the ACL2
  loop unless guard-checking was off (i.e., ~ilc[set-guard-checking] was
  invoked with ~c[nil] or ~c[:none]), in which case it returned ~c[(3 2)].
  But we observed evaluation of this form to return ~c[(3 2 . 7)] in every host
  Lisp on which ACL2 runs (Allegro CL, CCL, CLISP, CMUCL, GCL, LispWorks, and
  SBCL).  Now, ACL2 behaves like these Lisps.

  A call of the ~ilc[theory] macro had previously returned ~c[nil] when applied
  to other than the name of name of a previously executed ~ilc[deftheory]
  event.  Now, a hard error occurs.

  The ~il[table] ~c[binop-table] has been replaced by the table
  ~ilc[untrans-table].  However, ~ilc[add-binop] and ~ilc[remove-binop]
  continue to have the same effect as before.  ~l[add-macro-fn], which is a new
  feature discussed below.

  The function ~ilc[booleanp] is now defined using ~ilc[eq] instead of
  ~ilc[equal], which may increase its efficiency.  Thanks to Jared Davis for
  this change.

  For pairs ~c[(key . val)] in the ~ilc[macro-aliases-table], there had been a
  requirement that ~c[val] is a known function symbol.  Now, it only needs to
  be a symbol.  (This change was made to support the new feature,
  ~ilc[defun-inline], described elsewhere in these release notes.)

  ~st[NEW FEATURES]

  A new ``tau system'' provides a kind of ``type checker.''  ~l[tau-system].
  Thanks to Dave Greve for supplying a motivating example (on which this system
  can provide significant speedup), and to Sol Swords for sending a very
  helpful bug report on a preliminary implementation.

  Users may now arrange for additional summary information to be printed at the
  end of ~il[events].  [Note added at Version_6.1: Formerly we pointed here to
  ~c[print-summary-user], but now, ~pl[finalize-event-user]; also
  ~pl[note-6-1]].  Thanks to Harsh Raju Chamarthi for requesting this feature
  and participating in a design discussion.

  A new, advanced ~il[proof-checker] command, ~c[geneqv], shows the generated
  equivalence relation at the current subterm.  Thanks to Dave Greve for an
  inquiry leading to this enhancement.

  A new reader macro, ~c[#u], permits the use of underscore characters in a
  number.  ~l[sharp-u-reader].  Thanks to Jared Davis for requesting this
  capability.

  New ~il[proof-checker] commands ~c[pl] and ~c[pr] provide interfaces to the
  ACL2 commands ~c[:]~ilc[pl] and ~c[:]~ilc[pr], respectively.  These can be
  useful if you want to see trivially-proved hypotheses, as now clarified in
  the ~il[proof-checker] documentation for its ~c[show-rewrites] command.
  ~l[proof-checker-commands].  Thanks to Pete Manolios for suggesting such
  clarification and capability.

  It is now legal to call ~il[non-executable] functions without the usual
  ~il[signature] restrictions imposed on executable code.  For example,
  the third event below was not admissible, but now it is.
  ~bv[]
  (defstobj foo fld)
  (defun-nx id (x)
    x)
  (defun f (foo)
    (declare (xargs :stobjs foo :verify-guards nil))
    (cons 3 (id foo)))
  ~ev[]
  Thanks to Jared Davis for requesting this enhancement, in particular for
  calling non-executable functions in the ~c[:logic] part of an ~ilc[mbe]
  call.  Here is Jared's example, which is admissible now but formerly was
  not.
  ~bv[]
  (defstobj foo (fld))
  (defun-nx my-identity (x) x)
  (defun my-fld (foo)
    (declare (xargs :stobjs foo))
    (mbe :logic (my-identity foo)
         :exec (let ((val (fld foo)))
                 (update-fld val foo))))
  ~ev[]

  A new macro, ~ilc[non-exec], allows the use of non-executable code, for
  example inside ordinary function definitions.  Thanks to Sol Swords for
  requesting this enhancement.

  A new ``provisional certification'' process is supported that can allow
  ~il[books] to be certified before their included sub-books have been
  certified, thus allowing for potentially much greater `make'-level
  parallelism.  ~l[provisional-certification].  Thanks to Jared Davis for
  requesting this feature and for helpful discussions, based in part on
  rudimentary provisional certification schemes that he developed first at
  Rockwell Collins and later for his `Milawa' project.  Also, thanks to Jared
  and to Sol Swords for testing this feature and for providing a fix for a bug
  in a preliminary implementation, and thanks to Sol for providing performance
  feedback and a crucial suggestion that led to an improved implementation.

  Event summaries now show the names of events that were mentioned in
  ~il[hints] of type ~c[:use], ~c[:by], or ~c[:clause-processor].
  ~l[set-inhibited-summary-types].  Thanks to Francisco J. Martin Mateos for
  requesting such an enhancement (actually thanks to the community, as his
  request is the most recent but this has come up from time to time before).

  ACL2 now stores a data structure representing the relation ``Event A is used
  in the proof of Event B.''  ~l[dead-events], which explains this data
  structure and mentions one application: to identify dead code and unused
  theorems.  Thanks to Shilpi Goel for requesting such a feature and for
  helpful feedback.

  A new ~il[documentation] topic provides a guide to programming with state;
  ~pl[programming-with-state].  Thanks to Sarah Weissman for suggesting that
  such a guide might be useful, and to David Rager for helpful feedback on a
  preliminary version.  There also has been some corresponding reorganization
  of the documentation as well as creation of additional documentation (e.g.,
  ~pl[state-global-let*]).  Now, most built-in functions and macros commonly
  used in programs (as opposed to ~il[events] like ~ilc[defun], for example)
  are subtopics of a new topic ~-[] ~pl[acl2-built-ins] ~-[] which is a
  subtopic of ~il[programming], a topic that in turn has considerably fewer
  direct subtopics than before.

  It is now possible to bind extra variables in a ~c[:USE] hint, thus avoiding
  the error message: ``The formula you wish to instantiate, ..., mentions only
  the variable(s) ...''.  ~l[lemma-instance], in particular the discussion of
  keyword ~c[:extra-bindings-ok].  Thanks to Sol Swords for requesting such an
  enhancement.

  The function ~c[read-object-suppress] is like ~c[read-object] except that it
  avoids errors and discards the value read.  ~l[io].

  A ~il[stobj] may now be passed as an argument where another stobj is expected
  if the two are ``congruent''.  ~l[defstobj], in particular, its discussion of
  the new ~c[:congruent-to] keyword of ~c[defstobj].  Thanks to Sol Swords for
  requesting this enhancement and for useful discussions contributing to its
  design.

  A new top-level utility has been provided that shows the assembly language
  for a defined function symbol; ~pl[disassemble$].  Thanks to Jared Davis for
  requesting such a utility and to Shilpi Goel for pointing out an
  inconvenience with the initial implementation.  Note that it uses the
  distributed book ~c[books/misc/disassemble.lisp], which users are welcome to
  modify (see ~url[http://www.cs.utexas.edu/users/moore/acl2/]).

  The macro ~c[set-accumulated-persistence] is an alias for
  ~ilc[accumulated-persistence].  Thanks to Robert Krug for suggesting this
  addition.

  A new ~il[documentation] topic lists lesser-known and advanced ACL2 features,
  intended for those with prior ACL2 experience who wish to extend their
  knowledge of ACL2 capabilities.  ~l[advanced-features].  Thanks to Warren
  Hunt and Anna Slobodova for requesting such information.

  A new macro, ~ilc[deftheory-static], provides a variant of ~ilc[deftheory]
  such that the resulting theory is the same at ~ilc[include-book] time as it
  was at ~ilc[certify-book] time.  Thanks to Robert Krug for helpful
  discussions on this new feature and for updating his ~c[books/arithmetic-5/]
  distributed books to use this feature.

  A new event, ~ilc[defabsstobj], provides a new way to introduce
  single-threaded objects (~pl[stobj] and ~pl[defstobj]).  These so-called
  ``abstract ~il[stobj]s'' permit user-provided logical definitions for
  primitive operations on stobjs, for example using an alist-based
  representation instead of a list-based representation for array fields.
  Moreover, the proof obligations guarantee that the recognizer is preserved;
  hence the implementation avoids executing the recognizer, which may be an
  arbitrarily complex invariant that otherwise would be an expensive part of
  ~il[guard] checks.  Thanks to Warren Hunt for a request leading us to design
  and implement this new feature, and thanks to Rob Sumners for a request
  leading us to implement a related utility, ~ilc[defabsstobj-missing-events].
  ~l[defabsstobj].  Also thanks to Sol Swords for sending an example exhibiting
  a bug in the initial implementation, which has been fixed.

  A new command, ~c[:psof <filename>], is like ~c[:pso] but directs proof
  replay output to the specified file.  For large proofs, ~c[:]~ilc[psof] may
  complete much more quickly than ~c[:]~ilc[pso].  ~pl[psof].  More generally,
  a new utility, ~ilc[wof] (an acronym for ``With Output File''), directs
  standard output and proofs output to a file; ~pl[wof].

  The new macro ~ilc[defnd] defines a function with ~c[:]~ilc[guard] ~c[t] and
  ~il[disable]s that function, in analogy to how ~ilc[defund] defines with
  ~ilc[defun] and then ~il[disable]s.  Thanks to Shilpi Goel for requesting
  this feature.

  The ~c[:]~ilc[pl2] command now shows ~c[:]~ilc[linear] rules; and a new
  ~il[proof-checker] command, ~c[show-linears] (equivalently, ~c[sls]), is an
  analogue of the ~il[proof-checker] ~c[show-rewrites] (~c[sr]) command, but
  for ~il[linear] rules.  Thanks to Shilpi Goel for requesting this new
  proof-checker command.  Finally, a corresponding new proof-checker command,
  ~c[apply-linear] (~c[al]), is an analogue of the ~il[proof-checker]
  ~c[rewrite] (~c[r]) command, but for ~il[linear] rules.

  The macros ~ilc[add-macro-fn] and ~ilc[remove-macro-fn] replace macros
  ~ilc[add-binop] and ~ilc[remove-binop], respectively, though the latter
  continue to work.  The new macros allow you to decide whether or not to
  display calls of binary macros as flat calls for right-associated arguments,
  e.g., ~c[(append x y z)] rather than ~c[(append x (append y z))].
  ~l[add-macro-fn].

  It is now possible to request that the host Lisp compiler inline calls of
  specified functions, or to direct that the host Lisp compiler not inline such
  calls.  ~l[defun-inline] and ~pl[defun-notinline].  We thank Jared Davis for
  several extensive, relevant conversations, and for finding a bug in a
  preliminary implementation.  We also thank others who have engaged in
  discussions with us about inlining for ACL2; besides Jared Davis, we recall
  such conversations with Rob Sumners, Dave Greve, and Shilpi Goel.

  ~st[HEURISTIC IMPROVEMENTS]

  Reading of ACL2 ~ilc[arrays] (~pl[aref1], ~pl[aref2]) has been made more
  efficient (as tested with CCL as the host Lisp) in the case of consecutive
  repeated reads of the same named array.  Thanks to Jared Davis and Sol Swords
  for contributing this improvement.

  Slightly modified the induction schemes stored, so that calls of so-called
  ``guard-holders'' (such as ~ilc[mbe] and ~ilc[prog2$] ~-[] indeed, any call
  of ~ilc[return-last] ~-[] and ~ilc[the]) are expanded away.  In particular,
  calls of equality variants such as ~ilc[member] are treated as their
  corresponding function calls, e.g., ~ilc[member-equal];
  ~pl[equality-variants].  Guard-holders are also now expanded away before
  storing ~il[constraint]s for ~ilc[encapsulate] ~il[events], which can
  sometimes result in simpler constraints.

  Improved the performance of ~ilc[dmr] (technical note: by modifying raw Lisp
  code for function ~c[dmr-flush], replacing ~c[finish-output] by
  ~c[force-output]).

  We now avoid certain rewriting loops.  A long comment about this change,
  including an example of a loop that no longer occurs, may be found in source
  function ~c[expand-permission-result].

  Slightly strengthened ~il[type-set] reasoning at the level of literals (i.e.,
  top-level hypotheses and conclusions).  See the comment in ACL2 source
  function ~c[rewrite-atm] about the ``use of dwp = t'' for an example of a
  theorem provable only after this change.

  Strengthened the ability of ~il[type-set] reasoning to make deductions about
  terms being integers or non-integer rationals.  The following example
  illustrates the enhancement: before the change, no simplification was
  performed, but after the change, the conclusion simplifies to ~c[(foo t)].
  Thanks to Robert Krug for conveying the problem to us and outlining a
  solution.
  ~bv[]
  (defstub foo (x) t)
  (thm ; should reduce conclusion to (foo t)
   (implies (and (rationalp x)
                 (rationalp y)
                 (integerp (+ x (* 1/3 y))))
            (foo (integerp (+ y (* 3 x))))))
  ~ev[]

  ~st[BUG FIXES]

  Fixed a class of soundness bugs involving each of the following functions:
  ~ilc[getenv$], ~ilc[get-wormhole-status], ~ilc[cpu-core-count],
  ~ilc[wormhole-p], ~ilc[random$], ~c[file-write-date$], and
  ~c[serialize-read-fn], and (for the HONS version of ACL2)
  ~ilc[clear-memoize-table] and ~ilc[clear-memoize-tables] as well as (possible
  soundness bug) ~c[serialize-write-fn].  For example, we were able to admit
  the following events, but that is no longer the case (neither for ~c[getenv$]
  as shown, nor analogously for other functions listed above).
  ~bv[]
  (defthm not-true
    (stringp (cadr (getenv$ \"PWD\" (build-state))))
    :rule-classes nil)

  (defthm contradiction
    nil
    :hints ((\"Goal\"
             :in-theory (disable (getenv$))
             :use not-true))
    :rule-classes nil)
  ~ev[]

  Fixed a soundness bug involving ~c[with-live-state], which could cause an
  error in the use of ~ilc[add-include-book-dir] or
  ~ilc[delete-include-book-dir] in a book or its ~il[portcullis] commands.
  ~l[with-live-state], as the documentation for this macro has been updated; in
  particular it is now untouchable (~pl[remove-untouchable]) and is intended
  only for system hackers.  Thanks to Jared Davis for reporting a bug in the
  use of ~ilc[add-include-book-dir] after our first attempt at a fix.

  Fixed a soundness bug based on the use of ~ilc[skip-proofs] together with the
  little-used argument ~c[k=t] for ~ilc[certify-book].  An example proof of
  ~c[nil] appears in a comment in the ACL2 sources, in
  ~c[(deflabel note-5-0 ...)].

  Fixed a soundness bug that allowed users to define new ~il[proof-checker]
  primitive commands.  Before this fix, a book proving ~c[nil] could be
  certified, as shown in a comment now in the introduction of the ~il[table]
  ~c[pc-command-table] in source file ~c[proof-checker-a.lisp].

  (Technical change, primarily related to ~ilc[make-event]:) Plugged a security
  hole that allowed ~il[books]' ~il[certificate]s to be out-of-date with
  respect to ~ilc[make-event] expansions, but not recognized as such.  The
  change is to include the so-called expansion-alist in the certificate's
  checksum.  An example appears in a comment in the ACL2 sources, in
  ~c[(deflabel note-5-0 ...)].

  Fixed a bug in ~il[guard] verification due to expanding calls of primitives
  when translating user-level terms to internal form, so called ``translated
  terms'' (~pl[term]).  While we have not observed a soundness hole due to this
  bug, we have not ruled it out.  Before the bug fix, the following event was
  admissible, as guard verification succeeded (but clearly should not have).
  ~bv[]
  (defun f ()
    (declare (xargs :guard t))
    (car (identity 3)))
  ~ev[]
  For those who want details about this bug, we analyze how ACL2 generates
  ~il[guard] proof obligations for this example.  During that process, it
  evaluates ground subexpressions.  Thus, ~c[(identity '3)] is first simplified
  to ~c['3]; so a term must be built from the application of ~c[car] to ~c['3].
  Guard-checking is always turned on when generating guard proof obligations,
  so now, ACL2 refuses to simplify ~c[(car '3)] to ~c['nil].  However, before
  this bug fix, when ACL2 was building a term by applying ~c[car] to argument
  ~c['3], it did so directly without checking guards; source code function
  ~c[cons-term] is `smart' that way.  After the fix, such term-building
  reduction is only performed when the primitive's guard is met.

  While calls of many event macros had been prohibited inside executable code,
  others should have been but were not.  For example, the following was
  formerly allowed.
  ~bv[]
  (defun foo (state)
    (declare (xargs :mode :program :stobjs state))
    (add-custom-keyword-hint :my-hint (identity nil)))
  (foo state) ; Caused hard raw Lisp error!
  ~ev[]
  Thus, several event macros (including for example
  ~ilc[add-custom-keyword-hint]) may no longer be called inside executable
  code.

  Fixed an assertion that could occur, for example, after reverting to prove
  the original goal by induction and generating a goal of ~c[NIL].  Thanks to
  Jared Davis for sending us a helpful example to bring this bug to our
  attention.

  It was possible for ~ilc[defstobj] to generate raw Lisp code with
  excessively restrictive type declarations.  This has been fixed.  Thanks to
  Warren Hunt for reporting this bug and sending an example that illustrates
  it.  ~l[stobj-example-2] for examples of such raw Lisp code; now, one finds
  ~c[(and fixnum (integer 0 *))] where formerly the type was restricted to
  ~c[(integer 0 268435455)].

  Fixed a bug in that was ignoring the use of ~c[:computed-hint-replacement] in
  certain cases involving a combination of computed hints and custom keyword
  hints.  Thanks to Robert Krug for reporting this bug and sending a very
  helpful example.

  Fixed a bug in the output from ~ilc[defattach], which was failing to list
  previous ~il[events] in the message about ``bypassing constraints that have
  been proved when processing the event(s)''.

  (GCL only) Fixed a bug in ~ilc[set-debugger-enable] (which was only a bug in
  GCL, not an issue for other host Lisps).

  Fixed ACL2 trace output to indent properly for levels above 99 (up to 9999).
  Thanks to Warren Hunt for bringing this bug to our attention.

  Fixed a bug in the reporting of times in event summaries ~-[] probably one
  that has been very long-standing!  The times reported had often been too
  small in the case of compound ~il[events], notably ~ilc[include-book].
  Thanks to everyone who reported this problem (we have a record of emails from
  Eric Smith and Jared Davis on this issue).

  Fixed a bug in ~c[:expand] ~il[hints], where the use of ~c[:lambdas] could
  prevent other parts of such a hint.  For example, the following invocation of
  ~ilc[thm] failed before this fix was made.
  ~bv[]
  (defund foo (x) (cons x x))
  (thm (equal (car (foo x)) x)
  :hints ((\"Goal\" :expand (:lambdas (foo x)))))
  ~ev[]

  Certain ``program-only'' function calls will now cause hard Lisp errors.
  (The rather obscure reason for this fix is to support logical modeling of the
  ACL2 evaluator.  A relevant technical discussion may be found in source
  function ~c[oneify-cltl-code], at the binding of variable
  ~c[fail_program-only-safe].)

  There was an unnecessary restriction that ~ilc[FLET]-bound functions must
  return all ~il[stobj]s among their inputs.  For example, the following
  definition was rejected because ~c[state] was not among the outputs of ~c[h].
  This restriction has been removed.
  ~bv[]
  (defun foo (state)
    (declare (xargs :stobjs state))
    (flet ((h (state) (f-boundp-global 'x state)))
      (h state)))
  ~ev[]

  We fixed a bug, introduced in the preceding release (Version  4.3), in the
  check for irrelevant formals (~pl[irrelevant-formals]).  That check had been
  too lenient in its handling of lambda (~il[LET]) expressions, for example
  allowing the following definition to be admitted in spite of its first formal
  parameter obviously being irrelevant.
  ~bv[]
  (defun foo (x clk)
    (if (zp clk)
        :diverge
      (let ((clk (1- clk)))
        (foo x clk))))
  ~ev[]

  Fixed a bug in the ~c[mini-proveall] target in ~c[GNUmakefile].  The fix
  includes a slight change to the ~c[:mini-proveall] ~il[command] (an extra
  event at the end).  Thanks to Camm Maguire for reporting this bug.

  Fixed a bug that occurred when ~ilc[certify-book] was called after using
  ~ilc[set-fmt-soft-right-margin] or ~ilc[set-fmt-hard-right-margin] to set a
  small right margin.

  Fixed ~ilc[set-inhibit-warnings] so that it takes effect for a subsequent
  ~ilc[include-book] event.  Thanks to Jared Davis and David Rager for queries
  that led to this fix.

  Hard Lisp errors are now avoided for certain ~c[:]~ilc[rewrite] rules: those
  whose ~il[equivalence] relation is other than ~c[equal] when the rule is
  originally processed, but is no longer a known equivalence relation when the
  rule is to be stored.  Thanks to Jared Davis for sending a useful example, a
  minor variant of which is included in a comment in source function
  ~c[interpret-term-as-rewrite-rule] (file ~c[defthm.lisp]).

  Fixed a bug in the ACL2 evaluator (source function ~c[raw-ev-fncall]), which
  was unlikely to be exhibited in practice.

  Fixed a hard Lisp error that could occur for ill-formed ~c[:]~ilc[meta]
  ~il[rule-classes], e.g., ~c[(:meta :trigger-fns '(foo))].

  It is now an error to include a ~il[stobj] name in the ~c[:renaming] alist
  (~pl[defstobj]).

  Some bogus warnings about non-recursive function symbols have been eliminated
  for rules of class ~c[:]~ilc[type-prescription].

  (Allegro CL host Lisp only) Fixed an obsolete setting of compiler variable
  ~c[comp:declared-fixnums-remain-fixnums-switch], which may have been
  responsible for intermittent (and infrequent) checksum errors encountered
  while including books during certification of the regression suite.

  Fixed a ~il[proof-checker] bug that could result in duplicate goal names in
  the case of forced hypotheses.  An example showing this bug, before the fix,
  appears in a comment in the ACL2 sources, in ~c[(deflabel note-5-0 ...)].

  We fixed a bug in a prover routine involved in ~il[type-set] computations
  involving linear arithmetic.  This bug has been around since at least as far
  back as Version_3.3 (released November, 2007).  We are not aware of any
  resulting unsoundness, though it did have the potential to weaken the prover.
  For example, the following is proved now, but was not proved before the bug
  was fixed.
  ~bv[]
  (thm
   (implies (and (rationalp x)
                 (rationalp y)
                 (integerp (+ (* 1/3 y) x)))
            (integerp (+ y (* 3 x))))
   :hints ((\"Goal\" :in-theory (disable commutativity-of-+))))
  ~ev[]

  Although all bets are off when using redefinition
  (~pl[ld-redefinition-action]), we wish to minimize negative effects of its
  use, especially raw Lisp errors.  The examples below had caused raw Lisp
  errors, but no longer.
  ~bv[]
  (defstobj st fld :inline t)
  (redef!)
  (defstobj st new0 fld)
  (u)
  (fld st) ; previously an error, which is now fixed

  ; Fresh ACL2 session:
  (redef!)
  (defun foo (x) x)
  (defmacro foo (x) `(quote ,x))
  (u)

  ; Fresh ACL2 session:
  (redef!)
  (defmacro foo (x) (cons 'list x))
  (defun foo (x) x)
  ~ev[]

  Fixed a bug that could cause hard Lisp errors in an ~ilc[encapsulate] event.
  Thanks to Sol Swords for sending an example that exhibited this bug.  Here is
  a simpler such example; the bug was in how it was checked whether the
  ~il[guard] for a guard-verified function (here, ~c[g]) depends on some
  function introduced in the ~il[signature] of the ~ilc[encapsulate] (here, the
  function ~c[f]).
  ~bv[]
  (encapsulate
   ((f (x) t))
   (local (defun f (x) (declare (xargs :guard t)) x))
   (defun g (x)
     (declare (xargs :guard (if (integerp x) (f x) t)))
     x))
  ~ev[]

  Fixed a bug in ~c[mfc-relieve-hyp] that we believe could prohibit its use on
  the last hypothesis.  Thanks to Sol Swords for reporting this bug and
  providing a fix.

  The syntax ~c[#!] (~pl[sharp-bang-reader]) was broken after a skipped
  readtime conditional.  For example, the following input line caused an
  error.
  ~bv[]
  #+skip #!acl2(quote 3)
  ~ev[]
  This bug has been fixed.

  Fixed a bug in the ~il[break-rewrite] utility, which was evidenced by error
  messages that could occur when dealing with free variables.  An example of
  such an error message is the following; we thank Robert Krug for sending us
  an example that produced this error and enabled us to produce a fix.
  ~bv[]
  HARD ACL2 ERROR in TILDE-@-FAILURE-REASON-PHRASE1:  Unrecognized failure
  reason, ((MEM-ARRAY . X86) (ADDR QUOTE 9)).
  ~ev[]

  We fixed an obscure bug that we believe could interfere with ~ilc[defproxy]
  because of an incorrect ~c[(declaim (notinline <function>))] form.

  ~st[CHANGES AT THE SYSTEM LEVEL AND TO DISTRIBUTED BOOKS]

  Improvements have been made related to the reading of characters.  In
  particular, checks are now done for ASCII encoding and for the expected
  ~ilc[char-code] values for ~c[Space], ~c[Tab], ~c[Newline], ~c[Page], and
  ~c[Rubout].  Also, an error no longer occurs with certain uses of
  non-standard characters.  For example, it had caused an error to certify a
  book after a single ~il[portcullis] ~il[command] of
  ~c[(make-event `(defconst *my-null* ,(code-char 0)))]; but this is no longer
  an issue.  Thanks to Jared Davis for helpful correspondence that led us to
  make these improvements.

  The character encoding for reading from files has been fixed at iso-8859-1.
  ~l[character-encoding].  Thanks to Jared Davis for bringing this portability
  issue to our attention (as this change arose in order to deal with a change
  in the default character encoding for the host Lisp, CCL), and pointing us in
  the right direction for dealing with it.  In many cases, the character
  encoding for reading from the terminal is also iso-8859-1; but this is not
  guaranteed.  In particular, when the host Lisp is SBCL this may not be the
  case.

  Although the HTML documentation is distributed with ACL2, it had not been
  possible for users to build that documentation without omitting graphics, for
  example on the ACL2 home page.  That has been fixed, as files
  ~c[graphics/*.gif] are now distributed.

  Compiler warnings are suppressed more completely than they had been before.
  For example, the following had produced a compiler warning when the host Lisp
  is CCL, but no longer does so.
  ~bv[]
  (defun f () (car 3))
  (trace$ f)
  ~ev[]

  Removed support for ``tainted'' ~il[certificate]s.  One reason is that there
  are rarely incremental releases.  A stronger reason is that for the
  compatibility of a new release is with the previous non-incremental release,
  it's not particularly relevant whether or not the new release is incremental.

  The `make' variable ~c[BOOKS] can now be defined above the line that includes
  Makefile-generic.  (For relevant background,
  ~pl[books-certification-classic].)

  (SBCL only) ACL2 images built on SBCL now have an option,
  ~c[--dynamic-space-size 2000], that can avoid space problems that could
  previously have caused the session to die.

  The default value for variable ~c[LISP] in file ~c[GNUmakefile] is now
  ~c[ccl].  Thus, if you use `make' in the standard way to build an ACL2
  executable, the default host Lisp is ~c[ccl] rather than ~c[gcl].

  ~st[EMACS SUPPORT]

  ~st[EXPERIMENTAL VERSIONS]

  For the version supporting the reals, ACL2(r) (~pl[real]), the supporting
  function ~c[floor1] has been defined in raw Lisp.  This avoids an error
  such as in the following case.
  ~bv[]
  (defun f () (declare (xargs :guard t)) (floor1 8/3))
  (f) ; had caused raw Lisp error, before the fix
  ~ev[]

  Among the enhancements for the parallel version, ACL2(p) (~pl[parallelism]),
  are the following.  We thank David Rager for his work in developing ACL2(p)
  and these improvements in particular.~bq[]

  The macro ~c[set-parallel-evaluation] has been renamed
  ~ilc[set-parallel-execution].

  Calls of the macro ~ilc[set-waterfall-printing] are no longer ~il[events], so
  may not be placed at the top level of ~il[books].  However, it is easy to
  create events that have these effects; ~pl[set-waterfall-printing].  Note
  that now, ~c[:]~ilc[ubt] and similar commands do not change the settings for
  either waterfall-parallelism or waterfall-printing.

  The implementation of ~ilc[deflock] has been improved.  Now, the macro it
  defines can provide a lock when invoked inside a ~il[guard]-verified or
  ~c[:]~ilc[program] mode function.  Previously, this was only the case if the
  function definition was loaded from raw Lisp, typically via a compiled file.

  The underlying implementation for waterfall parallelism
  (~pl[set-waterfall-parallelism]) has been improved.  As a result, even the
  largest proofs in the regression suite can be run efficiently in
  ~c[:resource-based] waterfall parallelism mode.  Among these improvements is
  one that can prevent machines from rebooting because operating system limits
  have been exceeded; thanks to Robert Krug for bringing this issue to our
  attention.

  There is also a new flag for configuring the way waterfall parallelism
  behaves once underlying system resource limits are reached.  This flag is
  most relevant to ~c[:full] waterfall parallelism.
  ~pl[set-total-parallelism-work-limit] for more information.

  The ~ilc[dmr] utility has the same behavior in ACL2(p) as it has in ACL2
  unless waterfall-parallelism has been set to a non-~c[nil] value
  (~pl[set-waterfall-parallelism]), in which case statistics about parallel
  execution are printed instead of the usual information.

  The user can now build the regression suite using waterfall ~il[parallelism].
  See the distributed file ~c[acl2-customization-files/README] for details, and
  ~pl[unsupported-waterfall-parallelism-features] for a disclaimer related to
  building the regression suite using waterfall parallelism.

  When building ACL2 with both the hons and parallelism extensions (what is
  called ``ACL2(hp)''), the functions that are automatically memoized by the
  hons extension are now automatically unmemoized and memoized when the user
  toggles waterfall parallelism on and off, respectively.

  Calling ~ilc[set-waterfall-parallelism] with a flag of ~c[t] now results in
  the same settings as if it were called with a flag of ~c[:resource-based],
  which is now the recommended mode for waterfall parallelism.  Thanks to
  Shilpi Goel for requesting this feature.

  The prover now aborts in a timely way in response to interrupts issued during
  a proof with waterfall parallelism enabled.  (This had often not been the
  case.)  Thanks to Shilpi Goel for requesting this improvement.
  ~eq[]

  Among the enhancements for the HONS extension (~pl[hons-and-memoization])
  are the following.~bq[]

  The compact-print code has been replaced by new serialization routines
  contributed by Jared Davis.  This may improve performance when including
  books that contain ~ilc[make-event]s that expand to very large constants.
  You can also now save objects to disk without going into raw lisp;
  ~pl[serialize] for details.

  Printing of certain messages has been sped up (by using Lisp function
  ~c[force-output] in place of ~c[finish-output]).  Thanks to Jared Davis for
  contributing this improvement.

  ~il[Stobj] array writes are perhaps twice as fast.

  It is now permitted to ~il[memoize] functions that take user-defined
  ~il[stobj]s as inputs, provided that no ~il[stobj]s are returned.  Even if
  stobjs are returned, memoization is permitted provided the condition is
  ~c[nil], as when profiling (~pl[profile]).  Thanks to Sol Swords for an
  observation that led to this improvement and for useful conversations,
  including follow-up leading us to improve our initial implementation.

  Fixes have been made for memoizing with a non-~c[nil] value of
  ~c[:ideal-okp].  Errors had occurred when memoizing with a ~c[:condition]
  other than ~c[t] for a ~c[:]~ilc[logic] mode function that had not been
  ~il[guard]-verified, even with a non-~c[nil] value of ~c[:ideal-okp]; and
  after successfully memoizing such a function (without such ~c[:condition]),
  it had not been possible to ~ilc[unmemoize] it.  Thanks to Sol Swords for
  reporting issues with the ~c[:ideal-okp] argument of ~ilc[memoize].

  If a book defined a function that was subsequently ~il[memoize]d in that
  book, the function would no longer behaves as memoized upon completion of
  book certification (unless that ~ilc[certify-book] command was undone and
  replaced by evaluation of a corresponding ~ilc[include-book] command).  This
  has been fixed.  Thanks to David Rager for pointing out the problem by
  sending an example.

  We now support ACL2(h) built not only on 64-bit CCL but also on all supported
  host Ansi Common Lisps (i.e., all supported host Lisps except GCL).  Thanks
  to Jared Davis for doing much of the work to make this improvement.  Note
  that performance will likely be best for 64-bit CCL; for some Lisps,
  performance may be much worse, probably depending in part on the underlying
  implementation of hash tables.
  ~eq[]

  ~/~/")

(deflabel note-6-0

; Total number of release note items: 53.

; Added analogues simple-translate-and-eval-error-double and
; simple-translate-and-eval-cmp of simple-translate-and-eval, which instead of
; (mv erp val state) return (mv erp val) and a context-message pair,
; respectively.  Note that error output is never inhibited for these functions
; as implemented.

; Two new functions, which are rarely if ever called by users, have been made
; untouchable: stobj-evisceration-alist and trace-evisceration-alist.  Before
; this change, (trace-evisceration-alist state) and (stobj-evisceration-alist
; nil state) could return non-ACL2 objects, and the former even caused a hard
; Lisp error when a user stobj had previously been introduced.

; Fixed an error message that was complaining about redefinition of a function
; previously defined at the top level, or at the top level of a book, when the
; function was actually built into ACL2.  For example:
;
; ; old:
; ACL2 !>(defun rewrite (x) x)
; ....
; Note: REWRITE was previously defined at the top level.
;
; ; new:
; ACL2 !>(defun rewrite (x) x)
; ....
; Note: REWRITE has already been defined as a system name; that is, it
; is built into ACL2.

; Fixed bugs in case-match calls: final branch cased on t instead of &.  The
; functions were translate-rule-class-alist and bdd-clause1; the former could
; be exploited to get a misleading error message, but that's all, while we
; didn't investigate the consequences of the bdd-clause1 bug.

; The defrec utility now defines a recognizer with guard t.  An optional
; argument specifies the name of the recognizer, which by default for (defrec
; foo ...) is the symbol WEAK-FOO-P, in the same package as foo.

; Among the lower-level changes made in support of meta-extract hypotheses are
; the following:

;   If x is an atom then (sublis-var1-lst alist x) is x; formerly it was nil.
;   This change was originally made to support meta-extract hypotheses, but
;   probably is no longer necessary.  Still, it seems like a good change, since
;   it "almost" allows us to prove that (sublis-var nil x) = x -- "almost"
;   because this is only true when x is in quote-normal form.

;   Some mfc-xx functions now have implicit constraints because of
;   meta-extract-contextual-fact, so *unattachable-primitives* has been
;   extended accordingly.

;   The Essay on Correctness of Meta Reasoning has been substantially
;   extended.

; The definition of constraint-info has changed very slightly in order to make
; it easier to make :common-lisp-compliant (as required in order to make
; meta-extract-formula :common-lisp-compliant).  The main change is that some
; zero-ary function constraints might not be in quote-normal form, though that
; actually seems impossible since the changes away from cons-term would only be
; relevant for executable zero-ary primitives, of which there are none!

; We eliminated the nonlinearp condition in add-polys1.  For details, see the
; comment there about this change.

; The file doc/texinfo.tex has been removed, the result being that if one
; builds one's own Postscript version of the documentation, then the look will
; quite possibly be somewhat different than it was previously.

; Here is an example from Sol Swords, as promised in the item below about a
; soundness bug in defabsstobj based on guards.

;   (defstobj my-stobj-impl (my-fld :type (integer 0 *) :initially 0))
;
;   (trace$ len)
;
;   (defun bad-accessor-logic (my-stobj-logic)
;      (declare (xargs :guard (equal (len my-stobj-logic) 0)))
;      (mbe :logic 0
;           :exec (len my-stobj-logic)))
;
;   (defun bad-accessor-exec (my-stobj-impl)
;      (declare (xargs :stobjs my-stobj-impl)
;               (ignorable my-stobj-impl))
;      1)
;
;   (defun create-my-stobj-logic ()
;      (declare (xargs :guard t))
;      (list 0))
;
;   (defun my-stobj-logicp (x)
;      (declare (xargs :guard t))
;      (AND (TRUE-LISTP X)
;           (= (LENGTH X) 1)
;           (MY-FLDP (NTH 0 X))))
;
;   (defun-nx my-stobj-corr (my-stobj-i my-stobj-l)
;      (and (my-stobj-implp my-stobj-i)
;           (equal my-stobj-i my-stobj-l)))
;
;   (DEFTHM CREATE-MY-STOBJ-ABS{CORRESPONDENCE}
;            (MY-STOBJ-CORR (CREATE-MY-STOBJ-IMPL)
;                           (CREATE-MY-STOBJ-LOGIC))
;            ;; added by Matt K.:
;            :hints (("Goal" :in-theory (disable (my-stobj-corr)))))
;
;   (DEFTHM CREATE-MY-STOBJ-ABS{PRESERVED}
;            (MY-STOBJ-LOGICP (CREATE-MY-STOBJ-LOGIC)))
;
;   (DEFTHM BAD-ACCESSOR{CORRESPONDENCE}
;            (IMPLIES (AND (MY-STOBJ-CORR MY-STOBJ-IMPL MY-STOBJ-ABS)
;                          (EQUAL (LEN MY-STOBJ-ABS) 0))
;                     (EQUAL (BAD-ACCESSOR-EXEC MY-STOBJ-IMPL)
;                            (BAD-ACCESSOR-LOGIC MY-STOBJ-ABS))))
;
;   (defabsstobj my-stobj-abs
;      :concrete my-stobj-impl
;      :recognizer (my-stobj-absp :logic my-stobj-logicp :exec my-stobj-implp)
;      :creator (create-my-stobj-abs :logic create-my-stobj-logic :exec
;                                    create-my-stobj-impl)
;      :corr-fn my-stobj-corr
;      :exports ((bad-accessor :logic bad-accessor-logic :exec bad-accessor-exec)))
;
;   ; Test added by Matt K.:
;   (bad-accessor my-stobj-abs) ; note trace of len here: (LEN |<my-stobj-abs>|)
;
;   (defun length-of-my-stobj-abs ()
;      (declare (xargs :guard t))
;      (with-local-stobj my-stobj-abs
;        (mv-let (len my-stobj-abs)
;          (let ((len (ec-call (bad-accessor my-stobj-abs))))
;            (mv len my-stobj-abs))
;          len)
;        create-my-stobj-abs))
;
;   (defthm length-of-my-stobj-abs-by-def
;      (equal (length-of-my-stobj-abs) 0)
;      :hints(("Goal" :in-theory (disable (length-of-my-stobj-abs))))
;      :rule-classes nil)
;
;   (defthm length-of-my-stobj-abs-by-exec
;      (equal (length-of-my-stobj-abs) 1))

; Replaced "data base" and "data-base" by "database".

; The following change supports certification of books/centaur/ books in ACL2,
; not just ACL2(h).  In order to eliminate errors upon reference to
; *never-profile-ht* in raw Lisp code such as is found in
; books/centaur/vl/util/print-htmlencode.lisp, *never-profile-ht* is now
; defined in ACL2.

; Changed equal to eq in the body of the definition of pos-listp.

; Fixed a typo (":type-prescrption", in chk-acceptable-type-prescription-rule)
; that could have affected warnings printed when :type-prescription rules are
; submitted.

; Modified the definition of mbe to use ignorable.  Some Lisps might warn, but
; they probably warned before.  With this change, CCL svn rev 15527 doesn't
; warn.

; Modified the layout of the history-entry record so that it's not dependent on
; #+acl2-par, and hence its generated macros are the same for ACL2 and
; ACL2(p).  As a consequence, made a few related changes.

  :doc
  ":Doc-Section release-notes

  ACL2 Version  6.0 (December, 2012) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  5.0 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, changes at the system level, Emacs support, and
  experimental versions.  Each change is described in just one category, though
  of course many changes could be placed in more than one category.

  NOTE.  But we start with one major change that is outside the usual
  categories:

  ~st[LICENSE change]

  The ACL2 license has been changed from GPL Version 2 to a 3-clause BSD
  license, found in the ~c[LICENSE] file distributed with ACL2.

  ~st[CHANGES TO EXISTING FEATURES]

  Function ~ilc[fmt-to-string] and similar functions (~pl[printing-to-strings])
  now use the default right margin settings; formerly the right margin had been
  set at 10,000.  If you want the former behavior, you can use the
  ~c[:fmt-control-alist], as illustrated below.
  ~bv[]
  (fmt-to-string \"~~x0\"
                 (list (cons #\\0 (make-list 30)))
                 :fmt-control-alist
                 `((fmt-soft-right-margin . 10000)
                   (fmt-hard-right-margin . 10000)))
  ~ev[]

  The use of attachments (~pl[defattach]) has been made more efficient,
  avoiding some low-level checks (Common Lisp `~c[boundp]' checks).  Thanks to
  Shilpi Goel for constructing an example that we used to help direct us to
  remove inefficiency.  The following results for that example ~-[] a Fibonacci
  program run on a machine interpreter in raw-mode (~pl[set-raw-mode]) ~-[]
  give a sense of the potential speedup, though we note that a full ACL2(h)
  regression showed no significant speedup.~bv[]

  ; Time before the change:
  ; 0.89 seconds realtime, 0.90 seconds runtime

  ; Time after the change:
  ; 0.75 seconds realtime, 0.75 seconds runtime

  ; Time when cheating to avoid the cost of attachments, by redefining a
  ; function to BE its attachment (so, this gives a lower bound on possible
  ; execution time):
  ; 0.72 seconds realtime, 0.72 seconds runtime
  ~ev[]

  Functions ~c[read-acl2-oracle] and ~c[read-acl2-oracle@par] are no longer
  untouchable (~pl[remove-untouchable]).  We reported this change for
  Version_5.0 but it was not made; thanks to Jared Davis for bringing this to
  our attention.  Function ~c[get-timer] also is no longer untouchable.

  The function ~ilc[butlast] now behaves more reasonably on arguments violating
  its ~il[guard].  For example, ~c[(butlast '(1 2 3) -1)] is now provably equal
  to ~c[(1 2 3)] instead of to ~c[(1 2 3 nil)].  Thanks to Jared Davis for
  suggesting a change to the definition of ~c[butlast].

  The utilities ~c[mfc-ts] and ~c[mfc-ap] (~pl[extended-metafunctions])
  formerly never used forcing (~pl[force]).  Now, by default, forcing is
  allowed during execution of these functions if and only if it is permitted in
  the rewriting environment where they are called.  Moreover, these and the
  ~c[mfc-xx] utilities ~-[] ~c[mfc-rw], ~c[mfc-rw+], and ~c[mfc-relieve-hyp]
  ~-[] are now macros that take (optional) keyword arguments ~c[:forcep] and
  ~c[:ttreep].  The ~c[:forcep] argument is ~c[:same] by default, providing the
  forcing behavior inherited from the environment (as described above); but it
  can be the symbol ~c[t] or ~c[nil], indicating that forcing is to be enabled
  or disabled, respectively.  The ~c[:ttree] argument is ~c[nil] by default,
  but when it is ~c[t], then a second value is returned, which is a tag-tree.
  ~l[extended-metafunctions].

  Many improvements have been made to the tau-system (~pl[tau-system]),
  including support for arithmetic intervals bounded by constants.  Thus, for
  example, ~c[(and (<= 0 x) (<= x 15))] is a tau predicate.  The
  ~il[documentation] has also been improved
  (~pl[introduction-to-the-tau-system]). Also ~pl[time-tracker-tau] for
  discussion of how the new ~ilc[time-tracker] utility can help discover ways
  to detect slowdown related to the tau-system.

  The ~ilc[defthm] ~il[events] printed by ~ilc[defabsstobj], namely those that
  remain to be proved, are now given with ~c[:rule-classes nil] since there is
  probably no intention to use them as rules.  Thanks to Robert Krug for
  suggesting that we consider this change.

  The formal parameters for a macro definition (~pl[defmacro]) may now include
  ~ilc[state] and user-defined ~ilc[stobj]s.  (However, macro formals may not
  be declared as stobjs; ~pl[xargs].)  Thanks to Jose Luis Ruiz-Reina for
  raising this issue and to Rob Sumners for helpful conversations ~-[] both of
  these nearly 10 years ago!

  The utilities ~ilc[defun-inline], ~ilc[defun-notinline], ~ilc[defund-inline],
  and ~ilc[defund-notinline] have been simplified, by taking advantage of the
  lifting of restrictions on formal parameters of macro definitions mentioned
  above (involving symbols that happen to be ~il[stobj] names).  Now, when any
  of the above four utilities is called with a given set of formal parameters,
  those formals will be used not only for the generated ~ilc[defun] event but
  also for the generated ~ilc[defmacro] event.  (Previously, they had been
  renamed for the ~ilc[defmacro] event in order to respect the stobj name
  restriction that no longer exists.)  Thanks to Jared Davis for pointing out
  the value of making this change.

  The ~il[events] ~ilc[add-invisible-fns] and ~ilc[remove-invisible-fns] now
  convert arguments as appropriate using the ~ilc[macro-aliases-table].  For
  example, the event ~c[(add-invisible-fns append car)] is now legal (though
  probably not a good idea), because ~c[add-invisible-fns] is now sensitive
  to the fact that ~ilc[append] maps to ~ilc[binary-append] in the
  ~ilc[macro-aliases-table].

  When ~c[:]~c[pe] is applied to a built-in function that does not have a
  defining event, such as ~ilc[symbolp], ~c[:pe] now gives more useful output
  that points to the documentation instead of printing a call of
  ~c[ENTER-BOOT-STRAP-MODE].  Thanks to Anthony Knape for bringing this issue
  to our attention.

  The macros ~ilc[memoize] and ~ilc[unmemoize] now cause a warning rather than
  an error in ACL2 (and work as before in ACL2(h)).

  Terms are now parsed into ~c[:]~ilc[type-prescription] rules in a manner that
  removes ~ilc[let] bindings both at the top level and in the conclusion (but
  still not in the hypotheses of the rule).  ~l[type-prescription].  Thanks to
  Jared Davis for requesting such an enhancement.

  Printing of numbers is now appropriately sensitive to the print radix;
  ~pl[set-print-radix].  Thanks to Shilpi Goel for requesting this enhancement.

  The system function ~c[explode-atom] no longer includes the radix indicator.
  The new function ~c[explode-atom+] may be used for that purpose.

  ~st[NEW FEATURES]

  Among the new features for system hackers are analogues of system function
  ~c[simple-translate-and-eval] that do not return ~ilc[state].  (Thanks to
  David Rager for requesting this feature and helpful conversations on its
  implementation.)  This and other low-level changes are typically documented
  in comments in the corresponding release note event, which in this case is
  ~c[(deflabel note-6-0 ...)].

  More built-in functions are now ~il[guard]-verified (and in ~c[:]~ilc[logic]
  mode).  Furthermore, a mechanism exists for marking yet more built-in
  functions as guard-verified based on ~il[books] contributed by users; see
  Part II of
  ~url[http://www.cs.utexas.edu/users/moore/acl2/open-architecture/].  The
  current state of that enterprise may be viewed by evaluating the constant
  ~c[*system-verify-guards-alist*], which associates a community book name with
  a list of functions.  When ACL2 is built in the normal way, each of those
  functions is marked as guard-verified when ACL2 is started up; but a special
  developer build can be used to check that the indicated book, together with
  its sub-books, proves that those functions are guard-verified.

  Metatheorems (~pl[meta]) may now have additional hypotheses, called
  ``meta-extract hypotheses'', that allow metafunctions to depend on the
  validity of certain terms extracted from the context or the logical
  ~il[world].  ~l[meta-extract].  Thanks to Sol Swords for providing an initial
  implementation, together with very helpful discussions as well as a community
  book, ~c[books/clause-processors/meta-extract-user.lisp], that extends the
  power of meta-extract hypotheses.

  New utilities ~ilc[oracle-funcall], ~ilc[oracle-apply], and
  ~ilc[oracle-apply-raw] call a function argument on specified arguments.
  Thanks to Jared Davis for requesting this utility.

  A new utility makes it convenient to track time spent inside specified
  function calls or, more generally, during specified evaluation.
  ~l[time-tracker].

  New runic designators make it easy to refer to macro names when building
  theories.  Thus, for example, the object ~c[(:i append)] may be used in
  theory expressions to designate the ~il[rune] ~c[(:induction binary-append)].
  ~l[theories].  Thanks to Jared Davis for a useful discussion leading to this
  enhancement.

  ~ilc[Defabsstobj] ~il[events] now take an optional ~c[:congruent-to] keyword
  argument, much like ~ilc[defstobj].  Thanks to Sol Swords for requesting this
  feature and for suggesting a very nice optimization that avoids the need to
  prove additional lemmas.

  ~ilc[Flet] may now include ~c[inline] and ~c[notinline] declarations.  Thanks
  to Jared Davis for requesting this feature.

  The utility ~c[gc-verbose] controls printing of messages by the garbage
  collector, for certain host Lisps.  ~l[gc-verbose].  Thanks to Shilpi Goel
  for requesting this utility.

  Added definitions of functions ~ilc[nat-listp] and ~ilc[acl2-number-listp].
  Thanks to Harsh Raju Chamarthi for requesting these additions.  Many
  community books had varying definitions of these functions; these additions
  guarantee that all books must agree on how these two functions are
  defined.  (Some community books have been changed in order that they remain
  certifiable, given these additions.)  Note that a few built-in
  ~c[:]~ilc[forward-chaining] rules were modified in order to accommodate these
  additions, and the definition of ~ilc[integer-listp] was modified to call
  ~ilc[eq] instead of ~ilc[equal], like the other such definitions.

  ~l[get-command-sequence] for a new utility that returns a list of
  ~il[command]s between two given command descriptors.

  ~st[HEURISTIC IMPROVEMENTS]

  We obtained a substantial speedup ~-[] 13% observed for the regression suite,
  and 8% observed for the ACL2(h) regression suite ~-[] by tweaking the
  ~il[break-rewrite] implementation to eliminate virtually all of its overhead
  when it is not in use (the default, which holds until ~c[:]~ilc[brr]~c[ t] is
  evaluated).  Thanks to David Rager for a conversation involving ACL2(p)
  performance statistics that suggested looking at changing ~il[break-rewrite]
  to boost performance.

  The heuristics for automatically expanding recursive function calls have been
  changed during proofs by induction.  Now, during induction, more terms that
  suggested the induction scheme are automatically expanded.  Thanks to David
  Rager for providing an example and having discussions with us that spurred us
  to develop this heuristic improvement.

  ~st[BUG FIXES]

  Fixed a soundness bug in ~ilc[defabsstobj] based on ~ilc[guard]s that
  violated single-threadedness restrictions.  Thanks to Sol Swords for bringing
  this bug to our attention and supplying a proof of ~c[nil], which we include
  as a comment in source file ~c[ld.lisp], in ~c[(deflabel note-6-0 ...)].  We
  also thank Sol for helpful discussions about ~il[guard]s of functions
  introduced by ~c[defabsstobj], which has led us to enhance the
  ~il[documentation]; ~pl[defabsstobj].

  Fixed a soundness bug in ~ilc[defabsstobj] based on interrupted updates of
  abstract stobjs.  As part of the fix a new keyword, ~c[:PROTECT], has been
  introduced for ~c[defabsstobj] exports, along with a new top-level
  ~c[defabsstobj] keyword, ~c[:PROTECT-DEFAULT]; ~pl[defabsstobj].  We do some
  analysis that we expect will avoid the use of ~c[:PROTECT] in many cases,
  which is fortunate since the use of ~c[:PROTECT t] may cause a slight
  slowdown in (abstract) stobj updates.  Thanks to Sol Swords for bringing this
  bug to our attention and supplying a proof of ~c[nil], which we include as a
  comment in source file ~c[other-events.lisp], in the definition of function
  ~c[set-absstobj-debug].

  Fixed a raw Lisp error that occurred when tracing a ~i[stobj] resize
  function, thanks to an error report from Warren Hunt, Marijn Heule, and
  Nathan Wetzler.

  Fixed a raw Lisp error that occurred for certain ill-formed signatures, as in
  the following example.
  ~bv[]
  ACL2 !>(encapsulate
             (((f (*) => * :guard t)))
             (local (defun f (x) (consp x))))

  ***********************************************
  ************ ABORTING from raw Lisp ***********
  Error:  value (F (*) => * :GUARD T) is not of the expected type SYMBOL.
  ***********************************************
  ~ev[]

  The notion of ``error triple'' (~pl[error-triples]) had been implemented
  ambiguously, with the result that for a ~il[stobj], ~c[st], the result of
  evaluating the following two forms was the same: ~c[(mv nil st state)] and
  ~c[(mv t st state)].  Of course, these are just examples; in general, a
  result of ~c[(mv erp val state)] was sometimes treated as an error triple
  even when ~c[val] is a ~il[stobj].  Now, ~c[(mv erp val state)] is an error
  triple only when ~c[erp] and ~c[val] are ordinary (non-~il[stobj]) values.
  Thanks to Warren Hunt and Marijn Heule for bringing this problem to our
  attention.

  The ``with-error-trace'' utility, ~ilc[wet], now works in the non-error case
  when given a form that returns multiple values.  (Note however that
  ~ilc[STATE] will be printed as ~c[REPLACED-STATE]; and similarly, a
  user-defined ~il[stobj], say ~c[ST], will be printed as ~c[REPLACED-ST].)

  Some possible error messages for ~ilc[defabsstobj] have been fixed that had
  been ill-formed.  Thanks to Sol Swords for bringing this bug to our
  attention.

  Fixed a bug that sometimes caused the times displayed in the summary for
  ~ilc[certify-book] to be smaller than the actual times.

  Fixed a bug in the ~il[guard]s to system functions ~c[fmt-char] and
  ~c[fmt-var], which are no longer ~c[:]~ilc[logic]-mode, guard-verified
  functions.

  (GCL only) Fixed a bug present in Gnu Common Lisp for
  ~c[#u] (~pl[sharp-u-reader]).

  ~st[CHANGES AT THE SYSTEM LEVEL]

  The ~il[state] global variable ~c['distributed-books-dir] has been renamed
  ~c['system-books-dir].  On a related note, the ~il[documentation] now refers
  to ``community books'' rather than ``distributed books'', and there is a
  corresponding new documentation topic; ~pl[community-books].

  Fixed a bug in the implementation of ~ilc[wet] (which is actually in the
  community book ~c[books/misc/wet.lisp]).

  A directory, ~c[interface/], is no longer part of the ACL2 distribution.
  Rather, it is a subdirectory of the ACL2 community books.  Thus, if you fetch
  those books in the usual way (see the installation instructions on the ACL2
  home page), you will find a directory ~c[books/interface/].  Subdirectory
  ~c[emacs/] of that ~c[interface] directory provides Emacs support for
  ~il[proof-tree]s as well an ~c[acl2-mode].  This change has been reflected in
  ACL2 file ~c[emacs/emacs-acl2.el], so users will probably not be impacted if
  they load that file into Emacs.

  The community books file ~c[books/Makefile-generic] now causes, by default, a
  backtrace to be printed when there is a raw Lisp error.

  Some changes have been made to how regressions are run, i.e., to how the
  community books are certified.  (1) The standard regression now includes
  community books directory ~c[books/centaur].  To skip these (for example, a
  Windows system has encountered difficulty with them even after installing
  Perl), include ~c[ACL2_CENTAUR=skip] with your `make' command.  (2) A new
  `make' (or environment) variable, ~c[ACL2_JOBS], specifies the number of
  parallel jobs to run, serving as a replacement for the ~c[-j] argument of
  `make' that works for all community books, including those under directory
  ~c[centaur]; ~pl[books-certification-classic].  (3) It is no longer necessary
  to do an ACL2(h) regression in order to build a copy of the documentation
  generated by Jared Davis's xdoc utility at
  ~c[books/xdoc-impl/manual/preview.html]; a vanilla ACL2 regression will build
  this manual.  (4) It is no longer necessary to set the ~c[ACL2] environment
  variable for ACL2(h) regressions if you want to use the executable
  ~c[saved_acl2h] in the ACL2 sources directory.

  The ACL2 home page now has a search utility for documentation and books.
  Thanks to Shilpi Goel and David Rager for feedback on a preliminary version
  of this utility.

  (only for SBCL with 64-bit ACL2(h)) The value of SBCL command line option
  ~c[--dynamic-space-size] for ACL2(h) on 64-bit platforms has been increased
  from 2000 to 16000 (as explained in a comment in the ACL2 source definition
  of ~c[*sbcl-dynamic-space-size*]).

  ~st[EMACS SUPPORT]

  ~st[EXPERIMENTAL/ALTERNATE VERSIONS]

  Among the enhancements for ACL2(r) (~pl[real]) are the following.~bq[]

  Thanks to Ruben Gamboa for his helpful role in making the following
  improvements made with Ruben Gamboa in support for non-standard analysis in
  ACL2(r).

  Constrained functions can now be introduce as non-classical.  ~l[signature].

  ~ilc[Defun-sk] now takes a new keyword argument, ~c[:CLASSICALP], that
  determines whether or not the named function is classical.  ~l[defun-sk].

  Incorporated a bug fix from Ruben Gamboa for ~ilc[ceiling].  The default (for
  `bad' arguments) had been 1, but now we follow normal ACL2 practice by
  returning 0 in that case.
  ~eq[]

  Among the enhancements for the HONS extension (~pl[hons-and-memoization])
  are the following.~bq[]

  Macros ~ilc[with-fast-alist], ~ilc[with-stolen-alist], and
  ~ilc[fast-alist-free-on-exit] are now defined in ACL2(h), rather than being
  defined in the community book ~c[\"books/centaur/misc/hons-extra.lisp\"].
  Thanks to Jared Davis and Sol Swords for donating this code, and thanks to
  Jared for helpful discussions leading to this change.
  ~eq[]

  Among the enhancements for ACL2(p) (~pl[parallelism]) are the following.  We
  thank David Rager for his work in developing ACL2(p) and for his helpful role
  in these improvements.~bq[]

  A bug has been fixed that could leave one in a ~il[wormhole], awaiting input,
  after an error, such as an error in an ~c[:in-theory] hint during a proof.
  Thanks to Shilpi Goel for bringing this bug to our attention.

  A key checkpoint for a given goal is now printed only once.  Previously, if a
  key checkpoint led to more than one goal pushed for proof by induction, the
  key checkpoint would be printed once for each such goal during the proof, and
  also once for each such goal in the summary at the end.
  ~eq[]

  ~/~/")

(deflabel note-6-1

; Total number of release note items: 26.

; As usual, we made a number of improvements to the documentation, based in
; part on useful feedback from users.

; "Theory" warnings about disabling primitives (such as mv-nth) have been made
; much more compact, pointing to a new :doc topic, theories-and-primitives.
; Thanks to David Rager and Jared Davis for helpful discussions leading to this
; change.

; Avoided a bogus call of all-vars in
; defstobj-component-recognizer-axiomatic-defs, and removed a false comment in
; translate-declaration-to-guard that had "justified" this.

; Removed argument value of :raw-lisp for defstobj-template.

; Regarding Expansion/Defstobj Bug (technical remarks followed by example): The
; problem was that the raw Lisp code for defstobj, which is called when loading
; an expansion file, in turn called function defstobj-raw-defs in a world that
; did not include the definition of the `satisfies' predicate, which in turn
; called defstobj-component-recognizer-axiomatic-defs on that world, which in
; turn called translate-declaration-to-guard on that world, which translated
; the `satisfies' type-spec (below) to nil.  Here is the promised example (file
; bug.lisp).

;   ; acl2
;   ; (assign save-expansion-file t)
;   ; (certify-book "bug")
;   ; (quit)
;   ; rm bug.lx64fsl
;   ; acl2
;   ; (include-book "bug")
;   ; (defthm bug
;   ;   nil
;   ;   :hints (("Goal" :use obvious))
;   ;   :rule-classes nil)
;
;   (in-package "ACL2")
;
;   (defun my-natp (x)
;     (declare (xargs :guard t))
;     (natp x))
;
;   (defstobj st1 (fld1 :type (satisfies my-natp) :initially 0))
;
;   (defthm obvious
;     (fld1p 3)
;     :rule-classes nil)
;
;   ; [End of file bug.lisp.]

; Regarding memoize/congruent stobj bug: The following example illustrates the
; bug.  Technical remark: The problem was that the true congruent stobj
; representative was not stored where expected.

;   ; acl2h
;   ; (certify-book "foo")
;   ; (quit)
;   ; acl2h
;   ; (include-book "foo")
;   ; (memoize 'fld3)
;   ; (defthm bug
;   ;   nil
;   ;   :hints (("Goal" :use foo-is-17))
;   ;   :rule-classes nil)
;
;   (in-package "ACL2")
;
;   (defstobj st1 fld1)
;   (defstobj st2 fld2 :congruent-to st1)
;   (defstobj st3 fld3 :congruent-to st2)
;
;   (defun foo ()
;     (with-local-stobj
;      st3
;      (mv-let (result st3)
;              (prog2$ (fld3 st3)
;                      (let ((st3 (update-fld3 17 st3)))
;                        (mv (fld3 st3) st3)))
;              result)))
;
;   (defthm foo-is-17
;     (equal (foo) 17)
;     :rule-classes nil)

; Here is a proof of nil for "a soundness bug involving system function
; canonical-pathname...."  This is a trivial modification (being very slightly
; simpler) of the example sent by Jared Davis and Sol Swords.

;   (in-package "ACL2")
;
;   (defchoose state-for-canonical-pathname (st) ()
;      (not (canonical-pathname nil nil st)))
;
;   (defevaluator ncp-ev ncp-ev-lst ((state-for-canonical-pathname)
;                                     (canonical-pathname a b c)
;                                     (if a b c)
;                                     (equal a b)))
;
;   (defun run-canonical-pathname-cp (clause hints state)
;      (declare (xargs :guard (pseudo-term-listp clause)
;                      :stobjs state)
;               (ignore hints))
;      (mv nil
;          (if (equal clause '((equal (canonical-pathname
;                                      'nil 'nil
;                                      (state-for-canonical-pathname))
;                                     'nil)))
;              (if (canonical-pathname nil nil state)
;                  (list clause)
;                nil)
;            (list clause))
;          state))
;
;   (defthm run-canonical-pathname-cp-correct
;      (implies (and (pseudo-term-listp clause)
;                    (alistp a)
;                    (ncp-ev (conjoin-clauses
;                             (clauses-result
;                              (run-canonical-pathname-cp clause hint state)))
;                            a))
;               (ncp-ev (disjoin clause) a))
;      :hints (("goal" :use ((:instance state-for-canonical-pathname
;                                       (st state)))))
;      :rule-classes :clause-processor)
;
;   (defthm canonical-pathname-is-nil
;      (equal (canonical-pathname
;              'nil 'nil
;              (state-for-canonical-pathname))
;             'nil)
;      :hints (("goal" :clause-processor
;               (run-canonical-pathname-cp clause nil state))))
;
;   (defun foo (x dir-p st)
;      ;; Matches the constraints of canonical-pathname but never returns NIL.
;      (declare (ignore x dir-p st))
;      "hello")
;
;   (defthm foo-never-returns-nil
;      (foo x dir-p st))
;
;   (defchoose state-for-foo (st) ()
;      (not (foo nil nil st)))
;
;   (defthm foo-sometimes-returns-nil
;      (not (foo nil nil (state-for-foo)))
;      :hints (("goal" :use ((:functional-instance
;                             canonical-pathname-is-nil
;                             (canonical-pathname foo)
;                             (state-for-canonical-pathname
;                              state-for-foo))))))
;
;   (defthm contradiction
;      nil
;      :rule-classes nil
;      :hints (("Goal" :use ((:instance foo-sometimes-returns-nil)))))

; Elaborating here on "the functions ~ilc[sys-call] and ~ilc[sys-call-status]
; are now ~il[guard]-verified ~c[:]~ilc[logic]-mode functions": Sys-call-status
; now modifies the oracle, not the file-clock.  It was misguided anyhow to
; expect that modifying the file-clock would provide a coherent story, since
; one might call sys-call to modify the file-system but never call
; sys-call-status to update the file-clock.  Since a trust-tag is necessary in
; order to call sys-call, there is no soundness problem here.  Indeed, :doc
; sys-call is already quite clear on such matters.

; As part of the process of fixing two soundness bugs -- one related to
; canonical-pathname and the other about defaxiom and defattach -- we made a
; number of changes, in particular improving the Essay on Correctness of Meta
; Reasoning and the Essay on Defattach.  In particular, the Essay on Defattach
; now comprehends the use of defaxiom.  Also note that now, as part of fixing
; that first bug, canonical-pathname and mfc-xx functions have been given
; unknown-constraints (by introducing them with dependent clause processors),
; and are no longer in *unattachable-primitives* (as their unattachability is
; enforced by their having unknown-constraints).

; We improved (and slowed down) the algorithm for computing the tau of a term.
; For example it dives into NOT now. The biggest change is that it in
; Version_6.0, preprocess-clause tried tau only before the first simplification
; (when hist=nil) and after the clause was stable under simplification.  The
; new one tries tau more aggressively: before the first three simplifications.
; We found proofs where the more aggressive use of tau -- try after a little
; rewriting got rid of functions the users means to expand -- helped.
;
; Some tests showed that the more aggressive use of tau slows down the
; regression a little compared to the less aggressive use of tau.  But because
; as more tau-based scripts are developed, we expect the more aggressive use of
; tau will pay for itself.
;
; Perhaps more important is the comparison between these two alternatives and
; Version_6.0.  They don't have identical regression suites (of course).  But
; they have 3,075 books in common as of Feb. 2013.  The new .out comparison
; utility can compare total reported book certification time for books in
; common.  Based on that, the less aggressive use of tau was measured at about
; 1% faster than Version_6.0 and the more aggressive use of tau was measured as
; about the same speed as Version_6.0.

; Source function simple-array-type formerly accepted '* as a valid array-etype
; for deducing a type of `(simple-vector *).  But since '* is not a valid
; type-spec, we should never hit this case, so we now cause a hard error in
; order to detect a mistake in that thinking.  Note that we tried evaluating
; (make-array '(5) :element-type *) in several (raw) Lisps, and often got an
; error (though it took two evaluations in CCL to get the error).

  :doc
  ":Doc-Section release-notes

  ACL2 Version  6.1 (February, 2013) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  6.0 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, changes at the system level, Emacs support, and
  experimental versions.  Each change is described in just one category, though
  of course many changes could be placed in more than one category.

  ~st[CHANGES TO EXISTING FEATURES]

  More system functions are in ~c[:]~ilc[logic] mode, ~il[guard]-verified.
  Evaluate
  ~bv[]
  (strip-cars (cdr (assoc-equal \"system/top\" *system-verify-guards-alist*)))
  ~ev[]
  for the list of functions checked to be guard-verifiable in the community
  books.  Thanks to those who have contributed to this effort, as shown in file
  headers in directory ~c[system/] of the community books.

  The macro ~ilc[defund] now avoids an error when ~c[:mode :program] has been
  specified in an ~ilc[xargs] form of a ~ilc[declare] form, for example:
  ~c[(defund f (x) (declare (xargs :mode :program)) x)].  It does this by
  avoiding the generation of ~ilc[in-theory] ~il[events] in such cases.  Thanks
  to David Rager and Jared Davis for requesting such a change, and for ensuing
  helpful discussions.

  Added a field ~c[:UNIFY-SUBST] to metafunction contexts
  (~pl[EXTENDED-METAFUNCTIONS]), accessed with function ~c[mfc-unify-subst].
  Thanks to Sol Swords for requesting this enhancement.

  The functions ~ilc[sys-call] and ~ilc[sys-call-status] are now
  ~il[guard]-verified ~c[:]~ilc[logic]-mode functions.

  It had been the case that if any supporter of a dependent clause processor
  (~pl[define-trusted-clause-processor]) is among the ancestors of a given
  formula, then it was illegal to apply functional instantiation
  (~pl[lemma-instance]) to that formula.  Now, this is illegal only if some
  such supporter is in the domain of the functional substitution.

  The tau system (~pl[tau-system], or if you are unfamiliar with the tau
  system, ~pl[introduction-to-the-tau-system]) now allows the user to define
  and verify functions that compute bounds on arithmetic expressions.
  ~l[bounders].

  The utility ~c[print-summary-user] has been replaced by
  ~ilc[finalize-event-user], which is described below.  If you previously
  attached a function to ~c[print-summary-user], say ~c[my-print-summary-user],
  then you can get the effect you had previously as follows.
  ~bv[]
  (defun my-finalize-event-user (state)
    (declare (xargs :mode :logic :stobjs state))
    (prog2$ (my-print-summary-user state)
            state))
  (defattach finalize-event-user my-finalize-event-user)
  ~ev[]

  It had been the case that when you ~ilc[LD] a file, the connected book
  directory (~pl[cbd]) was set to the canonical pathname of that file's
  directory for the duration of the ~c[LD] call.  This could cause problems,
  however, if the file is actually a soft link: an ~ilc[include-book] form in
  the book with a relative pathname for the book would be resolved with respect
  to the absolute pathname for that link, which is probably not what was
  intended.  So soft links are no longer followed when computing the above
  connected book directory.  The following example, which is how we discovered
  this problem, may clarify.  We attempted to execute the form
  ~c[(ld \"top.lisp\")] using ACL2(r) (~pl[real]) in community books directory
  ~c[nonstd/arithmetic/], where all of the ~c[.lisp] files are soft links to
  files in ~c[arithmetic/].  Thus, the form ~c[(include-book \"equalities\")]
  attempted to include ~c[arithmetic/equalities] instead of
  ~c[nonstd/arithmetic/equalities], which caused an error.

  We no longer document the use of value ~c[:START] for
  ~ilc[with-prover-step-limit].  This value has always been used by the ACL2
  implementation and may have semantics that change with new ACL2 versions.  If
  you have reason to use this value, please contact the ACL2 implementors.

  ~st[NEW FEATURES]

  By default, the prover now gives information about case splits.
  ~l[splitter].  Thanks to many ACL2 users, most recently David Rager, for
  requesting such a capability.  Also thanks to David Rager and Jared Davis for
  helpful discussions, and thanks to Robert Krug for feedback on the initial
  implementation and documentation that led us to make improvements.

  New utilities ~ilc[initialize-event-user] and ~ilc[finalize-event-user] allow
  the user to run ~il[state]-modifying code at the start and end of
  ~il[events].  Thanks to Harsh Raju Chamarthi for requesting these
  capabilities.  Note that ~ilc[finalize-event-user] replaces
  ~c[print-summary-user].

  ~st[HEURISTIC IMPROVEMENTS]

  Several heuristic improvements have been made to the tau system, even if you
  do not explicitly use the new capability for computing bounds on arithmetic
  expressions, mentioned above.  ~l[tau-system], or if you are unfamiliar with
  the tau system, ~pl[introduction-to-the-tau-system].

  ~st[BUG FIXES]

  A soundness bug has been fixed that exploited the use of expansion files
  (~pl[book-compiled-file]) together with ~ilc[defstobj].  For an example
  illustrating this bug, see the comment about ``Expansion/Defstobj Bug'' in
  the form ~c[(deflabel note-6-1 ...)] in ACL2 source file ~c[ld.lisp].

  We fixed a soundness bug involving system function ~ilc[canonical-pathname]
  and (most likely) other functions in the former value of constant
  ~c[*unattachable-primitives*].  Thanks to Jared Davis and Sol Swords for
  bringing this bug to our attention by way of an example.  We include a very
  slight variant of that example in a comment within the form
  ~c[(deflabel note-6-1 ...)] in ACL2 source file ~c[ld.lisp].

  There was a soundness bug that allowed attachments to prove ~c[nil] in a
  consistent logical ~il[world] involving ~ilc[defaxiom] ~il[events].  This has
  been fixed, by requiring that no function symbol ancestral in a
  ~ilc[defaxiom] formula is allowed to get an attachment.  ~l[defattach], in
  particular discussion of ``a restriction based on a notion of a function
  symbol syntactically supporting an event'', which concludes with a proof of
  ~c[nil] that is no longer possible.

  (ACL2(h) only) We fixed a soundness bug in the interaction of memoization
  with congruent stobjs, in cases where the ~c[:congruent-to] field of
  ~ilc[defstobj] was not the canonical representative in the congruence class.
  For an example illustrating this bug, see the comment about
  ``memoize/congruent stobj bug'' in the form ~c[(deflabel note-6-1 ...)]  in
  ACL2 source file ~c[ld.lisp].

  Functions defined by ~ilc[defstobj] had failed to be compiled when certifying
  books, except in host Lisps that compile on-the-fly (CCL, SBCL).  This has
  been fixed for all host Lisps.  A related change, probably less significant,
  was made for ~ilc[defabsstobj].  Thanks to Sol Swords for reporting bugs that
  turned out to be mistakes in a preliminary implementation of this change.

  Fixed an assertion error involving linear arithmetic.  Thanks to Sol Swords
  for sending an example illustrating the bug (now appearing as a comment in
  ACL2 source function ~c[linearize1]).

  Fixed a bug that was breaking the ACL2s build mechanism (~pl[acl2-sedan]) by
  causing certain needless evaluation of ``hidden ~ilc[defpkg]'' forms in
  ~il[certificate] files when executing a call of ~ilc[include-book].  The bug
  could also affect rare error messages arising from ill-formed
  ~il[certificate] files.  Thanks to Harsh Raju Chamarthi for bringing this bug
  to our attention by sending us an example script of the sort that was
  breaking during an ACL2s build.

  Fixed handling of pathnames by some low-level code (system function
  ~c[our-truename]) that could cause errors, for example for host-Lisp GCL on
  some platforms when environment variable ~c[HOME] points to a non-existent
  directory.  Thanks to Camm Maguire for bringing this issue to our attention
  and helping with the debugging.

  Fixed a coding bug in generation of stobj resizing functions for a stobj
  named ~c[OLD].  The following example illustrates the bug.
  ~bv[]
  (defstobj old
    (fld :type (array (unsigned-byte 31) (8))
          :initially 0 :resizable t))
  (resize-fld 10 old)
  ; The following returned 8 but should have returned 10:
  (fld-length old)
  ~ev[]

  Fixed a bug in ~ilc[defabsstobj-missing-events] (which macroexpanded
  incorrectly).  Thanks to Sol Swords for bringing this bug to our attention.

  Fixed two bugs in the handling of step-limits.  Thanks to Hanbing Liu for
  bringing the main such bug to our attention, which was that ACL2 could report
  a step-limit violation during ~ilc[certify-book] (in fact, during any
  compound event such as a call of ~ilc[encapsulate] or ~ilc[progn]), even
  without direct user involvement in managing step-limits
  (~pl[set-prover-step-limit] and ~pl[with-prover-step-limit]).  The other bug
  was that a bad argument to ~ilc[set-prover-step-limit] could result in a raw
  Lisp error, for example: ~c[(progn (set-prover-step-limit '(a b)))].

  ~st[CHANGES AT THE SYSTEM LEVEL]

  The ~c[books/] directory no longer needs to exist in order to build an ACL2
  executable.  Thanks to Robert Krug for pointing out that the installation
  instructions had suggested that this was already the case.

  Many changes have been made to the community books (~pl[community-books]).
  For example, some community books now include ~c[std/lists/rev.lisp], which
  contains the rule ~c[revappend-removal], which may cause some proofs
  involving ~ilc[revappend] to fail where they formerly succeeded, or
  vice-versa.  When a proof fails that formerly succeeded, it may be useful for
  you to look over the ~il[rune]s printed in the event summary.

  ~st[EMACS SUPPORT]

  ~st[EXPERIMENTAL/ALTERNATE VERSIONS]

  For ACL2(p), ~ilc[wormhole-eval] is now locked by default; thanks to David
  Rager for suggesting this change.  But there is a way to avoid the lock;
  ~pl[wormhole-eval].  In particular, the lock is avoided in the
  implementations of ~il[accumulated-persistence] and
  ~il[forward-chaining-reports], which are not supported in ACL2(p)
  (~pl[unsupported-waterfall-parallelism-features]).

  ~/~/")

(deflabel note-6-2

; Total number of release note items: 41.

; Here is the example from Sol Swords for the item below about acl2-magic-mfc
; and acl2-magic-canonical-pathname.  The problem was that these "placeholder"
; functions are really clause processors, and hence should return a list of
; clauses, not (as erroneously done before) a single clause.
;
;  (defun foo (x)
;     (declare (ignore x))
;     nil)
;
;  (defthm foo-of-t
;     (foo t)
;     :hints (("goal" :clause-processor acl2-magic-mfc)))
;
;  (thm nil :hints (("goal" :use foo-of-t)))

; Defined function set-new-dispatch-macro-character, which we use instead of
; set-dispatch-macro-character to extend the reader with #\@, #\!, #\u, and in
; ACL2(h), also #\Y and #\Z.  This way, we check that the character hasn't
; already being appropriated for the reader by the host Lisp.  Now, for
; example, CLISP causes an error when trying to build ACL2(h), because #\Y is
; already defined as a dispatch macro character.

; Here is Robert Krug's example, trivially modified, for the heuristic
; improvement pertaining to the ancestors check.

;   (defstub rm-low-32 (addr x86) t)
;
;   (defaxiom n32p-rm-low-32-axiom
;     (implies (and (integerp addr)
;                   (<= 0 addr)
;                   (force (< (+ 3 addr) 1000)))
;              (and (integerp (rm-low-32 addr x86))
;                   (<= 0 (rm-low-32 addr x86)))))
;
;   (defstub foo-p (x) t)
;
;   (defaxiom axiom-2
;     (implies (<= 0 (rm-low-32 addr x86))
;              (foo-p (rm-low-32 addr x86))))
;
;   ; The following fails, which is to be expected.  But we expect to see a forcing
;   ; round.  This is indeed the case now, but it was not in Version_6.1 (and
;   ; probably many versions preceding that one).
;   (thm (implies (and (integerp addr)
;                      (<= 0 addr))
;                 (foo-p (rm-low-32 addr x86))))

; Improved error messages and documentation pertaining to certain ill-formed
; uses of syntaxp, in response to feedback from Robert Krug.

; Regarding the change for THE, note that the-error has been replaced by
; the-check, a 3-place function that results in a macroexpansion of THE forms
; that differs from what we got previously.

; In the process of modifying the ancestors-check heuristic to use var-counts,
; as mentioned below and explained further in the definition of macro
; var-or-fn-count-<, we changed var-fn-count to be a macro defined in terms of
; (partly) tail-recursive "flag" function var-fn-count-1, in analogy to what we
; already did for fn-count and fn-count-1.  Var-fn-count-1 is in :logic mode;
; the old var-fn-count nest was not.

; It had been possible to violate the first invariant on type-alists: no quotep
; is bound in a type-alist, but we fixed that.  See the long comment in
; subst-type-alist1.

; Added macro our-ignore-errors in raw Lisp, to ignore errors except for CLtL1
; (i.e., non-ANSI GCL).  Used it to define safe-open, which is open wrapped
; with our-ignore-errors.  Used safe-open to implement the change mentioned
; below for open-input-channel open-output-channel: "no longer cause an error
; when failing to open a channel".

; The Essay on Correctness of Meta Reasoning has been greatly improved, in
; particular with respect to its handling of meta-extract hypotheses.

; Here is an example that formerly broke into raw lisp, but no longer after the
; fix for "A hard Lisp error was possible for certain illegal functional
; substitutions", mentioned in the :doc below.
;
;   (encapsulate
;    ((f (x) t))
;    (local (defun f (x) (cons x x)))
;    (defthm f-prop
;      (consp (f x))))
;
;   (defthm main
;     (let ((y x))
;       (listp (f y))))
;
;   (defun g (x y)
;     (cons x y))
;
;   (defthm g-prop
;     (listp (g x y))
;     :hints (("Goal"
;              :use
;              (:functional-instance main
;                                    (f (lambda (v) (g v y)))))))

; In the course of modifying ACL2 to run on top of ANSI GCL, we did
; miscellaneous clean-up of various comments and documentation topics, and in a
; few cases, code.  In particular, ANSI GCL exposed a flaw in
; intern-in-package-of-symbol, which we slightly reworked as a result.

; In support of mfc-xx fixes documented below, mfc-relieve-hyp-raw now
; returns two values, as expected by call from mfc-relieve-hyp macro.

; We made the following changes during the process of adding support for
; building ACL2(h) on ANSI GCL.  While we have run a significant portion of the
; ACL2(h) regression suite on ACL2(h) built on a version of ANSI GCL, which was
; gracefully built for us by Camm Maguire, it stalled out with
; books/models/y86/y86-basic/common/x86-state.lisp.  Here is a summary of those
; changes, some of which might benefit other Lisps, although currently we only
; do automatic proclaiming for defuns for ANSI GCL.
;
; - We improved output-type-for-declare-form-rec and
;   output-type-for-declare-form for gcl, but as a result, no longer
;   attempt to do such declaims for *1* functions (see
;   install-defs-for-add-trip).
;
; - We now avoid function declaims for an abstract stobj export, which
;   is defined as a macro.
;
; - We now avoid using defun-one-output for functions like mfc-ts-raw
;   that return two values (the second of which is a ttree).
;
; - We rationalized saving the system in GCL (function
;   save-acl2-in-akcl), in particular to use the function
;   acl2-default-restart.
;
; - With-print-controls no longer messes with *print-pprint-dispatch*.
;
; - We did miscellaneous cleanup, including changing #+DRAFT-ANSI-CL-2
;   to #+cltl2.
;
; - The definition of global *float-internal-units-per-second* was
;   clearly intended to be the definition of global
;   *float-internal-time-units-per-second*.  This has been fixed.
;
; - For ANSI GCL, we added a workaround for undefinedness of
;   compiler-macro-function.
;
; - For ANSI GCL, we fixed memoize-fn to quote the symbol-function
;   passed to funcall.  Perhaps that could be done in other Lisps too,
;   but it seemed unwise to risk it.
;
; - We guarded an occurrence of (start-sol-gc) with #+Clozure, since
;   start-sol-gc is undefined otherwise.
;
; - We moved a type declaration on formals in the definition of
;   ser-decode-nat-large to be just after the formals (which avoids a
;   complaint by ANSI GCL, but is probably a good thing to do
;   regardless).

; Fixed bug in rare error message in check-certificate-file-exists, for the
; case: "argument k is t for certify-book".

; Arranged that character encoding for files in LispWorks is always as expected
; (latin-1 with linefeed for :EOL-STYLE).

; Strengthened the identification of built-ins with raw Lisp definitions (see
; constant *primitive-logic-fns-with-raw-code*, as well as source function
; fns-different-wrt-acl2-loop-only and related functions).

; Added documentation about guards in :doc princ$ and :doc io.

; Removed trailing whitespace from the ACL2 sources.  Thanks to Warren Hunt for
; suggesting this change.

; We improved :doc redundant-events, especially to clarify that progn and
; make-event forms are never redundant.  Thanks to Harsh Raju Chamarthi for an
; email exchange that led us to make this improvement.

; For GCL only, further increased the binding stack (except, still, on
; Windows); see the setting of si::*multiply-stacks* in init.lisp.

; Modified *home-page* for searching books, to point to tag
; #Searching_and_browsing_the_books.

; For SBCL only, increased --control-stack-size from 4 to 8.

  :doc
  ":Doc-Section release-notes

  ACL2 Version  6.2 (June, 2013) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version 6.1 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, changes at the system level, Emacs support, and
  experimental versions.  Each change is described in just one category, though
  of course many changes could be placed in more than one category.

  ~st[CHANGES TO EXISTING FEATURES]

  The macro ~ilc[top-level] has been changed, so that evaluation of a form
  ~c[(top-level x)] results in an error when evaluation of ~c[x] results in an
  error.  Thanks to Jared Davis for observing that when evaluating a file using
  ~ilc[ld], an interrupt of a call of a ~ilc[top-level] call in that file would
  not prevent evaluation of later forms in the file.

  The macro ~ilc[THE] no longer causes an error when ~il[guard]-checking is
  ~c[:NONE].  For example, it had been the case that evaluation of
  ~c[(the integer t)] always caused an error; but now, there is no error after
  executing command ~c[:]~ilc[set-guard-checking]~c[ :NONE].  Thanks to Jared
  Davis for asking for a way to avoid such errors.

  The error printed when attempting to ``reincarnate'' a package ~-[] that is,
  to define a package that is not part of the ACL2 logical ~il[world] but
  exists in raw Lisp because it was once part of the world ~-[] is now much
  more instructive.  In particular, it shows pathnames for the previous and
  proposed ~ilc[defpkg] events, and it shows the symbols that are imported by
  one but not the other.  Thanks to Jared Davis for requesting this
  improvement.

  Functions ~ilc[open-input-channel] and ~ilc[open-output-channel] no longer
  cause an error when failing to open a channel because of a permissions
  problem, but instead return ~c[(mv nil state)].  Thanks to Jared Davis for
  requesting this change.  (Note: this change does not apply if the host Lisp
  is non-ANSI, i.e., if the host Lisp is non-ANSI GCL.)

  The advanced ~il[meta-extract] mechanisms, provided for using facts from the
  ~il[world] or metafunction context, have been enhanced in the following
  ways, in collaboration with Sol Swords.  ~l[meta-extract] for more
  details.~bq[]

  It is now permissible to use calls of ~c[meta-extract-global-fact] in
  hypotheses of ~ilc[clause-processor] rules, much as they are used in
  hypotheses of ~ilc[meta] rules.  ~l[meta-extract].  Thanks to Sol Swords for
  requesting this feature.

  The utility ~c[meta-extract-global-fact] is now a macro, which expands to a
  corresponding call of the new function, ~c[meta-extract-global-fact+].  This
  new function takes an alternate, extra ~il[state] as an argument; it is not
  to be executed, and it operates on the alternate state, whose logical
  ~il[world] is intended to be the same as that of the ``live'' (usual) state.

  A new sort of value for the ~c[obj] argument is supported for
  ~c[meta-extract-global-fact] (and ~c[meta-extract-global-fact+]), which
  results in a term equating a function application to its result.
  ~l[meta-extract], in particular the discussion of ~c[:fncall].
  ~eq[]

  It is now possible for ~c[trace$] to avoid printing prefixes of the form
  ~c[\"n> \"] and ~c[\"<n \"], while also (optionally) avoiding indentation.
  ~l[trace$], in particular the discussion of ~c[:fmt!].  Thanks to Shilpi Goel
  for requesting this feature.

  It was possible for the ~il[guard-debug] feature to generate duplicate calls
  of ~c[extra-info] in hypotheses generated for guard verification.  We have
  eliminated duplicates of this sort.

  When ~ilc[in-theory] returns without error, it returns a value
  ~c[(:NUMBER-OF-ENABLED-RUNES k)], where ~c[k] is the length of the new
  current theory.  (Formerly, ~c[k] was returned.)  This value is thus printed
  when an ~c[in-theory] event is submitted at the top level.  Thanks to Gisela
  Rossi for feedback that led us to make this change.

  A new keyword parameter for ~ilc[ld] is ~c[:ld-missing-input-ok].  Its
  default value is ~c[nil], which causes an error, as before, upon failure to
  open a specified file.  Other legal values are ~c[t] and ~c[:WARN];
  ~pl[ld-missing-input-ok] and ~pl[ld].

  Extended ~c[*acl2-exports*], in particular adding ~c[UNSIGNED-BYTE-P] and
  ~c[SIGNED-BYTE-P] (thanks to a suggestion by Jared Davis)

  Even if the function ~c[f] is defined to take one or more ~il[stobj]
  arguments, the form ~c[(ec-call (f ...))] is now legal if all arguments of
  the call of ~c[f] are non-stobj objects, in any context where only ordinary
  object return values are expected.

  When the second argument of ~ilc[certify-book] is a symbol, that symbol
  formerly needed to be ~c[?] or ~c[t], in the ~c[\"ACL2\"] package.  Now, the
  ~il[symbol-package-name] of the second argument is ignored: any symbol whose
  ~il[symbol-name] is ~c[\"?\"] or ~c[\"T\"] is treated the same in that
  argument position as the symbol ~c[?] or ~c[t] in the ~c[\"ACL2\"] package,
  respectively.  Thanks to Warren Hunt and Nathan Wetzler for suggesting
  consideration of a more relaxed criterion for that second argument.

  (For system hackers, not standard ACL2 users:) Utilities
  ~ilc[initialize-event-user] and ~ilc[finalize-event-user] now each take a
  list of three arguments, ~c[(ctx body state)].  Thanks to Harsh Raju
  Chamarthi for requesting this change.

  ~st[NEW FEATURES]

  It is now permissible to specify a ~il[stobj] field that is itself either a
  stobj or an array of stobjs.  A new primitive, ~ilc[stobj-let], is provided
  in order to access or update such fields; ~pl[stobj-let].  Thanks to Warren
  Hunt and Sol Swords for requesting the ability to specify nested stobjs.

  New accessor function ~c[(mfc-world mfc)] returns the world component of a
  metafunction context.  ~l[extended-metafunctions].

  A new ~ilc[xargs] keyword, ~c[:SPLIT-TYPES], can be used to ``split'' the
  ~il[type] declarations from the ~il[guard] in the following sense.  By
  default, or when ~c[:SPLIT-TYPES] has value ~c[nil], one gets the existing
  behavior: the terms corresponding to type declarations are conjoined into the
  guard.  However, if ~c[:SPLIT-TYPES t] is specified, then that is not the
  case; instead, guard verification will require that these terms are proved
  under the hypothesis that the guard holds.  In this way, one can add type
  declarations to assist the host Lisp compiler without cluttering the
  function's guard.  ~l[xargs].  Thanks to Jared Davis for requesting this
  feature.

  Advanced users may want to ~pl[quick-and-dirty-subsumption-replacement-step]
  for a way to turn off a prover heuristic.  Thanks to those who have mentioned
  to us potential issues with this heuristic, most recently Dave Greve.

  ~st[HEURISTIC IMPROVEMENTS]

  We made changes to the ``ancestors check'' heuristic (source function
  ~c[ancestors-check-builtin]), as follows.~bq[]

  The heuristic could prevent a ~il[rewrite] rule's hypothesis from being
  rewritten to true, even when that hypothesis is of the form
  ~c[(force <term>)].  Now, forcing will take place as expected; ~pl[force].
  Thanks to Robert Krug for bringing this issue to our attention and sending an
  example, which we include as a comment in the ACL2 source code (see
  ~c[(deflabel note-6-2 ...)]).

  The heuristic is now delayed until after we check whether the hypothesis is
  already known, using ~il[type-set] reasoning alone (in particular, not using
  rewriting), to be true or to be false.  We believe that this is now the
  ``right'' order for those two operations.  We saw a slight speed up in the
  regression tests (about a percent) with this change, but that might be in the
  noise.

  A technical change makes the heuristic slightly less aggressive in preventing
  backchaining.  Roughly speaking, ordering checks based on function symbol
  counts could suffice to permit backchaining, where now variable counts also
  suffice.  Thanks to Robert Krug for showing us an example where backchaining
  led to a term with no free variables that was nevertheless subject to the
  ancestors check, preventing it from being rewritten.

  (For those who use ~ilc[defattach] to attach to ~c[ancestors-check])  We have
  used ~c[defrec] to introduce an `~c[ancestor]' data structure.  A new function,
  ~c[strip-ancestor-literals], should be used to obtain the literals from a
  list of ancestors, although ~c[strip-cars] will still work at this time.
  ~eq[]

  When we rewrite the current literal of the current clause we assume the
  falsity of the other literals and of the conclusions produced by forward
  chaining.  We have changed the order in which those assumptions are made,
  which affects the type-alist used during rewriting.  This has three effects:
  the new type-alist, which is sometimes stronger than the old one, may allow
  additional rules to fire, the choice of free vars may be different, and the
  order of the literals in forced subgoals may be different.  Should ``legacy''
  proofs fail under the new type-alist, we recommend looking for rules that are
  fired in the new proof that were not fired (on that same subgoal) in the old
  one.  Thanks to Dave Greve for sending us an example that led us to make this
  change.

  ~st[BUG FIXES]

  We fixed a soundness bug that could be exploited by calling system functions
  ~c[acl2-magic-mfc] or ~c[acl2-magic-canonical-pathname].  Thanks to Sol
  Swords for bringing this bug to our attention.

  We fixed a soundness bug in the handling of ~il[stobj]s, in which strings
  were recognized as stobjs in raw Lisp.  Thanks to Jared Davis for sending us
  a proof of ~c[nil] that exploited this bug.  We now have a much simpler
  example of this bug, as follows.
  ~bv[]
  (defstobj st fld)
  (defthm bad (stp \"abc\") :rule-classes nil)
  (defthm contradiction
    nil
    :hints ((\"Goal\" :in-theory (disable (stp)) :use bad))
    :rule-classes nil)
  ~ev[]

  We fixed bugs in extended metafunctions (~pl[extended-metafunctions]).  The
  macro ~c[mfc-ap] no longer takes a ~c[:TTREEP] keyword argument, because this
  argument could allow returning a tag tree that does not properly account for
  forcing.  The remaining ~c[mfc-xx] macros ~-[] ~c[mfc-relieve-hyp],
  ~c[mfc-rw+], ~c[mfc-rw], and ~c[mfc-ts] ~-[] still take a ~c[:TTREEP] keyword
  argument, but the corresponding functions when ~c[:TTREEP] is ~c[t] ~-[]
  ~c[mfc-relieve-hyp-ttree], ~c[mfc-rw+-ttree], ~c[mfc-rw-ttree], and
  ~c[mfc-ts-ttree] ~-[] were introduced with incorrect output signatures.  A
  complication is that ~c[mfc-relieve-hyp-ttree] was improperly defined in raw
  Lisp in a way that actually matched the incorrect signature!  All of these
  bugs have been fixed.  Perhaps any of them could have made it possible to
  prove ~c[nil], though we have not tried to do so.

  (Windows only) On Windows, it had been possible for ACL2 not to consider two
  pathnames to name the same file when the only difference is the case of the
  drive, e.g., `~c[C:]' vs. `~c[c:]'.  This has been fixed.  Thanks to Sol
  Swords for reporting this issue.

  Fixed a bug in the storing of rules for the tau system; ~pl[tau-system].
  (The error message mentions
  PARTITION-SIGNATURE-HYPS-INTO-TAU-ALIST-AND-OTHERS.)  Thanks to Sol Swords
  for reporting this bug and sending a simple example to illustrate it.

  It had been possible to admit the missing ~ilc[defthm] events printed by
  ~ilc[defabsstobj], and yet get an error when subsequently submitting the same
  ~c[defabsstobj] event, stating: ``Note discrepancy with existing formula''.
  The problem could occur when an expression of the form ~c[(or X Y)] occurred
  in one of those missing events, because ACL2 created it from the term
  ~c[(if X 't Y)] but then translated ~c[(or X Y)] to ~c[(if X X Y)], resulting
  in a mismatch.  This has been fixed.  Thanks to Jared Davis for reporting
  this bug using a simple example.

  A hard Lisp error was possible for certain illegal functional
  substitutions (~pl[lemma-instance]).  Thanks to Sol Swords for reporting this
  bug.

  We fixed a bug in the case that an exported function of a ~ilc[defabsstobj]
  event had a ~il[guard] of ~c[t].  Thanks to Jared Davis for sending a simple
  example when reporting this bug.

  We now avoid an infinite loop that could occur when attempting to close the
  standard character output channel (~pl[standard-co]).  Instead, an error
  message explains how to accomplish what was probably intended.  Thanks to
  Shilpi Goel for bringing this issue to our attention.

  (Windows only) Fixed a bug that was causing a hard error on Windows when ACL2
  encountered filenames starting with the tilde character (~c[~~]), for
  example, ~c[(ld \"~~/acl2-customization.lsp\")].  Thanks to Sol Swords for
  bringing this bug to our attention.  Also thanks to Harsh Raju Chamarthi for
  a useful conversation that led to a better fix than our first one.

  ~st[CHANGES AT THE SYSTEM LEVEL]

  ACL2 may now be built on recent versions of a new host Lisp, ANSI Gnu Common
  Lisp (GCL).  Traditional (non-ANSI) GCL was the original host Lisp underlying
  ACL2, and we are grateful for GCL support that we received from the late Bill
  Schelter and, more recently and particularly for ANSI GCL, from Camm Maguire.

  The `make' process suggested for book certification has changed
  substantially, thanks in large part to contributions from Jared Davis and Sol
  Swords.  We have seen the new process provide better performance on machines
  with many cores, and we expect maintenance advantages such as eliminating the
  need for Makefiles in individual book directories.  The ``classic'' process,
  which was based on community books file ~c[books/Makefile-generic], is still
  supported (~pl[books-certification-classic]) but may disappear in a future
  release of ACL2.  ~l[books-certification].  Most changes should be invisible
  to the user, other than improved `make'-level parallelism, with the exception
  of the following.
  ~bq[]
  o Variable ~c[ACL2_JOBS] is no longer supported, nor is it necessary; simply
  use `make' option `-j' instead.

  o Regressions now use `make' option ~c[-k] by default, which causes the
  regression to keep going after errors, rather than ~c[-i], which ignores
  errors.  If you encounter problems because of this change, use
  ~c[ACL2_IGNORE=-i] with your `make' command.

  o The `regression' target works for the experimental extension, ACL2(h)
  (~pl[hons-and-memoization]); target `regression-hons' no longer exists.
  ~eq[]
  Please let us know if you run into problems with the new infrastructure, as
  we consider the legacy infrastructure to be deprecated and we will probably
  eliminate much of it in the future.  In particular, circular dependencies
  were formerly prohibited at the directory level, but that is no longer the
  case, and we expect such cycles to occur in the future.

  Although ACL2 users don't typically modify raw Lisp variables, we have
  arranged to reset Lisp variable ~c[*default-pathname-defaults*] if necessary
  at startup so that it will not interfere with ACL2, in particular by messing
  up the initial connected book directory (~pl[cbd]).  Thanks to Jared Davis,
  Sol Swords, and Raymond Toy for helping us to identify this issue.

  ~st[EMACS SUPPORT]

  ~st[EXPERIMENTAL/ALTERNATE VERSIONS]

  In ACL2(h), ~ilc[print-object$] no longer uses the serialize printer except
  in system applications as before (e.g., write out ~c[.cert] files).  Thanks
  to Dave Greve for bringing this issue to our attention.

  Jared Davis contributed changes related to the ~ilc[memoize] utility of
  ACL2(h), including some low-level changes as well as the following.

  o ~ilc[Never-memoize] specifies that a given function should never
  be memoized.

  o Removed ~c[memoize-let], which may never have ever been used.

  o Removed the ~c[:inline] keyword option to memoize, which was just an alias
    for the ~c[:recursive] option.

  For ACL2(p), some anomalous behavior may no longer occur because prover
  calls (more specifically, trips through the ACL2 ``waterfall'') will return
  only after all sub-computations (threads) have finished.  Thanks to David
  Rager for contributing this improvement.

  ACL2(pr), which includes ~il[parallelism] (as for ACL2(p)) and non-standard
  analysis support for the ~il[real]s (as for ACL2(r)), now builds and can
  certify the community ~c[nonstd/] books.  Thanks to David Rager for his
  contribution to this capability.

  ~/~/")

(deflabel note-6-3

; Total number of release note items: 36.

; (CMUCL only) Allocated larger stack for CMUCL on 64-bit x86 installations.

; Evaluation of a redundant encapsulate event, in the case that the event is
; not equal to the corresponding earlier encapsulate event, now causes an
; additional message to be printed that directs the user to a new :doc topic --
; redundant-events -- which explains the situation (and includes some :doc that
; was formerly in :doc encapsulate).  Because of this, stop-redundant-event has
; been modified.  Thanks to Carl Eastlund for suggesting improvements along
; these lines.

; Modified functions stobjs-out and get-stobjs-out-for-declare-form according
; to a suggestion from Jared Davis, to cause an error when attempting to
; determine the stobjs-out for IF just as we already do for RETURN-LAST.

; Regarding the "bug in the case of a field of a (concrete) stobj that is an
; abstract stobj": this was a bug in the #-acl2-loop-only case of
; get-stobj-creator.

; Here is a little example showing the change to :by hints involving
; quote-normal form.  The proof of the THM failed until this change was made.
;
;   (defthm my-thm
;     (equal (car '(3 . 4)) 3)
;     :rule-classes nil)
;
;   (thm (equal (car (cons 3 4)) 3)
;        :hints (("Goal" :by my-thm)))

; (GCL only) Just below is a book, certifiable in ACL2 Version_6.2,
; illustrating the "obscure soundness bug due to an error in the GCL
; implementation of set-debugger-enable".  The problem was that the definition
; of set-debugger-enable-fn ended with #-acl2-loop-only code of the form (when
; (live-state-p state) ... state), which erroneously returns nil for a non-live
; state.
;
;   (in-package "ACL2")
;
;   (defthm false-formula
;     (equal (set-debugger-enable-fn nil (build-state))
;            nil)
;     :rule-classes nil)
;
;   (defthm true-formula
;     (implies (state-p1 s)
;              (state-p1 (set-debugger-enable-fn nil s)))
;     :hints (("Goal" :in-theory (enable state-p1)))
;     :rule-classes nil)
;
;   (defthm contradiction
;     nil
;     :hints (("Goal"
;              :use (false-formula
;                    (:instance true-formula
;                               (s (build-state))))))
;     :rule-classes nil)

; Improved our-truename (definition and calls) so that in case of an error, we
; get additional information.

; Added new :doc topic, ignored-attachment, pointing to it in the message
; printed by function ignored-attachment-msg.

; Changed all-equal to have a test of atom instead of endp, so that it
; could be guard-verifiable as requested by Eric Smith.

; Updated :doc make-event to clarify that it is legal to call make-event during
; make-event expansion, even outside an event context.

; Modified function maybe-add-command-landmark to avoid giving special
; treatment to with-prover-time-limit, since with-prover-time-limit is
; a full-fledged event constructor and hence this is not necessary.

; Removed definition of obsolete function, called acl2, from the
; sources.

; Slightly changed wording pertaining to the "combined" (xdoc) manual on the
; ACL2 home page.

; Fixed TAGS! target in GNUmakefile to do what it has been claming to
; do: rebuild TAGS even when TAGS is up-to-date with respect to the
; source files.

; Added new state global, acl2-sources-dir, in support of the new ACL2 startup
; banner in the case of an svn version.

; The guard for function ENDP has potentially been made trivially more
; efficient by using EQ to test against nil instead of EQUAL.

; In support of :by hint processing (see the :doc string below),
; remove-guard-holders now returns a term in quote normal form, even when the
; input term contains no guard holders.

; The startup banner can now be suppressed, though only by unsanctioned hacking
; in raw Lisp, as requested by Jared Davis.  See *print-startup-banner*, which
; has a comment explaining more about how this may not be appropriate and what
; needs to be done to suppress startup information.

; Removed support for "make fast", "make very-fast", and "make fast-meter",
; which as far as we know are no longer in use.  Thus, deleted
; quick-compile-acl2.  Also deleted no-object-file-or-out-of-date-object-file,
; since it was there only to support quick-compile-acl2.

; Improved the error message for #.expr when expr hasn't been defined by
; defconst.

; Modified printing of the banner for GCL, especially for versions 2.6.9 and
; later, due to a change in how GCL prints such banners.  (Printing of the
; banner still works fine with older versions of GCL.)

; For SBCL only, increased --control-stack-size from 8 to 16.

; For CMUCL only, declaimed len to be notinline, to avoid what appears to be a
; CMUCM compiler bug that shows up when attempting to certify the community
; book books/models/jvm/m1/defsys.lisp.

; Incorporated fix from Sol Swords for ACL2(h): "memoize time
; tracking: fix some subtle RDTSC-related problems that could cause
; safe-incf errors on machines that often switch your thread across
; cores".

; Just below is a book containing a proof of nil, which exploits the soundness
; bug reported by Jen Davis and Dave Greve, permitting a stobj to be bound by a
; let or mv-let form without being among the outputs of that form.  The bug was
; in translate11-let (and also, similarly, in translate11-mv-let); it was a
; coding bug that failed to distinguish between the original value of a formal,
; tbody, and an updated version of that formal.

;   (in-package "ACL2")
;
;   (defstobj st halt)
;
;   (defun foo (st)
;     (declare (xargs :stobjs st))
;     (let ((st (update-halt 0 st)))
;       (halt st)))
;
;   (defun bar ()
;     (declare (xargs :guard t))
;     (with-local-stobj
;      st
;      (mv-let (result st)
;              (let ((x (foo st)))
;                (declare (ignore x))
;                (mv (halt st) st))
;              result)))
;
;   (defthm thm1
;     (equal (bar) 0)
;     :rule-classes nil)
;
;   (defthm thm2
;     (equal (bar) nil)
;     :hints (("Goal" :in-theory (disable (bar))))
;     :rule-classes nil)
;
;   (defthm contradiction
;     nil
;     :hints (("Goal" :use (thm1 thm2)))
;     :rule-classes nil)

  :doc
  ":Doc-Section release-notes

  ACL2 Version  6.3 (October, 2013) Notes~/

  NOTE!  New users can ignore these release notes, because the
  ~il[documentation] has been updated to reflect all changes that are recorded
  here.

  Below we roughly organize the changes since Version  6.2 into the following
  categories of changes: existing features, new features, heuristic
  improvements, bug fixes, changes at the system level, Emacs support, and
  experimental versions.  Each change is described in just one category, though
  of course many changes could be placed in more than one category.

  ~st[CHANGES TO EXISTING FEATURES]

  The evaluation of a term from a ~ilc[bind-free] hypothesis had been expected
  to produce an alist binding free variables to terms.  While that is still
  legal, it is also legal for that evaluation to produce a list of such alists:
  then each is considered, until one of them permits all remaining hypotheses
  to be relieved.  ~l[bind-free].  Thanks to Sol Swords for requesting this
  enhancement.

  ACL2 continues to provide a way to specify keyword command abbreviations for
  the top-level loop; ~pl[ld-keyword-aliases].  However,
  ~ilc[ld-keyword-aliases] is now a ~il[table] rather than a ~il[state] global;
  it is thus no longer a so-called ~il[LD] special.  The functionality of
  ~c[set-ld-keyword-aliases] has essentially been preserved, except that it is
  now an event (~pl[events]), hence it may appear in a book; it is ~il[local]
  to a book (or ~ilc[encapsulate] event); and the ~ilc[state] argument is
  optional, and deprecated.  A non-local version (~c[set-ld-keyword-aliases!])
  has been added, along with corresponding utilities ~c[add-keyword-alias] and
  ~c[add-keyword-alias!] for adding a single keyword alias.
  ~l[ld-keyword-aliases].  Thanks to Jared Davis for correspondence that led us
  to make this change.

  The ~il[proof-checker] command ~c[(exit t)] now exits without a query (but
  still prints an event to show the ~c[:INSTRUCTIONS]).  Thanks to Warren Hunt
  for feedback leading us to make this change.

  We made the following minor changes to the behavior or ~c[dmr]; ~pl[dmr].
  First, if ~c[dmr] monitoring is enabled, then ~c[(dmr-start)] will have no
  effect other than to print a corresponding observation, and if monitoring is
  disabled, then ~c[(dmr-stop)] will have no effect other than to print a
  corresponding observation.  Second, it had been the case that when
  ~c[(dmr-start)] is invoked, the debugger was always automatically enabled
  with value ~c[t] (~pl[set-debugger-enable]), and the debugger remained
  enabled when ~c[(dmr-stop)] was invoked.  Now, the debugger is only enabled
  by ~c[(dmr-start)] if it is not already enabled and does not have setting
  ~c[:never].  Moreover, if such automatic enabling takes place, then the old
  setting for the debugger is restored by ~c[(dmr-stop)] unless
  ~ilc[set-debugger-enable] has first been called after that automatic
  enabling.  Finally, if the value of ~il[state] global variable
  ~c['debugger-enable] is ~c[:bt], then the new value will be ~c[:break-bt],
  not ~c[t].

  When a call of ~ilc[progn] is executed in the ACL2 loop, its constituent
  ~il[events] and their results are printed, just as was already done for calls
  of ~ilc[encapsulate].  Thanks to Jared Davis for a conversation causing us to
  consider this change.

  (CCL only) When ~ilc[set-debugger-enable] is invoked with an argument that
  prints a backtrace and CCL is the host Lisp, the backtrace will be limited to
  10,000 stack frames.  (We have seen more than 65,000 stack frames before this
  change.)  This limit is the value of raw Lisp variable
  ~c[*ccl-print-call-history-count*], which may be assigned another positive
  integer value to serve as the maximum number of stack frames to be printed.

  Improvements have been made pertaining to the disabling (inhibiting) of
  individual types of warning.  Now, inhibited warnings are implemented in a
  straightforward way using a separate ~il[table] for this purpose, the
  ~c[inhibit-warnings-table], rather than using the ~ilc[acl2-defaults-table].
  ~l[set-inhibit-warnings], and ~pl[set-inhibit-warnings!] for a variant that
  is not ~ilc[local] to an ~ilc[encapsulate] or a book in which it occurs.
  Thanks to Sol Swords for sending examples showing how
  ~ilc[set-inhibit-warnings] did not always behave as one might reasonably
  expect when books are involved.

  It had been the case that ~ilc[lp] took a single argument, ~c['raw].  This
  argument was not documented and also caused an error, so it has been
  eliminated.

  The functionality of ~ilc[make-event] has been significantly expanded.
  First: if the expansion is of the form ~c[(:OR e1 e2 ...)], then event forms
  ~c[e1], ~c[e2], and so on are evaluated, in order, until the evaluation of
  some ~c[ek] completes without error.  In that case, the expansion is treated
  simply as ~c[ek].  With this capability, alternative expansions can be
  attempted and the successful one does not need to be evaluated again.  See
  the new version of community book ~c[books/make-event/proof-by-arith.lisp]
  for an example.  Second, an expansion may be of the form ~c[(:DO-PROOFS e)],
  in which case the event ~c[e] is evaluated with proofs ~st[not] skipped;
  ~pl[ld-skip-proofsp].  Third, new keyword ~c[:EXPANSION?] can be used to
  avoid storing expansions in certificate files.  ~l[make-event].

  When a ~ilc[defun] event prints a failure message in the summary, that
  message now indicates when the failure is due to a failed proof of guard
  verification or a failed proof of the measure theorem.  Thanks to Shilpi Goel
  for requesting this enhancement.

  ~st[NEW FEATURES]

  ACL2 can now be instructed to time activities using real time (wall clock
  time) instead of run time (typically, cpu time).  ~l[get-internal-time].
  Thanks to Jared Davis for asking to be able to obtain real-time reports in
  event summaries.

  A new utility, ~ilc[sys-call+], is similar to existing utility ~ilc[sys-call]
  in that it executes a command.  Unlike ~c[sys-call], however, ~c[sys-call+]
  returns values that include output from the command (in addition to the exit
  status), rather than simply printing the command.  ~l[sys-call+].

  The new macro ~ilc[verify-guards+] extends the functionality of
  ~ilc[verify-guards] by permitting macro-aliases (~pl[macro-aliases-table]).
  ~l[verify-guards+].  Thanks to Jared Davis for requesting this feature and
  suggesting the use of ~ilc[make-event] in its implementation.  We have also
  modified ~ilc[verify-guards] to print a friendlier error message when its
  argument is a macro-alias.

  ~l[last-prover-steps] for a new utility that returns the number of prover
  steps most recently taken.

  ~st[HEURISTIC IMPROVEMENTS]

  The processing of ~c[:use] and ~c[:by] ~il[hints] has been changed in the
  following two rather subtle ways, thanks to suggestions from Sol Swords.
  ~bq[]

  o For ~c[:by] hints, the simplest check was an equality check, rather than a
  more general subsumption check.  That equality check was made after removing
  so-called ``guard holders'' (~ilc[must-be-equal], ~ilc[prog2$],
  ~ilc[ec-call], ~ilc[the]) from both the previous theorem and the purported
  theorem.  Now, guard-holder removal has been strengthened, so that the
  results are also put into so-called quote-normal form, for example replacing
  ~c[(cons '3 '4)] by ~c['(3 . 4)].

  o For a ~il[lemma-instance] provided to a ~c[:use] or ~c[:by] hint that
  is a ~c[:functional-instance], if a ~c[:do-not] hint (~pl[hints]) has
  specified that ~c[preprocess-clause] is not to be used, then preprocessing
  will not be used on the constraints.~eq[]

  We eliminated certain warnings about being ``weak'' for every
  ~c[:]~ilc[type-prescription] rule whose conclusion designates that the
  function call can be equal to one of its arguments, e.g.,
  ~c[(or (integerp (foo y)) (equal (foo y) y))].  In many cases (such as the
  one above), such warnings about ``weak'' simply aren't correct.

  ~st[BUG FIXES]

  Fixed a soundness bug that was permitting a ~il[stobj] to be bound by a
  ~ilc[let] or ~ilc[mv-let] form, without being among the outputs of that form.
  Thanks to Jen Davis and Dave Greve for reporting this bug.  Their report
  included an example which forms the basis for a proof of ~c[nil], included as
  a comment in the form ~c[(deflabel note-6-3 ...)] in ACL2 source file
  ~c[ld.lisp].

  (GCL only) Fixed an obscure soundness bug due to an error in the GCL
  implementation of ~ilc[set-debugger-enable].  For details, see the relevant
  comment in the ACL2 source code under ~c[(deflabel note-6-3 ...)].

  Fixed a bug in the case of a field of a (concrete) stobj that is an abstract
  stobj (~pl[nested-stobjs]).  Thanks to David Rager for bringing this bug to
  our attention.

  Splitter output for type ~c[if-intro] (~pl[splitter]) could formerly occur
  even when at most one subgoal is generated.  This has been fixed.

  Fixed a bug in ~ilc[wof], hence in ~ilc[psof] (which uses ~c[wof]), that was
  causing the printing of a bogus error message.

  A small logical bug has been fixed in the logical definition of
  ~ilc[sys-call-status].  Formerly it always returned ~c[(mv nil state)]
  whenever the oracle of the state is non-empty (~pl[state]).

  Fixed a bug that was causing an error upon evaluation of the form
  ~c[(set-prover-step-limit nil)].  Thanks to David Russinoff for reporting
  this error.

  The ~c[:measure] (if supplied) is now ignored when checking redundancy with
  respect to a non-recursive definition that is not defined within a
  ~ilc[mutual-recursion].  (~l[redundant-events] and ~pl[xargs].)  It had been
  possible to get a low-level ACL2 error in this situation.  Thanks to Jared
  Davis for reporting this bug with a helpful example.

  Eliminated a potential error when using ~ilc[comp] to compile an uncompiled
  function defined under ~ilc[progn!], which we observed in LispWorks.

  ~st[CHANGES AT THE SYSTEM LEVEL]

  The ACL2 sources are now publicly available between ACL2 releases, using svn;
  see the new ``~c[acl2-devel]'' project hosted by Google code at
  ~url[http://acl2-devel.googlecode.com].  Although such a copy of ACL2 is
  likely to work well with the latest svn (trunk) revision of the ACL2
  community books (~pl[community-books]), please take seriously the warning
  message printed at startup: ``The authors of ACL2 consider svn distributions
  to be experimental; they may be incomplete, fragile, and unable to pass our
  own regression.''  That message also provides instructions for bug reports.
  If you decide to use svn versions of either the community books or ACL2, then
  you should use both, as they tend to be kept in sync.  We fully expect ACL2
  releases to continue from time to time, as usual.  Thanks to Jared Davis for
  his efforts in setting up the new acl2-devel project and svn repository, and
  to him and David Rager for convincing us to distribute ACL2 sources via svn
  between releases.

  Thanks to a suggestion from Jared Davis, over 30 built-in functions are now
  declared to be inline in order to boost performance.  (The list may be found
  by searching ACL2 source file ~c[axioms.lisp] for ``~c[(declaim (inline]''.)

  Better support has been provided for command line arguments, especially those
  supplied directly by the user when calling ACL2.  For one, problems with
  quoting have been solved using ~c[\"$@\"] in place of ~c[$*].  Also, the
  function ~ilc[save-exec] now allows specification of arguments, both for the
  host Lisp as well as ``inert'' arguments that can be passed along to calls of
  programs (as with ~ilc[sys-call]).  A keyword argument, ~c[:return-from-lp],
  specifies a form to evaluate before quitting the read-eval-print loop at
  startup.  ~l[save-exec].  Also see the source function ~c[user-args-string]
  and its comments, source file ~c[acl2-init.lisp], for more information.
  Thanks to Jared Davis for suggesting the use of ~c[\"$@\"], as well as
  modifications to ~ilc[save-exec] and helpful conversations about that.

  A rather extensive overhaul has taken place for the function proclaiming
  mechanism.  As before, this is only used when the host Lisp is GCL.  However,
  building an executable is now faster for some Lisps, including GCL, by
  avoiding repeated recompilation and perhaps repeated initialization.

  (CCL only) We increased stack sizes when the host Lisp is CCL.  The default
  for recent CCL versions is equivalent to specifying `~c[-Z 2M]' on the
  command line, but saved ACL2 scripts (including experimental versions
  ACL2(h), ACL2(p), ACL2(r), and combinations of them) to `~c[-Z 64M]',
  representing a 32-fold increase.  Thanks to Jared Davis for pointing us to
  community books file ~c[books/centaur/ccl-config.lsp] and to Sol Swords for
  helpful discussions.

  (SBCL only) Fixed ~c[save-exec] for host Lisp SBCL to provide the same export
  of variable ~c[SBCL_HOME] that was provided in the original ~c[saved_acl2]
  script.

  (GCL only) We made changes, following suggestions from Camm Maguire (whom we
  thank for these suggestions), to support ACL2 builds on recent versions of
  GCL (2.6.8 and 2.6.10; we recommend against using GCL 2.6.9, since issues
  there were fixed in 2.6.10).  Specifically, we no longer set the hole size,
  and we allocate contiguous pages sufficient to run an ACL2 regression without
  failing due to memory limitations.

  ~st[EMACS SUPPORT]

  Modified file ~c[emacs/emacs-acl2.el] to eliminate some warnings that were
  appearing in a recent Emacs version, replacing ~c[(end-of-buffer)] by
  ~c[(goto-char (point-max))] and ~c[next-line] by ~c[forward-line].  Thanks to
  Warren Hunt for bringing the warnings to our attention.

  ~st[EXPERIMENTAL/ALTERNATE VERSIONS]

  (Allegro CL only) ACL2(h) now avoids blow-ups in hash table sizes that could
  be caused by ~il[hons-shrink-alist].  Thanks to Jared Davis for helping to
  debug this problem, and to David Rager for contributing the community book
  ~c[books/parsers/earley/earley-parser.lisp], which highlighted this problem.

  (SBCL only) Fixed a bug that was causing a Lisp break after turning on
  ~il[waterfall-parallelism].  Thanks to David Rager for confirming that our
  proposed fix is correct.

  ~/~/")

(deflabel the-method
  :doc
  ":Doc-Section Miscellaneous

  how to find proofs~/

  Also ~pl[introduction-to-the-theorem-prover] for a more detailed tutorial on
  how to prove theorems with ACL2.

  Many users develop proof scripts in an Emacs buffer and submit one event at a
  time to the theorem prover running in a ~c[*shell*] buffer.  The script
  buffer is logically divided into two regions: the events that have been
  accepted by the theorem prover and those that have not yet been accepted.  An
  imaginary ``barrier'' divides these two regions.  The region above the
  barrier describes the state of the ~c[*shell*] buffer (and ACL2's logical
  world).  The region below the barrier is the ``to do'' list.

  We usually start a proof project by typing the key lemmas, and main goal into
  the to do list.  Definitions are here just regarded as theorems to
  prove (i.e., the measure conjectures).  Then we follow ``The Method.''

  (1) Think about the proof of the first theorem in the to do list.  Structure
  the proof either as an induction followed by simplification or just
  simplification.  Have the necessary lemmas been proved? That is, are the
  necessary lemmas in the done list already?  If so, proceed to Step 2.
  Otherwise, add the necessary lemmas at the front of the to do list and repeat
  Step 1.

  (2) Call the theorem prover on the first theorem in the to do list and let
  the output stream into the *shell* buffer.  Abort the proof if it runs more
  than a few seconds.

  (3) If the theorem prover succeeded, advance the barrier past the successful
  command and go to Step 1.

  (4) Otherwise, inspect the failed proof attempt, starting from the beginning,
  not the end.  Basically you should look for the first place the proof attempt
  deviates from your imagined proof.  If your imagined proof was inductive,
  inspect the induction scheme used by ACL2.  If that is ok, then find the
  first subsequent subgoal that is stable under simplification and think about
  why it was not proved by the simplifier.  If your imagined proof was not
  inductive, then think about the first subgoal stable under simplification, as
  above.  Modify the script appropriately.  It usually means adding lemmas to
  the to do list, just in front of the theorem just tried.  It could mean
  adding hints to the current theorem.  In any case, after the modifications go
  to Step 1.~/

  We do not seriously suggest that this or any rotely applied algorithm will
  let you drive ACL2 to difficult proofs.  Indeed, to remind you of this we
  call this ``The Method'' rather than ``the method.''  That is, we are aware
  of the somewhat pretentious nature of any such advice.  But these remarks
  have helped many users approach ACL2 in a constructive and disciplined way.

  We say much more about The Method in the ACL2 book.  See the home page.  Also
  ~pl[set-gag-mode] for a discussion of a way for ACL2 to help you to use The
  Method.  And again, ~pl[introduction-to-the-theorem-prover] for a more
  detailed tutorial.

  Learning to read failed proofs is a useful skill.  There are several kinds of
  ``checkpoints'' in a proof: (1) a formula to which induction is being (or
  would be) applied, (2) the first formula stable under simplification, (3) a
  formula that is possibly generalized, either by cross-fertilizing with and
  throwing away an equivalence hypothesis or by explicit generalization of a
  term with a new variable.

  At the induction checkpoint, confirm that you believe the formula being
  proved is a theorem and that it is appropriately strong for an inductive
  proof.  Read the selected induction scheme and make sure it agrees with your
  idea of how the proof should go.

  At the post-simplification checkpoint, which is probably the most commonly
  seen, consider whether there are additional rewrite rules you could prove to
  make the formula simplify still further.  Look for compositions of function
  symbols you could rewrite.  Look for contradictions among hypotheses and
  prove the appropriate implications: for example, the checkpoint might contain
  the two hypotheses ~c[(P (F A))] and ~c[(NOT (Q (G (F A))))] and you might
  realize that ~c[(implies (p x) (q (g x)))] is a theorem.  Look for signs that
  your existing rules did not apply, e.g., for terms that should have been
  rewritten, and figure out why they were not.  Possible causes include that
  they do not exactly match your old rules, that your old rules have hypotheses
  that cannot be relieved here -- perhaps because some other rules are missing,
  or perhaps your old rules are disabled.  If you cannot find any further
  simplifications to make in the formula, ask yourself whether it is valid.  If
  so, sketch a proof.  Perhaps the proof is by appeal to a combination of
  lemmas you should now prove?

  At the two generalization checkpoints --- where hypotheses are discarded or
  terms are replaced by variables --- ask yourself whether the result is a
  theorem.  It often is not.  Think about rewrite rules that would prove the
  formula.  These are often restricted versions of the overly-general formulas
  created by the system's heuristics.

  ~l[proof-tree] for a discussion of a tool to help you navigate through ACL2
  proofs.")

(deflabel lp
  :doc
  ":Doc-Section Miscellaneous

  the Common Lisp entry to ACL2~/

  To enter the ACL2 ~il[command] loop from Common Lisp, call the Common
  Lisp program ~c[lp] (which stands for ``loop,'' as in ``read-eval-print
  loop'' or ``~il[command] loop.'')  The ACL2 ~il[command] loop is actually
  coded in ACL2 as the function ~ilc[ld] (which stands for ``load'').  The
  ~il[command] loop is just what you get by loading from the standard
  object input channel, ~ilc[*standard-oi*].  Calling ~ilc[ld] directly from
  Common Lisp is possible but fragile because hard lisp errors or
  aborts throw you out of ~ilc[ld] and back to the top-level of Common Lisp.
  ~c[Lp] calls ~ilc[ld] in such a way as to prevent this and is thus the
  standard way to get into the ACL2 ~il[command] loop.  Also
  ~pl[acl2-customization] for information on the loading of an
  initialization file.~/

  All of the visible functionality of ~c[lp] is in fact provided by ~ilc[ld],
  which is written in ACL2 itself.  Therefore, you should ~pl[ld]
  for detailed ~il[documentation] of the ACL2 ~il[command] loop.  We sketch it
  below, for novice users.

  Every expression typed to the ACL2 top-level must be an ACL2
  expression.

  Any ACL2 expression may be submitted for evaluation.  Well-formedness is checked.
  Some well-formed expressions cannot be evaluated because they involve (at some level)
  undefined constrained functions (~pl[encapsulate]).  In addition, ACL2 does not
  allow ``global variables'' in expressions to be evaluated.  Thus, ~c[(car '(a b c))]
  is legal and evaluates to ~c[A], but ~c[(car x)] is not, because there is no
  ``global context'' or binding environment that gives meaning to the variable symbol
  ~c[x].

  There is an exception to the global variable rule outlined above:
  single-threaded objects (~pl[stobj]) may be used as global variables
  in top-level expressions.  The most commonly used such object is the
  ACL2 ``current state,'' which is the value of the variable symbol
  ~ilc[state].  This variable may occur in the top-level form to be
  evaluated, but must be passed only to ACL2 functions ``expecting'' ~c[state]
  as described in the documentation for ~ilc[state] and for ~ilc[stobj]s in general.
  If the form returns a new ~il[state] object as one of its
  values, then that is considered the new ``current'' ~il[state] for
  the evaluation of the subsequent form.  ~l[state].

  ACL2 provides some support for the functionality usually provided by
  global variables in a read-eval-print loop, namely the saving of the
  result of a computation for subsequent re-use in another expression.
  ~l[assign] and ~pl[@].

  If the form read is a single keyword, e.g., ~c[:]~ilc[pe] or ~c[:]~ilc[ubt], then
  special procedures are followed.  ~l[keyword-commands].

  The ~il[command] loop keeps track of the ~il[command]s you have typed and
  allows you to review them, display them, and roll the logical ~il[state]
  back to that created by any ~il[command].  ~l[history].

  ACL2 makes the convention that if a top-level form returns three
  values, the last of which is an ACL2 ~il[state], then the first is
  treated as a flag that means ``an error occurred,'' the second is
  the value to be printed if no error occurred, and the third is (of
  course) the new ~il[state].  When ``an error occurs'' no value is
  printed.  Thus, if you execute a top-level form that happens to
  return three such values, only the second will be printed (and that
  will only happen if the first is ~c[nil]!).  ~l[ld] for details.")

#-acl2-loop-only
(defun-one-output compiled-function-p! (fn)

; In CMU Lisp, compiled-function-p is braindead.  It seems that the
; symbol-function of every defun'd function is a ``compiled'' object.
; Some are #<Interpreted Function ...> and others are #<Function ...>.
; I think the following test works.  Fn is assumed to be a symbol.

  #+cmu
  (not (eq (type-of (symbol-function fn)) 'eval:interpreted-function))
  #-cmu
  (compiled-function-p (symbol-function fn)))

(defun compile-function (ctx fn0 state)

; Fn0 can either be a symbol, (:raw sym), or (:exec sym).

  (declare (xargs :guard
                  (and (or (symbolp fn0)
                           (and (consp fn0)
                                (member-eq (car fn0) '(:raw :exec))
                                (consp (cdr fn0))
                                (null (cddr fn0))))
                       (state-p state))))
  (let ((wrld (w state))
        (fn (if (consp fn0)
                (cadr fn0)
              fn0)))
    (cond
     ((not (eq (f-get-global 'compiler-enabled state) t))
      (value (er hard ctx
                 "Implementation error: Compile-function called when ~x0."
                 '(not (eq (f-get-global 'compiler-enabled state) t)))))
     ((eq (getprop fn 'formals t 'current-acl2-world wrld)
          t)
      (er soft ctx
          "~x0 is not a defined function in the current ACL2 world."
          fn))
     (t
      (state-global-let*
       ((trace-specs (f-get-global 'trace-specs state))
        (retrace-p t))
       (prog2$
        #+acl2-loop-only
        nil
        #-acl2-loop-only
        (let ((trace-spec
               (assoc-eq fn (f-get-global 'trace-specs state))))
          (when trace-spec
            (untrace$-fn (list fn) state))
          (let* ((form (cltl-def-from-name fn wrld))
                 (*1*fn (*1*-symbol fn))
                 (raw-only-p  (and (consp fn0) (eq (car fn0) :raw)))
                 (exec-only-p (and (consp fn0) (eq (car fn0) :exec))))
            (cond
             ((not (or exec-only-p
                       (compiled-function-p! fn)))
              (cond (form
                     (eval (make-defun-declare-form fn form))))
              (compile fn)))
            (cond
             ((and (not raw-only-p)
                   (fboundp *1*fn)
                   (not (compiled-function-p! *1*fn)))
              #-acl2-mv-as-values ; may delete this restriction in the future
              (eval
               (make-defun-declare-form
                fn
                (cons 'defun (oneified-def fn wrld))))
              (compile *1*fn)))
            (when trace-spec
              (trace$-fn trace-spec ctx state))))
        (value fn)))))))

#-acl2-loop-only
(defun getpid$ ()

; This function is intended to return the process id.  But it may return nil
; instead, depending on the underlying lisp platform.

  (let ((fn
         #+allegro 'excl::getpid
         #+gcl 'si::getpid
         #+sbcl 'sb-unix::unix-getpid
         #+cmu 'unix::unix-getpid
         #+clisp (or (let ((fn0 (find-symbol "PROCESS-ID" "SYSTEM")))
                       (and (fboundp fn0) ; CLISP 2.34
                            fn0))
                     (let ((fn0 (find-symbol "PROGRAM-ID" "SYSTEM")))
                       (and (fboundp fn0) ; before CLISP 2.34
                            fn0)))
         #+ccl 'ccl::getpid
         #+lispworks 'system::getpid
         #-(or allegro gcl sbcl cmu clisp ccl lispworks) nil))
    (and fn
         (fboundp fn)
         (funcall fn))))

#-acl2-loop-only
(defun-one-output tmp-filename (dir suffix)

; Warning: If this function is changed, look at its call in save-gprof.lsp.

; Dir should be a filename in Unix-style syntax, possibly "".  We return a
; filename in Unix-style syntax.

  (let ((pid (and (not (eq (f-get-global 'keep-tmp-files *the-live-state*)
                           :no-pid))
                  (getpid$)))
        (dir (if (and (not (equal dir ""))
                      (not (eql (char dir (1- (length dir)))
                                *directory-separator*)))
                 (concatenate 'string dir *directory-separator-string*)
               dir)))
    (coerce (packn1 (list* dir
                           "TMP"
                           (if pid
                               (if suffix
                                   (list "@" pid "@" suffix)
                                 (list "@" pid "@"))
                             (if suffix
                                 (list suffix)
                               nil))))
            'string)))

(defun keep-tmp-files (state)
  (f-get-global 'keep-tmp-files state))

(defun comp-fn (fns gcl-flg tmp-suffix state)

; Gcl-flg should only be used with GCL, and causes .c and .h files to be left
; around after compilation.

  (declare (xargs :guard (and (state-p state)
                              (or (and (true-listp fns) fns)
                                  (symbolp fns))
                              (stringp tmp-suffix)
                              (not (equal tmp-suffix ""))))
           #+acl2-loop-only
           (ignore tmp-suffix))
  (cond
   ((eql 0 (f-get-global 'ld-level state))
    (pprogn (warning$ 'comp "Comp"
                      "Comp is ignored outside the ACL2 loop.")
            (value nil)))
   #-gcl
   (gcl-flg
    (er soft 'comp
        "Comp-gcl may only be used in GCL implementations."))
   ((not (eq (f-get-global 'compiler-enabled state) t))
    (value nil))
   (t
    (let ((fns (cond
                ((or (and (symbolp fns)
                          (not (eq fns t))
                          (not (eq fns :raw))
                          (not (eq fns :exec))
                          (not (eq fns nil)))
                     (and (consp fns)
                          (member-eq (car fns) '(:raw :exec))
                          (consp (cdr fns))
                          (null (cddr fns))))
                 (list fns))
                (t fns))))
      (cond
       ((and (consp fns)
             (null (cdr fns))
             (not gcl-flg))
        (compile-function 'comp (car fns) state))
       ((null fns)
        (er soft 'comp
            "We do not allow the notion of compiling the empty list of ~
             functions.  Perhaps you meant to do something else."))
       (t
        #+acl2-loop-only
        (value t)
        #-acl2-loop-only
        (state-global-let*
         ((retrace-p t))
         (let ((*package* *package*)
               (dir (or (f-get-global 'tmp-dir state)
                        (f-get-global 'connected-book-directory state)
                        ""))
               (raw-fns nil)
               (exec-fns nil)
               (trace-specs nil))
           (cond
            ((consp fns)
             (dolist (fn fns)
               (cond
                ((and (consp fn)
                      (member-eq (car fn) '(:raw :exec)))
                 (cond ((and (consp (cdr fn))
                             (null (cddr fn))
                             (symbolp (cadr fn)))
                        (cond ((eq (car fn) :raw)
                               (setq raw-fns (cons (cadr fn) raw-fns)))
                              (t ; :exec
                               (setq exec-fns (cons (cadr fn) exec-fns)))))
                       (t
                        (er hard 'comp
                            "Unexpected function specifier, ~x0."
                            fn))))
                ((symbolp fn)
                 (setq raw-fns (cons fn raw-fns))
                 (setq exec-fns (cons fn exec-fns)))
                (t (er hard 'comp
                       "Unexpected function specifier, ~x0."
                       fn)))
               (setq raw-fns (nreverse raw-fns))
               (setq exec-fns (nreverse exec-fns))))
            (t (setq raw-fns fns)
               (setq exec-fns fns)))
           (when (not (eq fns :exec))
             (setq trace-specs
                   (f-get-global 'trace-specs state))
             (untrace$)
             (let ((tmpfile (tmp-filename dir nil)))
               (compile-uncompiled-defuns
                tmpfile
                (if (or (eq fns t)
                        (eq fns :raw))
                    :some
                  raw-fns)
                gcl-flg)))
           (when (not (eq fns :raw))
             (when (and (null trace-specs)
                        (f-get-global 'trace-specs state))
               (setq trace-specs
                     (f-get-global 'trace-specs state))
               (untrace$))
             (let ((tmpfile (tmp-filename dir tmp-suffix)))
               (compile-uncompiled-*1*-defuns
                tmpfile
                (if (member-eq fns '(t :exec))
                    :some
                  exec-fns)
                gcl-flg)))
           (when trace-specs
             (trace$-lst trace-specs 'comp state))
           (value t)))))))))

#-acl2-loop-only
(defmacro comp (fns)
  (declare (ignore fns))
  nil)

#+acl2-loop-only
(defmacro comp (fns)

  ":Doc-Section Events

  compile some ACL2 functions~/

  NOTE: ~c[Comp] is a no-op if explicit compilation is suppressed;
  ~pl[compilation].  The documentation here assumes that this is not the case.

  ~bv[]
  Examples:
  :comp t          ; compile all uncompiled ACL2 functions
  (comp t)         ; same as above, but can be put into a book
  (comp :exec)     ; compile all uncompiled logic (``*1*'') definitions
  :comp foo        ; compile the defined function foo
  :comp (:raw foo) ; compile the raw Lisp version of the defined function foo
                     but not the corresponding logic definition
  :comp (foo bar)  ; compile the defined functions foo and bar
  :comp (foo (:raw bar))  ; compile the defined functions foo and bar, but for
                          ; bar do not compile the corresponding logic definition

  General Form:
  :comp specifier
  where specifier is one of the following:

    t                     compile all user-defined ACL2 functions that are
                            currently uncompiled (redefined built-in functions
                            are not recompiled)
    :exec                 same as t, except that only logic versions are
                            compiled (see below), not raw Lisp definitions
    :raw                  same as t, except that only raw Lisp definitions are
                            compiled, not logic version (see below)
    (name-1 ... name-k)   a non-empty list of names of functions defined by
                            DEFUN in ACL2, except that each name-i can be of
                            the form (:raw sym) or (:exec sym), where sym is
                          the name of such a function
    name                  same as (name)
  ~ev[]

  When you define a function in ACL2, you are really causing two definitions to
  be made ``under the hood'' in Common Lisp: the definition is submitted
  explicitly to raw Lisp, but so is a corresponding ``logic definition''.  If
  guards have not been verified, then only the logic definition will be
  evaluated; ~pl[guards-and-evaluation], in particular the section titled
  ``Guards and evaluation V: efficiency issues''.

  Thus, if you are not verifying ~il[guard]s and you want the benefit of Lisp
  compilation for speed and space efficiency, then you may want to place the
  form ~c[(comp :exec)] in your ~il[books].

  Generally it is not necessary to place the form ~c[(comp t)], or the form
  ~c[(comp :raw)], in a book, because ~ilc[certify-book] compiles the raw Lisp
  definitions anyhow, by default.  But you may wish to put ~c[(comp t)] or
  ~c[(comp fn1 fn2 ... fnk)] in a book when such a form precedes expensive
  calls of functions, for example for proofs involving calls of functions on
  large constants, or to support computationally expensive macroexpansion.

  As suggested by the examples above, if a function specifier is of the form
  ~c[(:raw fn)], then ~c[fn] will be compiled in raw Common Lisp but its
  corresponding logic definition will not be compiled; and for ~c[(:exec fn)],
  it's the other way around.

  The use of ~c[:comp] may create various files whose names start with
  ``~c[TMP*]'', but it then deletes them.  If you want to save these files,
  evaluate ~c[(assign keep-tmp-files t)].~/

  Also ~pl[set-compile-fns] for a way to compile each function as it is
  defined.  But note that ~c[set-compile-fns] is ignored during
  ~ilc[include-book].

  Note that if functions are traced (~pl[trace$]), then ~c[comp] will first
  untrace the functions that are to be compiled, then will do the compile(s),
  and finally will re-trace the functions that it untraced (using their
  original trace specs).  In particular, if you have traced a function and then
  you compile it using ~c[:comp], the resulting traced function will be
  compiled as well unless you specified ~c[:compile nil] in your trace spec;
  and after you untrace the function it will definitely run compiled.

  We conclude with a technical remark only for those who use trust tags to
  write raw Lisp code.  ~c[:Comp] generally creates files to compile unless it
  is given a single function to compile.  Those files contain the ACL2
  definitions of all functions to compile, omitting those in the lists obtained
  by evaluating the forms ~c[(@ logic-fns-with-raw-code)] and
  ~c[(@ program-fns-with-raw-code)].  ~c[:Comp] skips compilation for functions
  that are already compiled, as is typically the case when you redefine
  functions in raw Lisp using the utility ~c[include-raw] defined in community
  book ~c[books/tools/include-raw.lisp].  But if you define interpreted (as
  opposed to compiled) functions with raw Lisp code, say by using trust
  tags (~pl[defttag]) and ~ilc[progn!], then you are advised to add all such
  symbols to one of the lists stored in the two ~il[state] globals above: to
  ~c[logic-fns-with-raw-code] if the function symbol is in ~c[:]~ilc[logic]
  mode, else to ~c[program-fns-with-raw-code].  Then, instead of the
  corresponding ACL2 definition (without raw Lisp code) being written to a
  file, the function symbol will be passed directly to the Lisp ~c[compile]
  function.  Note that the above two state globals are both untouchable, so you
  may need to deal with that before modifying them, for example as
  follows (also ~pl[remove-untouchable]).
  ~bv[]
  (defttag t)
  (state-global-let*
   ((temp-touchable-vars t set-temp-touchable-vars))
   (progn!