//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Beam/FootprintItems.cpp
//! @brief     Implements FootprintItem classes
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Beam/FootprintItems.h"
#include "Device/Beam/FootprintGauss.h"
#include "Device/Beam/FootprintSquare.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString GaussianFootprint("GaussianFootprint");
const QString SquareFootprint("SquareFootprint");

} // namespace Tag
} // namespace

std::unique_ptr<IFootprint> FootprintNoneItem::createFootprint() const
{
    return {};
}

FootprintGaussianItem::FootprintGaussianItem(double value)
{
    m_gaussianFootprintValue.init("Width ratio", "The ratio of beam and sample full widths", value,
                                  Unit::unitless, 3 /* decimals */, RealLimits::nonnegative(),
                                  "ratio");
}

void FootprintGaussianItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));
    m_gaussianFootprintValue.writeTo(w, Tag::GaussianFootprint);
}

void FootprintGaussianItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        if (tag == Tag::GaussianFootprint)
            m_gaussianFootprintValue.readFrom(r, tag);
        else
            r->skipCurrentElement();
    }
}

std::unique_ptr<IFootprint> FootprintGaussianItem::createFootprint() const
{
    return std::make_unique<FootprintGauss>(m_gaussianFootprintValue);
}

FootprintSquareItem::FootprintSquareItem(double value)
{
    m_squareFootprintValue.init("Width ratio", "The ratio of beam and sample full widths", value,
                                Unit::unitless, 3 /* decimals */, RealLimits::nonnegative(),
                                "ratio");
}

void FootprintSquareItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));
    m_squareFootprintValue.writeTo(w, Tag::SquareFootprint);
}

void FootprintSquareItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        if (tag == Tag::SquareFootprint)
            m_squareFootprintValue.readFrom(r, tag);
        else
            r->skipCurrentElement();
    }
}

std::unique_ptr<IFootprint> FootprintSquareItem::createFootprint() const
{
    return std::make_unique<FootprintSquare>(m_squareFootprintValue);
}
