#include "GUI/Model/Project/ProjectUtil.h"
#include "GUI/Support/Util/ItemFileNameUtil.h"
#include "Tests/GTestWrapper/google_test.h"
#include "Utils.h"
#include <QDir>
#include <QFile>
#include <QTextStream>
#include <iostream>

namespace {

QStringList nonXMLDataInDir(const QString& dirname)
{
    QDir dir(dirname);

    if (!dir.exists())
        throw std::runtime_error("GUI::Project::Util::nonXMLDataInDir -> Error. Non existing "
                                 "directory '"
                                 + dirname.toLatin1() + "'.");

    return dir.entryList(GUI::Model::FilenameUtil::nonXMLFileNameFilters());
}

} // namespace

class TestProjectUtils : public ::testing::Test {
protected:
    //! Helper function to create test file in a given directory (directory should exist).
    void createTestFile(const QString& dirname, const QString& fileName)
    {
        QString filename = dirname.isEmpty() ? fileName : dirname + "/" + fileName;

        QFile file(filename);
        if (!file.open(QIODevice::WriteOnly | QIODevice::Text))
            throw std::runtime_error("TestGUI::Project::Util::createTestFile -> Error. "
                                     "Cannot create file");

        QTextStream out(&file);
        out << "Test file " << 42 << "\n";
        file.close();
    }
};

TEST_F(TestProjectUtils, nonXMLDataInDir)
{
    const QString projectDir = "test_ProjectUtils";

    QDir dir(projectDir);
    if (dir.exists()) {
        EXPECT_TRUE(QDir(projectDir).removeRecursively());
        EXPECT_FALSE(dir.exists());
    }

    UTest::GUI::create_dir(projectDir);
    EXPECT_TRUE(QFile::exists(projectDir));

    EXPECT_EQ(1, 1);
    QStringList test_nonxml_files{"jobdata_job2_0.int.gz", "refdata_job2_0.int.gz",
                                  "realdata_box_nanodots_0.int.gz", "realdata_untitled_0.int.gz"};

    const QString ext = QString(GUI::Project::Util::projectFileExtension);
    QStringList test_some_other_files{"user_0.int.gz", "b.txt", "untitled" + ext};

    std::cout << "create files ..." << std::endl;
    for (const auto& name : test_nonxml_files + test_some_other_files) {
        createTestFile(projectDir, name);
        EXPECT_TRUE(QFile::exists(projectDir + "/" + name));
    }

    std::cout << "check nonXMLDataInDir ..." << std::endl;
    // Check nonXMLDataInDir method, should contain only files from test_nonxml_files.
    QStringList nonxml = nonXMLDataInDir(projectDir);

    std::cout << "sort files ..." << std::endl;
    nonxml.sort();
    test_nonxml_files.sort();
    EXPECT_EQ(nonxml.size(), 4);
    EXPECT_EQ(test_nonxml_files, nonxml);

    std::cout << "remove nonxml files ..." << std::endl;
    for (auto& name : nonxml)
        QFile::remove(projectDir + "/" + name);

    std::cout << "check that no files left ..." << std::endl;
    nonxml = nonXMLDataInDir(projectDir);
    EXPECT_EQ(nonxml.size(), 0);
}
