#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "canlxx.h"

namespace AuthN {

  Context::Context(ContextType env):valid_(false) { //scan the application env. 
    if(env == EmptyContext) return;
    if((env == ClientMinimalContext) || (env == ClientFullContext)) {
      char* val = NULL;
      std::string home;
      val = getenv("X509_USER_PROXY");
      home = getenv("HOME");
      if(val) {
        certpath_ = val;
        keypath_ = val;
      } else {
        val = getenv("X509_USER_CERT");
        if(val) {
          certpath_ = val;
        } else {
          certpath_ = home + "/.emi/usercert.pem"; //"~/.emi/usercert.pem";
        };
        val = getenv("X509_USER_KEY");
        if(val) {
          keypath_ = val;
        } else {
          keypath_ = home + "/.emi/userkey.pem"; //"~/.emi/userkey.pem";
        };
      };
      val = getenv("X509_CERT_DIR");
      if(val) {
        capath_ = val;
      } else {
        capath_ = "/etc/grid-security/certificates";
      };
      val = getenv("X509_CRL_DIR");
      if(val) {
        crlpath_ = val;
      } else {
        crlpath_ = capath_;
      };
    };
    if((env == ServiceMinimalContext) || (env == ServiceFullContext)) {
      certpath_ = "/etc/grid-security/hostcert.pem";
      keypath_ = "/etc/grid-security/hostkey.pem";
      capath_ = "/etc/grid-security/certificates";
      crlpath_ = capath_;
    };
    if((env == ClientFullContext) || (env == ServiceFullContext)) {
      struct stat st;
      if((certpath_.empty()) || (stat(certpath_.c_str(),&st) != 0) || (!S_ISREG(st.st_mode))) {
        last_error_ = Status(-1,"Certificate file not accessible"); return;
      };
      if((keypath_.empty()) || (stat(keypath_.c_str(),&st) != 0) || (!S_ISREG(st.st_mode))) {
        last_error_ = Status(-1,"Private file not accessible"); return;
#ifndef WIN32
      } else if(st.st_mode & (S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)) {
        last_error_ = Status(-1,"Private key file has unsafe permissions"); return;
#endif
      };
      if((capath_.empty()) || (stat(capath_.c_str(),&st) != 0) || (!S_ISDIR(st.st_mode))) {
        last_error_ = Status(-1,"CA directory not accessible"); return;
      };
      if((crlpath_.empty()) || (stat(crlpath_.c_str(),&st) != 0) || (!S_ISDIR(st.st_mode))) {
        last_error_ = Status(-1,"CRL directory not accessible"); return;
      };
    };
    valid_ = true;
    return;
  }

  Context::~Context(void) {
  }

  Context::operator bool(void) const {
    return valid_;
  }

  bool Context::operator!(void) const {
    return !valid_;
  }

  Status Context::GetStatus(void) const {
    return last_error_;
  }

  Context& Context::Copy(void) const {
    return *(new Context(*this));
  }

  std::string Context::GetCAPath(void) const {
    return capath_;
  }

  void Context::SetCAPath(const std::string& path) {
    valid_ = true;
    capath_ = path;
  }

  std::string Context::GetCRLPath(void) const {
    return crlpath_;
  }

  void Context::SetCRLPath(const std::string& path) {
    valid_ = true;
    crlpath_ = path;
  }

  void Context::SetCredentials(const std::string& certpath,const std::string& keypath){
    valid_ = true;
    certpath_ = certpath;
    keypath_ = keypath;
  }

  void Context::SetCredentials(const std::string& certpath) {
    valid_ = true;
    certpath_ = certpath;
    keypath_ = certpath;
  }

  std::string Context::GetCertPath(void) const {
    return certpath_;
  }

  std::string Context::GetKeyPath(void) const {
    return keypath_;
  }

  std::string Context::Password(const std::string& /*type*/, const std::string& /*source*/) {
    return std::string();
  }

  std::string Context::Password(const std::string& /*message*/) {
    return std::string();
  }

  void Context::Message(const std::string& /*message*/) {
  }

  void Context::LogFormat(LogLevel level, const std::string& format, ...) {
    va_list args;
    va_start(args,format);
    std::string buf(1024,'\0');
    vsnprintf((char*)buf.c_str(),1024,format.c_str(),args);
    va_end(args);
    Log(level,buf);
  }

  void Context::Log(LogLevel level, const std::string& message) {
    std::cout<<message.substr(0,message.find('\0'))<<std::endl;
  }
}
