/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "cdw_task.h"
#include "cdw_string.h"
#include "gettext.h"
#include "cdw_widgets.h"
#include "cdw_debug.h"
#include "cdw_config.h"
#include "cdw_ext_tools.h"
#include "cdw_utils.h"
#include "cdw_main_window.h"
#include "cdw_erase_disc.h"

extern cdw_config_t global_config;


static cdw_rv_t cdw_task_set_allowed_modes(cdw_task_t *task, cdw_disc_t *disc);
static void     cdw_task_set_allowed_modes_message(cdw_id_t id);
static void     cdw_task_set_allowed_disc_modes(cdw_task_t *task, cdw_disc_t *disc);
static void     cdw_task_set_allowed_disc_modes_growisofs(cdw_task_t *task);
static void     cdw_task_set_allowed_disc_modes_cdrecord(cdw_task_t *task, cdw_disc_t *disc);
static void     cdw_task_set_allowed_disc_modes_xorriso(cdw_task_t *task, cdw_disc_t *disc);

static cdw_rv_t cdw_task_get_initial_options(cdw_task_t *task, cdw_disc_t *disc);
static void     cdw_task_reset_state(cdw_task_t *task);
static cdw_rv_t cdw_task_resolve_allowed_writing_modes(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_task_select_tools_for_task(cdw_task_t *task, cdw_disc_t *disc);
static bool     cdw_task_select_tool(cdw_task_t *task, cdw_disc_t *disc);

//static cdw_id_t cdw_task_select_tool_family(cdw_task_t *task, cdw_disc_t *disc);
//static bool cdw_task_select_tool_sub(cdw_task_t *task, cdw_id_t tool_family);
static void cdw_task_reset_tools(cdw_task_t *task);
static const char *cdw_task_get_task_label(cdw_id_t taks_id);
static void cdw_task_debug_print_tools(cdw_task_t *task);
static void cdw_task_e(const char *title, const char *message);

enum {
	IMAGE_TO_NONEMPTY,
	NONEMPTY_DVD_RP_DL,
	NONEMPTY_DVD_AND_CDRECORD
} session_mode_message_ids;


/* can be called explicitly in some situations, but
   when you create new task, you can use CDW_TASK_CREATE_TASK macro */
void cdw_task_reset_state(cdw_task_t *task)
{
	task->id = CDW_TASK_NONE;
	cdw_task_reset_tool_status(task);

	task->burn.session_mode = CDW_SESSION_MODE_INIT;
	task->burn.session_modes_n_labels = 0;
	task->burn.disc_mode = CDW_DISC_MODE_INIT;
	task->burn.disc_modes_n_labels = 0;

	task->burn.speed_id = 0;
	task->burn.verify = false;
	task->burn.dummy = false;
	task->burn.eject = false;

	task->burn.data_size_mb = 0.0;

	task->burn.growisofs.other_growisofs_options = (char *) NULL;

	task->burn.cdrecord.burnproof = true;
	task->burn.cdrecord.pad = false;
	task->burn.cdrecord.pad_size = 0;
	task->burn.cdrecord.other_cdrecord_options = (char *) NULL;

	task->burn.xorriso.other_xorriso_burn_options = (char *) NULL;

	task->burn.iso9660_file_fullpath = (char *) NULL;


	task->erase.speed_id = 0;
	task->erase.erase_mode = CDW_ERASE_MODE_ALL;
	task->erase.format_mode = CDW_ERASE_DVD_RW_FORMAT_NONE; /* it can stay as "NONE" when we use cdrecord for erasing */
	task->erase.erase_time = 0;
	task->erase.eject = false;

	task->calculate_digest.file_size = 0;
	task->calculate_digest.source_fd = -1;
	task->calculate_digest.target_fd = -1;
	task->calculate_digest.read_and_write = (void * (*)(void *)) NULL;

	task->create_image.iso9660_file_fullpath = (char *) NULL;
	task->create_image.volume_id[0] = '\0';
	task->create_image.mkisofs.iso_level = 0;
	task->create_image.mkisofs.root_dir = (char *) NULL;
	task->create_image.mkisofs.other_mkisofs_options = (char *) NULL;
	task->create_image.xorriso.other_xorriso_iso_options = (char *) NULL;
	task->create_image.pad = true;

	task->disc = (cdw_disc_t *) NULL;

	cdw_task_reset_tools(task);

	return;
}






void cdw_task_reset_tools(cdw_task_t *task)
{
	task->erase.tool.label = (char *) NULL;
	task->erase.tool.id = CDW_TOOL_NONE;
	task->burn.tool.label = (char *) NULL;
	task->burn.tool.id = CDW_TOOL_NONE;
	task->calculate_digest.tool.label = (char *) NULL;
	task->calculate_digest.tool.id =  CDW_TOOL_NONE;
	task->create_image.tool.label = (char *) NULL;
	task->create_image.tool.id = CDW_TOOL_NONE;

	return;
}




void cdw_task_reset_tool_status(cdw_task_t *task)
{
	task->tool_status.ok = true;
	task->tool_status.mkisofs = CDW_TOOL_STATUS_OK;
	task->tool_status.cdrecord = CDW_TOOL_STATUS_OK;
	task->tool_status.calculate_digest = CDW_TOOL_STATUS_OK;
	task->tool_status.growisofs = CDW_TOOL_STATUS_OK;
	task->tool_status.dvd_rw_mediainfo = CDW_TOOL_STATUS_OK;
	task->tool_status.libcdio = CDW_TOOL_STATUS_OK;
	task->tool_status.xorriso = CDW_TOOL_STATUS_OK;

	return;
}




/**
   \brief Check tool status field, display errors messages (if any)

   This function checks task->tool_status.* fields. If value of any of the
   fields is not CDW_TOOL_STATUS_OK then suitable error message is displayed
   and task->tool_status.ok is set to false. Otherwise task->tool_status.ok
   is set to true.

   Values of task->tool_status.* fields are not modified to CDW_TOOL_STATUS_OK
   before returning.

   It is a good idea to call this function after calling any
   run_command_*() function.

   \param task - task that you want to check
*/
cdw_rv_t cdw_task_check_tool_status(cdw_task_t *task)
{
	if (task->tool_status.mkisofs == CDW_TOOL_STATUS_OK
	    && task->tool_status.cdrecord == CDW_TOOL_STATUS_OK
	    && task->tool_status.calculate_digest == CDW_TOOL_STATUS_OK
	    && task->tool_status.growisofs == CDW_TOOL_STATUS_OK
	    && task->tool_status.dvd_rw_mediainfo == CDW_TOOL_STATUS_OK
	    && task->tool_status.libcdio == CDW_TOOL_STATUS_OK
	    && task->tool_status.xorriso == CDW_TOOL_STATUS_OK) {

		task->tool_status.ok = true;
		cdw_sdm ("INFO: task->tool_status.ok = true\n");
		return CDW_OK;
	} else {
		task->tool_status.ok = false;
		cdw_vdm ("WARNING: task->tool_status.ok = false\n");
	}

	/* *** cdrecord *** */

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_DEVICE);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("cdrecord can't access CD/DVD drive. Please check your settings in Configuration -> Hardware."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_NO_SUCH_FILE_OR_DIR);
		/* this is similar to CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR:
		   cdrecord either writes iso image file (provided by option)
		   or data stream from mkisofs. So this is eiter wrong
		   image file (but we checked this in check_write_preconditions())
		   or some invalid value in "other cdrecord options" */
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("cdrecord reports problems - it can't find some files. Please make sure that string in Configuration -> Writing -> \"Other cdrecord options\" field is correct."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_SEND_CUE_SHEET);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("cdrecord reports errors when attempting to write to optical disc, probably selected session mode is not acceptable. Please consult log file ('L' hotkey in main window)."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_BLANK_DVD_RWP_NON_RICOH);
		/* 2TRANS: this is title of dialog window */
		cdw_task_e(_("Warning"),
			   /* 2TRANS: this is message in dialog window */
			   _("cdrecord reported some problems with erasing optical disc, the disc may be still non-empty, please check it."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_CANNOT_OPEN_NEXT);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("cdrecord reported some problems with writing to optical disc, the disc was not appendable (probably it was closed)."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_ERROR_AFTER);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("cdrecord reported some problems with writing to optical disc, most probably data was not written. Try using different tool."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_DATA_MAY_NOT_FIT);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Size of selected files or ISO image exceeds size of free space available on optical disc."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_BAD_OPTION) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_BAD_OPTION\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_BAD_OPTION);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("One of cdrecord options is incorrect. Please check \"Other cdrecord options\" field in Configuration -> Writing."));
	}

	if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK) {
		cdw_vdm ("ERROR: masked tool_status.cdrecord = 0x%08x / CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK\n",
			 (unsigned int) task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("cdw can't find valid disc in drive. Perhaps drive tray is open."));
	}


	/* *** libcdio *** */

	if (task->tool_status.libcdio & CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL) {
		cdw_vdm ("ERROR: masked tool_status.libcdio = 0x%08x / CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL\n",
			 (unsigned int) task->tool_status.libcdio & CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't open CD/DVD drive device. Please check your settings in Configuration -> Hardware."));
	}

	if (task->tool_status.libcdio & CDW_TOOL_STATUS_LIBCDIO_READ_FAIL) {
		cdw_vdm ("ERROR: masked tool_status.libcdio = 0x%08x / CDW_TOOL_STATUS_LIBCDIO_READ_FAIL\n",
			 (unsigned int) task->tool_status.libcdio & CDW_TOOL_STATUS_LIBCDIO_READ_FAIL);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't read optical disc. Please check the disc."));
	}


	/* *** growisofs *** */

	if (task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_FILES) {
		cdw_vdm ("ERROR: masked tool_status.growisofs = 0x%08x / CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_FILES\n",
			 (unsigned int) task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_FILES);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't burn these files to optical disc. Too little space or no space left on optical disc."));
	}

	if (task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_IMAGE) {
		cdw_vdm ("ERROR: masked tool_status.growisofs = 0x%08x / CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_IMAGE\n",
			 (unsigned int) task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_IMAGE);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't burn this ISO image file to optical disc. Too little space or no space left on optical disc."));
	}

	if (task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE) {
		cdw_vdm ("ERROR: masked tool_status.growisofs = 0x%08x / CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE\n",
			 (unsigned int) task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't write to optical disc, it is not appendable. Please check the disc."));
	}

	if (task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION) {
		cdw_vdm ("ERROR: masked tool_status.growisofs = 0x%08x / CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION\n",
			 (unsigned int) task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window
			      don't make an impression that the disc *really*
			      is empty - only suggest it to user */
			   _("Optical disc is probably empty. Check the disc and consider using \"Start new, appendable disc\" option in \"Session writing mode\" dropdown in write wizard."));
	}

	if (task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_IO_ERROR) {
		cdw_vdm ("ERROR: masked tool_status.growisofs = 0x%08x / CDW_TOOL_STATUS_GROWISOFS_IO_ERROR\n",
			 (unsigned int) task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_IO_ERROR);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Unknown error occurred. Please retry with different options."));
	}

	if (task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY) {
		cdw_vdm ("ERROR: masked tool_status.growisofs = 0x%08x / CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY\n",
			 (unsigned int) task->tool_status.growisofs & CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Action failed because some files were larger than 4GB.\n\nYou may want to retry the action with \"-allow-limited-size   -udf\" options in Configuration -> ISO filesystem -> Other mkisofs options field, but that may result in data being unreadable on some systems that don't support UDF filesystem.\n\nConsult log file ('L' hotkey in main window) to check which file caused problems."));
	}


	/* *** mkisofs *** */

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't create image: some file names are too long. Try toggling \"Long Joliet names\" checkbox in Configuration -> ISO filesystem."));
	}

	if ( (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT)
	     && (!(task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_THE_SAME_JOLIET_NAME))) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_JOLIET_TREE_SORT);
		/* this condition is to display this message only when
		   "the same joliet name" wasn't signaled before */
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't create image. Check your settings in Configuration -> ISO filesystem."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_CANT_FIND_BOOT_IMAGE);
		/* since the task has been started, then there is _some_
		   path to boot image, but it is incorrect (it should be
		   relative to files specified as main arg to mkisofs) */
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Can't create image. Path to boot image file is incorrect. Please consult \"mkisofs\" or \"cdw\" manual page."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_NO_SUCH_FILE_OR_DIR);
		/* Since all files are fed to mkisofs through graft file,
		   and I think that there can be no error (other than files
		   suddenly disappearing from hard disc), the problem is
		   with some words in "other mkisofs options" field. */
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("mkisofs reports problems: it can't find some files. Possible reasons:\none of selected files has been deleted or is a broken symbolic link, or there is incorrect option in \"Other mkisofs options\" field in Configuration -> ISO filesystem tab. Check log for details ('L' key)."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_BAD_OPTION) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_BAD_OPTION\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_BAD_OPTION);
		/* options provided to mkisofs are set either by checkboxes
		   and input fields that are validated, or in "other mkisofs
		   options" input field that is not checked for correctness
		   of mkisofs options. This error is probably a result of
		   invalid string in "other mkisofs options" field. */
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("mkisofs reports invalid option. Please check content of \"Other mkisofs options\" field in Configuration -> ISO filesystem tab."));
	}


	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_INODE_NUMBER_OVERFLOW);
		/* I'm not sure what this error means, probably some
		   problems with links in filesystem */
		if (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_CREATE_IMAGE) {
			cdw_task_e((char *) NULL,
				   /* 2TRANS: this is message in dialog window */
				   _("mkisofs reports problems with selected files. Can't write these files. Try creating ISO image and then burn the image to optical disc."));
		} else {
			cdw_task_e((char *) NULL,
				   /* 2TRANS: this is message in dialog window */
				   _("mkisofs reports problems with selected files. Can't write these files."));
		}
		/* 2TRANS: this is title of dialog window */
		cdw_task_e(_("Error - more info"),
			   /* 2TRANS: this is message in dialog window */
			   _("If you are appending data to non-empty optical disc, make sure to use the same mkisofs options as for data already written to the disc."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_NO_SPACE_LEFT);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("No space left on hard disc. Select other hard disc as target for ISO image file."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_ISO_LEVEL_MISMATCH);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("mkisofs reported some problems, probably current \"ISO level\" value set in Configuration -> \"ISO filesystem\" does not match the one used for writing previous session."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME);

		if (task->id == CDW_TASK_BURN_FROM_FILES) {
			cdw_task_e((char *) NULL,
				   /* 2TRANS: this is message in dialog window */
				   _("mkisofs reported some problems when trying to append data to optical disc. Probably you can't append more data to the disc."));
		} else if (task->id == CDW_TASK_CREATE_IMAGE) {
			cdw_task_e((char *) NULL,
				   /* 2TRANS: this is message in dialog window */
				   _("mkisofs reported some problems when trying to create ISO image, probably some files have very similar names or paths."));
		} else {
			cdw_vdm ("ERROR: mkisofs regexp code captured CDW_TOOL_STATUS_MKISOFS_THE_SAME_RR_NAME, but current task \"%lld\" does not involve mksiofs\n",
				 task->id);
		}

	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW);

		if (task->id == CDW_TASK_CREATE_IMAGE) {
			cdw_task_e((char *) NULL,
				   /* 2TRANS: this is message in dialog window */
				   _("mkisofs failed to create ISO image because one of selected files changed size. Check log file for details ('L' hotkey in main window).\n"));
		} else if (task->id == CDW_TASK_BURN_FROM_FILES) {
			cdw_task_e((char *) NULL,
				   /* 2TRANS: this is message in dialog window */
				   _("Process of writing files to optical disc failed because one of selected files changed size. Check log file for details ('L' hotkey in main window).\n"));
		} else {
			cdw_vdm ("ERROR: mkisofs regexp code captured CDW_TOOL_STATUS_MKISOFS_FILE_DID_GROW, but current task \"%lld\" does not involve mksiofs\n",
				 task->id);
		}
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_FATAL_ERROR);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("mkisofs reported fatal error, current task was finished incorrectly. Check log file for details ('L' hotkey in main window).\n"));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_ALLOW_LIMITED_SIZE);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Action failed because some files were larger than 2GiB.\n\nYou may want to retry the action with \"-allow-limited-size  -udf\" option in Configuration -> ISO filesystem -> \"Other mkisofs options\" field, but that may result in data being unreadable on some systems that don't support UDF filesystem.\nAlternatively ensure that \"ISO level\" in Configuration -> ISO filesystem is at least 3.\n\nConsult log file ('L' hotkey in main window) to check which file caused problems."));
	}


	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_LOW_ISO_LEVEL);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Action failed because some files were too large for current settings. Please ensure that \"ISO level\" in Configuration -> ISO filesystem is at least 3.\n\nConsult log file ('L' hotkey in main window) to check which file caused problems."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_SHORT_READ) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_SHORT_READ\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_SHORT_READ);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Action failed, but you may want to open and close drive tray and try again."));
	}

	if (task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_IMPLEMENTATION_BOTCH) {
		cdw_vdm ("ERROR: masked tool_status.mkisofs = 0x%08x / CDW_TOOL_STATUS_MKISOFS_IMPLEMENTATION_BOTCH\n",
			 (unsigned int) task->tool_status.mkisofs & CDW_TOOL_STATUS_MKISOFS_IMPLEMENTATION_BOTCH);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Action failed, you probably need to select \"xorriso\" in Configuration -> Tools to burn to this disc. Check log file for details ('L' hotkey in main window), search for \"Implementation botch\"."));
	}

	/* *** code working with md5sum *** */

	if (task->tool_status.calculate_digest & CDW_TOOL_STATUS_CALCULATE_DIGEST_WRONG_TRACK) {
		cdw_vdm ("ERROR: masked tool_status.calculate_digest = 0x%08x / CDW_TOOL_STATUS_CALCULATE_DIGEST_WRONG_TRACK\n",
			 (unsigned int) task->tool_status.calculate_digest & CDW_TOOL_STATUS_CALCULATE_DIGEST_WRONG_TRACK);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Tried to check md5sum of non-data track or track from multi-track disc. cdw can't do that."));
	}


	/* *** xorriso *** */

	if (task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_SIMULATION_NOT_SUPPORTED) {
		cdw_vdm ("ERROR: masked tool_status.xorriso = 0x%08x / CDW_TOOL_STATUS_XORRISO_SIMULATION_NOT_SUPPORTED\n",
			 (unsigned int) task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_SIMULATION_NOT_SUPPORTED);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("This disc/drive combo does not accept simulation of burning or erasing. Please adjust the options of the task."));
	}

	if (task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_NOT_ENOUGH_SPACE) {
		cdw_vdm ("ERROR: masked tool_status.xorriso = 0x%08x / CDW_TOOL_STATUS_XORRISO_NOT_ENOUGH_SPACE\n",
			 (unsigned int) task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_NOT_ENOUGH_SPACE);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("There is too little space on optical disc to burn selected data."));
	}

	if (task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_CANNOT_READ_ISO_IMAGE_TREE) {
		cdw_vdm ("ERROR: masked tool_status.xorriso = 0x%08x / CDW_TOOL_STATUS_XORRISO_CANNOT_READ_ISO_IMAGE_TREE\n",
			 (unsigned int) task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_CANNOT_READ_ISO_IMAGE_TREE);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("cdw can't read any information from the disc using xorriso. Please adjust your settings in Configuration -> Tools."));
	}

	if (task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_NOT_A_KNOWN_OPTION) {
		cdw_vdm ("ERROR: masked tool_status.xorriso = 0x%08x / CDW_TOOL_STATUS_XORRISO_NOT_A_KNOWN_OPTION\n",
			 (unsigned int) task->tool_status.xorriso & CDW_TOOL_STATUS_XORRISO_NOT_A_KNOWN_OPTION);
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window */
			   _("Invalid option passed to xorriso. Check options in text fields in Configuration window, and under wizard's \"Other options\". See log file ('L' hotkey) for details."));
	}

	return CDW_ERROR;
}





cdw_rv_t cdw_task_select_tools_for_task(cdw_task_t *task, cdw_disc_t *disc)
{
	if (task->id == CDW_TASK_ERASE_DISC
	    || task->id == CDW_TASK_MEDIA_INFO
	    || task->id == CDW_TASK_BURN_FROM_FILES
	    || task->id == CDW_TASK_BURN_FROM_IMAGE) {

		cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc cannot be NULL\n");
	}

	bool success = cdw_task_select_tool(task, disc);
	if (success) {
		cdw_task_debug_print_tools(task);
		return CDW_OK;
	}

	/* error handling */
	cdw_task_reset_tools(task);
	if (cdw_ext_tools_config_is_manual()) {
		cdw_vdm ("ERROR: failed to get tool using manual method\n");
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window;
			      "tool" means "cdrecord", "growisofs", etc.
			      "manual" means "made by user in config window";
			      "automatic" means "not made by user in config window" */
			   _("There was some error with manual selection of a tool for current task. You may need to check your system or use automatic selection of tool (Configuration -> Tools). Please refer to previous error messages and cdw log file for details."));
	} else {
		cdw_vdm ("ERROR: failed to get tool using intelligent method\n");
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message in dialog window;
			      "tool" means "cdrecord", "growisofs", etc.
			      "manual" means "made by user in config window";
			      "automatic" means "not made by user in config window" */
			   _("There was some problem with automatic selection of a tool for current task. You may need to check your system or use manual selection of tool (Configuration -> Tools). Please refer to previous error messages and cdw log file for details."));
	}

	return CDW_ERROR;
}





void cdw_task_e(const char *title, const char *message)
{
	/* 2TRANS: this is title of dialog window */
	const char *local_title = (title == (char *) NULL) ? _("Error") : title;
	cdw_buttons_dialog(local_title, message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	cdw_main_ui_main_window_wrefresh();

	return;
}





cdw_rv_t cdw_task_resolve_allowed_writing_modes(cdw_task_t *task, cdw_disc_t *disc)
{
	int i = 0;
	for (i = 0; i < CDW_SESSION_MODE_N_MAX; i++) {
		task->burn.session_modes[i] = CDW_SESSION_MODE_INIT;
	}
	task->burn.session_modes_n_labels = 0;
	for (i = 0; i < CDW_DISC_MODE_N_MODES; i++) {
		task->burn.disc_modes[i] = CDW_DISC_MODE_INIT;
	}
	task->burn.disc_modes_n_labels = 0;

	cdw_rv_t crv = cdw_task_set_allowed_modes(task, disc);

	if (crv == CDW_OK) {
		for (i = 0; i < CDW_SESSION_MODE_N_MAX; i++) {
			cdw_sdm ("resolved allowed_session_modes[%d] = %d\n", i, task->burn.allowed_session_modes[i]);
		}
		for (i = 0; i < CDW_DISC_MODE_N_MODES; i++) {
			cdw_sdm ("resolved allowed_disc_modes[%d] = %d\n", i, task->burn.allowed_disc_modes[i]);
		}
		return CDW_OK;

	} else if (crv == CDW_NO) {
		return CDW_NO;
	} else {
		cdw_vdm ("ERROR: failed to set allowed modes\n");
		cdw_task_e((char *) NULL,
			   /* 2TRANS: this is message dialog window;
			      "mode" is a type of writing to disc: here
			      it is a generic term that includes
			      both "tao/dao" and "single session/multi session" */
			   _("Cannot guess allowed modes for the disc, can't write to the disc. Try with different disc type."));
		return CDW_ERROR;
	}
}





cdw_rv_t cdw_task_set_allowed_modes(cdw_task_t *task, cdw_disc_t *disc)
{
	/* When there are two or more allowed session modes, and two of
	   them being "create single-session" and "start multi-session",
	   put "create single-session" first, as default option. This
	   decision is a result of discussion with user rogerx on cdw-devel
	   mailing list ~6th of June 2010; */

	cdw_assert (disc->state_writable == CDW_TRUE,
		    "ERROR: disc state should be appendable\n");
	cdw_assert (disc->type_writable == CDW_TRUE,
		    "ERROR: disc type should be writable\n");
	cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD
		    || task->burn.tool.id == CDW_TOOL_GROWISOFS
		    || task->burn.tool.id == CDW_TOOL_XORRISO,
		    "ERROR: unknown burn tool %lld\n", task->burn.tool.id);
	cdw_assert (disc->state_empty == CDW_TRUE
		    || disc->state_empty == CDW_FALSE,
		    "ERROR: disc->state_empty should be either true or false\n");

	/* this is the only place in application where we decide which modes
	   are allowed for given disc type and state, and source of data
	   (files/iso); write wizard uses results of this code, stored in
	   session_mode_ids */

	bool error = false;
	task->burn.session_modes_n_labels = 0;
	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		/* If we are writing image to disc, then checking for
		   target disc (other than DVD+RW and DVD-RW Res) being
		   blank was already made in check_write_preconditions(). */

		if (disc->type == CDW_CD_R
		    || disc->type == CDW_CD_RW) {

			if (disc->state_empty == CDW_TRUE) {
				/* previously it was possible to write iso
				   image file as first session of multi-session
				   disc, but then I found that this may cause
				   some problems with disc read. Until I
				   investigate this closer, I'm disabling
				   starting multisession disc */
				task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
				task->burn.session_modes_n_labels = 1;
			} else {
				task->burn.session_modes_n_labels = 0;
				cdw_vdm ("INFO: can't write image file to non-empty disc\n");
				cdw_task_set_allowed_modes_message(IMAGE_TO_NONEMPTY);
			}

		} else if (disc->type == CDW_DVD_R
			   || disc->type == CDW_DVD_R_SEQ
			   || disc->type == CDW_DVD_R_RES
			   || disc->type == CDW_DVD_RP) {

			if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {

				if (disc->state_empty == CDW_TRUE) {
					task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
					task->burn.session_modes[1] = CDW_SESSION_MODE_START_MULTI;
					task->burn.session_modes_n_labels = 2;
				} else {
					task->burn.session_modes_n_labels = 0;
					cdw_vdm ("INFO: can't write image file to non-empty disc\n");
					cdw_task_set_allowed_modes_message(IMAGE_TO_NONEMPTY);
				}
			} else if (task->burn.tool.id == CDW_TOOL_CDRECORD
				   || task->burn.tool.id == CDW_TOOL_XORRISO) {
				/* cdrecord offers pretty limited support
				   for DVD-R and DVD+R - so we provide only
				   "create single" option for empty discs;

				   cdw can't use all features provided by xorriso, so
				   we provide only "create single" option for empty discs */
				if (disc->state_empty == CDW_FALSE) {
					/* cdrecord Cdrecord-ProDVD-ProBD-Clone 2.01.01a37
					   can correctly recognize non-empty disc,
					   but wodim Wodim 1.1.10 has problems with this */
					task->burn.session_modes_n_labels = 0;
					cdw_vdm ("INFO: can't write image file to non-empty disc\n");
					cdw_task_set_allowed_modes_message(IMAGE_TO_NONEMPTY);
				} else { /* since wodim may have problems with
					    recognizing empty=no, this branch has
					    to cover both empty=yes and empty=unknown */
					/* It seems that the only option available for
					   burning image to DVD-/+R with cdrecord is
					   "create single session" */

					task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
					task->burn.session_modes_n_labels = 1;
				}
			} else {
				cdw_assert (0, "ERROR: unknown tool type %lld\n", task->burn.tool.id);
			}

		} else if (disc->type == CDW_DVD_RW
			   || disc->type == CDW_DVD_RW_SEQ) {

			if (disc->state_empty == CDW_TRUE) {
				task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
				task->burn.session_modes[1] = CDW_SESSION_MODE_START_MULTI;
				task->burn.session_modes_n_labels = 2;
			} else {
				task->burn.session_modes_n_labels = 0;
				cdw_vdm ("INFO: can't write image file to non-empty disc\n");
				cdw_task_set_allowed_modes_message(IMAGE_TO_NONEMPTY);
			}
		} else if (disc->type == CDW_DVD_RP_DL
			   && global_config.support_dvd_rp_dl) {

			if (disc->state_empty == CDW_TRUE) {
				task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
				task->burn.session_modes_n_labels = 1;
			} else {
				task->burn.session_modes_n_labels = 0;
				cdw_vdm ("INFO: can't write image file to non-empty disc\n");
				cdw_task_set_allowed_modes_message(IMAGE_TO_NONEMPTY);
			}

		} else { /* (disc->type == CDW_DVD_RWP
			    || disc->type == CDW_DVD_RW_RES); you can
			    always start new session for these discs, even
			    if there is already data on the disc */

			task->burn.session_modes[0] = CDW_SESSION_MODE_START_MULTI;
			task->burn.session_modes_n_labels = 1;
		}

	} else { /* task->id == CDW_TASK_BURN_FROM_FILES */
		/* if disc is not empty then always put continue_multi
		   before write_final: I guess that user for most of time
		   will want to keep disc appendable */
		if (disc->type == CDW_CD_R
		    || disc->type == CDW_CD_RW) {

			if (disc->state_empty == CDW_TRUE) {
				task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
				task->burn.session_modes[1] = CDW_SESSION_MODE_START_MULTI;
				task->burn.session_modes_n_labels = 2;
			} else { /* disc->state_empty == CDW_FALSE */
				task->burn.session_modes[0] = CDW_SESSION_MODE_CONTINUE_MULTI;
				task->burn.session_modes[1] = CDW_SESSION_MODE_WRITE_FINAL;
				task->burn.session_modes_n_labels = 2;
			}
		} else if (disc->type == CDW_DVD_RW
			   || disc->type == CDW_DVD_RW_SEQ) {

			if (disc->state_empty == CDW_TRUE) {
				task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
				task->burn.session_modes[1] = CDW_SESSION_MODE_START_MULTI;
				task->burn.session_modes_n_labels = 2;
			} else { /* disc->state_empty == CDW_FALSE */
				task->burn.session_modes[0] = CDW_SESSION_MODE_CONTINUE_MULTI;
				task->burn.session_modes[1] = CDW_SESSION_MODE_WRITE_FINAL;
				task->burn.session_modes_n_labels = 2;
			}
		} else if (disc->type == CDW_DVD_RP_DL) {
			/* experimental! */
			if (disc->state_empty == CDW_TRUE) {
				task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
				task->burn.session_modes_n_labels = 1;
			} else { /* disc->state_empty == CDW_FALSE */
				task->burn.session_modes_n_labels = 0;
				cdw_vdm ("INFO: can't write files to non-empty DVD+R DL\n");
				cdw_task_set_allowed_modes_message(NONEMPTY_DVD_RP_DL);
			}

		} else if (disc->type == CDW_DVD_RWP
			   || disc->type == CDW_DVD_RW_RES) {

			if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {

				/* it seems that with DVD+RW and DVD-RW Rest. you can:
				   1. create only one-session disc (to be more precise:
				      DVD+RW / DVD-RW Restricted does not have any
				      notion of multiple sessions, filesystem on disc
				      can be grown to accept more data within
				      boundaries of one, single session)
				   2. write to DVD+RW / DVD-RW Restricted only in
				      appendable mode, -dvd-compat doesn't seem to
				      work neither for image nor for files
				   3. start writing from scratch any time you want
				      with "-Z" option

				   All this may mean that DVD+RW and DVD-RW Res will
				   receive 'special treatment' when creating command
				   for writing.	*/

				if (disc->state_empty == CDW_TRUE) {
					task->burn.session_modes[0] = CDW_SESSION_MODE_START_MULTI;
					task->burn.session_modes_n_labels = 1;
				} else {
					/* "continue multi" must be first; rationale:
					   if there is already some data on disc, it's
					   more probable that user will want to append
					   more data than overwrite the existing data */
					task->burn.session_modes[0] = CDW_SESSION_MODE_CONTINUE_MULTI;
					task->burn.session_modes[1] = CDW_SESSION_MODE_START_MULTI;
					task->burn.session_modes_n_labels = 2;
				}
			} else if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
				/* it is impossible (?) to get correct msinfo
				   for non-empty disc (cdrecord returns "0,0")
				   so there is no possibility to burn
				   second, or following sessions */
				task->burn.session_modes[0] = CDW_SESSION_MODE_START_MULTI;
				task->burn.session_modes_n_labels = 1;
			} else {
				error = true;
				cdw_assert (0, "ERROR: unknown burn tool %lld\n", task->burn.tool.id);
			}
		} else if (disc->type == CDW_DVD_R
			   || disc->type == CDW_DVD_R_SEQ
			   || disc->type == CDW_DVD_R_RES
			   || disc->type == CDW_DVD_RP) {

			if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {

				if (disc->state_empty == CDW_TRUE) {
					task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
					task->burn.session_modes[1] = CDW_SESSION_MODE_START_MULTI;
					task->burn.session_modes_n_labels = 2;
				} else { /* disc->state_empty == CDW_FALSE */
					/* seems that there is no way to close
					   appendable dvd disc other than by filling
					   it with zeros
					   http://osdir.com/ml/audio.cd-record/2004-03/msg00097.html */
					task->burn.session_modes[0] = CDW_SESSION_MODE_CONTINUE_MULTI;
					task->burn.session_modes_n_labels = 1;
				}
			} else if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
				/* again cdrecord and its limited support for DVD-/+R */
				if (disc->state_empty == CDW_TRUE) {
					/* TODO: wodim may not close DVD+R disc even
					   when burning in single session mode, provide
					   a workaround for this */
					task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
					task->burn.session_modes_n_labels = 1;
				} else {
					/* there were some problems with getting cdrecord/wodim to
					   burn to non-empty DVD-R (first session burned with
					   growisofs, non-closed disc), so I guess that it may be
					   impossible to burn second session with cdrecord */
					task->burn.session_modes_n_labels = 0;
					cdw_vdm ("INFO: can't write files to non-empty DVD-/+R with cdrecord\n");
					cdw_task_set_allowed_modes_message(NONEMPTY_DVD_AND_CDRECORD);
				}
			} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
				if (disc->state_empty == CDW_TRUE) {
					task->burn.session_modes[0] = CDW_SESSION_MODE_CREATE_SINGLE;
					task->burn.session_modes[1] = CDW_SESSION_MODE_START_MULTI;
					task->burn.session_modes_n_labels = 2;
				} else { /* disc->state_empty == CDW_FALSE */
					task->burn.session_modes[0] = CDW_SESSION_MODE_CONTINUE_MULTI;
					task->burn.session_modes[1] = CDW_SESSION_MODE_WRITE_FINAL;
					task->burn.session_modes_n_labels = 2;
				}
			} else {
				cdw_assert (0, "ERROR: unknown tool type %lld\n", task->burn.tool.id);
			}

		} else {
			error = true;
			cdw_assert (0, "ERROR: unknown disc type %d\n", disc->type);
		}
	}

	if (task->burn.session_modes_n_labels > 0) {
		cdw_task_set_allowed_disc_modes(task, disc);
	}

	cdw_assert (task->burn.session_modes_n_labels <= CDW_SESSION_MODE_N_MAX,
		    "there are more labels in table (%d) than defined modes (%d)\n",
		    task->burn.session_modes_n_labels, CDW_SESSION_MODE_N_MAX);

        if (error) {
		cdw_vdm ("ERROR: there is no allowed session mode for given pair \"disc type\":\"source of files\"\n");
		return CDW_ERROR;
	} else {
		if (task->burn.session_modes_n_labels == 0) {
			/* it's not an error, it's just that for these
			   conditions there is no valid solution; */
			cdw_vdm ("INFO: there is no allowed session mode for given conditions\n");
			return CDW_NO;
		} else {
			return CDW_OK;
		}
	}
}





void cdw_task_set_allowed_disc_modes(cdw_task_t *task, cdw_disc_t *disc)
{
	if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		cdw_task_set_allowed_disc_modes_growisofs(task);
	} else if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		cdw_task_set_allowed_disc_modes_cdrecord(task, disc);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		cdw_task_set_allowed_disc_modes_xorriso(task, disc);
	} else {
		;
	}
	return;
}





void cdw_task_set_allowed_disc_modes_growisofs(cdw_task_t *task)
{
	int i = 0;
	task->burn.disc_modes[i++] = CDW_DISC_MODE_UNSPECIFIED;

	for (int j = 0; j < task->burn.session_modes_n_labels; j++) {
		if (task->burn.session_modes[j] == CDW_SESSION_MODE_CREATE_SINGLE) {
			task->burn.disc_modes[i++] = CDW_DISC_MODE_DAO;
			break;
		}
	}
	task->burn.disc_modes_n_labels = i;
	return;
}





void cdw_task_set_allowed_disc_modes_cdrecord(cdw_task_t *task, cdw_disc_t *disc)
{
	int i = 0;
	task->burn.disc_modes[i++] = CDW_DISC_MODE_UNSPECIFIED;

	/* first try dao/sao, it's more probable that people
	   burn data in "create single" mode */
	if (disc->accepted_disc_modes & CDW_DISC_MODE_DAO) {
		task->burn.disc_modes[i++] = CDW_DISC_MODE_DAO;
	}
	if (disc->accepted_disc_modes & CDW_DISC_MODE_SAO) {
		task->burn.disc_modes[i++] = CDW_DISC_MODE_SAO;
	}

	/* then try also TAO, as this may also be a valid option */
	if (disc->accepted_disc_modes & CDW_DISC_MODE_TAO) {
		task->burn.disc_modes[i++] = CDW_DISC_MODE_TAO;
	}

	task->burn.disc_modes_n_labels = i;
	return;
}





void cdw_task_set_allowed_disc_modes_xorriso(cdw_task_t *task, cdw_disc_t *disc)
{
	int i = 0;
	task->burn.disc_modes[i++] = CDW_DISC_MODE_UNSPECIFIED;

	/* first try dao/sao, it's more probable that people
	   burn data in "create single" mode */
	if (disc->accepted_disc_modes & CDW_DISC_MODE_DAO) {
		task->burn.disc_modes[i++] = CDW_DISC_MODE_DAO;
	}
	if (disc->accepted_disc_modes & CDW_DISC_MODE_SAO) {
		task->burn.disc_modes[i++] = CDW_DISC_MODE_SAO;
	}

	/* then try also TAO, as this may also be a valid option */
	if (disc->accepted_disc_modes & CDW_DISC_MODE_TAO) {
		task->burn.disc_modes[i++] = CDW_DISC_MODE_TAO;
	}

	task->burn.disc_modes_n_labels = i;
	return;
}



void cdw_task_set_allowed_modes_message(cdw_id_t id)
{
	/* 2TRANS: this is title of dialog window */
	const char *title = _("Error");
	char *message = (char *) NULL;

	if (id == IMAGE_TO_NONEMPTY) {
		/* 2TRANS: this is message in dialog window */
		message = _("cdw can't write image file to non-empty disc.");
	} else if (id == NONEMPTY_DVD_RP_DL) {
		/* 2TRANS: this is message in dialog window */
		message = _("cdw can't write to non-empty DVD+R DL disc.");
	} else if (id == NONEMPTY_DVD_AND_CDRECORD) {
		/* 2TRANS: this is message in dialog window */
		message = _("cdw can't write to non-empty DVD-R or DVD+R disc with current tool. Please adjust your settings in Configuration -> Tools.");
	} else {
		cdw_assert (0, "ERROR: unknown message id = %lld\n", id);
		return;
	}

	cdw_buttons_dialog(title, message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	cdw_main_ui_main_window_wrefresh();

	return;
}





bool cdw_task_select_tool(cdw_task_t *task, cdw_disc_t *disc)
{
	bool success = false;
	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		success = cdw_ext_tools_set_burn_tool(disc, &(task->burn.tool), &(task->create_image.tool));
		if (!success) {
			return false;
		}
		success = cdw_ext_tools_set_digest_tool(&(task->calculate_digest.tool));
		if (!success) {
			cdw_vdm ("WARNING: task is \"burn form image\", but no digest tool is available\n");
		}
		return true;

	} else if (task->id == CDW_TASK_BURN_FROM_FILES) {
		if (disc->state_empty != CDW_TRUE
		    && disc->cdio->ofs->cdw_tool == CDW_TOOL_NONE
		    && task->burn.session_mode == CDW_SESSION_MODE_CONTINUE_MULTI) {

			/* file system present on disc has been created using
			   tool other than xorriso or mkisofs; this can lead
			   to problems with adding new data */

			/* 2TRANS: this is a dialog window title */
			cdw_buttons_dialog(_("Warning"),
					   /* 2TRANS: this is a message in dialog window */
					   _("cdw can't recognize tool previously used to burn to this disc. You may experience serious trouble (including data loss) if you decide to continue."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		}
		success = cdw_ext_tools_set_iso9660_otf_tool(disc, &(task->create_image.tool));
		if (!success) {
			cdw_vdm ("ERROR: failed to get ISO9660 otf tool\n");
			return false;
		}
		success = cdw_ext_tools_set_burn_tool(disc, &(task->burn.tool), &(task->create_image.tool));
		if (!success) {
			cdw_vdm ("ERROR: failed to get burn tool (ISO9660 otf tool = \"%s\")\n", cdw_ext_tools_get_tool_name(task->create_image.tool.id));
			return false;
		}
		return true;
#if 0
	} else if (task->id == CDW_TASK_READ_DISC) {
		return true;
#endif
	} else if (task->id == CDW_TASK_ERASE_DISC) {
		success = cdw_ext_tools_set_erase_tool(disc, &(task->erase.tool));
		if (!success) {
			return false;
		}
		return true;

	} else if (task->id == CDW_TASK_CREATE_IMAGE) {
		success = cdw_ext_tools_set_iso9660_sa_tool(&(task->create_image.tool));
		if (!success) {
			return false;
		}
		return true;

	} else if (task->id == CDW_TASK_MEDIA_INFO) {
		success = cdw_ext_tools_set_media_info_tool(disc, &(task->media_info.tool));
		if (!success) {
			return false;
		}
		return true;

	} else if (task->id == CDW_TASK_CALCULATE_DIGEST) {
		success = cdw_ext_tools_set_digest_tool(&(task->calculate_digest.tool));
		if (!success) {
			return false;
		}
		return true;
	} else {
		cdw_vdm ("ERROR: invalid or unknown task id %lld\n", task->id);
		return false;
	}
}





static cdw_id_clabel_t cdw_session_modes[] = {
	{ CDW_SESSION_MODE_INIT,             "CDW_SESSION_MODE_INIT" },
	{ CDW_SESSION_MODE_START_MULTI,      "CDW_SESSION_MODE_START_MULTI" },
	{ CDW_SESSION_MODE_CREATE_SINGLE,    "CDW_SESSION_MODE_CREATE_SINGLE" },
	{ CDW_SESSION_MODE_CONTINUE_MULTI,   "CDW_SESSION_MODE_CONTINUE_MULTI" },
	{ CDW_SESSION_MODE_WRITE_FINAL,      "CDW_SESSION_MODE_WRITE_FINAL" },
	{ CDW_SESSION_MODE_ERROR,            "CDW_SESSION_MODE_ERROR" },
	/* { CDW_SESSION_MODE_N_MAX,            "CDW_SESSION_MODE_N_MAX" }, */
	{ 0,                                 (char *) NULL }};



const char *cdw_task_get_session_mode_label(cdw_id_t mode)
{
	return cdw_utils_id_label_table_get_label(cdw_session_modes, mode);
}



static cdw_id_clabel_t cdw_disc_modes[] = {
	{ CDW_DISC_MODE_INIT,            "CDW_DISC_MODE_INIT" },
	{ CDW_DISC_MODE_UNSPECIFIED,     "CDW_DISC_MODE_UNSPECIFIED" },
	{ CDW_DISC_MODE_TAO,             "CDW_DISC_MODE_TAO" },
	{ CDW_DISC_MODE_DAO,             "CDW_DISC_MODE_DAO" },
	{ CDW_DISC_MODE_SAO,             "CDW_DISC_MODE_SAO" },
	/* { CDW_DISC_MODE_RAW,             "CDW_DISC_MODE_RAW" },
	{ CDW_DISC_MODE_RAW96P,          "CDW_DISC_MODE_RAW96P" },
	{ CDW_DISC_MODE_RAW96R,          "CDW_DISC_MODE_RAW96R" },
	{ CDW_DISC_MODE_RAW16,           "CDW_DISC_MODE_RAW16" }, */
	{ CDW_DISC_MODE_ERROR,           "CDW_DISC_MODE_ERROR" },
	{ CDW_DISC_MODE_N_MODES,           "CDW_DISC_MODE_N_MAX" },
	{ 0,                             (char *) NULL }};



const char *cdw_task_get_disc_mode_label(cdw_id_t mode)
{
	return cdw_utils_id_label_table_get_label(cdw_disc_modes, mode);
}


static cdw_id_clabel_t cdw_task_labels[] = {
	{ CDW_TASK_NONE,                 "CDW_TASK_NONE" },
	{ CDW_TASK_BURN_FROM_IMAGE,      "CDW_TASK_BURN_FROM_IMAGE" },
	{ CDW_TASK_BURN_FROM_FILES,      "CDW_TASK_BURN_FROM_FILES" },
	{ CDW_TASK_READ_DISC,            "CDW_TASK_READ_DISC" },
	{ CDW_TASK_ERASE_DISC,           "CDW_TASK_ERASE_DISC" },
	{ CDW_TASK_CLONE_DISC,           "CDW_TASK_CLONE_DISC" },
	{ CDW_TASK_CREATE_IMAGE,         "CDW_TASK_CREATE_IMAGE" },
	{ CDW_TASK_CALCULATE_DIGEST,     "CDW_TASK_CALCULATE_DIGEST" },
	{ CDW_TASK_MEDIA_INFO,           "CDW_TASK_MEDIA_INFO" },
	{ 0,                             (char *) NULL }};


const char *cdw_task_get_task_label(cdw_id_t task_id)
{
	return cdw_utils_id_label_table_get_label(cdw_task_labels, task_id);
}


void cdw_task_debug_print_tools(cdw_task_t *task)
{
	cdw_assert (task->id != CDW_TASK_NONE, "ERROR: selecting tool: task id == NONE\n");
	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		cdw_vdm ("INFO: selecting tool: %s, burning tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->burn.tool.label);
		cdw_vdm ("INFO: selecting tool: %s, md5sum tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->calculate_digest.tool.label);
	} else if (task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_vdm ("INFO: selecting tool: %s, burning tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->burn.tool.label);
		cdw_vdm ("INFO: selecting tool: %s, iso fs tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->create_image.tool.label);
	} else if (task->id == CDW_TASK_READ_DISC) {
		cdw_vdm ("INFO: selecting tool: %s, no info about read tool\n", cdw_task_get_task_label(task->id));
	} else if (task->id == CDW_TASK_ERASE_DISC) {
		cdw_vdm ("INFO: selecting tool: %s, erase tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->erase.tool.label);
	} else if (task->id == CDW_TASK_CREATE_IMAGE) {
		cdw_vdm ("INFO: selecting tool: %s, iso fs tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->create_image.tool.label);
	} else if (task->id == CDW_TASK_CALCULATE_DIGEST) {
		cdw_vdm ("INFO: selecting tool: %s, digest tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->calculate_digest.tool.label);
	} else if (task->id == CDW_TASK_MEDIA_INFO) {
		cdw_vdm ("INFO: selecting tool: %s, mediainfo tool is \"%s\"\n", cdw_task_get_task_label(task->id), task->media_info.tool.label);
	} else {
		/* it may be "clone" task, but it is not supported for now */
		cdw_assert (0, "ERROR: unknown task id %lld\n", task->id);
	}
	cdw_vdm ("INFO: selecting tool: ----------------------------------\n");

	return;
}





cdw_rv_t cdw_task_get_initial_options(cdw_task_t *task, cdw_disc_t *disc)
{
	if (task->id == CDW_TASK_CREATE_IMAGE
	    || task->id == CDW_TASK_BURN_FROM_FILES) {

		task->create_image.joliet_information = global_config.joliet;
		task->create_image.follow_symlinks = global_config.follow_symlinks;
		task->create_image.rock_ridge = global_config.rock_ridge;
		task->create_image.pad = true;

		strncpy(task->create_image.volume_id, global_config.volume_id, CDW_ISO9660_VOLI_LEN);
		task->create_image.volume_id[CDW_ISO9660_VOLI_LEN] = '\0';
		strncpy(task->create_image.volume_set_id, global_config.volume_set_id, CDW_ISO9660_VOLS_LEN);
		task->create_image.volume_set_id[CDW_ISO9660_VOLS_LEN] = '\0';
		strncpy(task->create_image.preparer, global_config.preparer, CDW_ISO9660_PREP_LEN);
		task->create_image.preparer[CDW_ISO9660_PREP_LEN] = '\0';
		strncpy(task->create_image.publisher, global_config.publisher, CDW_ISO9660_PUBL_LEN);
		task->create_image.publisher[CDW_ISO9660_PUBL_LEN] = '\0';
		strncpy(task->create_image.system_id, global_config.system_id, CDW_ISO9660_SYSI_LEN);
		task->create_image.system_id[CDW_ISO9660_SYSI_LEN] = '\0';
		strncpy(task->create_image.copyright, global_config.copyright, CDW_ISO9660_COPY_LEN);
		task->create_image.copyright[CDW_ISO9660_COPY_LEN] = '\0';
		strncpy(task->create_image.abstract, global_config.abstract, CDW_ISO9660_ABST_LEN);
		task->create_image.abstract[CDW_ISO9660_ABST_LEN] = '\0';

		if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
			task->create_image.mkisofs.joliet_long = global_config.joliet_long;
			task->create_image.mkisofs.iso_level = global_config.iso_level;
			cdw_string_set(&(task->create_image.mkisofs.root_dir), global_config.mkisofs_root_dir);
			cdw_string_set(&(task->create_image.mkisofs.other_mkisofs_options), global_config.other_mkisofs_options);
		} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
			cdw_string_set(&(task->create_image.xorriso.other_xorriso_iso_options), global_config.other_xorriso_iso_options);
		} else {
			cdw_assert(0, "ERROR: unknown tool %lld for task \"create image\"\n", task->create_image.tool.id);
		}
	}

	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		cdw_string_set(&(task->burn.iso9660_file_fullpath), global_config.iso_image_full_path);
	} else if (task->id == CDW_TASK_CREATE_IMAGE) {
		cdw_string_set(&(task->create_image.iso9660_file_fullpath), global_config.iso_image_full_path);
	} else {
		;
	}

	if (task->id == CDW_TASK_BURN_FROM_FILES
	    || task->id == CDW_TASK_BURN_FROM_IMAGE) {

		task->burn.eject = global_config.eject;
		task->burn.verify = true;
		task->burn.dummy = false;
		task->burn.speed_id = cdw_disc_get_initial_write_speed_id(disc);

		if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
			cdw_string_set(&(task->burn.growisofs.other_growisofs_options), global_config.other_growisofs_options);
		} else if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
			task->burn.cdrecord.burnproof = global_config.burnproof;
			task->burn.cdrecord.pad = global_config.write_pad;
			task->burn.cdrecord.pad_size = global_config.write_pad_size;
			cdw_string_set(&(task->burn.cdrecord.other_cdrecord_options), global_config.other_cdrecord_options);
		} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
			cdw_string_set(&(task->burn.xorriso.other_xorriso_burn_options), global_config.other_xorriso_burn_options);
		} else {
			;
		}
	}

	if (task->id == CDW_TASK_ERASE_DISC) {
		task->erase.eject = global_config.eject;
		task->erase.erase_mode = global_config.erase_mode;
		task->erase.format_mode = CDW_ERASE_DVD_RW_FORMAT_NONE;
		task->erase.speed_id = cdw_disc_get_initial_write_speed_id(disc);
	} else {
		;
	}


	return CDW_OK;
}





cdw_task_t *cdw_task_new(cdw_id_t id, cdw_disc_t *disc)
{
	cdw_task_t *task = (cdw_task_t *) malloc(sizeof (cdw_task_t));
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: failed to malloc memory for task\n");
		return (cdw_task_t *) NULL;
	}

	cdw_task_reset_state(task);
	task->id = id;

	cdw_rv_t crv = cdw_task_select_tools_for_task(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to select tools for task\n");
		cdw_task_delete(&task);
		return (cdw_task_t *) NULL;
	}

	crv = cdw_task_get_initial_options(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get options for task\n");
		cdw_task_delete(&task);
		return (cdw_task_t *) NULL;
	}

	if (task->id == CDW_TASK_BURN_FROM_FILES
	    || task->id == CDW_TASK_BURN_FROM_IMAGE) {

		/* check if current state of disc allows us to write what we want
		   (iso file or files from disc), and specify allowed modes of writing
		   (start multisession, append and close, etc.) */
		crv = cdw_task_resolve_allowed_writing_modes(task, disc);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to get options for task\n");
			cdw_task_delete(&task);
			return (cdw_task_t *) NULL;
		}
	}

	if (disc) {
		cdw_assert (!task->disc, "ERROR: disc is already set\n");
		task->disc = disc;
	}

	return task;
}





void cdw_task_delete(cdw_task_t **task)
{
	cdw_assert (task, "ERROR: passing pointer to NULL task to the function\n");

	if (*task == (cdw_task_t *) NULL) {
		cdw_vdm ("WARNING passing NULL pointer to task\n");
		return;
	}

	if ((*task)->id == CDW_TASK_BURN_FROM_FILES
	    || (*task)->id == CDW_TASK_BURN_FROM_IMAGE) {

		if ((*task)->burn.tool.id == CDW_TOOL_CDRECORD) {

		} else if ((*task)->burn.tool.id == CDW_TOOL_GROWISOFS) {
			if ((*task)->burn.growisofs.other_growisofs_options != (char *) NULL) {
				free((*task)->burn.growisofs.other_growisofs_options);
				(*task)->burn.growisofs.other_growisofs_options = (char *) NULL;
			}
		} else {
			;
		}
		if ((*task)->id == CDW_TASK_BURN_FROM_IMAGE) {
			if ((*task)->burn.iso9660_file_fullpath != (char *) NULL) {
				free((*task)->burn.iso9660_file_fullpath);
				(*task)->burn.iso9660_file_fullpath = (char *) NULL;
			}
		}
	} else if ((*task)->id == CDW_TASK_CREATE_IMAGE) {
		if ((*task)->create_image.tool.id == CDW_TOOL_MKISOFS) {
			if ((*task)->create_image.mkisofs.root_dir != (char *) NULL) {
				free((*task)->create_image.mkisofs.root_dir);
				(*task)->create_image.mkisofs.root_dir = (char *) NULL;
			}
			if ((*task)->create_image.mkisofs.other_mkisofs_options != (char *) NULL) {
				free((*task)->create_image.mkisofs.other_mkisofs_options);
				(*task)->create_image.mkisofs.other_mkisofs_options = (char *) NULL;
			}
		} else if ((*task)->create_image.tool.id == CDW_TOOL_XORRISO) {
			if ((*task)->create_image.xorriso.other_xorriso_iso_options != (char *) NULL) {
				free((*task)->create_image.xorriso.other_xorriso_iso_options);
				(*task)->create_image.xorriso.other_xorriso_iso_options = (char *) NULL;
			}
		} else {
			;
		}
		if ((*task)->create_image.iso9660_file_fullpath != (char *) NULL) {
			free((*task)->create_image.iso9660_file_fullpath);
			(*task)->create_image.iso9660_file_fullpath = (char *) NULL;
		}
	} else {
		;
	}

	free(*task);
	*task = (cdw_task_t *) NULL;

	return;
}





void cdw_task_save(cdw_task_t *task)
{
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: task is NULL\n");
		return;
	}



	if (task->id == CDW_TASK_ERASE_DISC) {
		global_config.eject = task->erase.eject;
		global_config.erase_mode = task->erase.erase_mode;

	} else if (task->id == CDW_TASK_BURN_FROM_FILES
		   || task->id == CDW_TASK_BURN_FROM_IMAGE) {
		global_config.eject = task->burn.eject;
	} else {
		;
	}

	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		cdw_string_set(&(global_config.iso_image_full_path), task->burn.iso9660_file_fullpath);
	} else if (task->id == CDW_TASK_CREATE_IMAGE) {
		cdw_string_set(&(global_config.iso_image_full_path), task->create_image.iso9660_file_fullpath);
	} else {
		;
	}

	if (task->id == CDW_TASK_CREATE_IMAGE
	    || task->id == CDW_TASK_BURN_FROM_FILES) {


		global_config.follow_symlinks = task->create_image.follow_symlinks;

		strncpy(global_config.volume_id, task->create_image.volume_id, CDW_ISO9660_VOLI_LEN);
		global_config.volume_id[CDW_ISO9660_VOLI_LEN] = '\0';
		strncpy(global_config.volume_set_id, task->create_image.volume_set_id, CDW_ISO9660_VOLS_LEN);
		global_config.volume_set_id[CDW_ISO9660_VOLS_LEN] = '\0';
		strncpy(global_config.preparer, task->create_image.preparer, CDW_ISO9660_PREP_LEN);
		global_config.preparer[CDW_ISO9660_PREP_LEN] = '\0';
		strncpy(global_config.publisher, task->create_image.publisher, CDW_ISO9660_PUBL_LEN);
		global_config.publisher[CDW_ISO9660_PUBL_LEN] = '\0';
		strncpy(global_config.system_id, task->create_image.system_id, CDW_ISO9660_SYSI_LEN);
		global_config.system_id[CDW_ISO9660_SYSI_LEN] = '\0';
		strncpy(global_config.copyright, task->create_image.copyright, CDW_ISO9660_COPY_LEN);
		global_config.copyright[CDW_ISO9660_COPY_LEN] = '\0';
		strncpy(global_config.abstract, task->create_image.abstract, CDW_ISO9660_ABST_LEN);
		global_config.abstract[CDW_ISO9660_ABST_LEN] = '\0';

		global_config.rock_ridge = task->create_image.rock_ridge;

		if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
			global_config.joliet_long = task->create_image.mkisofs.joliet_long;
			global_config.iso_level = task->create_image.mkisofs.iso_level;
			cdw_string_set(&(global_config.mkisofs_root_dir), task->create_image.mkisofs.root_dir);

			cdw_string_set(&(global_config.other_mkisofs_options), task->create_image.mkisofs.other_mkisofs_options);
		} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
			cdw_string_set(&(global_config.other_xorriso_iso_options), task->create_image.xorriso.other_xorriso_iso_options);
		} else {
			;
		}
	}

	if (task->id == CDW_TASK_BURN_FROM_FILES
	    || task->id == CDW_TASK_BURN_FROM_IMAGE) {

		if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
			global_config.write_pad = task->burn.cdrecord.pad;
			global_config.write_pad_size = task->burn.cdrecord.pad_size;
			global_config.burnproof = task->burn.cdrecord.burnproof;
			cdw_string_set(&(global_config.other_cdrecord_options), task->burn.cdrecord.other_cdrecord_options);
		} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
			cdw_string_set(&(global_config.other_growisofs_options), task->burn.growisofs.other_growisofs_options);
		} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
			cdw_string_set(&(global_config.other_xorriso_burn_options), task->burn.xorriso.other_xorriso_burn_options);
		} else {
			;
		}

	}

	return;
}





void cdw_task_debug_print_burn_options(cdw_task_t *task)
{
	cdw_vdm ("INFO:            session mode = \"%s\"\n", cdw_task_get_session_mode_label(task->burn.session_mode));
	cdw_vdm ("INFO:               disc mode = \"%s\"\n", cdw_task_get_disc_mode_label(task->burn.disc_mode));
	cdw_vdm ("INFO:                   speed = \"%lld\"\n", task->burn.speed_id);
	cdw_vdm ("INFO:          verify burning = \"%s\"\n", task->burn.verify ? "true" : "false");
	cdw_vdm ("INFO:              dummy burn = \"%s\"\n", task->burn.dummy ? "true" : "false");
	cdw_vdm ("INFO:       eject after write = \"%s\"\n", task->burn.eject ? "true" : "false");

	if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		cdw_vdm ("INFO: other growisofs options = \"%s\"\n", task->burn.growisofs.other_growisofs_options);
	} else if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		cdw_vdm ("INFO:               burnproof = \"%s\"\n", task->burn.cdrecord.burnproof ? "true" : "false");
		cdw_vdm ("INFO:                     pad = \"%s\"\n", task->burn.cdrecord.pad ? "true" : "false");
		cdw_vdm ("INFO:                pad size = \"%d\"\n", task->burn.cdrecord.pad_size);
		cdw_vdm ("INFO:  other cdrecord options = \"%s\"\n", task->burn.cdrecord.other_cdrecord_options);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		cdw_vdm ("INFO:   other xorriso options = \"%s\"\n", task->burn.xorriso.other_xorriso_burn_options);
	} else {
		cdw_vdm ("ERROR: unknown burn tool %lld\n", task->burn.tool.id);
	}

	if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		cdw_vdm ("INFO:             ISO9660 file = \"%s\"\n", task->burn.iso9660_file_fullpath);
	}

	return;
}





void cdw_task_debug_print_iso9660_options(cdw_task_t *task)
{
	if (task->id == CDW_TASK_BURN_FROM_FILES
	    || task->id == CDW_TASK_CREATE_IMAGE) {
		if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
			cdw_vdm ("INFO:               joliet long = \"%s\"\n", task->create_image.mkisofs.joliet_long ? "true" : "false");
			cdw_vdm ("INFO:                 iso level = \"%lld\"\n", task->create_image.mkisofs.iso_level);
			cdw_vdm ("INFO:                  root dir = \"%s\"\n", task->create_image.mkisofs.root_dir);
			cdw_vdm ("INFO:     other mkisofs options = \"%s\"\n", task->create_image.mkisofs.other_mkisofs_options);
		} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
			cdw_vdm ("INFO: other xorriso ISO options = \"%s\"\n", task->create_image.xorriso.other_xorriso_iso_options);
		} else {
			cdw_vdm ("ERROR: unknown ISO9660 tool %lld\n", task->create_image.tool.id);
		}

		cdw_vdm ("INFO:                    joliet = \"%s\"\n", task->create_image.joliet_information ? "true" : "false");
		cdw_vdm ("INFO:           follow symlinks = \"%s\"\n", task->create_image.follow_symlinks ? "true" : "false");
		cdw_vdm ("INFO:                       pad = \"%s\"\n", task->create_image.pad ? "true" : "false");
		cdw_vdm ("INFO:                rock ridge = %lld\n", task->create_image.rock_ridge);

		cdw_vdm ("INFO:                 volume ID = \"%s\"\n", task->create_image.volume_id);
		cdw_vdm ("INFO:             volume set ID = \"%s\"\n", task->create_image.volume_set_id);
		cdw_vdm ("INFO:                  preparer = \"%s\"\n", task->create_image.preparer);
		cdw_vdm ("INFO:                 publisher = \"%s\"\n", task->create_image.publisher);
		cdw_vdm ("INFO:                 system id = \"%s\"\n", task->create_image.system_id);
		cdw_vdm ("INFO:                 copyright = \"%s\"\n", task->create_image.copyright);
		cdw_vdm ("INFO:                  abstract = \"%s\"\n", task->create_image.abstract);

		if (task->id == CDW_TASK_CREATE_IMAGE) {
			cdw_vdm ("INFO:              ISO9660 file = \"%s\"\n", task->create_image.iso9660_file_fullpath);
		}
	} else if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		;
	} else {
		cdw_vdm ("ERROR: incorrect task id %lld\n", task->id);
	}

	return;
}




/* unused code below */

#if 0


cdw_id_t cdw_task_select_tool_family(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_id_t tool_family = CDW_TOOL_NONE;
	if (task->id == CDW_TASK_MEDIA_INFO
	    || task->id == CDW_TASK_BURN_FROM_IMAGE
	    || task->id == CDW_TASK_BURN_FROM_FILES
	    || task->id == CDW_TASK_ERASE_DISC) {

		/* note on cdw_ext_tools_config_get_X_family(): if manual
		   selection is in effect, these two functions simply take
		   value selected in "tool family" menu in configuration panel;
		   if "intelligent" selection is in effect, returned values
		   are selected with some intelligent method */
		if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			tool_family = cdw_ext_tools_config_get_cd_handler_id();
			if (tool_family == CDW_TOOL_NONE) {
				cdw_task_e((char *) NULL,
					   /* 2TRANS: this is message in dialog window;
					      "tools" means "cdrecord", "growisofs" etc. */
					   _("cdw can't find in your system any tools for handling CD discs. Current action can't be performed."));
			}
			cdw_assert (tool_family == CDW_TOOL_CDRECORD, "ERROR: tool family for CD is not cdrecord\n");

		} else if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
			tool_family = cdw_ext_tools_config_get_dvd_handler_id();
			if (tool_family == CDW_TOOL_NONE) {
				cdw_task_e((char *) NULL,
					   /* 2TRANS: this is message in dialog window;
					      "tools" means "cdrecord", "growisofs" etc. */
					   _("cdw can't find in your system any tools for handling DVD discs. Current action can't be performed."));
			}
			cdw_assert (tool_family == CDW_TOOL_CDRECORD || tool_family == CDW_TOOL_GROWISOFS,
				    "ERROR: tool family for DVD is neither cdrecord nor growisofs\n");
		} else {
			cdw_assert (0, "ERROR: incorrect disc simple type %d\n", disc->simple_type);
		}
	} else if (task->id == CDW_TASK_CREATE_IMAGE) {
		tool_family = cdw_ext_tools_config_get_iso9660_id();
		if (tool_family == CDW_TOOL_NONE) {
			cdw_task_e((char *) NULL,
				   /* 2TRANS: this is message in dialog window;
				      "tools" means "mkisofs", "xorriso" etc. */
				   _("cdw can't find in your system any tools for creating ISO9660 files. Current action can't be performed."));
		}
		cdw_assert (tool_family == CDW_TOOL_MKISOFS || tool_family == CDW_TOOL_XORRISO,
			    "ERROR: too family for creating ISO9660 is invalid: %ld\n", tool_family);
	} else {
		;
	}
	return tool_family;
}





bool cdw_task_select_tool_sub(cdw_task_t *task, cdw_id_t tool_family)
{
	bool success = true;
	if (task->id == CDW_TASK_MEDIA_INFO) {
		if (tool_family == CDW_TOOL_GROWISOFS) {
			task->media_info.tool.id = CDW_TOOL_DVD_RW_MEDIAINFO;
		} else if (tool_family == CDW_TOOL_CDRECORD) {
			task->media_info.tool.id = CDW_TOOL_CDRECORD;
		/* } else if (tool_family == CDW_TOOL_XORRISO) {
		        task->media_info.tool.id = CDW_TOOL_XORRISO; */
		} else {
			success = false;
			cdw_assert (0, "ERROR: incorrect tool family %ld\n", tool_family);
		}
		if (success) {
			task->media_info.tool.fullpath = cdw_ext_tools_get_tool_fullpath(task->media_info.tool.id);
		}

	} else if (task->id == CDW_TASK_ERASE_DISC) {
		if (tool_family == CDW_TOOL_GROWISOFS) {
			task->erase.tool.id = CDW_TOOL_DVD_RW_FORMAT;
		} else if (tool_family == CDW_TOOL_CDRECORD) {
			task->erase.tool.id = CDW_TOOL_CDRECORD;
		/* } else if (tool_family == CDW_TOOL_XORRISO) {
		        task->erase.tool.id = CDW_TOOL_XORRISO; */
		} else {
			success = false;
			cdw_assert (0, "ERROR: incorrect tool family %ld\n", tool_family);
		}
		if (success) {
			task->erase.tool.fullpath = cdw_ext_tools_get_tool_fullpath(task->erase.tool.id);
		}
	} else if (task->id == CDW_TASK_BURN_FROM_FILES) {
		   if (tool_family == CDW_TOOL_GROWISOFS
		       || tool_family == CDW_TOOL_CDRECORD) {

			   if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)) {
				   task->create_image.tool.id = CDW_TOOL_MKISOFS;
				   task->create_image.tool.fullpath = cdw_ext_tools_get_tool_fullpath(CDW_TOOL_MKISOFS);

				   if (tool_family == CDW_TOOL_GROWISOFS) {
					   task->burn.tool.id = CDW_TOOL_GROWISOFS;
				   } else {
					   task->burn.tool.id = CDW_TOOL_CDRECORD;
				   }
			   } else {
				   success = false;
			   }
		   /* } else if (tool_family == CDW_TOOL_XORRISO) {
		              task->burn.tool.id = CDW_TOOL_XORRISO; */
		   } else {
			   success = false;
			   cdw_assert (0, "ERROR: incorrect tool family %ld\n", tool_family);
		   }
		   if (success) {
			   task->burn.tool.fullpath = cdw_ext_tools_get_tool_fullpath(task->burn.tool.id);
		   }
	} else if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		if (tool_family == CDW_TOOL_GROWISOFS) {
			task->burn.tool.id = CDW_TOOL_GROWISOFS;
		} else if (tool_family == CDW_TOOL_CDRECORD) {
			task->burn.tool.id = CDW_TOOL_CDRECORD;
		/* } else if (tool_family == CDW_TOOL_XORRISO) {
		        task->burn.tool.id = CDW_TOOL_XORRISO;*/
		} else {
			success = false;
			cdw_assert (0, "ERROR: incorrect tool family %ld\n", tool_family);
		}
		if (success) {
			task->burn.tool.fullpath = cdw_ext_tools_get_tool_fullpath(task->burn.tool.id);
		}

	} else if (task->id == CDW_TASK_CREATE_IMAGE) {
		if (tool_family == CDW_TOOL_MKISOFS) {
			task->create_image.tool.id = CDW_TOOL_MKISOFS;
		} else if (tool_family == CDW_TOOL_XORRISO) {
			task->create_image.tool.id = CDW_TOOL_XORRISO;
		} else {
			success = false;
			cdw_assert (0, "ERROR: incorrect tool family %ld\n", tool_family);
		}
		if (success) {
			task->create_image.tool.fullpath = cdw_ext_tools_get_tool_fullpath(task->create_image.tool.id);
		}
	} else if (task->id == CDW_TASK_CHECK_MD5SUM) {
		if (cdw_ext_tools_is_tool_available(CDW_TOOL_MD5SUM)) {
			task->calculate_digest.tool.id = CDW_TOOL_MD5SUM;
			task->calculate_digest.tool.fullpath = cdw_ext_tools_get_tool_fullpath(CDW_TOOL_MD5SUM);
		} else {
			success = false;
		}
	} else {
		cdw_assert (0, "ERROR: incorrect task id = %d\n", task->id);
	}

	return success;
}


#endif

