/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */



/**
   \file cdw_ofs.c

   Module containing some functions related to ofs - optical file system.
   The module makes use of cdio library calls to collect data about ofs.
   ofs data structure can be a part of cdio disc, providing information
   about file system on optical disc.
*/



#include <stdlib.h>
#include <errno.h>
#include <string.h>

/* open() */
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <cdio/iso9660.h>

#include <fcntl.h>

#include "cdw_ofs.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_string.h"


/* Look-up table for labels/names of optical file systems */
struct {
	cdio_fs_t   cdio_fs;               /* a key; you can use it to index the table */
	const char *cdio_fs_label;         /* a string looking the same as the key; may be used in debug messages */
	const char *cdio_fs_short_label;   /* the label; use this if you have limited space */
	const char *cdio_fs_long_label;    /* the label; use this if you have plenty of space */
} cdw_ofs_types[] = {
	/* CDIO_FS_* constants come from cdio/cd_types.h */
	{ 0,                              "none",                         "none",         "none"        },
	{ CDIO_FS_AUDIO,                  "CDIO_FS_AUDIO",                "AUDIO",        "CD-Audio"    },  /* CDIO_FS_AUDIO = 1 */
	{ CDIO_FS_HIGH_SIERRA,            "CDIO_FS_HIGH_SIERRA",          "HS",           "High Sierra" },
	{ CDIO_FS_ISO_9660,               "CDIO_FS_ISO_9660",             "ISO9660",      "ISO 9660"    },
	{ CDIO_FS_INTERACTIVE,            "CDIO_FS_INTERACTIVE",          "INT.",         "Interactive" },
	{ CDIO_FS_HFS,                    "CDIO_FS_HFS",                  "HFS",          "HFS (Hierarchical File System)" },  /* MacOS 6 through MacOS 9 */
	{ CDIO_FS_UFS,                    "CDIO_FS_UFS",                  "UFS",          "UFS (Unix File System" },  /* Generic Unix file system */
	{ CDIO_FS_EXT2,                   "CDIO_FS_EXT2",                 "EXT2",         "EXT2" },
	{ CDIO_FS_ISO_HFS,                "CDIO_FS_ISO_HFS",              "ISO+HFS",      "ISO 9660 + HFS (Hierarchical File System)"  },  /* both HFS & ISO-9660 file system */
	{ CDIO_FS_ISO_9660_INTERACTIVE,   "CDIO_FS_ISO_9660_INTERACTIVE", "ISO INT.",     "ISO 9660 Interactive" },  /* CD-RTOS and ISO filesystem */
	{ CDIO_FS_3DO,                    "CDIO_FS_3DO",                  "3DO",          "3DO"  },  /* 3DO Company */
	{ CDIO_FS_XISO,                   "CDIO_FS_XISO",                 "XISO",         "XISO" },  /* Microsoft X-BOX CD */
	{ CDIO_FS_UDFX,                   "CDIO_FS_UDFX",                 "UDFX",         "UDFX" },
	{ CDIO_FS_UDF,                    "CDIO_FS_UDF",                  "UDF",          "UDF (Universal Disk Format)" },
	{ CDIO_FS_ISO_UDF,                "CDIO_FS_ISO_UDF",              "ISO+UDF",      "ISO 9660 Interactive + UDF (Universal Disk Format"  },
	{ CDIO_FS_UNKNOWN,                "CDIO_FS_UNKNOWN",              "UNKNOWN",      "Unknown file system" },
	{ -1,                             (char *) NULL,                  (char *) NULL,  (char *) NULL }};   /* guard */




static cdw_rv_t cdw_ofs_get_iso(cdw_ofs_t *ofs, const char *fullpath);
static void cdw_ofs_resolve_tool(cdw_ofs_t *ofs);


/**
   \brief Allocate new ofs data structure, initialize it

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Function allocates new data structure of type cdw_ofs_t,
   and initializes its fields with default values.
   The function does not read any data from any optical file
   system.

   Call cdw_ofs_delete() to deallocate value returned by
   this function.

   \return NULL pointer on malloc failure
   \return pointer to new ofs data structure otherwise
*/
cdw_ofs_t *cdw_ofs_new(void)
{
	cdw_ofs_t *ofs = (cdw_ofs_t *) malloc(sizeof (cdw_ofs_t));
	if (!ofs) {
		cdw_vdm ("ERROR: malloc()\n");
		return (cdw_ofs_t *) NULL;
	}

	ofs->fd = -1;
	ofs->type = 0; /* none */
	ofs->type_label = strdup(cdw_ofs_short_label(ofs->type));
	ofs->n_sectors = 0; /* Initially assume empty disc with no OFS. Its size would be zero, not -1. */

	ofs->cdw_tool = -1; /* external_tools/cdw_ext_tools.h/enum cdw_tool/CDW_TOOL_NONE */

	ofs->preparer_id = (char *) NULL;
	ofs->app_id = (char *) NULL;
	ofs->publisher_id = (char *) NULL;
	ofs->volume_id = (char *) NULL;
	ofs->volumeset_id = (char *) NULL;

	// ofs->joliet_ucs_level = -1;

	ofs->cdio_iso = (iso9660_t *) NULL;

	return ofs;
}





/**
   \brief Fill ofs data structure with information about optical file system

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Fill \p ofs data structure with information about optical file system
   pointed to by \p fullpath. Currently this only works for ISO9660.

   Caller should pass to the function expected \p type of file
   system. The function isn't intelligent enough to know the type
   by itself, and won't do any checks. It is up to caller to check this.

   Function will *attempt* to return CDW_OK when checking supported file system.
   It will return CDW_ERROR if it will fail.
   It will return CDW_NO when optical file system is currently unsupported
   by this function.

   List of currently supported file systems: ISO9660.

   \p ofs should be created with cdw_ofs_new().

   \param ofs - data structure to fill with information
   \param fullpath - full path to optical file system to be checked
   \param type - expected type of file system

   \return CDW_NO if give ofs type is not supported
   \return CDW_ERROR if getting information about supported ofs won't succeed
   \return CDW_OK if getting information about supported ofs will succeed
*/
cdw_rv_t cdw_ofs_get(cdw_ofs_t *ofs, const char *fullpath, cdio_fs_t type)
{
	cdw_assert (type <= (cdio_fs_t) CDIO_FS_UNKNOWN, "ERROR: fs out of bounds: %d\n", type);
	if (type > (cdio_fs_t) CDIO_FS_UNKNOWN) {
		type = CDIO_FS_UNKNOWN;
	}
	cdw_assert (ofs, "ERROR: ofs is NULL\n");
	cdw_assert (fullpath, "ERROR: fullpath is NULL\n");

	ofs->fd = open(fullpath, O_RDONLY);
	if (ofs->fd == -1) {
		int e = errno;
		cdw_vdm ("ERROR: can't open ISO file system on a disc, errno = \"%s\"\n", strerror(e));
		return CDW_ERROR;
	}

	ofs->type = type;

	cdw_string_set(&(ofs->type_label), cdw_ofs_short_label(type));
	if (!ofs->type_label) {
		cdw_vdm ("ERROR: cdw_string_set()\n");
		return CDW_ERROR;
	}

	if (cdw_ofs_is_iso(ofs->type)) {
		return cdw_ofs_get_iso(ofs, fullpath);
	} else {
		;
	}

	return CDW_NO; /* no supported ofs caught */
}





/**
   \brief Get information about ISO9660 file system

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Function will treat file system pointed to by \p fullpath as ISO9660
   optical file system, and will attempt to collect information about it.
   Information will be stored in \p ofs.

   \param ofs - data structure to be filled with information
   \param fullpath - path to optical file system

   \return CDW_ERROR - if function will fail to get full information
   \return CDW_OK otherwise
*/
cdw_rv_t cdw_ofs_get_iso(cdw_ofs_t *ofs, const char *fullpath)
{
	ofs->cdio_iso = iso9660_open_ext(fullpath, ISO_EXTENSION_ALL);
	if (!ofs->cdio_iso) {
		cdw_vdm ("ERROR: iso9660_open\n");
		return CDW_ERROR;
	}

	iso9660_pvd_t pvd;
	bool rv = iso9660_ifs_read_pvd(ofs->cdio_iso, &pvd);
	if (!rv) {
		cdw_vdm ("ERROR: iso9660_ifs_read_pvd\n");
		return CDW_ERROR;
	}

	ofs->preparer_id = iso9660_get_preparer_id(&pvd);
	if (!ofs->preparer_id) {
		cdw_vdm ("ERROR: iso9660_ifs_get_preparer_id\n");
		return CDW_ERROR;
	}

	ofs->app_id = iso9660_get_application_id(&pvd);
	if (!ofs->app_id) {
		cdw_vdm ("ERROR: iso9660_ifs_get_application_id\n");
		return CDW_ERROR;
	}

	ofs->publisher_id = iso9660_get_publisher_id(&pvd);
	if (!ofs->publisher_id) {
		cdw_vdm ("ERROR: iso9660_ifs_get_publisher_id\n");
		return CDW_ERROR;
	}

	ofs->volume_id = iso9660_get_volume_id(&pvd);
	if (!ofs->volume_id) {
		cdw_vdm ("ERROR: iso9660_get_volume_id\n");
		return CDW_ERROR;
	}

	ofs->volumeset_id = iso9660_get_volumeset_id(&pvd);
	if (!ofs->volumeset_id) {
		cdw_vdm ("ERROR: iso9660_get_volumeset_id\n");
		return CDW_ERROR;
	}

#if 0   /* maybe someday */
	ofs->joliet_ucs_level = iso9660_ifs_get_joliet_level(ofs->cdio_iso);

 	CdioList_t *list = iso9660_ifs_readdir(ofs->cdio_iso, "/");
	void *node;
	_CDIO_LIST_FOREACH(node, list) {
		iso9660_stat_t *st = _cdio_list_node_data(node);
		cdw_vdm ("INFO: size = %d bytes * %d, %s\n", st->size, st->secsize, st->filename);
	}
	_cdio_list_free(list, true);
#endif

	cdw_ofs_resolve_tool(ofs);

	return CDW_OK;
}





/**
   \brief Print information stored in ofs data structure

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   This is a debug function.
   Print to stderr value of most of fields of cdw_ofs_t data structure.

   \param ofs - data structure to be printed
*/
void cdw_ofs_print_debug(cdw_ofs_t *ofs)
{
	cdw_assert (ofs, "ERROR: ofs is NULL\n");

	cdw_vdm ("INFO: fd              = %d\n",     ofs->fd);
	cdw_vdm ("INFO: type            = %d\n",     ofs->type);
	cdw_vdm ("INFO: type label      = \"%s\"\n", ofs->type_label);
	cdw_vdm ("INFO: n sectors       = %ld\n",    ofs->n_sectors);
	cdw_vdm ("INFO: cdw tool        = %d / \"%s\"\n", ofs->cdw_tool, cdw_ext_tools_get_tool_name(ofs->cdw_tool));
	cdw_vdm ("INFO: preparer id     = \"%s\"\n", ofs->preparer_id);
	cdw_vdm ("INFO: app id          = \"%s\"\n", ofs->app_id);
	cdw_vdm ("INFO: publisher id    = \"%s\"\n", ofs->publisher_id);
	cdw_vdm ("INFO: volume id       = \"%s\"\n", ofs->volume_id);
	cdw_vdm ("INFO: volumeset id    = \"%s\"\n", ofs->volumeset_id);
	//cdw_vdm ("INFO: joliet ucs level = %d\n",     ofs->joliet_ucs_level);

	return;
}





/**
   \brief Destroy and deallocate ofs data structure

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Destroy and deallocate data structure returned by cdw_ofs_new().
   Function sets \p ofs to NULL.

   \param ofs - pointer to data to be deleted
*/
void cdw_ofs_delete(cdw_ofs_t **ofs)
{
	cdw_assert (ofs, "ERROR: NULL pointer to ofs\n");

	if (!(*ofs)) {
		cdw_vdm ("WARNING: ofs is NULL\n");
		return;
	}

	if ((*ofs)->fd != -1) {
		close((*ofs)->fd);
		(*ofs)->fd = -1;
	}
	if ((*ofs)->type_label) {
		free((*ofs)->type_label);
		(*ofs)->type_label = (char *) NULL;
	}
	if ((*ofs)->preparer_id) {
		free((*ofs)->preparer_id);
		(*ofs)->preparer_id = (char *) NULL;
	}
	if ((*ofs)->app_id) {
		free((*ofs)->app_id);
		(*ofs)->app_id = (char *) NULL;
	}
	if ((*ofs)->publisher_id) {
		free((*ofs)->publisher_id);
		(*ofs)->publisher_id = (char *) NULL;
	}
	if ((*ofs)->volume_id) {
		free((*ofs)->volume_id);
		(*ofs)->volume_id = (char *) NULL;
	}
	if ((*ofs)->volumeset_id) {
		free((*ofs)->volumeset_id);
		(*ofs)->volumeset_id = (char *) NULL;
	}
	if ((*ofs)->cdio_iso) {
		iso9660_close((*ofs)->cdio_iso);
		(*ofs)->cdio_iso = (iso9660_t *) NULL;
	}

	free(*ofs);
	*ofs = (cdw_ofs_t *) NULL;

	return;
}





/**
   \brief Return readable representation of file system type

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Return a string that is a readable representation (a label) of
   given file system type \p type.
   Returned pointer is owned by this module. Caller should
   probably make a copy of the string.

   Make sure to pass valid value of \p type, otherwise function will
   return "Unknown file system" string.

   \param type - file system type

   \return string with label of given file system
*/
const char *cdw_ofs_long_label(cdio_fs_t type)
{
	cdw_assert (CDIO_FS_AUDIO == 1, "ERROR: CDIO_FS_AUDIO has changed, this will break indexing of cdw_ofs_types[]\n");
	cdw_assert (type <= (cdio_fs_t) CDIO_FS_UNKNOWN, "ERROR: fs out of bounds: %d\n", type);

	if (type > (cdio_fs_t) CDIO_FS_UNKNOWN) {
		type = CDIO_FS_UNKNOWN;
	}
	return cdw_ofs_types[type].cdio_fs_long_label;
}





/**
   \brief Return short version of readable representation of file system type

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Return a string that is a short version of readable representation
   (a short label) of given file system type \p type.
   Returned pointer is owned by this module. Caller should
   probably make a copy of the string.

   Make sure to pass valid value of \p type, otherwise function will
   return "UNKNOWN" string.

   \param type - file system type

   \return string with short label of given file system
*/
const char *cdw_ofs_short_label(cdio_fs_t type)
{
	cdw_assert (CDIO_FS_AUDIO == 1, "ERROR: CDIO_FS_AUDIO has changed, this will break indexing of cdw_ofs_types[]\n");
	cdw_assert (type <= (cdio_fs_t) CDIO_FS_UNKNOWN, "ERROR: fs out of bounds: %d\n", type);

	if (type > (cdio_fs_t) CDIO_FS_UNKNOWN) {
		type = CDIO_FS_UNKNOWN;
	}
	return cdw_ofs_types[type].cdio_fs_short_label;
}





/**
   \brief Check if given file system type is ISO9660 file system

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   \param type - file system type to be checked

   \return true if give file system \p type is a ISO9660 file system
   \return true if give file system \p type is not a ISO9660 file system
*/
bool cdw_ofs_is_iso(cdio_fs_t type)
{
	if (type == CDIO_FS_ISO_9660
	    || type == CDIO_FS_ISO_UDF
	    || type == CDIO_FS_ISO_HFS
	    || type == CDIO_FS_ISO_9660_INTERACTIVE) {

		cdw_vdm ("INFO: file system %d / \"%s\" is ISO9660\n", type, cdw_ofs_long_label(type));
		return true;
	} else {
		cdw_vdm ("INFO: file system %d / \"%s\" is not ISO9660\n", type, cdw_ofs_long_label(type));
		return false;
	}
}





/**
   \brief Create almost exact copy of given ofs data structure

   \date Function's top-level comment reviewed on 2012-01-18
   \date Function's body reviewed on 2012-01-18

   Create new ofs data structure, copy values from given \p ofs to
   the new copy. Return the copy.

   Function copies all fields, with exception of cdio_iso9660.
   If any of fields is copied incorrectly, function returns NULL pointer.

   Call cdw_ofs_delete() to destroy data returned by this function.

   \param ofs - ofs data structure to be copied

   \return NULL pointer on copy errors
   \return pointer to copy of ofs otherwise
*/
cdw_ofs_t *cdw_ofs_copy(cdw_ofs_t *ofs)
{
	cdw_ofs_t *copy = cdw_ofs_new();
	if (!copy) {
		cdw_vdm ("ERROR: cdw_ofs_new()\n");
		return (cdw_ofs_t *) NULL;
	}

	bool success = true;

	copy->fd = ofs->fd;

	copy->type = ofs->type;

	if (success && ofs->type_label) {
		cdw_string_set(&(copy->type_label), ofs->type_label);
		if (!copy->type_label) {
			success = false;
		}
	}

	copy->n_sectors = ofs->n_sectors;

	copy->cdw_tool = ofs->cdw_tool;

	if (success && ofs->preparer_id) {
		copy->preparer_id = strdup(ofs->preparer_id);
		if (!copy->preparer_id) {
			success = false;
		}
	}
	if (success && ofs->app_id) {
		copy->app_id = strdup(ofs->app_id);
		if (!copy->app_id) {
			success = false;
		}
	}
	if (success && ofs->publisher_id) {
		copy->publisher_id = strdup(ofs->publisher_id);
		if (!copy->publisher_id) {
			success = false;
		}
	}
	if (success && ofs->volume_id) {
		copy->volume_id = strdup(ofs->volume_id);
		if (!copy->volume_id) {
			success = false;
		}
	}
	if (success && ofs->volumeset_id) {
		copy->volumeset_id = strdup(ofs->volumeset_id);
		if (!copy->volumeset_id) {
			success = false;
		}
	}

	// copy->joliet_ucs_level = ofs->joliet_ucs_level;

	/* TODO: copy this as well, somehow */
	//cdw_cdio_iso9660_t cdio_iso9660;

	if (!success) {
		cdw_ofs_delete(&copy);
	}

	return copy;
}





/**
   \brief Set proper value of cdw_ofs_t->cdw_tool

   \date Function's top-level comment reviewed on 2012-02-11
   \date Function's body reviewed on 2012-02-11

   Examine text fields of \ofs (set by cdio function calls), and
   set proper value of ofs->cdw_tool field. The value will be
   CDW_TOOL_XORRISO, CDW_TOOL_MKISOFS, or CDW_TOOL_NONE. The value
   can be used later by higher level code, e.g. to determine which
   tool to select for adding data to already non-empty disc.

   \param ofs - data structure to examine and alter
*/
void cdw_ofs_resolve_tool(cdw_ofs_t *ofs)
{
	if (strstr(ofs->preparer_id, "XORRISO")) {
		/* "Preparer   : XORRISO-0.5.6 2010.05.04.100001, LIBISOBURN-0.5.6, LIBISOFS-0.6." */
		ofs->cdw_tool = CDW_TOOL_XORRISO;

	} else if (strstr(ofs->app_id, "GENISOIMAGE")) {
		/* "Application: GENISOIMAGE ISO 9660_HFS FILESYSTEM CREATOR (C) 1993 E.YOUNGDALE" */
		/* TODO: also check image created by original mkisofs */
		ofs->cdw_tool = CDW_TOOL_MKISOFS;

	} else {
		ofs->cdw_tool = CDW_TOOL_NONE;
	}

	return;
}





bool cdw_ofs_is_iso9660_file(const char *fullpath)
{
	iso9660_t *iso = iso9660_open(fullpath);
	if (!iso) {
		cdw_vdm ("INFO: %s is not ISO\n", fullpath);
		return false;
	} else {
		cdw_vdm ("INFO: %s is ISO\n", fullpath);
		iso9660_close(iso);
		return true;
	}
}
