/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
*/

#include "cameraresolution.h"
#include "camera.h"
#include "../cordova.h"


#include <QCamera>
#include <QCameraViewfinder>
#include <QCameraImageCapture>
#include <QGraphicsObject>
#include <QCloseEvent>
#include <QTemporaryFile>
#include <QQuickItem>

#include <QDebug>

#define CAMERA_STATE_NAME "camera"
#define RECORDVIDEO_STATE_NAME "recordVideo"

Camera::Camera(Cordova *cordova):
    CPlugin(cordova),
    m_lastScId(0),
    m_lastEcId(0) {
}

bool Camera::preprocessImage(QString &path) {
    bool convertToPNG = (*m_options.find("encodingType")).toInt() == Camera::PNG;
    int quality = (*m_options.find("quality")).toInt();
    int width = (*m_options.find("targetWidth")).toInt();
    int height = (*m_options.find("targetHeight")).toInt();

    QImage image(path);
    if (width && height)
        image = image.scaled(width, height, Qt::KeepAspectRatio, Qt::SmoothTransformation);

    QFile oldImage(path);
    QTemporaryFile newImage;

    const char *type;
    if (convertToPNG) {
        newImage.setFileTemplate("imgXXXXXX.png");
        type = "png";
    } else {
        newImage.setFileTemplate("imgXXXXXX.jpg");
        type = "jpg";
    }

    newImage.open();
    newImage.setAutoRemove(false);
    image.save(newImage.fileName(), type, quality);

    path = newImage.fileName();
    oldImage.remove();

    return true;
}

void Camera::cancel() {
    bool captureAPI = m_options.find("captureAPI")->toBool();

    leaveState(m_state);

    if (captureAPI && !m_result.isEmpty()) {
        this->callback(m_lastScId, QString("[%1]").arg(m_result));
    } else {
        this->cb(m_lastEcId, "canceled");
    }
    m_result = "";

    m_lastEcId = m_lastScId = 0;
}

void Camera::onImageSaved(int, QString path) {
    bool captureAPI = m_options.find("captureAPI")->toBool();
    bool dataURL = m_options.find("destinationType")->toInt() == Camera::DATA_URL;
    int limit = m_options.find("limit")->toInt();

    QString cbParams;
    if (preprocessImage(path)) {
        QString absolutePath = QFileInfo(path).absoluteFilePath();
        if (dataURL) {
            QFile image(absolutePath);
            image.open(QIODevice::ReadOnly);
            QByteArray content = image.readAll().toBase64();
            cbParams = QString("\"%1\"").arg(content.data());
            image.remove();
        } else {
            cbParams = QString("\"%1\"").arg(QUrl::fromLocalFile(absolutePath).toString());
        }
    }

    if (!captureAPI) {
        leaveState(CAMERA_STATE_NAME);
        this->callback(m_lastScId, cbParams);
    } else {
        if (!m_result.isEmpty())
            m_result += ", ";
        m_result += cbParams;
        if (limit <= 1) {
            leaveState(CAMERA_STATE_NAME);
            this->callback(m_lastScId, QString("[%1]").arg(m_result));
            m_result = "";
        }
    }

    QObject *object = m_cordova->rootObject()->findChild<QObject*>(CAMERA_STATE_NAME);
    QObject *imageCapture = object->property("imageCapture").value<QObject*>();
    QObject::disconnect(imageCapture, SIGNAL(imageSaved(int, const QString)), this,  SLOT(onImageSaved(int, const QString)));

    if (--limit > 0) {
        m_options.insert("limit", limit);
        getPicture(m_lastScId, m_lastEcId, m_options);
    } else {
        m_lastEcId = m_lastScId = 0;
    }
}

void Camera::getPicture(int scId, int ecId, QVariantMap p_options) {
    if (m_camera.isNull()) {
        m_camera = QSharedPointer<QCamera>(new QCamera());
    }

    if (((m_lastScId || m_lastEcId) && (m_lastScId != scId && m_lastEcId != ecId)) || !m_camera->isAvailable() || m_camera->lockStatus() != QCamera::Unlocked) {
        this->cb(m_lastEcId, "Device is busy");
        return;
    }

    m_options = p_options;
    m_lastScId = scId;
    m_lastEcId = ecId;

    if (m_state != CAMERA_STATE_NAME)
        setState(CAMERA_STATE_NAME);

    QObject *object = m_cordova->rootObject()->findChild<QObject*>("camera");
    QObject *imageCapture = object->property("imageCapture").value<QObject*>();
    QObject::connect(imageCapture, SIGNAL(imageSaved(int, const QString)), this,  SLOT(onImageSaved(int, QString)));
}

void Camera::recordVideo(int scId, int ecId, QVariantMap p_options) {
    if (m_camera.isNull()) {
        m_camera = QSharedPointer<QCamera>(new QCamera());
    }

    if (((m_lastScId || m_lastEcId) && (m_lastScId != scId && m_lastEcId != ecId)) || !m_camera->isAvailable() || m_camera->lockStatus() != QCamera::Unlocked) {
        this->cb(m_lastEcId, "Device is busy");
        return;
    }

    m_options = p_options;
    m_lastScId = scId;
    m_lastEcId = ecId;

    setState(RECORDVIDEO_STATE_NAME);
}


void Camera::onRecordEnd() {
    leaveState(RECORDVIDEO_STATE_NAME);

//    QObject *object = view->rootObject()->findChild<QObject*>("camera");
//    QObject *videoRecording = object->property("videoRecording").value<QObject*>();
}

void Camera::setState(const QString &state) {
    Q_ASSERT(m_state=="");
    m_state = state;
    m_cordova->pushViewState(m_state);
}

void Camera::leaveState(const QString &state) {
    Q_ASSERT(state == m_state);
    m_cordova->popViewState(m_state);
    m_state = "";
}
