/*
 * Copyright 2015 The Error Prone Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.errorprone.bugpatterns;

import static com.google.errorprone.BugPattern.SeverityLevel.WARNING;
import static com.google.errorprone.matchers.Matchers.assertionWithCondition;
import static com.google.errorprone.matchers.Matchers.booleanLiteral;

import com.google.errorprone.BugPattern;
import com.google.errorprone.VisitorState;
import com.google.errorprone.bugpatterns.BugChecker.AssertTreeMatcher;
import com.google.errorprone.fixes.SuggestedFix;
import com.google.errorprone.matchers.Description;
import com.google.errorprone.matchers.Matcher;
import com.sun.source.tree.AssertTree;

/**
 * @author sebastian.h.monte@gmail.com (Sebastian Monte)
 */
@BugPattern(
    summary =
        "Assertions may be disabled at runtime and do not guarantee that execution will "
            + "halt here; consider throwing an exception instead",
    severity = WARNING)
public class AssertFalse extends BugChecker implements AssertTreeMatcher {

  private static final Matcher<AssertTree> ASSERT_FALSE_MATCHER =
      assertionWithCondition(booleanLiteral(false));

  @Override
  public Description matchAssert(AssertTree tree, VisitorState state) {
    if (!ASSERT_FALSE_MATCHER.matches(tree, state)) {
      return Description.NO_MATCH;
    }

    return describeMatch(tree, SuggestedFix.replace(tree, "throw new AssertionError()"));
  }
}
