/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ab_fritzfon.c
 * \brief FritzFon addressbook plugin
 */

#include <ffgtk.h>
#include <sys/stat.h>

extern const gchar *getProfilesDir( void );

#define VALID_NUMBER( x ) ( ( x != NULL ) && ( strlen( x ) > 0 ) )

/** Swap first/last name */
static gboolean fritzfonGetSwap( void ) {
	return prefsGetBool( getActiveProfile(), "/plugins/fritzfon/swap" );
}

/** Swap first/last name */
static void fritzfonSetSwap( gboolean bSwap ) {
	prefsSetBool( getActiveProfile(), "/plugins/fritzfon/swap", bSwap );
}

/**
 * \brief Get profile book file name
 * \return file name
 */
static gchar *getBookFile( void ) {
	gchar *pnFile;
	
	pnFile = g_build_filename( getProfilesDir(), getActiveProfile() -> pnName, "fritzfon.xml", NULL );

	return pnFile;
}


/**
 * \brief Convert person structure to xml node
 * \param psPerson person structure
 * \return xml node
 */
static xmlnode *contactToXmlnode( struct sPerson *psPerson ) {
	xmlnode *psNode;
	xmlnode *psPersonNode;
	xmlnode *psRealName;
	xmlnode *psImage;
	xmlnode *psTelephony;
	xmlnode *psCategory;
	gchar *pnTmp = NULL;

	/* Main contact entry */
	psNode = xmlnode_new( "contact" );

	/* Category */
	psCategory = xmlnode_new( "category" );
	xmlnode_insert_data( psCategory, psPerson -> pnCategory, -1 );
	xmlnode_insert_child( psNode, psCategory );

	/* Person */
	psPersonNode = xmlnode_new( "person" );

	/* RealName */
	if ( fritzfonGetSwap() == FALSE ) {
		pnTmp = g_strdup_printf( "%s %s", psPerson -> pnFirstName, psPerson -> pnLastName );
	} else {
		pnTmp = g_strdup_printf( "%s %s", psPerson -> pnLastName, psPerson -> pnFirstName );
	}

	psRealName = xmlnode_new( "realName" );
	xmlnode_insert_data( psRealName, pnTmp, -1 );
	xmlnode_insert_child( psPersonNode, psRealName );
	g_free( pnTmp );

	/* ImageURL stub */
	psImage = xmlnode_new( "ImageURL" );
	xmlnode_insert_child( psPersonNode, psImage );

	/* Insert person to main node */
	xmlnode_insert_child( psNode, psPersonNode );

	/* Telephony */
	psTelephony = xmlnode_new( "telephony" );

	/* Add home phone */
	if ( psPerson -> pnPrivatePhone != NULL && strlen( psPerson -> pnPrivatePhone ) > 0 ) {
		xmlnode *psNumber;

		psNumber = xmlnode_new( "number" );
		xmlnode_set_attrib( psNumber, "type", "home" );

		/* For the meantime set priority to 0, TODO */
		xmlnode_set_attrib( psNumber, "prio", "0" );
	
		xmlnode_insert_data( psNumber, psPerson -> pnPrivatePhone, -1 );
		xmlnode_insert_child( psTelephony, psNumber );
	}

	/* Add mobile phone */
	if ( psPerson -> pnPrivateMobile != NULL && strlen( psPerson -> pnPrivateMobile ) > 0 ) {
		xmlnode *psNumber;

		psNumber = xmlnode_new( "number" );
		xmlnode_set_attrib( psNumber, "type", "mobile" );

		/* For the meantime set priority to 0, TODO */
		xmlnode_set_attrib( psNumber, "prio", "0" );
		xmlnode_insert_data( psNumber, psPerson -> pnPrivateMobile, -1 );
		xmlnode_insert_child( psTelephony, psNumber );
	}

	/* Add work phone */
	if ( psPerson -> pnBusinessPhone != NULL && strlen( psPerson -> pnBusinessPhone ) > 0 ) {
		xmlnode *psNumber;

		psNumber = xmlnode_new( "number" );
		xmlnode_set_attrib( psNumber, "type", "work" );

		/* For the meantime set priority to 0, TODO */
		xmlnode_set_attrib( psNumber, "prio", "0" );
		xmlnode_insert_data( psNumber, psPerson -> pnBusinessPhone, -1 );
		xmlnode_insert_child( psTelephony, psNumber );
	}
	xmlnode_insert_child( psNode, psTelephony );

	return psNode;
}

/**
 * \brief Convert phonebooks to xml node
 * \return xml node
 */
static xmlnode *phonebooksToXmlnode( void ) {
	xmlnode *psNode;
	xmlnode *psChild;
	xmlnode *psBook;
	GList *psList = psPersonsList;

	/* Create general phonebooks node */
	psNode = xmlnode_new( "phonebooks" );

	/* Currently we only support one phonebook, TODO */
	psBook = xmlnode_new( "phonebook" );
	xmlnode_set_attrib( psBook, "name", "Telefonbuch" );
	xmlnode_insert_child( psNode, psBook );

	/* Loop through persons list and add only non-deleted entries */
	for ( psList = psPersonsList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		struct sPerson *psPerson = psList -> data;

		/* Skip deleted persons */
		if ( psPerson -> nFlags & PERSON_FLAG_DELETED ) {
			psPerson -> nFlags = PERSON_FLAG_UNCHANGED;
			continue;
		}

		if ( !VALID_NUMBER( psPerson -> pnPrivatePhone ) &&
			!VALID_NUMBER( psPerson -> pnPrivateMobile ) &&
			!VALID_NUMBER( psPerson -> pnBusinessPhone ) ) {
			continue;
		}

		/* Convert each contact and add it to current phone book */
		psChild = contactToXmlnode( psPerson );
		xmlnode_insert_child( psBook, psChild );
	}

	return psNode;
}

/**
 * \brief Add contact to persons list
 * \param psNode xml contact node
 * \param nCount unique person id
 */
static void addContact( xmlnode *psNode, int nCount ) {
	xmlnode *psPerson;
	xmlnode *psRealName;
	xmlnode *psImage;
	xmlnode *psTelephony;
	xmlnode *psChild;
	xmlnode *psCategory;
	gchar *pnFirstName;
	gchar *pnLastName;
	gchar *pnHome = NULL;
	gchar *pnWork = NULL;
	gchar *pnMobile = NULL;
	gchar *pnTmp;
	gchar *pnRealName;
	gchar *pnImage;
	gchar *pnCategory = NULL;
	GdkPixbuf *psImageBuf = NULL;

	/* Get person entry */
	psPerson = xmlnode_get_child( psNode, "person" );
	if ( psPerson == NULL ) {
		return;
	}

	/* Get real name entry */
	psRealName = xmlnode_get_child( psPerson, "realName" );
	if ( psRealName == NULL ) {
		return;
	}	
	pnRealName = xmlnode_get_data( psRealName );

	pnTmp = strchr( pnRealName, ' ' );
	if ( pnTmp != NULL ) {
		gint nLen = 0;
		pnLastName = strrchr( pnRealName, ' ' ) + 1;
		nLen = strlen( pnRealName ) - strlen( pnLastName ) - 1;
		pnFirstName = g_malloc0( nLen + 1 );
		strncpy( pnFirstName, pnRealName, nLen );
		pnFirstName[ nLen ] = '\0';
	} else {
		pnFirstName = NULL;
		pnLastName = pnRealName;
	}

	psTelephony = xmlnode_get_child( psNode, "telephony" );
	if ( psTelephony != NULL ) {
		/* Check for numbers */
		for ( psChild = xmlnode_get_child( psTelephony, "number" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
			const gchar *pnType;

			pnType = xmlnode_get_attrib( psChild, "type" );
			if ( pnType == NULL ) {
				continue;
			}

			if ( strcmp( pnType, "mobile" ) == 0 ) {
				pnMobile = xmlnode_get_data( psChild );
			} else if ( strcmp( pnType, "home" ) == 0 ) {
				pnHome = xmlnode_get_data( psChild );
			} else if ( strcmp( pnType, "work" ) == 0 ) {
				pnWork = xmlnode_get_data( psChild );
			}
		}
	}

	/* Get image */
	psImage = xmlnode_get_child( psPerson, "imageURL" );
	if ( psImage != NULL ) {
		pnImage = xmlnode_get_data( psImage );
		if ( pnImage != NULL ) {
			/* file:///var/InternerSpeicher/FRITZ/fonpix/946684999-0.jpg */
			if ( strlen( pnImage ) > 28 ) {
				struct sUrlHandler *psHandler = NULL;
				int nPos = findString( pnImage, 0, "/FRITZ/" );
				nPos = findStringReverse( pnImage, nPos, "/" );

				gchar *pnFile = g_strdup_printf( "ftp://%s/%s", routerGetHost( getActiveProfile() ), pnImage + nPos );
				int nError;
				int nHandle;
				int nResult;

				psHandler = urlHandler( pnFile, 21 );
				nError = readUrl( psHandler, getActiveProfile() );
				if ( nError == 0 ) {
					nHandle = open( "/tmp/test.jpg", O_RDWR | O_CREAT | O_TRUNC, 0660 );

					if ( nHandle != -1 ) {
						nResult = write( nHandle, psHandler -> pnData, psHandler -> nSize );
						if ( nResult != psHandler -> nSize ) {
							Debug( KERN_DEBUG, "Could not save image\n" );
						}
						close( nHandle );
						psImageBuf = gdk_pixbuf_new_from_file( "/tmp/test.jpg", NULL );
					}
					freeHandler( psHandler );
				}
			}
		}
	}

	psCategory = xmlnode_get_child( psNode, "category" );
	if ( psCategory != NULL ) {
		pnCategory = xmlnode_get_data( psCategory );
	}

	/* Create person hash table, fill structure and add them to internal list */
	GHashTable *psTable = g_hash_table_new( NULL, NULL );
	gchar *pnId = g_strdup_printf( "%d", nCount );

	AddInfo( psTable, PERSON_ID, pnId );
	if ( fritzfonGetSwap() == FALSE ) {
		AddInfo( psTable, PERSON_FIRST_NAME, pnFirstName );
		AddInfo( psTable, PERSON_LAST_NAME, pnLastName );
	} else {
		AddInfo( psTable, PERSON_FIRST_NAME, pnLastName );
		AddInfo( psTable, PERSON_LAST_NAME, pnFirstName );
	}
	AddInfo( psTable, PERSON_DISPLAY_NAME, pnRealName );
	AddInfo( psTable, PERSON_BUSINESS_PHONE, pnWork );
	AddInfo( psTable, PERSON_PRIVATE_PHONE, pnHome );
	AddInfo( psTable, PERSON_PRIVATE_MOBILE, pnMobile );
	AddInfo( psTable, PERSON_IMAGE, ( const gchar * ) psImageBuf );
	AddInfo( psTable, PERSON_CATEGORY, pnCategory );

	AddPerson( psTable, FALSE );
	g_free( pnId );

	g_hash_table_destroy( psTable );
}

/**
 * \brief Add phonebook
 * \param psNode phonebook xml node
 */
static void addPhonebook( xmlnode *psNode ) {
	xmlnode *psChild;
	int nCount = 0;

	for ( psChild = xmlnode_get_child( psNode, "contact" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
		addContact( psChild, nCount++ );
	}
}
		
/**
 * \brief Read ebook list
 * \return error code
 */
static int fritzfonReadBook( void ) {
	struct sUrlHandler *psHandler;
	gchar anBuffer[ 1024 ];
	gint nError = -1;
	xmlnode *psNode = NULL;
	xmlnode *psChild;
	struct curl_httppost* post = NULL;
	struct curl_httppost* last = NULL;
	struct curl_slist *m_headerlist = NULL;

	if ( routerLogin( getActiveProfile() ) == -1 ) {
		return -1;
	}

	snprintf( anBuffer, sizeof( anBuffer ), "%s/cgi-bin/firmwarecfg", routerGetHost( getActiveProfile() ) );
	psHandler = urlHandler( anBuffer, 80 );

	m_headerlist = curl_slist_append(m_headerlist, "ENCTYPE=\"multipart/form-data\""); 	
	curl_easy_setopt(psHandler -> psHandle, CURLOPT_HTTPHEADER, m_headerlist);

	curl_formadd(&post, &last, CURLFORM_COPYNAME, "sid",   CURLFORM_COPYCONTENTS, getActiveProfile() -> sRouterInfo.pnSessionId, CURLFORM_END); 
	curl_formadd(&post, &last, CURLFORM_COPYNAME, "PhonebookId",   CURLFORM_COPYCONTENTS, "0", CURLFORM_END); 
	curl_formadd(&post, &last, CURLFORM_COPYNAME, "PhonebookExportName",   CURLFORM_COPYCONTENTS, "Telefonbuch", CURLFORM_END); 
	curl_formadd(&post, &last, CURLFORM_COPYNAME, "PhonebookExport",   CURLFORM_COPYCONTENTS, "", CURLFORM_END); 

	curl_easy_setopt(psHandler -> psHandle, CURLOPT_HTTPPOST, post); 

	/* set timeout to 3sec */
	//curl_easy_setopt( psHandler -> psHandle, CURLOPT_TIMEOUT, 3 );
	
	nError = readUrl( psHandler, getActiveProfile() );
	if ( nError == 0 ) {
		gchar *pnFile;

		pnFile = getBookFile();
		saveData( pnFile, psHandler -> pnData, psHandler -> nSize );

		psNode = readXmlFromFile( pnFile, "phonebooks" );
		if ( psNode == NULL ) {
			Debug( KERN_DEBUG, "Could not read %s\n", pnFile );
			g_free( pnFile );
			freeHandler( psHandler );
			return -1;
		}
		g_free( pnFile );

		for ( psChild = xmlnode_get_child( psNode, "phonebook" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
			addPhonebook( psChild );
		}
	}
	freeHandler( psHandler );

	routerLogout( getActiveProfile() );
		
	return 0;
}

/**
 * \brief Write ebook list
 * \return error code
 */
static int fritzfonWriteBook( void ) {
	struct sUrlHandler *psHandler;
	gchar anBuffer[ 1024 ];
	gint nError = -1;
	xmlnode *node;
	char *pnData;

	if ( routerLogin( getActiveProfile() ) == -1 ) {
		return -1;
	}

	node = phonebooksToXmlnode();

	pnData = xmlnode_to_formatted_str( node, NULL );
	if ( pnData != NULL ) {
		gchar *pnFile;

		pnFile = getBookFile();
		saveData( pnFile, pnData, -1 );

		struct curl_httppost* post = NULL;
		struct curl_httppost* last = NULL;
		struct curl_slist *m_headerlist = NULL;


		snprintf( anBuffer, sizeof( anBuffer ), "%s/cgi-bin/firmwarecfg", routerGetHost( getActiveProfile() ) );
		psHandler = urlHandler( anBuffer, 80 );

		m_headerlist = curl_slist_append(m_headerlist, "ENCTYPE=\"multipart/form-data\""); 	

		curl_easy_setopt(psHandler -> psHandle, CURLOPT_HTTPHEADER, m_headerlist);
		curl_formadd(&post, &last, CURLFORM_COPYNAME, "sid",   CURLFORM_COPYCONTENTS, getActiveProfile() -> sRouterInfo.pnSessionId, CURLFORM_END); 
		curl_formadd(&post, &last, CURLFORM_COPYNAME, "PhonebookId",   CURLFORM_COPYCONTENTS, "0", CURLFORM_END); 
		curl_formadd(&post, &last, CURLFORM_COPYNAME, "PhonebookImportFile", CURLFORM_FILE, pnFile, CURLFORM_END); 

		curl_easy_setopt(psHandler -> psHandle, CURLOPT_HTTPPOST, post); 
	
		nError = readUrl( psHandler, getActiveProfile() );
		freeHandler( psHandler );
		g_free(pnData);
		g_free( pnFile );
	}
	xmlnode_free(node);

	routerLogout( getActiveProfile() );

	return nError;
}

void prefsAddNone( struct sProfile *psProfile, const char *pnName );

/**
 * \brief Display fritzfon preferences window
 */
static void fritzfonPreferences( void ) {
	GtkWidget *psDialog = gtk_dialog_new_with_buttons( _( "FritzFon Preferences" ), NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	GtkWidget *psCheckBox = gtk_check_button_new_with_label( _( "Swap first/lastname" ) );
	GtkWidget *psBox;

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCheckBox ), fritzfonGetSwap() );
	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psCheckBox, FALSE, TRUE, 15 );
	gtk_widget_set_size_request( psDialog, 300, 80 );
	gtk_widget_show( GTK_WIDGET( psCheckBox ) );
	gtk_dialog_run( GTK_DIALOG( psDialog ) );

	prefsAddNone( getActiveProfile(), "/plugins/fritzfon" );

	fritzfonSetSwap( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCheckBox ) ) );
	gtk_widget_destroy( psDialog );
	SavePreferences( getActiveProfile() );

	freePersons();
	fritzfonReadBook();
}

/** book definition */
static struct sAddressBook sFritzFon = {
	fritzfonReadBook,
	fritzfonWriteBook,
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME |
	PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_BUSINESS_PHONE,
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME |
	PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_BUSINESS_PHONE,
	fritzfonPreferences
};

MODULE_INIT( PLUGIN_TYPE_BOOK, _( "FritzFon Addressbook" ), &sFritzFon, NULL, NULL );
