-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-driver.ads,v $
--  Description     : Ada Database Object - Driver Interface                 --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/02/14 20:46:59 $
--  Version         : $Revision: 1.1 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This is the abstract driver interface for the ADO package. Any           --
--  implmentation of this interface provides interfaces for the following    --
--  packages:                                                                --
--                                                                           --
--                                                                           --
--      +----------------+ +---------------+ +---------------+               --
--      | Ado.Connection | | Ado.Statement | | Ado.Resultset |               --
--      +----------------+ +---------------+ +---------------+               --
--                                                                           --
--      +----------------------------------------------------+               --
--      |                    Ado.Driver                      |               --
--      +----------------------------------------------------+               --
--                                                                           --
--  As a general rule the resource management is always implemented in the   --
--  Ado packages.                                                            --
--                                                                           --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
with GNU.DB.ADBC.Hostvariable;       use GNU.DB.ADBC.Hostvariable;

package GNU.DB.ADBC.Driver is

   type Object is abstract tagged null record;
   type Handle is access all Object'Class;

   Null_Driver_Handle : constant Handle := null;

   Connection_Failure   : exception;
   Wrong_Database_Name  : exception;
   Empty_Resultset      : exception;
   End_Of_Resultset     : exception;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Connect to the data base. user and password may be used to
   --#    authorize the user in the RDBMS.
   --#    The data base name may be used to select the correct data base
   --#    instance.
   --# Preconditions  :
   --#    R.1 - Database not connected
   --# Postconditions :
   --#    P.1 - All data base related insdtances are created by the driver
   --#          implementation.
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Connect(
      This     : in out Object;
      User     : in String;
      Password : in String;
      Database : in String) is abstract;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Disconnect from the data base.
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --#    P.1 - All driver internal resources related to the connection are
   --#          released.
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Disconnect(
      This : in out Object ) is abstract;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    This procedure is intended to prepare the actual statement. The
   --#    query syntax is expected to be pared by the connection module.
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --#    P.1 - A statement id is allocated to the statement.
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Prepare(
      This : in out Object;
      stmt : in Statement_ID  ) is abstract;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Execute the referenced statement and store the result set information
   --#    in the referenced result set.
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Execute_Statement(
      This   : in Object;
      Result : in Resultset_ID;
      Stmt   : in Statement_ID  ) is abstract;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Create/delete an instance of the statement implementation
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   function Create_Statement(
      This : in Object;
      Con  : in Connection_ID ) return Statement_ID is abstract;

   procedure Delete_Statement(
      This : in Object;
      Stmt : in Statement_ID ) is abstract;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Create/Delete an instance of the implementation of an result set.
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   function Create_Resultset(
      This   : in Object;
      Stmt   : in Statement_ID ) return Resultset_ID is abstract;

   procedure Delete_Resultset(
      This   : in Object;
      Result : in Resultset_ID ) is abstract;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Expand the host variable into a SQL compatiable represenation
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   function Expand(
      This   : in Object;
      V      : in Hostvariable.Handle ) return String is abstract;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Bind the host variable with the statement. The method is used in
   --#    order to comply to the ODBC protocol which allowes the usage of
   --#    deferred buffers.
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Bind_Host_Variable(
      This   : in Object;
      Stmt   : in Statement_ID;
      V      : in Hostvariable.Handle ) is abstract;

   procedure Get_Host_Value(
      This : in Object;
      Stmt : in Statement_ID;
      V    : in Hostvariable.Handle) is abstract;

end GNU.DB.ADBC.Driver;

