//*****************************************************************************
//                               PnlAnaBase.cpp                               *
//                              ----------------                              *
// Started     : 26/04/2004                                                   *
// Last Update : 09/08/2011                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "base/PnlAnaBase.hpp"

//*****************************************************************************
// Allocate storage for static data members.

bool       PnlAnaBase::m_bSyncSwpSrcs = FALSE;
Component  PnlAnaBase::m_oCpntSwpSrcs;

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlAnaBase, wxPanel )

  EVT_CHOICE( ID_CHO_SRCNAME, PnlAnaBase::OnSrcName )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List :
//   poWin - A pointer to the parent window

PnlAnaBase::PnlAnaBase( wxWindow * poWin ) : wxPanel( poWin )
{
  m_eSimEng  = eSIMR_NONE;
  m_eAnaType = eCMD_NONE;

  CreateBase( );  // Create the analysis panel

  bClear( );      // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlAnaBase::~PnlAnaBase( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bClear( void )
{
  bool  bRtn=TRUE;

  // Set sweep default values
  if( ! m_oPnlStart.bSetDefaults( ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetDefaults( ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetDefaults( ) ) bRtn = FALSE;

  // Set input source default values
  if( bIsCreated( eCTLGRP_SIGSRC ) && m_oSbxSrc.IsShown( ) )
  {
    m_oChoSrcName.Clear( );
    m_oChoSrcName.Append( wxT("None") );
    m_oChoSrcName.SetSelection( 0 );
    if( m_oPnlSrcLvl.GetParent( ) != NULL )
    {
      m_oPnlSrcLvl.bSetValue( (float) 0.0 );
      m_oPnlSrcLvl.bSetUnitsType( eUNITS_NONE );
    }
  }

  // Set parameters check box default values
  m_oCbxVoltage.SetValue( TRUE );
  m_oCbxCurrent.SetValue( FALSE );
  m_oCbxPower  .SetValue( FALSE );
  m_oCbxResist .SetValue( FALSE );

  // Set the analysis temperature default value
  if( bIsCreated( eCTLGRP_TEMP ) && m_oSbxTemp.IsShown( ) )
    if( ! m_oPnlTemp.bSetDefaults( ) )
      bRtn = FALSE;

  m_osErrMsg.Empty( ); // Clear the error string

  return( bRtn );
}

//*****************************************************************************
// Create the basic display objects.

void  PnlAnaBase::CreateBase( void )
{
  wxPanel * poPnl;

  if( bIsCreated( eCTLGRP_BASE ) ) return;

  // Create the sweep panel and controls
#ifdef LAYOUT_MNGR
  poPnl = new wxPanel( this );
#else
  poPnl = this; // ???
#endif // LAYOUT_MNGR

  // Create and add sweep parameter labels
#ifdef LAYOUT_MNGR
  m_oPnlStart.bCreate( poPnl, ID_PNL_START, 120 );
  m_oPnlStop .bCreate( poPnl, ID_PNL_STOP,  120 );
  m_oPnlStep .bCreate( poPnl, ID_PNL_STEP,  120 );
#else
  m_oSbxSwpPars.Create( poPnl, ID_UNUSED, wxT(" Sweep Parameters "),
                        wxPoint( 6, 5 ), wxSize( 330, 120 ) );
  m_oPnlStart.bCreate( poPnl, ID_PNL_START, 120, wxPoint( 18, 28 ) );
  m_oPnlStop .bCreate( poPnl, ID_PNL_STOP,  120, wxPoint( 18, 58 ) );
  m_oPnlStep .bCreate( poPnl, ID_PNL_STEP,  120, wxPoint( 18, 88 ) );
#endif // LAYOUT_MNGR
  m_oPnlStart.bSetName( wxT("Start Value") );
  m_oPnlStop .bSetName( wxT("Stop Value") );
  m_oPnlStep .bSetName( wxT("Step Increment") );
  m_oSbxSwpPars.SetOwnFont( FONT_SLANT );

  // Create the parameter (probe) panel and controls
#ifdef LAYOUT_MNGR
  poPnl = new wxPanel( this );
#else
  poPnl = this; // ???
#endif // LAYOUT_MNGR

  // Create and add simulation parameter check boxes
#ifdef LAYOUT_MNGR
  m_oCbxVoltage .Create( poPnl, ID_CBX_VOLT, wxT("Voltage")    );
  m_oCbxCurrent .Create( poPnl, ID_CBX_CURR, wxT("Current")    );
  m_oCbxPower   .Create( poPnl, ID_CBX_PWR,  wxT("Power")      );
  m_oCbxResist  .Create( poPnl, ID_CBX_RES,  wxT("Resistance") );
#else
  m_oSbxCalcPars.Create( poPnl, ID_UNUSED  , wxT(" Parameters "),
                         wxPoint( 343,   5 ), wxSize( 105, 122 ) );
  m_oCbxVoltage .Create( poPnl, ID_CBX_VOLT, wxT("Voltage"),
                         wxPoint( 349,  25 ) );
  m_oCbxCurrent .Create( poPnl, ID_CBX_CURR, wxT("Current"),
                         wxPoint( 349,  50 ) );
  m_oCbxPower   .Create( poPnl, ID_CBX_PWR,  wxT("Power"),
                         wxPoint( 349,  75 ) );
  m_oCbxResist  .Create( poPnl, ID_CBX_RES,  wxT("Resistance"),
                         wxPoint( 349, 100 ) );
#endif // LAYOUT_MNGR
  m_oSbxCalcPars.SetOwnFont( FONT_SLANT );

  // Create and add .OPTIONS configuration dialog button
#ifdef LAYOUT_MNGR
  m_oBtnOPTIONS.Create( this, ID_BTN_OPTIONS, wxT("  .OPTIONS ...  ") );
#else
  m_oBtnOPTIONS.Create( this, ID_BTN_OPTIONS, wxT(".OPTIONS ..."),
                        wxPoint( 343, 137 ), wxSize( 105, 29 ) );
#endif // LAYOUT_MNGR
}

//*****************************************************************************
// Create the scale display objects.

void  PnlAnaBase::CreateScale( void )
{
  wxPanel * poPnl;

  if( bIsCreated( eCTLGRP_SCALE ) ) return;

  // Create and add the scale radio buttons
  poPnl = (wxPanel *) m_oPnlStart.GetParent( );
  wxString  osScale[ 4 ] = { wxT("Lin  "), wxT("Log "), wxT("Dec "), wxT("Oct ") };
  m_oRbxScale.Create( poPnl, ID_RBX_SCALE, wxT(" Step Scale "),
                      wxPoint( 13, 123 ), wxDefaultSize, 4, osScale, 4 );
  m_oRbxScale.SetSelection( eSCALE_LIN );
}

//*****************************************************************************
// Create the initial conditions display objects.

void  PnlAnaBase::CreateInitC( void )
{
  wxPanel * poPnl;

  if( bIsCreated( eCTLGRP_INITC ) ) return;

  // Create and add the initial conditions radio buttons
  poPnl = (wxPanel *) m_oPnlStart.GetParent( );
  wxString  osInitC[ 3 ] = { wxT("Warm "), wxT("Use ICs"), wxT("Cold ") };
  m_oRbxInitC.Create( poPnl, ID_UNUSED, wxT(" Initial Conditions "),
                      wxPoint( 13, 123 ), wxDefaultSize, 3, osInitC, 3 );
  m_oRbxInitC.SetSelection( eINITC_WARM );
}

//*****************************************************************************
// Create the source component display objects.

void  PnlAnaBase::CreateSrc( void )
{
  wxPanel * poPnl;

  if( bIsCreated( eCTLGRP_SIGSRC ) ) return;

  // Create the input source panel and controls
#ifdef LAYOUT_MNGR
  poPnl = new wxPanel( this );
#else
  poPnl = this; // ???
#endif // LAYOUT_MNGR

  // Create and add input source
#ifdef LAYOUT_MNGR
  m_oChoSrcName.Create( poPnl, ID_CHO_SRCNAME );
  m_oPnlSrcLvl.bCreate( poPnl, ID_PNL_SRCLVL  );
#else
  m_oSbxSrc    .Create( poPnl, ID_UNUSED, wxT(" Signal Source "),
                        wxPoint(  6, 175 ), wxSize( 330, 55 ) );
  m_oChoSrcName.Create( poPnl, ID_CHO_SRCNAME,
                        wxPoint( 18, 195 ), wxSize( 121, -1 ) );
  m_oPnlSrcLvl.bCreate( poPnl, ID_PNL_SRCLVL, -1, wxPoint( 143, 193 ) );
#endif // LAYOUT_MNGR

  // Set static box font
  m_oSbxSrc.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Create the simulation parameter complex part display objects.

void  PnlAnaBase::CreateCpxPrt( void )
{
  wxPanel * poPnl;

  if( bIsCreated( eCTLGRP_CPXPRT ) ) return;

  // Create the parameter (probe) complex part panel and controls
#ifdef LAYOUT_MNGR
  poPnl = new wxPanel( this );
#else
  poPnl = this; // ???
#endif // LAYOUT_MNGR

  // Create and add simulation parameter complex part check boxes
#ifdef LAYOUT_MNGR
  m_oCbxMag   .Create( poPnl, ID_CBX_MAG  , wxT("Magnitude")  );
  m_oCbxPhase .Create( poPnl, ID_CBX_PHASE, wxT("Phase")      );
  m_oCbxReal  .Create( poPnl, ID_CBX_REAL , wxT("Real Part")  );
  m_oCbxImag  .Create( poPnl, ID_CBX_IMAG , wxT("Imag. Part") );
  m_oCbxMagDb .Create( poPnl, ID_CBX_MAGDB, wxT("Mag. in dB") );
#else
  m_oSbxCpxPrt.Create( poPnl, ID_UNUSED,    wxT(" Complex Parts "),
                       wxPoint( 455,  5 ), wxSize( 110, 146 ) );
  m_oCbxMag   .Create( poPnl, ID_CBX_MAG,   wxT("Magnitude"),
                       wxPoint( 461, 24 ) );
  m_oCbxPhase .Create( poPnl, ID_CBX_PHASE, wxT("Phase"),
                       wxPoint( 461, 49 ) );
  m_oCbxReal  .Create( poPnl, ID_CBX_REAL,  wxT("Real Part"),
                       wxPoint( 461, 74 ) );
  m_oCbxImag  .Create( poPnl, ID_CBX_IMAG,  wxT("Imag. Part"),
                       wxPoint( 461, 99 ) );
  m_oCbxMagDb .Create( poPnl, ID_CBX_MAGDB, wxT("Mag. in dB"),
                       wxPoint( 461,124 ) );
#endif // LAYOUT_MNGR

  // Set static box font
  m_oSbxCpxPrt.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Create the analysis temperature display objects.

void  PnlAnaBase::CreateTemp( void )
{
  wxPanel * poPnl;

  if( bIsCreated( eCTLGRP_TEMP ) ) return;

  // Create the temperature panel and controls
#ifdef LAYOUT_MNGR
  poPnl = new wxPanel( this );
#else
  poPnl = this;
#endif // LAYOUT_MNGR

  // Create and add analysis temperature
#ifdef LAYOUT_MNGR
  m_oPnlTemp.bCreate( poPnl, ID_PNL_TEMP );
#else
  m_oSbxTemp. Create( poPnl, ID_UNUSED, wxT(" Temperature "),
                      wxPoint( 343, 175 ), wxSize( 205, 55 ) );
  m_oPnlTemp.bCreate( poPnl, ID_PNL_TEMP, -1,
                      wxPoint( 355, 193 ) );
#endif // LAYOUT_MNGR
  m_oPnlTemp.bSetUnitsType( eUNITS_TEMP );

  // Set static box font
  m_oSbxTemp.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Test if a certain group of controls has been created.
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlAnaBase::bIsCreated( enum eTypeCtlGrp eCtlGrp )
{
  wxWindow * poCtlGrp;

  if( eCtlGrp<eCTLGRP_FST || eCtlGrp>eCTLGRP_LST ) return( FALSE );

  switch( eCtlGrp )
  {
    case eCTLGRP_BASE   : poCtlGrp = &m_oPnlStart;   break;
    case eCTLGRP_SCALE  : poCtlGrp = &m_oRbxScale;   break;
    case eCTLGRP_INITC  : poCtlGrp = &m_oRbxInitC;   break;
    case eCTLGRP_SIGSRC : poCtlGrp = &m_oChoSrcName; break;
    case eCTLGRP_CPXPRT : poCtlGrp = &m_oCbxMag;     break;
    case eCTLGRP_TEMP   : poCtlGrp = &m_oPnlTemp;    break;
    default             :                          return( FALSE );
  }

  return( poCtlGrp->GetParent( ) != NULL );
}

//*****************************************************************************
// Layout the panel display objects.

void  PnlAnaBase::DoLayout( void )
{
#ifndef LAYOUT_MNGR
  return;
#endif

  wxWindow       * poWin;
  wxSizer        * poSzr;
  wxGridBagSizer * poSzrGB;
  wxSizerFlags     oFlags;
  wxGBPosition     oGBPosn;
  wxGBSpan         oGBSpan;
  int              iFlags;
  int              iBorder;

  //*****************************************************************
  // Create and set the sizer for the sweep panel
  poWin = m_oPnlStart.GetParent( );
  poSzr = new wxStaticBoxSizer( wxVERTICAL, poWin, wxT(" Sweep Parameters ") );
  poWin->SetSizer( poSzr );

  // Layout the sweep panel
  oFlags.Align( wxALIGN_LEFT );
  oFlags.Border( wxALL & ~wxBOTTOM, 3 );
  poSzr->Add( &m_oPnlStart, oFlags );
  poSzr->Add( &m_oPnlStop , oFlags );
  poSzr->Add( &m_oPnlStep , oFlags );
  poSzr->AddSpacer( 3 );
  if( bIsCreated( eCTLGRP_SCALE ) ) poSzr->Add( &m_oRbxScale, oFlags );
  if( bIsCreated( eCTLGRP_INITC ) ) poSzr->Add( &m_oRbxInitC, oFlags );

  //*****************************************************************
  if( bIsCreated( eCTLGRP_SIGSRC ) )
  {
    // Create and set the sizer for the signal source panel
    poWin = m_oChoSrcName.GetParent( );
    poSzr = new wxStaticBoxSizer( wxHORIZONTAL, poWin, wxT(" Signal Source ") );
    poWin->SetSizer( poSzr );

    // Layout the signal source panel
    oFlags.Align( wxALIGN_LEFT );
    oFlags.Border( wxALL, 3 );
    poSzr->Add( &m_oChoSrcName, oFlags );
    poSzr->Add( &m_oPnlSrcLvl , oFlags );
  }

  //*****************************************************************
  // Create and set the sizer for the parameter (probe) panel
  poWin = m_oCbxVoltage.GetParent( );
  poSzr = new wxStaticBoxSizer( wxVERTICAL, poWin, wxT(" Parameters ") );
  poWin->SetSizer( poSzr );

  // Layout the parameter (probe) panel
  oFlags.Align( wxALIGN_LEFT );
  oFlags.Border( wxALL & ~wxBOTTOM, 3 );
  poSzr->Add( &m_oCbxVoltage, oFlags );
  poSzr->Add( &m_oCbxCurrent, oFlags );
  poSzr->Add( &m_oCbxPower  , oFlags );
  poSzr->Add( &m_oCbxResist , oFlags );
  poSzr->AddSpacer( 3 );

  //*****************************************************************
  if( bIsCreated( eCTLGRP_CPXPRT ) )
  {
    // Create and set the sizer for the complex part panel
    poWin = m_oCbxMag.GetParent( );
    poSzr = new wxStaticBoxSizer( wxVERTICAL, poWin, wxT(" Complex Parts ") );
    poWin->SetSizer( poSzr );

    // Layout the complex part panel
    oFlags.Align( wxALIGN_LEFT );
    oFlags.Border( wxALL & ~wxBOTTOM, 3 );
    poSzr->Add( &m_oCbxMag  , oFlags );
    poSzr->Add( &m_oCbxPhase, oFlags );
    poSzr->Add( &m_oCbxReal , oFlags );
    poSzr->Add( &m_oCbxImag , oFlags );
    poSzr->Add( &m_oCbxMagDb, oFlags );
    poSzr->AddSpacer( 3 );
  }

  //*****************************************************************
  if( bIsCreated( eCTLGRP_TEMP ) )
  {
    // Create and set the sizer for the complex part panel
    poWin = m_oPnlTemp.GetParent( );
    poSzr = new wxStaticBoxSizer( wxVERTICAL, poWin, wxT(" Temperature ") );
    poWin->SetSizer( poSzr );

    // Layout the complex part panel
    oFlags.Align( wxALIGN_LEFT );
    oFlags.Border( wxALL, 3 );
    poSzr->Add( &m_oPnlTemp, oFlags );
  }

  //*****************************************************************
  // Create and set the underlying panel's sizer
  poSzrGB = new wxGridBagSizer( 1, 1 );
  SetSizer( poSzrGB );

  // Add the sweep panel
  poWin = m_oPnlStart.GetParent( );
  oGBPosn.SetCol( 0 );     oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 3 ); oGBSpan.SetRowspan( 4 );
  iFlags  = wxALL | wxALIGN_CENTER;
  iBorder = 5;
  poSzrGB->Add( poWin, oGBPosn, oGBSpan, iFlags, iBorder );

  if( bIsCreated( eCTLGRP_SIGSRC ) )
  {
    // Add the signal source panel
    poWin = m_oChoSrcName.GetParent( );
    oGBPosn.SetCol( 0 );     oGBPosn.SetRow( 4 );
    oGBSpan.SetColspan( 3 ); oGBSpan.SetRowspan( 1 );
    iFlags = wxALL | wxALIGN_LEFT;
    poSzrGB->Add( poWin, oGBPosn, oGBSpan, iFlags, iBorder );
  }

  // Add the parameter (probe) panel
  poWin = m_oCbxVoltage.GetParent( );
  oGBPosn.SetCol( 3 );     oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 ); oGBSpan.SetRowspan( 3 );
  poSzrGB->Add( poWin, oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the options button
  oGBPosn.SetCol( 3 );     oGBPosn.SetRow( 3 );
  oGBSpan.SetColspan( 1 ); oGBSpan.SetRowspan( 1 );
  iFlags = wxALL | wxALIGN_CENTER;
  poSzrGB->Add( &m_oBtnOPTIONS, oGBPosn, oGBSpan, iFlags, iBorder );

  if( bIsCreated( eCTLGRP_CPXPRT ) )
  {
    // Add the complex parts panel
    poWin = m_oCbxMag.GetParent( );
    oGBPosn.SetCol( 4 );     oGBPosn.SetRow( 0 );
    oGBSpan.SetColspan( 1 ); oGBSpan.SetRowspan( 4 );
    poSzrGB->Add( poWin, oGBPosn, oGBSpan, iFlags, iBorder );
  }

  if( bIsCreated( eCTLGRP_TEMP ) )
  {
    // Add the temperature panel
    poWin = m_oPnlTemp.GetParent( );
    oGBPosn.SetCol( 3 );     oGBPosn.SetRow( 4 );
    oGBSpan.SetColspan( 2 ); oGBSpan.SetRowspan( 1 );
    poSzrGB->Add( poWin, oGBPosn, oGBSpan, iFlags, iBorder );
  }
}

//*****************************************************************************
// Set the state of the step scale radio box.
//
// Argument List :
//   eScale - The enumerated scale specifier
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlAnaBase::bSetScale( eScaleType eScale )
{
  if( eScale<eSCALE_FST || eScale>eSCALE_LST )      return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxScale.GetCount( ) < (uint) eScale + 1 ) return( FALSE );
#else
  if( m_oRbxScale.GetCount( ) < (int)  eScale + 1 ) return( FALSE );
#endif

  m_oRbxScale.SetSelection( (int) eScale );

  InitScale( );

  return( TRUE );
}

//*****************************************************************************
// Set the state of the initial conditions radio box.
//
// Argument List :
//   eInitC - The enumerated initial conditions specifier
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PnlAnaBase::bSetInitC( eInitCType eInitC )
{
  if( eInitC<eINITC_FST || eInitC>eINITC_LST )      return( FALSE );
#if wxCHECK_VERSION( 2,8,0 )
  if( m_oRbxInitC.GetCount( ) < (uint) eInitC + 1 ) return( FALSE );
#else
  if( m_oRbxInitC.GetCount( ) < (int)  eInitC + 1 ) return( FALSE );
#endif

  m_oRbxInitC.SetSelection( (int) eInitC );

  return( TRUE );
}

//*****************************************************************************
// Load the source choice box with the components names which may be chosen as
// a source.
//
// Argument List :
//   roSimn   - A reference to an array on Component objects
//   osPrefix - A string containing the first letter of the desired components

void  PnlAnaBase::LoadSrcNames( ArrayComponent & roaCpnts, wxString osPrefixes )
{
  wxArrayString  oasSrcs;
  wxString       osTmpSrc;
  wxString       os1;
  wxChar         oc1;
  uint           ui1;

  // Temporarily record the current selection
  osTmpSrc = m_oChoSrcName.GetStringSelection( );

  // Clear the signal source choice box
  m_oChoSrcName.Clear( );
  m_oChoSrcName.Append( wxT("None") );

  // Load the selected components into the signal source choice box
  for( ui1=0; ui1<roaCpnts.GetCount( ); ui1++ )
  {
    os1 = roaCpnts.Item( ui1 ).m_osName;
    if( os1.IsEmpty( ) )                        continue;

    oc1 = os1.GetChar( 0 );
    if( osPrefixes.Find( oc1 ) == wxNOT_FOUND ) continue;

    oasSrcs.Add( os1 );
  }
  oasSrcs.Sort( iStrCmpSrc );
  m_oChoSrcName.Append( oasSrcs );

  // Restore the original selection if possible
  if( ! m_oChoSrcName.SetStringSelection( osTmpSrc ) )
    m_oChoSrcName.SetStringSelection( wxT("None") );
}

//*****************************************************************************
// Set the source component.
//
// Argument List :
//   roCpnt - A component object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetSrcCpnt( Component & roCpnt )
{
  // Can't do anything if the signal source controls haven't been created
  if( ! bIsCreated( eCTLGRP_SIGSRC ) )                         return( FALSE );

  if( ! roCpnt.m_osName.IsEmpty( ) )
  {
    // Set the source component label, value and units
    if( ! m_oChoSrcName.SetStringSelection(roCpnt.m_osName)  ) return( FALSE );
    if( ! m_oPnlSrcLvl.bSetUnitsType(roCpnt.eGetUnitsType()) ) return( FALSE );
    if( ! m_oPnlSrcLvl.bSetValue( roCpnt.rosGetNumValue( ) ) ) return( FALSE );
  }
  else
  {
    // If the components isn't valid set default values
    m_oChoSrcName.SetStringSelection( wxT("None") );
    m_oPnlSrcLvl.bSetValue( (float) 0.0 );
    m_oPnlSrcLvl.bSetUnitsType( eUNITS_NONE );
  }

  return( roCpnt.m_osName == wxT("None") );
}

//*****************************************************************************
// Set the label for the sweep panel.
//
// Argument List :
//   rosLabel - The new panel label

void  PnlAnaBase::SetSwpPnlLbl( const wxString & rosLabel )
{
  wxStaticBoxSizer * poSzr=NULL;

  poSzr = (wxStaticBoxSizer *) m_oPnlStart.GetParent( )->GetSizer( );

  if( poSzr != NULL ) poSzr->GetStaticBox( )->SetLabel( rosLabel );
}

//*****************************************************************************
// Select the simulator engine to be used.
//
// Argument list :
//   eSimEng - The simulator engine type
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetSimrEng( eSimrType eSimEng )
{
  if( eSimEng<eSIMR_FST || eSimEng>eSIMR_LST ) return( FALSE );

  m_eSimEng = eSimEng;

  return( TRUE );
}

//*****************************************************************************
// Select the analysis to be performed.
//
// Argument List :
//   eAnalysis - The analysis type
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlAnaBase::bSetAnaType( eCmdType eAnaType )
{
  if( eAnaType<eCMD_ANA_FST || eAnaType>eCMD_ANA_LST ) return( FALSE );

  m_eAnaType = eAnaType;

  return( TRUE );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Source component choice box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlAnaBase::OnSrcName( wxCommandEvent & roEvtCmd )
{
  // Set the sweep source component
  if( m_oChoSrcName.GetSelection( ) > 0 )
  {
    // Get the sweep source name
    m_oCpntSwpSrcs.bSetName( m_oChoSrcName.GetStringSelection( ) );

    // Get the sweep source value
    if( m_oPnlSrcLvl.GetParent( ) != NULL )
         m_oCpntSwpSrcs.bSetValue( m_oPnlSrcLvl.rosGetValue( ) );
    else m_oCpntSwpSrcs.bSetValue( wxT("1.0") );
  }
  else m_oCpntSwpSrcs.bClear( );
}

//*****************************************************************************
