-- Arbitrary.hs: QuickCheck instances
-- Copyright © 2014  Clint Adams
-- This software is released under the terms of the Expat license.
-- (See the LICENSE file).

module Codec.Encryption.OpenPGP.Arbitrary () where

import Codec.Encryption.OpenPGP.Types
import qualified Data.ByteString as B
import Test.QuickCheck (Arbitrary(..), choose, elements, frequency, getPositive, oneof, vector)
import Test.QuickCheck.Instances ()

instance Arbitrary PKESK where
    arbitrary = do pv <- arbitrary
                   eoki <- arbitrary
                   pka <- arbitrary
                   mpis <- arbitrary
                   return $ PKESK pv eoki pka mpis

instance Arbitrary Signature where
    arbitrary = arbitrary >>= return . Signature

instance Arbitrary UserId where
    arbitrary = arbitrary >>= return . UserId

--

instance Arbitrary SignaturePayload where
    arbitrary = frequency [(2,three),(3,four),(1,other)]
        where
            three = do
                st <- arbitrary
                w32 <- arbitrary
                eoki <- arbitrary
                pka <- arbitrary
                ha <- arbitrary
                w16 <- arbitrary
                mpis <- arbitrary
                return (SigV3 st w32 eoki pka ha w16 mpis)
            four = do
                st <- arbitrary
                pka <- arbitrary
                ha <- arbitrary
                has <- arbitrary
                uhas <- arbitrary
                w16 <- arbitrary
                mpis <- arbitrary
                return (SigV4 st pka ha has uhas w16 mpis)
            other = do
                v <- choose (5, maxBound)
                bs <- arbitrary
                return (SigVOther v bs)

instance Arbitrary SigSubPacket where
    arbitrary = do
        crit <- arbitrary
        pl <- arbitrary
        return (SigSubPacket crit pl)

instance Arbitrary SigSubPacketPayload where
    arbitrary = oneof [sct, set, ec, ts, {- re, -} ket, psa]
        where
            sct = arbitrary >>= return . SigCreationTime
            set = arbitrary >>= return . SigExpirationTime
            ec = arbitrary >>= return . ExportableCertification
            ts = arbitrary >>= \tl -> arbitrary >>= \ta -> return (TrustSignature tl ta)
            re = arbitrary >>= return . RegularExpression -- FIXME: figure out why RegularExpression fails to serialize properly
            ket = arbitrary >>= return . KeyExpirationTime
            psa = arbitrary >>= return . PreferredSymmetricAlgorithms
	    -- FIXME: finish this

--

instance Arbitrary PubKeyAlgorithm where
    arbitrary = elements [RSA, DSA, EC, ECDSA, DH]

instance Arbitrary EightOctetKeyId where
    arbitrary = vector 8 >>= return . EightOctetKeyId . B.pack

instance Arbitrary MPI where
    arbitrary = arbitrary >>= return . MPI . getPositive

instance Arbitrary SigType where
    arbitrary = elements [BinarySig, CanonicalTextSig, StandaloneSig, GenericCert, PersonaCert, CasualCert, PositiveCert, SubkeyBindingSig, PrimaryKeyBindingSig, SignatureDirectlyOnAKey, KeyRevocationSig, SubkeyRevocationSig, CertRevocationSig, TimestampSig, ThirdPartyConfirmationSig]

instance Arbitrary HashAlgorithm where
    arbitrary = elements [DeprecatedMD5, SHA1, RIPEMD160, SHA256, SHA384, SHA512, SHA224]

instance Arbitrary SymmetricAlgorithm where
    arbitrary = elements [Plaintext, IDEA, TripleDES, CAST5, Blowfish, ReservedSAFER, ReservedDES, AES128, AES192, AES256, Twofish]
