/*
 *   ALSA command line mixer utility
 *   Copyright (c) 1999-2000 by Jaroslav Kysela <perex@suse.cz>
 *   Modified for hotkeys by Joey Hess <joeyh@debian.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include <errno.h>
#include <assert.h>
#include <alsa/asoundlib.h>
#include <sys/poll.h>

#include "hotkeys.h"
#include "xmalloc.h"

char *card;
char **controls;
int numcontrols=0;
snd_mixer_t *handle = NULL;

static int parse_simple_id(const char *str, snd_mixer_selem_id_t *sid) {
	int c, size;
	char buf[128];
	char *ptr = buf;

	while (*str == ' ' || *str == '\t')
		str++;
	if (!(*str))
		return -EINVAL;
	size = 1;	/* for '\0' */
	if (*str != '"' && *str != '\'') {
		while (*str && *str != ',') {
			if (size < (int)sizeof(buf)) {
				*ptr++ = *str;
				size++;
			}
			str++;
		}
	} else {
		c = *str++;
		while (*str && *str != c) {
			if (size < (int)sizeof(buf)) {
				*ptr++ = *str;
				size++;
			}
			str++;
		}
		if (*str == c)
			str++;
	}
	if (*str == '\0') {
		snd_mixer_selem_id_set_index(sid, 0);
		*ptr = 0;
		goto _set;
	}
	if (*str != ',')
		return -EINVAL;
	*ptr = 0;	/* terminate the string */
	str++;
	if (!isdigit(*str))
		return -EINVAL;
	snd_mixer_selem_id_set_index(sid, atoi(str));
       _set:
	snd_mixer_selem_id_set_name(sid, buf);
	return 0;
}

snd_mixer_elem_t *getelem(char *control) {
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;
	snd_mixer_selem_id_alloca(&sid);

	if (control == NULL)
		return NULL;

	if (parse_simple_id(control, sid)) {
		uError("Bad ALSA control '%s'", control);
		return NULL;
	}
	elem = snd_mixer_find_selem(handle, sid);
	if (!elem) {
		uError("Unable to find ALSA control '%s',%i", snd_mixer_selem_id_get_name(sid), snd_mixer_selem_id_get_index(sid));
		return NULL;
	}

	return elem;
}

int gethandle(void) {
	int err=0;

	if (handle) {
		snd_mixer_close(handle);
		handle=NULL;
	}
	
	if ((err = snd_mixer_open(&handle, 0)) < 0) {
		uError("Mixer %s open error: %s", card, snd_strerror(err));
		return err;
	}
	if ((err = snd_mixer_attach(handle, card)) < 0) {
		uError("Mixer attach %s error: %s", card, snd_strerror(err));
		snd_mixer_close(handle);
		handle = NULL;
		return err;
	}
	if ((err = snd_mixer_selem_register(handle, NULL, NULL)) < 0) {
		uError("Mixer register error: %s", snd_strerror(err));
		snd_mixer_close(handle);
		handle = NULL;
		return err;
	}
	err = snd_mixer_load(handle);
	if (err < 0) {
		uError("Mixer %s load error: %s", card, snd_strerror(err));
		snd_mixer_close(handle);
		handle = NULL;
		return err;
	}

	return 0;
}

int configALSA(char *alsacard, char *alsacontrols) {
        int i = 0;
	char *c = alsacontrols;

	if (alsacard == NULL || alsacontrols == NULL) {
		return 1;
	}

	card=alsacard;

	/* Split controls into array and parse. */
        do {
		c = strchr( c+1, '|' );
		numcontrols++;
        } while ( c != NULL );
        controls = XMALLOC( char *, numcontrols );
        /* dup needed since strtok modifies the string */
        c = (char*) xstrdup( alsacontrols );
        controls[0] = strtok( c, "|" );
        while ( i < numcontrols ) {
            i++;
            controls[i] = strtok( NULL, "|" );
        }

	return 0; /* success */
}

#define check_range(val, min, max) \
	((val < min) ? (min) : (val > max) ? (max) : (val)) 

/* Fuction to convert from volume to percentage. val = volume */

static int convert_prange(int val, int min, int max) {
	int range = max - min;
	int tmp;

	if (range == 0)
		return 0;
	val -= min;
	tmp = rint((double)val/(double)range * 100);
	return tmp;
}

#define convert_prange1(val, min, max) \
	ceil((val) * ((max) - (min)) * 0.01 + (min))

static int set_volume_simple(snd_mixer_elem_t *elem,
			     snd_mixer_selem_channel_id_t chn,
			     int percent) {
	long val, pmin, pmax;
	int invalid = 0, err = 0;

	if (! snd_mixer_selem_has_playback_volume(elem))
		invalid = 1;

	if (! invalid &&
	    snd_mixer_selem_get_playback_volume_range(elem, &pmin, &pmax) < 0)
		invalid = 1;

	if (! invalid)
		val = (long)convert_prange1(percent, pmin, pmax);
	
	if (! invalid) {
		val = check_range(val, pmin, pmax);
		err = snd_mixer_selem_set_playback_volume(elem, chn, val);
	}
	return err ? err : (invalid ? -ENOENT : 0);
}

/* Pass percent to change mixer, or zero to toggle mute. Returns new
 * mixer volume if changed, or mute status if mute changed, or negative 
 * on error. (Returned values are for the first control only.) */
int amixer(int percent) {
	snd_mixer_selem_channel_id_t chn;
	unsigned int channels = ~0U;
	long pvol;
	long pmin = 0, pmax = 0;
	int i;
	int err=0, ret = -1;
 
	/* Need to reopen the handle every time to detect changes to the
	 * mixer state. This is probably not the best way.. */
	if (gethandle() != 0) {
		err=-1;
	}

	if (! numcontrols) {
		uError("internal error: no controls");
		err=-1;
	}

	int firstcontrol = 1;
	for (i = 0; i < numcontrols; i++ ) {
		snd_mixer_elem_t *elem = getelem(controls[i]);

		if (elem == NULL)
			continue;
		
		int firstchn = 1;
		for (chn = 0; chn <= SND_MIXER_SCHN_LAST; chn++) {
			int ival;
			if (!(channels & (1 << chn)))
				continue;
			if (snd_mixer_selem_has_playback_channel(elem, chn)) {
				if (percent == 0) {
					if (snd_mixer_selem_has_playback_switch(elem) &&
					   (firstchn || !snd_mixer_selem_has_playback_switch_joined(elem))) {
						snd_mixer_selem_get_playback_switch(elem, chn, &ival);
						snd_mixer_selem_set_playback_switch(elem, chn, (ival ? 1 : 0) ^ 1);
						if (firstcontrol) {
							ret = ival;
							firstcontrol=0;
						}
					}
				}
				else {
					int current_percent;
					
					snd_mixer_selem_get_playback_volume_range(elem, &pmin, &pmax);
					snd_mixer_selem_get_playback_volume(elem, chn, &pvol);
					current_percent = convert_prange(pvol, pmin, pmax);

					current_percent += percent;
					if (current_percent > 100)
						current_percent = 100;
					/* 1 not 0 to avoid muting */
					else if (current_percent < 1)
						current_percent = 1;

					if (set_volume_simple(elem, chn, current_percent) > 0) {
						uError("volume set failed");
						err=-1;
					}

					if (firstcontrol) {
						ret = current_percent;
						firstcontrol=0;
					}
				}
			}
			firstchn = 0;
		}
	}

	if (err != 0)
		return err;
	else
		return ret;
}
