/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


/***************************************************************************
   grid-proxy-verify.c

 Sample C program that verifies a Globus GSI proxy.
 The following checks are performed:
 - certificate chain is verified , including proxy certs
 - proxy itself is verified (SUBJ/CN=proxy vs ISSUER etc)
 - proxy private key is matched against proxy public key
 - file permissions of proxy file are checked

 Build instructions:
    gcc -o grid-proxy-verify grid-proxy-verify.c \
        -I<OPENSSL-INCLUDE> -L<OPENSSL-LIB> -lssl -lcrypto
 (code is CFLAGS="-Wall -g -Wuninitialized -O" clean)

 ***************************************************************************/

#define _GNU_SOURCE

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "log.h"
#include "verify_x509.h"


char  *fileName  = NULL;

void print_usage( void );


void print_usage( void )
{
    printf( "grid-proxy-verify\n");
    printf( "Usage:\n" );
    printf( "  grid-proxy-verify [-h|--help] [-d|--debug] [-q||--quiet] [proxy]\n\n" );
    printf( "Repeat -d/--debug multiple times to get more debugging output.\n" );
    printf( "If no proxy is specified then %s is used.\n", fileName );

    exit(0);
}


int main( int argc, char **argv )
{
    char             *CA_dir = NULL;

    int               i = 0;
    unsigned long     result = 0;
    char             *long_opt;
    struct stat       my_stat;
    int               log_lvl = 0;

    internal_verify_x509_data_t * verify_data;


    fileName = getenv( "X509_USER_PROXY" );
    if ( fileName == NULL )
    {
        fileName = calloc( 255, sizeof( char ) );
        snprintf( fileName, 255, "/tmp/x509up_u%d", getuid() );
    }

    for (i = 1; i < argc; i++)
    {
        if ( (strlen(argv[i]) >= 2 ) && ( argv[i][0] == '-' ) )
        {
            switch (argv[i][1])
            {
                case '-': long_opt = argv[i]+2;
                          if ( strcmp( long_opt, "help" ) == 0 )
                              print_usage();
                          else if ( strcmp( long_opt, "debug") == 0 )
                          {
                              log_lvl = get_log_level();
                              log_lvl++;
                              set_log_level (log_lvl);
                          }
                          else if ( strcmp( long_opt, "quiet") == 0 )
                              set_log_level (0);
                          else
                          {
                              fprintf( stderr, "Unknown option: %s\n", argv[i] );
                              print_usage();
                          }
                          break;
                case 'h': print_usage();
                          break;
                case 'd': 
                          log_lvl = get_log_level();
                          log_lvl++;
                          set_log_level (log_lvl);
                          break;
                case 'q':
                          set_log_level (0);
                          break;
                default:  fprintf( stderr, "Unknown option: %s\n", argv[i] );
                          print_usage();
            }
        }
        else
        {
            fileName = argv[i];
        }
    }


    /* First, find the trusted CA directory */
    CA_dir = getenv( "X509_CERT_DIR" );
    if ( CA_dir == NULL ) CA_dir = "/etc/grid-security/certificates/";

    Log ( L_DEBUG, "Testing CA directory %s", CA_dir );

    /* Check the file permissions on the proxy */
    Log( L_INFO, "Checking file permissions for %s", fileName );

    stat( fileName, &my_stat );
    if ( my_stat.st_mode & (S_IRWXG | S_IRWXO ) )
    {
        Error( "Checking file permissions",
               "should be 0600, are currently %04o.",
                my_stat.st_mode & 0xFFF);
    }


    /* The fileName is the proxy file and shown (above tests) to be accessible, including the CA_dir.
     * The CA_dir is used for the CA, sub-CA and CRL files.
     * Note: must implement check that make use of the CA namespace files
     */
    
    result = verify_X509_init(&verify_data);

    result = verify_X509_setParameter (&verify_data, VERIFY_X509_CERTIFICATE_FILEPATH, fileName);
    if (result == ERR_VERIFY_X509_PARAMS_ALREADY_SET)
        printf ("VERIFY_X509_CERTIFICATE_FILEPATH already set...\n");
    else if (result == ERR_VERIFY_X509_PARAMS_ACCESS_FAILURE)
    {
        verify_X509_term(&verify_data);
        return result;
    }
    
    result = verify_X509_setParameter (&verify_data, VERIFY_X509_CA_PATH, CA_dir);
    if (result == ERR_VERIFY_X509_PARAMS_ALREADY_SET)
        printf ("VERIFY_X509_CA_PATH already set...\n");
    else if (result == ERR_VERIFY_X509_PARAMS_ACCESS_FAILURE)
    {
        verify_X509_term(&verify_data);
        return result;
    }
    
    
    /* result = startVerifyProcess (fileName, CA_dir); */
    result = verify_X509_verify(&verify_data);
    
    verify_X509_term(&verify_data);
    return ( !( result == X509_V_OK ) );
}

