/***********************************************************************************

    Copyright (C) 2007-2020 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_BASE_HEADER
#define LIFEOGRAPH_BASE_HEADER


#ifdef _WIN32
#include <winsock2.h> // to silence warnings on Windows
#endif

#include <gtkmm.h>

extern "C"
{
#include <enchant.h>
}

#include "settings.hpp"
#include "diary.hpp"


//#define LIFEOGRAPH_BETA_STAGE


namespace LIFEO
{
    static const char   PROGRAM_NAME[]      = "Lifeograph";

    static const int    BACKUP_INTERVAL     = 30;    // seconds
    static const int    LOGOUT_COUNTDOWN    = 10;    // seconds

    static const char   TE_ENTRY_ROW[]      = "GTK_TREE_MODEL_ROW_ENTRY";
    static const char   TE_THEME[]          = "Lifeograph/theme";
    static const char   TE_TEXT_PLAIN[]     = "text/plain;charset=utf-8";

struct Icons
{
    Icon                            diary_32;
    Icon                            entry_16;
    Icon                            entry_32;
    Icon                            entry_parent_16;
    Icon                            entry_parent_32;
    Icon                            chapter_16;
    Icon                            chapter_32;
    Icon                            tag_16;
    Icon                            tag_32;
    Icon                            favorite_16;
    Icon                            filter_16;
    Icon                            todo_auto_16;
    Icon                            todo_auto_32;
    Icon                            todo_open_16;
    Icon                            todo_open_32;
    Icon                            todo_progressed_16;
    Icon                            todo_progressed_32;
    Icon                            todo_done_16;
    Icon                            todo_done_32;
    Icon                            todo_canceled_16;
    Icon                            todo_canceled_32;
};

enum DRAG_TYPE
{
    DT_ENTRY_ROW, DT_TEXT
};

class AppWindow; // forward declaration

// Lifeograph Application
class Lifeograph : public Gtk::Application
{
    public:
        static const char           PROGRAM_VERSION_STRING[];

        static Glib::RefPtr< Lifeograph >
                                    create();

        bool                        quit();

        virtual                     ~Lifeograph();

        static void                 init_paths();

        static void                 START_INTERNAL_OPERATIONS()
        { s_internal_operation_depth++; }
        static void                 FINISH_INTERNAL_OPERATIONS()
        { s_internal_operation_depth--; }
        static bool                 is_internal_operations_ongoing()
        { return( s_internal_operation_depth > 0 ); }

        static void                 update_gtk_theme()
        {
            Gtk::Settings::get_default()->property_gtk_application_prefer_dark_theme() =
                settings.use_dark_theme;
        }

        static Glib::RefPtr< Gtk::Builder >&
                                    get_builder()
        { return s_builder; }
        static Glib::RefPtr< Gtk::Builder >&
                                    get_builder2();

        static std::string          get_icon_dir();
        static Icon&                get_todo_icon( ElemStatus );
        static Icon&                get_todo_icon32( ElemStatus );

        static void                 show_manual();

        // DRAG & DROP
        static const Gtk::TargetEntry
                                    DTE_entry_row;
        static const Gtk::TargetEntry
                                    DTE_theme;
        static const Gtk::TargetEntry
                                    DTE_text_plain;
        static DiaryElement*        get_dragged_elem()
        { return s_elem_dragged; }
        static void                 set_dragged_elem( DiaryElement* elem )
        { s_elem_dragged = elem; }

        // COLORS
        static void                 set_color_insensitive( const std::string& color )
        { s_color_insensitive = color; }
        static const std::string&   get_color_insensitive()
        { return s_color_insensitive; }

        // ENCHANT
        static EnchantBroker*       s_enchant_broker;
        static SetStrings           s_lang_list;

        static bool                 load_gui( Glib::RefPtr< Gtk::Builder >&, const std::string& );

        static SignalVoid           signal_logout()
        { return s_signal_logout; }

        // CONFIGURATION
        static std::string          BINPATH;
        static std::string          SHAREDIR;
        static std::string          MANUALPATH;
        static std::string          EXAMPLEDIARYPATH;
        static std::string          LOCALEDIR;

        // OBJECTS
        static Lifeograph*          p; // static pointer to itself

        static Settings             settings;
        static Icons*               icons;

    protected:
                                    Lifeograph();

        static void                 fill_lang_list_cb( const char* const l, const char* const,
                                                       const char* const, const char* const,
                                                       void* )
        { s_lang_list.insert( l ); }

        static bool                 load_gui( const std::string& path )
        { return load_gui( s_builder, path ); }

        void                        on_startup();
        void                        on_activate();
        int                         on_command_line(
                                        const Glib::RefPtr< Gio::ApplicationCommandLine >& );

        static DiaryElement*        s_elem_dragged;

        static std::string          s_color_insensitive;

        static volatile int         s_internal_operation_depth;

        static SignalVoid           s_signal_logout;

        // BUILDER
        static Glib::RefPtr< Gtk::Builder >
                                    s_builder;
        static bool                 m_flag_ui2_loaded;

        // APPLICATION MENU
        //Glib::RefPtr< Gio::Menu >   m_menu;
};

} // end of namespace LIFEO

#endif

