
/*
  CoreLinux++ 
  Copyright (C) 2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__DOOR_HPP)
#include <Door.hpp>
#endif

#if   !defined(__ROOM_HPP)
#include <Room.hpp>
#endif

using namespace corelinux;

//
// Constructor door defaults to open
//

Door::Door( RoomPtr aFirstRoom, RoomPtr aSecondRoom )
   :
   theFirstRoom( aFirstRoom ),
   theSecondRoom( aSecondRoom ),
   theDoorState( DOOROPEN )
{
   ; // do nothing
}

//
// Copy Constructor
//

Door::Door( DoorCref aDoor )
   :
   theFirstRoom( aDoor.getFirstRoom() ),
   theSecondRoom( aDoor.getSecondRoom() ),
   theDoorState( aDoor.isOpen() ? DOOROPEN : DOORCLOSED )
{
   ;  // do nothing
}

//
// Destructor
//

Door::~Door( void )
{
   theFirstRoom = theSecondRoom = NULLPTR;
}

//
// Assignment operator
//

DoorRef  Door::operator=( DoorCref aRef ) throw( Assertion )
{
   if( *this == aRef )
   {
      ;  // do nothing
   }
   else
   {
      setFirstRoom(aRef.getFirstRoom());
      setSecondRoom(aRef.getSecondRoom());
      theDoorState =  aRef.isOpen() ? DOOROPEN : DOORCLOSED ;
   }

   return (*this);
}

/// Equality operator

bool  Door::operator==( DoorCref aRef ) const
{
   return ( theFirstRoom == aRef.getFirstRoom() &&
            theSecondRoom == aRef.getSecondRoom() );
}


// Is door open - true is yes

bool  Door::isOpen( void ) const
{
   return (theDoorState == DOOROPEN);
}

// Is door closed - true is yes

bool  Door::isClosed( void ) const
{
   return (theDoorState == DOORCLOSED);
}

// Get the first room

RoomPtr  Door::getFirstRoom( void ) const
{
   return theFirstRoom;
}

// Get the second room

RoomPtr  Door::getSecondRoom( void ) const
{
   return theSecondRoom;
}

// Get the opposite room from the argument

RoomPtr  Door::otherSideFrom( RoomPtr aPtr )  const throw( Assertion )
{
   REQUIRE( aPtr != NULLPTR );

   return ( aPtr == getFirstRoom() ? 
            getSecondRoom() : getFirstRoom() );
}


// Opens door if closed

void  Door::setOpen( void )
{
  theDoorState = DOOROPEN;
}

// Closes door if open

void  Door::setClosed( void )
{
   theDoorState = DOORCLOSED;
}

// Set the first room

void  Door::setFirstRoom( RoomPtr aPtr ) throw( Assertion )
{
   REQUIRE( aPtr != NULLPTR );
   theFirstRoom = aPtr;
}

// Set the second room

void  Door::setSecondRoom( RoomPtr aPtr ) throw( Assertion )
{
   REQUIRE( aPtr != NULLPTR );
   theSecondRoom = aPtr;
}


//
// Pure virtual, I don't know what to do as I enter
// a door? Do I check if it is open?
//
   
void Door::enter( void )
{
   ;  //  do nothing
}

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.1 $
   $Date: 2000/04/21 02:38:47 $
   $Locker:  $
*/


