/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2011 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2011 Sun Microsystems, Inc.
 */
package org.netbeans.modules.j2ee.weblogic9.ui.nodes;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.netbeans.modules.j2ee.weblogic9.WLConnectionSupport;
import org.netbeans.modules.j2ee.weblogic9.deploy.WLDeploymentManager;
import org.netbeans.modules.j2ee.weblogic9.ui.nodes.JdbcChildrenFactory.JDBCDataBean;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.Lookup;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 *
 * @author Petr Hejl
 */
public class WebProfileJdbRetriever implements JdbcChildrenFactory.Retriever {

    private static final Logger LOGGER = Logger.getLogger(WebProfileJdbRetriever.class.getName());

    private static final Integer JDBC = Integer.valueOf(4);

    private final AtomicReference<List<JDBCDataBean>> data = new AtomicReference<List<JDBCDataBean>>();

    private final Lookup lookup;

    private boolean isRetrieveStarted;

    public WebProfileJdbRetriever(Lookup lookup) {
        this.lookup = lookup;
    }

    @Override
    public void clean() {
        data.set(null);
    }

    @Override
    public List<JDBCDataBean> get() {
        return data.get();
    }

    @Override
    public void retrieve() {
        synchronized (this) {
            if (isRetrieveStarted) {
                return;
            }
            isRetrieveStarted = true;
        }

        data.set(null);

        WLDeploymentManager manager = lookup.lookup(WLDeploymentManager.class);

        WLConnectionSupport support = new WLConnectionSupport(manager);
        List<JDBCDataBean> list = Collections.emptyList();

        try {
            list = support.executeAction(new WLConnectionSupport.
                    JMXAction<List<JDBCDataBean>>() {

                @Override
                public List<JDBCDataBean> call(MBeanServerConnection con) throws Exception {
                    List<JDBCDataBean> list = new LinkedList<JDBCDataBean>();

                    Set<ObjectName> servers = con.queryNames(new ObjectName(
                            "com.bea:Name=*,Type=ServerRuntime"), null); // NOI18N
                    Set<String> adminNames = new HashSet<String>();
                    for (ObjectName adminServer : servers) {
                        adminNames.add(con.getAttribute(adminServer, "Name").toString()); // NOI18N
                    }

                    Set<ObjectName> domain = con.queryNames(new ObjectName(
                            "com.bea:Name=*,Type=Domain"), null); // NOI18N
                    if (domain.isEmpty()) {
                        return Collections.emptyList();
                    }
                    ObjectName config = domain.iterator().next();

                    findSystemJdbc(con, list, adminNames, config);
                    findDeployedJdbc( con, list , adminNames, config);
                    return list;
                }

                @Override
                public String getPath() {
                    return ""; // NOI18N
                }

                private void findDeployedJdbc( MBeanServerConnection con,
                        List<JDBCDataBean> list, Set<String> adminNames,
                        ObjectName config ) throws MBeanException,
                    AttributeNotFoundException, InstanceNotFoundException,
                    ReflectionException, IOException, MalformedObjectNameException {

                    ObjectName applications[] = (ObjectName[]) con.getAttribute(
                            config, "Applications"); // NOI18N
                    for (ObjectName application : applications) {
                        Integer type = (Integer) con.getAttribute( application, "InternalType"); // NOI18N
                        if (JDBC.equals(type)) {
                            // TODO should we check servers on web profile ?
                            String path = (String) con.getAttribute( application,
                                    "FullPath"); // NOI18N
                            String name = (String) con.getAttribute( application,
                                    "Name"); // NOI18N
                            if (path != null) {
                                loadDeployedDataSource( path , list , name );
                            }
                        }
                    }

                }

                private void findSystemJdbc(MBeanServerConnection con, List<JDBCDataBean> list,
                        Set<String> adminNames, ObjectName objectName) throws MBeanException,
                    AttributeNotFoundException, InstanceNotFoundException,
                    ReflectionException, IOException {
                    // TODO implement fo DWP
                }

            });
        } catch (Exception e) {
            LOGGER.log(Level.INFO, null, e);
        } finally {
            data.compareAndSet(null, list);
            synchronized (this) {
                isRetrieveStarted = false;
                notifyAll();
            }
        }
    }

    @Override
    public void waitForCompletion() {
        synchronized (this) {
            while (isRetrieveStarted) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    LOGGER.log(Level.FINE, null, e);
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    private void loadDeployedDataSource(String path, List<JDBCDataBean> list,
            String deplName) {
        try {
            if (!path.endsWith(".xml")) { // NOI18N
                return;
            }

            SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
            JdbcConfigHandler handler = new JdbcConfigHandler();
            FileObject jdbcConfig = FileUtil.toFileObject( FileUtil.
                    normalizeFile( new File(path)));
            parser.parse(new BufferedInputStream(
                    jdbcConfig.getInputStream()), handler);
            List<String> jndiNames = handler.getJndiNames();
            list.add( new JDBCDataBean( handler.getName(),
                    jndiNames.toArray(new String[jndiNames.size()]), deplName));
        } catch (ParserConfigurationException e) {
            LOGGER.log(Level.INFO, null, e);
        } catch (SAXException e) {
            LOGGER.log(Level.INFO, null, e);
        } catch (FileNotFoundException e) {
            LOGGER.log(Level.INFO, null, e);
        } catch (IOException e) {
            LOGGER.log(Level.INFO, null, e);
        }
    }

    private static class JdbcConfigHandler extends DefaultHandler {

        private static final String DATA_SOURCE_PARAMS = "jdbc-data-source-params"; // NOI18N

        private final List<String> jndiNames = new LinkedList<String>();

        private String content;

        private String name;

        private boolean dataSourceParamsStarted;

        @Override
        public void startElement(String uri, String localName, String qName,
                Attributes attributes) throws SAXException {
            content = null;
            if (DATA_SOURCE_PARAMS.equals(getUnprefixedName(qName))) {
                dataSourceParamsStarted = true;
            }
        }

        @Override
        public void endElement(String uri, String localName, String qName)
                throws SAXException {
            if (name == null && "name".equals(getUnprefixedName(qName))) { // NOI18N
                name = content;
            } else if (DATA_SOURCE_PARAMS.equals(getUnprefixedName(qName))) { // NOI18N
                dataSourceParamsStarted = false;
            } else if ( dataSourceParamsStarted
                    && "jndi-name".equals(getUnprefixedName(qName))) { // NOI18N
                jndiNames.add( content );
            }
        }

        @Override
        public void characters(char[] ch, int start, int length) {
            content = new String(ch, start, length);
        }

        String getName() {
            return name;
        }

        List<String> getJndiNames() {
            return jndiNames;
        }

        private String getUnprefixedName(String name) {
            if (name.contains(":")) { // NOI18N
                return name.substring(name.indexOf(":") + 1); // NOI18N
            } else {
                return name;
            }
        }
    }
}
