'use strict';

var isEqual = require('../internal/isequal');

/**
 * Creates a new object recording the production of the specified value at the given virtual time.
 *
 * @constructor
 * @param {Number} time Virtual time the value was produced on.
 * @param {Mixed} value Value that was produced.
 * @param {Function} comparer An optional comparer.
 */
function Recorded (time, value, comparer) {
  this.time = time;
  this.value = value;
  this.comparer = comparer || isEqual;
}

/**
 * Checks whether the given recorded object is equal to the current instance.
 *
 * @param {Recorded} other Recorded object to check for equality.
 * @returns {Boolean} true if both objects are equal; false otherwise.
 */
Recorded.prototype.equals = function (other) {
  return this.time === other.time && this.comparer(this.value, other.value);
};

/**
 * Returns a string representation of the current Recorded value.
 *
 * @returns {String} String representation of the current Recorded value.
 */
Recorded.prototype.toString = function () {
  return this.value.toString() + '@' + this.time;
};

module.exports = Recorded;
