/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import "Constants.js" as Const

ListView {
    id: noteList
    spacing: units.gu(2)

    currentIndex: -1

    delegate: NoteItem {
        id: noteDelegate
        width: noteList.width
        expandedHeight: units.gu(24)
        collapsedHeight: units.gu(11)

        noteText: note
        dateText: date
        focus: isExpanded
        isExpanded: noteList.currentIndex === index

        onNoteDeleted: dataModel.deleteNote(index) // after deleting, the NoteItem should be removed
        onActivated: noteList.setActiveIndex(index);
        onDeactivated: noteList.setActiveIndex(-1);

        /* After editing an expanded note we used to collapse it and save it at the same time, and this
           was causing the note to be moved at the start of the list. The two animations (move and collapse)
           were happening at the same time.
           However with ListView the move animation shapshots the height of the items at the start, so it is
           not possible to do the animations at the same time or the items will end up in the wrong place.
           So for now we start saving the note and thus starting the move animation only when the collapsed
           animation has finished */
        onCollapsed: noteList.commitNote(index, true);

        Connections {
            target: Qt.application
            onActiveChanged: {
                if (!Qt.application.active && noteList.currentIndex == index)
                    noteList.commitNote(index, true);
            }
        }

        property int itemIndex: index
    }

    move: Transition {
        NumberAnimation { property: "y"; duration: Const.animationDuration }
    }

    displaced: Transition {
        NumberAnimation { property: "y"; duration: Const.animationDuration }
    }

    // Given an index makes the corresponding note the selected one.
    // Any negative index will deselect any current note and not select any other
    function setActiveIndex(newIndex) {
        if (newIndex < 0) {
            noteList.currentIndex = -1;
        } else {
            noteList.currentIndex = newIndex;
        }
    }

    // Find actual note item from its index, returns the note if
    // found, null otherwise. It might be that the delegate does not
    // exist at the moment since it's out of view and not created yet.
    function getNoteAtIndex(noteIndex) {
        for (var i = 0; i < noteList.contentItem.children.length; i++) {
            var current = noteList.contentItem.children[i];
            if (current.itemIndex === noteIndex) {
                return current;
            }
        }
        return null;
    }

    // Save to the database the note if dirty or delete it completely if
    // it's totally empty (and the caller is ok to have it deleted).
    // Return the text of the note that was saved.
    function commitNote(index, deleteIfEmpty) {
        var note = noteList.getNoteAtIndex(index);
        if (note) {
            var theData = note.readNoteData(note);
            if (theData.length === 0) {
                if (deleteIfEmpty) {
                    dataModel.deleteNote(note.itemIndex);
                }
            } else if (note.dirty) {
                note.dirty = false;
                dataModel.touchNote(note.itemIndex, theData);
            }
            return theData;
        } else return null;
    }
}
