///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file CylinderCreationMode.h 
 * \brief Contains the definition of the StdObjects::CylinderCreationMode class. 
 */

#include <core/Core.h>
#include <core/scene/objects/CreationMode.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/snapping/SnappingManager.h>

#include <stdobjects/StdObjects.h>
#include "CylinderObject.h"

namespace StdObjects {

/**
 * \brief This input mode lets the user creat a procedural CylinderObject in the viewports.
 * 
 * \author Alexander Stukowski
 */
class CylinderCreationMode : public SimpleCreationMode
{
public:
	
	/// \brief Default constructor.
	CylinderCreationMode() : SimpleCreationMode(tr("Create Cylinder"), tr("Cylinder")) {}
	
protected:

	/// \brief This creates the actual scene object.
	/// \return The scene object to be inserted into the scene.
	virtual SceneObject::SmartPtr createObject() {
		return new CylinderObject();
	}

	/// \brief Will be called when the user presses the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMousePressed(QMouseEvent* event) {
		SimpleCreationMode::onMousePressed(event);

		if(clickCount() == 1) {
			// Take click point as base point.
			basePointScreen = radiusPointScreen = event->pos();
			basePoint = ORIGIN;
			if(!viewport()->snapPoint(basePointScreen, basePoint))
				onAbort();
			else
				onMouseDrag(event);	// This moves the new object to the current click point.
		}
		else if(clickCount() == 2) {
			onFinish();
		}
	}

	/// \brief Will be called when the user releases the left mouse button.
	/// \param event The mouse event to be handled.
	virtual void onMouseReleased(QMouseEvent* event) {
		SimpleCreationMode::onMouseReleased(event);
		if(clickCount() == 2) {
			if(basePointScreen != radiusPointScreen) {
				beginAdjustOperation();
				SNAPPING_MANAGER.clearLastSnapPoint();
			}
			else
				onAbort();
		}
	}

	/// \brief Will be called when the user moves the mouse while the operation is active.
	/// \param event The mouse event to be handled.
	virtual void onMouseDrag(QMouseEvent* event) {
		SimpleCreationMode::onMouseDrag(event);
		if(clickCount() == 1) {
			// Take mouse position as second box corner.
			radiusPointScreen = event->pos();
			radiusPoint = ORIGIN;
			if(!viewport()->snapPoint(radiusPointScreen, radiusPoint))
				return;
			
			abortAdjustOperation();
			beginAdjustOperation();

			// Position scene node.
			AffineTransformation tm = viewport()->grid().gridMatrix() * AffineTransformation::translation(basePoint - ORIGIN);			
			objectNode()->transformationController()->setValue(0, tm);

			// Resize cylinder.
			static_object_cast<CylinderObject>(object())->radiusController()->setValue(0, Distance(radiusPoint, basePoint));
		}
		else {
			abortAdjustOperation();
			beginAdjustOperation();

			FloatType h = viewport()->grid().computeConstructionLength(Ray3(radiusPoint, Vector3(0,0,1)), radiusPointScreen, event->pos());
			static_object_cast<CylinderObject>(object())->heightController()->setValue(0, h);
		}

		// Update viewports immediately.
		VIEWPORT_MANAGER.processViewportUpdates();
	}

private:

	/// The base point of the cylinder in construction plane coordinates.
    Point3 basePoint;

	/// The screen coordinates of the base point.
	QPoint basePointScreen;

	/// The second point on the radius of the cylinder in construction plane coordinates.
	Point3 radiusPoint;

	/// The screen coordinates of the second point.
	QPoint radiusPointScreen;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CylinderCreationMode)
};

};
