/*	Query_DB

PIRL CVS  ID: Query_DB.java,v 3.9 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2008  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;

import java.util.Vector;
import java.util.Iterator;

/** <i>Query_DB</i> sends a SQL query to a Database and lists the results.
<p>
	The Database server that is queried may be any for which the Database
	can obtain a Data_Port. Access information for the Database is provided
	in a Configuration file, which may be supplemented by command line
	options.
<p>
    Database queries use standard SQL syntax. Depending on the specific
	database server that is accessed the syntax may have limitations and/or
	extensions.
<p>
	The command line {@link #Usage() usage} description provides details
	on the application operation.
<p>
	@author     Bradford Castalia, Sean Whitsitt - UA/PIRL
	@version    3.9
	@see        PIRL.Database
	@see        PIRL.Configuration
*/
public class Query_DB
{
/**	The Class identification with revision number and date.
*/
public static final String
	ID = "PIRL.Database.Query_DB (3.9 2012/04/16 06:08:57)";


/**	The default Configuration filename.
*/
public static final String
	DEFAULT_CONFIGURATION_FILENAME
		= Database.DEFAULT_CONFIGURATION_FILENAME;

private static boolean
	Verbose								= false;


/** Prefix for the listing line that contains the SQL expression sent to
	the database server.
*/
public static final String
	SQL_QUERY_PREFIX					= ">>> ";

/**	Prefix for the listing line that indicates a failed SQL query.
*/
public static final String
	SQL_QUERY_FAILED					= "!!! Query failed.";

/**	Prefix for the data table field names line.
*/
public static final String
	FIELD_NAMES_PREFIX					= "=== ";

/**	Delimiter between a data record number and its field values.
*/
public static final String
	DATA_RECORD_DELIMITER				= " - ";

/**	Success exit status.
*/
public static final int
	EXIT_SUCCESS						= 0;

/**	Exit status when invalid command line syntax was encounterd.
*/
public static final int
	EXIT_INVALID_COMMAND_LINE_SYNTAX	= 1;

/**	Application exit status when there is a Configuration file problem.
*/
public static final int
	EXIT_CONFIGURATION_PROBLEM			= 2;

/**	Application exit status when there is a problem accessing the Database.
*/
public static final int
	EXIT_DATABASE_ERROR					= 3;

/**	Application exit status when one or more Database queries failed.
*/
public static final int
	EXIT_QUERY_FAILURE					= 4;


/**	System new line sequence.
*/
private static final String
	NL									= Database.NL;

/*==============================================================================
    Application main
*/
/** Run the Query_DB application.
<p>
    Exit status values:
<p>
<dl>
<dt>{@link #EXIT_SUCCESS}
<dd>All queries succeeded.

<dt>{@link #EXIT_INVALID_COMMAND_LINE_SYNTAX}
<dd>The command line contains invalid syntax.

<dt>{@link #EXIT_CONFIGURATION_PROBLEM}
<dd>There was a problem with the configuration source. Either it couldn't
	be found or its contents are not valid PVL.

<dt>{@link #EXIT_DATABASE_ERROR}
<dd>A connection with the database server could not be established. This
	could be due to incorrect access information in the configuration.

<dt>{@link #EXIT_QUERY_FAILURE}
<dd>One or more queries failed. If multiple queries were specified some
	may have succeeded.
</dl>
<p>
	@param arguments	The {@link #Usage() command line} arguments.
*/
public static void main
	(
	String[]	arguments
	) 
{
String
	configuration_source = null,
	server_name = null,
	catalog_name = null;
Vector<String>
	queries = new Vector<String> ();
boolean
	verbose = false;

if (arguments.length == 0)
	Usage ();

for (int
		count = 0;
		count < arguments.length;
		count++)
	{
	if (arguments[count].length () > 1 &&
		arguments[count].charAt (0) == '-')
		{
		switch (arguments[count].toUpperCase ().charAt (1))
			{
            case 'C':
				if (arguments[count].length () > 2 &&
					arguments[count].toUpperCase ().charAt (2) == 'A')
					{
					//	-CAtalog name
    		    	if (++count == arguments.length ||
        	        	arguments[count].length () == 0 ||
                    	arguments[count].charAt (0) == '-')
						{
                    	System.err.println
							("Missing catalog name for "
								+ arguments[--count] + " argument.");
                    	Usage ();
        				}
					if (catalog_name != null)
						{
                    	System.err.println
							("Multiple catalog names -" + NL
                    		+ catalog_name + NL
                    		+ "and" + NL
                    		+ arguments[count]);
                    	Usage ();
                		}
                	catalog_name = arguments[count];
					break;
					}

				// -Configuration name
    		    if (++count == arguments.length ||
        	        arguments[count].length () == 0 ||
                    arguments[count].charAt (0) == '-')
					{
                    System.err.println
						("Missing configuration source name for "
							+ arguments[--count] + " argument.");
                    Usage ();
        			}
				if (configuration_source != null)
					{
                    System.err.println
						("Multiple configuration files -" + NL
                    	+ configuration_source + NL
                    	+ "and" + NL
                    	+ arguments[count]);
                    Usage ();
                	}
                configuration_source = arguments[count];
				break;

			case 'S':	//	-Server name
				if (++count == arguments.length ||
					arguments[count].length () == 0 ||
					arguments[count].charAt (0) == '-')
					{
					System.err.println
						("Missing server name for " + arguments[--count]
							+ "argument.");
					Usage ();
					}
				if (server_name != null)
					{
					System.err.println
						("Multiple server names -" + NL
						+ server_name + NL
						+ "and" + NL
						+ arguments[count]);
					Usage ();
					}
				server_name = arguments[count];
				break;

			case 'Q':	//	-Query SQL
				if (++count == arguments.length ||
				    arguments[count].length() == 0 ||
				    arguments[count].charAt(0) == '-')
				    {
				    System.err.println
						("Missing query for "
							+ arguments[--count] + " argument.");
				    Usage ();
				    }
				queries.add (arguments[count]);
				break;

			case 'V':	//	-Verbose
				verbose = true;
				break;

			default:
				System.err.println
					("Unknown option: " + arguments[count]);

			case 'H':	//	Help.
				Usage ();
			}
		}
	else
		queries.add (arguments[count]);
	}

if (queries.isEmpty ())
	{
    System.err.println
		("No query was specified.");
    Usage ();
	}

//	Configuration.
if (configuration_source == null)
	configuration_source = DEFAULT_CONFIGURATION_FILENAME;
Configuration
	configuration = null;
try {configuration = new Configuration (configuration_source);}
catch (Configuration_Exception exception)
	{
	System.err.println
		("A configuration could not be created" + NL
		+"from " + configuration_source + NL
		+ exception.getMessage ());
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}
configuration.Case_Sensitive (false);

//	Database.
Database
	database = null;
try
	{
	database = new Database (configuration);
	database.Connect (server_name, catalog_name);
	}
catch (Database_Exception exception)
	{
	System.err.println
		("A database connection could not be established." + NL
		+ exception.getMessage ());
	System.exit (EXIT_DATABASE_ERROR);
	}
catch (Configuration_Exception exception)
	{
	System.err.println
		("There was a problem with the configuration"
			+ " while connecting to the database" + NL
		+ exception.getMessage ());
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}

if (verbose)
	database.Add_SQL_Listener (new SQL_Listener ()
		{public void SQL_Statement (String SQL)
		{System.out.println (SQL_QUERY_PREFIX + SQL);}});

//	Queries.
Vector<Vector<String>>
	table;
int
	status = EXIT_SUCCESS,
	index = -1,
	size = queries.size ();
while (++index < size)
	{
	if (index != 0)
		System.out.println ();
	try
		{
		table = database.Query (queries.get (index));
		Print_Table (table, true, true);
		}
	catch (Database_Exception exception)
		{
		status = EXIT_QUERY_FAILURE;
		if (verbose)
			System.out.println (SQL_QUERY_FAILED);
		System.err.println (exception.getMessage ());
		}
	}

try {database.Disconnect ();}
catch (Database_Exception exception)
	{
	System.err.println
		("There was a problem while disconnecting from the database." + NL
		+ exception.getMessage ());
	}

System.exit (status);
}

/*==============================================================================
    Utilities
*/
/** Print a table of field records.
<p>
	Each record in the table is listed as fields separated by a single
	horizontal tab character. Each record listed is terminated by a
	new-line sequence defined by the "line.separator" {@link
	System#getProperty(String) System property}.
<p>
	<b>N.B.</b>: While data tables are typically rectangular, no
	presumption is made that each record will have the same number of
	field values.
<p>
	If the prefix argument is true each line of the table listing will
	contain a prefix that identifies the table record. If the
	first_record_field_names argument is also true the prefix for the
	first record will be the {@link #FIELD_NAMES_PREFIX}. All other records
	will have as their prefix the record number followed by the {@link
	#DATA_RECORD_DELIMITER}.
<p>
	@param  table   A Vector of Vector of Strings. If null nothing is
		done.
	@param	prefix	If true, each record listed will have an indentifying
		prefix; if false, no prefix will be listed.
	@param	first_record_field_names	If true, the first table record
		is taken to contain the column names of the field values that
		will be in the remaining records; if false, all table records
		are taken to contain field values.
*/
public static void Print_Table
	(
	Vector<Vector<String>>	table,
	boolean					prefix,
	boolean					first_record_field_names
	)
{
if (table == null)
	return;

Vector<String>
	record;
int
	record_index = -1,
	record_index_adjust = first_record_field_names ? 0 : 1,
	total_records = table.size (),
	field_index,
	total_fields;
while (++record_index < total_records)
	{
	if (prefix)
		{
		if (record_index == 0 &&
			first_record_field_names)
			System.out.print (FIELD_NAMES_PREFIX);
		else
			System.out.print
				((record_index + record_index_adjust) + DATA_RECORD_DELIMITER);
		}
	record = table.get (record_index);
	field_index = -1;
	total_fields = record.size ();
	while (++field_index < total_fields)
		{
		if (field_index != 0)
			System.out.print ("\t");
		System.out.print (record.get (field_index));
		}
	System.out.println ();
	}
}

/** Command line usage syntax.
<p>
<blockquote><pre>
Usage: <b>Query_DB</b> [&lt;<i>Options</i>&gt;] [<b>-<u>Q</u>uery</b>] &lt;<i>query</i>&gt; [...]
&nbsp;&nbsp;Each <i>query</i> is an SQL query expression.
&nbsp;&nbsp;Options -
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>C</u>onfiguration</b> [&lt;<i>filename</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>S</u>erver</b> &lt;<i>server name</i>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>CA</u>talog</b> &lt;<i>catalog name</i>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>V</u>erbose</b>
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>H</u>elp</b>
</pre></blockquote>
<p>
<h4>
    Queries:
</h4><p>
	Each query is an SQL expression that will be run on the database
	server. Depending on the specific database server that is accessed
	the SQL query expression syntax may have limitations and/or
	extensions to the standard SQL syntax.
<p>
	If the query contains spaces then it must be quoted; If it contains
	quotes then these must be escaped from shell interpretation (by a
	preceeding backslash) or the specification is already quoted then the
	specification quotes must be different from the enclosing quotes (use
	double quotes to enclose  the specification and single  quotes in the
	specification content). Other shell metacharacters - such as '*' or
	'!' - must also be escaped or enclosed in single quotes.
<p>
	An example of a query expression to obtain all the records from the
	"Table" table in the "Catalog" catalog is:
<p>
	'select * from Catalog.Table'
<p>
	If the -catalog option had specified the "Catalog" catalog as the
	default then the "Catalog." portion of the expression would not be
	needed.
<p>
	An example of a query expression that will list the "Field_1" and
	"Field_2" field values of "Table", using the default catalog, with a
	conditional clause that selects only those records having a "Text"
	field value of "text" and a "Number" field value of "123" is:
<p>
	'select Field_1,Field_2 from Table where Text="text" and Number=123'
<p>
	The result of each query is listed to the standard output. The first
	line of each query listing contains the selected field names; this
	line will be prefixed with {@link #FIELD_NAMES_PREFIX "=== "}. The
	following lines contain the values of the fields that were selected;
	each record line is prefixed with its record number (starting with 1)
	followed by the {@link #DATA_RECORD_DELIMITER " - "} delimiter.
	If no table records were obtained from the query only the field names
	will be listed. Each record line, including the line with the field
	names, separates each field with a single horizonatal tab character.
	If the -verbose option is specified the table listing is preceeded by
	a line beginning with {@link #SQL_QUERY_PREFIX ">>> "} listing the
	SQL query that was sent to the database server. If the query fails
	the SQL query line is followed by a {@link #SQL_QUERY_FAILED "!!!
	Query failed."} line. If the -verbose option is not specified and the
	query fails nothing will be listed for that query. A query failure
	will not prevent other queries from being sent to the database
	server. When multiple queries are specified each query result will be
	separated from the next query result by a single empty line.
<p>
	All error reports are listed to the standard error output. If a query
	fails the failure report is listed to the standard error output.
<p>
<h4>
    Configuration:
</h4><p>
    If the name of the {@link PIRL.Configuration.Configuration
	Configuration} file is not specified the {@link
	#DEFAULT_CONFIGURATION_FILENAME} will be used. If the configuration
	file is not in the current working directory, it will be looked for
	in the user's home directory. The configuration file must contain
	the necessary information needed to identify and connect with the
	database server (as required by the {@link
	PIRL.Database.Database#Database(Configuration) Database} constructor
	and its {@link PIRL.Database.Database#Connect() Connect} method).
	These are typically the server "Host" name and database "Type",
	"User" and "Password" access values.
<p>
	Only one configuration file may be used.
<p>
<h4>
	Database server name:
</h4><p>
	The Configuration file may contain connection information for more
	than one database. The information for each database is organized
	by {@link Database#SERVER Server} name, which may be specified. If
	no server name is provided, the Database will attempt to use the
	default (first) Server specified in the Configuration.
<p>
	Only one database server may be used.
<p>
<h4>
    Database catalog name:
</h4><p>
	The default database catalog to use. This will override any {@link
	Database#CATALOG Catalog} parameter value from the configuration.
	The default catalog will only be used when a query expression does
	not specify a catalog.
<p>
<h4>
    Verbose
</h4><p>
    With the verbose option each query result is preceeded by listing
	the SQL expression sent to the database server. And, if the query
	fails, a "!!! Query failed." message will be listed instead of
	nothing.
<p>
<h4>
    Help:
</h4><p>
    The command line syntax usage is listed and the program exits.
    <b>N.B.</b>: If the -Help option occurs anywhere on the command line
    no other command line arguments will be used. If the command line is
	empty the usage will be listed.
<p>
    <b>N.B.</b>The usage is printed to stderr. This method always results
    in a System.exit with a status of <code>{@link 
    #EXIT_INVALID_COMMAND_LINE_SYNTAX
    EXIT_INVALID_COMMAND_LINE_SYNTAX}</code>.
*/
public static void Usage ()
{
System.err.println
	("Usage: Query_DB [<Options>] [-Query] <query> [...]" + NL
	+"  Each query is an SQL query expression." + NL
	+"  Options -" + NL
	+"  -Configuration <source>"
		+ " (default: " + DEFAULT_CONFIGURATION_FILENAME + ")" + NL
	+"  -Server <server name>"
		+ " (default: first configuration Server)" + NL
	+"  -CAtalog <catalog name>"
		+ " (default: Configuration Catalog parameter value)" + NL
    +"  -Verbose"
        + " (default: false)" + NL
	+"  -Help" + NL
	);
System.exit (EXIT_INVALID_COMMAND_LINE_SYNTAX);
}


}
