/* PSPP - a program for statistical analysis.
   Copyright (C) 2009, 2011 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#ifndef OUTPUT_TABLE_ITEM_H
#define OUTPUT_TABLE_ITEM_H 1

/* Table items.

   A table item is a subclass of an output item (see output-item.h) that
   contains a table (see table.h) and some formatting properties (currently
   just a caption). */

#include "libpspp/compiler.h"
#include "output/output-item.h"

/* A table item.

   The members of struct table_item should not be accessed directly.  Use one
   of the accessor functions defined below. */
struct table_item
  {
    struct output_item output_item; /* Superclass. */
    struct table *table;        /* The table to be rendered. */
    char *caption;              /* May be null if there is no caption. */
  };

struct table_item *table_item_create (struct table *, const char *caption);

const struct table *table_item_get_table (const struct table_item *);

const char *table_item_get_caption (const struct table_item *);
void table_item_set_caption (struct table_item *, const char *);

/* This boilerplate for table_item, a subclass of output_item, was
   autogenerated by mk-class-boilerplate. */

#include <assert.h>
#include "libpspp/cast.h"

extern const struct output_item_class table_item_class;

/* Returns true if SUPER is a table_item, otherwise false. */
static inline bool
is_table_item (const struct output_item *super)
{
  return super->class == &table_item_class;
}

/* Returns SUPER converted to table_item.  SUPER must be a table_item, as
   reported by is_table_item. */
static inline struct table_item *
to_table_item (const struct output_item *super)
{
  assert (is_table_item (super));
  return UP_CAST (super, struct table_item, output_item);
}

/* Returns INSTANCE converted to output_item. */
static inline struct output_item *
table_item_super (const struct table_item *instance)
{
  return CONST_CAST (struct output_item *, &instance->output_item);
}

/* Increments INSTANCE's reference count and returns INSTANCE. */
static inline struct table_item *
table_item_ref (const struct table_item *instance)
{
  return to_table_item (output_item_ref (&instance->output_item));
}

/* Decrements INSTANCE's reference count, then destroys INSTANCE if
   the reference count is now zero. */
static inline void
table_item_unref (struct table_item *instance)
{
  output_item_unref (&instance->output_item);
}

/* Returns true if INSTANCE's reference count is greater than 1,
   false otherwise. */
static inline bool
table_item_is_shared (const struct table_item *instance)
{
  return output_item_is_shared (&instance->output_item);
}

void table_item_submit (struct table_item *);

#endif /* output/table-item.h */
