# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 David Planella <david.planella@ubuntu.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

from . GtkHelpers import clear_text_entry, show_clear_icon
from qreator_lib.i18n import _
from qreator_lib.helpers import get_data_file, get_cache_file
from gi.repository import Gtk, GdkPixbuf, GObject  # pylint: disable=E0611
import sys
sys.path.insert(0, "/usr/share/software-center/")
import softwarecenter.db.database  # pylint: disable=F0401
import defer
import threading


class QRCodeSoftwareCenterAppGtk(object):
    def __init__(self, qr_code_update_func):
        self.qr_code_update_func = qr_code_update_func
        self.builder = Gtk.Builder()

        self.builder.add_from_file(
            get_data_file('ui', '%s.ui' % ('QrCodeSoftwareCentreApp',)))
        self.grid = self.builder.get_object('qr_code_softwarecenterapp')
        self.builder.connect_signals(self)

        # Initialize the Software Center apps autocompletion
        self.softwarecenterapps = None

        self.usc_apps_complete = Complete()

        # pylint: disable=E1101
        self.usc_apps_complete.set_hexpand(True)
        self.usc_apps_complete.connect(
            "changed", self.on_usc_apps_complete_changed, None)
        self.usc_apps_complete.connect(
            "icon-press", self.on_usc_apps_complete_icon_pressed)
        self.usc_apps_complete.set_icon_from_icon_name(
            Gtk.EntryIconPosition.PRIMARY, 'edit-find-symbolic')
        self.usc_apps_complete.set_icon_from_stock(
            Gtk.EntryIconPosition.SECONDARY, None)
        #self.builder.get_object('labelApp').grab_focus()

        self.grid.attach(self.usc_apps_complete, 1, 0, 1, 1)
        self.softwarecenter_apps_init = False
        self.grid.show_all()

    def on_activated(self):
        pass

    # handle changes to the entry
    # pylint: disable=W0613
    def on_usc_apps_complete_icon_pressed(self, widget, icon, mouse_button):
        clear_text_entry(widget, icon)

    def on_usc_apps_complete_changed(self, widget, data=None):
        show_clear_icon(widget)
        USC_APPS_BASE_URL = 'https://apps.ubuntu.com/cat/applications/'
        self.qr_code_update_func(USC_APPS_BASE_URL + widget.get_text())

    # handle autocompletion with deferred calls
    def on_prepared(self):
        if not self.softwarecenter_apps_init:
            GObject.idle_add(self.init_softwarecenter_apps)

    def init_softwarecenter_apps(self):
        deferred = defer.defer(self.get_softwarecenterapps)
        deferred.add_callback(self.on_get_softwarecenterapps)
        self.softwarecenter_apps_init = True

    def on_get_softwarecenterapps(self, result):
        # gets called when the icons and app names are ready
        self.usc_apps_complete.add(self.softwarecenterapps)

    def get_softwarecenterapps(self):
        # actually gets the icons and names
        db = softwarecenter.db.database.StoreDatabase()
        db._aptcache.open()  # pylint: disable=W0212
        db.open(use_axi=False, use_agent=True)
        apps = []
        for doc in db:
            app = db.get_application(doc)
            icon = db.get_iconname(doc)
            appinfo = App(app.appname, app.pkgname, icon)
            apps.append(appinfo)
        self.softwarecenterapps = apps
        return apps


class App(object):
    def __init__(self, name, package, icon):
        self.name = name
        self.package = package
        self.icon = icon


class Complete(Gtk.Entry):
    '''a class to autocomplete'''
    # pylint: disable=W0613
    def __init__(self, apps=None):
        Gtk.Entry.__init__(self)
        self.completion = Gtk.EntryCompletion()
        self.set_completion(self.completion)
        self.model = Gtk.ListStore(str, GdkPixbuf.Pixbuf, str)
        self.completion_img = Gtk.CellRendererPixbuf()
        self.completion.pack_start(self.completion_img, True)
        self.completion.add_attribute(self.completion_img, "pixbuf", 1)
        self.completion.set_text_column(0)
        self.completion.set_popup_set_width(False)
        self.completion.set_inline_completion(True)

        # TRANSLATORS: this message appears for a few seconds on the
        # Software Center apps text entry until the app names to
        # suggest for autocompletion have been read from the Software
        # Center database
        self.set_placeholder_text(
            _('[Wait a few seconds to enable autocompletion...]'))

    def add(self, apps):
        # pylint: disable=W0201
        self.loadthread = threading.Thread(target=self.load_pixbuf_sprites,
                                           args=(apps,)).start()

    def load_pixbuf_sprites(self, apps):
        icontheme = IconThemeSprite(apps, 16)
        icons = [icontheme[app.name] for app in apps]
        GObject.idle_add(self._finished, apps, icons)

    def _finished(self, apps, pixbufs):
        '''Add a value to the list of strings to suggest'''
        for app, icon in zip(apps, pixbufs):
            self.model.append([app.package, icon, app.name])
        self.completion.set_model(self.model)
        self.set_placeholder_text(
            _('[Type the name of an app]'))
        threading.Thread(target=self._recreate_cache,
                         args=(apps,)).start()

    def _recreate_cache(self, apps):
        # after everything is done:
        # recreate the sprite map for the next time
        icontheme = IconThemeSprite(apps, 16)
        icontheme.create_sprite(apps)


class IconThemeSprite(object):
    """The sprite is divided into two parts,
    the actual icons in one png file (-> sprite) and
    the position of the icons based on the appname in a
    text file (-> spritemap).
    """
    @property
    def spritepath(self):
        return get_cache_file("sprite_{}.png".format(self.size))

    @property
    def spritemappath(self):
        return get_cache_file("sprite_{}.txt".format(self.size))

    def __init__(self, apps, size=32):
        self.size = size
        try:
            self._sprite = GdkPixbuf.Pixbuf.new_from_file(self.spritepath)
            self._spritemap = dict((l.strip(), n) for n, l in enumerate(
                open(self.spritemappath).readlines()))
        except:
            # FIXME actually check the exception type
            self.create_sprite(apps)

    def __getitem__(self, item):
        try:
            src_y = (self._spritemap[item] + 1) * self.size
            src_x = 0
            return GdkPixbuf.Pixbuf.new_subpixbuf(
                self._sprite,
                src_x,
                src_y,
                self.size,
                self.size)
        except KeyError:
            # sprite is not found in the spritemap
            # so we have to replace it with a default
            icontheme = Gtk.IconTheme.get_default()
            return icontheme.load_icon(
                "stock_not",
                self.size,
                Gtk.IconLookupFlags.USE_BUILTIN)

    def create_sprite(self, apps):
        icontheme = Gtk.IconTheme.get_default()
        icontheme.append_search_path("/usr/share/app-install/icons/")
        # create a large sprite
        sprite = GdkPixbuf.Pixbuf.new(
            GdkPixbuf.Colorspace.RGB, True, 8, self.size, self.size * len(apps))
        for n, app in enumerate(apps):
            # load images to pixbufs
            try:
                icon = icontheme.load_icon(
                    "{0}".format(app.icon), self.size,
                    Gtk.IconLookupFlags.USE_BUILTIN)
            except:
                icon = icontheme.load_icon(
                    "stock_not", self.size,
                    Gtk.IconLookupFlags.USE_BUILTIN)
            # paste each at the end of the sprite
            icon.copy_area(0,
                           0,
                           icon.get_width(),
                           icon.get_height(),
                           sprite,
                           0,
                           (n + 1) * self.size)
        # save the sprite
        sprite.savev(self.spritepath, "png", [], [])  # undocumented gi @$#*%!
        self._sprite = sprite

        # save the sprite map
        self._spritemap = {}
        with open(self.spritemappath, "w") as spritemap:
            for n, app in enumerate(apps):
                spritemap.write(app.name + "\n")
                self._spritemap[app.name] = n
