///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/form_element_rep.h"
#include "rheolef/ublas_matrix_range.h"
using namespace rheolef;
using namespace std;

form_element_rep::form_element_rep ()
 : _X(),
   _Y(),
   _quad(),
   _bx_table(),
   _by_table(),
   _tr_p1_table(),
   _initialized(false),
   _wh(),
   _is_weighted(false),
   _bw_table(),
   _use_coordinate_system_weight(true),
   _use_coordinate_system_dual_weight(false),
   _skip_geo_compatibility_check(false)
{
}
form_element_rep::~form_element_rep()
{
}
void
form_element_rep::initialize (
   const space& X, 
   const space& Y) const
{
  _X = X;
  _Y = Y;
  _X.freeze();
  _Y.freeze();
  check_macro (_skip_geo_compatibility_check || _X.get_global_geo() == _Y.get_global_geo(),
	"form assembly: global spaces based on different meshes: unsupported");

  size_type quad_order = get_first_basis().degree() + get_second_basis().degree();
  if (coordinate_system_type() != fem_helper::cartesian) quad_order++;
  if (_is_weighted) quad_order += get_weight_basis().degree();
  quad_order -= n_derivative();
  _quad.set_order (quad_order);
  _bx_table.set (_quad, get_first_basis());
  _by_table.set (_quad, get_second_basis());
  _tr_p1_table.set (_quad, get_p1_transformation());
  if (_is_weighted) {
    check_macro (_skip_geo_compatibility_check || get_weight().get_space().get_global_geo() == get_global_geo(),
	"weighted form assembly: weight space based on a different mesh: not supported");
    _bw_table.set (_quad, get_weight_basis());
  }
  _initialized = true;
  check_after_initialize();
}
void
form_element_rep::check_after_initialize () const
{
    // this fct can be overloaded by concrete forms
}
form_element_rep::size_type
form_element_rep::n_derivative() const
{
    // this fct can be overloaded by concrete forms:
    // mass (L2 scalar product) have 0 derivative
    // d_dx have 1 derivative
    // grad_grad have 2 derivatives
    // => this virtual fct decreases the quadrature formulae order
    //    so that the integration is exact
    return 0;
}
void
form_element_rep::set_weight (const field& wh) const
{
  _wh = wh;
  if (_initialized) {
    // TODO: recompute _quad and all quad tables
    error_macro ("set_weight must be called prior to initialize"); 
    _bw_table.set (_quad, get_weight_basis());
  }
  _is_weighted = true;
}
// F (hat_xq)
point
form_element_rep::piola_transformation (const geo_element& K, size_type q) const
{
    size_type nt = get_p1_transformation().size(K);
    check_macro (nt == K.size(), "unsupported transformation " << get_p1_transformation().name());
    basis_on_quadrature::const_iterator tr = _tr_p1_table.begin (K, q);
    point xq (0,0,0);
    size_type coord_d = coordinate_dimension();
    geo::const_iterator_node p = begin_vertex();
    for (size_type k = 0; k < nt; k++, tr++) {
      Float coeff = *tr;
      const point& ak = p[K[k]];
      for (size_type i = 0; i < coord_d; i++) {
        xq[i] += ak[i] * coeff;
      }
    }
    return xq;
}
// DF (hat_xq)
void
form_element_rep::jacobian_piola_transformation (const geo_element& K, size_type q, tensor& DF) const
{
    size_type nt = get_p1_transformation().size(K);
    check_macro (nt == K.size(), "unsupported transformation " << get_p1_transformation().name());
    DF.fill (0);
    geo::const_iterator_node p = begin_vertex();
    basis_on_quadrature::const_iterator_grad grad_tr
     = _tr_p1_table.begin_grad (K, q);
    size_type coord_d = coordinate_dimension();
    for (size_type k = 0; k < nt; k++, grad_tr++)
      ::cumul_otimes (DF, p[K[k]], *grad_tr, coord_d, K.dimension());
}
// The pseudo inverse extend inv(DF) for face in 3d or edge in 2d
//  i.e. usefull for Laplacian-Beltrami and others surfacic forms.
//
// pinvDF (hat_xq) = inv DF, if tetra in 3d, tri in 2d, etc
//                  = pseudo-invese, when tri in 3d, edge in 2 or 3d
// e.g. on 3d face : pinvDF*DF = [1, 0, 0; 0, 1, 0; 0, 0, 0]
//
// let DF = [u, v, w], where u, v, w are the column vectors of DF
// then det(DF) = mixt(u,v,w)
// and det(DF)*inv(DF)^T = [v^w, w^u, u^v] where u^v = vect(u,v)
//
// application:
// if K=triangle(a,b,c) then u=ab=b-a, v=ac=c-a and w = n = u^v/|u^v|.
// Thus DF = [ab,ac,n] and det(DF)=|ab^ac|
// and inv(DF)^T = [ac^n/|ab^ac|, -ab^n/|ab^ac|, n]
// The pseudo-inverse is obtained by remplacing the last column n by zero.
//
tensor
form_element_rep::pseudo_inverse_jacobian_piola_transformation (const tensor& DF, size_type map_dim) const
{
    size_t d = coordinate_dimension();
    if (d == map_dim) {
      return inv(DF, map_dim);
    }
    tensor invDF;
    switch (map_dim) {
      case 0: { // point in 1D
          invDF(0,0) = 1;
          return invDF;
      }
      case 1: { // segment in 2D
          point t = DF.col(0);
          invDF.set_row (t/norm2(t), 0, d);
          return invDF;
      }
      case 2: {
          point t0 = DF.col(0);
          point t1 = DF.col(1);
          point n = vect(t0,t1);
          Float det2 = norm2(n);
          point v0 =   vect(t1,n)/det2;
          point v1 = - vect(t0,n)/det2;
          invDF.set_row (v0, 0, d);
          invDF.set_row (v1, 1, d);
          return invDF;
      }
      default:
        error_macro ("psuedo_inverse_jacobian_piola_transformation: unsupported element dimension "
	    << map_dim << " in " << coordinate_dimension() << "D mesh.");
    }
}
Float
form_element_rep::det_jacobian_piola_transformation (const tensor& DF, size_type map_dim) const
{
    if (coordinate_dimension() == map_dim) {
      return DF.determinant (map_dim);
    }
    /* surface jacobian: references:
     * Spectral/hp element methods for CFD
     * G. E. M. Karniadakis and S. J. Sherwin
     * Oxford university press
     * 1999
     * page 165
     */
    switch (map_dim) {
      case 0: return 1;
      case 1: return norm(DF.col(0));
      case 2: return norm(vect(DF.col(0), DF.col(1)));
      default:
        error_macro ("det_jacobian_piola_transformation: unsupported element dimension "
	    << map_dim << " in " << coordinate_dimension() << "D mesh.");
    }
}
Float
form_element_rep::det_jacobian_piola_transformation (const geo_element& K, size_type q) const
{
    tensor DF;
    jacobian_piola_transformation (K, q, DF);
    return det_jacobian_piola_transformation (DF, K.dimension());
} 
// ----------------------------------------------------------
// add to elementary matrix : (phi otimes psi) * w
// ----------------------------------------------------------
void
form_element_rep::cumul_otimes (
    ublas::matrix<Float>& m, 
    const Float& w,
    basis_on_quadrature::const_iterator phi,
    basis_on_quadrature::const_iterator last_phi,
    basis_on_quadrature::const_iterator first_psi,
    basis_on_quadrature::const_iterator last_psi)
{
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      basis_on_quadrature::const_iterator psi = first_psi;
      Float phi_w = (*phi)*w;
      for (size_type j = 0; psi != last_psi; psi++, j++)
        m(i,j) += (*psi)*phi_w;
    }
}
// ----------------------------------------------------------
// add to elementary matrix : (phi otimes d psi/ds) * w
// ----------------------------------------------------------
void
form_element_rep::cumul_otimes (
    ublas::matrix<Float>& m, 
    const Float& w,
    basis_on_quadrature::const_iterator phi,
    basis_on_quadrature::const_iterator last_phi,
    basis_on_quadrature::const_iterator_grad first_grad_psi,
    basis_on_quadrature::const_iterator_grad last_grad_psi)
{
    // Only the first component of psi is relevant (curvilinear derivative
    // along an edge)
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      Float phi_w = (*phi)*w;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++)
        m(i,j) += (*grad_psi)[0]*phi_w;
    }
}
// ----------------------------------------------------------
// add to elementary matrix : (phi otimes d^2 psi/ds^2) * w
// ----------------------------------------------------------
void
form_element_rep::cumul_otimes (
    ublas::matrix<Float>& m, 
    const Float& w,
    basis_on_quadrature::const_iterator phi,
    basis_on_quadrature::const_iterator last_phi,
    basis_on_quadrature::const_iterator_hessian first_hessian_psi,
    basis_on_quadrature::const_iterator_hessian last_hessian_psi)
{
    // Only the 0,0 component of hessian of psi is relevant (curvilinear derivative
    // along an edge)
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      basis_on_quadrature::const_iterator_hessian hessian_psi = first_hessian_psi;
      Float phi_w = (*phi)*w;
      for (size_type j = 0; hessian_psi != last_hessian_psi; hessian_psi++, j++)
        m(i,j) += (*hessian_psi)(0,0)*phi_w;
    }
}
// ----------------------------------------------------------
// add to elementary matrix : (d phi/ds otimes d^2 psi/ds^2) * w
// ----------------------------------------------------------
void
form_element_rep::cumul_otimes (
    ublas::matrix<Float>& m, 
    const Float& w,
    basis_on_quadrature::const_iterator_grad grad_phi,
    basis_on_quadrature::const_iterator_grad last_grad_phi,
    basis_on_quadrature::const_iterator_hessian first_hessian_psi,
    basis_on_quadrature::const_iterator_hessian last_hessian_psi)
{
    // Only the 0,0 component of hessian of psi is relevant (curvilinear derivative
    // along an edge), and the 0 component of grad phi (d phi/ds)
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      basis_on_quadrature::const_iterator_hessian hessian_psi = first_hessian_psi;
      Float phi_w = (*grad_phi)[0]*w;
      for (size_type j = 0; hessian_psi != last_hessian_psi; hessian_psi++, j++)
        m(i,j) += (*hessian_psi)(0,0)*phi_w;
    }
}
// ----------------------------------------------------------
// add to elementary matrix : (d^2 phi/ds^2 otimes d^2 psi/ds^2) * w
// ----------------------------------------------------------
void
form_element_rep::cumul_otimes (
    ublas::matrix<Float>& m, 
    const Float& w,
    basis_on_quadrature::const_iterator_hessian hessian_phi,
    basis_on_quadrature::const_iterator_hessian last_hessian_phi,
    basis_on_quadrature::const_iterator_hessian first_hessian_psi,
    basis_on_quadrature::const_iterator_hessian last_hessian_psi)
{
    // Only the 0,0 component of hessian of psi is relevant (curvilinear derivative
    // along an edge), and the 0 component of grad phi (d phi/ds)
    for (size_type i = 0; hessian_phi != last_hessian_phi; hessian_phi++, i++) {
      basis_on_quadrature::const_iterator_hessian hessian_psi = first_hessian_psi;
      Float phi_w = (*hessian_phi)(0,0)*w;
      for (size_type j = 0; hessian_psi != last_hessian_psi; hessian_psi++, j++)
        m(i,j) += (*hessian_psi)(0,0)*phi_w;
    }
}
// ----------------------------------------------------------
// add to elementary matrix : (dphi/ds otimes d psi/ds) * w
// ----------------------------------------------------------
void
form_element_rep::cumul_otimes (
    ublas::matrix<Float>& m, 
    const Float& w,
    basis_on_quadrature::const_iterator_grad grad_phi,
    basis_on_quadrature::const_iterator_grad last_grad_phi,
    basis_on_quadrature::const_iterator_grad first_grad_psi,
    basis_on_quadrature::const_iterator_grad last_grad_psi)
{
    // Only the first component of psi and phi are relevant (curvilinear derivative
    // along an edge)
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      Float grad_phi0_w = ((*grad_phi)[0])*w;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++)
        m(i,j) += (*grad_psi)[0]*grad_phi0_w;
    }
}
// t(u,v) = (t*u,v)
static
Float
tensor_apply (
  const tensor& t,
  const point& u,
  const point& v,
  tensor::size_type map_d)
{
  typedef tensor::size_type size_type;
  Float sum = 0;
  for (size_type k = 0; k < map_d; k++) {
    for (size_type l = 0; l < map_d; l++) {
      sum += v[k]*t(k,l)*u[l];
    }
  }
  return sum;
}
// --------------------------------------------------------------------
// Mij += trans(grad_phi_i) * Dw * grad_psi_j
// --------------------------------------------------------------------
void
form_element_rep::cumul_otimes (
    ublas::matrix<Float>& m, 
    const tensor& Dw,
    basis_on_quadrature::const_iterator_grad grad_phi,
    basis_on_quadrature::const_iterator_grad last_grad_phi,
    basis_on_quadrature::const_iterator_grad first_grad_psi,
    basis_on_quadrature::const_iterator_grad last_grad_psi,
    size_type map_d)
{
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	m(i,j) += tensor_apply (Dw, *grad_psi, *grad_phi, map_d);
      }
    }
}
Float
form_element_rep::weight_coordinate_system (const geo_element& K, size_type q) const
{
    if (coordinate_system_type() == fem_helper::cartesian
        || ! use_coordinate_system_weight()) {
  	return 1;
    }
    point xq = piola_transformation (K, q);
    switch (coordinate_system_type()) {
      case fem_helper::axisymmetric_rz: 
	  return use_coordinate_system_dual_weight() ? 1/xq[0] : xq[0];
      case fem_helper::axisymmetric_zr:
	  return use_coordinate_system_dual_weight() ? 1/xq[1] : xq[1];
      default: {
	fatal_macro ("unsupported coordinate system `" 
	<<  fem_helper::coordinate_system_name (coordinate_system_type()) << "'");
	return 0;
      }
    }
}
// weighted forms: i-th component on node x_q in K
Float
form_element_rep::weight (const geo_element& K, size_type q, size_t i_comp) const
{
    Float wq = 0;
    tiny_vector<Float> wdof;
    _wh.get_dof_value_from_global (get_weight_basis(), K, wdof, i_comp);
    size_type n = wdof.size();
    check_macro (n == get_weight_basis().size(K), "unconsistent basis and dof table (fatal)");
    basis_on_quadrature::const_iterator iw = _bw_table.begin (K, q);
    for (size_type i = 0; i < n; i++, iw++) {
      wq += wdof[i] * (*iw); // sum_i w_coef(i)*hat_phi(x_i)
    }
    return wq;
}
// tensor-weighted forms
void
form_element_rep::weight (const geo_element& K, size_type q, tensor& W) const
{
    size_t                        d         = _wh.dimension();
    fem_helper::coordinate_type   sys_coord = _wh.coordinate_system_type();
    fem_helper::valued_field_type valued    = _wh.get_valued_type();
    for (size_t i = 0; i < d; i++) {
      for (size_t j = 0; j < d; j++) {
        size_t i_comp = fem_helper::tensor_index (valued, sys_coord, i, j);
	W(i,j) = weight (K, q, i_comp);
      }
    }
}

// map non-Lagrange reference basis functions to element basis function 
template<class InputIterator,class OutputIterator>
InputIterator 
mypiola_basis (const reference_element& K, const basis& b, const tensor& DF, 
	InputIterator first_in, InputIterator last_in, OutputIterator first_out)
{
  if      (b.family() == element_constant::Lagrange) return first_in;
  else if (b.family() == element_constant::Hermite) 
   {
   	for (piola::size_type i=0; first_in!=last_in; first_in++, i++)
	 {
	 	if (b.dof_family(K, i) == element_constant::Lagrange)
			first_out[i]=*first_in;
		else // Hermite
			{
			DF(0,0)*(*first_in); 
			first_out[i]=DF(0,0)*(*first_in); 
			}
	 }
   }
  else error_macro("Unknown family of finite elements, " << b.family());
  return first_out;
}

void 
form_element_rep::build_scalar_mass (const geo_element& K, ublas::matrix<Float>& m, size_t weight_i_comp) const
{
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  vector<Float> F_bx_table (get_first_basis().size(hat_K));
  vector<Float> F_by_table (get_second_basis().size(hat_K));
  tensor DF;
  for (size_type q = 0; first != last; first++, q++) {
    Float wq = is_weighted() ? weight(K,q,weight_i_comp) : Float(1);
    wq *= weight_coordinate_system (K, q);
    jacobian_piola_transformation (K, q, DF);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w; 
    vector<Float>::const_iterator Fx_begin = piola_basis (get_first_space(), K, get_first_basis(), DF,
    	_bx_table.begin (hat_K, q), _bx_table.end (hat_K, q), F_bx_table.begin());
    vector<Float>::const_iterator Fy_begin = piola_basis (get_second_space(), K, get_second_basis(), DF,
    	_by_table.begin (hat_K, q), _by_table.end (hat_K, q), F_by_table.begin());
    cumul_otimes (m, wq, 
      Fy_begin, Fy_begin+get_second_basis().size(hat_K),
      Fx_begin, Fx_begin+get_first_basis().size(hat_K));
  }
}
// --------------------------------------------------------------------
// general local mass matrix : multi-component spaces and weight
// --------------------------------------------------------------------
void 
form_element_rep::build_general_mass (const geo_element& K, ublas::matrix<Float>& m) const
{
  size_type n_comp = get_first_space().n_component();
  if (n_comp == 1) {
    build_scalar_mass (K, m);
    return;
  }
  check_macro (get_first_basis().family() == element_constant::Lagrange &&
  	      get_second_basis().family() == element_constant::Lagrange, 
	"Only Lagrange elements supported");
  // space is vectorial or tensorial: weight could be scalar, tensor or tensor4
  reference_element hat_K = K;
  size_type n1 = get_second_basis().size(hat_K);
  size_type n2 = get_first_basis().size(hat_K);
  m.resize (n_comp*n1, n_comp*n2);
  m.clear();
  fem_helper::coordinate_type   sys_coord     = get_first_space().coordinate_system_type();
  fem_helper::valued_field_type space_valued  = get_first_space().get_valued_type();
  fem_helper::valued_field_type weight_valued = fem_helper::scalar;
  if (is_weighted()) {
    weight_valued = _wh.get_valued_type();
  }
  if (weight_valued == fem_helper::scalar) {
    // --------------------------------------------------------------------------
    // all components have the same weight and the local matrix is block-diagonal
    // --------------------------------------------------------------------------
    ublas::matrix<Float> m_ij;
    build_scalar_mass (K, m_ij);
    switch (space_valued) {
     case fem_helper::scalar:
     case fem_helper::vectorial:
     case fem_helper::unsymmetric_tensorial: {
      for (size_type i = 0; i < n_comp; i++)
        mr_set (m, ublas::range(i*n1,(i+1)*n1), ublas::range(i*n2,(i+1)*n2), m_ij);
      break;
     }
     case fem_helper::tensorial: {
      // symmetric tensor => 2 factor for (12,13,23) subscripts
      for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
        fem_helper::pair_size_type ij = fem_helper::tensor_subscript (space_valued, sys_coord, ij_comp);
        size_t i = ij.first;
        size_t j = ij.second;
        Float factor_ij = ((i == j) ? 1 : 2); // symmetry => multiplicity factor: when i!=j : 2*w_ij
  	mr_set (m, ublas::range(ij_comp*n1,(ij_comp+1)*n1), ublas::range(ij_comp*n2,(ij_comp+1)*n2), factor_ij*m_ij);
      }
      break;
     }
     default: {
      string space_valued_name = get_first_space().get_valued();
      error_macro ("unexpected `" << space_valued_name << "'-valued space for the `mass' form.");
     }
    }
  } else { // weight_valued != fem_helper::scalar
    // ------------------------------------------------------------------------------
    // components have different weight : the local matrix is no more block-diagonal
    // ------------------------------------------------------------------------------
    switch (space_valued) {
      case fem_helper::vectorial: {
        // first case : vector space and tensor weight
        //     m(u,v) = int_Omega w_ij u_j v_i dx
	//     and w_ij = w_ji
        ublas::matrix<Float> m_ij;
        for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
          for (size_type j_comp = 0; j_comp < n_comp; j_comp++) {
	    size_type ij_comp = fem_helper::tensor_index (weight_valued, sys_coord, i_comp, j_comp);
            build_scalar_mass (K, m_ij, ij_comp);
            mr_set (m, ublas::range(i_comp*n1,(i_comp+1)*n1), ublas::range(j_comp*n2,(j_comp+1)*n2), m_ij);
          }
        }
	break;
      }
      case fem_helper::tensorial: {
        // second case : tensor space and tensor4 weight
        //     m(tau,gamma) = int_Omega w_ijkl tau_kj gamma_ij dx
	//     and w_ijkl = w_jikl, w_ijkl = w_ijlk and w_ijkl = w_klij
	// symmetry => multiplicity factor: 4 when i!=j and k!=l or 2 when i!=j xor k!=l
        ublas::matrix<Float> m_ijkl;
        for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
          fem_helper::pair_size_type ij = fem_helper::tensor_subscript (space_valued, sys_coord, ij_comp);
          size_t i = ij.first;
          size_t j = ij.second;
          Float factor_ij = ((i == j) ? 1 : 2);
          for (size_type kl_comp = 0; kl_comp < n_comp; kl_comp++) {
            fem_helper::pair_size_type kl = fem_helper::tensor_subscript (space_valued, sys_coord, kl_comp);
            size_t k = kl.first;
            size_t l = kl.second;
            Float factor_kl = ((k == l) ? 1 : 2);
	    Float factor_ijkl = factor_ij*factor_kl;
	    size_type ijkl_comp = fem_helper::tensor4_index (weight_valued, sys_coord, i, j, k, l);
            build_scalar_mass (K, m_ijkl, ijkl_comp);
            mr_set (m, ublas::range(ij_comp*n1,(ij_comp+1)*n1), ublas::range(kl_comp*n2,(kl_comp+1)*n2), factor_ijkl*m_ijkl);
          }
        }
	break;
      }
      default: {
        error_macro ("unexpected " << fem_helper::valued_field_name(space_valued) << "-valued space and `"
		<< fem_helper::valued_field_name(weight_valued)
		<< "'-valued weight for the `mass' form.");
      }
    }
  }
}
// --------------------------------------------------------------------
// (d_dxi phi, psi)
// Mij += trans(grad_phi_i) * Dw * grad_psi_j
// --------------------------------------------------------------------
void
form_element_rep::d_dx (const geo_element& K, ublas::matrix<Float>& m, size_type idx) const
{
  check_macro (get_first_basis().family() == element_constant::Lagrange &&
  	      get_second_basis().family() == element_constant::Lagrange, 
	"Only Lagrange elements supported");
  size_type map_d = K.dimension();
  check_macro (idx < map_d, "unexpected `d_dx" << idx << "' on element type "
	<< "`" << K.name() << "'");
  check_macro (coordinate_dimension() == K.dimension(),
    "unsupported `d_dx" << idx << "' form on element type `" << K.name()
    << " in " << coordinate_dimension() << "D geometry");
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  tensor DF;
  tensor invDF;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = inv(DF,map_d);
    Float wq = is_weighted() ? weight(K,q) : Float(1);
    wq *= weight_coordinate_system (K, q);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w;
    point invDFt_idx_wq = invDF.col(idx) * wq;
  
    // put in a subroutine:
    basis_on_quadrature::const_iterator phi      = _by_table.begin(hat_K, q);
    basis_on_quadrature::const_iterator last_phi = _by_table.end(hat_K, q);
    basis_on_quadrature::const_iterator_grad first_grad_psi = _bx_table.begin_grad (hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_psi  = _bx_table.end_grad (hat_K, q);
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      Float phi_i = *phi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	Float sum = 0;
	const point& grad_psi_j = *grad_psi;
        for (size_type k = 0; k < map_d; k++) {
          sum += phi_i * invDFt_idx_wq[k] * grad_psi_j[k];
	}
	m(i,j) += sum;
      }
    }
  }
}
// --------------------------------------------------------------------
//   (d_ds phi, psi) along an edge e
// --------------------------------------------------------------------
void 
form_element_rep::build_d_ds (const geo_element& K, ublas::matrix<Float>& m) const
{
  check_macro(K.dimension()==1, 
  	"unsupported `d_ds' form on element type `" << K.name() << "'.");
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  vector<point> F_bx_table_grad (get_first_basis().size(hat_K));
  vector<Float> F_by_table (get_second_basis().size(hat_K));
  tensor DF;
  for (size_type q = 0; first != last; first++, q++) {
    Float wq = is_weighted() ? weight(K,q) : Float(1);
    wq *= weight_coordinate_system (K, q);
    wq *= (*first).w; 
    // Jacobian of Piola transform cancels out in
    //    int_e psi dphi/ds ds 
    jacobian_piola_transformation (K, q, DF);
    vector<point>::const_iterator Fx_begin_grad = piola_basis (get_first_space(), K, get_first_basis(), DF,
    	_bx_table.begin_grad (hat_K, q), _bx_table.end_grad (hat_K, q), F_bx_table_grad.begin());
    vector<Float>::const_iterator Fy_begin = piola_basis (get_second_space(), K, get_second_basis(), DF,
    	_by_table.begin (hat_K, q), _by_table.end (hat_K, q), F_by_table.begin());
    cumul_otimes (m, wq, 
      Fy_begin, Fy_begin+get_second_basis().size(hat_K),
      Fx_begin_grad, Fx_begin_grad+get_first_basis().size(hat_K));
  }
}
// --------------------------------------------------------------------
//   (d^2_ds^2 phi, psi) along an edge e
// --------------------------------------------------------------------
void 
form_element_rep::build_d2_ds2 (const geo_element& K, ublas::matrix<Float>& m) const
{
  check_macro(K.dimension()==1, 
  	"unsupported `d2_ds2' form on element type `" << K.name() << "'.");
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  vector<tensor> F_bx_table (get_first_basis().size(hat_K)); 
  vector<Float> F_by_table (get_second_basis().size(hat_K));
  tensor DF;
  tensor invDF;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, map_d);
    Float wq = weight_coordinate_system (K, q);
    wq *= (*first).w;
    wq *= (is_weighted() ? weight(K,q) : Float(1));
    // Jacobian of Piola transform appears inverted in
    //    int_e d^2 phi/ds^2 psi ds 
    wq /= det_jacobian_piola_transformation (DF, K.dimension());
    vector<tensor >::const_iterator Fx_begin_hessian 
    	= piola_basis (get_first_space(), K, get_first_basis(), DF,
    	_bx_table.begin_hessian (hat_K, q), _bx_table.end_hessian (hat_K, q), F_bx_table.begin());
    vector<Float>::const_iterator Fy_begin = piola_basis (get_second_space(), K, get_second_basis(), DF,
    	_by_table.begin (hat_K, q), _by_table.end (hat_K, q), F_by_table.begin());
    cumul_otimes (m, wq, 
      Fy_begin, Fy_begin+get_second_basis().size(hat_K),
      Fx_begin_hessian, Fx_begin_hessian+get_first_basis().size(hat_K));
  }
}
// --------------------------------------------------------------------
//   (d^2_ds^2 phi, d_ds psi) along an edge e
// --------------------------------------------------------------------
void 
form_element_rep::build_d2_ds2_d_ds (const geo_element& K, ublas::matrix<Float>& m) const
{
  check_macro(K.dimension()==1, 
  	"unsupported `d2_ds2_d_ds' form on element type `" << K.name() << "'.");
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  vector<tensor > F_bx_table (get_first_basis().size(hat_K)); 
  vector<point> F_by_table (get_second_basis().size(hat_K));
  tensor DF;
  tensor invDF;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, map_d);
    Float wq = weight_coordinate_system (K, q);
    wq *= (*first).w;
    wq *= (is_weighted() ? weight(K,q) : Float(1));
    // Jacobian of Piola transform appears at power -2 in
    //    int_e d^2 phi/ds^2 d psi/ds ds 
    wq /= sqr(det_jacobian_piola_transformation (DF, K.dimension()));
    vector<tensor >::const_iterator Fx_begin_hessian 
    	= piola_basis (get_first_space(), K, get_first_basis(), DF,
    	_bx_table.begin_hessian (hat_K, q), _bx_table.end_hessian (hat_K, q), F_bx_table.begin());
    vector<point>::const_iterator Fy_begin_grad = piola_basis (get_second_space(), K, get_second_basis(), DF,
    	_by_table.begin_grad (hat_K, q), _by_table.end_grad (hat_K, q), F_by_table.begin());
    cumul_otimes (m, wq, 
      Fy_begin_grad, Fy_begin_grad+get_second_basis().size(hat_K),
      Fx_begin_hessian, Fx_begin_hessian+get_first_basis().size(hat_K));
  }
}
// --------------------------------------------------------------------
//   (d^2_ds^2 phi, d^2_ds^2 psi) along an edge e
// --------------------------------------------------------------------
void 
form_element_rep::build_d2_ds2_d2_ds2 (const geo_element& K, ublas::matrix<Float>& m) const
{
  check_macro(K.dimension()==1, 
  	"unsupported `d2_ds2_d2_ds2' form on element type `" << K.name() << "'.");
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  vector<tensor > F_bx_table (get_first_basis().size(hat_K));
  vector<tensor > F_by_table (get_second_basis().size(hat_K)); 
  tensor DF;
  tensor invDF;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, map_d);
    Float wq = weight_coordinate_system (K, q);
    wq *= (*first).w;
    wq *= (is_weighted() ? weight(K,q) : Float(1));
    // Jacobian of Piola transform appears at power -3 in
    //    int_e d^2 phi/ds^2 d^2 psi/ds^2 ds 
    Float det=det_jacobian_piola_transformation (DF, K.dimension());
    wq /= det*det*det;
    vector<tensor >::const_iterator Fx_begin_hessian 
    	= piola_basis (get_first_space(), K, get_first_basis(), DF,
    	_bx_table.begin_hessian (hat_K, q), _bx_table.end_hessian (hat_K, q), F_bx_table.begin());
    vector<tensor >::const_iterator Fy_begin_hessian 
    	= piola_basis (get_second_space(), K, get_second_basis(), DF,
    	_by_table.begin_hessian (hat_K, q), _by_table.end_hessian (hat_K, q), F_by_table.begin());
    cumul_otimes (m, wq, 
      Fy_begin_hessian, Fy_begin_hessian+get_second_basis().size(hat_K),
      Fx_begin_hessian, Fx_begin_hessian+get_first_basis().size(hat_K));
  }
}
// ----------------------------
// scalar grad_grad (Laplacian)
// ----------------------------
void 
form_element_rep::build_grad_grad (const geo_element& K, ublas::matrix<Float>& m) const
{
  bool scalar_diffusion = !is_weighted() || (is_weighted() && _wh.n_component() == 1);
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  vector<point> F_bx_table (get_first_basis().size(hat_K));
  vector<point> F_by_table (get_second_basis().size(hat_K));
  tensor DF;
  tensor invDF;
  tensor Dw;
  tensor W;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, map_d);
    Float wq = weight_coordinate_system (K, q);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w;
    if (scalar_diffusion) {
       wq *= (is_weighted() ? weight(K,q) : Float(1));
       Dw = wq*invDF*trans(invDF);
    } else { // tensorial diffusion
       weight (K, q, W);
       Dw = wq*invDF*W*trans(invDF);
    }
    vector<point>::const_iterator Fx_begin_grad = piola_basis (get_first_space(), K, get_first_basis(), DF,
    	_bx_table.begin_grad (hat_K, q), _bx_table.end_grad (hat_K, q), F_bx_table.begin());
    vector<point>::const_iterator Fy_begin_grad = piola_basis (get_second_space(), K, get_second_basis(), DF,
    	_by_table.begin_grad (hat_K, q), _by_table.end_grad (hat_K, q), F_by_table.begin());
    cumul_otimes (m, Dw, 
      Fy_begin_grad, Fy_begin_grad+get_second_basis().size(hat_K),
      Fx_begin_grad, Fx_begin_grad+get_first_basis().size(hat_K),
      map_d);
  }
}
// --------------------------------------------------------------------
//   (d_ds phi, d_ds psi) along an edge e
// --------------------------------------------------------------------
void
form_element_rep::build_d_ds_d_ds (const geo_element& K, ublas::matrix<Float>& m) const
{
    warning_macro ("obsolete `d_ds_d_ds' form: ussing `grad_grad' instead...");
    build_grad_grad (K, m);
}
// --------------------------------------------------------------------
// int_K (gradt u : grad v) w dK = int_K [(M U).V] w dK,
//		with u = Sum_j,l U_{nx*l+j} psi_j
//			 v = Sum_i,k V_{ny*k+i} phi_i
// so M_{ny*k+i,nx*l+j} = int_K d psi_j/dk * d phi_i/dl w dK 
// --------------------------------------------------------------------
void
form_element_rep::build_gradt_grad (const geo_element& K, ublas::matrix<Float>& m) const
{
  check_macro (get_first_basis().family() == element_constant::Lagrange &&
  	      get_second_basis().family() == element_constant::Lagrange, 
	"Only Lagrange elements supported");
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  check_macro (coordinate_dimension() == K.dimension(),
    "unsupported `gradt_grad' form on element type `" << K.name()
    << " in " << coordinate_dimension() << "D geometry");
  size_type nx = get_first_basis().size(hat_K);
  size_type ny = get_second_basis().size(hat_K);
  m.resize (map_d*ny, map_d*nx);
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  tensor DF;
  tensor invDF;
  tensor invDFt;
  tensor Dw;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = inv(DF,map_d);
    invDFt = trans(invDF);
    Float wq = is_weighted() ? weight(K,q) : Float(1);
    wq *= weight_coordinate_system (K, q);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w;
    
    basis_on_quadrature::const_iterator_grad grad_phi       = _by_table.begin_grad(hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_phi  = _by_table.end_grad(hat_K, q);
    basis_on_quadrature::const_iterator_grad first_grad_psi = _bx_table.begin_grad (hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_psi  = _bx_table.end_grad (hat_K, q);
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      const point& hat_grad_phi_i = *grad_phi;
      point grad_phi_i = invDFt*hat_grad_phi_i;
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	const point& hat_grad_psi_j = *grad_psi;
        point grad_psi_j = invDFt*hat_grad_psi_j;
        for (size_type k = 0; k < map_d; k++) {
          for (size_type l = 0; l < map_d; l++) {
	    m(i+ny*k,j+nx*l) += grad_psi_j[k]*grad_phi_i[l]*wq;
	  }
	}
      }
    }
  }
}
// --------------------------------------------------------------------
// vector-valued : <div_div phi, psi> = (div phi, div psi)
// Mij += trans(grad_phi_i) * Dw * grad_psi_j
// --------------------------------------------------------------------
void
form_element_rep::build_div_div (const geo_element& K, ublas::matrix<Float>& m) const
{
  check_macro (get_first_basis().family() == element_constant::Lagrange &&
  	      get_second_basis().family() == element_constant::Lagrange, 
	"Only Lagrange elements supported");
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  check_macro (coordinate_dimension() == K.dimension(),
    "unsupported `gradt_grad' form on element type `" << K.name()
    << " in " << coordinate_dimension() << "D geometry");
  size_type nx = get_first_basis().size(hat_K);
  size_type ny = get_second_basis().size(hat_K);
  m.resize (map_d*ny, map_d*nx);
  m.clear();
  quadrature<Float>::const_iterator first = _quad.begin(hat_K);
  quadrature<Float>::const_iterator last  = _quad.end  (hat_K);
  tensor DF;
  tensor invDF;
  tensor invDFt;
  tensor Dw;
  for (size_type q = 0; first != last; first++, q++) {
    jacobian_piola_transformation (K,q,DF);
    invDF = inv(DF,map_d);
    invDFt = trans(invDF);
    Float wq = is_weighted() ? weight(K,q) : Float(1);
    wq *= weight_coordinate_system (K, q);
    wq *= det_jacobian_piola_transformation (DF, K.dimension());
    wq *= (*first).w;
    
    basis_on_quadrature::const_iterator_grad grad_phi       = _by_table.begin_grad(hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_phi  = _by_table.end_grad(hat_K, q);
    basis_on_quadrature::const_iterator_grad first_grad_psi = _bx_table.begin_grad (hat_K, q);
    basis_on_quadrature::const_iterator_grad last_grad_psi  = _bx_table.end_grad (hat_K, q);
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      const point& hat_grad_phi_i = *grad_phi;
      point grad_phi_i = invDFt*hat_grad_phi_i;
      basis_on_quadrature::const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	const point& hat_grad_psi_j = *grad_psi;
        point grad_psi_j = invDFt*hat_grad_psi_j;
        for (size_type k = 0; k < map_d; k++) {
          for (size_type l = 0; l < map_d; l++) {
	    m(i+ny*k,j+nx*l) += grad_psi_j[l]*grad_phi_i[k]*wq;
	  }
	}
      }
    }
  }
}
