/*
 * ===========================
 * VDK Visual Develeopment Kit
 * Version 0.6.2
 * May 1999
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _vdkcustombutton_h
#define _vdkcustombutton_h
#include <vdk/vdkobj.h>
#include <vdk/boxes.h>
#include <vdk/label.h>
#include <vdk/image.h>
#include <vdk/menu.h>
// #include <vdk/gtkcombobutton.h>
/*!
  \class VDKCustomButton
  \brief This class provides an unified wrapper for almost all kind
  of GtkButton
  \par Signals
  \arg clicked signal, emitted when user clicks over button
  \arg enter_signal, emitted when mouse goes over button
  \arg leave_signal. emitted when mouse leaves button

  \par Examples
  In ./testvdk/testvdk.cc
 */
class VDKCustomButton: public VDKObject
{
 protected:
  void ConnectButtonSignals();
  static void ToggleEvent(GtkWidget *wid, gpointer gp);
 public:
// properties
  /*!
    <read-only>
    Permits to access embedded VDKBox that already contains
    a VDKLabel and probably a VDKImage
   */
  VDKReadOnlyValueProp<VDKCustomButton,  VDKBox*> ButtonBox;
  /*!
    <read-only>
    Permits to access embedded VDKLabel
    \code
    VDKLabel* label = button->Label;
    if(label)
      {
      label->Text = "new text";
      label->Justify = GTK_JUSTIFY_RIGHT;
      }
    \endcode
   */
  VDKReadOnlyValueProp<VDKCustomButton,  VDKLabel*> Label;
  /*!
    <read-only>
    Permits to access embedded VDKImage
  */
  VDKReadOnlyValueProp<VDKCustomButton,  VDKImage*> Pixmap;
  /*!
    Permits to change button caption
  */
  VDKReadWriteValueProp<VDKCustomButton, const char*> Caption;
  /*!
    Permits to change button caption wrap
  */
  VDKReadWriteValueProp<VDKCustomButton, bool> CaptionWrap;
  /*
    !
    Permits to set/get checked state (meaningless if isn't
    a toggle button
  */
  VDKReadWriteValueProp<VDKCustomButton, bool> Checked;
  /*
    !
    Permits to set/get button relief (meaningless if is
    a toggle button
  */
  VDKReadWriteValueProp<VDKCustomButton, GtkReliefStyle> Relief;
  /*!
    Constructor, makes a labeled button
    \param owner
    \param label
    \param type
    \par type and toggle
    <type> arg is used to determine if the button is toggled and if it is
    a combo button or not
    - toggled can be VDK_CBUTTON_TOGGLED or VDK_CBUTTON_UNTOGGLED
    - type can be: VDK_CBUTTON_NORMAL or VDK_CBUTTON_COMBO
    - toggled and type can be or'ed togheter except that
    VDK_CBUTTON_TOGGLED | VDK_CBUTTON_COMBO does not produce a combo
    toggled button.

    \par accelerators
    custom button support accelerators, setting an underline into button label
    makes an accelerator.
    \code
    VDKCustomButton *button = new VDKCustomButton(this,"_A button");
    \endcode
    this makes that a clicked or toggled signal will be emitted using
    alt+a keystroke.
   */
  VDKCustomButton(VDKForm* owner, 
		  const char* label,
		  unsigned int type = 
		  VDK_CBUTTON_UNTOGGLED |  VDK_CBUTTON_NORMAL);
  /*!
    Constructor, makes a pixmapped button.
    \param pixfile
    \param label
    \param type
    \param position, refers to label position into button box
   */
  VDKCustomButton(VDKForm* owner, 
		  const char* pixfile, 
		  const char* label,
		  unsigned int type = 
		  VDK_CBUTTON_UNTOGGLED |  VDK_CBUTTON_NORMAL,
		  GtkPositionType position = GTK_POS_RIGHT);
  /*!
    Constructor, same as above but using pixdata instead a pix file.
  */
  VDKCustomButton(VDKForm* owner, 
		  const char** pixdata, 
		  const char* label,
		  unsigned int type = 
		  VDK_CBUTTON_UNTOGGLED |  VDK_CBUTTON_NORMAL,
		  GtkPositionType position = GTK_POS_RIGHT);

  ~VDKCustomButton();
  /*!
    Sets a menu into combo button.

    Meaningless if button isn't a combo button
  */
  void SetComboMenu(VDKMenu* menu);

  /*
   */
  virtual void SetForeground(VDKRgb color, 
			     GtkStateType state = GTK_STATE_NORMAL) 
    {
      VDKLabel *label = Label;
      if( label)
        _setForeground_(label->Widget(),
			color.red,
			color.green,
			color.blue, 
			state);
    }
  /*
   */
  virtual void SetFont(VDKFont* font) 
    {
      VDKLabel *label = Label;
      if( label)
        _setFont_(label->Widget(),font);
    }
  /*
   */
  void SetCaption(const char* str) 
    {
      VDKLabel *label = Label;
      if( label)
        label->Caption = str;
    }
  /*
   */
  const char* GetCaption () 
    {
      VDKLabel *label = Label;
      if( label)
	return label->Caption;
      else
	return (const char*) NULL;
    }
  /*
   */
  void SetCaptionWrap (bool flag) 
    {
      VDKLabel *label = Label;
      if( label)
	gtk_label_set_line_wrap (GTK_LABEL (label->Widget()), flag);
    }
  /*
   */
  bool GetCaptionWrap () 
    {
      VDKLabel *label = Label;
      if( label)
	return GTK_LABEL (label->Widget())->wrap;
      else
	return false;
    }
  /*
   */
  void SetRelief(GtkReliefStyle style) 
    {
      if(GTK_IS_BUTTON(widget))
	gtk_button_set_relief(GTK_BUTTON(widget), style);
    }
  /*
   */
  GtkReliefStyle GetRelief() 
    {
      if(GTK_IS_BUTTON(widget))
	return gtk_button_get_relief(GTK_BUTTON(widget));
      else
	return GTK_RELIEF_NORMAL;
    }
  /*
   */
  void Toggle() { 
    if(GTK_IS_TOGGLE_BUTTON(widget))
      Checked = Checked ? false : true; 
  }
  /*
   */
  void SetChecked(bool flag)
    {
      if(GTK_IS_TOGGLE_BUTTON(widget))
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widget),flag);
    }
  /*
   */
  bool GetChecked()
    { 
      if(GTK_IS_TOGGLE_BUTTON(widget))
	return GTK_TOGGLE_BUTTON(widget)->active ? true : false; 
      else
	return false;
    }
  /*
   */
  virtual void Enable(bool flag = true);
#ifdef USE_SIGCPLUSPLUS
 public:
  /*!
    Extended LS signal system:
    Received when button is toggled
    \param bool state
  */
  VDKSignal1<void, bool> OnButtonToggled;
  /*!
    Extended LS signal system:
    Received when button is clicked
  */
  VDKSignal0<void> OnButtonClicked;
/*!
    Extended LS signal system:
    Received when button is pressed
  */
  VDKSignal0<void> OnButtonPressed;
/*!
    Extended LS signal system:
    Received when mouse leaves button
  */
  VDKSignal0<void> OnButtonLeave;
 private:
  static void _handle_clicked(GtkWidget*, gpointer);
  static void _handle_pressed(GtkWidget*, gpointer);
  static void _handle_leave(GtkWidget*, gpointer);
#endif /* USE_SIGCPLUSPLUS */
};

#endif
