%define OT_Function_doc
R"RAW(Function base class.

Notes
-----
A function acts on points to produce points: :math:`f: \Rset^d \rightarrow \Rset^{d'}`.

A function enables to evaluate its gradient and its hessian when mathematically defined.


Examples
--------
Create a *Function* from a list of analytical formulas and
descriptions of the input vector and the output vector :

>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x0', 'x1'],
...                         ['x0 + x1', 'x0 - x1'])
>>> print(f([1, 2]))
[3,-1]


Create a *Function* from strings:

>>> import openturns as ot
>>> f = ot.SymbolicFunction('x', '2.0*sqrt(x)')
>>> print(f(([16],[4])))
    [ y0 ]
0 : [ 8  ]
1 : [ 4  ]


Create a *Function* from a Python function:

>>> def a_function(X):
...     return [X[0] + X[1]]
>>> f = ot.PythonFunction(2, 1, a_function)
>>> print(f(((10, 5),(6, 7))))
    [ y0 ]
0 : [ 15 ]
1 : [ 13 ]


See :class:`~openturns.PythonFunction` for further details.

Create a *Function* from a Python class:

>>> class FUNC(ot.OpenTURNSPythonFunction):
...     def __init__(self):
...         super(FUNC, self).__init__(2, 1)
...         self.setInputDescription(['R', 'S'])
...         self.setOutputDescription(['T'])
...     def _exec(self, X):
...         Y = [X[0] + X[1]]
...         return Y
>>> F = FUNC()
>>> func = ot.Function(F)
>>> print(func((1.0, 2.0)))
[3]


See :class:`~openturns.PythonFunction` for further details.

Create a *Function* from another *Function*:

>>> f = ot.SymbolicFunction(ot.Description.BuildDefault(4, 'x'),
...                         ['x0', 'x0 + x1', 'x0 + x2 + x3'])

Then create another function by setting x1=4 and x3=10:

>>> g = ot.ParametricFunction(f, [3, 1], [10.0, 4.0], True)
>>> print(g.getInputDescription())
[x0,x2]
>>> print(g((1, 2)))
[1,5,13]

Or by setting x0=6 and x2=5:

>>> g = ot.ParametricFunction(f, [3, 1], [6.0, 5.0], False)
>>> print(g.getInputDescription())
[x3,x1]
>>> print(g((1, 2)))
[6,8,12]


Create a *Function* from another *Function*
and by using a comparison operator:

>>> analytical = ot.SymbolicFunction(['x0','x1'], ['x0 + x1'])
>>> indicator = ot.IndicatorFunction(ot.LevelSet(analytical, ot.Less(), 0.0))
>>> print(indicator([2, 3]))
[0]
>>> print(indicator([2, -3]))
[1]

Create a *Function* from a collection of functions:

>>> functions = list()
>>> functions.append(ot.SymbolicFunction(['x1', 'x2', 'x3'],
...                                      ['x1^2 + x2', 'x1 + x2 + x3']))
>>> functions.append(ot.SymbolicFunction(['x1', 'x2', 'x3'],
...                                      ['x1 + 2 * x2 + x3', 'x1 + x2 - x3']))
>>> function = ot.AggregatedFunction(functions)
>>> print(function([1.0, 2.0, 3.0]))
[3,6,8,0]

Create a *Function* which is the linear combination *linComb*
of the functions defined in  *functionCollection* with scalar weights
defined in *scalarCoefficientColl*:

:math:`functionCollection  = (f_1, \hdots, f_N)`
where :math:`\forall 1 \leq i \leq N, \,     f_i: \Rset^n \rightarrow \Rset^{p}`
and :math:`scalarCoefficientColl = (c_1, \hdots, c_N) \in \Rset^N`
then the linear combination is:

.. math::

    linComb: \left|\begin{array}{rcl}
                  \Rset^n & \rightarrow & \Rset^{p} \\
                  \vect{X} & \mapsto & \displaystyle \sum_i c_if_i (\vect{X})
              \end{array}\right.

>>> function2 = ot.LinearCombinationFunction(functions, [2.0, 4.0])
>>> print(function2([1.0, 2.0, 3.0]))
[38,12]


Create a *Function* which is the linear combination
*vectLinComb* of the scalar functions defined in
*scalarFunctionCollection* with vectorial weights defined in
*vectorCoefficientColl*:

If :math:`scalarFunctionCollection = (f_1, \hdots, f_N)`
where :math:`\forall 1 \leq i \leq N, \,    f_i: \Rset^n \rightarrow \Rset`
and :math:`vectorCoefficientColl = (\vect{c}_1, \hdots, \vect{c}_N)`
where :math:`\forall 1 \leq i \leq N, \,   \vect{c}_i \in \Rset^p`

.. math::

    vectLinComb: \left|\begin{array}{rcl}
                     \Rset^n & \rightarrow & \Rset^{p} \\
                     \vect{X} & \mapsto & \displaystyle \sum_i \vect{c}_if_i (\vect{X})
                 \end{array}\right.

>>> functions=list()
>>> functions.append(ot.SymbolicFunction(['x1', 'x2', 'x3'],
...                                      ['x1 + 2 * x2 + x3']))
>>> functions.append(ot.SymbolicFunction(['x1', 'x2', 'x3'],
...                                      ['x1^2 + x2']))
>>> function2 = ot.DualLinearCombinationFunction(functions, [[2., 4.], [3., 1.]])
>>> print(function2([1, 2, 3]))
[25,35]

If the input and output dimensions of two functions are equal
then we can perform arithmetic operations on these functions
using the `+`, `-` and `*` operators.

>>> g1 = ot.SymbolicFunction(['x1', 'x2', 'x3'],
...                          ['x1 + 2 * x2 + x3'])
>>> g2 = ot.SymbolicFunction(['x1', 'x2', 'x3'],
...                          ['x1^2 + x2'])
>>> g_sum = g1 + g2
>>> g_difference = g1 - g2
>>> g_product = g1 * g2

Create a *Function* from values of the inputs and the outputs:

>>> inputSample = [[1.0, 1.0], [2.0, 2.0]]
>>> outputSample = [[4.0], [5.0]]
>>> database = ot.DatabaseFunction(inputSample, outputSample)
>>> x = [1.8]*database.getInputDimension()
>>> print(database(x))
[5]


Create a *Function* which is the composition function
:math:`f\circ g`:

>>> g = ot.SymbolicFunction(['x1', 'x2'],
...                         ['x1 + x2','3 * x1 * x2'])
>>> f = ot.SymbolicFunction(['x1', 'x2'], ['2 * x1 - x2'])
>>> composed = ot.ComposedFunction(f, g)
>>> print(composed([3, 4]))
[-22])RAW"
%enddef
%feature("docstring") OT::FunctionImplementation
OT_Function_doc

// ---------------------------------------------------------------------

%define OT_Function_getCallsNumber_doc
"Accessor to the number of direct calls to the function.

Returns
-------
calls_number : int
    Integer that counts the number of times the function has been called
    directly through the `()` operator."
%enddef
%feature("docstring") OT::FunctionImplementation::getCallsNumber
OT_Function_getCallsNumber_doc

// ---------------------------------------------------------------------

%define OT_Function_getEvaluationCallsNumber_doc
"Accessor to the number of times the evaluation of the function has been called.

Returns
-------
evaluation_calls_number : int
    Integer that counts the number of times the evaluation of the function
    has been called since its creation.
    This may include indirect calls via finite-difference gradient or Hessian."
%enddef
%feature("docstring") OT::FunctionImplementation::getEvaluationCallsNumber
OT_Function_getEvaluationCallsNumber_doc

// ---------------------------------------------------------------------

%define OT_Function_getGradientCallsNumber_doc
"Accessor to the number of times the gradient of the function has been called.

Returns
-------
gradient_calls_number : int
    Integer that counts the number of times the gradient of the
    Function has been called since its creation.
    Note that if the gradient is implemented by a finite difference method,
    the gradient calls number is equal to 0 and the different calls are
    counted in the evaluation calls number."
%enddef
%feature("docstring") OT::FunctionImplementation::getGradientCallsNumber
OT_Function_getGradientCallsNumber_doc

// ---------------------------------------------------------------------

%define OT_Function_getHessianCallsNumber_doc
"Accessor to the number of times the hessian of the function has been called.

Returns
-------
hessian_calls_number : int
    Integer that counts the number of times the hessian of the
    Function has been called since its creation.
    Note that if the hessian is implemented by a finite difference method,
    the hessian calls number is equal to 0 and the different calls are counted
    in the evaluation calls number."
%enddef
%feature("docstring") OT::FunctionImplementation::getHessianCallsNumber
OT_Function_getHessianCallsNumber_doc

// ---------------------------------------------------------------------
%define OT_Function_getMarginal_doc
R"RAW(Accessor to marginal.

Parameters
----------
indices : int or list of ints
    Set of indices for which the marginal is extracted.

Returns
-------
marginal : :class:`~openturns.Function`
    Function corresponding to either :math:`f_i` or
    :math:`(f_i)_{i \in indices}`, with :math:`f:\Rset^n \rightarrow \Rset^p`
    and :math:`f=(f_0 , \dots, f_{p-1})`.)RAW"
%enddef
%feature("docstring") OT::FunctionImplementation::getMarginal
OT_Function_getMarginal_doc

// ---------------------------------------------------------------------

%define OT_Function_getImplementation_doc
"Accessor to the evaluation, gradient and hessian functions.

Returns
-------
function : :class:`~openturns.FunctionImplementation`
    A copy of the evaluation, gradient and hessian functions.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getImplementation())
input  : [x1,x2]
output : [y0]
evaluation : 2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6
gradient   :
| d(y) / d(x1) = (1)+(4*x1)+((-4*((x2)*(sin(x1)))))
| d(y) / d(x2) = (8)+((4*(cos(x1)))) 

hessian    :
|    d^2(y) / d(x1)^2 = (4)+((-4*((x2)*(cos(x1)))))
| d^2(y) / d(x2)d(x1) = (-4*(sin(x1))) 
|    d^2(y) / d(x2)^2 = 0"
%enddef
%feature("docstring") OT::FunctionImplementation::getImplementation
OT_Function_getImplementation_doc

// ---------------------------------------------------------------------

%define OT_Function_getEvaluation_doc
"Accessor to the evaluation function.

Returns
-------
evaluation : :class:`~openturns.EvaluationImplementation`
    The evaluation function.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getEvaluation())
[x1,x2]->[2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6]"
%enddef
%feature("docstring") OT::FunctionImplementation::getEvaluation
OT_Function_getEvaluation_doc

// ---------------------------------------------------------------------

%define OT_Function_getGradient_doc
R"RAW(Accessor to the gradient function.

Returns
-------
gradient : :class:`~openturns.GradientImplementation`
    The gradient function.

Examples
--------
To get the gradient formulas when the function is a  :class:`~openturns.SymbolicFunction`.
The indices are :math:`(i,j)`. The corresponding formula is
:math:`\left( \dfrac{\partial f_j(\vect{x})}{\partial x_i} \right)`.

>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> formula_deriv_x1 = f.getGradient().getImplementation().getFormula(0,0)
>>> formula_deriv_x2 = f.getGradient().getImplementation().getFormula(1,0)
)RAW"
%enddef
%feature("docstring") OT::FunctionImplementation::getGradient
OT_Function_getGradient_doc

// ---------------------------------------------------------------------

%define OT_Function_getHessian_doc
R"RAW(Accessor to the hessian function.

Returns
-------
hessian : :class:`~openturns.HessianImplementation`
    The hessian function.

Examples
--------
To get the hessian formulas when the function is a  :class:`~openturns.SymbolicFunction`.
The indices are :math:`(i,j,k)`. The corresponding formula is :math:`\left( \dfrac{\partial^2 f_k(\vect{x})}{\partial x_i \partial x_j} \right)`.

>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> formula_hessian_000 = f.getHessian().getImplementation().getFormula(0,0,0)
>>> formula_hessian_010 = f.getHessian().getImplementation().getFormula(0,1,0)
>>> formula_hessian_110 = f.getHessian().getImplementation().getFormula(1,1,0)
>>> formula_hessian_001 = f.getHessian().getImplementation().getFormula(0,0,0)
>>> formula_hessian_011 = f.getHessian().getImplementation().getFormula(0,1,0)
>>> formula_hessian_111 = f.getHessian().getImplementation().getFormula(1,1,0)
)RAW"
%enddef
%feature("docstring") OT::FunctionImplementation::getHessian
OT_Function_getHessian_doc

// ---------------------------------------------------------------------

%define OT_Function_setEvaluation_doc
"Accessor to the evaluation function.

Parameters
----------
function : :class:`~openturns.EvaluationImplementation`
    The evaluation function."
%enddef
%feature("docstring") OT::FunctionImplementation::setEvaluation
OT_Function_setEvaluation_doc

// ---------------------------------------------------------------------

%define OT_Function_setGradient_doc
"Accessor to the gradient function.

Parameters
----------
gradient_function : :class:`~openturns.GradientImplementation`
    The gradient function.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                          ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> f.setGradient(ot.CenteredFiniteDifferenceGradient(
...  ot.ResourceMap.GetAsScalar('CenteredFiniteDifferenceGradient-DefaultEpsilon'),
...  f.getEvaluation()))"
%enddef
%feature("docstring") OT::FunctionImplementation::setGradient
OT_Function_setGradient_doc

// ---------------------------------------------------------------------

%define OT_Function_setHessian_doc
"Accessor to the hessian function.

Parameters
----------
hessian_function : :class:`~openturns.HessianImplementation`
    The hessian function.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> f.setHessian(ot.CenteredFiniteDifferenceHessian(
...  ot.ResourceMap.GetAsScalar('CenteredFiniteDifferenceHessian-DefaultEpsilon'),
...  f.getEvaluation()))"
%enddef
%feature("docstring") OT::FunctionImplementation::setHessian
OT_Function_setHessian_doc

// ---------------------------------------------------------------------

%define OT_Function_gradient_doc
R"RAW(Return the Jacobian transposed matrix of the function at a point.

Parameters
----------
point : sequence of float
    Point where the Jacobian transposed matrix is calculated.

Returns
-------
gradient : :class:`~openturns.Matrix`
    The Jacobian transposed matrix of the function at *point*.

Notes
-----
Let :math:`f : \Rset^\inputDim \rightarrow \Rset^p` where the
:math:`f_i` are the components functions. Then, the gradient evaluated at :math:`\vect{x}` is
the matrix with :math:`\inputDim` rows and :math:`p` columns defined by:

.. math::
    \nabla f(\vect{x}) & = \left( \dfrac{\partial f_j(\vect{x})}{\partial x_i} \right)_{i,j} \\
     & = \left(
     \begin{array}{lcr}
    \dfrac{\partial f_1(\vect{x})}{\partial x_1} & \hdots & \dfrac{\partial f_p(\vect{x})}{\partial x_1}\\
    vdots & \vdots & \vdots \\
    \dfrac{\partial f_1(\vect{x})}{\partial x_\inputDim} & \hdots & \dfrac{\partial f_p(\vect{x})}{\partial x_\inputDim}
    \end{array}
    \right)

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6','x1 + x2'])
>>> print(f.gradient([3.14, 4]))
[[ 13.5345   1       ]
 [  4.00001  1       ]])RAW"
%enddef
%feature("docstring") OT::FunctionImplementation::gradient
OT_Function_gradient_doc

// ---------------------------------------------------------------------

%define OT_Function_hessian_doc
R"RAW(Return the hessian of the function at a point.

Parameters
----------
point : sequence of float
    Point where the hessian of the function is calculated.

Returns
-------
hessian : :class:`~openturns.SymmetricTensor`
    Hessian of the function at *point*.

Notes
-----
Let :math:`f : \Rset^\inputDim \rightarrow \Rset^p` where the
:math:`f_i` are the components functions. Then, the hessian evaluated at :math:`\vect{x}` is
a symmetric tensor with :math:`p` sheets, each one composed of :math:`\inputDim` rows and :math:`\inputDim`
columns. The :math:`k` -th sheet is defined by:

.. math::
    \left( \dfrac{\partial^2 f_k(\vect{x})}{\partial x_i \partial x_j} \right)_{i,j}

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6','x1 + x2'])
>>> print(f.hessian([3.14, 4]))
sheet #0
[[ 20          -0.00637061 ]
 [ -0.00637061  0          ]]
sheet #1
[[  0           0          ]
 [  0           0          ]])RAW"
%enddef
%feature("docstring") OT::FunctionImplementation::hessian
OT_Function_hessian_doc

// ---------------------------------------------------------------------

%define OT_Function_getDescription_doc
"Accessor to the description of the inputs and outputs.

Returns
-------
description : :class:`~openturns.Description`
    Description of the inputs and the outputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                         ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getDescription())
[x1,x2,y0]"
%enddef
%feature("docstring") OT::FunctionImplementation::getDescription
OT_Function_getDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_setDescription_doc
"Accessor to the description of the inputs and outputs.

Parameters
----------
description : sequence of str
    Description of the inputs and the outputs.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                          ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getDescription())
[x1,x2,y0]
>>> f.setDescription(['a','b','y'])
>>> print(f.getDescription())
[a,b,y]"
%enddef
%feature("docstring") OT::FunctionImplementation::setDescription
OT_Function_setDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_setInputDescription_doc
"Accessor to the description of the input vector.

Parameters
----------
description : :class:`~openturns.Description`
    Description of the input vector."
%enddef
%feature("docstring") OT::FunctionImplementation::setInputDescription
OT_Function_setInputDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_getInputDescription_doc
"Accessor to the description of the input vector.

Returns
-------
description : :class:`~openturns.Description`
    Description of the input vector.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                          ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getInputDescription())
[x1,x2]"
%enddef
%feature("docstring") OT::FunctionImplementation::getInputDescription
OT_Function_getInputDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_setOutputDescription_doc
"Accessor to the description of the output vector.

Parameters
----------
description : :class:`~openturns.Description`
    Description of the output vector."
%enddef
%feature("docstring") OT::FunctionImplementation::setOutputDescription
OT_Function_setOutputDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_getOutputDescription_doc
"Accessor to the description of the output vector.

Returns
-------
description : :class:`~openturns.Description`
    Description of the output vector.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                          ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getOutputDescription())
[y0]"
%enddef
%feature("docstring") OT::FunctionImplementation::getOutputDescription
OT_Function_getOutputDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_getInputDimension_doc
"Accessor to the dimension of the input vector.

Returns
-------
inputDim : int
    Dimension of the input vector :math:`d`.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                          ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getInputDimension())
2"
%enddef
%feature("docstring") OT::FunctionImplementation::getInputDimension
OT_Function_getInputDimension_doc

// ---------------------------------------------------------------------

%define OT_Function_getOutputDimension_doc
"Accessor to the number of the outputs.

Returns
-------
number_outputs : int
    Dimension of the output vector :math:`d'`.

Examples
--------
>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x1', 'x2'],
...                          ['2 * x1^2 + x1 + 8 * x2 + 4 * cos(x1) * x2 + 6'])
>>> print(f.getOutputDimension())
1"
%enddef
%feature("docstring") OT::FunctionImplementation::getOutputDimension
OT_Function_getOutputDimension_doc

// ---------------------------------------------------------------------

%define OT_Function_getParameterDimension_doc
"Accessor to the dimension of the parameter.

Returns
-------
parameterDimension : int
    Dimension of the parameter."
%enddef
%feature("docstring") OT::FunctionImplementation::getParameterDimension
OT_Function_getParameterDimension_doc

// ---------------------------------------------------------------------

%define OT_Function_draw_doc
R"RAW(Draw the output of function as a :class:`~openturns.Graph`.

Available usages:
    draw(*inputMarg, outputMarg, centralPoint, xiMin, xiMax, ptNb, scale*)

    draw(*firstInputMarg, secondInputMarg, outputMarg, centralPoint, xiMin_xjMin, xiMax_xjMax, ptNbs, scale, isFilled*)

    draw(*xiMin, xiMax, ptNb, scale*)

    draw(*xiMin_xjMin, xiMax_xjMax, ptNbs, scale*)

Parameters
----------
outputMarg, inputMarg : int, :math:`outputMarg, inputMarg \geq 0`
    *outputMarg* is the index of the marginal to draw as a function of the marginal
    with index *inputMarg*.
firstInputMarg, secondInputMarg : int, :math:`firstInputMarg, secondInputMarg \geq 0`
    In the 2D case, the marginal *outputMarg* is drawn as a function of the
    two marginals with indexes *firstInputMarg* and *secondInputMarg*.
centralPoint : sequence of float
    Central point with dimension equal to the input dimension of the function.
xiMin, xiMax : float
    Define the interval where the curve is plotted.
xiMin_xjMin, xiMax_xjMax : sequence of float of dimension 2.
    In the 2D case, define the intervals where the curves are plotted.
ptNb : int :math:`ptNb > 0`
    The number of points to draw the curves.
ptNbs : list of int of dimension 2 :math:`ptNbs_k > 0, k=1,2`
    The number of points to draw the contour in the 2D case.
scale : bool
    scale indicates whether the logarithmic scale is used either for one or both axes:

    - ot.GraphImplementation.NONE or 0: no log scale is used,

    - ot.GraphImplementation.LOGX or 1: log scale is used only for horizontal data,

    - ot.GraphImplementation.LOGY or 2: log scale is used only for vertical data,

    - ot.GraphImplementation.LOGXY or 3: log scale is used for both data.
isFilled : bool
    isFilled indicates whether the contour graph is filled or not

Notes
-----
We note :math:`f: \Rset^n \rightarrow \Rset^p`
where :math:`\vect{x} = (x_1, \dots, x_n)` and
:math:`f(\vect{x}) = (f_1(\vect{x}), \dots, f_p(\vect{x}))`,
with :math:`n\geq 1` and :math:`p\geq 1`.

- In the first usage:

Draws graph of the given 1D *outputMarg* marginal
:math:`f_k: \Rset^n \rightarrow \Rset` as a function of the given 1D *inputMarg*
marginal with respect to the variation of :math:`x_i` in the interval
:math:`[x_i^{min}, x_i^{max}]`, when all the other components of
:math:`\vect{x}` are fixed to the corresponding components of the *centralPoint* :math:`\vect{c}`.
Then OpenTURNS draws the graph:

.. math::
    y = f_k^{(i)}(s)

for any :math:`s \in [x_i^{min}, x_i^{max}]` where :math:`f_k^{(i)}(s)` is defined by the equation:

.. math::
    f_k^{(i)}(s) = f_k(c_1, \dots, c_{i-1}, s,  c_{i+1} \dots, c_n).

- In the second usage:

Draws the iso-curves of the given *outputMarg* marginal :math:`f_k` as a
function of the given 2D *firstInputMarg* and *secondInputMarg* marginals
with respect to the variation of :math:`(x_i, x_j)` in the interval
:math:`[x_i^{min}, x_i^{max}] \times [x_j^{min}, x_j^{max}]`, when all the
other components of :math:`\vect{x}` are fixed to the corresponding components of the
*centralPoint* :math:`\vect{c}`.
Then OpenTURNS draws the graph:

.. math::
    y = f_k^{(i,j)}(s, t)

for any :math:`(s, t) \in [x_i^{min}, x_i^{max}] \times [x_j^{min}, x_j^{max}]` where :math:`f_k^{(i,j)}` 
is defined by the equation:

.. math::
    f_k^{(i,j)}(s,t) = f_k(c_1, \dots, c_{i-1}, s, c_{i+1}, \dots, c_{j-1}, t,  c_{j+1} \dots, c_n).

- In the third usage:

The same as the first usage but only for function :math:`f: \Rset \rightarrow \Rset`.

- In the fourth usage:

The same as the second usage but only for function :math:`f: \Rset^2 \rightarrow \Rset`.


Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> f = ot.SymbolicFunction('x', 'sin(2*pi_*x)*exp(-x^2/2)')
>>> graph = f.draw(-1.2, 1.2, 100)
>>> View(graph).show()
)RAW"
%enddef
%feature("docstring") OT::FunctionImplementation::draw
OT_Function_draw_doc

// ---------------------------------------------------------------------

%define OT_Function_drawCrossCuts_doc
R"RAW(Draw the 2D and 1D cross cuts of a 1D output function as a :class:`~openturns.GridLayout`.

Parameters
----------
centralPoint : list of float
    Central point with dimension equal to the input dimension of the function.
xMin, xMax : list of float
    Define the interval where the curve is plotted.
pointNumber : :class:`~openturns.Indices`
    The number of points to draw the contours and the curves.
withMonoDimensionalCuts : bool, optional
    withMonoDimensionalCuts indicates whether the mono dimension cuts are drawn or not
    Default value is specified in the `CrossCuts-DefaultWithMonoDimensionalCuts` ResourceMap key.
isFilled : bool, optional
    isFilled indicates whether the contour graphs are filled or not
    Default value is specified in the `Contour-DefaultIsFilled` ResourceMap key
vMin, vMax : float, optional
    Define the interval used to build the color map for the contours
    If not specified, these values are computed to best fit the graphs.
    Either specify both values or do not specify any.

Notes
-----
We note :math:`f: \Rset^n \rightarrow \Rset^p`
where :math:`\vect{x} = (x_1, \dots, x_n)` and
:math:`f(\vect{x}) = (f_1(\vect{x}), \dots, f_p(\vect{x}))`,
with :math:`n\geq 1` and :math:`p\geq 1`.

In all usages, draw the 1D and 2D cross cuts of
:math:`f_k: \Rset^n \rightarrow \Rset` as a function of all input coordinates for 1D cuts
and all couples of coordinates for 2D cuts. Variable coordinates :math:`x_i` are sampled
regularly using :math:`ptNb[i]` points in the interval :math:`[x_i^{min}, x_i^{max}]`,
when all the other components of :math:`\vect{x}` are fixed to the corresponding components
of the *centralPoint* :math:`\vect{c}`. In the first usage, vMin and vMax are evaluated as
the min and max of all samples of the function value calculated in all cross cuts performed.

For 1D cross cuts the graph shows:

.. math::
    y = f_k^{(i)}(s)

for any :math:`s \in [x_i^{min}, x_i^{max}]` where :math:`f_k^{(i)}(s)` is defined by the equation:

.. math::
    f_k^{(i)}(s) = f_k(c_1, \dots, c_{i-1}, s,  c_{i+1} \dots, c_n).

- For 2D cross cuts:

.. math::
    y = f_k^{(i,j)}(s, t)

for any :math:`(s, t) \in [x_i^{min}, x_i^{max}] \times [x_j^{min}, x_j^{max}]` where :math:`f_k^{(i,j)}` 
is defined by the equation:

.. math::
    f_k^{(i,j)}(s,t) = f_k(c_1, \dots, c_{i-1}, s, c_{i+1}, \dots, c_{j-1}, t,  c_{j+1} \dots, c_n).

Examples
--------
>>> import openturns as ot
>>> from openturns.viewer import View
>>> f = ot.SymbolicFunction(['x0', 'x1', 'x2'], ['sin(1*pi_*x0) + x1 - x2 ^ 2'])
>>> grid = f.drawCrossCuts([0., 0., 0.], [-3., -3, -3], [3, 3, 3], [100, 20, 20], True, True)
>>> View(grid).show()
)RAW"
%enddef
%feature("docstring") OT::FunctionImplementation::drawCrossCuts
OT_Function_drawCrossCuts_doc

// ---------------------------------------------------------------------

%define OT_Function_getParameter_doc
"Accessor to the parameter values.

Returns
-------
parameter : :class:`~openturns.Point`
    The parameter values."
%enddef
%feature("docstring") OT::FunctionImplementation::getParameter
OT_Function_getParameter_doc

// ---------------------------------------------------------------------

%define OT_Function_setParameter_doc
"Accessor to the parameter values.

Parameters
----------
parameter : sequence of float
    The parameter values."
%enddef
%feature("docstring") OT::FunctionImplementation::setParameter
OT_Function_setParameter_doc

// ---------------------------------------------------------------------

%define OT_Function_getParameterDescription_doc
"Accessor to the parameter description.

Returns
-------
parameter : :class:`~openturns.Description`
    The parameter description."
%enddef
%feature("docstring") OT::FunctionImplementation::getParameterDescription
OT_Function_getParameterDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_setParameterDescription_doc
"Accessor to the parameter description.

Parameters
----------
parameter : :class:`~openturns.Description`
    The parameter description."
%enddef
%feature("docstring") OT::FunctionImplementation::setParameterDescription
OT_Function_setParameterDescription_doc

// ---------------------------------------------------------------------

%define OT_Function_parameterGradient_doc
"Accessor to the gradient against the parameter.

Returns
-------
gradient : :class:`~openturns.Matrix`
    The gradient."
%enddef
%feature("docstring") OT::FunctionImplementation::parameterGradient
OT_Function_parameterGradient_doc

// ---------------------------------------------------------------------

%define OT_Function_isLinear_doc
"Accessor to the linearity of the function.

Returns
-------
linear : bool
    *True* if the function is linear, *False* otherwise."
%enddef
%feature("docstring") OT::FunctionImplementation::isLinear
OT_Function_isLinear_doc

// ---------------------------------------------------------------------

%define OT_Function_isLinearlyDependent_doc
"Accessor to the linearity of the function with regard to a specific variable.

Parameters
----------
index : int
    The index of the variable with regard to which linearity is evaluated.

Returns
-------
linear : bool
    *True* if the function is linearly dependent on the specified variable, *False* otherwise."
%enddef
%feature("docstring") OT::FunctionImplementation::isLinearlyDependent
OT_Function_isLinearlyDependent_doc

// ---------------------------------------------------------------------

%define OT_Function_setStopCallback_doc
"Set up a stop callback.

Can be used to programmatically stop an evaluation.

Parameters
----------
callback : callable
    Returns a bool deciding whether to stop or continue.
"
%enddef
%feature("docstring") OT::FunctionImplementation::setStopCallback
OT_Function_setStopCallback_doc

// ---------------------------------------------------------------------

%define OT_Function_isParallel_doc
"Accessor to the parallel flag.

Returns
-------
isParallel : bool
    Whether the object is considered thread-safe.
"
%enddef
%feature("docstring") OT::FunctionImplementation::isParallel
OT_Function_isParallel_doc
