/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdint.h>
#include <stdio.h>
#include <errno.h>
#include <sys/time.h>
#include <pthread.h>

#include <xine.h>
#include <xine/xineutils.h>

#include "common.h"
#include "config_wrapper.h"
#include "kbindings.h"
#include "event_sender.h"
#include "panel.h"
#include "actions.h"
#include "event.h"
#include "xine-toolkit/image.h"
#include "xine-toolkit/labelbutton.h"

#define WINDOW_WIDTH    250
#define WINDOW_HEIGHT   221

#define _ES_TYPE_MASK 0xffff
#define _ES_STYLE_MASK 0xffff0000

typedef enum {
  _W_0 = 0, _W_1, _W_2, _W_3, _W_4, _W_5, _W_6, _W_7, _W_8, _W_9, _W_plus10,
  _W_menu1, _W_menu2, _W_menu3, _W_menu4, _W_menu5, _W_menu6, _W_menu7,
  _W_left, _W_right, _W_up, _W_down, _W_select,
  _W_anglenext, _W_angleprev,
  _W_exit, _W_LAST
} _W_t;

static const struct {
  uint8_t x, y, w, h;
  int type;
  char label[8], skin[16];
} _es_event_types[] = {
  [_W_menu1] = {  5,  5, 80, 23, ACTID_EVENT_MENU1,                             "", "NavMenu1" },
  [_W_menu2] = {  5, 28, 80, 23, ACTID_EVENT_MENU2 | XITK_DRAW_R,               "", "NavMenu2" },
  [_W_menu3] = {  5, 52, 80, 23, ACTID_EVENT_MENU3 | XITK_DRAW_G,               "", "NavMenu3" },
  [_W_menu4] = { 85,  5, 80, 23, ACTID_EVENT_MENU4 | XITK_DRAW_R | XITK_DRAW_G, "", "NavMenu4" },
  [_W_menu5] = {165,  5, 80, 23, ACTID_EVENT_MENU5 | XITK_DRAW_G | XITK_DRAW_B, "", "NavMenu5" },
  [_W_menu6] = {165, 28, 80, 23, ACTID_EVENT_MENU6,                             "", "NavMenu6" },
  [_W_menu7] = {165, 52, 80, 23, ACTID_EVENT_MENU7,                             "", "NavMenu7" },
  [_W_9] = { 57,124, 23, 23, ACTID_EVENT_NUMBER_9, "9", "NavNum9" },
  [_W_8] = { 34,124, 23, 23, ACTID_EVENT_NUMBER_8, "8", "NavNum8" },
  [_W_7] = { 11,124, 23, 23, ACTID_EVENT_NUMBER_7, "7", "NavNum7" },
  [_W_6] = { 57,147, 23, 23, ACTID_EVENT_NUMBER_6, "6", "NavNum6" },
  [_W_5] = { 34,147, 23, 23, ACTID_EVENT_NUMBER_5, "5", "NavNum5" },
  [_W_4] = { 11,147, 23, 23, ACTID_EVENT_NUMBER_4, "4", "NavNum4" },
  [_W_3] = { 57,170, 23, 23, ACTID_EVENT_NUMBER_3, "3", "NavNum3" },
  [_W_2] = { 34,170, 23, 23, ACTID_EVENT_NUMBER_2, "2", "NavNum2" },
  [_W_1] = { 11,170, 23, 23, ACTID_EVENT_NUMBER_1, "1", "NavNum1" },
  [_W_0] = { 11,193, 23, 23, ACTID_EVENT_NUMBER_0, "0", "NavNum0" },
  [_W_plus10] = { 34,193, 46, 23, ACTID_EVENT_NUMBER_10_ADD, "+10", "NavNumPlus10" },
  [_W_up]     = { 90, 39, 70, 40, ACTID_EVENT_UP,     N_("Up"), "NavUp" },
  [_W_left]   = { 20, 79, 70, 40, ACTID_EVENT_LEFT,   N_("Left"), "NavLeft" },
  [_W_select] = { 95, 84, 60, 30, ACTID_EVENT_SELECT, N_("Select"), "NavSelect" },
  [_W_right]  = {160, 79, 70, 40, ACTID_EVENT_RIGHT,  N_("Right"), "NavRight" },
  [_W_down]   = { 90,119, 70, 40, ACTID_EVENT_DOWN,   N_("Down"), "NavDown" },
  [_W_anglenext] = {165,124, 80, 23, ACTID_EVENT_ANGLE_NEXT,  N_("Angle +"), "NavAnglePlus" },
  [_W_angleprev] = {165,147, 80, 23, ACTID_EVENT_ANGLE_PRIOR, N_("Angle -"), "NavAngleMinus" },
  [_W_exit] = {175,193, 70, 23, ACTID_EVENT_SENDER, N_("Close"), "NavDismiss" }
};

/* nicely visible echo hotkeys matching our buttons.
 * simpliest way would be a 2kbyte full table.
 * to save space, just use the low 7 bits (128 byte), then reverse test again.
 * compiler will bail when this is ambigous.
 * the bit 7 set prevents ambiguity witn compiler auto zeroed fields. */
static const uint8_t _es_reverse[128] = {
  [127 & ACTID_EVENT_MENU1] = _W_menu1 | 0x80,
  [127 & ACTID_EVENT_MENU2] = _W_menu2 | 0x80,
  [127 & ACTID_EVENT_MENU3] = _W_menu3 | 0x80,
  [127 & ACTID_EVENT_MENU4] = _W_menu4 | 0x80,
  [127 & ACTID_EVENT_MENU5] = _W_menu5 | 0x80,
  [127 & ACTID_EVENT_MENU6] = _W_menu6 | 0x80,
  [127 & ACTID_EVENT_MENU7] = _W_menu7 | 0x80,
  [127 & ACTID_EVENT_NUMBER_9] = _W_9 | 0x80,
  [127 & ACTID_EVENT_NUMBER_8] = _W_8 | 0x80,
  [127 & ACTID_EVENT_NUMBER_7] = _W_7 | 0x80,
  [127 & ACTID_EVENT_NUMBER_6] = _W_6 | 0x80,
  [127 & ACTID_EVENT_NUMBER_5] = _W_5 | 0x80,
  [127 & ACTID_EVENT_NUMBER_4] = _W_4 | 0x80,
  [127 & ACTID_EVENT_NUMBER_3] = _W_3 | 0x80,
  [127 & ACTID_EVENT_NUMBER_2] = _W_2 | 0x80,
  [127 & ACTID_EVENT_NUMBER_1] = _W_1 | 0x80,
  [127 & ACTID_EVENT_NUMBER_0] = _W_0 | 0x80,
  [127 & ACTID_EVENT_NUMBER_10_ADD] = _W_plus10 | 0x80,
  [127 & ACTID_EVENT_UP]      = _W_up | 0x80,
  [127 & ACTID_EVENT_LEFT]    = _W_left | 0x80,
  [127 & ACTID_EVENT_SELECT]  = _W_select | 0x80,
  [127 & ACTID_EVENT_RIGHT]   = _W_right | 0x80,
  [127 & ACTID_EVENT_DOWN]    = _W_down | 0x80,
  [127 & ACTID_EVENT_ANGLE_NEXT]  = _W_anglenext | 0x80,
  [127 & ACTID_EVENT_ANGLE_PRIOR] = _W_angleprev | 0x80,
  [127 & ACTID_EVENT_SENDER] = _W_exit | 0x80
};

static const char _es_menu_names[][12] = {
  /*  0 */ N_("Menu"),
  /*  1 */ N_("Menu 1"), N_("Menu 2"), N_("Menu 3"), N_("Menu 4"),
           N_("Menu 5"), N_("Menu 6"), N_("Menu 7"),
  /*  8 */ N_("Menu toggle"), N_("Title"), N_("Root"), N_("Subpicture"),
           N_("Audio"), N_("Angle"), N_("Part"),
  /* 15 */ N_("Top Menu"), N_("Popup Menu"),
  /* 17 */ N_("Red"), N_("Green"), N_("Yellow"), N_("Blue")
};

static const uint8_t _es_menu_index[][8] = {
  /* "dvd:/", "dvdnav:/" */
  {  8, 9,10,11,12,13,14, 0 },
  /* "bd:/" */
  { 15,16, 3, 4, 5, 6, 7, 0 },
  /* "vdr:/", "xvdr" */
  {  0,17,18,19,20, 6, 7, 0 },
  /* default */
  {  1, 2, 3, 4, 5, 6, 7, 0 }
};

static const char _es_menu_modes[][8] = { "DVD", "Bluray", "VDR", "FILE" };

struct xui_event_sender_st {
  gui_new_window_t      nw;

  const char           *label[_W_LAST];
  xitk_widget_t        *wdgts[_W_LAST + 1];

  int                   skin;

  uint32_t              shown_menu; /** << 0 .. sizeof (_es_menu_index) / sizeof (_es_menu_index[0]) */
};

static int _es_win_move (xui_event_sender_t *es, int move) {
  xitk_rect_t _panel = { .x = 0 };
  int x, y1, y2, diff;
  int display_width = 0, display_height;

  /* simulste an actual magnet behaviour. */
  xitk_get_display_size (es->nw.gui->xitk, &display_width, &display_height);
  panel_get_window_position (es->nw.gui->panel, &_panel);
  x = _panel.x + _panel.width;
  if (x + es->nw.wr.width > display_width)
    x = _panel.x - es->nw.wr.width;
  y1 = y2 = _panel.y;
  diff = _panel.height - es->nw.wr.height;
  if (diff < 0)
    y1 += diff;
  else
    y2 += diff;
  diff = es->nw.wr.y;
  if (diff < y1)
    diff = y1;
  else if (diff > y2)
    diff = y2;

  if ((x != es->nw.wr.x) || (diff != es->nw.wr.y)) {
    es->nw.wr.x = x;
    es->nw.wr.y = diff;
    if (!move)
      return 1;
    _panel.x = es->nw.wr.x;
    _panel.y = es->nw.wr.y;
    _panel.width = XITK_INT_KEEP;
    _panel.height = XITK_INT_KEEP;
    xitk_window_move_resize (es->nw.xwin, &_panel);
    return 1;
  }
  return 0;
}

void event_sender_sticky_cb (void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  unsigned int old_flags = gui->flags;

  gui->flags &= ~XUI_FLAG_eventer_sticky;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_eventer_sticky);
  if (gui->eventer) {
    xitk_labelbutton_set_state (gui->eventer->wdgts[_W_LAST], cfg->num_value);
    if (gui->flags & ~old_flags & XUI_FLAG_eventer_sticky)
      _es_win_move (gui->eventer, 1);
  }
}

/* Send given event to xine engine */
static void _es_send (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_event_sender_t *es = (xui_event_sender_t *)data;
  int type = xitk_widget_user_id (w);

  (void)state;
  (void)modifier;
  gui_action_id (es->nw.gui, _es_event_types[type].type & _ES_TYPE_MASK);
}

static void _es_magnet (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_event_sender_t *es = data;

  (void)w;
  (void)modifier;
  if (state == (int)xitk_bitmove (es->nw.gui->flags, XUI_FLAG_eventer_sticky, 1))
    return;

  /* will call event_sender_sticky_cb () above. */
  config_update_num (es->nw.gui->xine, "gui.eventer_sticky", state);
}

static void _es_exit (xui_event_sender_t *es) {
  gui_window_delete (&es->nw);
  es->nw.gui->eventer = NULL;
  free (es);
}

static int _es_event (void *data, const xitk_be_event_t *e) {
  xui_event_sender_t *es = data;
  action_id_t id = 0;
  unsigned int id2 = 0;
  static const uint8_t map[XITK_KEY_LASTCODE] = {
    [XITK_KEY_UP]    = _W_up | 0x80,
    [XITK_KEY_DOWN]  = _W_down | 0x80,
    [XITK_KEY_LEFT]  = _W_left | 0x80,
    [XITK_KEY_RIGHT] = _W_right | 0x80,
    [XITK_KEY_RETURN] = _W_select | 0x80,
    [XITK_KEY_NEXT] = ACTID_EVENT_NEXT & 0x7f,
    [XITK_KEY_PREV] = ACTID_EVENT_PRIOR & 0x7f,
    [XITK_KEY_ESCAPE] = _W_exit | 0x80
  };

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      id = ACTID_EVENT_SENDER;
      break;
    case XITK_EV_POS_SIZE:
      es->nw.wr.x = e->x;
      es->nw.wr.y = e->y;
      es->nw.wr.width = e->w;
      es->nw.wr.height = e->h;
      return 1;
    case XITK_EV_BUTTON_UP:
      /* If we tried to move sticky window, move it back to stored position. */
      if ((es->nw.gui->flags & XUI_FLAG_eventer_sticky) && (panel_is_visible (es->nw.gui->panel) > 1))
        _es_win_move (es, 1);
      return 1;
    case XITK_EV_KEY_DOWN:
    case XITK_EV_KEY_UP:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        id2 = map[((const uint8_t *)e->utf8)[1]];
      } else if ((uint8_t)(e->utf8[0] ^ '0') < 10u) {
        id2 = (_W_0 + (e->utf8[0] - '0')) | 0x80;
      } else if (e->utf8[0] == '+') {
        id2 = _W_plus10 | 0x80;
      }
      if (!(id2 & 0x80)) {
        if (id2) {
          id = id2 | ACTID_IS_INPUT_EVENT;
          id2 = 0;
        } else {
          id = kbindings_aid_from_be_event (es->nw.gui->kbindings, e, es->nw.gui);
          if ((_es_event_types[_es_reverse[id & 127] & 127].type & _ES_TYPE_MASK) == id)
            id2 = _es_reverse[id & 127];
        }
      }
      if (id2 & 0x80)
        return xitk_widget_key_event (es->wdgts[id2 & 127], e, 1);
      if (e->type == XITK_EV_KEY_UP)
        id = 0;
      break;
    default: ;
  }
  if (id > 0) {
    gui_action_id (es->nw.gui, id);
    return 1;
  }
  return gui_handle_be_event (es->nw.gui, e);
}

static uint32_t _es_menu_mode (gGui_t *gui) {
  char b[12];

  b[0] = 0;
  gui_playlist_lock (gui);
  if (gui->mmk.mrl)
    strlcpy (b, gui->mmk.mrl, sizeof (b));
  gui_playlist_unlock (gui);

  if ((!memcmp (gui->mmk.mrl, "dvd:/", 5)) || (!memcmp (gui->mmk.mrl, "dvdnav:/", 8)))
    return 0;
  if (!memcmp (gui->mmk.mrl, "bd:/", 4))
    return 1;
  if (!memcmp (gui->mmk.mrl, "xvdr", 4) || !memcmp (gui->mmk.mrl, "vdr:/", 5))
    return 2;
  return 3;
}

int event_sender_menu_names (gGui_t *gui, const char *names[7]) {
  const uint8_t *x;
  uint32_t i;

  if (!gui || !names)
    return 0;

  x = _es_menu_index[_es_menu_mode (gui)];
  for (i = 0; i < 7; i++)
    names[i] = gettext (_es_menu_names[x[i]]);
  return 7;
}

void event_sender_update_menu_buttons (gGui_t *gui) {
  const uint8_t *x;
  uint32_t mode, i;

  if (!gui)
    return;
  if (!gui->eventer)
    return;

  mode = _es_menu_mode (gui);
  if (gui->eventer->shown_menu == mode)
    return;

  gui->eventer->shown_menu = mode;
  x = _es_menu_index[mode];
  for (i = 0; i < 7; i++)
    xitk_labelbutton_change_label (gui->eventer->wdgts[_W_menu1 + i],
      gui->eventer->label[_W_menu1 + i] = gettext (_es_menu_names[x[i]]));
  if (gui->verbosity >= 2)
    printf ("gui.eventer: menu mode %s.\n", _es_menu_modes[mode]);
}

void event_sender_move (gGui_t *gui) {
  if (gui && gui->eventer && (gui->flags & XUI_FLAG_eventer_sticky))
    _es_win_move (gui->eventer, 1);
}

void event_sender_change_skins (xui_event_sender_t *es, int synthetic) {
  (void)synthetic;
  if (es)
    _es_exit (es);
}

static void _es_adjust (gui_new_window_t *nw) {
  xui_event_sender_t *es;

  xitk_container (es, nw, nw);
  if ((es->nw.gui->flags & XUI_FLAG_eventer_sticky) && (panel_is_visible (es->nw.gui->panel) > 1))
    _es_win_move (es, 0);
}

void event_sender_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_event_sender_t *es;
  uint32_t i;

  if (!gui)
    return;

  es = gui->eventer;
  if (mode == XUI_W_OFF) {
    if (!es)
      return;
    _es_exit (es);
    return;
  } else if (mode == XUI_W_ON) {
    if (es) {
      gui_raise_window (gui, es->nw.xwin);
      return;
    }
  } else { /* toggle */
    if (es) {
      _es_exit (es);
      return;
    }
  }

  es = (xui_event_sender_t *)calloc (1, sizeof (*es));
  if (!es)
    return;

  /* Create window */
  es->nw.gui = gui;
  es->nw.title = _("xine Event Sender");
  es->nw.id = "eventer";
  es->nw.skin = "NavBG";
  es->nw.wfskin = "NavWF";
  es->nw.adjust = _es_adjust;
  es->nw.wr.x = 80;
  es->nw.wr.y = 80;
  es->nw.wr.width = WINDOW_WIDTH;
  es->nw.wr.height = WINDOW_HEIGHT;
  es->skin = gui_window_new (&es->nw);
  if (es->skin < 0) {
    free (es);
    return;
  }

  gui->eventer = es;
  xitk_window_get_window_position (es->nw.xwin, &es->nw.wr);

  if (!es->skin) {
    xitk_image_t *wf = xitk_image_new (es->nw.gui->xitk, XINE_SKINDIR "/nav.png", 0, 0, 0);

    if (wf) {
      int w = xitk_image_width (wf), h = xitk_image_height (wf);
      xitk_image_widget_t iw = {
        .nw = {
          .wl = es->nw.wl,
          .tips = "",
          .userdata = es,
          .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
        }
      };
      xitk_part_image_t image = {
        .image = wf,
        .x = 0,
        .y = 0,
        .width = w,
        .height = h,
        .num_states = 2
      };
      xitk_widget_t *sym = xitk_noskin_part_image_create (&iw,
        &image, (2 * WINDOW_WIDTH - w) >> 2, 119 + 40 + ((WINDOW_HEIGHT - 2 - 119 - 40 - h) >> 1));

      xitk_widget_set_window_focusable (sym);
      xitk_image_free_image (&wf);
    }
  }

  es->shown_menu = _es_menu_mode (es->nw.gui);
  {
    const uint8_t *x = _es_menu_index[es->shown_menu];
    for (i = 0; i < 7; i++)
      es->label[_W_menu1 + i] = gettext (_es_menu_names[x[i]]);
  }
  if (es->nw.gui->verbosity >= 2)
    printf ("gui.eventer: menu mode %s.\n", _es_menu_modes[es->shown_menu]);
  for (i = _W_0; i <= _W_plus10; i++)
    es->label[i] = _es_event_types[i].label;
  for (i = _W_left; i < _W_LAST; i++)
    es->label[i] = gettext (_es_event_types[i].label);

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = es->nw.wl,
        .userdata = es,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
      },
      .button_type       = CLICK_BUTTON,
      .align             = ALIGN_CENTER,
      .callback          = _es_send
    };

    if (es->skin) {
      for (i = 0; i < _W_LAST; i++) {
        lb.nw.skin_element_name = _es_event_types[i].skin;
        lb.nw.user_id = i;
        lb.nw.tips = lb.label = es->label[i];
        es->wdgts[i] = xitk_labelbutton_create (&lb, es->nw.gui->skin_config);
      }
      lb.nw.skin_element_name = "NavMagnet";
      lb.nw.add_state = (es->nw.gui->flags & XUI_FLAG_eventer_sticky)
                      ? (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON)
                      : (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
      lb.nw.tips = _("Event sender window stick to main panel"); /** << same as in event.c */
      lb.button_type = RADIO_BUTTON;
      lb.callback = _es_magnet;
      lb.label = "<<";
      es->wdgts[_W_LAST] = xitk_labelbutton_create (&lb, es->nw.gui->skin_config);
    } else {
      uint32_t style = XITK_DRAW_SAT (es->nw.gui->gfx_saturation);

      for (i = 0; i < _W_LAST; i++) {
        lb.nw.user_id = i;
        lb.label = es->label[i];
        lb.style = _es_event_types[i].type & _ES_STYLE_MASK;
        if (lb.style)
          lb.style |= style;
        es->wdgts[i] = xitk_noskin_labelbutton_create (&lb,
          _es_event_types[i].x, _es_event_types[i].y,
          _es_event_types[i].w, _es_event_types[i].h,
          XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
      }
      lb.style = 0;
      lb.nw.tips = _("Event sender window stick to main panel"); /** << same as in event.c */
      lb.nw.add_state = (es->nw.gui->flags & XUI_FLAG_eventer_sticky)
                      ? (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON)
                      : (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
      lb.button_type = RADIO_BUTTON;
      lb.callback = _es_magnet;
      lb.label = "<<";
      es->wdgts[_W_LAST] = xitk_noskin_labelbutton_create (&lb,
        WINDOW_WIDTH - 23 - 70 - 5, WINDOW_HEIGHT - 23 - 5, 23, 23,
        XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
    }
  }

  es->nw.key = xitk_be_register_event_handler ("eventer", es->nw.xwin, _es_event, es, NULL, NULL);
  gui_raise_window (es->nw.gui, es->nw.xwin);
}
